'Test suite for LCDI2CLib
GOTO TestSuite

'------------------------------------------------------------------
'
' LCDI2CLib  V7 2015-February-25
'
' Micromite to LCD using I2C module
'
' (c) Peter Carnegie 2014,2015
'
' This code is a heavily modified version of I2CLCD.BAS by John Gerrard
'
' This LIB can use several different models of PCF8574 based I2C to LCD interfaces
' The most important difference between such modules is whether they use P0~P3 or
' P4~P7 for the 4 bit data nibble to the LCD. This LIB can ONLY deal with LO or HI nibbles. If the 
' data bits are spread across the PCF8574 port bits in other ways, then the user will
' have to edit the code in SUB LCDI2C_DirectSend
'
' The most important call is LCDI2C_Init which configures the library appropriately.
' LCDI2C_Init I2CAddr,NibPos,RSBitNo,ENBitNo,BLBitNo,BLInv 
' 
' Where I2CAddr is the I2C Address  
'       NibPos=0 use low 4 bits ie P0~P3, NibPos=1 use high 4 bits ie P4~P7  
'       RSBitNo is PCF8574 bit number for RS 0~7  
'       ENBitNo is PCF8574 bit number for EN 0~7  
'       BLBitNo is PCF8574 bit number for Backlight control 0~7  
'       BLInv specifies if Backlight turned ON by a 1 (default) or ON by a 0 (inverted)
'
'  
' Using the I2C -> LCD Interface from eBay priced about US$1.60
' http://www.ebay.co.uk/itm/291116790762?ssPageName=STRK:MEWNX:IT&_trksid=p3984.m1439.l2649
' http://www.youtube.com/watch?v=wmTWk4Rwfw0
'
' Uses PCF8574 - 8 bit I2C Port Expander
' I2CAddress is &H27 with unjumpered address pads
' PCF8574 port is wired as follows
' P0 - RS
' P1 - RW
' P2 - EN
' P3 - Backlight On/Off
' P4 - D4 LCD
' P5 - D5 LCD
' P6 - D6 LCD
' P7 - D7 LCD
'
' LCDI2C_Init I2CAddr,NibPos,RSBitNo,ENBitNo,BLBitNo 
' LCDI2C_Init &H27,1,0,2,3 

'
' Using BigMik board (see http://www.thebackshed.com/forum/forum_posts.asp?TID=6715&PN=1&TPN=2)
' Uses PCF8574 - 8 bit I2C Port Expander
' I2CAddress is &H20~&H27 depending on jumper settings
' PCF8574 port is wired as follows
' P0 - D4
' P1 - D5
' P2 - D6
' P3 - D7
' P4 - RS
' P5 - Backlight 
' P6 - RW
' P7 - EN
'
' LCDI2C_Init I2CAddr,NibPos,RSBitNo,ENBitNo,BLBitNo,BLInv 
' LCDI2CInit &H2X,0,4,7,5,0 
'
' Functions in the Library are modelled after the native MMBasic LCD commands
' to ease code conversion
'
' SUB LCDI2C(LineNum,CharPos,Text$) Display string Text$ on LineNum of the display starting at CharPos
' SUB LCDI2C_Clear()                Clears the LCD
' SUB LCDI2C_Init I2CAddr,NibPos,RSBitNo,ENBitNo,BLBitNo  Initialise the I2C and LCD
' SUB LCDI2C_Close()                Close the I2C LCD

' SUB LCDI2C_BackLight(State)       Turn LCD backlight ON | OFF

' SUB LCDI2C_CMD(Byte)              Send a COMMAND byte to the LCD, ie RS=0
' SUB LCDI2C_DATA(Byte)             Send a DATA byte to the LCD, ie RS=1
'
' SUB LCDI2C_DirectSend(Byte)       Send a BYTE to the LCD - used internally by the lib
' SUB LCDI2C_WireWrite(Byte)        Write a byte onto the I2C bus to the LCD
'
'
' Creates 8 Global Variables
' LCDI2C_LCDBackLight, LCDI2C_I2CAddr, LCDI2C_RSDataMask, 
' LCDI2C_EMask, LCDI2C_BckLight, LCDI2C_NibPos, LCDI2C_BLOn, LCDI2C_BLOff
'  
'
' V3 - Added SUB LCDI2C_BackLight(State) where State=0=Backlight OFF, NonZero=BacklightON.
'	     Deleted SUB LCDI2C_BacklightOn and SUB LCDI2C_BacklightOFF
'
' V4 - Changed Test Suite to use SUB LCDI2C_BackLight(State) instead of LCDI2C_BacklightOn/LCDI2C_BacklightOFF
'
' V5 - Changed SUB LCDI2C_DATA(Byte) to allow for Low nibble i/f's like the MUP
'      LCDI2C_DirectSend now takes 2 arguments, the second is the state of RS, 0 for CMD, 1 for DATA
'      Moved I2C Open out from the library
' 
' V6 - Updated LCDI2C_Init to include optional BLInv parameter to control if backlight control
'      is inverted.
'
' V7 - Renamed variable LCDI2C_Backlight to LCDI2C_Bcklight because in MMBasic V 4.6 a variable 
'      and a SUB should have different names
'



'------------------------------------------------------------------
'
' Print String to LCD
'
'
SUB LCDI2C(LineNum,CharPos,Text$)
  LOCAL I
  IF LineNum=1 THEN I=(&H80 + CharPos-1)
  IF LineNum=2 THEN I=(&HC0 + CharPos-1)
  IF LineNum=3 THEN I=(&H94 + CharPos-1)
  IF LineNum=4 THEN I=(&HD4 + CharPos-1)
  LCDI2C_CMD(I)

  FOR I=1 TO LEN(Text$)
    LCDI2C_DATA(ASC(MID$(Text$,I,1)))
  NEXT I

END SUB


'------------------------------------------------------------------
'
' Clear LCD
' 
'
SUB LCDI2C_Clear()
  LCDI2C_CMD(&H01)
END SUB


'------------------------------------------------------------------
'
' INITIALIZE LCD
' 
' Where I2CAddr is the I2C Address 
'       NibPos=0 use low 4 bits ie P0~P3, NibPos=1 use high 4 bits ie P4~P7 
'       RSBitNo is PCF8574 bit number for RS 0~7 
'       ENBitNo is PCF8574 bit number for EN 0~7 
'       BLBitNo is PCF8574 bit number for Backlight control 0~7 
'       BLInv specifies Backlight is turned ON by a 1 or ON by a 0 
'             If BLInv=0 or is ommitted, then LCDI2C_BackLight(1) turn ON backlight 
'             If BLInv=1 then LCDI2C_BackLight(0)will turn ON backlight 
'
SUB LCDI2C_Init I2CAddr,NibPos,RSBitNo,ENBitNo,BLBitNo,BLInv

  'Backlight Control bit
  DIM LCDI2C_BckLight
  LCDI2C_BckLight=2^BLBitNo
  
  'Backlight Inversion controls
  DIM LCDI2C_BLOn, LCDI2C_BLOff
  IF BLInv<>0 THEN
    'Inverting Backlight Control
    LCDI2C_BLOn=&H0
    LCDI2C_BLOff=LCDI2C_BckLight
  ELSE
    'Non-Inverting Backlight control - Default
    LCDI2C_BLOn=LCDI2C_BckLight
    LCDI2C_BLOff=&H0
  ENDIF
  
  'Current Backlight state - default to ON
  DIM LCDI2C_LCDBackLight
  LCDI2C_LCDBackLight=LCDI2C_BckLight
  
  'I2C Bus Address
  DIM LCDI2C_I2CAddr
  LCDI2C_I2CAddr=I2CAddr
  
  'RS Control bit
  DIM LCDI2C_RSDataMask
  LCDI2C_RSDataMask=2^RSBitNo
  
  'EN Control Bit
  DIM LCDI2C_EMask
  LCDI2C_EMask=2^ENBitNo
  
  'Save if P0~P3 or P4~P7 bits of PCF8574 port
  DIM LCDI2C_NibPos
  LCDI2C_NibPos=NiBPos

  'Switch into 4 bit bus mode
  
  '%0011---- %0011----   8-bit / 8-bit
  LCDI2C_CMD(&H33)
  '%0011---- %0010----   8-bit / 4-bit
  LCDI2C_CMD(&H32)

  ' Byte commands - To configure the LCD

  ' Display Format
  ' 4bit mode, 2 lines, 5x7
  '
  ' 001LNF00
  ' %00101000
  LCDI2C_CMD(&H28)

  ' L : 0 = 4-bit Mode    1 = 8-bit Mode
  ' N : 0 = 1 Line        1 = 2 Lines
  ' F : 0 = 5x7 Pixels    1 = N/A

  
  ' Setup Display
  ' Display ON, Cursor On, Cursor Steady
  '
  ' 00001DCB
  ' %00001110
  LCDI2C_CMD(&H0C)
  
  ' D : 0 = Display Off   1 = Display On
  ' C : 0 = Cursor Off    1 = Cursor On
  ' B : 0 = Cursor Steady 1 = Cursor Flash

  
  ' Setup Cursor/Display
  ' Inc Cursor Cursor Move
  '
  ' 000001IS
  LCDI2C_CMD(&H06)
  ' I : 0 = Dec Cursor    1 = Inc Cursor
  ' S : 0 = Cursor Move   1 = Display Shift

  LCDI2C_CMD(&H01)
  
  'Turn Off LCDBacklight
  LCDI2C_BackLight(0)

END SUB


'------------------------------------------------------------------
'
' Close the I2C LCD
'
'
SUB LCDI2C_Close()

  'Close the I2C Bus
  I2C CLOSE

END SUB           

'------------------------------------------------------------------
'
' Turn Backlight On | Off
' If State=0 then OFF Else ON
'
SUB LCDI2C_BackLight(State)
  
  IF State<>0 THEN
  
    'Request to turn ON backlight
    LCDI2C_LCDBacklight=LCDI2C_BLOn
  
  ELSE

    'Request to turn OFF backlight
    LCDI2C_LCDBacklight=LCDI2C_BLOff
    
  ENDIF

  'Send control out to LCD
  LCDI2C_WireWrite(0)
  
END SUB


'------------------------------------------------------------------
'
' Send Command Byte to LCD
'
'
SUB LCDI2C_CMD(BYTE)

  'Send Hi Nibble
  LCDI2C_DirectSend(BYTE AND &HF0,0) 
  
  'Send Low Nibble
  LCDI2C_DirectSend((BYTE AND &H0F) * 16,0)
  
END SUB


'------------------------------------------------------------------
'
' Send Data Byte to LCD
'
'
SUB LCDI2C_DATA(BYTE)

  'Send Hi Nibble
  LCDI2C_DirectSend((BYTE AND &HF0), LCDI2C_RSDataMask)
  
  'Send Lo Nibble
  LCDI2C_DirectSend(((BYTE AND &H0F) * 16), LCDI2C_RSDataMask)
  
END SUB


'------------------------------------------------------------------
'
' Send Byte to LCD over I2C
' NB, we don't call LCDI2C_WireWrite in this SUB because that results in 
' MUCH slower execution time than inlin'ing the I2C Write's
'
'
SUB LCDI2C_DirectSend(BYTE, RSState)

  LOCAL B,B1
  
  'Use a copy so that we don't mess up caller's argument
 
  'Are D4-D7 of LCD mapped to P0~P3 of PCD8574 ?
  IF LCDI2C_NibPos=0 THEN
    B1=(BYTE\16) OR RSState
  ELSE
    B1=BYTE OR RSState
  ENDIF    
  
  'Take EN high, use var B 'cos it's quicker than doing the OR's in the I2C Write command itself
  B=B1 OR LCDI2C_EMask OR LCDI2C_LCDBacklight
  I2C WRITE LCDI2C_I2CAddr,0,1,B
  
  'Take EN Low
  I2C WRITE LCDI2C_I2CAddr,0,1,B1 OR LCDI2C_LCDBacklight
  
END SUB


'------------------------------------------------------------------
'
' Send Byte over I2C
'
'
SUB LCDI2C_WireWrite(BYTE)
  I2C WRITE LCDI2C_I2CAddr,0,1,BYTE OR LCDI2C_LCDBacklight
END SUB

'End Of LCDI2C Library code

'<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
'Cut Here and discard the Test Suite code when U include the library
'into your own programs

TestSuite:

'Start I2C
I2C OPEN 400,100 

'Initialise the I2C LCD device
'LCDI2C_Init I2CAddr,NibPos,RSBitNo,ENBitNo,BLBitNo,BLInv
LCDI2C_Init &H27,1,0,2,3,0
WaitForAKey("Press any key for next test - Turn On Backlight")

'Turn On the Backlight
LCDI2C_Backlight 1
WaitForAKey("Press any key for next test - Turn OFF Backlight")

'Turn OFF the Backlight
LCDI2C_Backlight 0
WaitForAKey("Press any key for next test - Turn On Backlight")

'Turn On the Backlight
LCDI2C_Backlight 1
WaitForAKey("Press any key for next test - Display Date on Row 1")

LCDI2C 1,3,DATE$
WaitForAKey("Press any key for next test - Display Time on Row 2")

LCDI2C 2,4,TIME$
WaitForAKey("Press any key for next test - Clear the Display")

LCDI2C_Clear
WaitForAKey("Press any key for next test - Digital Clock Mode")

DO

'  L=TIMER 'DEBUG
  LCDI2C 1,3,DATE$
  LCDI2C 2,4,TIME$
'  PRINT TIMER-L 'DEBUG
  
LOOP

END

SUB WaitForAKey(Msg$)
  IF Msg$="" THEN Msg$="Press any key to continue"
  PRINT Msg$;
  DO
  LOOP WHILE INKEY$=""
  PRINT
END SUB
