'
' Copyright (c) 2020, Anton Staaf.  All rights reserved.
' Use of this source code is governed by a BSD-style license that can be
' found in the LICENSE file.
'

option explicit
option default none
option console screen

#include "../include/display.inc"
#include "../include/math.inc"
#include "../include/range.inc"
#include "../include/transform.inc"
#include "ifs_csub.inc"

'
' Include a set of hard coded IFS fractal coefficients.
'
ifs_data:
#include "ifs_data.inc"

'------------------------------------------------------------------------------
' Read header and transform data statements from the included ifs_data.inc file
'
' TODO: Add file loading/saving and online editing of IFS transforms.
'
sub ifs.read_header name as string, count as integer
    read name$
    read count
end sub

sub ifs.read_transform transforms() as float, index as integer
    local integer i

    for i = 0 to 8
        read transforms(i,index)
    next
end sub

'------------------------------------------------------------------------------
' Read and discard all IFS data, keeping track of the number of IFS fractals
' read and the total number of transforms in all of the fractals.
'
sub ifs.read_counts num_ifs as integer, num_transforms as integer
    local string  name
    local integer count
    local float   transforms(10,1)

    restore ifs_data

    num_ifs        = 0
    num_transforms = 0

    do
        ifs.read_header name, count

        if count = 0 then exit

        num_ifs        = num_ifs + 1
        num_transforms = num_transforms + count

        ' Read and discard the transforms to move on to the next IFS data
        do while count > 0
            ifs.read_transform transforms(), 0
            count = count - 1
        loop
    loop
end sub

'------------------------------------------------------------------------------
' Read all of the IFS data into three arrays, one for the IFS names, one for
' the number of transforms in each IFS, and one that contains all of the
' transforms for all of the IFS fractals.
'
sub ifs.read_data names() as string, counts() as integer, transforms() as float
    local integer index = 0
    local integer i, j, k

    restore ifs_data

    for i = 0 to min(bound(names()), bound(counts()), bound(transforms(), 2))
        ifs.read_header names(i), counts(i)

        for j = index to index + counts(i) - 1
            ifs.read_transform transforms(), j
        next

        index = index + counts(i)
    next
end sub

'------------------------------------------------------------------------------
' Pick one transform randomly from the IFS and apply it to the given x,y point.
'
sub ifs.iterate transforms() as float, first as integer, last as integer, x as float, y as float
    local float   p = rnd
    local integer i

    for i = first to last
        p = p - transforms(9,i)

        if p <= 0 then
            transform.apply transforms(), i, x, y
            exit for
        end if
    next
end sub

'------------------------------------------------------------------------------
' Initialize the two transform weights, the first is the probability that the
' transform is selected out of all of the transforms that make up the IFS.  The
' second is the color lerp factor, or how much this transforms color is added
' to the color of the point that we are transforming.
'
sub ifs.initialize_weights transforms() as float, first as integer, last as integer
    local integer i
    local float total_det = 0

    for i = first to last
        ' The probability that a transform is taken is computed to be
        ' proportional to the determinant of the affine transform.  This
        ' ensures that transformations that cover a larger portion of the
        ' IFS will be selected more often.
        transforms(9,i) = abs(transform.determinant(transforms(), i))

        total_det = total_det + transforms(9,i)
    next

    '
    ' Since it is possible for a determinant to be zero (because the map is
    ' not invertible) we apply a laplace smoothing factor that is proportional
    ' to the sum of all of the absolute values of the transforms determinants.
    ' This ensures, that even if a transforms area is in zero it will have a
    ' non-zero probability.
    '
    local integer count = last - first + 1
    local float   alpha = total_det / 100

    for i = first to last
        transforms( 9,i) = (transforms(9,i) + alpha) / (total_det + count * alpha)

        '
        ' I've found that the fourth root of the transform probability makes
        ' for very nice images.  This could very well be changed to any other
        ' function and produce different nice results.
        '
        transforms(10,i) = sqr(sqr(transforms(9,i)))
    next
end sub

'------------------------------------------------------------------------------
' Compute the starting state for a given IFS.
'
' The Banach fixed-point theorem says that if the transformation is a
' contraction, which should always be true for an IFS, then we can compute the
' fixed-point of the transformation by iterating the transformation a large
' enough number of times.  I'm selecting the first transformation of the given
' IFS and iterating 100 times.
'
' We initialize the X and Y coordinates of the state to the resulting
' approximate fixed-point.  The points color is initialized to the color of the
' transform, which is what we would expect from repeated lerping toward that
' color.  And the random number generator state is initialized arbitrarily to
' zero and one.  The random number generator actually uses the two 64-bit
' doubles as uint64_t types, but all that matters is that both uint64_ts are
' non-zero to start.
'
sub ifs.initialize_state transforms() as float, index as integer, state() as float
    local integer i
    local float   x = 0
    local float   y = 0

    for i = 0 to 100
        transform.apply transforms(), index, x, y
    next

    state(0) = x
    state(1) = y
    state(2) = transforms(6,index)
    state(3) = transforms(7,index)
    state(4) = transforms(8,index)
    state(5) = 0
    state(6) = 1
end sub

'------------------------------------------------------------------------------
' Given a starting state and an IFS we generate 1000 points on the attractor
' and track the smallest axis aligned bounding box that contains all points.
' This gives us a reasonable bound for the attractor.
'
sub ifs.initialize_range transforms() as float, first as integer, last as integer, state() as float, range() as float
    local integer i
    local float   x = state(0)
    local float   y = state(1)

    range.reset range()

    for i = 1 to 300
        ifs.iterate transforms(), first, last, x, y
        range.expand range(), x, y
    next
end sub

'------------------------------------------------------------------------------
' Given the approximate bound of an IFS we compute the display transform that
' will center and scale the attractor to the current screen resolution.
'
sub ifs.initialize_display range() as float, shape() as float, display() as float
    '
    ' Make a copy of the target range so it can be scaled to give a nice
    ' border.  We can swap the min and max values for the Y-axis now as
    ' well to orient the fractal as expected.
    '
    local float from_range(3) = (range(0), range(3), range(2), range(1))

    range.scale from_range(), 1.25

    display.initialize display(), shape(), from_range()
end sub

'------------------------------------------------------------------------------
' Given a set of affine transformations compute the relative probabilities,
' initial state for rendering, and the approximate range of the IFS fractal.
'
sub ifs.initialize transforms() as float, first as integer, last as integer, state() as float, range() as float
    ifs.initialize_weights transforms(), first, last
    ifs.initialize_state   transforms(), first, state()
    ifs.initialize_range   transforms(), first, last, state(), range()
end sub

'------------------------------------------------------------------------------
' Rotate the given transform about the center of the range by theta radians.
'
sub ifs.rotate transforms() as float, i as integer, range() as float, theta as float
    local float x = range.center_x(range())
    local float y = range.center_y(range())

    transform.apply     transforms(), i, x, y
    transform.translate transforms(), i, -x, -y
    transform.rotate    transforms(), i, theta
    transform.translate transforms(), i, x, y
end sub

'------------------------------------------------------------------------------
' Shear the given transform about the center of the range by shear.
'
sub ifs.shear transforms() as float, i as integer, range() as float, shear as float
    local float x = range.center_x(range())
    local float y = range.center_y(range())

    transform.apply     transforms(), i, x, y
    transform.translate transforms(), i, -x, -y
    transform.shear     transforms(), i, shear
    transform.translate transforms(), i, x, y
end sub

'------------------------------------------------------------------------------
' Scale the given transform about the center of the range by sx,sy.
'
sub ifs.scale transforms() as float, i as integer, range() as float, sx as float, sy as float
    local float x = range.center_x(range())
    local float y = range.center_y(range())

    transform.apply     transforms(), i, x, y
    transform.translate transforms(), i, -x, -y
    transform.scale     transforms(), i, sx, sy
    transform.translate transforms(), i, x, y
end sub

'------------------------------------------------------------------------------
' Constrain the given transform to have a maximum singular value less than 0.9.
' This ensures that the transform is a contraction, and thus a valid transform
' for an IFS fractal.
'
sub ifs.constrain transforms() as float, index as integer, range() as float
    const maximum! = 0.9

    local integer i, j
    local float   gradient(3)
    local float   sigma

    '
    ' Compute the transformed center of the range before applying twp gradient
    ' descent steps to the transform matrix.
    '
    local float x = range.center_x(range())
    local float y = range.center_y(range())

    transform.apply transforms(), index, x, y

    '
    ' These gradient descent steps first ensure that the largest singular value
    ' is less than maximum (0.9).  The second iteration is necessary in case
    ' the smaller singular value was also larger than the maximum.
    '
    for i = 0 to 1
        transform.gradient transforms(), index, gradient(), sigma

        if sigma > maximum! then
            for j = 0 to 3
                transforms(j,index) = transforms(j,index) - gradient(j) * (sigma - maximum!)
            next
        end if
    next

    '
    ' The transformed center of the range will have changed, so we translate
    ' the new transform by the difference of the transformed centers before
    ' and after the gradient descent steps.  This ensures that the center of
    ' the range will still transform to the same position as it did before
    ' ifs.constrain was called.
    '
    local float cx = range.center_x(range())
    local float cy = range.center_y(range())

    transform.apply     transforms(), index, cx, cy
    transform.translate transforms(), index, x - cx, y - cy
end sub

'------------------------------------------------------------------------------
' Draw the transform by applying it to the range and drawing the result.
'
sub ifs.draw_transform transforms() as float, index as integer, display() as float, range() as float
    local float ax(3) = (range(0), range(2), range(2), range(0))
    local float ay(3) = (range(1), range(1), range(3), range(3))
    local integer j

    for j = 0 to 3
        transform.apply transforms(), index, ax(j), ay(j)

        ax(j) = display.screen_x(display(), ax(j))
        ay(j) = display.screen_y(display(), ay(j))
    next

    local float r = transforms(6,index) * 255
    local float g = transforms(7,index) * 255
    local float b = transforms(8,index) * 255

    polygon 4, ax(), ay(), rgb(r,g,b)
end sub

'------------------------------------------------------------------------------
' Render a set of affine transformations using the ifs_csub routine.  This
' function manages the parameter passing and splitting up into small chunks
' the numer of points to render.
'
sub ifs.render transforms() as float, first as integer, last as integer, state() as float, display() as float, count as integer, blend as float
    local integer buffer          = MM.INFO(page address 0)
    local integer resolution(1)   = (MM.HRES, MM.VRES)
    '
    ' The CSUB expects pointers to the first and last transform passed as
    ' parameters, instead of the full transforms array and first/last indicies.
    '
    local integer first_transform = peek(varaddr transforms(0, first))
    local integer last_transform  = peek(varaddr transforms(0, last))

    ' Don't let the CSUB run for too long as it will interfere with keyboard
    ' event handling.  Instead, process the IFS fractal 500 points at a time.
    do while count > 0
        ifs_csub buffer, resolution(), first_transform, last_transform, state(), display(), 500, blend
        count = count - 500
    loop
end sub

'------------------------------------------------------------------------------
' The main loop of the IFS application is a state machine that is managed by
' the update_state routine.  It handles the rendering of the status bar based
' on the current state as well as transitioning to new states based on key
' presses.  It also passes the key press detected back to the state routines
' so they can implement their UI appropriately.
'
' I think this is a reasonable architecture given the language and platform,
' but this is the first application with a UI that I've written for MMBasic.
'
const STATE_SELECT_IFS% = 0
const STATE_RENDER_IFS% = 1
const STATE_EDIT_IFS%   = 2
const STATE_QUIT%       = 3

'
' Human readable key code constants.
'
const KEY_NONE%      =   0
const KEY_TAB%       =   9
const KEY_CTRL_Q%    =  17
const KEY_ESCAPE%    =  27
const KEY_UP%        = 128
const KEY_DOWN%      = 129
const KEY_LEFT%      = 130
const KEY_RIGHT%     = 131
const KEY_INSERT%    = 132
const KEY_HOME%      = 134
const KEY_END%       = 135
const KEY_PAGE_UP%   = 136
const KEY_PAGE_DOWN% = 137
const KEY_F1%        = 145

sub update_state state as integer, key as integer
    local string status

    key = asc(inkey$)

    '
    ' Flush the keyboard input buffer to prevent double taps and repeated keys
    ' from accumulating.
    '
    do while inkey$ <> "" : loop

    select case state
        case STATE_SELECT_IFS%
            status = "^Q=Quit, Enter=Select IFS, Arrows=Change highlighted IFS"

            select case key
                case      10, 13 : state = STATE_RENDER_IFS%
                case KEY_CTRL_Q% : state = STATE_QUIT%
            end select

        case STATE_RENDER_IFS%
            status = "^Q=Quit, Escape=Back, F4=Edit IFS, +/-=Zoom, Arrows=Move, Space=Reset View"

            select case key
                case KEY_F1% + 3 : state = STATE_EDIT_IFS%
                case KEY_CTRL_Q% : state = STATE_QUIT%
                case KEY_ESCAPE% : state = STATE_SELECT_IFS%
            end select

        case STATE_EDIT_IFS%
            status = "^Q=Quit, Escape=Back, +/-=Scale, [/]=Rotate, </>=Shear, Arrows=Move, Space=Reset View, Tab=Next Transform"

            select case key
                case KEY_CTRL_Q% : state = STATE_QUIT%
                case KEY_ESCAPE% : state = STATE_RENDER_IFS%
            end select

        case ELSE
            status = "^Q=Quit"

            select case key
                case KEY_CTRL_Q% : state = STATE_QUIT%
            end select
    end select

    text 0, MM.VRES - MM.INFO(FONTHEIGHT), status, "LT", 1, 1, rgb(255,255,255), rgb(0,0,0)
end sub

'------------------------------------------------------------------------------
function do_select_ifs(names() as string, counts() as integer, transforms() as float, first as integer, last as integer) as integer
    local integer i
    local integer index  = 0
    local integer state  = STATE_SELECT_IFS%
    local integer key    = KEY_NONE%
    local integer update = 1
    local float   blend  = 1
    local float   ifs_state(6)
    local float   ifs_range(3)
    local float   display(3)
    local float   preview(3) = (MM.HRES / 2, 0, MM.HRES, MM.VRES / 2)

    range.scale preview(), 0.9

    cls

    do
        select case key
            case 128, 130 : index = index - 1 : update = 1
            case 129, 131 : index = index + 1 : update = 1
        end select

        index = math.wrap(index, 0, bound(names()) + 1)

        for i = 0 to bound(names())
            print @(0, i * MM.INFO(FONTHEIGHT), (i = index) * 2) names(i)
        next

        if update then
            update = 0
            blend  = 1
            last   = -1

            for i = 0 to index
                first = last  + 1
                last  = first + counts(i) - 1
            next

            ifs.initialize transforms(), first, last, ifs_state(), ifs_range()
            ifs.initialize_display ifs_range(), preview(), display()

            box preview(0), preview(1), preview(2) - preview(0), preview(3) - preview(1), 1, rgb(255,255,255), rgb(0,0,0)
        end if

        '
        ' Render IFS preview
        '
        ifs.render transforms(), first, last, ifs_state(), display(), 20000, blend

        blend = blend * 0.95

        update_state(state, key)
    loop until state <> STATE_SELECT_IFS%

    do_select_ifs = state
end function

'------------------------------------------------------------------------------
function do_render_ifs(transforms() as float, first as integer, last as integer) as integer
    local integer state = STATE_RENDER_IFS%
    local integer key   = KEY_NONE%
    local float   blend = 1
    local float   ifs_state(6)
    local float   ifs_range(3)
    local float   display(3)
    local float   screen(3) = (0, 0, MM.HRES, MM.VRES)

    cls

    ifs.initialize transforms(), first, last, ifs_state(), ifs_range()
    ifs.initialize_display ifs_range(), screen(), display()

    do
        update_state(state, key)

        select case key
            case KEY_UP%            : range.translate ifs_range(),  0,  range.height(ifs_range()) / 30
            case KEY_DOWN%          : range.translate ifs_range(),  0, -range.height(ifs_range()) / 30
            case KEY_LEFT%          : range.translate ifs_range(), -range.width(ifs_range()) / 30,   0
            case KEY_RIGHT%         : range.translate ifs_range(),  range.width(ifs_range()) / 30,   0
            case asc("-"), asc("_") : range.scale ifs_range(), 1.1
            case asc("="), asc("+") : range.scale ifs_range(), 0.9
            case asc(" ")           : ifs.initialize transforms(), first, last, ifs_state(), ifs_range()
        end select

        if key then
            cls
            blend = 1

            ifs.initialize_display ifs_range(), screen(), display()
        end if

        ifs.render transforms(), first, last, ifs_state(), display(), 20000, blend

        '
        ' Slowly decay the blend factor, reducing how much color each rendering
        ' pass adds to each pixel.  The result is that areas of the IFS that
        ' are covered by many different combinations of transforms, and thus
        ' will have very different colors, will be blended nicely together.
        ' But initially with a blend of 1 the first few passes will set the
        ' pixel colors immediately, making for a fast but noisy initial
        ' rendering and then a more blended slower final image.
        '
        ' TODO: This decay factor is a magic number that happens to work well
        '       for the default zoom, but not so great when you zoom in or out.
        '       I think a better solution would be to take into account the
        '       zoom factor.
        '
        ' TODO: This also depends on the IFS itself, long spirals take many
        '       iterations to fill in and would need a slower decay.
        '
        blend = blend * 0.99
    loop until state <> STATE_RENDER_IFS%

    do_render_ifs = state
end function

'------------------------------------------------------------------------------
function do_edit_ifs(transforms() as float, first as integer, last as integer) as integer
    local integer state = STATE_EDIT_IFS%
    local integer key   = KEY_NONE%
    local integer index = first
    local float   blend = 1
    local float   ifs_state(6)
    local float   ifs_range(3)
    local float   display(3)
    local float   screen(3) = (0, 0, MM.HRES, MM.VRES)
    local float   x, y

    '
    ' This has to have two copies of the null transform because I can't make
    ' the second dimension contain only one entry.
    '
    local float null_transform(10,1) = (1,0,0,1,0,0,0.5,0.5,0.5,0,0,1,0,0,1,0,0,0.5,0.5,0.5,0,0)

    cls

    ifs.initialize transforms(), first, last, ifs_state(), ifs_range()
    ifs.initialize_display ifs_range(), screen(), display()

    do
        update_state(state, key)

        select case key
            case KEY_UP%    : transform.translate transforms(), index, 0,  range.height(ifs_range()) / 50
            case KEY_DOWN%  : transform.translate transforms(), index, 0, -range.height(ifs_range()) / 50
            case KEY_LEFT%  : transform.translate transforms(), index, -range.width(ifs_range()) / 50, 0
            case KEY_RIGHT% : transform.translate transforms(), index,  range.width(ifs_range()) / 50, 0

            case asc("["), asc("{") : ifs.rotate transforms(), index, ifs_range(),  0.02
            case asc("]"), asc("}") : ifs.rotate transforms(), index, ifs_range(), -0.02

            case asc(","), asc("<")
                ifs.shear     transforms(), index, ifs_range(),  0.05
                ifs.constrain transforms(), index, ifs_range()

            case asc("."), asc(">")
                ifs.shear     transforms(), index, ifs_range(), -0.05
                ifs.constrain transforms(), index, ifs_range()

            case asc("+"), asc("=")
                ifs.scale     transforms(), index, ifs_range(), 1.05, 1.05
                ifs.constrain transforms(), index, ifs_range()

            case asc("_"), asc("-")
                ifs.scale     transforms(), index, ifs_range(), 0.95, 0.95
                ifs.constrain transforms(), index, ifs_range()

            case asc(" ") : ifs.initialize transforms(), first, last, ifs_state(), ifs_range()
            case KEY_TAB% : index = math.wrap(index + 1, first, last + 1)
        end select

        if key then
            cls
            blend = 1

            ifs.initialize_display ifs_range(), screen(), display()
        end if

        ifs.render transforms(), first, last, ifs_state(), display(), 20000, blend

        ifs.draw_transform null_transform(), 0, display(), ifs_range()
        ifs.draw_transform transforms(), index, display(), ifs_range()

        x = 0
        y = 0

        transform.apply transforms(), index, x, y

        x = display.screen_x(display(), x)
        y = display.screen_y(display(), y)

        circle x, y, 3

        blend = blend * 0.99
    loop until state <> STATE_EDIT_IFS%

    do_edit_ifs = state
end function

'------------------------------------------------------------------------------
' From here down are commands that are run immediately.
'
' The mode can be changed to any 16-bit mode.
'
mode 9,16

'
' Scan the ifs_data statements to gather enough information to allocate the
' names, counts, and transforms arrays.  Then populate those arrays.
'
dim integer num_ifs, num_transforms

ifs.read_counts num_ifs, num_transforms

dim string  names(num_ifs - 1)
dim integer counts(num_ifs - 1)
dim float   transforms(10, num_transforms - 1)

ifs.read_data names(), counts(), transforms()

'
' Main rendering and UI loop along with additional global state
'
dim integer first ' index of first transform for the current IFS
dim integer last  ' index of last transform for the current IFS
dim integer state = STATE_SELECT_IFS%
dim integer key   = KEY_NONE%

do while state <> STATE_QUIT%
    select case state
        case STATE_SELECT_IFS% : state = do_select_ifs(names(), counts(), transforms(), first, last)
        case STATE_RENDER_IFS% : state = do_render_ifs(transforms(), first, last)
        case STATE_EDIT_IFS%   : state = do_edit_ifs(transforms(), first, last)
        case ELSE              : update_state(state, key)
    end select
loop

