'************************************************************************************* 
' 
' Peter Mather
' 
' 2.0 
' 2015-20-6 
' 
' Graphics library with drivers for SSD1983 display 
' 9 bit parallel output from Micromite MX170 - 44-pin fixed pinout
'
' drawing primitives assume a graphics display where top left is point 0, 0
'
' set screen orientation: sub SSD.setrot(x) portrait, portrait180, landscape, landscape180
' plot pixel: sub SSD_pixel(x, y, colour) x, y coordinates of pixel, colour
' draw circle: sub SSD.dcirc(x0, y0, r, colour) x, y coordinates of centre, radius, colour
' draw filled circle: sub SSD.fcirc(x0, y0, r, colour) x, y coordinates of centre, radius, colour
' draw triangle:  sub SSD.dtri(x0, y0, x1, y1, 	x2, y2, colour) three coordinate pairs, colour
' draw filled triangle sub SSD.ftri (x0, y0, x1, y1, x2, y2, colour) three coordinate pairs, colour
' print string: sub SSD.pstring(font, xx0, yy0, text$, sz, orientation, forecolour, backcolour) font number, x, y coordinates of top left of first character, text string, size, character orientation,  foreground colour,  background colour
' print bitmap: sub SSD.pbitmap(xx0, yy0, width, height, scale, forecolour, backcolour) x, y coordinates of top left of first character, bitmap, size, character orientation,  foreground colour,  background colour
' draw line: sub SSD.dline(xx0, yy0, xx1, yy1, colour) x, y of start line, x, y of end line, colour
' draw rectangle: sub SSD.drect(x, y, w, h, colour) x, y coordinate of top left, width, height, colour
' draw filled rectangle: sub SSD.frect(x, y, w, h, colour) x, y coordinate of top left, width, height, colour
' draw rounded rectangle: sub SSD.drndrect(x, y, w, h, r, colour) x, y coordinates of top left, width, height, radius of corner, colour
' draw filled rounded rectangle: sub SSD.frndrect(x, y, w, h, r, colour) x, y coordinates of top left, width, height, radius of corner, colour
' draw bitmap: sub SSD.bitmap(x, y , w , h , size, forecolour, backcolour, bitmap()) x, y coordinates of top left, width, height, scale, foreground colour, background colour, bitmap as integer array
'
option explicit
option default NONE
const displaysize=4.3 'choose the size of display 4.3 or 5 or 7
Const RS=12 'define the pin to use for RS
const RD=13 'define the pin to use for RD
const Master_reset=32 'define the pin to use for REST
dim integer rbuf(800) ' buffer for implementing scrolling sized to cater for largest display
cpu 48
  print "start Init"
  SSD.initdisplay
  print "Start main"

Do
   SSD_test
loop

end
'
' 
' Display specific routines
'
sub SSD.initdisplay
'Fixed Pins
  setpin 25,dout 'C0 - connect to D0
  setpin 26,dout 'C1 - connect to D1
  setpin 27,dout 'C2 - connect to D2
  setpin 36,dout 'C3 - connect to D3
  setpin 37,dout 'c4 - connect to D4
  setpin 38,dout 'C5 - connect to D5
  setpin 2,dout 'C6 - connect to D6
  setpin 3,dout 'C7 - connect to D7
  setpin 4,dout 'C8 - connect to D8
  pin(5)=1 :  setpin 5,DOUT 'C9 - connect to WR
' Connect RD to VCC
' Connect CS to GND
'
'Flexible pins
  pin(RS)=1: setpin RS,DOUT   
  pin(RD)=1: setpin RD,DOUT   
  pin(Master_reset)=1: SETPIN Master_reset, dout
'
  dim integer SSD.pix=0
  dim integer SSD.ffr=1
  dim integer SSD.filltri=2
  dim integer SSD.drawline=3
  dim integer SSD.writecommand=4
  dim integer SSD.writedata=5
  dim integer SSD.fpstring=7
  dim integer SSD.fcirch=8
  dim integer SSD.dcirch=9
  dim integer SSD.Bits=10
  dim integer SSD.scroll=11
  dim integer landscape=2
  dim integer landscape180=3
  dim integer portrait=0
  dim integer portrait180=1
  dim integer displaywidth,displayheight
  dim integer  SSD1963_PAG_COL_ADDR_ORDER = (1 << 5)
  dim integer  SSD1963_COL_ADDR_ORDER     = (1 << 6)
  dim integer  SSD1963_PAGE_ADDR_ORDER    = (1 << 7)
  ' Global variables
  dim integer SSD_params(8) ' Global variables
  DIM integer SSD_wi=0,SSD_ht=1,SSD_rot=2,SSD_nr=3,SSD_X=4,SSD_Y=5,SSD_seed=6,SSD_RS=7,SSD_RD=8
  local integer i,j
  if displaysize=4.3 then
    displaywidth=480
    displayheight=272
  else
    displaywidth=800
    displayheight=480
  endif
  SSD_params(SSD_wi)=displaywidth
  SSD_params(SSD_ht)=displayheight
  SSD_params(SSD_RS)=RS
  SSD_params(SSD_RD)=RD
  pin(Master_RESET)=0
  pause 100
  pin(Master_RESET)=1
  pause 100
	SSDwritecommand(&HE2)		'PLL multiplier, set PLL clock to 120M
	if displaysize=4.3 then
    SSDwritedata(&H23)	   
  else
    SSDwritedata(&H1E)	    
  endif
	SSDwritedata(&H02)
	SSDwritedata(&H54)
	SSDwritecommand(&HE0)		' PLL enable
	SSDwritedata(&H01)
	pause 10)
	SSDwritecommand(&HE0)
	SSDwritedata(&H03)
	pause 10)
	SSDwritecommand(&H01)		' software reset
	pause 100)
	SSDwritecommand(&HE6)		'PLL setting for PCLK, depends on resolution
	if displaysize=4.3 then
    SSDwritedata(&H01)
    SSDwritedata(&H1F)
  else
    SSDwritedata(&H03)
    SSDwritedata(&HFF)
  endif
	SSDwritedata(&HFF)

	SSDwritecommand(&HB0)		'LCD SPECIFICATION
	if displaysize=4.3 then
    SSDwritedata(&H20)
    SSDwritedata(&H00)
    SSDwritedata(&H01)	
    SSDwritedata(&HDF)
    SSDwritedata(&H01)
    SSDwritedata(&H0F)
    SSDwritedata(&H00)
    SSDwritecommand(&HB4)	
    SSDwritedata(&H02)
    SSDwritedata(&H13)
    SSDwritedata(&H00)
    SSDwritedata(&H08)
    SSDwritedata(&H2B)
    SSDwritedata(&H00)
    SSDwritedata(&H02)
    SSDwritedata(&H00)
    SSDwritecommand(&HB6)	
    SSDwritedata(&H01)
    SSDwritedata(&H20)
    SSDwritedata(&H00)
    SSDwritedata(&H04)
    SSDwritedata(&H0C)
    SSDwritedata(&H00)
    SSDwritedata(&H02)
  else
    SSDwritedata(&H24)
    SSDwritedata(&H00)
    SSDwritedata(&H03)
    SSDwritedata(&H1F)
    SSDwritedata(&H01)
    SSDwritedata(&HDF)
    SSDwritedata(&H00)
    SSDwritecommand(&HB4)
    SSDwritedata(&H03)
    SSDwritedata(&HA0)
    SSDwritedata(&H00)
    SSDwritedata(&H2E)
    SSDwritedata(&H30)
    SSDwritedata(&H00)
    SSDwritedata(&H0F)
    SSDwritedata(&H00)
    SSDwritecommand(&HB6)	
    SSDwritedata(&H02)
    SSDwritedata(&H0D)
    SSDwritedata(&H00)
    SSDwritedata(&H10)
    SSDwritedata(&H10)
    SSDwritedata(&H00)
    SSDwritedata(&H08)
  endif
	SSDwritecommand(&HBA)
	SSDwritedata(&H0F)		'GPIO[3:0] out 1
	SSDwritecommand(&HB8)
	SSDwritedata(&H07)	    'GPIO3=input, GPIO[2:0]=output
	SSDwritedata(&H01)		'GPIO0 normal
	SSDwritecommand(&H36)		'rotation
	SSDwritedata(&B00000011)
	SSDwritecommand(&HF0)		'pixel data interface
	SSDwritedata(&H06)
	SSDwritecommand(&H3A)		'pixel data interface
	SSDwritedata(&B01100000)
	SSDwritecommand(&H26)		'gamma curve
	SSDwritedata(&B00000001)
	pause 1
	SSDwritecommand(&H29)		'display on
	SSDwritecommand(&HBE)		'set PWM for B/L
	SSDwritedata(&H06)
	SSDwritedata(&Hf0)
	SSDwritedata(&H01)
	SSDwritedata(&Hf0)
	SSDwritedata(&H00)
	SSDwritedata(&H00)
	SSDwritecommand(&Hd0) 
	SSDwritedata(&H0d)	
	SSDwritecommand(&H2C) 
end sub
'
sub SSDwritedata(d as integer)
  local integer i=ssddraw(SSD.writedata, d, SSD_params())
end sub
'
sub SSDwritecommand(d as integer)
  local integer i=ssddraw(SSD.writecommand, d, SSD_params())
end sub
'
sub SSD.fillscreen(scol as integer)
  local integer i=SSD_params(SSD_rot)
  SSD_params(SSD_rot)=landscape
  local integer j=ssddraw(SSD.ffr,scol, 0,0,SSD_params(SSD_wi),SSD_params(SSD_ht), SSD_params())
  SSD_params(SSD_rot)=i
end sub
'
' Generic drawing routines
'
sub SSD.dcirc(x0 as integer, y0 as integer, r as integer, colour as integer) 'x, y coordinates of centre, radius
  local integer i=ssddraw(SSD.dcirch, colour,x0,y0,r,31,SSD_params())

end sub
 '
sub SSD.dtri(x0 as integer, y0 as integer, x1 as integer, y1 as integer, x2 as integer, y2 as integer, colour as integer) 'three coordinate pairs
  SSD.dline(x0, y0, x1, y1, colour)
  SSD.dline(x1, y1, x2, y2, colour)
  SSD.dline(x2, y2, x0, y0, colour)
end sub
'
sub SSD.pixel(x0 as integer, y0 as integer,  colour as integer) 'three coordinate pairs
  local integer i=ssddraw(SSD.pix, colour,x0,y0,SSD_params())
end sub
 '
sub SSD.ftri(x0 as integer, y0 as integer, x1 as integer, y1 as integer, x2 as integer, y2 as integer, colour as integer) 'three coordinate pairs
  local integer i=ssddraw(SSD.filltri, x0,y0,x1,y1,x2,y2,colour,SSD_params())
end sub
'
sub SSD.pstring(font as integer, xx0 as integer, yy0 as integer, text$, sz as integer, fontcolour as integer, backgroundcolour as integer) 'x, y coordinates of top left of first character, text string, size, orientation
  local integer j=font and 1
  local integer i=ssddraw(SSD.fpstring,j,xx0,yy0,sz,fontcolour,backgroundcolour,SSD_params(),text$)
end sub
'
sub SSD.dline(xx0 as integer, yy0 as integer, xx1 as integer, yy1 as integer, colour as integer) ' x, y of start line, x, y of end line
   local integer dp=ssddraw(SSD.drawline, colour, xx0, yy0, xx1, yy1,  SSD_params())
end sub
'
sub SSD.drect(x as integer, y as integer, w as integer, h as integer, rcolour as integer) 'x, y coordinate of top left, wi, ht
  local integer a, dp
  dp=ssddraw(SSD.ffr, rcolour, x, y, w, 1,  SSD_params()) 
  a=y+h-1
  dp=ssddraw(SSD.ffr, rcolour, x, a, w, 1,  SSD_params()) 
  dp=ssddraw(SSD.ffr, rcolour, x, y, 1, h,  SSD_params()) 
  a=x+w-1
  dp=ssddraw(SSD.ffr, rcolour, a, y, 1, h,  SSD_params()) 
end sub
'
sub SSD.bitmap(x as integer, y as integer, w as integer, h as integer, size as integer, fcolour as integer, bcolour as integer, array() as integer) 'x, y coordinate of top left, wi, ht
  local integer  j=ssddraw(SSD.Bits, x,y,w,h,size,fcolour,bcolour,array(),SSD_params())  
end sub
'
sub SSD.frect(x as integer, y as integer, w as integer, h as integer, rcolour as integer) 'x, y coordinate of top left, width, height
  local integer dp
  dp=ssddraw(SSD.ffr, rcolour, x, y, w, h,  SSD_params()) 
end sub
'
sub SSD.fcirc(x0 as integer, y0 as integer, r as integer, ccolour as integer)  'x, y coordinates of centre, radius
  local integer dp
  dp=ssddraw(SSD.ffr, ccolour, x0, y0-r, 1, 2*r+1,  SSD_params())
  dp=ssddraw(SSD.fcircH,x0, y0, r, 3, 0, ccolour, SSD_params())
end sub
 '
sub SSD.drndrect(x as integer, y as integer, w as integer, h as integer, r as integer, rcolour as integer) 'x, y coordinates of top left, width, height, radius of corner
  local integer dp
  dp=ssddraw(SSD.ffr, rcolour, x+r, y, w-2*r, 1,  SSD_params()) 
  dp=ssddraw(SSD.ffr, rcolour, x+r, y+h-1, w-2*r, 1,  SSD_params()) 
  dp=ssddraw(SSD.ffr, rcolour, x, y+r, 1, h-2*r,  SSD_params()) 
  dp=ssddraw(SSD.ffr, rcolour, x+w-1, y+r, 1, h-2*r,  SSD_params()) 
  dp=ssddraw(SSD.dcirch, rcolour, x+r, y+r, r, 1,SSD_params())
  dp=ssddraw(SSD.dcirch,  rcolour, x+w-r-1, y+r, r, 2,SSD_params())
  dp=ssddraw(SSD.dcirch,  rcolour, x+w-r-1, y+h-r-1, r, 4,SSD_params())
  dp=ssddraw(SSD.dcirch,  rcolour, x+r, y+h-r-1, r, 8,SSD_params())
end sub
'
sub SSD.frndrect(x as integer, y as integer, w as integer, h as integer, r as integer, colour as integer) 'x, y coordinates of top left, width, height, radius of corner
  local integer dp
  dp=ssddraw(SSD.ffr, colour, x+r, y, w-2*r, h,  SSD_params()) 
  dp=ssddraw(SSD.fcircH,x+w-r-1, y+r, r, 1, h-2*r-1, colour, SSD_params())
  dp=ssddraw(SSD.fcircH,x+r, y+r, r, 2, h-2*r-1, colour, SSD_params())
end sub
'
sub SSD.bitmap(x as integer, y as integer, w as integer, h as integer, r as integer, colour as integer) 'x, y coordinates of top left, width, height, radius of corner
  local integer dp
  dp=ssddraw(SSD.ffr, colour, x+r, y, w-2*r, h,  SSD_params()) 
  dp=ssddraw(SSD.fcircH,x+w-r-1, y+r, r, 1, h-2*r-1, colour, SSD_params())
  dp=ssddraw(SSD.fcircH,x+r, y+r, r, 2, h-2*r-1, colour, SSD_params())
end sub
 '
sub SSD.setrot(x%) 'sets screen rotation: 0=portrait, 1=portrait180, 2=landscape, 3-landscape180
    SSD_params(SSD_rot)=(x% and 3)
    local integer orient = &H10
    select case x%
        case portrait
           orient = SSD1963_PAGE_ADDR_ORDER OR SSD1963_PAG_COL_ADDR_ORDER
        case portrait180
            orient = SSD1963_COL_ADDR_ORDER OR SSD1963_PAG_COL_ADDR_ORDER
        case landscape
        case landscape180
            orient = SSD1963_PAGE_ADDR_ORDER OR  SSD1963_COL_ADDR_ORDER
    end select
    SSDwritecommand(&H36) ' set addresss mode
    SSDwritedata(orient)
end sub
'
CFunction SSDdraw
	00000685
	'writeCommand
	27bdffe8 afbf0014 afb00010 00808021 3c029d00 8c42001c 8ca40038 0040f809 
	24050005 36100200 3c02bf88 ac506230 24030200 ac436234 ac436238 8fbf0014 
	8fb00010 03e00008 27bd0018 

	'writeData
	27bdffe8 afbf0014 afb00010 00808021 3c029d00 8c42001c 8ca40038 0040f809 
	24050006 36100200 3c02bf88 ac506230 24030200 ac436234 ac436238 8fbf0014 
	8fb00010 03e00008 27bd0018 

	'setxy
	27bdffc8 afbf0034 afb70030 afb6002c afb50028 afb40024 afb30020 afb2001c 
	afb10018 afb00014 0080a021 00a0a821 00c0b021 00e0b821 8fb3004c 3c10bf88 
	ae006210 3c129d00 8e42001c 8e640038 0040f809 24050005 2402002a ae026230 
	24110200 ae116234 ae116238 8e42001c 8e640038 0040f809 24050006 00141202 
	ae026230 ae116234 ae116238 8e42001c 8e640038 0040f809 24050006 ae146230 
	ae116234 ae116238 8e42001c 8e640038 0040f809 24050006 00151202 ae026230 
	ae116234 ae116238 8e42001c 8e640038 0040f809 24050006 ae156230 ae116234 
	ae116238 8e42001c 8e640038 0040f809 24050005 2402002b ae026230 ae116234 
	ae116238 8e42001c 8e640038 0040f809 24050006 00161202 ae026230 ae116234 
	ae116238 8e42001c 8e640038 0040f809 24050006 ae166230 ae116234 ae116238 
	8e42001c 8e640038 0040f809 24050006 00171202 ae026230 ae116234 ae116238 
	8e42001c 8e640038 0040f809 24050006 ae176230 ae116234 ae116238 8fa20048 
	1040000d 3c029d00 8c42001c 8e640038 0040f809 24050005 3c02bf88 2403002c 
	ac436230 24030200 ac436234 ac436238 1000000e 8fbf0034 8c42001c 8e640038 
	0040f809 24050005 3c02bf88 2403002e ac436230 24030200 ac436234 ac436238 
	240301ff ac436210 8fbf0034 8fb70030 8fb6002c 8fb50028 8fb40024 8fb30020 
	8fb2001c 8fb10018 8fb00014 03e00008 27bd0038 

	'localfastfillrect
	27bdffc0 afbf003c afb70038 afb60034 afb50030 afb4002c afb30028 afb20024 
	afb10020 afb0001c 00808021 8fa20050 8fb10068 24c4ffff 8fa30058 00832021 
	2457ffff 8fa30060 02e31821 8e280000 8e250010 14a0000b 8e270008 00409021 
	00609821 30e9ffff 0004a027 0134a021 3294ffff 0006a827 0135a821 10000015 
	32b5ffff 24090001 14a9000b 24090002 3108ffff 00039027 01129021 3252ffff 
	00029827 01134021 3113ffff 00c0a021 10000018 0080a821 14a90007 24090003 
	00c09021 00809821 0040a021 10000011 0060a821 24090003 14a9000e 3108ffff 
	00049027 01129021 3252ffff 00069827 01134021 3113ffff 30e7ffff 0003a027 
	00f4a021 3294ffff 0002a827 00f53821 30f5ffff 24160001 02c63023 00c42021 
	00621023 24420001 7082b802 24040044 02202821 0411FF07 00000000 afb60010 
	afb10014 02402021 02602821 02803021 02a03821 0411FF25 00000000 7e021340 
	001093c3 325201f8 00529025 7e022880 00108103 321001c0 00508025 3c029d00 
	8c42001c 8e240038 0040f809 24050006 12e0000f 26f7ffff 3c02bf88 36520200 
	24030200 36100200 2404ffff ac526230 ac436234 ac436238 ac506230 ac436234 
	ac436238 26f7ffff 16e4fff8 00000000 24020001 00001821 8fbf003c 8fb70038 
	8fb60034 8fb50030 8fb4002c 8fb30028 8fb20024 8fb10020 8fb0001c 03e00008 
	27bd0040 

	'DBitmap
	27bdffa0 afbf005c afbe0058 afb70054 afb60050 afb5004c afb40048 afb30044 
	afb20040 afb1003c afb00038 0080a021 00a0f021 00c09021 afa7006c 8fb00070 
	8fa70074 8fa60078 8fb3007c 8fb50080 248affff 72122002 008a5021 24abffff 
	8fa4006c 72042802 00ab5821 8ea40000 8eac0010 1580000d 8ea50008 30a2ffff 
	000a1827 00431821 3063ffff 00144027 00481021 3042ffff afa40018 00a08821 
	01604021 10000012 03c04821 240d0001 158d000d 240d0002 3082ffff 000b4827 
	00494821 3129ffff 001e4027 00481021 3048ffff afa40018 00a08821 01401021 
	1000001c 02801821 518d0015 afa50018 240d0003 558d0018 7ce51340 3082ffff 
	000a4827 00494821 3129ffff 00144027 00481021 3048ffff 30a2ffff 000b1827 
	00431821 3063ffff 001e5027 004a1021 3042ffff afa50018 10000006 00808821 
	00808821 01601021 03c01821 01404021 02804821 7ce51340 000723c3 308401f8 
	00a42025 afa40030 7cf92880 00073903 30e701c0 03273825 afa70034 7cd71340 
	000623c3 308401f8 02e4b825 7cd62880 00063143 30c601c0 02c6b025 24040001 
	afa40010 afb50014 01202021 01002821 00603021 00403821 0411FE8B 00000000 
	3c029d00 8c42001c 8ea40038 0040f809 24050006 8fa2006c 18400063 02007821 
	afbe0024 00121823 afa30028 70521002 2442ffff afa2001c afb2002c 00007021 
	afa00020 24180001 36f70200 3c05bf88 24060200 36d60200 8fa20030 34550200 
	8fa30034 1000004c 34790200 04400019 0051202b 50800018 24630001 05220016 
	24630001 51600014 24630001 91a40000 008c2024 10800009 00000000 acb56230 
	aca66234 aca66238 acb96230 aca66234 aca66238 10000008 24630001 acb76230 
	aca66234 aca66238 acb66230 aca66234 aca66238 24630001 1470ffe4 24420001 
	24e70001 2508ffff 10f20010 014f5021 00ee1021 24430007 284d0000 006d100b 
	000268c3 026d6821 000817c3 00021742 01026021 318c0007 01826023 01986004 
	01401021 1000ffd1 00001821 27de0001 17d00004 25290001 10000009 8fa20020 
	0000f021 1a40fff9 02805021 8fa8001c 00003821 8fa40018 1000ffe5 0124582b 
	24420001 afa20020 8fa30024 006f1821 afa30024 8fa4001c 8fa20028 00822021 
	afa4001c 8fa3002c 8fa40020 8fa2006c 10820005 01c37021 1e00ffe9 8fa90024 
	1000ffef 8fa20020 8fbf005c 8fbe0058 8fb70054 8fb60050 8fb5004c 8fb40048 
	8fb30044 8fb20040 8fb1003c 8fb00038 03e00008 27bd0060 

	'swap
	8c820000 8ca30000 ac830000 03e00008 aca20000 

	'drawline
	27bdffa8 afbf0054 afbe0050 afb7004c afb60048 afb50044 afb40040 afb3003c 
	afb20038 afb10034 afb00030 00809821 afa5005c afa60060 afa70064 8fa20068 
	14c2001f 8fb6006c 00e5102a 10400006 8fa7005c 27a4005c 27a50064 0411FFE3 
	00000000 8fa7005c 8fa20060 afa20010 000217c3 afa20014 8fa20064 24420001 
	00471023 afa20018 000217c3 afa2001c 24020001 00001821 afa20020 afa30024 
	afb60028 02602021 00e03021 00073fc3 0411FE67 00000000 1000005b 8fbf0054 
	8fb7005c 8fa30064 16e3001f 00779023 0046102a 10400006 8fa40060 27a40060 
	27a50068 0411FFC1 00000000 8fa40060 8fa7005c afa40010 000417c3 afa20014 
	24020001 00001821 afa20018 afa3001c 8fa20068 24420001 00442023 afa40020 
	000427c3 afa40024 afb60028 02602021 00e03021 00073fc3 0411FE45 00000000 
	10000039 8fbf0054 001227c3 00929026 02449023 02e3b82a 24040001 2405ffff 
	00b7200a 0080b821 00468823 00111fc3 00718826 00718823 00c2f02a 24030001 
	2404ffff 009e180a 0060f021 02328021 24140001 0000a821 8fa7005c 8fa20060 
	afa20010 000217c3 afa20014 afb40018 afb5001c afb40020 afb50024 afb60028 
	02602021 00e03021 00073fc3 0411FE20 00000000 00101040 0051182a 54600008 
	0242102a 8fa3005c 8fa40064 1064000d 02118021 02e31821 afa3005c 0242102a 
	1440ffe6 8fa7005c 8fa20060 8fa30068 10430004 02128021 03c21021 1000ffde 
	afa20060 8fbf0054 8fbe0050 8fb7004c 8fb60048 8fb50044 8fb40040 8fb3003c 
	8fb20038 8fb10034 8fb00030 03e00008 27bd0058 

	'drawchar
	27bdffc8 afbf0034 8fa20048 8fa9004c 8fa80050 8fab0054 7c042420 3c039d00 
	8c630054 00052880 00a31821 8c650000 90a30001 90ac0002 008c682a 15a00019 
	90aa0000 90ad0003 01ac6821 008d682a 51a00015 afa90010 afa80010 afa60014 
	afa70018 706a3002 000630c3 008c6023 70cc2002 24840004 00a42821 afa5001c 
	afab0020 00402021 01202821 01403021 00603821 0411FE5A 00000000 10000013 
	8fbf0034 afa90010 00094fc3 afa90014 71485002 afaa0018 000a57c3 afaa001c 
	70681802 afa30020 00031fc3 afa30024 afab0028 00e02021 00403021 00023fc3 
	0411FDC6 00000000 8fbf0034 03e00008 27bd0038 

	'testchar
	8fa30010 8c680010 8c690000 70e61002 00442021 2502fffe 2c420002 10400009 
	8c630008 04800011 00001021 04a0000f 0124302a 14c0000d 0065302a 14c0000b 
	00000000 2d080002 11000008 24020001 04800006 00001021 04a00004 0064182a 
	14600002 0125282a 38a20001 03e00008 00000000 

	'pstring
	27bdffa8 afbf0054 afbe0050 afb7004c afb60048 afb50044 afb40040 afb3003c 
	afb20038 afb10034 afb00030 afa40020 8fa20068 afa2002c 8fb00070 8fb30088 
	8fb7008c 00c01821 0040a821 8fa50078 afa50024 8fa20080 afa20028 3c029d00 
	8c420054 00042080 00821021 8c420000 905e0000 92e20000 18400021 0200a021 
	73d08002 0200b021 00d08021 24110001 02169023 afb30010 02402021 02a02821 
	02803021 03c03821 0411FFB8 00000000 1040001c 02f11021 80440000 afb20010 
	afb50014 afb40018 afb3001c 8fa50020 8fa60024 8fa70028 0411FF6F 00000000 
	00404021 00604821 02001821 26310001 92e20000 0051102a 1040ffe5 02168021 
	ae630020 00031fc3 ae630024 8fa2002c ae620028 8fa3002c 000317c3 10000003 
	ae62002c 00004021 00004821 01001021 01201821 8fbf0054 8fbe0050 8fb7004c 
	8fb60048 8fb50044 8fb40040 8fb3003c 8fb20038 8fb10034 8fb00030 03e00008 
	27bd0058 

	'dcirch
	27bdff88 afbf0074 afbe0070 afb7006c afb60068 afb50064 afb40060 afb3005c 
	afb20058 afb10054 afb00050 00808021 00c0b021 8fb50088 8fb20090 8fa20098 
	8fb700a0 02408821 afa60038 afb5003c afa20034 30420010 1040003f 0080f021 
	00069fc3 02551021 afa20010 000217c3 afa20014 24020001 00001821 afa20018 
	afa3001c afa20020 afa30024 afb70028 02603821 0411FD26 00000000 02b21023 
	afa20010 000217c3 afa20014 24020001 00001821 afa20018 afa3001c afa20020 
	afa30024 afb70028 02002021 02c03021 02603821 0411FD16 00000000 00159fc3 
	02563821 afb50010 afb30014 24020001 00001821 afa20018 afa3001c afa20020 
	afa30024 afb70028 02002021 00e03021 00073fc3 0411FD06 00000000 02d23823 
	afb50010 afb30014 24020001 00001821 afa20018 afa3001c afa20020 afa30024 
	afb70028 02002021 00e03021 00073fc3 0411FCF7 00000000 1a2000bb 24110001 
	02328823 00121023 00021040 afa20030 26b3ffff 26d4ffff 26b50001 26d60001 
	02408021 00009021 8fa30034 30630004 afa30040 8fa20034 30420002 afa20044 
	8fa30034 30630008 afa30048 8fa20034 30420001 afa20034 06200005 8fa30030 
	2610ffff 24630002 afa30030 02238821 8fa20040 10400023 26520001 8fa3003c 
	02031021 afa20010 000217c3 afa20014 24020001 00001821 afa20018 afa3001c 
	afa20020 afa30024 afb70028 03c02021 02c03021 00163fc3 0411FCC5 00000000 
	8fa30038 02033821 afb50010 001517c3 afa20014 24020001 00001821 afa20018 
	afa3001c afa20020 afa30024 afb70028 03c02021 00e03021 00073fc3 0411FCB4 
	00000000 8fa30044 10600024 8fa30048 8fa3003c 00701023 afa20010 000217c3 
	afa20014 24020001 00001821 afa20018 afa3001c afa20020 afa30024 afb70028 
	03c02021 02c03021 00163fc3 0411FCA0 00000000 8fa30038 02033821 afb30010 
	001317c3 afa20014 24020001 00001821 afa20018 afa3001c afa20020 afa30024 
	afb70028 03c02021 00e03021 00073fc3 0411FC8F 00000000 8fa30048 10600024 
	8fa30034 8fa20038 00503823 afb50010 001517c3 afa20014 24020001 00001821 
	afa20018 afa3001c afa20020 afa30024 afb70028 03c02021 00e03021 00073fc3 
	0411FC7B 00000000 8fa3003c 02031021 afa20010 000217c3 afa20014 24020001 
	00001821 afa20018 afa3001c afa20020 afa30024 afb70028 03c02021 02803021 
	00143fc3 0411FC6A 00000000 8fa30034 50600024 2673ffff 8fa20038 00503823 
	afb30010 001317c3 afa20014 24020001 00001821 afa20018 afa3001c afa20020 
	afa30024 afb70028 03c02021 00e03021 00073fc3 0411FC56 00000000 8fa3003c 
	00701023 afa20010 000217c3 afa20014 24020001 00001821 afa20018 afa3001c 
	afa20020 afa30024 afb70028 03c02021 02803021 00143fc3 0411FC45 00000000 
	2673ffff 2694ffff 26b50001 0250102a 10400005 26d60001 00121040 24420001 
	1000ff5d 02228821 8fbf0074 8fbe0070 8fb7006c 8fb60068 8fb50064 8fb40060 
	8fb3005c 8fb20058 8fb10054 8fb00050 03e00008 27bd0078 

	'fcirch
	27bdff90 afbf006c afbe0068 afb70064 afb60060 afb5005c afb40058 afb30054 
	afb20050 afb1004c afb00048 8fb10080 8fb40090 8fbe0098 afa40038 afa6003c 
	8fa20088 1a200076 02801821 24100001 02118023 0011b823 0017b840 2495ffff 
	24d3ffff 26940003 24960001 00009021 30440001 afa40030 24630001 afa30040 
	30420002 afa20034 06000004 8fa20030 2631ffff 26f70002 02178021 1040002a 
	26520001 8fa3003c 00711023 afa20010 000217c3 afa20014 24020001 00001821 
	afa20018 afa3001c 00111040 8fa30040 00621021 afa20020 000217c3 afa20024 
	8fa400a0 afa40028 03c02021 02c03021 00163fc3 0411FBF0 00000000 8fa20038 
	02223821 afb30010 001317c3 afa20014 24020001 00001821 afa20018 afa3001c 
	afb40020 001417c3 afa20024 8fa300a0 afa30028 03c02021 00e03021 00073fc3 
	0411FBDD 00000000 8fa40034 5080002b 26b5ffff 8fa3003c 00711023 afa20010 
	000217c3 afa20014 24020001 00001821 afa20018 afa3001c 00111040 8fa30040 
	00621021 afa20020 000217c3 afa20024 8fa400a0 afa40028 03c02021 02a03021 
	00153fc3 0411FBC4 00000000 8fa20038 00513823 afb30010 001317c3 afa20014 
	24020001 00001821 afa20018 afa3001c afb40020 001417c3 afa20024 8fa300a0 
	afa30028 03c02021 00e03021 00073fc3 0411FBB1 00000000 26b5ffff 2673ffff 
	26940002 0251102a 10400005 26d60001 00121040 24420001 1000ff9b 02028021 
	8fbf006c 8fbe0068 8fb70064 8fb60060 8fb5005c 8fb40058 8fb30054 8fb20050 
	8fb1004c 8fb00048 03e00008 27bd0070 

	'ftri
	27bdff90 afbf006c afb50068 afb40064 afb30060 afb2005c afb10058 afb00054 
	8fa20088 8fb100a0 8fb200a8 afa40038 afa6003c 8fa30080 afa30040 afa20044 
	8fa30090 afa30048 8fa30098 0046102a 10400009 afa3004c 27a4003c 27a50044 
	0411FCE8 00000000 27a40038 27a50040 0411FCE4 00000000 8fa2004c 8fa30044 
	0043102a 1040000a 8fa20044 27a4004c 27a50044 0411FCDB 00000000 27a40048 
	27a50040 0411FCD7 00000000 8fa20044 8fa3003c 0043102a 1040000a 8fb0003c 
	27a4003c 27a50044 0411FCCE 00000000 27a40038 27a50040 0411FCCA 00000000 
	8fb0003c 8fa2004c 1602002d 8fb30044 8fa20038 afa20030 afa20034 8fa30040 
	0062202a 10800003 0043102a 10000003 afa30030 54400001 afa30034 8fa20048 
	8fa30030 0043182a 10600003 8fa30034 10000004 afa20030 0062182a 54600001 
	afa20034 8fa70030 8fa2003c afa20010 000217c3 afa20014 24020001 00001821 
	afa20018 afa3001c 8fa20034 24420001 00471023 afa20020 000217c3 afa20024 
	afb20028 02202021 00e03021 00073fc3 0411FB35 00000000 10000077 8fbf006c 
	00531026 0002102b 02629823 0270102a 14400036 24140001 0000a821 8fa20038 
	8fa3003c 02032823 8fa60040 00c23023 70a63002 8fa40044 00832023 00c4001a 
	008001f4 00002012 00822021 afa40030 8fa60048 00c23023 70a62802 8fa6004c 
	00c31823 00a3001a 006001f4 00001812 00621021 afa20034 0044102a 10400006 
	8fa70030 27a40030 27a50034 0411FC75 00000000 8fa70030 afb00010 001017c3 
	afa20014 8fa20034 24420001 00471023 afa20018 000217c3 afa2001c afb40020 
	afb50024 afb20028 02202021 00e03021 00073fc3 0411FAFC 00000000 26100001 
	0270102a 1040ffce 8fa20038 8fa2004c 0050182a 14600037 24140001 0000a821 
	8fa60040 8fa40038 8fa50048 8fa80044 02083823 00a61823 70e33802 00484023 
	00e8001a 010001f4 00001812 00661821 afa30030 8fa7003c 02073023 00a42823 
	70c53002 00471023 00c2001a 004001f4 00001012 00442021 afa40034 0083202a 
	10800006 8fa70030 27a40030 27a50034 0411FC3C 00000000 8fa70030 afb00010 
	001017c3 afa20014 8fa20034 24420001 00471023 afa20018 000217c3 afa2001c 
	afb40020 afb50024 afb20028 02202021 00e03021 00073fc3 0411FAC3 00000000 
	26100001 8fa2004c 0050182a 1060ffcd 8fa60040 8fbf006c 8fb50068 8fb40064 
	8fb30060 8fb2005c 8fb10058 8fb00054 03e00008 27bd0070 

	'scroll
	27bdffb8 afbf0044 afbe0040 afb7003c afb60038 afb50034 afb40030 afb3002c 
	afb20028 afb10024 afb00020 0080f021 afa5004c 8fb70060 24b3ffff 02679821 
	2482ffff 00461021 afa20018 3c109d00 8e020028 0040f809 8ee40040 24110001 
	00518804 8e020024 8ee40040 0040f809 24050005 0040a821 8e020024 8ee40040 
	0040f809 24050006 0040a021 27d60001 8fa30018 0076102a 1440005b 8fa2004c 
	0053102a afa2001c 24120200 afa00010 afb70014 02c02021 02c02821 8fa6004c 
	02603821 0411F9FD 00000000 3c039d00 8c62001c 8ee40038 0040f809 24050006 
	8fa2001c 10400038 8fa4004c 8fb0005c 02001821 3c02bf88 3c05bf81 aeb10000 
	8c466210 8ca6f220 ae910000 8c466220 ac660000 aeb10000 8c466210 8ca6f220 
	ae910000 8c466220 ac660004 24840001 1493fff2 24630008 24030001 afa30010 
	afb70014 03c02021 03c02821 8fa6004c 02603821 0411F9D9 00000000 3c039d00 
	8c62001c 8ee40038 0040f809 24050006 8fa3004c 3c02bf88 8e040000 34840200 
	ac446230 ac526234 ac526238 8e040004 34840200 ac446230 ac526234 ac526238 
	24630001 1473fff4 26100008 26d60001 8fa30018 0076102a 1040ffbc 27de0001 
	10000012 8fbf0044 24020001 afa20010 afb70014 03c02021 03c02821 8fa6004c 
	02603821 0411F9B5 00000000 3c039d00 8c62001c 8ee40038 0040f809 24050006 
	1000ffeb 26d60001 8fbf0044 8fbe0040 8fb7003c 8fb60038 8fb50034 8fb40030 
	8fb3002c 8fb20028 8fb10024 8fb00020 03e00008 27bd0048 

	'main
	27bdffb8 afbf0044 afb00040 00808021 00e01021 8faa0058 8fa9005c 8fa80060 
	8fab0064 8c830000 8c840004 00643825 14e00015 8fac0068 8ca40000 8cc70000 
	8c420000 afa20010 000217c3 afa20014 24020001 00001821 afa20018 afa3001c 
	afa20020 afa30024 afaa0028 00e03021 00073fc3 0411FA08 00000000 00402021 
	100000e0 00602821 24070001 14670018 24070002 14800016 00000000 8ca40000 
	8cc70004 8cc60000 8c430004 8c420000 afa20010 afa30014 8d420000 8d430004 
	afa20018 afa3001c 8d220000 8d230004 afa20020 afa30024 afa80028 0411F9EE 
	00000000 00402021 100000c6 00602821 14670021 24070003 1480001f 00000000 
	8ca40000 8ca50004 8cc70004 8cc60000 8c430004 8c420000 afa20010 afa30014 
	8d420000 8d430004 afa20018 afa3001c 8d220000 8d230004 afa20020 afa30024 
	8d020000 8d030004 afa20028 afa3002c 8d620000 8d630004 afa20030 afa30034 
	afac0038 0411FE32 00000000 8e040000 100000a4 8e050004 1467000f 24070004 
	1480000d 00000000 8ca40000 8cc50000 8c460000 8d470000 8d220000 afa20010 
	afa80014 0411FB28 00000000 8e040000 10000094 8e050004 1467000a 24070005 
	14800008 00000000 8ca40000 00c02821 0411F906 00000000 8e040000 10000089 
	8e050004 1467000a 24070007 14800008 00000000 8ca40000 00c02821 0411F90E 
	00000000 8e040000 1000007e 8e050004 1467001e 24070008 1480001c 00000000 
	8ca40000 8ca50004 8cc70004 8cc60000 8c430004 8c420000 afa20010 afa30014 
	8d420000 8d430004 afa20018 afa3001c 8d220000 8d230004 afa20020 afa30024 
	8d020000 8d030004 afa20028 afa3002c afab0030 afac0034 0411FBE2 00000000 
	00402021 1000005f 00602821 1467001d 24070009 1480001b 00000000 8ca40000 
	8ca50004 8cc70004 8cc60000 8c430004 8c420000 afa20010 afa30014 8d420000 
	8d430004 afa20018 afa3001c 8d220000 8d230004 afa20020 afa30024 8d020000 
	8d030004 afa20028 afa3002c afab0030 0411FD3B 00000000 8e040000 10000041 
	8e050004 14670019 2407000a 14800017 00000000 8ca40000 8ca50004 8cc70004 
	8cc60000 8c430004 8c420000 afa20010 afa30014 8d420000 8d430004 afa20018 
	afa3001c 8d220000 8d230004 afa20020 afa30024 afa80028 0411FC03 00000000 
	8e040000 10000027 8e050004 14670015 2407000b 14800013 00000000 8ca40000 
	8cc50000 8c460000 8d470000 8d220000 afa20010 8d020000 afa20014 8d620000 
	afa20018 afac001c 8fa2006c afa20020 0411F9BA 00000000 00002021 10000011 
	00002821 14670012 8fbf0044 14800011 8fb00040 8ca40000 8cc50000 8c460000 
	8d470000 8d220000 afa20010 afa80014 afab0018 0411FE74 00000000 00002021 
	00002821 00801021 00a01821 8fbf0044 8fb00040 03e00008 27bd0048 
End CFunction   'MIPS32 M4K
'
sub SSD_test
  const scrollxstart=100, scrollxwidth=280,newpoint=scrollxstart+scrollxwidth-1
  const scrollystart=20,scrollyheight=232, scrolldirection=0
  local integer i, rp=130, bp=100, yp=160, wp=190, cp=70, xp=40, zp=220
  SSD.setrot(landscape)
  SSD.fillscreen(rgb(black))
  ssd.drect(scrollxstart-1,scrollystart-1,scrollxwidth+2,scrollyheight+2,rgb(red))
  ssd.pstring(0,scrollxstart+8,0,"Micromite MKII 4.7b8 with SSD1963",1,rgb(white),rgb(black))
  ssd.pstring(0,newpoint+10,scrollystart-13+(scrollyheight\2),"Fixed",2,rgb(green),rgb(black))
  ssd.pstring(0,newpoint+10,scrollystart+13+(scrollyheight\2),"Area",2,rgb(green),rgb(black))
  ssd.pstring(0,10,scrollystart-13+(scrollyheight\2),"Fixed",2,rgb(green),rgb(black))
  ssd.pstring(0,10,scrollystart+13+(scrollyheight\2),"Area",2,rgb(green),rgb(black))
  ssd.pstring(0,scrollxstart+12,scrollystart+scrollyheight+5,"Demonstration of scrolling data",1,rgb(white),rgb(black))

 do
   ' scroll an area defined by the x-y coordinates of the top left point
   ' the width of the area to be scrolled
   ' the height of the are to be scrolled
   ' the scroll driection (NOT YET IMPLEMENTED)
   ' a buffer used to store the data as it is read and written back to the display
   ' the array containing display specific information set up during initialisation
   i=ssddraw(ssd.scroll,scrollxstart,scrollystart,scrollxwidth,scrollyheight,scrolldirection,rbuf(),SSD_params())
   ssd.pixel(newpoint,xp,rgb(black)) ' clear the old data after the scroll
   ssd.pixel(newpoint,rp,rgb(black))
   ssd.pixel(newpoint,yp,rgb(black))
   ssd.pixel(newpoint,bp,rgb(black))
   ssd.pixel(newpoint,wp,rgb(black))
   ssd.pixel(newpoint,cp,rgb(black))
   ssd.pixel(newpoint,zp,rgb(black))
   rp=rp+(rnd()*4)-2 'get the new data points
   bp=bp+(rnd()*4)-2
   yp=yp+(rnd()*4)-2
   xp=xp+(rnd()*2)-1
   zp=zp+(rnd()*4)-2
   wp=wp+(rnd()*4)-2
   cp=cp+(rnd()*4)-2
   ssd.pixel(newpoint,xp,rgb(yellow)) 'write them to the space left
   ssd.pixel(newpoint,rp,rgb(blue))
   ssd.pixel(newpoint,yp,rgb(red))
   ssd.pixel(newpoint,bp,rgb(green))
   ssd.pixel(newpoint,wp,rgb(magenta))
   ssd.pixel(newpoint,cp,rgb(cyan))
   ssd.pixel(newpoint,zp,rgb(white))
loop

end SUB
