/*
 *
 * converts a PS/2 mouse to a HobbyTronics port and serial port
 * TassyJim December 2020
 */

#include "Wire.h" 
#include "PS2Mouse.h"

// PS2 mouse pins
#define DATA_PIN 2
#define CLOCK_PIN 3

#define I2C_PORT 41
#define SERIAL_BAUD 19200
#define HT_MODE 1    // 1 = HobbyTronics serial, 0 = condensed
#define DO_SERIAL 1  // 1 = serial enabled
#define POLL_TIME 60 // serial polling interval in mS
#define MOUSE_GAIN 6 // larger = less responsive mouse

// pin for switch to change between mouse and joystick
// high or open for mouse, ground for joystick
#define SEL_Pin 12

// joystick gain
#define J_GAIN 22
#define J_DEADZONE 8

// 1 = IBM, 2 = linear, 3 = atari
#define DEF_JOYSTICK 1

// joystick pins
#define axPin A0
#define ayPin A1
#define bxPin A2
#define byPin A3
#define ab1Pin 4
#define ab2Pin 5
#define bb1Pin 10
#define bb2Pin 11
#define axPinPD 9
#define ayPinPD 8
#define bxPinPD 7
#define byPinPD 6
#define LEDpin 13

int dx,dy,dw,stat,statold;  // used in PS/2 routines
int ab1, ab2, bb1, bb2;     // joystick buttons
long ax, ay, bx, by;        // joystick analog readings
long axf, ayf, bxf, byf;    // joystick filtered values
long midax = 512, miday = 512, midbx = 512, midby = 512; // joystick centre reading

long X,Y,LB,RB,SB,XS,YS, WS;
long Hmax = 800, Vmax = 600, ms = 6;
long Hstart = 400, Vstart = 300;
byte addr = 0;
byte ht_registers[0x40];
long HTS = HT_MODE;
unsigned long readStart, readTime; // the time the read started
volatile boolean receive_flag;
volatile boolean handle_flag;
byte in_mode = 0;
byte mouseOK;
byte initial_reads;

PS2Mouse mouse(CLOCK_PIN, DATA_PIN);

void setup() {
  pinMode(SEL_Pin, INPUT_PULLUP);
  pinMode(LEDpin, OUTPUT);
  digitalWrite(LEDpin, LOW);
  Serial.begin(SERIAL_BAUD);
  in_mode = !digitalRead(SEL_Pin) * DEF_JOYSTICK;
	ht_init();
  if (in_mode == 0) {
    readStart = millis();
    mouse.initialize();
    readTime = (millis() - readStart);
    mouseOK = checkMouse();
     Serial.print("mouseOK ");
     Serial.print(mouseOK);
     Serial.print(",");
     Serial.println(readTime);
  } else {
    joy_initialize();
  }
  readStart = millis();
}

// The I2C should interrupt every 16 mS
// if no I2C for POLL_TIME mS, assume that we want serial instead.

void loop() {
 if (handle_flag){
    // interrupt has occurred, position data is being sent
    // wait a little longer to make sure that the I2C has really finished sending
    delayMicroseconds(950);
    readStart = millis();
    digitalWrite(LEDpin, HIGH); // used for checking timing
    if (in_mode == 0) {
      getMouse();
    } else {
      getJoystick();
    }
    digitalWrite(LEDpin, LOW);
    handle_flag = false;
  }
if (receive_flag){
  receive_flag = false;
  delayMicroseconds(200);
  ht_configure();
}
// send serial every 60mS if I2C not active
 if (DO_SERIAL && (millis() - readStart) > POLL_TIME){
   readStart = millis();
   initial_reads = 0; // ignore I2C startup period
   digitalWrite(LEDpin, HIGH);
  if (in_mode == 0) {
    getMouse();
  } else {
    getJoystick();
  }
    doSerial();
    digitalWrite(LEDpin, LOW);
   }
}

// called whenever the CMM2 changes display mode 
// and has to reconfigure X and Y limits
 void ht_configure() {
 //  digitalWrite(LEDpin, HIGH); // used for checking timing
  Hmax = ht_registers[20]*256 + ht_registers[21] -1;
  Vmax = ht_registers[22]*256 + ht_registers[23] -1;
  Hstart = ht_registers[24]*256 + ht_registers[25];
  Vstart = ht_registers[26]*256 + ht_registers[27];
  ht_registers[0] = ht_registers[24];
  ht_registers[1] = ht_registers[25];
  ht_registers[2] = ht_registers[26];
  ht_registers[3] = ht_registers[27];
  ms = ht_registers[28]; 
  X = Hstart;
  Y = Vstart;
  WS = 0;
  initial_reads = 6; // we want to ignore the first 5 mouse readings
  // digitalWrite(LEDpin, LOW);
  // check mode switch to allow for hot switching between mouse and joystick
  // only during startup
 // in_mode = !digitalRead(SEL_Pin);
 // if (in_mode == 0) {
 //   mouse.initialize();
 // } else {
 //   joy_initialize();
 // }
 
  // printout for debug only
  /*
     Serial.print(Hmax);
    Serial.print(',');
    Serial.print(Vmax);
    Serial.print(',');
    Serial.print(Hstart);
    Serial.print(',');
    Serial.print(Vstart);
    Serial.print(',');
    Serial.println(ms);
   */
 }

 // getmouse should get called AFTER the last data set has been sent.
 // (This prevents timing issues)
void getMouse() {
  
   readStart = millis();
   MouseData data = mouse.readData();
   stat = data.status;
   dx = data.position.x;
   dy = data.position.y;
   dw = data.wheel;
  // convert data to suit HT
  if (initial_reads > 0){
    initial_reads--;
    X = Hstart;
    Y = Vstart;
    XS = 0;
    YS = 0;
    WS = 0;
    stat = 0;
  } else {
    XS = dx*ms/MOUSE_GAIN;
    X = X + XS;
    X = constrain(X,0,(Hmax));

    YS = -dy*ms/MOUSE_GAIN;
    Y = Y + YS;
    Y = constrain(Y,0,(Vmax));
  
   LB = stat & 1;          // left mouse button
   RB = (stat>>1) & 1;     // right mouse button
   SB = (stat>>2) & 1;     // middle mouse button

   WS = constrain(dw,-1,1);
  }
   ht_registers[0] = X >> 8;
   ht_registers[1] = X % 256;
   ht_registers[2] = Y >> 8;
   ht_registers[3] = Y % 256;
   ht_registers[4] = LB;
   ht_registers[5] = RB;
   ht_registers[6] = SB;
   ht_registers[7] = XS;
   ht_registers[8] = YS;
   ht_registers[9] = WS;

}

void doSerial() {
  // send data out serial port only if something has changed
  // there are two modes - HTS which sends in the HobbyTronics format
  // and Compressed which send 4 bytes
  // normally called after getmouse/getjoy when the I2C polling is not present.
  // not enough time to do both I2C and serial when polled at 16mS intervals.
    if(dx != 0 || dy != 0 || dw != 0 || statold != stat){
      statold = stat;
      if(HTS = 1){
        // send serial data in HobbyTronics format
     /* 
      Serial.print(dx);
      Serial.print(',');
      Serial.print(dy);
      Serial.print(',');
      Serial.print(dw);
      Serial.print(',');  
      */
    Serial.print(X);
    Serial.print(',');
    Serial.print(Y);
    Serial.print(',');
    Serial.print(LB);
    Serial.print(',');
    Serial.print(RB);
    Serial.print(',');
    Serial.print(SB);
    Serial.print(',');
    Serial.print(XS);
    Serial.print(',');
    Serial.print(YS);
    Serial.print(',');
    Serial.println(WS);
    }else{
      // send serial data in compressed form
    Serial.write(stat);
    Serial.write(dx);
    Serial.write(dy);
    Serial.write(dw); 
        }
    }
}
void ht_set_byte(int index, byte value) {
  ht_registers[index] = value;
}

// called whenever data is ready in the I2C stream
 void receive_bytes(int count) {
  if (count == 1) {
    addr = Wire.read();
    //  Serial.print(addr);
    //  Serial.print(',');
  } else if (count > 1) {
    addr = Wire.read();
    //  Serial.print(addr);
    //  Serial.print(',');
    byte curr = addr;
    for (int i = 1; i < count; i++) {
      byte d = Wire.read();
      ht_registers[curr++] = d;
      //    Serial.print(d);
      //  Serial.print(',');
      }
      //    Serial.println("done");
      if(curr > 36){
        //    in_mode = ht_registers[36];
      Serial.println(ht_registers[36]);
      }
      else if(curr > 28){
        receive_flag = true;
     // Serial.print("setup");
   //   ht_configure();
    }
    }
}

void send_data(uint8_t* data, uint8_t size) {
    Wire.write(data, size);
}

// called whenever a request for data has been received
void handle_request() {
  send_data(ht_registers + addr, 12);
  //  set flag to initiate the mouse read ready for next call
  handle_flag = true;
  }

  // init is called once on first startup
void ht_init() {
  X = Hstart;
  Y = Vstart;
    ht_registers[0] = X >> 8;
    ht_registers[1] = X % 256;
    ht_registers[2] = Y >> 8;
    ht_registers[3] = Y % 256;
    ht_registers[4] = LB;
    ht_registers[5] = RB;
    ht_registers[6] = SB;
    ht_registers[7] = XS;
    ht_registers[8] = YS;
    ht_registers[9] = WS;
 
  // Join I2C bus
  Wire.begin(41);
  // Serial.println("INIT");
  Wire.onReceive(receive_bytes);
  Wire.onRequest(handle_request);
}
// set up the joystick
void joy_initialize() {
  in_mode = checkJoy();
  // Serial.print("Joy ");
  // Serial.println(in_mode);
  // setup buttons
  pinMode(ab1Pin, INPUT_PULLUP);
  pinMode(ab2Pin, INPUT_PULLUP);
  pinMode(bb1Pin, INPUT_PULLUP);
  pinMode(bb2Pin, INPUT_PULLUP);
  if (in_mode == 1) {
  // switch the 10k resistors to ground for IBM style joystick
    pinMode(axPinPD, OUTPUT);
    pinMode(ayPinPD, OUTPUT);
    pinMode(bxPinPD, OUTPUT);
    pinMode(byPinPD, OUTPUT);
    digitalWrite(axPinPD, LOW);
    digitalWrite(ayPinPD, LOW);
    digitalWrite(bxPinPD, LOW);
    digitalWrite(byPinPD, LOW);
  } else if (in_mode == 2) {
  // leave the 10k resistors floating for linear style joystick
    pinMode(axPinPD, INPUT);
    pinMode(ayPinPD, INPUT);
    pinMode(bxPinPD, INPUT);
    pinMode(byPinPD, INPUT);
    delay(10);
    midax = analogRead(axPin);
    miday = analogRead(ayPin);
    midbx = analogRead(bxPin);
    midby = analogRead(byPin);
  } else if (in_mode == 3) {
    // use the 10k resistors as pullups for Atari style joystick
    pinMode(axPinPD, OUTPUT);
    pinMode(ayPinPD, OUTPUT);
    pinMode(bxPinPD, OUTPUT);
    pinMode(byPinPD, OUTPUT);
    digitalWrite(axPinPD, HIGH);
    digitalWrite(ayPinPD, HIGH);
    digitalWrite(bxPinPD, HIGH);
    digitalWrite(byPinPD, HIGH);
  }
}

void getJoystick() {
//  Serial.println("joystick");
  ax = analogRead(axPin);
  ay = analogRead(ayPin);
  bx = analogRead(bxPin);
  by = analogRead(byPin);
  if (in_mode == 1) {
    // IBM style
  axf = (1024 * J_GAIN)/ax - J_GAIN - 128;
  ayf = (1024 * J_GAIN)/ay - J_GAIN - 128;
  bxf = (1024 * J_GAIN)/bx - J_GAIN - 128;
  byf = (1024 * J_GAIN)/by - J_GAIN - 128;
  if (bxf < -132) {
    bxf = 0;
  }
  if (byf < -132) {
    byf = 0;
  }
  dx = axf;
  dy = ayf;
  dw = byf;
  } else if (in_mode == 2) {
    // linear potentometer
    axf = ax - midax;
    ayf = ay - miday;
    bxf = by - midbx;
    byf = by - midby;
    dx = axf;
    dy = ayf;
    dw = byf;
  } else if (in_mode == 3) {
    // digital style                            
    if (ax < 200) {
    axf = 48;
    } else if (ay < 200){
      axf = -48;
    } else {
      axf = 0;
    }
    if (bx < 200) {
    ayf = 48;
    } else if (by < 200){
      ayf = -48;
    } else {
      ayf = 0;
    }    
    bxf = 0;
    dx = axf;
    dy = ayf;
    dw = 0;                
  }
  // set a dead zone to reduce slow drift
  if (abs(axf) < J_DEADZONE) {
    axf = 0;
  }
  if (abs(ayf) < J_DEADZONE) {
    ayf = 0;
  }
  if (abs(bxf) < J_DEADZONE) {
    bxf = 0;
  }
  if (abs(byf) < J_DEADZONE) {
    byf = 0;
  }  
  ab1 = digitalRead(ab1Pin);
  ab2 = digitalRead(ab2Pin);
  bb1 = digitalRead(bb1Pin);
  bb2 = digitalRead(bb2Pin);

   // convert data to suit HT
  if (initial_reads > 0){
    initial_reads--;
    X = Hstart;
    Y = Vstart;
    XS = 0;
    YS = 0;
    WS = 0;
    stat = 0;
  } else {
    XS = axf/ms;
    X = X + XS;
    X = constrain(X,0,(Hmax));

   YS = ayf/ms;
   Y = Y + YS;
   Y = constrain(Y,0,(Vmax));
  
   LB = 1 - ab1;     // left mouse button
   RB = 1 - ab2;     // right mouse button
   SB = 1 - bb1;     // middle mouse button

   WS = constrain((byf >> 7),-1,1);
  }
   ht_registers[0] = X >> 8;
   ht_registers[1] = X % 256;
   ht_registers[2] = Y >> 8;
   ht_registers[3] = Y % 256;
   ht_registers[4] = LB;
   ht_registers[5] = RB;
   ht_registers[6] = SB;
   ht_registers[7] = XS;
   ht_registers[8] = YS;
   ht_registers[9] = WS;

/*
Serial.print(ax);
    Serial.print(',');
    Serial.print(axf);
    Serial.print(',');
    Serial.print(ay);
    Serial.print(',');
    Serial.print(ayf);
    Serial.print(',');
  Serial.print(bx);
    Serial.print(',');
    Serial.print(bxf);
    Serial.print(',');
    Serial.print(by);
    Serial.print(',');
    Serial.println(byf);
    */
}

// try and determine which type of joystick is attached.
byte checkJoy() {
  long joyTestH, joyTestL, joyTestOC;
  byte test_mode;
  pinMode(axPinPD, OUTPUT);
  digitalWrite(axPinPD, HIGH);
  delay(10);
  joyTestH = analogRead(axPin);
  digitalWrite(axPinPD, LOW);
  delay(10);
  joyTestL = analogRead(axPin);
  pinMode(axPinPD, INPUT);
  delay(10);
  joyTestOC = analogRead(axPin);
  // Serial.print(joyTestH);
  // Serial.print(',');
  // Serial.print(joyTestL);
  // Serial.print(',');
  // Serial.println(joyTestOC);
  
  test_mode = DEF_JOYSTICK;
  if ((joyTestH > 900) && (joyTestL < 30)) { // Opencircuit = digital joystick
    test_mode = 3;
  } 
  else if ((joyTestH > 900) && (joyTestOC > 900) && (joyTestL > 30)) { // IBM style joystick
    test_mode = 1;
  } 
  else if ((joyTestH < 900) && (joyTestL > 100) && (joyTestOC > 400) && (joyTestOC < 700)) { // linear style joystick
    test_mode = 2;
  }
  return test_mode;
}


byte checkMouse() {
  unsigned long duration;
    pinMode(CLOCK_PIN, INPUT);
    digitalWrite(CLOCK_PIN, HIGH);
    pinMode(DATA_PIN, INPUT);
    digitalWrite(DATA_PIN, HIGH);
    delay(10);
    pinMode(CLOCK_PIN, OUTPUT);
    digitalWrite(CLOCK_PIN, LOW);
    delayMicroseconds(150);
    pinMode(DATA_PIN, OUTPUT);
    digitalWrite(DATA_PIN, LOW);
    delayMicroseconds(15);
    pinMode(CLOCK_PIN, INPUT);
    digitalWrite(CLOCK_PIN, HIGH);
    delayMicroseconds(50);
    duration = pulseIn(CLOCK_PIN, LOW);
    return min(duration, 255);
}
