'MMEDIT!!! Basic Version = Micromite_X_V5.3
'MMEDIT!!! Port = COM4:38400:10,300
'MMEDIT!!! Device = Micromite_X_V5.3
'MMEDIT!!! Config = 100111101121010110020000110001
  ' lightning detector code translated from Arduino code
  ' by Jim Hiley tassyjim on TBS forum
  '
  '*   SEN-39001-R01 (universal applications)
  '*   ---> http://www.playingwithfusion.com/productview.php?pdid=22
  '*
  '* Copyright © 2015 Playing With Fusion, Inc.
  '* SOFTWARE LICENSE AGREEMENT: This code is released under the MIT License.
  '*
  '* Permission is hereby granted, free of charge, to any person obtaining a
  '* copy of this software and associated documentation files (the "Software"),
  '* to deal in the Software without restriction, including without limitation
  '* the rights to use, copy, modify, merge, publish, distribute, sublicense,
  '* and/or sell copies of the Software, and to permit persons to whom the
  '* Software is furnished to do so, subject to the following conditions:
  '*
  '* The above copyright notice and this permission notice shall be included in
  '* all copies or substantial portions of the Software.
  '*
  '* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
  '* IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  '* FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  '* AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  '* LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
  '* FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
  '* DEALINGS IN THE SOFTWARE.
  '* **************************************************************************
  '* REVISION HISTORY:
  '* Author		Date		Comments
  '* J. Steinlage		2015Jul20       I2C release based on SPI example
  '*
  '* Playing With Fusion, Inc. invests time and resources developing open-source
  '* code. Please support Playing With Fusion and continued open-source
  '* development by buying products from Playing With Fusion!
  '*
  '* **************************************************************************
  '* APPLICATION SPECIFIC NOTES (READ THIS!!!):
  '* - This file configures then runs a program on a micromite to interface with
  '*   an AS3935 Franklin Lightning Sensor manufactured by AMS.
  '*    - Perform setup for AS3935 chip
  '*      --- capacitance registers for tuning (based on cal value provided)
  '*      --- configurations for your application specifics (indoor/outdoor, etc)
  '*    - read status/info from sensor
  '*    - Write formatted information to serial port
  '* - Set configs for your specific needs using the #defines for wiring, and
  '*   review the setup() function for other settings (indoor/outdoor, for example)
  '* - I2C specific note: This example uses the I2C interface via the I2C lib, not
  '*   the 'Wire' lib included with the Arduino IDE.
  '*
  '* Circuit:
  '     I2C             SPI         -->  SEN-39001: AS3935 Breakout
  '*    SDA             MOSI        -->  MOSI/SDA
  '      nc             MISO        -->  MISO
  '*    SCK             SCK         -->  SCK/SCL
  '*    3.3V            GND         -->  SI  select interface; GND=SPI, VDD=I2C
  '*    IRQ_PIN         IRQ_PIN     -->  IRQ use a count pin for reading freq when calibrating
  '*    GND             CS_PIN      -->  CS  pull CS to ground for I2C
  '*    GND             GND         -->  GND
  '*    3.3V            3.3V        -->  Micromite I/O is at 3.3V, so power board from 3.3V
  '**************************************************************************/
  OPTION EXPLICIT
  'option default none
  DIM AS INTEGER I2Cmode              = 0   ' 1 = I2C  0 = SPI
  ' defines for hardware config
  DIM AS INTEGER CS_PIN               = 4        ' CS used for SPI mode, GND for I2C
  DIM AS INTEGER IRQ_PIN              = 6        ' any suitable pin for interrupt
  DIM AS INTEGER AS3935_ADD           = &H03     ' x03 - standard PWF SEN-39001-R01 config
  DIM AS INTEGER AS3935_CAPACITANCE   = 112       ' <-- SET THIS VALUE TO THE NUMBER LISTED ON YOUR BOARD
  SETPIN CS_PIN, DOUT
  PIN(CS_PIN) = 1-I2Cmode
  
  ' defines for general chip settings
  'dim as integer AS3935_INDOORS       = 0
  DIM AS INTEGER AS3935_OUTDOORS      = 0        ' 1 = outdoors 0 = indoor
  'dim as integer AS3935_DIST_DIS      = 0
  DIM AS INTEGER AS3935_DIST_EN       = 1
  DIM AS INTEGER re, dp, AS3935_ISR_Trig
  dim newtune, k$
  AS3935_setup
  SETPIN IRQ_PIN, INTH, AS3935_ISR
  '
  DO
    ' This program only handles an AS3935 lightning sensor. It does nothing until
    ' an interrupt is detected on the IRQ pin.
    IF AS3935_ISR_Trig = 1 THEN ' lightning sensor triggered
      checkInt
    endif
    pause 5
    k$ = inkey$
    if ucase$(k$) = "C" then setcap
  LOOP
  
END
  
sub checkInt
  local re, int_src, lightning_dist_km
  PAUSE 5
  ' reset interrupt flag
  AS3935_ISR_Trig = 0
  ' now get interrupt source
  int_src = AS3935_GetInterruptSrc()
  'print date$;" ";time$;" ";
  SELECT CASE int_src
    CASE 0
      if dp>0 then :dp = 0 : print "": endif
      print date$;" ";time$;" ";
      PRINT "interrupt source result not expected"
    CASE 1
      if dp>0 then :dp = 0 : print "": endif
      print date$;" ";time$;" ";
      lightning_dist_km = AS3935_GetLightningDistKm()
      PRINT "Lightning detected! Distance to strike: ";
      PRINT lightning_dist_km;
      PRINT " kilometers"
    CASE 2
      'print date$;" ";time$;" ";
      PRINT "D"; '"Disturber detected"
      dp = dp + 1
      if dp>60 then
        print ""
        dp=0
      endif
    CASE 3
      if dp>0 then :dp = 0 : print "": endif
      print date$;" ";time$;" ";
      PRINT "Noise level too high"
    CASE ELSE
      if dp>0 then :dp = 0 : print "": endif
      print date$;" ";time$;" ";
      PRINT "OOPS!! The module shouldn't return  "+STR$(int_src)
  END SELECT
  're = AS3935_PrintAllRegs() ' for debug...
end sub
  
  ' check tuning
sub setcap
  setpin IRQ_PIN, fin
  re = AS3935_SetIRQ_Output_Source(3)
  re = AS3935_CAPACITANCE/8
  do
    pause 2000
    print "Frequency: ";str$(pin(IRQ_PIN)*16)
    input "New tuning capacitor (1 - 120pF)? ", newtune
    if newtune > 0 and newtune <121 then
      re = AS3935_SetTuningCaps(newtune)
      print "Capacitor set to ";str$(re*8);"pF"
      print ""
    endif
    
  loop until newtune <0
  AS3935_CAPACITANCE = re*8
  re = AS3935_SetIRQ_Output_Source(0) ' turn the osc. output off again
  pause 1000
  AS3935_setup
end sub
  
  
SUB AS3935_setup
  PRINT "Playing With Fusion: AS3935 Lightning Sensor, SEN-39001-R01"
  PRINT "beginning boot procedure...."
  re = AS3935_DefInit()   ' set registers to default
  ' now update sensor cal for your application and power up chip
  re = AS3935_ManualCal(AS3935_CAPACITANCE, AS3935_OUTDOORS, AS3935_DIST_EN)
  ' AS3935_ManualCal Parameters:
  '   --> capacitance, in pF (marked on package)
  '   --> indoors/outdoors (AS3935_INDOORS:0 / AS3935_OUTDOORS:1)
  '   --> disturbers (AS3935_DIST_EN:1 / AS3935_DIST_DIS:2)
  ' function also powers up the chip
  re = AS3935_PrintAllRegs()
  AS3935_ISR_Trig = 0           ' clear trigger
END SUB
  
  
SUB AS3935_ISR
  ' this is irq handler for AS3935 interrupts,
  ' always make code in interrupt handlers fast and short
  AS3935_ISR_Trig = 1
END SUB
  
FUNCTION AS3935_reg_read(RegAdd)
  LOCAL RegData
  IF I2Cmode = 1 THEN
    I2C OPEN 400, 1000
    I2C WRITE AS3935_ADD, 1, 1, RegAdd
    PAUSE 1
    I2C READ AS3935_ADD, 1, 1, RegData
    I2C CLOSE
  ELSE
    SPI OPEN 1000000, 1, 8
    PIN(CS_PIN) = 0
    PAUSE 1
    RegData = SPI((RegAdd AND &H3F) OR &H40)
    RegData = SPI(&H00)
    PIN(CS_PIN) = 1
    SPI CLOSE
  ENDIF
  AS3935_reg_read = RegData
END FUNCTION
  
FUNCTION AS3935_reg_write(RegAdd, DataMask, RegData)
  LOCAL OrigRegData, NewRegData, re
  ' start by reading original register data (only modifying what we need to)
  OrigRegData = AS3935_reg_read(RegAdd)
  ' calculate new register data... 'delete' old targeted data, replace with new data
  ' note: 'DataMask' must be bits targeted for replacement
  ' add'l note: this function does NOT shift values into the proper place... they need to be there already
  NewRegData = ((OrigRegData AND (&HFF XOR DataMask)) OR (RegData AND DataMask))
  ' finally, write the data to the register
  IF I2Cmode = 1 THEN
    I2C OPEN 400, 1000
    PAUSE 1
    I2C WRITE AS3935_ADD, 1, 2, RegAdd, NewRegData
    AS3935_reg_write = MM.I2C
    I2C CLOSE
  ELSE
    SPI OPEN 1000000, 1, 8
    PIN(CS_PIN) = 0
    PAUSE 1
    SPI WRITE 2, (RegAdd AND &H3F),NewRegData
    PIN(CS_PIN) = 1
    SPI CLOSE
  ENDIF
  PRINT "wrt: ";HEX$(NewRegData,2);" Act: ";HEX$(AS3935_reg_read(RegAdd),2)' for debugging
END FUNCTION
  
FUNCTION AS3935_DefInit()
  LOCAL re
  ' run PRESET_DEFAULT Direct Command to set all registers in default state
  IF I2Cmode = 1 THEN
    I2C OPEN 400, 1000
    PAUSE 1
    I2C WRITE AS3935_ADD, 1, 2, &H3C, &H96
    AS3935_DefInit = MM.I2C
    I2C CLOSE
  ELSE
    SPI OPEN 1000000, 1, 8
    PIN(CS_PIN) = 0
    PAUSE 1
    SPI WRITE 2, &H3C,&H96
    PIN(CS_PIN) = 1
    SPI CLOSE
  ENDIF
  PAUSE 2					' wait 2ms to complete
END FUNCTION
  
FUNCTION CalRCO()
  ' run ALIB_RCO Direct Command to cal internal RCO
  IF I2Cmode = 1 THEN
    I2C OPEN 400, 1000
    PAUSE 1
    I2C WRITE AS3935_ADD, 1, 2, &H3D, &H96
    CalRCO = MM.I2C
    I2C CLOSE
  ELSE
    SPI OPEN 1000000, 1, 8
    PIN(CS_PIN) = 0
    PAUSE 1
    SPI WRITE 2, &H3D,&H96
    PIN(CS_PIN) = 1
    SPI CLOSE
  ENDIF
  PAUSE 2					' wait 2ms to complete
END FUNCTION
  
FUNCTION AS3935_PowerUp()
  LOCAL re
  ' power-up sequence based on datasheet, pg 23/27
  ' register 0x00, PWD bit: 0 (clears PWD)
  re = AS3935_reg_write(&H00, &H01, &H00)
  re = CalRCO()							              ' run RCO cal cmd
  re = AS3935_reg_write(&H08, &H20, &H20)	' set DISP_SRCO to 1
  PAUSE 2
  re = AS3935_reg_write(&H08, &H20, &H00)	' set DISP_SRCO to 0
  AS3935_PowerUp = 1
END FUNCTION
  
FUNCTION AS3935_PowerDown()
  LOCAL re
  ' register 0x00, PWD bit: 0 (sets PWD)
  re = AS3935_reg_write(&H00, &H01, &H01)
  PRINT "AS3935 powered down"
  AS3935_PowerDown = re
END FUNCTION
  
FUNCTION AS3935_DisturberEn()
  LOCAL re
  ' register 0x03, PWD bit: 5 (resets MASK_DIST)
  re = AS3935_reg_write(&H03, &H20, &H00)
  PRINT "disturber detection enabled"
  AS3935_DisturberEn = re
END FUNCTION
  
FUNCTION AS3935_DisturberDis()
  LOCAL re
  ' register 0x03, PWD bit: 5 (sets MASK_DIST)
  re = AS3935_reg_write(&H03, &H20, &H20)
  PRINT "disturber detection disabled"
  AS3935_DisturberDis = re
END FUNCTION
  
FUNCTION AS3935_SetIRQ_Output_Source(irq_select)
  LOCAL re
  ' set interrupt source - what to displlay on IRQ pin
  ' reg 0x08, bits 5 (TRCO), 6 (SRCO), 7 (LCO)
  ' only one should be set at once, I think
  ' 0 = NONE, 1 = TRCO, 2 = SRCO, 3 = LCO
  SELECT CASE irq_select
    CASE 1
      re = AS3935_reg_write(&H08, &HE0, &H20)			' set only TRCO bit
    CASE 2
      re = AS3935_reg_write(&H08, &HE0, &H40)			' set only SRCO bit
    CASE 3
      re =AS3935_reg_write(&H08, &HE0, &H80)			' set only LCO bit
    CASE ELSE ' assume 0
      re =AS3935_reg_write(&H08, &HE0, &H00)			' clear IRQ pin display bits
  END SELECT
  AS3935_SetIRQ_Output_Source = re
END FUNCTION
  
FUNCTION AS3935_SetTuningCaps(cap_val)
  LOCAL re
  ' Assume only numbers divisible by 8 (because that's all the chip supports)
  IF cap_val >= 120 THEN	' cap_value out of range, assume highest capacitance
    re =AS3935_reg_write(&H08, &H0F, &H0F)			' set capacitance bits to maximum
  ELSE
    re =AS3935_reg_write(&H08, &H0F, (cap_val>>3))	' set capacitance bits
  ENDIF
  re = AS3935_reg_read(&H08) AND &H0F
  PRINT "capacitance set to 8x";STR$(re);" = ";STR$(re * 8);"pF"
  AS3935_SetTuningCaps = re
END FUNCTION
  
FUNCTION AS3935_GetInterruptSrc()
  LOCAL re, int_src
  ' definition of interrupt data on table 18 of datasheet
  ' for this function:
  ' 0 = unknown src, 1 = lightning detected, 2 = disturber, 3 = Noise level too high
  PAUSE 3						' wait 3ms before reading (min 2ms per pg 22 of datasheet)
  int_src = (AS3935_reg_read(&H03) AND &H0F)	' read register, get rid of non-interrupt data
  SELECT CASE int_src
    CASE &H08
      re =  1					' lightning caused interrupt
    CASE &H04
      re = 2					' disturber detected
    CASE &H01
      re = 3					' Noise level too high
    CASE ELSE
      re = 0					' interrupt cause unknown
  END SELECT
  AS3935_GetInterruptSrc = re
END FUNCTION
  
FUNCTION AS3935_GetLightningDistKm()
  LOCAL strike_dist
  strike_dist = (AS3935_reg_read(&H07) AND &H3F)	' read register, get rid of non-distance data
  AS3935_GetLightningDistKm = strike_dist         ' 63 equals out of range
END FUNCTION
  
FUNCTION AS3935_GetStrikeEnergyRaw()
  LOCAL energy_raw
  energy_raw = (AS3935_reg_read(&H06) AND &H1F) << 8	' MMSB, shift 8  bits left, make room for MSB
  energy_raw = energy_raw OR AS3935_reg_read(&H05)		' read MSB
  energy_raw = energy_raw << 8												' shift 8 bits left, make room for LSB
  energy_raw = energy_raw OR AS3935_reg_read(&H04)		' read LSB, add to others
  AS3935_GetStrikeEnergyRaw = energy_raw
END FUNCTION
  
FUNCTION AS3935_SetMinStrikes(min_strk)
  LOCAL re
  ' This function sets min strikes to the closest available number, rounding to the floor,
  ' where necessary, then returns the physical value that was set. Options are 1, 5, 9 or 16 strikes.
  ' see pg 22 of the datasheet for more info (#strikes in 17 min)
  SELECT CASE min_strk
    CASE 1
      re = AS3935_reg_write(&H02, &H30, &H00)
      re = 1
    CASE min_strk < 9
      re = AS3935_reg_write(&H02, &H30, &H10)
      re = 5
    CASE min_strk < 16
      re = AS3935_reg_write(&H02, &H30, &H20)
      re = 9
    CASE ELSE
      re = AS3935_reg_write(&H02, &H30, &H30)
      re = 16
  END SELECT
  AS3935_SetMinStrikes = re
END FUNCTION
  
FUNCTION AS3935_SetIndoors()
  ' AFE settings addres 0x00, bits 5:1 (10010, based on datasheet, pg 19, table 15)
  ' this is the default setting at power-up (AS3935 datasheet, table 9)
  AS3935_SetIndoors = AS3935_reg_write(&H00, &H3E, &H24)
  PRINT "set up for indoor operation"
END FUNCTION
  
FUNCTION AS3935_SetOutdoors()
  ' AFE settings addres 0x00, bits 5:1 (01110, based on datasheet, pg 19, table 15)
  AS3935_SetOutdoors = AS3935_reg_write(&H00, &H3E, &H1C)
  PRINT "set up for outdoor operation"
END FUNCTION
  
FUNCTION AS3935_ClearStatistics()
  LOCAL re
  ' clear is accomplished by toggling CL_STAT bit 'high-low-high' (then set low to move on)
  re = AS3935_reg_write(&H02, &H40, &H40)			' high
  re = AS3935_reg_write(&H02, &H40, &H00)			' low
  re = AS3935_reg_write(&H02, &H40, &H40)			' high
  AS3935_ClearStatistics = 1
END FUNCTION
  
FUNCTION AS3935_GetNoiseFloorLvl()
  LOCAL reg_raw
  ' NF settings addres 0x01, bits 6:4
  ' default setting of 010 at startup (datasheet, table 9)
  reg_raw = AS3935_reg_read(&H01)		' read register 0x01
  AS3935_GetNoiseFloorLvl = (reg_raw AND &H70)>>4				' should return value from 0-7, see table 16 for info
END FUNCTION
  
FUNCTION AS3935_SetNoiseFloorLvl(nf_sel)
  ' NF settings addres 0x01, bits 6:4
  ' default setting of 010 at startup (datasheet, table 9)
  IF nf_sel <= 7 THEN								' nf_sel within expected range
    AS3935_SetNoiseFloorLvl = AS3935_reg_write(&H01, &H70, ((nf_sel AND &H07)<<4))
  ELSE                              ' out of range, set to default (power-up value 010)
    AS3935_SetNoiseFloorLvl = AS3935_reg_write(&H01, &H70, &H20)
  ENDIF
END FUNCTION
  
FUNCTION AS3935_GetWatchdogThreshold()
  LOCAL reg_raw
  ' This function is used to read WDTH. It is used to increase robustness to disturbers,
  ' though will make detection less efficient (see page 19, Fig 20 of datasheet)
  ' WDTH register: add 0x01, bits 3:0
  ' default value of 0001
  ' values should only be between 0x00 and 0x0F (0 and 7)
  reg_raw = AS3935_reg_read(&H01)
  AS3935_GetWatchdogThreshold = reg_raw AND &H0F
END FUNCTION
  
FUNCTION AS3935_SetWatchdogThreshold(wdth)
  ' This function is used to modify WDTH. It is used to increase robustness to disturbers,
  ' though will make detection less efficient (see page 19, Fig 20 of datasheet)
  ' WDTH register: add 0x01, bits 3:0
  ' default value of 0001
  ' values should only be between 0x00 and 0x0F (0 and 7)
  AS3935_SetWatchdogThreshold = AS3935_reg_write(&H01, &H0F, (wdth AND &H0F))
END FUNCTION
  
FUNCTION AS3935_GetSpikeRejection()
  LOCAL reg_raw
  ' This function is used to read SREJ (spike rejection). Similar to the Watchdog threshold,
  ' it is used to make the system more robust to disturbers, though will make general detection
  ' less efficient (see page 20-21, especially Fig 21 of datasheet)
  ' SREJ register: add 0x02, bits 3:0
  ' default value of 0010
  ' values should only be between 0x00 and 0x0F (0 and 7)
  reg_raw = AS3935_reg_read(&H02)
  AS3935_GetSpikeRejection = reg_raw AND &H0F
END FUNCTION
  
  
FUNCTION AS3935_SetSpikeRejection(srej)
  ' This function is used to modify SREJ (spike rejection). Similar to the Watchdog threshold,
  ' it is used to make the system more robust to disturbers, though will make general detection
  ' less efficient (see page 20-21, especially Fig 21 of datasheet)
  ' WDTH register: add 0x02, bits 3:0
  ' default value of 0010
  ' values should only be between 0x00 and 0x0F (0 and 7)
  AS3935_SetSpikeRejection = AS3935_reg_write(&H02, &H0F, (srej & &H0F))
END FUNCTION
  
FUNCTION AS3935_SetLCO_FDIV(fdiv)
  ' This function sets LCO_FDIV register. This is useful in the tuning of the antenna
  ' LCO_FDIV register: add 0x03, bits 7:6
  ' default value: 00
  ' set 0, 1, 2 or 3 for ratios of 16, 32, 64 and 128, respectively.
  ' See pg 23, Table 20 for more info.
  AS3935_SetLCO_FDIV = AS3935_reg_write(&H03, &HC0, ((fdiv AND &H03) << 5))
END FUNCTION
  
FUNCTION AS3935_PrintAllRegs()
  'local energy_val
  PRINT ""
  PRINT TIME$;"  ";DATE$
  PRINT "Reg &H00: ";HEX$(AS3935_reg_read(&H00),2)
  PRINT "Reg &H01: ";HEX$(AS3935_reg_read(&H01),2)
  PRINT "Reg &H02: ";HEX$(AS3935_reg_read(&H02),2)
  PRINT "Reg &H03: ";HEX$(AS3935_reg_read(&H03),2)
  PRINT "Reg &H04: ";HEX$(AS3935_reg_read(&H04),2)
  PRINT "Reg &H05: ";HEX$(AS3935_reg_read(&H05),2)
  PRINT "Reg &H06: ";HEX$(AS3935_reg_read(&H06),2)
  PRINT "Reg &H07: ";HEX$(AS3935_reg_read(&H07),2)
  PRINT "Reg &H08: ";HEX$(AS3935_reg_read(&H08),2)
  '
  PRINT "AFE Gain Boost:       ";BIN$((AS3935_reg_read(&H00)AND &B00111110)>>1,5)
  PRINT "Power-down:           ";BIN$((AS3935_reg_read(&H00)AND &B00000001),1)
  PRINT "Noise Floor Level:    ";BIN$((AS3935_reg_read(&H01)AND &B01110000)>>4,3)
  PRINT "Watchdog threshold:   ";BIN$((AS3935_reg_read(&H01)AND &B00001111),4)
  PRINT "Clear statistics:     ";BIN$((AS3935_reg_read(&H02)AND &B01000000)>>6,1)
  PRINT "Minimum num lightning:";BIN$((AS3935_reg_read(&H02)AND &B00110000)>>4,2)
  PRINT "Spike rejection:      ";BIN$((AS3935_reg_read(&H02)AND &B00001111),4)
  PRINT "Freq div -ant tuning: ";BIN$((AS3935_reg_read(&H03)AND &B11000000)>>6,2)
  PRINT "Mask Disturber:       ";BIN$((AS3935_reg_read(&H03)AND &B00100000)>>5,1)
  PRINT "Interrupt:            ";BIN$((AS3935_reg_read(&H03)AND &B00001111),4)
  PRINT "Energy LSB:           ";BIN$(AS3935_reg_read(&H04)AND &B11111111,8)
  PRINT "Energy MSB:           ";BIN$(AS3935_reg_read(&H05)AND &B11111111,8)
  PRINT "Energy MMSB:          ";BIN$(AS3935_reg_read(&H06)AND &B00011111,5)
  PRINT "Distance estimation:  ";BIN$(AS3935_reg_read(&H07)AND &B00111111,6)
  PRINT "Display LCO on IRQ:   ";BIN$((AS3935_reg_read(&H08)AND &B10000000)>>7,1)
  PRINT "Display SRCO on IRQ:  ";BIN$((AS3935_reg_read(&H08)AND &B01000000)>>6,1)
  PRINT "Display TRCO on IRQ:  ";BIN$((AS3935_reg_read(&H08)AND &B00100000)>>5,1)
  PRINT "Internal Tuning Caps: ";BIN$(AS3935_reg_read(&H08)AND &B00001111,4)
  PRINT "Energy:   ";STR$(AS3935_GetStrikeEnergyRaw())
END FUNCTION
  
FUNCTION AS3935_ManualCal(capacitance, location, disturber)
  LOCAL re
  ' start by powering up
  re = AS3935_PowerUp()
  ' indoors/outdoors next...
  IF location = 1 THEN    ' set outdoors if 1
    re = AS3935_SetOutdoors()
  ELSE    ' set indoors if anything but 1
    re = AS3935_SetIndoors()
  ENDIF
  ' disturber cal
  IF disturber = 0 THEN							' disabled if 0
    re = AS3935_DisturberDis()
  ELSE										' enabled if anything but 0
    re = AS3935_DisturberEn()
  ENDIF
  re = AS3935_SetIRQ_Output_Source(0)
  
  PAUSE 500
  ' capacitance first... directly write value here
  re = AS3935_SetTuningCaps(capacitance)
  PRINT "AS3935 manual cal complete"
END FUNCTION
  
