'Multi function counter using RP2040 PIO
  
  
  
'Version control -------------------------------------------------------------------
'Version    Features
'  6       Single frequency measurement, posted on TBS in PIO training course
' 10       Dual frequency counter using PIO 1.1, 1.2, 1.3 and ratio calculation
' 11       Added Period measurement and Pulse measurement, on TBS
' 12       Code compression, PIO 1.0 and 1.1 share line 5 and 6. PIO config in SUB
' 13       Autoselect time/frequency with dynamic averaging in time
  
  
  
'todo ------------------------------------------------------------------------------
' adapt print statement to show only relevant info (MMBasic)
' make LCD GUI (develop on VGA picomite ?)
' develop hardware (input circuits using 74LVC132 or 74LVC14 and amplifier for AC-A)
' START/STOP on A/B function (MMBasic ? / PIO ?)
' Run pico at higher clock and adapt f0...f3 for more bandwidth/resolution
  
  
  
'PIO usage -------------------------------------------------------------------------
'
'PIO machine 1.0 measures period GP0 signal
'
'PIO machine 1.1 has 2 configurations
'   uses the GP2 gate signal to count GP1 pulses, GP4 is NO GP1 SIGNAL
'   -or-
'   measures PULSE width for time measurements (PAUSE = PERIOD-PULSE)
'
'PIO machine 1.2 generates a progr. gate signal on GP2 (side set pin GP2)
'    PIO machine 1.2 sets GP2,GP3,GP4 output for all state machines 1.1, 1.2 and 1.3
'
'PIO machine 1.3 uses the GP2 gate signal to count GP0 pulses, GP3 is NO GP0 SIGNAL
'
'PIO 1.0, 1.2 and 1.3 always run, to measure parameters of GP0 signal PIO 1.1 changes
'function depending on the choice of measurement.
  
  
  
  
'PIO program ---------------------------------------------------------------------
'
'PIO 1.0 (period)
'adress    data   mnemonics                         comment
'.wrap target PIO 1.0
' &h00     A02B    Mov X=-NULL                      'clear X
' &h01     0042    JMP (x<>0) x--, 2                'always go to next instruction, decrement x
' &h02     00C1    JMP (GP0=1), 1                   'high count loop
' &h03     00C5    JMP (GP0=1) &h05                 'jump out of the low count loop when GP0=1
' &h04     0043    JMP (X<>0) X--, &h03             'decrement X and always jump back to 3
'.wrap target PIO 1.1
' &h05     A0C9    MOV ISR = -X                     'X invert result -> ISR
' &h06     8000    Push noblock                     'ISR->FIFO
'.wrap PIO 1.0
'
'PIO 1.1 (pulse) runs this code -or- the code under PIO 1.3 depending the configuration
'adress    data   mnemonics                         comment
'.wrap target
' &h07     A02B    Mov X=-NULL                      'clear X
' &h08     2080    Wait until GP0=1                 'wait for pulse rising edge to start counting
' &h09     004A    JMP (x<>0) x--, &h0A             'always go to next instruction, decrement x
' &h0A     00C9    JMP (GP0=1), 2&h09               'high count loop
'.wrap PIO1.1
' MOV ISR=-X and PUSH (program code at &h05 and &h06) are shared between PIO 1.0 and 1.1
' dirty, but effectively saving 2 program locations. PIO 1.1 must start at address 7 !!
'
' &h0B
' -------- not used, 9 words free
' &h13
'
'PIO 1.2 (gate)
'adress    data   mnemonics                         comment
'  &h14    E087   set pindir 00111                  'set GP2,GP3,GP4 to output for both state machines
'.wrap target
'  &h15    90A0   pull block .side(1)               'pull data from FIFO into OSR when available
'  &h16    B027   mov(x,osr) .side(1)               'load gate time (in clock pulses) from osr
'  &h17    0057   jmp(x_dec, this line) .side(0)    '1 cycle count down + gate low
'.wrap
'
'PIO 1.3 (count) and also PIO 1.1 when ratio is chosen
'adress    data   mnemonics                         comment
'.wrap target
'  &h18    A02B   mov(x,-1) .side(0)                'x = -1 (clear counter)
'  &h19    00D9   jmp(gp2, to 0x19) .side(0)        'wait for gate pulse to become low
'  &h1A    3020   wait(0,gpx) .side(1)              'wait rising edge input, index+base IN
'  &h1B    30A0   wait(1,gpx) .side(1)              'wait falling edge input, index+base IN
'  &h1C    00DE   jmp(gp2, to 0x1E) .side(0)        'continue counting (gate pulse ended ?)
'  &h1D    005A   jmp(x_dec, to 0x1A) .side(0)      'yes, count next pulse input
'  &h1E    A0C9   mov(isr,-x) .side(0)              'copy counter X inverted to ISR
'  &h1F    8000   push noblock .side(0)             'get value to FIFO
'.wrap
  
'this is above program in data statements
  dim a%(7)=(&h00C500C10042A02B,&hA02B8000A0C90043,&h00C9004A2080,0,0,&h0057B02790A0E087,&h30A0302000D9A02B,&h8000A0C9005A00DE)
  
'program above code in the chip in PIO 1
  pio program 1,a%()
  
  
  
' PIO SETUP -------------------------------------------------------------------------
  
'SETUP code
  setpin gp0,pio1          'frequency counter input A
  setpin gp1,pio1          'frequency counter input B
  setpin gp2,pio1          'gate signal
  setpin gp3,pio1          'no signal A output used to display input signal missing
  setpin gp4,pio1          'no signal B output used to display input signal missing
  
  
  
  
  
  
'MMBASIC MAIN CODE --------------------------------------------------------------------------
  
  
'generate a test frequencies on GP5 and GP22, for testing of the counter
  setpin gp5,pwm
  pwm 2,50000,,50              'change 15000000 to test different frequencies.
'15MHz PLL generates 15.75MHz dues to divider settings inside the PLL
'setpin gp22,pwm
'pwm 3,1000,50
  setpin gp10,pwm 'my breadboard
  pwm 5,1000,50   'my breadboard
  
'define PIO 1.1 function
  function_pulse = 1               '1=pulse, 0=ratio
  
'config PIO
  configPio(function_pulse)
  
  
'variables and constants used in the counter
  dim r%(3), ca%, cb%, p%, pt%, m%, s%  'define variables, r%() only used to empty fifo
  dim per, pw, freq
  gate_time = 0.2                     'seconds
  gate_clocks% = gate_time*f2 - 1     'for 1MHz PIO 1.2 clock as integer (n counts (n-1...0))
  
  
'first empty the fifo's from rubbish
  pio read 1,1,4,r%()               'empty fifo 1.1 from rubbish (only needed at start)
  pio read 1,3,4,r%()               'empty fifo 1.3 from rubbish (only needed at start)
  
  
'this is the re-structured loop
  do
    
'put gate time in PIO 1.2 OSR (gate_time is specified in PIO cycles: gate_clocks%)
    pio write 1,2,1,gate_clocks%      'this starts the gate pulse generator, single pulse
    
'read the period time value from PIO 1.0
    pio read 1,0,1,pt%                       'read period from PIO 1.0 fifo
    pt%=pt%+2                                'correct for PIO cycles at adress 5 and 6
    per = 2*pt%/f0                           'estimate how many periods would fit in the gate time
    avg%=min(int(gate_time/per),1000)        'with a maximum of 1000
    
    for i%=2 to avg%                          'we take avg% samples (one is already in pt%
      pio read 1,0,1,p%                       'during the gate time
      pt%=pt%+p%+2                            'and sum them all up, including correction
      pause 1000*per                          'pause in ms
    next i%                                   'now we have in pt% the sum of avg% readings
    
'wait gate time + few ms for getting data after input signal cycle ends
    'pause (1000*gate_time + 10)
    pause 100
    
'reset the count variables
    ca%=0:cb%=0
    
'check if the input did not have input signal
    if pin(GP3)=0 then
      pio read 1,3,1,ca%                  'read value from FIFO (there is only 1 value in it)
      digits=autoselect (ca%,pt%,avg%)    'send both values to decision function
    end if
    
'depending freq ratio selected or not
    if function_pulse=0 then
      if pin(GP4)=0 then
        pio read 1,1,1,cb%                  'read value from FIFO (there is only 1 value in it)
        cb%=cb%/gate_time                   'convert or Hz
      end if
    else
      pio read 1,1,1,p%                     'read period from PIO 1.0 fifo
      pw = 2*(1e6/f1)*p%                    'calculate pulse width from reading FIFO
    end if
    
    
'print frequency value
    print "GP0 frequency   = ";freq;" Hz     "; digits; " relevant digits"; avg%; " averages"
    print "GP1 frequency   = ";cb%;" Hz"
    if cb%>0 then
      print "frequency ratio = ";freq/cb%
    end if
    print "GP0 period      = ";per*1e6;" us"
    if function_pulse=1 then
      print "GP0 high width  = ";pw;" us"
      print "GP0 low width   = ";(per*1e6)-pw;" us"
    end if
    
  loop
  
  
'subs -------------------------------------------------------------------------------
  
sub configPio (func)
  
'stop all PIO's
  pio stop 1,0: pio stop 1,1: pio stop 1,2: pio stop 1,3
  
'PIO state machine configurations
  
'PIO 1.0 (measure period from GP0)
  f0=63e6                               '63 MHz frequency
  e0=pio(execctrl 0,0,6)                'GP0 is PIN for cond jmp. wrap target 0, wrap 6
  p0=0                                  'default value
  pio init machine 1,0,f0,p0,e0,0,0     'start at 0
  
  if func=1 then
'PIO 1.1 (measure pulse width from GP0) in case pulse width is selected
    f1=63e6                               '63 MHz frequency
    e1=pio(execctrl 0,5,&h0A)             'GP0 is PIN for cond jmp. wrap target 7, wrap &h0C
    p1=0                                  'default value
    pio init machine 1,1,f1,p1,e1,0,7     'start at 7
  else
'PIO 1.1 (frequency B from GP1 in case frequency ratio is selected)
    f1=63e6                               '63 MHz frequency
    e1=pio(execctrl 2,&h18,&h1F)          'GP2 is PIN for cond jmp. wrap target &h18, wrap &h1F
    p1=Pio(pinctrl 1,,,gp1,gp4,,)         'GP1 base for inputs, GP4 out for SIDE SET
    pio init machine 1,1,f1,p1,e1,0,&h18  'start at &h18
  end if
  
'PIO 1.2 (gate pulse)
  f2=1e6                                '1 MHz frequency gate resolution
  e2=pio(execctrl 0,&h15,&h17)          'wrap target &h11, wrap &h13
  p2=Pio(pinctrl 1,3,,,gp2,gp2,)        'GP2 side set GATE puls, GP2/GP3/GP4 SET (for pindirs)
  pio init machine 1,2,f2,p2,e2,0,&h14  'start from 0x14
  
'PIO 1.3 (frequency A from GP0)
  f3=63e6                               '63 MHz frequency
  e3=pio(execctrl 2,&h18,&h1F)          'gGP2 is PIN for cond jmp. wrap target &h18, wrap &h1F
  p3=Pio(pinctrl 1,,,gp0,gp3,,)         'GP0 base for inputs, GP3 out for SIDE SET
  pio init machine 1,3,f3,p3,e3,0,&h18  'start at &h18
  
  
'Start all PIO sequencers
  PIO start 1,0 'this will start period measurement from GP0
  PIO start 1,1 'depending selection will start pulse measurement GP0 or frequency GP1
  PIO start 1,2 'this will wait for data to arrive in FIFO, then generate 1 gate
  PIO start 1,3 'this will start counter, waiting on adress &h17 for GP2 gate to become low
  
end sub
 
 
  
function autoselect(count%,time%,average%)
'the autoselect function determines what function (period/frequency) is used to give best resolution
'the return valyue is the number of relevant digits. Global variables per and ca% are also updated.
  
'select what value has most relevant digits by using a log function
'  res_f = log(count%)
'  res_t = log(time%)
' if res_f > res_t then
  
  if count% > 1e5 then
    freq = count%/gate_time                  'convert to Hz
    per = 1/freq                             'derive period from frequency
    autoselect = 1 + int(log(count%)/log(10))
  else
    per = 2*time%/(f0*average%)              'p% to time in s
    freq = 1/per                             'derive frequency from period (or from averaged period)
    autoselect = 1 + int(log(time%)/log(10))
  end if
end function
  
