'+----------------------------------------------------+
'|     logic analyzer framework research program      |
'|                                                    |
'| Volhout                                 25-4-2023  |
'+----------------------------------------------------+
'samples GP0...GP15 with triggering on GP22
'uses PIO and ring buffer under DMA.
'requires V50707RCx or newer (function names changed)


'---------------------- version control ----------------
' LA_3  uses DMA after manual trigger, serial output
' LA_4  uses DMA and ring buffer, serial output
' LA_5  testbed to debug DMA
' LA_6  restructure (this version) tbd PIO is ready
' LA_7  unroll circular buffer to linear array
' LA_8  triggering works
' LA_9  basic VGA display added
' LA_10 pan/scan added on cursor keys
' LA_11 8 chan, bar, sampling rate implemented +/-
' LA_12 6/8 channel, trigger overhaul.
' LA_13 single/run, set trigger by numeric keys
' LA_14 bug fixes, channel name edit
' LA_15 t changes trigger in packed buffer, bug fixes
' LA_16 m measures time
' LA_17 fix accuracy
' LA_20 add raw decoders from Pluto, debug version
' LA_21 I2C decoder implemented
' LA_21_b24 =LA_21 adapted for 8 pixel wide tiles in V50707b24
' LA_22 Adapted for the V5.07.07 release, functional identical to LA_21
' LA_23 Help text in 6 channel mode

'---------------------------keyboard control ------------------------

'   read keyboard
'   if "<" then move centre, display
'   if ">" then move centre, display
'   if "^" then zoom in, display
'   if "V" then zoom out, display
'   if "+/-" then unarm, change sample setup
'   if number then change channel trigger or name
'   if "m" measure (control arrows, exit "m")
'   if "t" change trigger point
'   if "s" single trigger
'   if "r" reapeat trigger (run)




'--------------  prepare and init the picomite for LA ---------------

  max_chan_no%=5'7                    'number of input channels 0..5 / 7
  samples% = 4096                     'must be 2^x, larger than 256
  posttrigger% = samples%*7/8         'default trigger in circular buffer
  st%=0:sp%=samples%-1                'whole buffer shown default
  frequency_index%=8                  '500kHz default sampling speed
  single%=&h7fffffff                  'large, anything > 0 for repreat
  generate_test_signal                '--debug-- test signal

  PIO_prog                            'program the PIO to sample

  init_memory                         'set up data buffers
  init_trigger_config                 'set up choice arrays
  init_sampling_rate                  'different sampling rates

  set_screen                          'set default screen parameters

  helptext

'------------------------------ main loop -------------------------------

  Do

    If aa$="" Then
      aa$=Inkey$
    EndIf

    If aa$="p" Then Save compressed image "image.bmp"

'pan/scan on cursor keys
    zoom aa$

'change sampling speed
    If aa$="+" Then
      frequency_index%=Min(frequency_index%+1,14)
      f0=sf%(frequency_index%)
      Box 56,42,52,16,,1,0             'full buffer
      Print @(62,44) Right$("  "+sr$(frequency_index%),4);"S"
      single%=single%+1                'another sample run
    EndIf
    If aa$="-" Then
      frequency_index%=Max(frequency_index%-1,0)
      f0=sf%(frequency_index%)
      Box 56,42,52,16,,1,0             'full buffer
      Print @(62,44) Right$("  "+sr$(frequency_index%),4);"S"
      single%=single%+1                'another sample run
    EndIf

'run/stop control
    If aa$="s" Then
      single%=1
      Print @(10,44)"SNGL"
    EndIf
    If aa$="r" Then
      single%=&h7fffffff
      Print @(10,44)" RUN"
    EndIf

'change trigger with numeric keys and change channel name
    If Asc(aa$)>=&h30 And Asc(aa$) <= (&h30 + max_chan_no%) Then
      cha%=Val(aa$)
      Inc tr%(cha%),1
      If tr%(cha%)=5 Then
        tr%(cha%)=0
        edit_channel_name
      EndIf
      trigger_legend
      trigger_create_program
    EndIf

'change trigger point in sampling buffer
    If aa$="t" Then
      posttrigger%=posttrigger%-(samples%*3/8)
      If posttrigger%<0 Then posttrigger%=samples%*7/8
    EndIf

'measure in waveforms
    If aa$="m" Then
      measure_time
    EndIf

'measure in waveforms
    If aa$="i" Then
      I2C_decoder
    EndIf

'end of key decoding
    aa$=""

'print zoom banner
    show_zoom
    If single% > 0 Then
'start la, wait for trigger
      arm_la

'calculate minimal circular buffer prefill time
      filltime%=Int(1000*3*(samples%-posttrigger%)/f0)+10
      Pause filltime%     'fill pre-trigger buffer

'generate_trigger
      enable_trigger

'wait for trigger to happen
      Print @(350,44) "ARM"
      Do
        aa$=Inkey$:Pause 50
      Loop While Pin(gp22)=1 And aa$=""
      Print @(350,44) "TRIG"

'the PIO should autocomplete, wait for post trigger length% written
      filltime%=Int(1000*3*posttrigger%/f0)+10
      Pause filltime%    'fill post-trigger buffer

'stop DMA, this will call ReadyInt that stops the PIO and re-init it.
      If DMA_RUN%=1 Then
        PIO DMA RX OFF                    'also unpacks the data
        If MM.Ver>=5.070807 Then ReadyInt 'needed if endless DMA
      EndIf

'signal that the acquisition has finished
      single%=single%-1
      If single%=0 Then Print @(10,44)"HALT"
    EndIf

'display the data
    scrnprint st%,sp%
  Loop

End



'-----------------------------------SUBS MMBasic ------------------------------


'add help text in lower part of the screen in 6 channel mode
Sub helptext
  If max_chan_no%=5 Then
    TILE 0,23,RGB(midgreen),0,80,7             'help text green
    Local a$=Chr$(146)+Chr$(147), b$=Chr$(148)+Chr$(149)
    Local l=360,h=30
    Print @(h,l) "Mode   : R(un)=automatic  S(ngl)=single shot from trigger"
    Print @(h,l+12) "Trigger: 0...5 changes trigger mode for each GP0...5 pin"
    Print @(h,l+24) "         T indicates Trigger in window and view banner"
    Print @(h,l+36) "         ARM=waiting for trigger  TRIG=trigger occurred"
    Print @(h,l+48) "Name   : 0...5 change channel name (or change trigger)"
    Print @(h,l+60) "Zoom   : ";a$;"=zoom(in/out)  ";b$;"=move window"
    Print @(h,l+72) "Sample : +/- = faster/slower   min=1kS/s   max=42MS/s"
    Print @(h,l+84) "Measure: M(easure)=time between edges ON/OFF ";a$;b$;"=move focus"
    Print @(h,l+96) "I2C    : I(2c)=decode ON/OFF ";b$;"=scroll   GP0=SCL  GP1=SDA"
  EndIf
End Sub



'use the cursor keys to change for zoomwindow
Sub zoom aa$
    If aa$=Chr$(128) Then
      cp%=(sp%+st%)/2:rn%=(sp%-st%)
      rn%=Max(rn%/4,64)
      st%=cp%-rn%:sp%=cp%+rn%
    EndIf
    If aa$=Chr$(129) Then
      cp%=(sp%+st%)/2:rn%=(sp%-st%)
      st%=Max(cp%-rn%,0):sp%=Min(cp%+rn%,samples%-1)
    EndIf
    If aa$=Chr$(130) Then
      cp%=(sp%+st%)/2:rn%=(sp%-st%)/2
      cp%=cp%-rn%/2
      st%=Max(cp%-rn%,0):sp%=Min(cp%+rn%,samples%-1)
    EndIf
    If aa$=Chr$(131) Then
      cp%=(sp%+st%)/2:rn%=(sp%-st%)/2
      cp%=cp%+rn%/2
      st%=Max(cp%-rn%,0):sp%=Min(cp%+rn%,samples%-1)
    EndIf
    scrnprint st%,sp%
End Sub


'decode the data in unpacked%() for valid I2C
'assume SCL=GP0, SDA=GP1
Sub I2C_decoder
  Local scl%=2^0,sda%=2^1,scls%=0,sdas%=1
  Local i%,c0%,c1%,d0%,d1%,byte%=0,bit%=0,bv%=0,da%
  nbr%=0

'init start condition
  c0%=(unpacked%(0) And scl%)>>scls%
  d0%=(unpacked%(0) And sda%)>>sdas%

'ripple through the bits and check for I2C typical transitions
  For i%=1 To samples%-2
    c1%=(unpacked%(i%) And scl%)>>scls%
    d1%=(unpacked%(i%) And sda%)>>sdas%
'now we can start detecting, start and stop are dominant
    If c0%=1 And c1%=1 Then
      If d0%=1 And d1%=0 Then
        I2C_wrt_inc "S",i%                  'start
        bit%=0:byte%=0                      'start of new message
      ElseIf d0%=0 And d1%=1 Then
        I2C_wrt_inc "P",i%                  'stop
      EndIf
'check the databits
    ElseIf c0%=0 And c1%=1 Then     'sample after leading clock edge
      Inc bit%,1
      If bit%=9 Then
        If byte%>0 Then
          da%=(i%+pnt%(nbr%-1))/2
          I2C_wrt_inc "0x"+Hex$(bv%),da%      'keep byte
        EndIf
        If d0%=1 Then
          I2C_wrt_inc "N",i%                  'NACK
        Else
          I2C_wrt_inc "A",i%                  'ACKN
        EndIf
        bv%=0:bit%=0:Inc byte%,1
      Else
        bv%=(bv%<<1)+d0%                      'shift bit in
      EndIf
      If bit%=8 And byte%=0 Then
        da%=(i%+pnt%(nbr%-1))/2
        I2C_wrt_inc "0x"+Hex$(bv%>>1),da%     '7bit address
        If (bv% And 1)=0 Then
          I2C_wrt_inc "W",i%
        Else
          I2C_wrt_inc "R",i%
        EndIf
      EndIf
    EndIf
    d0%=d1%:c0%=c1%                         'new samples, keep old values
  Next i%


  Do
    For j%=0 To nbr%
      If pnt%(j%)>st% And pnt%(j%)<sp%-(sp%-st%)/10 Then
        Print @(116+512*(pnt%(j%)-st%)/(sp%-st%),116) deco$(j%)
      EndIf
    Next j%

    Do :a$=Inkey$:Loop While a$=""
    Print @(120,116) Space$(60)
    zoom a$
    show_zoom
  Loop Until a$="i"

End Sub


'add data to the I2C decode arrays
Sub I2C_wrt_inc a$,i%
  pnt%(nbr%)=i%
  deco$(nbr%)=a$
  Inc nbr%,1
End Sub


'calculate timing from data in unpacked array and highlight on screen
'start position is mid screen channel 0. cursor keys can select
'other channel or other section. Measuring limitted to view window.
Sub measure_time
  Local ch%=0, lv%, i%, j%, tim, cs%, cw%, a$, t$

'determine where we are initially (mid screen)
  i% = (st%+sp%)/2

  Do
    j%=i% 'second pointer

'calculate what is logic level @i% and window zoom
    lv%=(unpacked%(i%) And 2^ch%)
    stp%=(sp%-st%)/512

'find edges left and right
    Do
      Inc i%,-1
    Loop While(lv% = (unpacked%(i%) And 2^ch%)) And i%>st%
    Do
      Inc j%,1
    Loop While(lv% = (unpacked%(j%) And 2^ch%)) And j%<sp%

'calculate time from found edges
'both edges are included, for accuracy drop 1 edge (-1)
    tim=3*1e9*(j%-i%-1)/f0

'calculate screen locations in tiles
    cs% = Min(79,Max(0,64*(i%-st%)/(sp%-st%)))
    cw% = Min(66-cs%,66*(j%-i%)/(sp%-st%)+1)

'color background
    TILE 14,5+ch%*3,RGB(yellow),0,66,2
    TILE 14+cs%,5+ch%*3,RGB(red),0,cw%,2

'print value and arrow
    If j%=sp% Or i%=st% Then
      t$="no edge"
    Else
      If tim>1e6 Then
        t$=Str$(tim/1e6,3,1)+"ms"
      ElseIf tim>1e3 Then
        t$=Str$(tim/1e3,3,1)+"us"
      Else
        t$=ste$(tim,3,1)+"ns"
      EndIf
    EndIf
    Print @(85+8*(cs%+cw%/2),90+ch%*48) t$

    Do :a$=Inkey$:Loop While a$=""

'control measurement focus
    If Asc(a$)=131 Then i%=Min(j%+2,sp%-2)
    If Asc(a$)=130 Then i%=Max(i%-2,st%+2)
    If Asc(a$)=128 Then
      TILE 14,5+ch%*3,RGB(yellow),0,66,2
      ch%=Max(ch%-1,0)
      i%=(st%+sp%)/2
    EndIf
    If Asc(a$)=129 Then
      TILE 14,5+ch%*3,RGB(yellow),0,66,2
      ch%=Min(ch%+1,max_chan_no%)
      i%=(st%+sp%)/2
    EndIf
    scrnprint st%,sp%

  Loop Until a$="m"

'restore colors
  TILE 14,5+ch%*3,RGB(yellow),0,66,2

End Sub


'edit the channel name. skip when <CR> or channel number. store
'when length = 9 or <CR> after chars are typed
Sub edit_channel_name
  Local nn$,newlen%,aa$
  Box 27,80+48*cha%,80,16,,0,1
  Print @(30,82+48*cha%); nam$(cha%)
  newlen%=0:nn$=""
  Do
    aa$=Inkey$
    Pause 50
    nn$=nn$+aa$
    Print @(30,82+48*cha%);Left$(nn$+"         ",9)
  Loop Until aa$=Chr$(13) Or Len(nn$)=9 Or (Asc(aa$)=cha%+&h30 And Len(nn$)=1)
  If Len(nn$)>2 Then nam$(cha%)=Left$(nn$,9)
  Box 27,80+48*cha%,80,16,,1,1
  channel_legend
End Sub


'start the trigger engine
Sub enable_trigger
  PIO init machine 1,1,f1,p1,e1,s1,10   'start address = 10 (0x0A)
  PIO start 1,1
End Sub


'trigger characters related to PIO instructions
Sub init_trigger_config
  Dim c$(4) length 2 = ("x","1","0","/","\")  'text for the 5trigger  modes
  Dim tr%(max_chan_no%)                       'trigger mode default 0

End Sub


'relation between PIO clock and sampling rates
Sub init_sampling_rate
  Dim sr$(14) length 4=("1k","2k","5k","10k","20k","50k","100k","200k","500k","1M","2M","5M","10M","21M","42M")
  Dim sf%(14)=(3e3,6e3,15e3,3e4,6e4,15e4,3e5,6e5,15e5,3e6,6e6,15e6,31e6,63e6,126e6)
End Sub


'initializes the screen (colors, font, lines)
Sub set_screen
  MODE 1:Colour RGB(WHITE),0:CLS :Font 1      'generic screen setup from Martin H
  TILE height 16                              'since V50707b24 tiles are 8xX where X is height
  TILE 0,0,RGB(white),0,80,30                 'default all tiles
  Dim nam$(max_chan_no%) length 9
  Load image "banner.bmp":Line 0,41,639,41,2,0'line=cosmetic
  Local f%

'indicate initial sampling speed and run mode
  Box 0,42,52,16,,1,0:Print @(10,44)" RUN"
  Box 56,42,52,16,,1,0:Print @(62,44) Right$("  "+sr$(frequency_index%),4);"S"

'default color pattern and channel names for LA from Martin H
  For f%=0 To max_chan_no%
    TILE 0,5+f%*3,RGB(WHITE),0,14,2:TILE 14,5+f%*3,RGB(yellow),0,66,2
    TILE 0,4+f%*3,RGB(midgreen),0,80,1:Line 0,64+f%*48,640,64+f%*48,,1
    nam$(f%)="Channel "+Str$(f%)
  Next f%

'posh for 6 channel mode
  TILE 0,22,RGB(midgreen),0,80,1:Line 0,352,640,352,,1

'show default startup information for channels and trigger
  channel_legend
  trigger_legend

End Sub


'shows the trigger settings on screen
Sub trigger_legend
  Local f%,i%
  For f%=0 To max_chan_no%               'for all channels
    Box 30,96+48*f%,72,16,,0,0           'erase all boxes and text
    For i%=0 To 4                        'step through all 5 trigger modes
      If tr%(f%)=i% Then
        Box 30+i%*14,96+48*f%,16,16      'write current trigger boxes
      EndIf
      Print @(34+i%*14,98+48*f%) c$(i%)  're-write text
    Next i%
  Next f%
End Sub


'shows the channel names on screen
Sub channel_legend
  Local f%
  For f%=0 To max_chan_no%
    Font 2:Print @(0,84+48*f%); f%
    Font 1:Print @(30,82+48*f%); nam$(f%)
  Next f%
End Sub


'unpack packed%() to unpacked%() such that trigger is in the centre
Sub unroll
  varindex%=(Pio(DMA RX POINTER)-Peek(VARADDR packed%()))/8   'bytes -> 64bits
'  beginsamples%=(Pio(DMA RX POINTER)-Peek(VARADDR packed%()))/4  'bytes->32bit
  beginsamples%=varindex%*2                                   '64->32bit

'in case DMA timeout, the varindex% exceeds array limits, correct it
  If varindex% > length%-1 Then
    varindex%=0                                             'in case timeout
    If beginsamples% > samples%-1 Then beginsamples%=0      'in case timeout
    DMA_RUN%=0
  EndIf

  endsamples% = samples% - beginsamples%
  Memory unpack Peek(varaddr packed%(varindex%)),unpacked%(),endsamples%,32
  Memory unpack packed%(),Peek(varaddr unpacked%(endsamples%)),beginsamples%,32
End Sub


'when DMA is ready stop PIO's and unroll buffer
Sub ReadyInt
  PIO STOP 1,0
  PIO stop 1,1
  unroll
  Math set 0,packed%()
End Sub


'Start the DMA to accept FIFO data from PIO 1.0, and set the posttrigger value
Sub arm_la
'prepare the PIO 1.0 for new DMA
  PIO init machine 1,0,f0,p0,e0,s0,0     'start address = 0

'the DMA will time out after &h7FFFFFFF samples, or when stopped
  If MM.Ver<5.070807 Then
    PIO DMA RX 1,0,&h7FFFFFFF,packed%(),ReadyInt,32,samples% 'start DMA ring buf
  Else
    PIO DMA RX 1,0,0,packed%(),,32,samples% 'start DMA ring buf
  EndIf
  DMA_RUN%=1

'write to PIO the number of post trigger samples, starts logging pre-trigger
  PIO WRITE 1,0,1,posttrigger%
End Sub


Sub init_memory
'define data buffers
'The logic analyzer captures samples% samples into the packed%() array'
'The DMA buffer has 64bit width, packed with 32bit FIFO (2 per cell) = length%
'The size of the ring buffer is specified in bytes, so size is 8 * length%
'the samples will be unpacked into unpacked%() at 32 bits per cell (=2*lenth%)

  length% = samples% / 2             'packed buffer = 64bit, data buffer 32bit
  Dim unpacked%(samples%-1)          'array to put the 32 bit samples (base 0)

  Dim packed%                        'name of array to become circular buffer
  PIO MAKE RING BUFFER packed%,8*length%

  Dim pnt%(100),deco$(100) length 10 'buffers for I2C decoding

End Sub


' Test signal: Three phase motor drive - picomite V50706+
Sub generate_test_signal
' runs autonomously on GP0...GP5 50Hz

  SetPin gp0,pwm  'CH 0a
  SetPin gp1,pwm  'CH 0b
  SetPin gp2,pwm  'CH 1a
  SetPin gp3,pwm  'CH 1b
  SetPin gp4,pwm  'CH 2a
  SetPin gp5,pwm  'CH 2b

  PWM 0, 50, 33.33, -50
  PWM 1, 500, 33.33, -50
  PWM 2, 5000, 33.33, -50

  Pause 20
End Sub


'program the PIO with the sampler code (PIO 1.0) and default trigger PIO 1.1
'PIO 1.1 is reprogrammed each time trigger is changed
Sub PIO_prog

'assign GP22 (trigger_not) to PIO
  SetPin gp22,pio1

'the PIO 1.0 program: the sampler PIO reads GP0..GP7 brute force
'and pushes data into FIFO. The clock speed determines the
'sampling rate. There are 3 instructions per cycle
'after trigger (line 6) is detected "posttrigger%" samples are added to FIFO


'clean start
  PIO clear 1


'program pio1.0 : sampler
  PIO program line 1,0,&hE081           'SET GP22 output

  PIO program line 1,1,&hE001           'SET GP22 high
  PIO program line 1,2,&h80A0           'PULL block
  PIO program line 1,3,&hA027           'mov(x,osr)

  PIO program line 1,4,&h4001+max_chan_no%  'IN pins 6/8
  PIO program line 1,5,&h8000           'PUSH noblock
  PIO program line 1,6,&h00C4           'JMP (GP22) 4

  PIO program line 1,7,&h4001+max_chan_no%  'IN pins 6/8
  PIO program line 1,8,&h8000           'PUSH noblock
  PIO program line 1,9,&h0047           'JMP (X<>0) 7 X--


'initial trigger program: run continuous (set GP22 low)
  PIO program line 1,10,&hE000          'set GP22 low
  PIO program line 1,11,&h000A          'jmp always to itself


'configuration PIO 1.0
  f0=15e5                             'PIO run at 500kHz
  p0=Pio(pinctrl 0,1,0,gp0,,GP22,)   'IN base = GP0, SET = GP22
  e0=Pio(execctrl gp22,1,9)          'wrap 9 to 1, gp22 is conditional JMP pin
  s0=Pio(shiftctrl 0,0,0,0,0,1)      'shift in through LSB, out is not used


'configuration PIO 1.1
  f1=126e6                           'PIO run at 126MHz
  p1=Pio(pinctrl 0,1,0,gp0,,GP22,)   'IN base = GP0, SET = GP22
  e1=Pio(execctrl gp0,10,30)         'wrap 30 to 10
  s1=Pio(shiftctrl 0,0,0,0,0,1)      'shift in through LSB, out through LSB


'write the default configurations
  PIO init machine 1,0,f0,p0,e0,s0,0    'start address = 0
  PIO init machine 1,1,f1,p1,e1,s1,10   'start address = 10 (0x0A)

End Sub


'print the traces on the serial output
Sub scrnprint st%,sp%
'print section st% to sp% of the linear array to serial (screen)
'Timer =0
  Local z%,y%,x%,stp%,mask%,bit%,trig%,old%
  trig%=samples%-posttrigger%
  stp%=Int(0.5+(sp%-st%)/128)
  For j%=0 To max_chan_no%
    z%=j%*48+85
    mask%=2^j%
    Box 112,z%-17,524,14,,0,0  'erase green bar
    For i%=st% To sp% Step stp%
      old%=bit%
      bit%=((unpacked%(i%) And mask%)=0)
      x%=116+4*(i%-st%)/stp%
      Box x%,z%,5,21,,0,0  'erase plot box
      y%=z%+20*bit%
      Line x%,y%,x%+4,y%,,1
      If bit%<>old% And i%>st% Then Line x%,z%,x%,z%+20,,1
      If i%>=trig% And i%<(trig%+stp%) And tr%(j%)<>0 Then
        Print @(x%-4,z%-17) "T"
      EndIf
    Next i%
  Next j%
'Print @(0,0) Timer
End Sub


'show bar st%-sp% in relation to full range 0-samples%
Sub show_zoom
  Local stp%,trg%
  Box 116,42,513,16,,1,0             'full buffer
  stp%=samples%/512
  Box 117+(st%/stp%),44,(sp%-st%)/stp%,12,,1,1
  trg%=samples%-posttrigger%
'box 117+(trg%/stp%),44,3,12,,0,1
  Print @(112+(trg%/stp%),44) "T"
End Sub


'from the tr%() array create the trigger program for PIO 1.1
Sub trigger_create_program
'when there is an edge trigger, it is placed at start of the program
'after that the logic conditions are added
'there can be only 1 edge trigger in this engine, the first in string

  Local trigger_start_address%, trigger_end_address%, i%
  trigger_start_address% = 10            'address &h0A
  trigger_end_address%= 31               'last location PIO program
  write_pio%=trigger_start_address%

'check edge trigger, for edge trigger use the PIO WAIT GPx instruction
'a falling edge is WAIT GPx=1, WAIT GPx=0. Rising edge WAIT GPx=0, WAIT GPx=1

  For i%=0 To max_chan_no%
    If tr%(i%) > 2 Then

'found edge trigger for channel i%
'below code can be optimized further

      If tr%(i%)=3 Then     'rising edge
        inc_write_pio &h2000+i%
        inc_write_pio &h2080+i%
      Else                  'falling edge
        inc_write_pio &h2080+i%
        inc_write_pio &h2000+i%
      EndIf
      i%=max_chan_no%                 'search no further, skip the rest
    EndIf
  Next i%


'check the static conditions. This shifts data in the ISR, copies to OSR
  inc_write_pio &h4001+max_chan_no%   'IN ISR 6/8
  inc_write_pio &hA0E6                'MOV ISR->OSR


'Each input processed by reading a new bit from OSR and checking it
'edge triggered pin is skipped (first one is processed, rest is ignored)

  For i%=0 To max_chan_no%
    Select Case tr%(i%)
      Case 0,3,4                      'x / \ do nothing, shift to Y trash
        inc_write_pio &h6041          'OUT OSR->Y 1
      Case 1                          '1
        inc_write_pio &h6021          'OUT OSR->X 1
        inc_write_pio &h002A          'trigger GPx=1
      Case 2                          '0
        inc_write_pio &h6021          'OUT OSR->X 1
        inc_write_pio &h004A          'trigger GPx=0
    End Select
  Next i%

'when all triggers are true, we set GP22 low as trigger for the sampler
  inc_write_pio &hE000                'set GP22 low
  inc_write_pio &h000A                'jmp top (needed when memory not full)
'print @(10,10) write_pio%

' maximum program length is:
' without edge trigger: 2 (shift in) + 8*2 (8 channels) + 2 (trigger+jump) = 20
' with edge trigger: 2 (edge) + 2 (shift in) + 7 * 2 (7 remaining channels)
'                                    + 1 (skipped edge) + 2 (trigger+jump) = 21
' start addres = 10, end = 31 -> 21 will always fit. No check needed.
End Sub


'this programs 1 PIO line and advances to the next line
Sub inc_write_pio instr%
  PIO program line 1,write_pio%,instr%
  Inc write_pio%,1
End Sub
    