  'mode 1
  'autosave
  '=============================================================================|
  ' LodeRunner Demo
  '
  ' written by Vegipete, Oct 2023
  '
  ' needs:
  ' OPTION CPUSPEED 252000 (KHz)
  ' OPTION SYSTEM I2C GP0,GP1     ' to use Wii controller on controller port
  '
  ' Send the levels data file to the PicoMite.
  ' In the Teraterm console, type the Picomite command:
  '     [xmodem receive "LRunnerLevels.bin"]
  ' Then use the Teraterm menus, use
  '     [File:Transfer:XMODEM:Send...]
  ' to select the desired level file and send it.
  '
  
  Option DEFAULT INTEGER
  MODE 2
  
  const cntl=2 '0=PS2, 1=Wiiclassic, 2=Wii nunchuck
  
  if cntl=1 then Device wii open
  ' device(wii funct)
  '
  ' Returns data from a Wii Classic controller.
  ' 'funct' is a 1 or 2 letter code indicating the info to return as follows:
  ' LX returns the position of the analog left joystick x axis
  ' LY returns the position of the analog left joystick y axis
  ' RX returns the position of the analog right joystick x axis
  ' RY returns the position of the analog right joystick y axis
  ' L returns the position of the analog left button
  ' R returns the position of the analog right button
  ' T returns the ID code of the controller - should be hex &H4200101
  ' B returns a bitmap of the state of all the buttons.
  '     A bit will be set to 1 if the button is pressed.
  ' The button bitmap is as follows:
  ' BIT 0: Button R
  ' BIT 1: Button start
  ' BIT 2: Button home
  ' BIT 3: Button select
  ' BIT 4: Button L
  ' BIT 5: Button down cursor
  ' BIT 6: Button right cursor
  ' BIT 7: Button up cursor
  ' BIT 8: Button left cursor
  ' BIT 9: Button ZR
  ' BIT 10: Button x
  ' BIT 11: Button a
  ' BIT 12: Button y
  ' BIT 13: Button b
  ' BIT 14: Button ZL

  const nunch = &h52
  if cntl=2 then init_nunch
  'register 0 = X joystick
  'register 1 = Y joystick
  'register 2,3,4 = XYZ acceleration (not used)
  'register 5 = buttons Y and C (xxxxxYC)

  
  'const MAXLEVELS = 10
  Const NUMMONK = 8     ' max number of monks
  Const NUMHOLES = 11   ' max number simultaneous holes
  Const LEVNAME$ = "LRunnerLevels.bin"
  Const SPRADDR = Peek(cfunaddr Lode)
  Const ASHX$ = "0123456789ABCDEF"
  Dim maxlevels
  
  ' Test for levels file and determine how many levels
  On error skip
  Open LEVNAME$ For input As #1
  If MM.Errno Then
    Print "Levels file not found.  :-("
    On error skip
    Close #1
End
  EndIf
  maxlevels = Val(Input$(4,#1))  ' read number of levels in data file
  Close #1
  
  Dim string crlv(27,16) length 1  ' current level
  
  Dim holes(NUMHOLES,4) ' 12 possible holes: time,x,y,tilex,tiley
  Dim act(NUMMONK,6)  ' actor state (0 = player)
  ' 0: direc 0=none/1=run r/2=run l/3=monkey r/4=monkey l
  '          5=ladder up/6=ladder down/7=fall r/8=fall l
  '          9=blast r/10=blast l
  ' 1: time until next move
  ' 2: reset time
  ' 3: x posn in pixels
  ' 4: y posn in pixels
  ' 5: pose number
  ' 6: prev direction: 0=down/1=right/2=left
  Dim soff(11) = (0,9,12,15,18,21,21,23,24,25,26,25) ' sprite offsets
  Dim bcnt    ' number of barrels (gold) on level
  Dim fire = 0  ' -1=fire left, 1=fire right
  
  'Init all
  Option LCDPANEL NOCONSOLE   ' print output to CDC only
  scor = 0
  men = 5
  lev = 9
  
  ' Prepare layers
  FRAMEBUFFER Create   ' hidden layer to draw actors
  FRAMEBUFFER Write F : CLS 0
  
  FRAMEBUFFER Layer 13 ' colour 13 (lilac) is transparent
  FRAMEBUFFER Write L : CLS RGB(lilac)
  Box 19,19,282,179,1,0,0 ' hide playing area
  
  FRAMEBUFFER Write n : CLS 0
  
  Do
    ReadLevel(lev)
    DrawTitleBox
    DrawCurLevel(0)  ' play view
    'DrawCurLevel(1)  ' edit view
    Iris(1,5)  ' open the iris to view new level
    CountLevel
    PlayLevel
    DrawActors  ' redraw on layer N for iris close
    Iris(0,5)  ' hide it
  Loop Until lev > maxlevels
  
  if cntl=1 then Device wii close
  FRAMEBUFFER close
  option lcdpanel console 7
  CLS 0
  Print "End of demo."
  Print "Would you like more?"
End
  
  '================================================================
  ' Find items on current level - valid only if freshly loaded.
  ' Player and monks are erased from the static (background) image
  ' and redrawn on the overlay layer
Sub CountLevel
  Local x,y
  bcnt = 0
  Math set 0,act()
  i = 0   ' monk count
  For y = 0 To 15
    For x = 0 To 27
      Select Case crlv(x,y)
        Case "7"  ' barrel
          Inc bcnt
        Case "8"  ' monk
          Inc i
          act(i,0) = 2    ' active, going left
          act(i,3) = x*10
          act(i,4) = y*11  ' location
          act(i,5) = 0    ' starting pose
          crlv(x,y) = "0"     ' erase monk from current level data
          s = soff(2)+34  ' locate base sprite location
          s = SPRADDR + s * 60
          FRAMEBUFFER Write L   ' draw monk on layer
          Sprite memory s,20+act(i,3),21+act(i,4),0  ' black is transparent
          FRAMEBUFFER Write n   ' erase monk on background
          Sprite memory SPRADDR,20+act(i,3),21+act(i,4) ' erase
        Case "9"  ' player
          act(0,0) = 1    ' active, going right
          act(0,3) = x*10
          act(0,4) = y*11  ' location
          act(0,5) = 0    ' starting pose
          act(0,6) = 1
          crlv(x,y) = "0"     ' erase player from current level data
          s = soff(1)     ' locate base sprite location
          s = SPRADDR + s * 60
          FRAMEBUFFER Write L
          Sprite memory s,20+act(0,3),21+act(0,4),0  ' black is transparent
          FRAMEBUFFER Write n
          Sprite memory SPRADDR,20+act(0,3),21+act(0,4) ' erase
      End Select
    Next x
  Next y
End Sub
  
  '' Test routine to count barrels on each level
  '
  '  bcnt = 0
  '  for y = 0 to 15
  '    for x = 0 to 27
  '      if crlv(x,y) = 7 then inc bcnt
  '    next x
  '  next y
  '  print lev,bcnt
  '  inc lev
  'loop until lev > maxlevels
  '
  'end
  '================================================================
  
Sub PlayLevel
  Local pt,mt,ht
  
  Math set 0,holes()
  BlinkPlayer   ' wake player, wait for input to start
  
  pt = Timer : mt = pt : ht = pt
  Do
    t = Timer
    If t > pt + 36 Then MovePlayer : pt = t
    If t > mt + 99 Then MoveMonks  : mt = t
    If t > ht + 50 Then FillHoles  : ht = t
    DrawMotion
    DrawScore
    if cntl=1 then
      If DEVICE(wii b) And 8 Then Inc lev : Exit ' select is pressed
      If DEVICE(wii b) And 2 Then Exit ' start is pressed
    else if cntl=2 then
      IF read_nunch(5)=1 THEN INC lev:EXIT
      IF read_nunch(5)=0 then exit
    else
      kyy$=inkey$ 'keybard control
      if kyy$="a" then inc lev:exit do 'press "a" as select
      if kyy$="b" then EXIT DO 'press "b" as start
    end if
    If bcnt = 0 And act(0,4) < 1 Then  ' got all gold, at top
      Pause 1000
      Inc lev
      Exit ' level done! Yah!
    EndIf
  Loop
End Sub
  
  '================================================================
  ' Blink the player's character until a button on the controller is pressed
Sub BlinkPlayer
  FRAMEBUFFER Write L
  Do
    Sprite memory SPRADDR,20+act(0,3),21+act(0,4) ' erase
    If WaitButton(1,333) > 0 Then Exit
    Sprite memory SPRADDR+soff(1)*60,20+act(0,3),21+act(0,4),0 ' draw player
    If WaitButton(1,333) > 0 Then Exit
  Loop
  Sprite memory SPRADDR+soff(1)*60,20+act(0,3),21+act(0,4),0 ' draw player
  if cntl=1 then
    Do : Loop Until DEVICE(wii b) = 0   ' wait for all buttons released
  else if cntl=2 then
    do : loop until read_nunch(5)=3     ' wait for all buttons released
  else
    do : loop until inkey$=""
  endif
End Sub
  
  ' Wait for controller buttons to be released (d=0) or pressed (d!=0)
  ' Timeout after t ms.
  ' Returns -1 if timed out, 0 or B if controller buttons
Function WaitButton(d,t)
  Local et,b
  d = (d<>0)  ' ensure d is 0 or 1 only
  et = Timer + t
  if cntl=1 then
    Do
      b = DEVICE(wii b)
      If ((b <> 0) = d) Then WaitButton = b : Exit Function
      If d=(DEVICE(wii LX) <> 128) Then WaitButton=&h10000*d:Exit Function
      If d=(DEVICE(wii LY) <> 128) Then WaitButton=&h20000*d:Exit Function
      If d=(DEVICE(wii rx) <> 128) Then WaitButton=&h40000*d:Exit Function
      If d=(DEVICE(wii ry) <> 128) Then WaitButton=&h80000*d:Exit Function
    Loop Until Timer > et
  else if cntl=2 then
    do
      if read_nunch(5)<>3 then WaitButton=read_nunch(5):exit function
      if read_nunch(0)<>128 then WaitButton=1:EXIT function
      if read_nunch(1)<>128 then WaitButton=1:EXIT function      
    loop until timer > et
  else
    do
      kyy$=inkey$
      if kyy$<>"" then WaitButton=asc(kyy$):exit function
    loop until timer > et
  end if
  WaitButton = -1   ' timed out
End Function
  
  ' Move the player according to user input or falling
  ' The right joystick of the Wii Classic is used for movement
  ' and must be held to continue movement
  ' fire right: Button right cursor 2^6 = 64
  ' fire left : Button left cursor 2^8 = 256
  ' Note for keyboard use: last player direction is maintained
  '       until motion in that direction is blocked.
Sub MovePlayer
  Local b,ox,oy,nx,ny,rx,ry   ' old,new,rounded
  local kyy$,ky,ty
  static wiix,wiiy
  
  if cntl=1 then
    wiix = DEVICE(wii rx)
    wiiy = DEVICE(wii ry)
    fire = (DEVICE(wii b) = 64) - (DEVICE(wii b) = 256)
  else if cntl=2 then
    wiix = read_nunch(0)
    wiiy = read_nunch(1)
    fire = (read_nunch(5)=2)
  else
    ty=timer + 30
    do
      kyy$=inkey$
    loop until kyy$<>"" or timer>ty
    ky=ASC(kyy$)
    wiiy=128 - 100*((ky=129)-(ky=128))
    wiix=128 + 100*((ky=131)-(ky=130))
    fire=(kyy$=" ")
  end if
  
  ox = act(0,3) \ 10  ' tile player is currently in
  oy = act(0,4) \ 11  ' tile player is currently in
  
  ' While falling, player cannot change motion.
  ' However, plMovePlayerayer can select direction to move at bottom of fall. (UDLR)
  If act(0,0) = 7 Or act(0,0) = 8 Then ' falling
    ' user input while falling changes previous motion
    If wiix > 135 Then act(0,6) = 1
    If wiix < 120 Then act(0,6) = 2
    If wiiy > 135 Then act(0,6) = 7 'or 8
    If wiiy < 120 Then act(0,6) = 7 'or 8
    
    ny = (act(0,4)+2) \ 11  ' tile player will enter next
    'Print "fall",act(0,3),act(0,4),ox,oy,ny,crlv(ox,oy) crlv(ox,ny) crlv(ox,ny+1)
    If oy = 15 Then ' hit the very bottom
      act(0,4) = oy * 11
      Select Case act(0,6)      ' move on bottom as per previous direction
          Case 1,5 : act(0,0) = 1   ' run right
          Case 2,6 : act(0,0) = 2   ' run left
          Case 7   : act(0,0) = 17  ' fall pose right
          Case 8   : act(0,0) = 18  ' fall pose left
      End Select
      Exit Sub
    EndIf
    
    If ny > oy Then ' about to fall into a new tile
      If crlv(ox,ny) = "7" Then GrabbedGold(ox,ny) ' fall onto barrel
      
      If crlv(ox,ny) = "4" Then ' tile is a monkey bar
        If wiiy < 120 Then  ' joystick down so keep falling
          Inc act(0,4),2
          Exit Sub
        Else    ' grab monkey bar
          act(0,4) = ny * 11
          Select Case act(0,6)      ' move on monkey bar as before
              Case 1,5,7 : act(0,0) = 3   ' monkey right
              Case 2,6,8 : act(0,0) = 4   ' monkey left
          End Select
          Exit Sub
        EndIf
      EndIf
      
      If Instr("123",crlv(ox,ny+1)) Then ' landing on brick, solid or ladder
        act(0,4) = ny * 11
        Select Case act(0,6)      ' move on surface as per previous direction
            Case 1,5 : act(0,0) = 1   ' run right
            Case 2,6 : act(0,0) = 2   ' run left
            Case 7   : act(0,0) = 17  ' fall pose right
            Case 8   : act(0,0) = 18  ' fall pose left
        End Select
        Exit Sub
      EndIf
    EndIf
    
    ' continue falling
    Inc act(0,4),2
    Exit Sub
  EndIf  ' falling
  
  ' Test if burning a hole active
  ' sgn(burn) indicates left or right
  If burn Then
    Inc burn,-Sgn(burn)  ' next frame of burning animation
    Exit Sub  ' can't move while burning
  Else
    ' Test if burn a hole requested
    ' Can only fire if brick is below open space
    If fire Then
      If ox >  0 And fire < 0 Then StartBurn(fire,ox,oy) ' room to burn left
      If ox < 27 And fire > 0 Then StartBurn(fire,ox,oy) ' room to butn right
    EndIf
    If burn Then Exit Sub  ' burning started so player can't move
  EndIf
  
  ' <<<<<<<<<<
  ' Test and handle player left command
  If wiix < 33 Then  ' left
    'Print "left ",act(0,3),act(0,4),ox,oy,crlv(ox,oy)
    act(0,6) = 2  ' last player input left
    
    If act(0,3) < 2 Then act(0,3) = 0 : Exit Sub   ' can't go further left
    nx = (act(0,3)-2) \ 10   ' new x tile posn if move goes ahead
    If nx < ox Then   ' entering a new tile left
      If Instr("125",crlv(nx,oy)) Then  ' path is blocked
        Exit Sub
      EndIf
    EndIf
    rx = (act(0,3)+3)\10  ' tile we're about to enter
    If Ins("067",rx,oy) And Ins("04567",rx,oy+1) Then
      StartFalling(rx)
      Exit Sub
    EndIf
    If crlv(rx,oy) = "7" Then GrabbedGold(rx,oy) ' run left into barrel
    act(0,0) = 2  ' running left
    If crlv(rx,oy) = "4" Then act(0,0) = 4  ' monkey left
    act(0,4) = ((act(0,4) + 5) \ 11) * 11   ' latch yposn to this row
    Inc act(0,5),-1 : If act(0,5) < 0 Then act(0,5) = 2  ' next pose
    Inc act(0,3),-2  ' move left
    ' maybe exit sub now?
    
    ' >>>>>>>>>>
    ' Test and handle player right command
  ElseIf wiix > 210 Then ' right
    'Print "right",act(0,3),act(0,4),ox,oy,crlv(ox,oy)
    act(0,6) = 1  ' last player input right
    
    If act(0,3) > 268 Then act(0,3) = 270 : Exit Sub ' can't go further right
    If ox < 27 Then
      If Instr("125",crlv(ox+1,oy)) Then Exit Sub  ' blocked
    EndIf
    
    rx = (act(0,3)+7)\10  ' tile we're about to enter
    If Ins("067",rx,oy) And Ins("04567",rx,oy+1) Then
      StartFalling(rx)
      Exit Sub
    EndIf
    If crlv(rx,oy) = "7" Then GrabbedGold(rx,oy) ' run right into barrel
    act(0,0) = 1  ' running right
    If crlv((act(0,3)+5)\10,oy) = "4" Then act(0,0) = 3  ' monkey right
    act(0,4) = ((act(0,4) + 5) \ 11) * 11   ' latch yposn to this row
    Inc act(0,5) : If act(0,5) > 2 Then act(0,5) = 0  ' next pose
    Inc act(0,3),2  ' move right
    Exit Sub
  EndIf
  
  ' ^^^^^^^^^^
  ' Test and handle player up command
  If wiiy > 210 Then  ' up
    'Print "up   ",act(0,3),act(0,4),ox,oy,crlv(ox,oy)
    If act(0,4) < 1 Then act(0,4) = 0 : Exit Sub  ' can't go higher
    
    ny = (act(0,4)-2) \ 11   ' new y tile posn if move goes ahead
    If "3" = crlv(ox,oy) Then   ' on ladder
      If Instr("125",crlv(ox,ny)) Then ' climbing into brick/solid/trap?
        act(0,4) = oy * 11
        Exit Sub
      EndIf
      
      act(0,0) = 5  ' ladder up
      If act(0,6) = 1 Or act(0,6) = 2 Then ' previous motion sideways
        act(0,5) = 0    ' starting ladder so starting pose
        Inc act(0,6),4  ' save up command, remember prev left or right
        Inc act(0,4),-1 ' start up the ladder
        act(0,3) = ox * 10
        Exit Sub
      EndIf
      
      ' not at very top of tile so keep moving up
      Inc act(0,4),-2
      If act(0,4) < 0 Then act(0,4) = 0   ' can't go off top
      act(0,5) = (act(0,5) = 0)  ' next pose
      Exit Sub
    EndIf
    
    ' into space/monkey bar above ladder
    If "3" = crlv(ox,oy+1) Then ' tile below is ladder
      If crlv(ox,oy) = "7" Then GrabbedGold(ox,oy) ' climbed up to barrel
      If ny < oy Then ' trying to climb above ladder
        act(0,4) = oy * 11  ' latch to top of ladder
        Exit Sub
      EndIf
      Inc act(0,4),-2
      act(0,5) = (act(0,5) = 0)  ' next pose
      Exit Sub
    EndIf
    
    ' vvvvvvvvvv
    ' Test and handle player down command
    ' Player can only move down when on ladder or monkey bar
  ElseIf wiiy < 33 Then  ' down
    'Print "down ",act(0,3),act(0,4),ox,oy,crlv(ox,oy),crlv(ox,oy+1)
    If act(0,4) > 164 Then act(0,4) = 165 : Exit Sub  ' can't go lower
    
    If Instr("12",crlv(ox,oy+1)) Then  ' blocked below
      act(0,4) = oy * 11
      Exit Sub
    EndIf
    
    ny = (act(0,4)+2) \ 11   ' new y tile posn if move goes ahead
    
    ' drop off monkey bar into open space, monkey bar, trap or barrel
    If crlv(ox,oy)="4" And Ins("0457",ox,oy+1) Then
      act(0,6) = 8 - (act(0,6) And 1)
      StartFalling(ox)
      Exit Sub
    EndIf
    
    ' drop off bottom of ladder into open space, trap or barrel
    If crlv(ox,oy)="3" And Ins("057",ox,oy+1) And act(0,4)>oy*11+7 Then
      act(0,6) = 8 - (act(0,6) And 1)
      StartFalling(ox)
      Exit Sub
    EndIf
    
    If ny > oy Then  ' about to enter new tile down
      Select Case crlv(ox,ny)
        Case "3"      ' ladder below
          ' continue down ladder
        Case "0","5","7"  ' open space, trap or barrel
          Inc act(0,6),6  ' save down command, remember prev left or right
          StartFalling(ox)
          Exit Sub
        Case "1","2"  ' brick or solid
          Exit Sub
        Case "4"      ' monkey bar
          act(0,4) = ny * 11
          Select Case act(0,6)      ' move on monkey bar as before
              Case 1,5,7 : act(0,0) = 3   ' monkey right
              Case 2,6,8 : act(0,0) = 4   ' monkey left
          End Select
          Exit Sub
      End Select
    EndIf
    
    act(0,0) = 6  ' ladder down
    If act(0,6) = 1 Or act(0,6) = 2 Then ' previous motion sideways
      act(0,5) = 0    ' starting ladder so starting pose
      Inc act(0,6),6  ' save down command, remember prev left or right
      Inc act(0,4)    ' start down the ladder
      act(0,3) = ox * 10
      Exit Sub
    EndIf
    Inc act(0,4),2
    act(0,5) = (act(0,5) = 0)  ' next pose
  EndIf
  
End Sub
  
  ' Grabbed another gold, test if that's all of 'em
Sub GrabbedGold(x,y)
  FRAMEBUFFER Write n
  Sprite memory SPRADDR,20+x*10,21+y*11 ' draw blank over barrel
  crlv(x,y) = "0"
  Inc scor,250
  If bcnt = 1 Then ShowExitLadders    ' player has all the gold!
  Inc bcnt,-1
End Sub
  
Sub StartFalling(x)
  act(0,3) = x * 10  ' set xposn to correct column
  act(0,0) = 8 - (act(0,6) And 1)
  Inc act(0,4)  ' and down we go
  act(0,5) = 0  ' pose 0
End Sub
  
  ' Try to start burning left or right from location x,y
Sub StartBurn(d,x,y)
  Local i
  If Ins("06",x+d,y) And crlv(x+d,y+1) = "1" Then ' accessible brick to burn
    burn = d*11       ' number of burning animation frames
    act(0,0) = 1 + (burn < 0)  ' player direction after finished firing
    act(0,3) = x*10   ' lock player to tile
    act(0,4) = y*11   ' lock player to tile
    act(0,5) = 0      ' starting pose
    For i = 0 To NUMHOLES   ' find an empty hole data slot
      If holes(i,0) = 0 Then
        holes(i,0) = 110    ' time
        holes(i,1) = 20+(x+d)*10  ' x
        holes(i,2) = 21+(y+1)*11  ' y
        holes(i,3) = x+d    ' tile x
        holes(i,4) = y+1    ' tile y
        Exit For
      EndIf
    Next
  EndIf
End Sub
  
  ' Move all monks one step in their current direction
  ' This must become monk AI!
Sub MoveMonks
  Local i,nt,m,x,y
  For i = 1 To NUMMONK
    If act(i,0) Then  ' monk exists so move
      Select Case act(i,0)  ' examine direction
        Case 1,3    ' run,monkey right
          m = act(i,3) Mod 10   ' prepare tile boundary test
          Inc act(i,5) : If act(i,5) > 2 Then act(i,5) = 0  ' new pose
          Inc act(i,3),2  ' move right
          If act(i,3) >= 270 Then    ' edge of world
            act(i,3) = 540 - act(i,3)
            Inc act(i,0)  ' turn monk around
            Continue For
          Else  ' examine the tile monk is about to enter, and the one below
            If act(i,3) Mod 10 < m Then   ' entering a new tile right
              x = act(i,3)\10 : y = act(i,4)\11   ' coords of tile entered
              Select Case crlv(x+1,y)
                Case "1","2","5"    ' brick, solid or trap
                  TurnLR(i)   ' turn around
                  act(i,3) = (x) * 10
                  Continue For
                Case "4"      ' monkey bar
                  act(i,0) = 3  ' change to monkey right mode
                  Continue For
                Case "0"    ' empty space - test what's below
                  If (y < 15) Then    ' not bottom of world
                    Select Case crlv(x+1,y+1)
                      Case "1","2","3"    ' floor or ladder
                        act(i,0) = 1  ' change to run right mode
                        Continue For
                      Case Else     ' non-solid
                        TurnLR(i)   ' turn around
                        act(i,3) = (x+0) * 10
                        Continue For
                    End Select
                  EndIf
                  act(i,0) = 1  ' change to run right mode
                  Continue For
                Case "3"    ' ladder
                  act(i,0) = 1  ' change to run right mode
                  Continue For
              End Select
            EndIf
          EndIf
        Case 2,4    ' run,monkey left
          m = act(i,3) Mod 10   ' prepare tile boundary test
          Inc act(i,5),-1 : If act(i,5) < 0 Then act(i,5) = 2
          Inc act(i,3),-2  ' move left
          If act(i,3) < 0 Then  ' left edge of world
            act(i,3) = -act(i,3)
            Inc act(i,0),-1  ' turn monk around
            Continue For
          Else  ' examine the tile monk is about to enter, and the one below
            If act(i,3) Mod 10 > m Then   ' entering a new tile left
              x = act(i,3)\10 : y = act(i,4)\11   ' coords of tile entered
              Select Case crlv(x,y)
                Case "1","2","5"    ' brick, solid or trap
                  TurnLR(i)   ' turn around
                  act(i,3) = (x+1) * 10
                  Continue For
                Case "4"      ' monkey bar
                  act(i,0) = 4  ' change to monkey mode
                  Continue For
                Case "0"    ' empty space - test what's below
                  If (y < 15) Then
                    Select Case crlv(x,y+1)
                      Case "1","2","3"    ' floor or ladder
                        act(i,0) = 2  ' change to run mode
                        Continue For
                      Case Else     ' non-solid
                        TurnLR(i)   ' turn around
                        act(i,3) = (x+1) * 10
                        Continue For
                    End Select
                  EndIf
                  act(i,0) = 2  ' change to run mode
                  Continue For
                Case "3"    ' ladder
                  act(i,0) = 2  ' change to run mode
                  Continue For
              End Select
            EndIf
          EndIf
        Case 5    ' climb up
        Case 6    ' climb down
        Case 7    ' fall right
        Case 8    ' fall left
        Case 11   ' spawn
      End Select
      EndIf
    Next i
End Sub
    
Sub TurnLR(n)
    If act(n,0) Mod 2 Then  ' moving right
      Inc act(n,0)  ' turn monk around
    Else
      Inc act(n,0),-1  ' turn monk around
    EndIf
End Sub
    
    ' Decrement timer for any active holes
Sub FillHoles
    Local i
    
    For i = 0 To NUMHOLES
      If holes(i,0) Then
        Inc holes(i,0),-1
        If holes(i,0) = 0 Then
          ' fill brick back into map - where? holes(i,3/4)
          crlv(holes(i,3),holes(i,4)) = "1"
        Else
          crlv(holes(i,3),holes(i,4)) = "0"
        EndIf
      EndIf
    Next
End Sub
    
    ' Return 1 if the tile (x,y) is contained in the string a$
Function Ins(a$,x,y) As integer
    Ins = (Instr(a$,crlv(x,y)) <> 0)
End Function
    
    ' Draw all kinds of stuff on the hidden layer, then copy to the overlay
    ' Dug holes and hole blasting cover over the background
    ' Monks and the player show the background behind them
Sub DrawMotion
    Local i
    FRAMEBUFFER Write F
    CLS RGB(lilac)  ' clear hidden to overlay's transparent colour
    
    ' draw barrels? - only placed in open space, 52 barrels on level 87
    
    ' draw holes - main layer must be covered over
    For i = 0 To NUMHOLES
      If holes(i,0) Then
        If holes(i,0) < 10 Then ' draw fill 2
          'If holes(i,0) = 14 then Print "Fill 2"
          Sprite memory SPRADDR+2520,holes(i,1),holes(i,2),-1
        ElseIf holes(i,0) < 20 Then ' draw fill 1
          'If holes(i,0) = 29 then Print "Fill 1 #######################"
          Sprite memory SPRADDR+2460,holes(i,1),holes(i,2),-1
        Else  ' draw blank
          Sprite memory SPRADDR,holes(i,1),holes(i,2),-1
        EndIf
      EndIf
    Next i
    
    ' draw actors - main layer must show through
    DrawActors  ' on layer F
    
    FRAMEBUFFER copy F,L,b  ' copy hidden to overlay during vbl
End Sub
    
    ' Draw player and monks on active layer
    ' Special draw if player is burning a hole
Sub DrawActors
    Local i,s,x,y,xb
    
    For i = 0 To NUMMONK
      If i = 0 And burn <> 0 Then   ' test if player is firing
        s = (burn < 0)  ' side to burn 0=right, 1=left
        x = 20+act(0,3)
        y = 21+act(0,4)
        xb = x + Sgn(burn) * 10
        Select Case Abs(burn)
          Case 1,2
            Sprite memory SPRADDR+(9+3*s)*60,x,y,0 ' run     (transparent)
            Sprite memory SPRADDR+32*60,xb,y+11    ' burn it down (overwrite)
          Case 3,4
            Sprite memory SPRADDR+(9+3*s)*60,x,y,0 ' run    (transparent)
            Sprite memory SPRADDR+31*60,xb,y+11    ' burn it down (overwrite)
          Case 5,6
            Sprite memory SPRADDR+(25+s)*60,x,y,0  ' fire     (transparent)
            Sprite memory SPRADDR+(36+4*s)*60,xb,y ' spray 4  (overwrite)
            Sprite memory SPRADDR+30*60,xb,y+11    ' burn it down (overwrite)
          Case 7,8
            Sprite memory SPRADDR+(25+s)*60,x,y,0  ' fire     (transparent)
            Sprite memory SPRADDR+(35+4*s)*60,xb,y ' spray 3  (overwrite)
            Sprite memory SPRADDR+29*60,xb,y+11    ' burn it down (overwrite)
          Case 9,10
            Sprite memory SPRADDR+(25+s)*60,x,y,0  ' fire     (transparent)
            Sprite memory SPRADDR+(34+4*s)*60,xb,y ' spray 2  (overwrite)
            Sprite memory SPRADDR+28*60,xb,y+11    ' burn it down (overwrite)
          Case 11,12
            Sprite memory SPRADDR+(25+s)*60,x,y,0  ' fire     (transparent)
            Sprite memory SPRADDR+(33+4*s)*60,xb,y ' spray 1  (overwrite)
            Sprite memory SPRADDR+27*60,xb,y+11    ' burn it down (overwrite)
        End Select
        Continue For
      EndIf
      If act(i,0) Then  ' actor exist so draw him/her
        If act(i,0) = 17 Or act(i,0) = 18 Then  ' landed after falling
          s = act(i,0) + 6    ' 23,24
        Else
          s = soff(act(i,0)) : If i Then Inc s,34  ' locate base sprite locn
          Inc s,act(i,5)  ' add pose number
        EndIf
        s = SPRADDR + s * 60
        Sprite memory s,20+act(i,3),21+act(i,4),0  ' black is transparent
      EndIf
    Next i
End Sub
    
    ' Read requested level from the level file
    ' This is fast enough to allow scanning from the beginning
    ' Bottom (invisible) layer of solid floor for data purposes
Sub ReadLevel(n)
    Local c,i,j
    Local b$
    Open LEVNAME$ For input As #1
    c = (n-1) * 17 + 1
    For i = 1 To c  ' read and ignore many lines
      b$ = Input$(14,#1)
    Next i
    For j = 0 To 15
      b$ = Input$(14,#1)  ' read one row of level
      For i = 0 To 13
        v = Str2bin(UINT8,Mid$(b$,i+1,1))
        crlv(0+i*2,j) = Mid$(ASHX$,1+v\16,1)
        crlv(1+i*2,j) = Mid$(ASHX$,1+v Mod 16,1)
      Next i
    Next j
    For i = 0 To 27 : crlv(i,16) = "2" : Next
    Close #1
End Sub
    
    ' open or close the iris
    ' d=0 to close, else open
    ' t=speed (ms delay between circles)
Sub Iris(d,t)
    Local i
    FRAMEBUFFER Write L
    If d Then ' open iris
      For i = 2 To 166  ' expanding circles of transparency
        Circle MM.HRes/2,MM.VRes/2-15,i,2,,RGB(lilac)
        Pause t
      Next i
    Else  ' close iris
      FRAMEBUFFER COPY n, L, b  ' copy display to front layer
      Box 19,19,282,179,1,RGB(lilac),RGB(lilac) ' transp playing area
      FRAMEBUFFER Write n ' erase back layer
      For i = 166 To 2 Step -1  ' shrinking circles of opaque
        Circle MM.HRes/2,MM.VRes/2-13,i,2,,0
        Pause t
      Next i
      CLS ' back layer
      DrawTitleBox  ' redraw on back layer
      FRAMEBUFFER Write L 'F
      CLS RGB(lilac)         ' clear all to transparent
      Box 19,19,282,179,1,0,0 ' hide playing area
      'FRAMEBUFFER COPY F, L, B
    EndIf
    FRAMEBUFFER Write n
End Sub
    
    ' Draw the play area frame and bottom text
    ' on hidden layer, then copy to main layer.
Sub DrawTitleBox
    FRAMEBUFFER Write F
    CLS
    Box 14,14,292,189,5,RGB(cobalt),0
    Text 70,206,"SCORE",RT,9,,RGB(cobalt)
    Text 72,206,Str$(scor,7,0,"0"),LT,9,,RGB(brown)
    Text 180,206,"MEN",RT,9,,RGB(cobalt)
    Text 182,206,Str$(men,3,0,"0"),LT,9,,RGB(brown)
    Text 270,206,"LEVEL",RT,9,,RGB(cobalt)
    Text 272,206,Str$(lev,3,0,"0"),LT,9,,RGB(brown)
    FRAMEBUFFER COPY F, n, b
    FRAMEBUFFER Write n
End Sub
    
Sub DrawScore
    FRAMEBUFFER Write n
    Text 72,206,Str$(scor,7,0,"0"),LT,9,,RGB(brown)
End Sub
    
    ' Change all exit ladder tiles (6) to normal ladders(3)
    ' and display them on normal layer.
    ' For aesthetic reasons, work bottom to top.
Sub ShowExitLadders
    Local x,y
    
    FRAMEBUFFER Write n
    For y = 15 To 0 Step -1
      For x = 0 To 27
        If crlv(x,y) = "6" Then  'end ladder
          crlv(x,y) = "3"
          Sprite memory SPRADDR+3*60,20+x*10,21+y*11
        EndIf
      Next x
    Next y
End Sub
    
    ' Draw current level
    ' If e=1 then show in edit mode
Sub DrawCurLevel(e)
    Local x,y
    Local string t length 1
    
    If e Then e = 1 ' only 0 and 1 are valid
    For y = 0 To 15
      For x = 0 To 27
        t = crlv(x,y)
        If t < "0" Or t > "9" Then t = "0"
        If (t = "6") And Not e Then t = "0" ' hide level end ladder
        If (t = "5") And Not e Then t = "1" ' hide trap door
        Sprite memory SPRADDR+Val(t)*60,20+x*10,21+y*11
      Next x
    Next y
End Sub


sub init_nunch
  I2C write nunch,0,2,&h40,&h00 'reset the unit
  pause 10
  I2C write nunch,0,2,&hF0,&h55 'program unencrytpted output
  I2C write nunch,0,2,&hFB,&h00  
end sub

function read_nunch(a)
  local d(5)
  I2C write nunch,0,1,0 'address 0
  pause 2
  I2C read nunch,0,6,d()
  read_nunch = d(a)
  if a=5 then read_nunch = read_nunch and 3
  ? a,read_nunch
end function

    
    ' Computerish font like the original
DefineFont #9
    40200C0A
    00000000 00000000 00000000 00000000 00000000 00000000 00000000 00000000
    00000000 00000000 00000000 00000000 00000000 00000000 00000000 00000000
    00000000 00000000 00000000 00000000 00000000 00000000 00000000 00000000
    00000000 00000000 00000000 00000000 00000000 00000000 F0C00300 030F3C30
    030CF0C0 0000F0C0 30C0030F F0C0030F 030F0C3C 000000C0 00000000 00000000
    00000000 00000000 00000000 00000000 00000000 00000000 00000000 07000000
    0018E081 00000000 F0C30F00 00000000 00000000 00000000 00000000 E0810700
    00000000 00000000 00000000 00000000 19E61F00 66986186 99679E79 0000F8E7
    30C0030F C000030C C30F0C30 1F0000F0 010618E6 8001E69F F8E71960 E61F0000
    87010618 010618E0 00F86798 99671E00 E09F79E6 80010618 00001860 8001E61F
    78E01F60 E781071E 1F0000F8 608019E6 9EF90718 F8E79967 E01F0000 81071E78
    1860E0E1 00800106 98E10700 E69F1966 98618619 0000F867 8619E61F 78E09F61
    E081071E 01000078 06186080 78000000 0080071E 80010000 00061860 1E780000
    10608007 18000000 071E1E1E 80077880 00000060 FEF90700 00000000 E09F7F00
    00000000 1EE00106 7878E001 00001878 E61F0000 81010618 070078E0 0078E081
    09E61F00 A69A6E82 1960BEA9 0000F8E7 6698E107 19E69F19 E6996186 1F000078
    66986186 8619E61F F8679861 E61F0000 18608019 78E08107 00F8671E 19861F00
    67986186 9E79E699 0000E067 E081E71F 01861F78 07186080 1F0000F8 78E081E7
    8001861F 00061860 E61F0000 18608019 609E7906 00F82798 19661800 E79F6186
    9E79E699 00009867 60800106 E0810718 81071E78 010000E0 06186080 1E78E001
    F8E79907 66180000 1F669879 79E6F987 0098679E 01061800 07186080 1E78E081
    0000F807 FE396718 19669B7F 679E7986 1C000098 6FF69967 8639E699 98679E79
    E61F0000 98679E79 61861966 00F86798 19E61F00 67986186 1E78E0F9 00008007
    9E79E61F 19669867 87996186 1F000098 410619E4 F881E79F 98671E7E E61F0000
    1F608019 071E78E0 00F8E799 80E11F00 01061860 071E78E0 0000E081 86196618
    99679861 679E79E6 1E0000F8 79E69967 E699679E 8081877F 671C0000 9D71C619
    3FFE5967 0030C30C 19661800 81876186 986186E1 00001866 9E79E619 E0E19F67
    81071E78 1F0000E0 068C19E6 C0800103 F8E79967 C30F0000 0C30C000 3CF0C003
    00F0030F 01060000 000738C0 80031CE0 00000060 0C30C00F F0C00003 C3030F3C
    030000F0 61CCE001 00000080 00000000 00000000 00000000 00000000 FE010000
End DefineFont
    
    ' Sprites and Tiles, stored as CSUB data. No compression
    ' 61 in total, 60 bytes each
    ' All tiles are 10 wide x 11 high
    ' Colour 'E' to change the ladders to yellow (3rd sprite)
CSub Lode
    00000000 000B000A 00000000 00000000 00000000 00000000 00000000 00000000
    00000000 00000000 00000000 00000000 00000000 00000000 00000000 00000000
    000B000A 00333333 33333333 33333300 33330033 33003333 00000000 33003300
    00333333 33333333 33333300 33330033 33003333 00003333 00000000 000B000A
    33333333 33333333 33333333 33333333 33333333 33333333 33333333 33333333
    33333333 33333333 33333333 33333333 00003333 00000000 000B000A F0000FF0
    000FF00F FFF00FF0 F00FFFFF 0FF0000F F0000FF0 000FF00F 0FF00FF0 F00FF000
    0FFFFFFF F0000FF0 000FF00F 0FF00FF0 000FF000 000B000A 00000000 FFFFFF00
    0000FFFF 00000000 00000000 00000000 00000000 00000000 00000000 00000000
    00000000 00000000 00000000 00000000 000B000A 33333333 33333333 00003333
    00000000 00FFFFFF 00FF0000 FF000000 00000000 000000FF 0000FF00 33333333
    33333333 00003333 00000000 000B000A 00000FF0 000FF000 FFF00000 F00FFFFF
    0FF0000F F0000000 0000000F 00000FF0 F00FF000 0FF0000F FFFFFFF0 000FF00F
    0FF00000 00000000 000B000A 00000000 00000000 00000000 00000000 00000000
    00000000 FFF00000 C00000FF 0000CCCC 00FFFFF0 CCCCC000 FFF00000 000000FF
    00000000 000B000A 00030000 88800000 80000000 00000088 00008000 88888000
    80088000 00008800 00000880 000FFF00 FF0FF000 0FF000FF F0000000 0000000F
    000B000A 00300000 FF000000 0000000F 00000FFF 0000FFF0 0FFF0FF0 0FF0FF00
    F0000FF0 0000000F 0000FFF0 0FF0FFFF F0000000 0000000F 00000FF0 000B000A
    00300000 FF000000 0000000F 00000FFF 0000FF00 00FFF000 FFFF0000 FF0000FF
    000F00FF 000FFF00 0FFFF000 0FFF0000 F0000000 0000000F 000B000A 00300000
    FF000000 0000000F 00000FFF 0000FF00 F0FFFF00 FF0FF000 000000FF 000000FF
    000FFFF0 FF00FF00 000FF000 0FF000FF 00000000 000B000A 00030000 FFF00000
    F0000000 000000FF 000FFF00 F0FFF000 F00FF00F 0000FF0F 00000FF0 000FFF00
    FF0FF000 0FF000FF F0000000 0000000F 000B000A 00030000 FFF00000 F0000000
    000000FF 0000FF00 FFFF0F00 FFFF0000 00000FF0 000000FF 000FFFF0 FF00FF00
    00FF0000 00000FF0 000FF000 000B000A 00030000 FFF00000 F0000000 000000FF
    0000FF00 0FFF0000 FFFF0000 00F000FF 0000FFFF 0000FFF0 0FFFF000 F0000000
    000000FF 00000FF0 000B000A F0000FF0 0000000F 0FF00000 F00FF0FF 00FFFF0F
    00FFFF00 0FF00000 F0000000 F000000F 00000FFF 000F00FF 0FF0FF00 FF0F0000
    00000000 000B000A F0000000 0000000F 00000000 000FFFF0 00FFF000 0FFFF000
    F0FF0000 0000000F 00000FF0 000FFF00 FF0FF000 0FF00000 F00000FF 0000FF0F
    000B000A 0000FF00 00000000 FF000000 000000FF 0FF0FFF0 FFFF0000 FF000000
    00000000 000000FF 0000FFF0 0FF0FF00 F0FF0000 FF00000F 00000FF0 000B000A
    F0000FF0 0000000F 0FF00000 000FF0FF 0FF0FFFF FFFF0000 F0000000 0000000F
    00000FF0 0FFFF000 00F00000 F00000FF 0000FF0F 00F0FF00 000B000A FF000000
    00000000 00000000 F000FFFF 000FFF0F 00FFFF00 FF000000 00000000 000000FF
    000FFF00 FF0FF000 0FF00000 F00000FF 0000FF0F 000B000A 00000FF0 00000000
    FFF00000 0000000F 00000FFF 0FFFF000 0FF00000 F00000FF 0000000F 0000FFF0
    0FF0FF00 F0FF0000 FF00000F 00000FF0 000B000A 00FF0000 FF000F00 FFFF0000
    000000FF 0F00FFF0 FFFFF000 FFF0000F F0000000 000000FF 000FF0FF 0FF0FFF0
    F0000000 0000000F 0000FFF0 000B000A 00FF0000 FF000000 0000F000 F0FFFFFF
    000FFF00 0FFFFFF0 FF000000 0000000F 00000FFF 00FF0FF0 FF0FF000 0FF0000F
    FF000000 0000000F 000B000A F00300FF FFF0FF0F F0FF0FF0 F00FF0FF 00FFFFFF
    000FF000 0FF00000 F0000000 00000FFF 00FF0FF0 FF0FF000 0FF00000 F00000FF
    0000000F 000B000A 00300FF0 FF0FF0FF 0FF0FF0F 00FF0FFF 0FFFFFFF 0FF00000
    F0000000 F000000F 00000FFF 000FF0FF 0FF0FF00 F0FF0000 0000000F 00000FF0
    000B000A 00030000 FFF00000 F0000000 000000FF 00000FF0 0FFFFFF0 0FF00000
    F000EEFF 00E0000F 0000FFF0 0FF0FF00 F0FF0000 FF00000F 00000FF0 000B000A
    00300000 FF000000 0000000F 00000FFF 000FF000 FFFFF000 F0FFEE0F 000E000F
    00000FF0 000FFF00 FF0FF000 0FF00000 F00000FF 0000FF0F 000B000A 0C003333
    33333333 33333300 33330033 33003333 00000000 33003300 00333333 33333333
    33333300 33330033 33003333 00003333 00000000 000B000A 0C000000 CC000000
    0033000C 33330000 33003333 00000000 33003300 00333333 33333333 33333300
    33330033 33003333 00003333 00000000 000B000A 00000000 C0000000 C0000000
    330000CC 3300CCC0 00000000 33003300 00333333 33333333 33333300 33330033
    33003333 00003333 00000000 000B000A 00000000 00000000 00000000 00000000
    0000C000 00CCC000 CCC00000 000000CC 33000000 33333300 33330033 33003333
    00003333 00000000 000B000A 00000000 00000000 00000000 00000000 00000000
    00000000 C0000000 00000000 3300CCC0 00CCCCC0 00000033 33003300 00003333
    00000000 000B000A 00000000 00000000 00000000 00000000 00000000 00000000
    00000000 00000000 00000000 0000C000 00CCC000 CCC00000 000000CC 00000000
    000B000A 00000000 00000000 00000000 00000000 00000000 00000000 00000000
    000E0000 E0000030 00300030 00300E00 00E00000 00000300 000000EE 000B000A
    00000000 00000000 00000000 00000000 00003000 00000000 00000000 330E3000
    E0000000 00000000 33300E00 00E00000 00000000 0000000E 000B000A 00030000
    00000000 00000000 00000000 00000000 00000000 03000300 00000000 03030000
    00000030 00000E00 00E00000 00000003 0000000E 000B000A 00000000 00000000
    00000000 00000000 00000300 00000000 00003000 00000000 00030000 00000000
    00000000 00000000 00000000 00000000 000B000A 00000000 00000000 00000000
    00000000 00000000 00000000 00000000 00000000 00E00003 0E030003 E0030000
    00003000 0000000E 000000EE 000B000A 00000000 00000000 00000000 00000000
    00000300 00000000 00000300 00000000 00E03300 0E000000 E0033300 00000000
    0000000E 000000E0 000B000A 00300000 00000000 00000000 00000000 00000000
    00000000 30000000 00303000 00000000 30030000 E0000000 00300000 0000000E
    000000E0 000B000A 00000000 00000000 00000000 00000000 00003000 00000000
    00000000 00300300 00000000 00000000 00000000 00000000 00000000 00000000
    000B000A 00000000 00000000 00000000 00000000 00000000 00000000 00000000
    00000000 00000000 00000000 00000033 00003333 00003300 00000000 000B000A
    00000000 00000000 00000000 00000000 00000000 00000000 00000000 00330000
    33330000 33000000 00000033 00003333 00003300 00000000 000B000A 00300000
    88000000 00000008 00000888 00000800 08888800 08008800 80000880 00000008
    0000FFF0 0FF0FFFF F0000000 0000000F 00000FF0 000B000A 00300000 88000000
    00000008 00000888 00000800 80088800 88880000 00000088 000000FF 000FFF00
    0FFFF000 0FFF0000 F0000000 0000000F 000B000A 00300000 88000000 00000008
    00000888 00000800 08888800 88080000 00000008 000000FF 000FFFF0 FF00FF00
    000FF000 0FF000FF 00000000 000B000A 00030000 88800000 80000000 00000088
    00008000 88888000 80088000 00008800 00000880 000FFF00 FF0FF000 0FF000FF
    F0000000 0000000F 000B000A 00030000 88800000 80000000 00000088 00008000
    88888000 88800000 00000080 000000FF 000FFFF0 FF00FF00 00FF0000 00000FF0
    000FF000 000B000A 00030000 88800000 80000000 00000088 00008000 88800800
    88880000 00000088 000000FF 0000FFF0 0FFFF000 F0000000 000000FF 00000FF0
    000B000A 00000800 00000008 08000000 00088800 08880088 08880000 08000000
    00000000 00000008 0000FFFF 00F00FF0 FF0FF000 F0000000 0000000F 000B000A
    08000000 00000000 00000000 00000888 00088800 00088800 08880000 00000000
    00000008 0000FFF0 0FF0FF00 F0FF0000 0000000F 00000FF0 000B000A 00000800
    00000000 88000000 00000008 08800800 08880000 08000000 00000000 00000008
    0000FFF0 0FF0FF00 F0FF0000 FF00000F 00000FF0 000B000A 80000080 00000000
    88800000 80008000 00880088 00888000 80000000 00000000 00000080 00FFFF00
    F00F0000 FF00000F 00000FF0 00000FF0 000B000A 80000000 00000000 00000000
    80008880 00008008 00888000 80000000 00000000 00000080 000FFF00 FF0FF000
    0FF00000 F00000FF 0000FF0F 000B000A 00008000 00000000 80000000 00000088
    00008880 88800000 80000000 00000088 00000080 000FFF00 FF0FF000 0FF00000
    F00000FF 0000000F 000B000A 00080000 08000800 88880000 00000008 08000800
    88880000 08000008 F0000000 000000FF 000FF0FF 0FF0FF00 F0000000 0000000F
    00000FF0 000B000A 00800000 80000000 00008000 80888880 00008000 00888880
    80000000 00000000 00000FFF 00FF0FF0 FF0FF000 0FF00000 F0000000 0000000F
    000B000A 00080008 08000808 00080800 80080008 00888888 00080000 08000000
    F0000000 00000FFF 00FF0FF0 FF0FF000 0FF00000 F00000FF 0000000F 000B000A
    00800080 80008080 00808000 00800080 08888888 00800000 80000000 F0000000
    00000FFF 000FF0FF 0FF0FF00 F0FF0000 0000000F 00000FF0 000B000A 00000000
    00000000 00000000 00000000 00000000 00000000 00000000 00000000 00000000
    00000000 00000000 88800000 88800000 00008888 000B000A 00000000 00000000
    00000000 00000000 00000000 00000000 00000000 00000000 00000000 00000000
    00888000 88888000 88800088 00008888
End CSub
    ' end
