  'MLX90641 IR camera support with up-scaling 4x
  
  'requires V5.90.00RC5 with M_MULT fix, and scalars w and h for BOX command
  'to execute BOX x(),y(),w,h,1,c(),c()
  'Use COLOUR MAP a(),b(),color%()
  'V18 has only 2kb RAM free in PicoMite LCD
  
  '-------------------- generic program defines ------------------------
  
  const SA=&h33             'I2C 7 bit slave address
  I2Cport=1                 'I2C2=2 anything other is I2C
  num_colors=15             '6,10,14,15,16 color grades in video interface
  dim i,x,y as integer      'in stead of replacing all i with i%
  
  
  '-------------------------system initialisation ----------------------
  
  vga%=instr(mm.device$,"VGA")          '0 or not 0
  if vga% then
    mode 2
    cls
  else
    font 7
    cls
    framebuffer create     'use framebuffer when LCD faster screen
    framebuffer write F    'use framebuffer when LCD faster screen
  endif

  
  
  'I2C bus open at 400kHz
  if I2Cport=2 then
    'assume system I2C on GP14/GP15, else uncomment
    'setpin gp14,gp15,I2C2
    'I2C2 open 400,100
  else
    setpin gp0,gp1,I2C
    I2C open 400,100
  end if
  
  
  'arrays with calibration values per pixel
  dim Po1(191)  'offset subbank 1
  dim Po0(191)  'offset subbank 0
  dim Pa(191)   'alpha
  dim Pkt(191)  'temperature impact
  dim Pkv(191)  'voltage impact
  dim Vir(191)  'voltage from sensor element and after conversion temperature
  dim Toffs(191):math set 2,Toffs()'Temperature offsets for manual calibration
  
  'debugging how much memory left
  'dim vzz(250) '2kb free on PicoMite (non-VGA)
  
  'define the colors
  dim col%(15)              'always need assign 15 colors, default=0 (black)
  if num_colors=6 then restore col6           'colors
  if num_colors=10 then restore col10         '10 colors
  if num_colors=14 then restore col14         '14 colors
  if num_colors=15  then restore col15        '16 colors  
  if num_colors=16  then restore col16        '16 colors  
  for i=0 to num_colors-1:read col%(i):next   'fill the array
  
  
  'scaling buffers and math matrices
  dim f_t(191),f_v(191)                     'scratchpad temp/voltage
  size%=5                                   'pixel size
  dim Toa(15,11)                            'this is Vir() in 2D form for math
  dim c(15,44)                              'array for intermediate results math
  dim d(11,44),b_e(30,15),b_o(30,15)        'conversion arrays for scaling
  dim x_s%(1394),y_s%(1394)                 'arrays for screen coordinates
  fill_matrix                               'fill conversion matrices
  dim scrn_2D(30,44)                        '2D buffer for math
  dim scrn_1De(1394),scrn_1Do(1394)         'even/odd 1D arrays for 61x45 screen
  
  'memory 'debug
  
  '--------------------------- read camera constants -------------------
  'this reads the EEPROM inside the sensor, to retrieve and calculate
  'calibration values.
  get_res_corr
  dummy=get_vdd()
  get_cal
  get_Kta_Kv
  get_Po0_Po1
  get_Pa
  get_Vir
  get_conf
  get_stat
  
  
  '---------------------------- screen setup -----------------------------
  
  'legend text boxes that are static
  box 0,0,60,15,1,rgb(white),rgb(black)
  box 65,0,60,15,1,rgb(white),rgb(black)
  box 296,0,24,15,1,rgb(white),rgb(black)
  circle 75,7,5,,,rgb(pink)                              'pink cirle 
  
  'place the bottom colour legend on screen
  box 130,0,24,15,1,rgb(white),rgb(white)
  box 154,0,142,15,1,rgb(white),rgb(black)
  h_size = 142 \ num_colors
  for i=0 to num_colors-1
    box 155+i*h_size,1,h_size,13,1,col%(num_colors-1-i),col%(num_colors-1-i)
  next

  
  '-------------------------- MAIN LOOP -------------------------------
  'this is based on MLX90641 datasheet chapter 10.5 without STEP mode
  'the MLX90641 runs autonomous at 500ms (2Hz) default speed
  
  do
    
    'wait for new data by checking data available flag
    do
      pause 10
      status% = get_reg(&h8000)
    loop until (status% and 8) = 8
    
    
    'get data from active sub-page and confirm
    page% = (status% and 1)   'what page
    get_vir(page%)            'read page
    set_reg(&h8000,page%)     'confirm read
    
    'convert IR readings to temperature according datasheet
    temp_conv
    
    
    'check if we need to re-calibrate
    if a$="c" then cal_array
    
    'calibrate the measured values to the reference
    math c_sub Vir(),Toffs(),Vir()
    inc vir(15),0.1  'to avoid math on 0 delta between pixelz
    
    'calculate extremes for mapping colors and legend
    Tmax=math(max Vir(),pix%):Tmin=math(min Vir())
    Tlow=min(tmin,Ta-2):Thigh=max(tmax,Tlow+10)
    
    
    'show the colorfull picture on screen
    scale_4d
    
    'place the marker's on screen
    text 20*(pix% mod 16)+6,20*(pix%\16)+13,"X"               'cross
    circle 159,117,10,,,rgb(pink)                             'ring
    
    'show the text that does not need immediate refresh
    text 134,4,str$(Thigh,3,0),,,,rgb(black),rgb(white)    'max scale temp
    text 300,4,str$(Tlow,3,0),,,,rgb(white),rgb(black)     'min scale temp
    text 10,4,"X "+str$(tmax,3,1),,,,rgb(white),rgb(black) 'highest temp
    text 85,4,str$(Vir(88),3,1),,,,rgb(white),rgb(black)   'centre temp
    
    'debug
    text 0,229,str$(timer,3,0):timer=0
    
    'fast copy of framebuffer to screen
    if vga%=0 then framebuffer copy f,n
    
    
    'do we want to save the current image ?
    if a$="s" then save image "screenshot.bmp"
    a$=inkey$
    
  loop while a$<>"q"
  
end
  
sub temp_conv
  
  'process data according 11.2.2 from datasheet
  Vdd=get_vdd()                     'get Vdd 11.2.2.2
  Ta=get_ta()                       'get Ta 11.2.2.3
  
  math scale vir(),Kgain,vir()      'scale gain 11.2.2.5.1
  
  'IRdatcomp 11.2.2.5.3
  math scale pkt(),Ta-25,f_t()      'calc 1+Kta*(Ta-T25)
  math add f_t(),1,f_t()
  math scale pkv(),vdd-3.3,f_v()    'calc 1+Kv*(Vdd-3.3)
  math add f_v(),1,f_v()
  math c_mul f_v(),f_t(),f_v()      'multiply and result in f_v()
  if page% then                     'compensate for page offset
    math c_mul f_v(),Po1(),f_v()
  else
    math c_mul f_v(),Po0(),f_v()
  end if
  math c_sub Vir(),f_v(),Vir()      'and add to pixel voltage
  
  'emissivity comp 11.2.2.5.4 and IR gradient comp 11.2.2.7
  math add Vir(),-tgc*cp_pix_os,Vir()
  math scale Vir(),1/em,Vir()             'this is Vir_compensated()
  
  'Sensitivity normalize 11.2.2.8 and put values in f_t() for temp conversion
  math add Pa(),-tgc*alpha_cp,f_t()
  math scale f_t(),1+ksta*(Ta-25),f_t()   'this is alpha_comp()
  
  
  'Temp conversion (0-80C) 11.2.2.9
  Tark4=calc_tark()
  'since there are no math commands for square root and raise to the power
  'this is still in a loop, needs 55ms @ 252MHz (all above math is only 10ms)
  for i=0 to 191
    Sx=KsTo_3*( ( Vir(i)*(f_t(i)^3) + (f_t(i)^4)*Tark4 )^(1/4) )
    Vir(i)=((tark4 + Vir(i)/(f_t(i)*(1-KsTo_3*273.15)+Sx))^(1/4))-273.15
  next
  
end sub
  
  '--------------------------- display routines -------------------------------
  'this routines perform the visual representation of the data in Vir()
  
  'show 192 temperatures on screen in a grid of 61x45 interleaved
sub scale_4d
  
  local y31,px,cs0,cs1,x,y as integer
  
  
  'copy linear array into 16x12 array, and perform math to scale up (=21ms)
  MEMORY COPY INTEGER PEEK( VARADDR Vir()),PEEK( VARADDR Toa()),192
  
  math m_mult d(),Toa(),c()           'c() is in correct vertical resolution
  
  'with low delta-T limit the color scaling for visual appearance
  max_colors = min(int(Tmax-Tmin)+2,num_colors-1)
  max_colors = 15
  math window c(),0,max_colors,c()  'x color
  'math window c(),0,max_colors+0.4,c()  'x colors + 0.4c for better distribution
  
  
  'calculate even pixels
  math m_mult c(),b_e(),scrn_2D()
  
  'flatten into 1D array
  MEMORY COPY float PEEK( VARADDR scrn_2D()),PEEK( VARADDR scrn_1De()),1395
  
  'convert to RGB
  colour map scrn_1De(),scrn_1De(),col%()
  
  
  'calculate odd pixels
  math m_mult c(),b_o(),scrn_2D()
  
  'flatten into 1D array
  MEMORY COPY INTEGER PEEK( VARADDR scrn_2D()),PEEK( VARADDR scrn_1Do()),1395
  
  'convert to RGB
  colour map scrn_1Do(),scrn_1Do(),col%()
  
  'plot all the boxes on screen, since VGA does not use a framebuffer
  'the plotting is done as fast and compact as possible. That is why there
  'are dual 1D screen buffers.
  
  '31 even boxes, original x() coordinates
  box x_s%(),y_s%(),size%,size%,1,scrn_1De(),scrn_1De()
  math add x_s%(),size%,x_s%() 'prepare for odd boxes
  
  '30 odd boxes
  box x_s%(),y_s%(),size%,size%,1,scrn_1Do(),scrn_1Do()
  math add x_s%(),-size%,x_s%() 'restore for next even boxes
  
end sub
  
  
  'determine the offset of each pixel from ambient. Requires camera
  'is pointed at surface that is Ta. I.e. metal block covering lens.
sub cal_array
  math add Vir(),-Ta,Toffs()
end sub
  
  
  
  ' ------------------- get pixel cal data from the chip ----------------------
  'these functions read data from the MLX90641 and do some formatting
  
  
  'get status register
sub get_stat
  Reg8000%=get_reg(&h8000)  'get value
end sub
  
  'get config register
sub get_conf
  Reg800D%=get_reg(&h800D)  'get value
end sub
  
  
  'get voltages for the pixel array from subpage n (0 or 1)
sub get_vir n
  local i,j,yoff,xoff as integer
  
  xoff=n*&h20
  
  for j=0 to 5      'all 6 banks
    yoff=j*&h40
    for i=0 to 31   'of 32 pixels
      'get value
      Vir(i+j*32)=get_reg_16_2(&h400 + i + yoff + xoff)
    next
  next
  
end sub
  
  
  'get alpha(i) and adjust for row (0-5)
sub get_Pa
  local i as integer
  
  'adjust per row (32 pix) according 11.2.2.8
  for i=0 to 31
    Pa(i)=a_ref_row_1*get_reg_11(&h2500+i)/2047
  next
  for i=32 to 63
    Pa(i)=a_ref_row_2*get_reg_11(&h2500+i)/2047
  next
  for i=64 to 95
    Pa(i)=a_ref_row_3*get_reg_11(&h2500+i)/2047
  next
  for i=96 to 127
    Pa(i)=a_ref_row_4*get_reg_11(&h2500+i)/2047
  next
  for i=128 to 159
    Pa(i)=a_ref_row_5*get_reg_11(&h2500+i)/2047
  next
  for i=160 to 191
    Pa(i)=a_ref_row_6*get_reg_11(&h2500+i)/2047
  next
  
end sub
  
  'get Kta(i) and Kv(i) and scale them to be used for math
sub get_Kta_Kv
  local x,i as integer
  
  for i=0 to 191 'all 192 pixels
    x=&h25c0+i
    
    'get value
    Pkt(i)=(get_reg(x) and &h7e0)/32
    if Pkt(i)>31 then inc Pkt(i),-64
    
    'scale
    Pkt(i)=(Pkt(i)*(2^kta_scale_2) + kta_avg) / (2^kta_scale_1)
    
    'get value
    Pkv(i)=get_reg(x) and &h1f
    if Pkv(i)>15 then inc Pkv(i),-32
    
    'scale
    Pkv(i)=(Pkv(i)*(2^kv_scale_2) + kv_avg) / (2^kv_scale_1)
  next
  
end sub
  
  'get pixel offset(i) for 0 and 1 compensated as in 11.2.2.5.2
sub get_Po0_Po1
  local x,i as integer
  
  for i=0 to 191 'all 192 pixels
    x=&h2440+i
    
    'get value
    Po0(i) = get_reg_11_2(x)
    
    'scale
    Po0(i) = (Po0(i) * 2^scale_os_r1) + pix_os_av
  next
  
  for i=0 to 191 'all 192 pixels
    x=&h2680+i
    
    'get value
    Po1(i) = get_reg_11_2(x)
    
    'scale
    Po1(i) = (Po1(i) * 2^scale_os_r2) + pix_os_av
  next
  
end sub
  
  'determine Tambient
function get_Ta()
  local dv,vbe,vptat,vptat_art as float
  
  dv = get_vdd()-3.3
  vbe = get_reg_16_2(&h580)
  vptat = get_reg_16_2(&h5a0)
  vptat_art = (vptat / (vptat*alpha_ptat + vbe)) * 2^18
  
  'formula from datasheet 11.1.2
  get_Ta = (((vptat_art/(1+kv_ptat*dv))-ptat)/kt_ptat)+25
  
end function
  
  
function calc_tark()
  Tr=Ta-5                               'from datasheet in absence of better
  local tvk4=(Tr+273.15)^4
  calc_tark=tvk4-(tvk4-(Ta+273.15)^4)/em
end function
  
  
  
  ' ------------------- get specific data from the chip ------------------------
  'these functions read data from the MLX90641 and do some formatting
  
  
  'resolution correction from the ADC
sub get_res_corr
  RC = (get_reg(&h2433) and &h600)>>9
  RC = RC/((get_reg(&h800D) and &hC00)>>10)
end sub
  
  
  'actual Vdd measured by the chips ADC
function get_vdd() as float
  local v=get_reg_16_2(&h5aa), k=get_reg_11_2(&h2427), va=get_reg_11_2(&h2426)
  va=va*32 : k=k*32
  get_vdd = 3.3 + (RC*v - va)/k
end function
  
  
  'this sub extracts calibration values from the chips EEPROM and formats these
  'for use in the math
  'formula's are derived from MLX90641 data sheets
sub get_cal
  local x
  local scale_row1
  local scale_row2
  local scale_row3
  local scale_row4
  local scale_row5
  local scale_row6
  local alpha_cp_sc, kta_cp_sc, kv_cp_sc
  
  
  x=get_reg_11(&h2410)
  scale_os_r1 = x>>5 : scale_os_r2 = x and 31
  
  x= get_reg_11_2(&h2411) * 32
  pix_os_av = get_reg_11_2(&h2412) : inc pix_os_av, x
  
  kta_avg = get_reg_11_2(&h2415)
  x=get_reg_11(&h2416)
  kta_scale_1 = x>>5 : kta_scale_2 = x and 31
  
  kv_avg = get_reg_11_2(&h2417)
  x=get_reg_11(&h2418)
  kv_scale_1 = x>>5 : kv_scale_2 = x and 31
  
  x=get_reg_11(&h2419)
  scale_row_1 = (x>>5) + 20 : scale_row_2 = (x and 31) + 20
  x=get_reg_11(&h241A)
  scale_row_3 = (x>>5) + 20 : scale_row_4 = (x and 31) + 20
  x=get_reg_11(&h241B)
  scale_row_5 = (x>>5) + 20 : scale_row_6 = (x and 31) + 20
  
  a_ref_row_1 = get_reg_11(&h241C) / (2^scale_row_1)
  a_ref_row_2 = get_reg_11(&h241D) / (2^scale_row_2)
  a_ref_row_3 = get_reg_11(&h241E) / (2^scale_row_3)
  a_ref_row_4 = get_reg_11(&h241F) / (2^scale_row_4)
  a_ref_row_5 = get_reg_11(&h2420) / (2^scale_row_5)
  a_ref_row_6 = get_reg_11(&h2421) / (2^scale_row_6)
  
  ksta = get_reg_11_2(&h2422) / 32768
  
  'emissivity
  em = get_reg_11_2(&h2423) / 512
  
  'general gain
  gain = get_reg_11(&h2424) * 32
  x = get_reg_11(&h2425) : inc gain,x
  Kgain = gain / get_reg_16_2(&h058A)
  
  'ptat
  ptat = get_reg_11(&h2428) * 32
  x = get_reg_11(&h2429) : inc ptat,x
  
  kt_ptat = get_reg_11(&h242A) / 8
  kv_ptat = get_reg_11(&h242B) / 4096
  alpha_ptat = get_reg_11(&h242C) / 128
  
  'control pixel
  alpha_cp_sc = get_reg_11(&h242E)
  alpha_cp = get_reg_11(&h242D) / (2 ^ alpha_cp_sc)
  
  x = get_reg_11(&h242F) * 32
  offset_cp = get_reg_11(&h2430) : inc offset_cp,x : map_16_2(offset_cp)
  gain_cp = get_reg_16_2(&h0588) * Kgain
  
  x = get_reg_11(&h2431)
  kta_cp_sc = x>>6
  kta_cp = x and 63 : if kta_cp > 31 then inc kta_cp,-64
  kta_cp = kta_cp / (2^kta_cp_sc)
  x = get_reg_11(&h2432)
  kv_cp_sc = x>>6
  kv_cp = x and 63 : if kta_cp > 31 then inc kta_cp,-64
  kv_cp = kv_cp / (2^kv_cp_sc)
  
  cp_pix_os = gain_cp - offset_cp*(1+kta_cp*(Ta-25))*(1+Kv_cp*(get_vdd()-3.3))
  
  'IR gradient compensation
  TGC = (get_reg(&h2433) and &h1ff) / 64
  RC_cal = (get_reg(&h2433) and &h600)>>9
  
  'Object temperature scaling
  KsTo_sc = get_reg_11(&h2434)
  KsTo_1 = get_reg_11_2(&h2435) / (2^KsTo_sc)
  KsTo_2 = get_reg_11_2(&h2436) / (2^KsTo_sc)
  KsTo_3 = get_reg_11_2(&h2437) / (2^KsTo_sc)
  KsTo_4 = get_reg_11_2(&h2438) / (2^KsTo_sc)
  KsTo_5 = get_reg_11_2(&h2439) / (2^KsTo_sc)
  KsTo_6 = get_reg_11_2(&h243B) / (2^KsTo_sc)
  KsTo_7 = get_reg_11_2(&h243D) / (2^KsTo_sc)
  KsTo_8 = get_reg_11_2(&h243F) / (2^KsTo_sc)
  CT6 = get_reg_11(&h243A)
  CT7 = get_reg_11(&h243C)
  CT8 = get_reg_11(&h243E)
  
end sub
  
  
  
  ' ------------------- conversion functions -----------------------------
  'this function makes use of the fact that changes in variable x reflect in
  'source variable
  
sub map_16_2(x)   '16 bit 2'th complement
  if x>32767 then inc x,-65536
end sub
  
  
  
  ' ------------------- read chip functions -----------------------------
  'these functions read data from the MLX90641 and do some formatting
  
  
  'return 16 bit value read from MLX90641 registers n, MSB first
function get_reg(n) as integer
  local a%(1)
  if I2Cport=2 then
    i2c2 write SA, 1, 2, n\256, n and 255
    i2c2 read SA,0,2,a%()
  else
    i2c write SA, 1, 2, n\256, n and 255
    i2c read SA,0,2,a%()
  end if
  get_reg=a%(1)+256*a%(0)
end function
  
  
  'return an 11 bit value (remove hamming code) from MLX90641 register n
function get_reg_11(n) as integer
  get_reg_11 = get_reg(n) and &h7ff
end function
  
  
  'return a 2'th complement 11 bit value (remove hamming code) from MLX90641 register n
function get_reg_11_2(n) as integer
  get_reg_11_2 = get_reg(n) and &h7ff
  if get_reg_11_2>1023 then inc get_reg_11_2,-2048
end function
  
  
  'return a 16 bit 2'th complement value from MLX90641 register n
function get_reg_16_2(n) as integer
  get_reg_16_2 = get_reg(n)
  if get_reg_16_2>32767 then inc get_reg_16_2,-65536
end function
  
  
  ' ------------------- write chip functions -----------------------------
  'this functions writes a 16 bit value v to MLX90641 register n
  
  
  'write 16 bit value to MLX90641 registers n, MSB first
sub set_reg n,v
  if I2Cport=2 then
    i2c2 write SA,0,4,n\256,n and 255,v\256,v and 255
  else
    i2c write SA,0,4,n\256,n and 255,v\256,v and 255
  end if
end sub
  
  
  '-----------------------matrices for math -----------------------------
  'this sections fills the multiplication matrics that do the up-scaling
  
sub fill_matrix
  local i,x,y as integer
  
  'matrix d(11,44) for scaling up 4x vertical
  for i=0 to 11 : d(i,4*i)=1 : next
  for i=0 to 10
    d(i,4*i+2)=0.5  : d(i+1,4*i+2)=0.5
    d(i,4*i+1)=0.75 : d(i+1,4*i+1)=0.25
    d(i,4*i+3)=0.25 : d(i+1,4*i+3)=0.75
  next
  
  'matrix b_e(30,15) and b_o(30,15) for scaling up 2x horizontal
  for i=0 to 15 : b_e(2*i,i)=1 : next         'references
  for i=0 to 14
    b_e(2*i+1,i)=0.5  : b_e(2*i+1,i+1)=0.5    'midpoints
    b_o(2*i,i)=0.75   : b_o(2*i,i+1)=0.25     'quarter
    b_o(2*i+1,i)=0.25 : b_o(2*i+1,i+1)=0.75   '3 quarter
  next
  
  'matrix x_s%() and y_s%() containg coordiantes for the boxes
  for x=0 to 30
    for y=0 to 44
      x_s%(31*y+x)=2*size%*x + 7   'shift 7 right to center picture
      y_s%(31*y+x)=size%*y + 15    'shift 15 down
    next
  next
  
end sub
  
  'color palette for the different color depths
col6:
  data rgb(blue),rgb(cerulean),rgb(red),rgb(orange),rgb(yellow),rgb(white)
col10:
data rgb(black),rgb(blue),rgb(cobalt),rgb(cerulean),rgb(cyan),rgb(midgreen),rgb(green),rgb(yellow),rgb(orange),rgb(red)
col14:
data rgb(black),rgb(blue),rgb(cobalt),rgb(cerulean),rgb(cyan),rgb(midgreen),rgb(green),rgb(yellow),rgb(orange),rgb(red),rgb(magenta),rgb(fuchsia),rgb(pink),rgb(white)
col15:
data rgb(black),rgb(blue),rgb(cobalt),rgb(cerulean),rgb(cyan),rgb(midgreen),rgb(green),rgb(yellow),rgb(orange),rgb(rust),rgb(red),rgb(magenta),rgb(fuchsia),rgb(pink),rgb(white)
col16:
data rgb(black),rgb(myrtle),rgb(blue),rgb(cobalt),rgb(cerulean),rgb(cyan),rgb(midgreen),rgb(green),rgb(yellow),rgb(orange),rgb(rust),rgb(red),rgb(magenta),rgb(fuchsia),rgb(pink),rgb(white)

