  'PIO 8n1 UART based on RP2040 datasheet 3.6.3 and 3.6.4
  'requires PicoMite MMBasic 6.00.02 beta 0 to PIO assemble
  
  'version control
  ' v0.0        sending from individual fifo writes works
  ' v0.1        sending from DMA works.
  ' v0.2        faster TX conversion string to dma fifo (idea: Peter)
  ' v0.3        UART RX experiments 8 deep fifo (needs V6.00.02b0 or newer)
  ' v0.4        UART_RX using DMA works, even at 115200
  ' v0.5        cleanup of the code
  ' v0.6        RX ringbuffer
  ' v0.7        improve RX robustness v0.6=+1/-5% 0.7=+4/-4% (16*rx_bau_rate)
  ' v0.8        Communicate framing errors through IRQ register
  ' v0.9        start PIO sequence changed, PIO stop for TX DMA
  ' v0.10       Change PIO ASSEMBLE to PIO PROGRAM
  
  
  option default integer
  
  'PIO program uses 2 state machines, one for RX and one for TX
  'baud_rate = 19200 '115200
  rx_buf_size = 128 '256 'characters
  tx_buf_size = 128 '256 'characters
  tx_baud_rate = 19200
  rx_baud_rate = tx_baud_rate '19200*0.96 to test robustness
  
  
  
  'memory buffer reservation
  
  'DMA buffers (packed) for 128 characters maximum
  dim tx_pck%(tx_buf_size/2 - 1)            'packed TX array, 2 char per integer
  const tx_pck_ad%=peek(varaddr tx_pck%())  'memory address of array start
  
  dim rx_pck%                               'packed RX array, 2 char per integer
  PIO make ring buffer rx_pck%,rx_buf_size*4    'as ring buffer, size in bytes
  const rx_pck_ad%=peek(varaddr rx_pck%())  'memory address of array start
  dim old_rx% = rx_pck_ad%                  'old read pointer, default at start
  const rx_pck_end%=rx_pck_ad%+rx_buf_size*4'last location of rx buffer in bytes
  
  dim rx$                                   'string for received characters
  const rx_ad%=peek(varaddr rx$)            'memory address of string
  
  
  'make sure all fifo's and program memory are cleared, then program PIO
  pio clear 1
  INIT_PIO
  
  
  '---------------------------- example code -------------------------------
  
  'test text string
  a$ = "Hello World, I admire your beautiful beaches" + chr$(13)
  
  
  'main loop: send string repeatedly
  do
    
    '---------------------- send a string -----------------
    UART_TX (a$)
    
    
    pause 50
    
    
    '--------------------- receive a string ----------------
    
    'print characters in the receive buffer if no framing error
    'UART_RX%() uses the global variable rx$ to assemble all data in.
    if UART_RX%() then
      print rx$                   'print the received string
    else
      print "framing error"
      poke word pio_1_irq,255     'clear framing error
    end if
    
  loop
  
  
  
  
  '-------------------------- subs -------------------------------------------
  
  'sends string using DMA to PIO TX
sub UART_TX (tx$)
  
  PIO STOP 1,0  'test as per phil99 advise
  
  'get start and size of string to be sent
  tx_l% = len(tx$) : tx_ad% = peek(varaddr tx$)
  
  'pack string for sending to UART
  memory copy tx_ad%+1, tx_pck_ad%, tx_l%, 1, 4
  
  'start the DMA
  PIO init machine 1,0,f0,p0,e0,s0,0        'SM0 start through DMA at defined state
  pio dma tx 1, 0, tx_l%, tx_pck%(),        'start SM0, send string through DMA
  
end sub
  
  
  
function UART_RX%()
  
  local LOF%, LOF1%, LOF2%, new_rx%
  
  new_rx% = Pio(DMA RX POINTER)   'get current position in ring buffer
  
  select case new_rx%-old_rx%
    case 0
      'no new data
      rx$=""
      
    case >0
      'new data, no wrap around
      LOF% = (new_rx% - old_rx%) \ 4
      
      'unpack RX buffer to string
      memory copy old_rx%,rx_ad%+1,LOF%,4,1           'fill string characters
      poke byte rx_ad%,LOF%                           'fill len(string)
      
    case <0
      'new data, with wrap around, needs split copying.
      LOF1% = (rx_pck_end% - old_rx%) \ 4             'begin text until end of buffer
      LOF2% = (new_rx% - rx_pck_ad%) \ 4              'start buffer to end of text
      LOF% = LOF1% + LOF2%                            'total size of text
      
      'unpack RX buffer to string
      memory copy old_rx%,rx_ad%+1,LOF1%,4,1          'fill start string characters
      memory copy rx_pck_ad%,rx_ad%+1+LOF1%,LOF2%,4,1 'fill end string characters
      poke byte rx_ad%,LOF%                           'fill len(string)
      
  end select
  
  old_rx% = new_rx%   'reposition read pointer
  
  UART_RX% = (peek(word pio_1_irq)=0)     'UART status 1=okay, 0=framing error
  
end function
  
  
  'program PIO 1 SM0 and 1, and start the UART
sub INIT_PIO
  
  'use PIO IRQ register to transfer framing errors
  
  pio_1_irq% = &h30 + &h50300000  'register + PIO1 offset
  poke word pio_1_irq,255         'erase all flag
  
  
  'State machine 0 is UART TX
  
  SetPin GP0,pio1
  
  ' note: side enable is implemented as the msb side_set bit since assembler
  ' instruction bits   12        11       10 9  8
  ' use              side_en   side_set     delay
  
  PIO ASSEMBLE 1,".program uart_tx"         'uart tx program from RP2040 datasheet
  PIO ASSEMBLE 1,".side_set 1 opt"          'side set 1 bit + opt=side enable
  pio assemble 1,".line 0"                  'start of PIO program memory
  PIO ASSEMBLE 1,"set pindirs, 1"           'gp0 is output
  PIO ASSEMBLE 1,".wrap target"             'return point for wrap
  PIO ASSEMBLE 1,"pull block side 1 [7]"    'get data from fifo
  PIO ASSEMBLE 1,"set x, 7  side 0 [7]"     'bit counter 7,6,5,....,0
  PIO ASSEMBLE 1,"bitloop:"                 'label
  PIO ASSEMBLE 1,"out pins, 1"              'shift 1 bit out
  PIO ASSEMBLE 1,"jmp x-- bitloop  [6]"     'check if 0, then decrement
  PIO ASSEMBLE 1,".wrap"                    'wrap to target
  PIO ASSEMBLE 1,".end program list"      'write program and list hex values
  
  'configure pio0 StateMachine 0
  f0=tx_baud_rate*8
  '              a b c d  e   f   g                     'a,e=side set   c,g=OUT  b,f=set
  p0=Pio(PINCTRL 2,1,1, ,gp0,gp0,gp0)                   'note:2=include side enable bit
  e0=Pio(execctrl gp1,Pio(.wrap target),Pio(.wrap),0,1 )'1=side set only when defined
  s0=Pio(shiftctrl 0,0,0,0,0,1,0,0)                     'shift OUT direction LSB first
  
  'write the configuration
  PIO init machine 1,0,f0,p0,e0,s0,0        'SM0 start at address 0
  
  
  
  'Statemachine 1 = UART RX
  
  PIO ASSEMBLE 1,".program uart_rx"         'uart tx program from RP2040 datasheet
  pio assemble 1,".line 5"                  '5=start of PIO program memory RX
  PIO ASSEMBLE 1,"start:"                   'label
  PIO ASSEMBLE 1,".wrap target"             'outer loop return
  PIO ASSEMBLE 1,"wait 0 pin 0"             'wait for a start bit
  PIO ASSEMBLE 1,"set x,7 [20]"             'set counter to 7,wait 10 cycles (=mid lsb)
  PIO ASSEMBLE 1,"bitloop2:"                'inner loop
  PIO ASSEMBLE 1,"in pins,1"                'shif in 1 bit
  PIO ASSEMBLE 1,"jmp x-- bitloop2 [14]"    'if x>0 next bit, delay 6, then x--
  PIO ASSEMBLE 1,"jmp pin good_stop"        'if valid stop bit -> good_stop
  PIO ASSEMBLE 1,"irq set 0"                'was "irq 4 rel" set flag for ARM
  PIO ASSEMBLE 1,"wait 1 pin 0"             'wait for a valid stop
  PIO ASSEMBLE 1,"jmp start"                'then goto wait for new start (loop)
  PIO ASSEMBLE 1,"good_stop:"               'label
  PIO ASSEMBLE 1,"in null,24"               'shift the data to LSB in the ISR
  PIO ASSEMBLE 1,"push"                     'shift data to fifo
  PIO ASSEMBLE 1,".wrap"                    'outer loop for new character
  PIO ASSEMBLE 1,".end program list"
  
  'configure pio0 StateMachine 1
  f1=rx_baud_rate*16
  '              a b c  d   e   f   g                   'a,e=side set c,g=OUT b,f=set
  p1=Pio(PINCTRL 0, , ,gp1,  ,   ,    )                 'gp1 = IN pin
  e1=Pio(execctrl gp1,Pio(.wrap target),Pio(.wrap))     'gp1 = JMP pin
  s1=Pio(shiftctrl 0,0,0,0,1,0,0,0)                     'shift IN LSB first
  
  'write the configuration
  PIO init machine 1,1,f1,p1,e1,s1,5        'SM1 start at address 5
  
  
  ' start the PIO UART
  pio start 1,0                                'start the TX UART to set GPIO direction
  pio dma rx 1,1,0,rx_pck%(),,32,rx_buf_size%  'start the RX PIO with DMA ringbuffer
  PIO STOP 1,0                                 'stop again, restart is in UART_TX DMA start
  '
end sub
  
  
  
  
  
  
