'Test suite for LCDI2CLib
GOTO TestSuite
 
'------------------------------------------------------------------
'
' LCDI2CLib  V5 2014-August-20
'
' Micromite to LCD using I2C module
'
' (c) Peter Carnegie 2014
'
' This code is a heavily modified version of I2CLCD.BAS by John Gerrard
'
' This LIB can use several different models of PCF8574 based I2C to LCD interfaces
' The most important difference between such modules is whether they use P0~P3 or
' P4~P7 for the 4 bit data nibble to the LCD. This LIB can ONLY deal with LO or HI nibbles. If the 
' data bits are spread across the PCF8574 port bits in other ways, then the user will
' have to edit the code in SUB LCDI2C_DirectSend
'
' The most important call is LCDI2C_Init which configures the library appropriately.
' LCDI2C_Init I2CAddr,NibPos,RSBitNo,ENBitNo,BLBitNo
'
' Where I2CAddr is the I2C Address 
'       NibPos=0 use low 4 bits ie P0~P3, NibPos=1 use high 4 bits ie P4~P7 
'       RSBitNo is PCF8574 bit number for RS 0~7 
'       ENBitNo is PCF8574 bit number for EN 0~7 
'       BLBitNo is PCF8574 bit number for Backlight control 0~7 
'
'  
' Using the I2C -> LCD Interface from eBay priced about US$1.60
' http://www.ebay.co.uk/itm/291116790762?ssPageName=STRK:MEWNX:IT&_trksid=p3984.m1439.l2649
' http://www.youtube.com/watch?v=wmTWk4Rwfw0
'
' Uses PCF8574 - 8 bit I2C Port Expander
' I2CAddress is &H27 with unjumpered address pads
' PCF8574 port is wired as follows
' P0 - RS
' P1 - RW
' P2 - EN
' P3 - Backlight On/Off
' P4 - D4 LCD
' P5 - D5 LCD
' P6 - D6 LCD
' P7 - D7 LCD
'
' LCDI2C_Init I2CAddr,NibPos,RSBitNo,ENBitNo,BLBitNo 
' LCDI2C_Init &H27,1,0,2,3 
 
'
' Using BigMik board (see http://www.thebackshed.com/forum/forum_posts.asp?TID=6715&PN=1&TPN=2)
' Uses PCF8574 - 8 bit I2C Port Expander
' I2CAddress is &H20~&H27 depending on jumper settings
' PCF8574 port is wired as follows
' P0 - D4
' P1 - D5
' P2 - D6
' P3 - D7
' P4 - RS
' P5 - Backlight 
' P6 - RW
' P7 - EN
'
' LCDI2C_Init I2CAddr,NibPos,RSBitNo,ENBitNo,BLBitNo 
' LCDI2CInit &H2X,0,4,7,5 
'
' Functions in the Library are modelled after the native MMBasic LCD commands
' to ease code conversion
'
' Sub LCDI2C(LineNum,CharPos,Text$) Display string Text$ on LineNum of the display starting at CharPos
' Sub LCDI2C_Clear()                Clears the LCD
' LCDI2C_Init I2CAddr,NibPos,RSBitNo,ENBitNo,BLBitNo          Initialise the I2C and LCD
' Sub LCDI2C_Close()                Close the I2C LCD
 
' SUB LCDI2C_BackLight(State)       Turn LCD backlight ON | OFF
 
' SUB LCDI2C_CMD(Byte)              Send a COMMAND byte to the LCD, ie RS=0
' SUB LCDI2C_DATA(Byte)             Send a DATA byte to the LCD, ie RS=1
'
' SUB LCDI2C_DirectSend(Byte)       Send a BYTE to the LCD - used internally by the lib
' SUB LCDI2C_WireWrite(Byte)        Write a byte onto the I2C bus to the LCD
'
'
' Uses 6 Global Variables
' LCDI2C_LCDBackLight, LCDI2C_I2CAddr, LCDI2C_RSDataMask, 
' LCDI2C_EMask, LCDI2C_BackLight, LCDI2C_NibPos
'  
'
' V3 - Added SUB LCDI2C_BackLight(State) where State=0=Backlight OFF, NonZero=BacklightON.
'            Deleted SUB LCDI2C_BacklightOn and SUB LCDI2C_BacklightOFF
'
' V4 - Changed Test Suite to use SUB LCDI2C_BackLight(State) instead of LCDI2C_BacklightOn/LCDI2C_BacklightOFF
'
' V5 - Changed SUB LCDI2C_DATA(Byte) to allow for Low nibble i/f's like the MUP
'      LCDI2C_DirectSend now takes 2 arguments, the second is the state of RS, 0 for CMD, 1 for DATA
 
 
'------------------------------------------------------------------
'
' Print String to LCD
'
'
SUB LCDI2C(LineNum,CharPos,Text$)
  LOCAL I
  IF LineNum=1 THEN I=(&H80 + CharPos-1)
  IF LineNum=2 THEN I=(&HC0 + CharPos-1)
  IF LineNum=3 THEN I=(&H94 + CharPos-1)
  IF LineNum=4 THEN I=(&HD4 + CharPos-1)
  LCDI2C_CMD(I)
 
  FOR I=1 TO LEN(Text$)
    LCDI2C_DATA(ASC(MID$(Text$,I,1)))
  NEXT I
 
END SUB
 
 
'------------------------------------------------------------------
'
' Clear LCD
' 
'
SUB LCDI2C_Clear()
  LCDI2C_CMD(&H01)
END SUB
 
 
'------------------------------------------------------------------
'
' INITIALIZE LCD
' 
' Where I2CAddr is the I2C Address 
'       NibPos=0 use low 4 bits ie P0~P3, NibPos=1 use high 4 bits ie P4~P7 
'       RSBitNo is PCF8574 bit number for RS 0~7 
'       ENBitNo is PCF8574 bit number for EN 0~7 
'       BLBitNo is PCF8574 bit number for Backlight control 0~7 
'
SUB LCDI2C_Init I2CAddr,NibPos,RSBitNo,ENBitNo,BLBitNo
 
  'Backlight Control bit
  DIM LCDI2C_BackLightMIK
  LCDI2C_BackLightMIK=2^BLBitNo
  
  'Current Backlight state - default to ON
  DIM LCDI2C_LCDBackLight
  LCDI2C_LCDBackLight=LCDI2C_BackLightMIK
  
  'I2C Bus Address
  DIM LCDI2C_I2CAddr
  LCDI2C_I2CAddr=I2CAddr
  
  'RS Control bit
  DIM LCDI2C_RSDataMask
  LCDI2C_RSDataMask=2^RSBitNo
  
  'EN Control Bit
  DIM LCDI2C_EMask
  LCDI2C_EMask=2^ENBitNo
  
  'Save if P0~P3 or P4~P7 bits of PCF8574 port
  DIM LCDI2C_NibPos
  LCDI2C_NibPos=NiBPos
 
  'Start I2C
  I2C OPEN 400,100 
  
  '%0011---- %0011----   8-bit / 8-bit
  LCDI2C_CMD(&H33)
  '%0011---- %0010----   8-bit / 4-bit
  LCDI2C_CMD(&H32)
 
  ' Byte commands - To configure the LCD
 
  ' Display Format
  ' 4bit mode, 2 lines, 5x7
  '
  ' 001LNF00
  ' %00101000
  LCDI2C_CMD(&H28)
 
  ' L : 0 = 4-bit Mode    1 = 8-bit Mode
  ' N : 0 = 1 Line        1 = 2 Lines
  ' F : 0 = 5x7 Pixels    1 = N/A
 
  
  ' Setup Display
  ' Display ON, Cursor On, Cursor Steady
  '
  ' 00001DCB
  ' %00001110
  LCDI2C_CMD(&H0C)
  
  ' D : 0 = Display Off   1 = Display On
  ' C : 0 = Cursor Off    1 = Cursor On
  ' B : 0 = Cursor Steady 1 = Cursor Flash
 
  
  ' Setup Cursor/Display
  ' Inc Cursor Cursor Move
  '
  ' 000001IS
  LCDI2C_CMD(&H06)
  ' I : 0 = Dec Cursor    1 = Inc Cursor
  ' S : 0 = Cursor Move   1 = Display Shift
 
  LCDI2C_CMD(&H01)
  
  'Turn Off LCDBacklight
  LCDI2C_BackLight(0)
 
END SUB
 
 
'------------------------------------------------------------------
'
' Close the I2C LCD
'
'
SUB LCDI2C_Close()
 
  'Close the I2C Bus
  I2C CLOSE
 
END SUB           
 
'------------------------------------------------------------------
'
' Turn Backlight On | Off
' If State=0 then OFF Else ON
'
SUB LCDI2C_BackLight(State)
  IF State<>0 THEN
    LCDI2C_LCDBacklight=LCDI2C_BacklightMIK
  ELSE
    LCDI2C_LCDBacklight=&H0
  ENDIF
  LCDI2C_WireWrite(0)
END SUB
 
'------------------------------------------------------------------
'
' Send Command Byte to LCD
'
'
SUB LCDI2C_CMD(Byte)
 
  'Send Hi Nibble
  LCDI2C_DirectSend(Byte AND &HF0,0) 
  
  'Send Low Nibble
  LCDI2C_DirectSend((Byte AND &H0F) * 16,0)
  
END SUB
 
 
'------------------------------------------------------------------
'
' Send Data Byte to LCD
'
'
SUB LCDI2C_DATA(Byte)
 
  'Send Hi Nibble
  LCDI2C_DirectSend((Byte AND &HF0), LCDI2C_RSDataMask)
  
  'Send Lo Nibble
  LCDI2C_DirectSend(((Byte AND &H0F) * 16), LCDI2C_RSDataMask)
  
END SUB
 
 
'------------------------------------------------------------------
'
' Send Byte to LCD over I2C
' NB, we don't call LCDI2C_WireWrite in this SUB because that results in 
' MUCH slower execution time than inlin'ing the I2C Write's
'
'
SUB LCDI2C_DirectSend(Byte, RSState)
 
  LOCAL B,B1
  
  'Use a copy so that we don't mess up caller's argument
 
  'Are D4-D7 of LCD mapped to P0~P3 of PCD8574 ?
  IF LCDI2C_NibPos=0 THEN
    B1=(Byte\16) OR RSState
  ELSE
    B1=Byte OR RSState
  ENDIF    
  
  'Take EN high, use var B 'cos it's quicker than doing the OR's in the I2C Write command itself
  B=B1 OR LCDI2C_EMask OR LCDI2C_LCDBacklight
  I2C WRITE LCDI2C_I2CAddr,0,1,B
  
  'Take EN Low
  I2C WRITE LCDI2C_I2CAddr,0,1,B1 OR LCDI2C_LCDBacklight
  
END SUB
 
 
'------------------------------------------------------------------
'
' Send Byte over I2C
'
'
SUB LCDI2C_WireWrite(Byte)
  I2C WRITE LCDI2C_I2CAddr,0,1,Byte OR LCDI2C_LCDBacklight
END SUB
 
'End Of LCDI2C Library code
 
'<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
'Cut Here and discard the Test Suite code when U include the library
'into your own programs
 
TestSuite:
'Initialise the I2C LCD device
'LCDI2C_Init I2CAddr,NibPos,RSBitNo,ENBitNo,BLBitNo
 LCDI2C_Init &H20,0,4,7,5
WaitForAKey("Press any key for next test")
 
'Turn On the Backlight
LCDI2C_Backlight 1
WaitForAKey("Press any key for next test")
 
'Turn OFF the Backlight
LCDI2C_Backlight 0
WaitForAKey("Press any key for next test")
 
'Turn On the Backlight
LCDI2C_Backlight 1
WaitForAKey("Press any key for next test")
 
LCDI2C 1,3,DATE$
WaitForAKey("Press any key for next test")
 
LCDI2C 2,4,TIME$
WaitForAKey("Press any key for next test")
 
LCDI2C_Clear
WaitForAKey("Press any key for next test")
 
DO
 
'  L=TIMER 'DEBUG
  LCDI2C 1,3,DATE$
  LCDI2C 2,4,TIME$
'  PRINT TIMER-L 'DEBUG
  
LOOP
 
END
 
SUB WaitForAKey(Msg$)
  IF Msg$="" THEN Msg$="Press any key to continue"
  PRINT Msg$;
  DO
  LOOP WHILE INKEY$=""
  PRINT
END Sub
