GOTO TestSuite


'=================================================================
'
' I2CKeypad_MUP.bas V1.0  (c) 2014 Peter Carnegie
'
' This Keypad library interfaces 4 x 4 keypads to uMite using
' a PCF8574 I2C chip.
'
' This library ONLY works with Keypad wiring as follows
'
' Row 1 -> P0, Row 2 -> P1, Row 3 -> P2, Row 4 -> P3
' Col 1 -> P4, Col 2 -> P5, Col 3 -> P6, Col 4 -> P7
' 
' The Library includes the following functions.
'       KeyPad_I2C_Init(I2CAddress)    Initialise the Library
'       KeyPad_I2C_Int                 Interrupt Service Routine
'       KeyPad_I2C_GetKey()             Called by KeyPad_I2C_Int
'
' Key strokes are returned to the main program in GLOBAL variable
' KeyPad_I2C_KeyCode. The value of &HFF indicates NO key press, 
' after reading out the key value, set KeyPad_I2C_KeyCode back to
' &HFF.
'
' Example Usage :
'
'     'Start I2C operations
'     I2C OPEN 400,100
'
'     'Initialise the Keypad Software
'     KeyPad_I2C_Init(&H26, 7)
'
'
'     'Do Forever
'     DO
'
'         IF KeyPad_I2C_KeyCode<>&HFF THEN
'             PRINT "Got the key ";KeyPad_I2C_KeyCode
'             KeyPad_I2C_KeyCode=&HFF
'         ENDIF
'  
'     LOOP
'
' 
'=================================================================


'=================================================================
'
' Initialise the I2C KeyPad software.
' You MUST call this function BEFORE carrying out any I2C Keypad 
' operations
'
' I2CAddr is the Address of the I2C Slave i/f to the KeyPad
' IntPin is the uMite pin number used for the PCF8574 IRQ
'
' Creates GLOBAL Vars
'   KeyPad_I2C_KeyCode      Var into which keypresses will be placed
'   KeyPad_I2C_Addr         Var holding the I2C Address of the slave 
'   KeyPad_I2C_RowMask(3)   Array of Row Bit Masks
'   KeyPad_I2C_ColMask(3)   Array of Column Bit Masks  
'   KeyPad_I2C_KeyCodes(15) Lookup table of Row,Col to Key code
'   KeyPad_I2C_Timer        Var used for autorepeat timing
'
SUB KeyPad_I2C_Init(I2C_Addr, IntPin)

  'Var into which key presses get put
  DIM KeyPad_I2C_KeyCode
  KeyPad_I2C_KeyCode=&HFF

  'I2C Address
  DIM KeyPad_I2C_Addr
  KeyPad_I2C_Addr=I2C_Addr

  
  'Row 1 -> P0, Row 2 -> P1, Row 3 -> P2, Row 4 -> P3

  'Make the Row Masks
  DIM KeyPad_I2C_RowMask(3)
  KeyPad_I2C_RowMask(0)=&B00000001
  KeyPad_I2C_RowMask(1)=&B00000010
  KeyPad_I2C_RowMask(2)=&B00000100
  KeyPad_I2C_RowMask(3)=&B00001000
  
  
  'Col 1 -> P4, Col 2 -> P5, Col 3 -> P6, Col 4 -> P7
   
  'Make the Column Masks
  DIM KeyPad_I2C_ColMask(3)
  KeyPad_I2C_ColMask(0)=&B11100000
  KeyPad_I2C_ColMask(1)=&B11010000
  KeyPad_I2C_ColMask(2)=&B10110000
  KeyPad_I2C_ColMask(3)=&B01110000
  
  'Lookup table - values as per uMite MMBasic Keypad i/f
  '4 cols x 4 rows
  DIM KeyPad_I2C_KeyCodes(3,3)
  KeyPad_I2C_KeyCodes(0,0)=1   :KeyPad_I2C_KeyCodes(1,0)=2
  KeyPad_I2C_KeyCodes(2,0)=3   :KeyPad_I2C_KeyCodes(3,0)=20
  KeyPad_I2C_KeyCodes(0,1)=4   :KeyPad_I2C_KeyCodes(1,1)=5
  KeyPad_I2C_KeyCodes(2,1)=6   :KeyPad_I2C_KeyCodes(3,1)=21
  KeyPad_I2C_KeyCodes(0,2)=7   :KeyPad_I2C_KeyCodes(1,2)=8
  KeyPad_I2C_KeyCodes(2,2)=9   :KeyPad_I2C_KeyCodes(3,2)=22
  KeyPad_I2C_KeyCodes(0,3)=10  :KeyPad_I2C_KeyCodes(1,3)=0
  KeyPad_I2C_KeyCodes(2,3)=11  :KeyPad_I2C_KeyCodes(3,3)=23
  
  'Auto repeat timer
  DIM KeyPad_I2C_Timer
  
  'Arm the PCF8574 Interrupt Output
  I2C WRITE KeyPad_I2C_Addr, 0, 1, &HF0
  
  'PCF8474 IRQs into the uMite
  SETPIN IntPin, INTL, KeyPad_I2C_Int

END SUB


'=================================================================
'
' Interrupt Service Routine for Keypad Interrupts
'
KeyPad_I2C_Int:

  LOCAL L
  
'  L=TIMER 'DEBUG
  KeyPad_I2C_KeyCode = KeyPad_I2C_GetKey()
'  L=TIMER-L 'DEBUG
  
IRETURN


'===================================================================
'
' Scan keypad and return keycode if pressed
' 
FUNCTION KeyPad_I2C_GetKey()

  IF TIMER-KeyPad_I2C_Timer<200 THEN

    KeyPad_I2C_GetKey=&HFF

    'Re-Arm the PCF8574 Interrupt
    I2C WRITE KeyPad_I2C_Addr, 0, 1, &HF0
    
    EXIT FUNCTION
    
  ENDIF

  LOCAL KeyCode, LastKeyCode
  
  'Key Debounce Buffer  
  LOCAL KeyCodeBuf(8)
  
  LOCAL Col, Cols, Row
  
  LOCAL I,J,K,L,M
  
  'Scan 8 times for keydown, debounce
  FOR K=0 TO 7
'    L=TIMER 'DEBUG
    
    'write all logic 0′s to the rows and all logic 1′s to the columns.
    I2C WRITE KeyPad_I2C_Addr, 0, 1, &HF0
    
    'Readback the state of the Cols
    I2C READ  KeyPad_I2C_Addr, 0, 1, Cols
    
    Cols=Cols AND &HF0

    'Is there a key down ?
    IF Cols <> &HF0 THEN

      'IF Cols=&B11100000 THEN Col=0:J=&B00010000
      IF Cols=KeyPad_I2C_ColMask(0) THEN Col=0:J=&B00010000
      
      'IF Cols=&B11010000 THEN Col=1:J=&B00100000
      IF Cols=KeyPad_I2C_ColMask(1) THEN Col=1:J=&B00100000
      
      'IF Cols=&B10110000 THEN Col=2:J=&B01000000
      IF Cols=KeyPad_I2C_ColMask(2) THEN Col=2:J=&B01000000
      
      'IF Cols=&B01110000 THEN Col=3:J=&B10000000
      IF Cols=KeyPad_I2C_ColMask(3) THEN Col=3:J=&B10000000
      
      'start writing logic 1′s to the rows sequentially and check if Col goes high high
      FOR I=0 TO 3
        Mask=KeyPad_I2C_RowMask(I) OR &HF0

        I2C WRITE KeyPad_I2C_Addr, 0, 1, Mask

        I2C READ  KeyPad_I2C_Addr, 0, 1, Cols
        
        'Has Col gone high
        IF (Cols AND J) > 0 THEN
          Row=I
          EXIT FOR
        ENDIF
        
      NEXT I
      
      'Lookup the keycode in the table
      KeyCode=KeyPad_I2C_KeyCodes(Col,Row)
     
      'Debounce the key stroke
      KeyCodeBuf(M)=KeyCode
      M=(M+1) MOD 8
      FOR I=0 TO 7
        IF KeyCode<>KeyCodeBuf(I) THEN
          EXIT FOR
        ENDIF
      NEXT I
  
      IF I>7 THEN
        IF KeyCode <> LastKeyCode THEN
          LastKeyCode=KeyCode
   
'          L=TIMER-l 'DEBUG
'          PRINT L;" ";"Col=";Col;" Row=";Row;" Key=";KeyPad_I2C_KeyCode 'DEBUG
          KeyPad_I2C_Timer=TIMER
          
          'Send back the key
          KeyPad_I2C_GetKey=KeyCode
          
          EXIT FOR
          
        ENDIF
      ENDIF
    
    ELSE
      'No Key
      KeyPad_I2C_GetKey=&HFF
      EXIT FOR
       
    ENDIF
   
  NEXT K
  
  'Rearm the IRQ
  I2C WRITE KeyPad_I2C_Addr, 0, 1, &HF0
  
END FUNCTION  

'<<<<<<<<<<<End of Library, Cut and throw away the Test Suite>>>>>>>>>>>>>>

TestSuite:

CPU 48

'Start I2C operations
I2C OPEN 400,100

'Initialise the Keypad Software
KeyPad_I2C_Init(&H26, 7)

'Do Forever
DO

  IF KeyPad_I2C_KeyCode<>&HFF THEN
    PRINT "Got the key ";KeyPad_I2C_KeyCode
    KeyPad_I2C_KeyCode=&HFF
  ENDIF
  
LOOP

END
