  ' moon_phases.bas         March 21, 2019
  
  ' this MMBASIC program calculates the calendar date
  ' and UTC time of the phases of the Moon
  
  ' MMBASIC eXtreme, Plus, RasPi and DOS version
  
  ''''''''''''''''''''''''''''''''''''''''''''''
  
  option default float
  
  option base 1
  
  ' dimension global arrays
  
  dim jdleap(28), leapsec(28), month$(12)
  
  dim iphase%, jdtdbi
  
  ' global constants
  
  const pi2 = 2.0 * pi, pidiv2 = 0.5 * pi, dtr = pi / 180.0
  
  CONST rtd = 180.0 / pi, atr = dtr / 3600.0
  
  ' read leap second data
  
  for i% = 1 to 28
    
    read jdleap(i%), leapsec(i%)
    
  next i%
  
  data 2441317.5,  10.0
  data 2441499.5,  11.0
  data 2441683.5,  12.0
  data 2442048.5,  13.0
  data 2442413.5,  14.0
  data 2442778.5,  15.0
  data 2443144.5,  16.0
  data 2443509.5,  17.0
  data 2443874.5,  18.0
  data 2444239.5,  19.0
  data 2444786.5,  20.0
  data 2445151.5,  21.0
  data 2445516.5,  22.0
  data 2446247.5,  23.0
  data 2447161.5,  24.0
  data 2447892.5,  25.0
  data 2448257.5,  26.0
  data 2448804.5,  27.0
  data 2449169.5,  28.0
  data 2449534.5,  29.0
  data 2450083.5,  30.0
  data 2450630.5,  31.0
  data 2451179.5,  32.0
  data 2453736.5,  33.0
  data 2454832.5,  34.0
  DATA 2456109.5,  35.0
  data 2457204.5,  36.0
  data 2457754.5,  37.0
  
  ' calendar months
  
  month$(1) = "January"
  month$(2) = "February"
  month$(3) = "March"
  month$(4) = "April"
  month$(5) = "May"
  month$(6) = "June"
  month$(7) = "July"
  month$(8) = "August"
  month$(9) = "September"
  month$(10) = "October"
  month$(11) = "November"
  month$(12) = "December"

  ' define root bracketing control parameters

  factor = 0.025

  dxmax = 1.0

  ' define root-finding tolerance

  rtol = 1.0e-8
  
  ''''''''''''''''''
  ' begin simulation
  ''''''''''''''''''
  
  print " "
  print "phases of the Moon"
  print "=================="
  print " "
  
  do
    
    ' request calendar month
    
    print "please input the calendar month (1 = January, 2 = February, etc.)"
    
    input cmonth
    
  loop until(cmonth >= 1 and cmonth <= 12)
  
  ' request calendar year
  
  print " "
  
  print "please input the calendar year (include all digits)"
  
  input cyear

  ' initial guess for julian day of new moon
    
  julian(cmonth, 1.0, cyear, jdtdb)

  z = (jdtdb - 2415021.065) / 29.53058868

  kp = int(z + 0.5)

  if (z < 0.0) then
  
     kp = int(z - 0.5)
  
  end if

  q = 29.53058868 * kp / 36525.0

  jdtdbi = 2415020.75933 + 29.53058868 * kp + 1.178e-4 * q * q - 1.55e-7 * q * q * q
  
  jdtdbi = jdtdbi + 3.3e-4 * sin(dtr * (166.56 + 132.87 * q - 9.173e-3 * q * q))
           
  ' process each lunar phase
  
  for iphase% = 0 to 3
    
    print " "
    
    select case iphase%
        
      case (0)
        
        print " "
        print "new Moon"
        print "--------"
       
        realroot1(0, 2.0, tol, xroot, froot)
        
      case (1)
        
        print " "
        print "first quarter"
        print "-------------"
        
        x1in = xroot
    
        x2in = x1in + 7.0
    
        broot(x1in, x2in, factor, dxmax, x1, x2)      

        realroot1(x1, x2, tol, xroot, froot)
            
      case (2)
        
        print " "
        print "full Moon"
        print "---------"
        
        x1in = xroot
    
        x2in = x1in + 7.0
    
        broot(x1in, x2in, factor, dxmax, x1, x2)     

        realroot1(x1, x2, tol, xroot, froot)
                
      case (3)
        
        print " "
        print "last quarter"
        print "------------"
        
        x1in = xroot
    
        x2in = x1in + 7.0
    
        broot(x1in, x2in, factor, dxmax, x1, x2)  

        realroot1(x1, x2, tol, xroot, froot) 
               
    end select
          
    ' TDB julian day of current event
    
    jdtdb = jdtdbi + xroot
    
    ' compute UTC julian day
    
    tdb2utc(jdtdb, jdutc)
    
    ' print results for each phase
    
    print " "
    
    jd2str(jdutc)
    
  next iphase%
  
  print " "
  
end
    
  '''''''''''''''''''
  '''''''''''''''''''
  
sub phase_func(x, fx)
  
  ' lunar phase objective function
  
  ' Celestial Computing with MMBASIC

  ''''''''''''''''''''''''''''''''''
  
  local jdtdb, plon_sun, plon_moon
  
  ' current tdb julian day
  
  jdtdb = jdtdbi + x
  
  ' compute solar and lunar geocentric ecliptic longitudes
  
  sun_moon(jdtdb, plon_sun, plon_moon)
  
  ' compute value of current objective function (-pi <= fx <= pi)
    
  fx = modulo((plon_sun - plon_moon) - iphase% * pidiv2 + pi) - pi
    
end sub
  
  '''''''''''''''''''''''''''''''''
  '''''''''''''''''''''''''''''''''
  
sub utc2tdb (jdutc, tai_utc, jdtdb)
  
  ' convert UTC julian date to TDB julian date
  
  ' input
  
  '  jdutc   = UTC julian day
  '  tai_utc = TAI-UTC (seconds)
  
  ' output
  
  '  jdtdb = TDB julian day
  
  ' Reference Frames in Astronomy and Geophysics
  ' J. Kovalevsky et al., 1989, pp. 439-442
  
  '''''''''''''''''''''''''''''''''''''''''
  
  local corr, jdtdt, t
  
  ' TDT julian date
  
  corr = (tai_utc + 32.184) / 86400.0
  
  jdtdt = jdutc + corr
  
  ' time argument for correction
  
  t = (jdtdt - 2451545.0) / 36525.0
  
  ' compute correction in microseconds
  
  corr = 1656.675 * sin(dtr * (35999.3729 * t + 357.5287))
  corr = corr + 22.418     * sin(dtr * (32964.467  * t + 246.199))
  corr = corr + 13.84      * sin(dtr * (71998.746  * t + 355.057))
  corr = corr +  4.77      * sin(dtr * ( 3034.906  * t +  25.463))
  corr = corr +  4.677     * sin(dtr * (34777.259  * t + 230.394))
  corr = corr + 10.216 * t * sin(dtr * (35999.373  * t + 243.451))
  corr = corr +  0.171 * t * sin(dtr * (71998.746  * t + 240.98 ))
  corr = corr +  0.027 * t * sin(dtr * ( 1222.114  * t + 194.661))
  corr = corr +  0.027 * t * sin(dtr * ( 3034.906  * t + 336.061))
  corr = corr +  0.026 * t * sin(dtr * (  -20.186  * t +   9.382))
  corr = corr +  0.007 * t * sin(dtr * (29929.562  * t + 264.911))
  corr = corr +  0.006 * t * sin(dtr * (  150.678  * t +  59.775))
  corr = corr +  0.005 * t * sin(dtr * ( 9037.513  * t + 256.025))
  corr = corr +  0.043 * t * sin(dtr * (35999.373  * t + 151.121))
  
  ' convert corrections to days
  
  corr = 0.000001 * corr / 86400.0
  
  ' TDB julian date
  
  jdtdb = jdtdt + corr
  
end sub
  
  ''''''''''''''''''''''''
  ''''''''''''''''''''''''
  
sub tdb2utc (jdtdb, jdutc)
  
  ' convert TDB julian day to UTC julian day subroutine
  
  ' input
  
  '  jdtdb = TDB julian day
  
  ' output
  
  '  jdutc = UTC julian day
  
  '''''''''''''''''''''''''
  
  local x1, x2, xroot, froot
  
  jdsaved = jdtdb
  
  ' set lower and upper bounds
  
  x1 = jdsaved - 0.1
  
  x2 = jdsaved + 0.1
  
  ' solve for UTC julian day using Brent's method
  
  realroot2(x1, x2, 1.0e-8, xroot, froot)
  
  jdutc = xroot
  
end sub
  
  '''''''''''''''''''
  '''''''''''''''''''
  
sub jdfunc (jdin, fx)
  
  ' objective function for tdb2utc
  
  ' input
  
  '  jdin = current value for UTC julian day
  
  ' output
  
  '  fx = delta julian day
  
  ''''''''''''''''''''''''
  
  local jdwrk, tai_utc
  
  find_leap(jdin, tai_utc)
  
  utc2tdb(jdin, tai_utc, jdwrk)
  
  fx = jdwrk - jdsaved
  
end sub
  
  ''''''''''''''''''''''''''''''''''''''
  ''''''''''''''''''''''''''''''''''''''
  
sub sun_moon (jdate, plon_sun, plon_moon)
  
  ' lunar and solar longitudes subroutine

  ' Celestial Computing with MMBASIC

  ''''''''''''''''''''''''''''''''''
  
  local djd, l, t
  
  djd = jdate - 2451545.0
  
  t = (djd / 36525.0) + 1.0
  
  gm = r2r(0.374897 + 0.03629164709 * djd)
  gm2 = modulo(2 * gm)
  gm3 = modulo(3 * gm)
  fm = r2r(0.259091 + 0.0367481952 * djd)
  fm2 = modulo(2 * fm)
  em = r2r(0.827362 + 0.03386319198 * djd)
  em2 = modulo(2 * em)
  em4 = modulo(4 * em)
  gs = r2r(0.993126 + 0.0027377785 * djd)
  lv = r2r(0.505498 + 0.00445046867 * djd)
  lm = r2r(0.606434 + 0.03660110129 * djd)
  ls = r2r(0.779072 + 0.00273790931 * djd)
  g2 = r2r(0.140023 + 0.00445036173 * djd)
  g4 = r2r(0.053856 + 0.00145561327 * djd)
  g5 = r2r(0.056531 + 0.00023080893 * djd)
  rm = r2r(0.347343 - 0.00014709391 * djd)
  
  l = 22640 * sin(gm) - 4586 * sin(gm - em2) + 2370 * sin(em2)
  l = l + 769 * sin(gm2) - 668 * sin(gs) - 412 * sin(fm2)
  l = l - 212 * sin(gm2 - em2) - 206 * sin(gm - em2 + gs)
  l = l + 192 * sin(gm + em2) + 165 * sin(em2 - gs)
  l = l + 148 * sin(gm - gs) - 125 * sin(em) - 110 * sin(gm + gs)
  l = l - 55 * sin(fm2 - em2) - 45 * sin(gm + fm2) + 40 * sin(gm - fm2)
  l = l - 38 * sin(gm - em4) + 36 * sin(gm3) - 31 * sin(gm2 - em4)
  l = l + 28 * sin(gm - em2 - gs) - 24 * sin(em2 + gs) + 19 * sin(gm - em)
  l = l + 18 * sin(em + gs) + 15 * sin(gm + em2 - gs) + 14 * sin(gm2 + em2)
  l = l + 14 * sin(em4) - 13 * sin(gm3 - em2) - 17 * sin(rm)
  l = l - 11 * sin(gm + 16 * ls - 18 * lv) + 10 * sin(gm2 - gs) + 9 * sin(gm - fm2 - em2)
  l = l + 9 * (cos(gm + 16 * ls - 18 * lv) - sin(gm2 - em2 + gs)) - 8 * sin(gm + em)
  l = l + 8 * (sin(2 * (em - gs)) - sin(gm2 + gs)) - 7 * (sin(2 * gs) + sin(gm - 2 * (em - gs)) - sin(rm))
  l = l - 6 * (sin(gm - fm2 + em2) + sin(fm2 + em2)) - 4 * (sin(gm - em4 + gs) - t * cos(gm + 16 * ls - 18 * lv))
  l = l - 4 * (sin(gm2 + fm2) - t * sin(gm + 16 * ls - 18 * lv))
  l = l + 3 * (sin(gm - 3 * em) - sin(gm + em2 + gs) - sin(gm2 - em4 + gs) + sin(gm - 2 * gs) + sin(gm - em2 - 2 * gs))
  l = l - 2 * (sin(gm2 - em2 - gs) + sin(fm2 - em2 + gs) - sin(gm + em4))
  l = l + 2 * (sin(4 * gm) + sin(em4 - gs) + sin(gm2 - em))
  
  plon_moon = lm + atr * l
  
  l = 6910 * sin(gs) + 72 * sin(2 * gs) - 17 * t * sin(gs)
  l = l - 7 * cos(gs - g5) + 6 * sin(lm - ls) + 5 * sin(4 * gs - 8 * g4 + 3 * g5)
  l = l - 5 * cos(2 * (gs - g2)) - 4 * (sin(gs - g2) - cos(4 * gs - 8 * g4 + 3 * g5))
  l = l + 3 * (sin(2 * (gs - g2)) - sin(g5) - sin(2 * (gs - g5)))
  
  plon_sun = ls + atr * (l - 17.0 * sin(rm))
  
end sub
  
  ''''''''''''''''''''''
  ''''''''''''''''''''''
  
function r2r(x) as float
  
  ' revolutions to radians function
  
  ' input
  
  '  x = argument (revolutions; 0 <= x <= 1)
  
  ' output
  
  '  r2r = equivalent x (radians; 0 <= y <= 2 pi)
  
  '''''''''''''''''''''''''''''''
  
  r2r = pi2 * (x - fix(x))
  
end function

''''''''''''''''''''''''''''''''''''''''''''''''''
''''''''''''''''''''''''''''''''''''''''''''''''''

sub broot(x1in, x2in, factor, dxmax, x1out, x2out)

  ' bracket a single root of a nonlinear equation

  ' input

  '  x1in   = initial guess for first bracketing x value
  '  x2in   = initial guess for second bracketing x value
  '  factor = acceleration factor (non-dimensional)
  '  dxmax  = rectification interval

  ' output

  '  x1out = final value for first bracketing x value
  '  x2out = final value for second bracketing x value

  ' Celestial Computing with MMBASIC

  ''''''''''''''''''''''''''''''''''

  ' evaluate objective function at initial value

  phase_func(x1in, f1)

  ' save initial value

  x3 = x1in
    
  ' save initial delta-x

  dx = x2in - x1in

  ' perform bracketing until the product of the
  ' two function values is negative

  do
  
   ' geometrically accelerate the second point
       
   x2in = x2in + factor * (x2in - x3)
   
   ' evaluate objective function at x2
       
   phase_func(x2in, f2)
       
   ' check to see if rectification is required
       
   if (abs(x2in - x3) > dxmax) then
     
      x3 = x2in - dx
      
   end if

   ' is the root bracketed?

  loop until ((f1 * f2) < 0.0)

  x1out = x1in

  x2out = x2in

end sub
  
  ''''''''''''''''''''''''''''''''''''''
  ''''''''''''''''''''''''''''''''''''''
  
sub realroot1(x1, x2, tol, xroot, froot)
  
  ' real root of a single non-linear function subroutine
  
  ' input
  
  '  x1  = lower bound of search interval
  '  x2  = upper bound of search interval
  '  tol = convergence criter%ia
  
  ' output
  
  '  xroot = real root of f(x) = 0
  '  froot = function value
  
  ' note: requires sub phase_func
  
  '''''''''''''''''''''''''''''''
  
  local eps, a, b, c, d, e, fa, fb, fcc, tol1
  
  local xm, p, q, r, s, xmin, tmp
  
  eps = 2.23e-16
  
  e = 0.0
  
  a = x1
  
  b = x2
  
  phase_func(a, fa)
  
  phase_func(b, fb)
  
  fcc = fb
  
  for iter% = 1 to 50
    
    if (fb * fcc > 0.0) then
      
      c = a
      
      fcc = fa
      
      d = b - a
      
      e = d
      
    end if
    
    if (abs(fcc) < abs(fb)) then
      
      a = b
      
      b = c
      
      c = a
      
      fa = fb
      
      fb = fcc
      
      fcc = fa
      
    end if
    
    tol1 = 2.0 * eps * abs(b) + 0.5 * tol
    
    xm = 0.5 * (c - b)
    
    if (abs(xm) <= tol1 or fb = 0.0) then exit for
    
    if (abs(e) >= tol1 and abs(fa) > abs(fb)) then
      
      s = fb / fa
      
      if (a = c) then
        
        p = 2.0 * xm * s
        
        q = 1.0 - s
        
      else
        
        q = fa / fcc
        
        r = fb / fcc
        
        p = s * (2.0 * xm * q * (q - r) - (b - a) * (r - 1.0))
        
        q = (q - 1.0) * (r - 1.0) * (s - 1.0)
        
      end if
      
      if (p > 0.0) then q = -q
      
      p = abs(p)
      
      min = abs(e * q)
      
      tmp = 3.0 * xm * q - abs(tol1 * q)
      
      if (min < tmp) then min = tmp
      
      if (2.0 * p < min) then
        
        e = d
        
        d = p / q
        
      else
        
        d = xm
        
        e = d
        
      end if
      
    else
      
      d = xm
      
      e = d
      
    end if
    
    a = b
    
    fa = fb
    
    if (abs(d) > tol1) then
      
      b = b + d
      
    else
      
      b = b + sgn(xm) * tol1
      
    end if
    
    phase_func(b, fb)
    
  next iter%
  
  froot = fb
  
  xroot = b
  
end sub
  
  ''''''''''''''''''''''''''''''''''''''
  ''''''''''''''''''''''''''''''''''''''
  
sub realroot2(x1, x2, tol, xroot, froot)
  
  ' real root of a single non-linear function subroutine
  
  ' input
  
  '  x1  = lower bound of search interval
  '  x2  = upper bound of search interval
  '  tol = convergence criter%ia
  
  ' output
  
  '  xroot = real root of f(x) = 0
  '  froot = function value
  
  ' note: requires sub jdfunc
  
  '''''''''''''''''''''''''''
  
  local eps, a, b, c, d, e, fa, fb, fcc, tol1
  
  local xm, p, q, r, s, xmin, tmp
  
  eps = 2.23e-16
  
  e = 0.0
  
  a = x1
  
  b = x2
  
  jdfunc(a, fa)
  
  jdfunc(b, fb)
  
  fcc = fb
  
  for iter% = 1 to 50
    
    if (fb * fcc > 0.0) then
      
      c = a
      
      fcc = fa
      
      d = b - a
      
      e = d
      
    end if
    
    if (abs(fcc) < abs(fb)) then
      
      a = b
      
      b = c
      
      c = a
      
      fa = fb
      
      fb = fcc
      
      fcc = fa
      
    end if
    
    tol1 = 2.0 * eps * abs(b) + 0.5 * tol
    
    xm = 0.5 * (c - b)
    
    if (abs(xm) <= tol1 or fb = 0) then exit for
    
    if (abs(e) >= tol1 and abs(fa) > abs(fb)) then
      
      s = fb / fa
      
      if (a = c) then
        
        p = 2.0 * xm * s
        
        q = 1.0 - s
        
      else
        
        q = fa / fcc
        
        r = fb / fcc
        
        p = s * (2.0 * xm * q * (q - r) - (b - a) * (r - 1.0))
        
        q = (q - 1.0) * (r - 1.0) * (s - 1.0)
        
      end if
      
      if (p > 0) then q = -q
      
      p = abs(p)
      
      xmin = abs(e * q)
      
      tmp = 3.0 * xm * q - abs(tol1 * q)
      
      if (xmin < tmp) then xmin = tmp
      
      if (2.0 * p < xmin) then
        
        e = d
        
        d = p / q
        
      else
        
        d = xm
        
        e = d
        
      end if
      
    else
      
      d = xm
      
      e = d
      
    end if
    
    a = b
    
    fa = fb
    
    if (abs(d) > tol1) then
      
      b = b + d
      
    else
      
      b = b + sgn(xm) * tol1
      
    end if
    
    jdfunc(b, fb)
    
  next iter%
  
  froot = fb
  
  xroot = b
  
end sub
  
  ''''''''''''''''''''''''''''''''
  ''''''''''''''''''''''''''''''''
  
sub julian(month, day, year, jday)
  
  ' Gregorian date to julian day subroutine
  
  ' input
  
  '  month = calendar month
  '  day   = calendar day
  '  year  = calendar year (all four digits)
  
  ' output
  
  '  jday = julian day
  
  ' special notes
  
  '  (1) calendar year must include all digits
  
  '  (2) will report October 5, 1582 to October 14, 1582
  '      as invalid calendar dates and exit

  ' Celestial Computing with MMBASIC

  ''''''''''''''''''''''''''''''''''
  
  local a, b, c, m, y
  
  y = year
  
  m = month
  
  b = 0.0
  
  c = 0.0
  
  if (m <= 2.0) then
    
    y = y - 1.0
    
    m = m + 12.0
    
  end if
  
  if (y < 0.0) then c = -0.75
  
  if (year < 1582.0) then
    
    ' null
    
  elseif (year > 1582.0) then
    
    a = fix(y / 100.0)
    
    b = 2.0 - a + fix(a / 4.0)
    
  elseif (month < 10.0) then
    
    ' null
    
  elseif (month > 10.0) then
    
    a = fix(y / 100.0)
    
    b = 2.0 - a + fix(a / 4.0)
    
  elseif (day <= 4.0) then
    
    ' null
    
  elseif (day > 14.0) then
    
    a = fix(y / 100.0)
    
    b = 2.0 - a + fix(a / 4.0)
    
  else
    
    print "this date does not exist!!"
    
    exit
    
  end if
  
  jday = fix(365.25 * y + c) + fix(30.6001 * (m + 1.0)) + day + b + 1720994.5
  
end sub
  
  ''''''''''''''''''''''''''''''''
  ''''''''''''''''''''''''''''''''
  
sub gdate (jday, month, day, year)
  
  ' Julian day to Gregorian date subroutine
  
  ' input
  
  '  jday = julian day
  
  ' output
  
  '  month = calendar month
  '  day   = calendar day
  '  year  = calendar year

  ' Celestial Computing with MMBASIC

  ''''''''''''''''''''''''''''''''''
  
  local a, b, c, d, e, f, z, alpha
  
  z = fix(jday + 0.5)
  
  f = jday + 0.5 - z
  
  if (z < 2299161) then
    
    a = z
    
  else
    
    alpha = fix((z - 1867216.25) / 36524.25)
    
    a = z + 1.0 + alpha - fix(alpha / 4.0)
    
  end if
  
  b = a + 1524.0
  
  c = fix((b - 122.1) / 365.25)
  
  d = fix(365.25 * c)
  
  e = fix((b - d) / 30.6001)
  
  day = b - d - fix(30.6001 * e) + f
  
  if (e < 13.5) then
    
    month = e - 1.0
    
  else
    
    month = e - 13.0
    
  end if
  
  if (month > 2.5) then
    
    year = c - 4716.0
    
  else
    
    year = c - 4715.0
    
  end if
  
end sub
  
  '''''''''''''''
  '''''''''''''''
  
sub jd2str(jdutc)
  
  ' convert julian day to calendar date and UTC time

  ' Celestial Computing with MMBASIC

  ''''''''''''''''''''''''''''''''''
  
  local cmonth, day, year
  
  gdate(jdutc, cmonth, day, year)
  
  print "calendar date  ", month$(cmonth) + " " + STR$(int(day)) + " " + str$(year)
  
  print " "
  
  thr0 = 24.0 * (day - int(day))
  
  thr = int(thr0)
  
  tmin0 = 60.0 * (thr0 - thr)
  
  tmin = int(tmin0)
  
  tsec = 60.0 * (tmin0 - tmin)
  
  ' fix seconds and minutes for rollover
  
  if (tsec >= 60.0) then
    
    tsec = 0.0
    
    tmin = tmin + 1.0
    
  end if
  
  ' fix minutes for rollover
  
  if (tmin >= 60.0) then
    
    tmin = 0.0
    
    thr = thr + 1.0
    
  end if
  
  print "UTC time       ", str$(thr) + " hours " + str$(tmin) + " minutes " + str$(tsec, 0, 2) + " seconds"
  
end sub
  
  '''''''''''''''''''''''''''''
  '''''''''''''''''''''''''''''
  
sub find_leap(jday, leapsecond)
  
  ' find number of leap seconds for utc julian day
  
  ' input
  
  '  jday = utc julian day
  
  ' input via global
  
  '  jdleap  = array of utc julian dates
  '  leapsec = array of leap seconds
  
  ' output
  
  '  leapsecond = number of leap seconds

  ' Celestial Computing with MMBASIC

  ''''''''''''''''''''''''''''''''''
  
  if (jday <= jdleap(1)) then
    
    ' date is <= 1972; set to first data element
    
    leapsecond = leapsec(1)
    
    exit sub
    
  end if
  
  if (jday >= jdleap(28)) then
    
    ' date is >= end of current data
    ' set to last data element
    
    leapsecond = leapsec(28)
    
    exit sub
    
  end if
  
  ' find data within table
  
  for i% = 1 to 27
    
    if (jday >= jdleap(i%) and jday < jdleap(i% + 1)) then
      
      leapsecond = leapsec(i%)
      
      exit sub
      
    end if
    
  next i%
  
end sub
  
  '''''''''''''''''''''''''
  '''''''''''''''''''''''''
  
function modulo(x) as float
  
  ' modulo 2 pi function

  ' Celestial Computing with MMBASIC

  ''''''''''''''''''''''''''''''''''  
  
  local a
  
  a = x - pi2 * fix(x / pi2)
  
  if (a < 0.0) then
    
    a = a + pi2
    
  end if
  
  modulo = a
  
end function
  
  '''''''''''''''''''''''''''
  '''''''''''''''''''''''''''
  
function atan3(a, b) as float
  
  ' four quadrant inverse tangent function
  
  ' input
  
  '  a = sine of angle
  '  b = cosine of angle
  
  ' output
  
  '  atan3 = angle (0 =< atan3 <= 2 * pi; radians)

  ' Celestial Computing with MMBASIC

  ''''''''''''''''''''''''''''''''''
  
  local c
  
  if (abs(a) < 1.0e-10) then
    
    atan3 = (1.0 - sgn(b)) * pidiv2
    
    exit function
    
  else
    
    c = (2.0 - sgn(a)) * pidiv2
    
  endif
  
  if (abs(b) < 1.0e-10) then
    
    atan3 = c
    
    exit function
    
  else
    
    atan3 = c + sgn(a) * sgn(b) * (abs(atn(a / b)) - pidiv2)
    
  endif
  
end function
  
  ''''''''''
  ''''''''''
  
sub keycheck
  
  ' check user response subroutine

  ' Celestial Computing with MMBASIC

  ''''''''''''''''''''''''''''''''''
  
  local a$
  
  print " "
  
  print "< press Enter key to continue >"
  
  a$ = ""
  
  do while a$ = ""
    
    a$ = inkey$
    
  loop
  
end sub
  
