' moon_phases.bas        November 3, 2021

' calculates the calendar date
' and UTC time of the phases of the Moon

' MMBASIC double precision

''''''''''''''''''''''''''

Option default float

Option base 1

' dimension global arrays

Dim jdleap(28), leapsec(28), month$(12)

Dim iphase%, jdtdbi

' global constants

Const pi2 = 2.0 * Pi, pidiv2 = 0.5 * Pi, dtr = Pi / 180.0

Const rtd = 180.0 / Pi, atr = dtr / 3600.0

' read leap second data

For i% = 1 To 28

Read jdleap(i%), leapsec(i%)

Next i%

Data 2441317.5,  10.0
Data 2441499.5,  11.0
Data 2441683.5,  12.0
Data 2442048.5,  13.0
Data 2442413.5,  14.0
Data 2442778.5,  15.0
Data 2443144.5,  16.0
Data 2443509.5,  17.0
Data 2443874.5,  18.0
Data 2444239.5,  19.0
Data 2444786.5,  20.0
Data 2445151.5,  21.0
Data 2445516.5,  22.0
Data 2446247.5,  23.0
Data 2447161.5,  24.0
Data 2447892.5,  25.0
Data 2448257.5,  26.0
Data 2448804.5,  27.0
Data 2449169.5,  28.0
Data 2449534.5,  29.0
Data 2450083.5,  30.0
Data 2450630.5,  31.0
Data 2451179.5,  32.0
Data 2453736.5,  33.0
Data 2454832.5,  34.0
Data 2456109.5,  35.0
Data 2457204.5,  36.0
Data 2457754.5,  37.0

' calendar months

month$(1) = "January"
month$(2) = "February"
month$(3) = "March"
month$(4) = "April"
month$(5) = "May"
month$(6) = "June"
month$(7) = "July"
month$(8) = "August"
month$(9) = "September"
month$(10) = "October"
month$(11) = "November"
month$(12) = "December"

' define root bracketing control parameters

factor = 0.025

dxmax = 1.0

' define root-finding tolerance

rtol = 1.0e-8

''''''''''''''''''
' begin simulation
''''''''''''''''''

CLS

Font 4

Colour RGB(yellow)

Text 0, 0, "----- PHASES OF THE MOON -----"

' request calendar month

GUI numberbox #1, 160, 60, 70, 40

CtrlVal(#1) = 0

Text 0, 73, "calendar month?"

Text 0, 130, "(1 = January, 2 = February, etc)"

Do

If (CtrlVal(#1) <> 0) Then

  cmonth = CtrlVal(#1)

  Exit

EndIf

Loop

CLS

Text 0, 0, "----- PHASES OF THE MOON -----"

' request calendar year

GUI numberbox #2, 160, 35, 70, 40

CtrlVal(#2) = 0

Text 0, 45, "calendar year?"

Text 0, 100, "(include all four digits)"

Do

If (CtrlVal(#2) <> 0) Then

  cyear = CtrlVal(#2)

  Exit

EndIf

Loop

CLS

' initial guess for julian day of new moon

julian(cmonth, 1.0, cyear, jdtdb)

z = (jdtdb - 2415021.065) / 29.53058868

kp = Int(z + 0.5)

If (z < 0.0) Then

 kp = Int(z - 0.5)

EndIf

q = 29.53058868 * kp / 36525.0

jdtdbi = 2415020.75933 + 29.53058868 * kp + 1.178e-4 * q * q - 1.55e-7 * q * q * q

jdtdbi = jdtdbi + 3.3e-4 * Sin(dtr * (166.56 + 132.87 * q - 9.173e-3 * q * q))

' process each lunar phase

For iphase% = 0 To 3

   Select Case iphase%

     Case (0)

       ' new moon

       realroot1(0, 2.0, tol, xroot, froot)

     Case (1)

       ' first quarter

       x1in = xroot

       x2in = x1in + 7.0

       broot(x1in, x2in, factor, dxmax, x1, x2)

       realroot1(x1, x2, tol, xroot, froot)

     Case (2)

       ' full Moon

       x1in = xroot

       x2in = x1in + 7.0

       broot(x1in, x2in, factor, dxmax, x1, x2)

       realroot1(x1, x2, tol, xroot, froot)

     Case (3)

       ' last quarter

       x1in = xroot

       x2in = x1in + 7.0

       broot(x1in, x2in, factor, dxmax, x1, x2)

       realroot1(x1, x2, tol, xroot, froot)

  End Select

' TDB julian day of current event

jdtdb = jdtdbi + xroot

Select Case iphase%

  Case (0)

    Text 0, 0, "NEW MOON"

    tdb2utc(jdtdb, jdutc)

    jd2str(jdutc, cdate$, utc$)

    Text 0, 18, "date        " + cdate$

    Text 0, 33, "UTC time    " + utc$

  Case (1)

    ' first quarter

    Text 0, 58, "FIRST QUARTER"

    tdb2utc(jdtdb, jdutc)

    jd2str(jdutc, cdate$, utc$)

    Text 0, 76, "date        " + cdate$

    Text 0, 91, "UTC time    " + utc$

  Case (2)

    ' full Moon

    Text 0, 116, "FULL MOON"

    tdb2utc(jdtdb, jdutc)

    jd2str(jdutc, cdate$, utc$)

    Text 0, 134, "date        " + cdate$

    Text 0, 149, "UTC time    " + utc$

  Case (3)

    ' last quarter

    Text 0, 174, "LAST QUARTER"

    tdb2utc(jdtdb, jdutc)

    jd2str(jdutc, cdate$, utc$)

    Text 0, 192, "date        " + cdate$

    Text 0, 207, "UTC time    " + utc$

    check4touch

  End Select

Next iphase%

' return to main menu

' Flash run 1

End

'''''''''''''''''''
'''''''''''''''''''

Sub phase_func(x, fx)

' lunar phase objective function

' Celestial Computing with MMBASIC

''''''''''''''''''''''''''''''''''

Local jdtdb, plon_sun, plon_moon

' current tdb julian day

jdtdb = jdtdbi + x

' compute solar and lunar geocentric ecliptic longitudes

sun_moon(jdtdb, plon_sun, plon_moon)

' compute value of current objective function (-pi <= fx <= pi)

fx = modulo((plon_sun - plon_moon) - iphase% * pidiv2 + Pi) - Pi

End Sub

'''''''''''''''''''''''''''''''''
'''''''''''''''''''''''''''''''''

Sub utc2tdb(jdutc, tai_utc, jdtdb)

' convert UTC julian date to TDB julian date

' input

'  jdutc   = UTC julian day
'  tai_utc = TAI-UTC (seconds)

' output

'  jdtdb = TDB julian day

' Reference Frames in Astronomy and Geophysics
' J. Kovalevsky et al., 1989, pp. 439-442

'''''''''''''''''''''''''''''''''''''''''

Local corr, jdtdt, t

' TDT julian date

corr = (tai_utc + 32.184) / 86400.0

jdtdt = jdutc + corr

' time argument for correction

t = (jdtdt - 2451545.0) / 36525.0

' compute correction in microseconds

corr = 1656.675 * Sin(dtr * (35999.3729 * t + 357.5287))
corr = corr + 22.418     * Sin(dtr * (32964.467  * t + 246.199))
corr = corr + 13.84      * Sin(dtr * (71998.746  * t + 355.057))
corr = corr +  4.77      * Sin(dtr * ( 3034.906  * t +  25.463))
corr = corr +  4.677     * Sin(dtr * (34777.259  * t + 230.394))
corr = corr + 10.216 * t * Sin(dtr * (35999.373  * t + 243.451))
corr = corr +  0.171 * t * Sin(dtr * (71998.746  * t + 240.98 ))
corr = corr +  0.027 * t * Sin(dtr * ( 1222.114  * t + 194.661))
corr = corr +  0.027 * t * Sin(dtr * ( 3034.906  * t + 336.061))
corr = corr +  0.026 * t * Sin(dtr * (  -20.186  * t +   9.382))
corr = corr +  0.007 * t * Sin(dtr * (29929.562  * t + 264.911))
corr = corr +  0.006 * t * Sin(dtr * (  150.678  * t +  59.775))
corr = corr +  0.005 * t * Sin(dtr * ( 9037.513  * t + 256.025))
corr = corr +  0.043 * t * Sin(dtr * (35999.373  * t + 151.121))

' convert corrections to days

corr = 0.000001 * corr / 86400.0

' TDB julian date

jdtdb = jdtdt + corr

End Sub

''''''''''''''''''''''''
''''''''''''''''''''''''

Sub tdb2utc(jdtdb, jdutc)

' convert TDB julian day to UTC julian day subroutine

' input

'  jdtdb = TDB julian day

' output

'  jdutc = UTC julian day

'''''''''''''''''''''''''

Local x1, x2, xroot, froot

jdsaved = jdtdb

' set lower and upper bounds

x1 = jdsaved - 0.1

x2 = jdsaved + 0.1

' solve for UTC julian day using Brent's method

realroot2(x1, x2, 1.0e-8, xroot, froot)

jdutc = xroot

End Sub

'''''''''''''''''''
'''''''''''''''''''

Sub jdfunc(jdin, fx)

' objective function for tdb2utc

' input

'  jdin = current value for UTC julian day

' output

'  fx = delta julian day

''''''''''''''''''''''''

Local jdwrk, tai_utc

find_leap(jdin, tai_utc)

utc2tdb(jdin, tai_utc, jdwrk)

fx = jdwrk - jdsaved

End Sub

''''''''''''''''''''''''''''''''''''''
''''''''''''''''''''''''''''''''''''''

Sub sun_moon(jdate, plon_sun, plon_moon)

' lunar and solar longitudes subroutine

' Celestial Computing with MMBASIC

''''''''''''''''''''''''''''''''''

Local djd, l, t

djd = jdate - 2451545.0

t = (djd / 36525.0) + 1.0

gm = r2r(0.374897 + 0.03629164709 * djd)
gm2 = modulo(2 * gm)
gm3 = modulo(3 * gm)
fm = r2r(0.259091 + 0.0367481952 * djd)
fm2 = modulo(2 * fm)
em = r2r(0.827362 + 0.03386319198 * djd)
em2 = modulo(2 * em)
em4 = modulo(4 * em)
gs = r2r(0.993126 + 0.0027377785 * djd)
lv = r2r(0.505498 + 0.00445046867 * djd)
lm = r2r(0.606434 + 0.03660110129 * djd)
ls = r2r(0.779072 + 0.00273790931 * djd)
g2 = r2r(0.140023 + 0.00445036173 * djd)
g4 = r2r(0.053856 + 0.00145561327 * djd)
g5 = r2r(0.056531 + 0.00023080893 * djd)
rm = r2r(0.347343 - 0.00014709391 * djd)

l = 22640 * Sin(gm) - 4586 * Sin(gm - em2) + 2370 * Sin(em2)
l = l + 769 * Sin(gm2) - 668 * Sin(gs) - 412 * Sin(fm2)
l = l - 212 * Sin(gm2 - em2) - 206 * Sin(gm - em2 + gs)
l = l + 192 * Sin(gm + em2) + 165 * Sin(em2 - gs)
l = l + 148 * Sin(gm - gs) - 125 * Sin(em) - 110 * Sin(gm + gs)
l = l - 55 * Sin(fm2 - em2) - 45 * Sin(gm + fm2) + 40 * Sin(gm - fm2)
l = l - 38 * Sin(gm - em4) + 36 * Sin(gm3) - 31 * Sin(gm2 - em4)
l = l + 28 * Sin(gm - em2 - gs) - 24 * Sin(em2 + gs) + 19 * Sin(gm - em)
l = l + 18 * Sin(em + gs) + 15 * Sin(gm + em2 - gs) + 14 * Sin(gm2 + em2)
l = l + 14 * Sin(em4) - 13 * Sin(gm3 - em2) - 17 * Sin(rm)
l = l - 11 * Sin(gm + 16 * ls - 18 * lv) + 10 * Sin(gm2 - gs) + 9 * Sin(gm - fm2 - em2)
l = l + 9 * (Cos(gm + 16 * ls - 18 * lv) - Sin(gm2 - em2 + gs)) - 8 * Sin(gm + em)
l = l + 8 * (Sin(2 * (em - gs)) - Sin(gm2 + gs)) - 7 * (Sin(2 * gs) + Sin(gm - 2 * (em - gs)) - Sin(rm))
l = l - 6 * (Sin(gm - fm2 + em2) + Sin(fm2 + em2)) - 4 * (Sin(gm - em4 + gs) - t * Cos(gm + 16 * ls - 18 * lv))
l = l - 4 * (Sin(gm2 + fm2) - t * Sin(gm + 16 * ls - 18 * lv))
l = l + 3 * (Sin(gm - 3 * em) - Sin(gm + em2 + gs) - Sin(gm2 - em4 + gs) + Sin(gm - 2 * gs) + Sin(gm - em2 - 2 * gs))
l = l - 2 * (Sin(gm2 - em2 - gs) + Sin(fm2 - em2 + gs) - Sin(gm + em4))
l = l + 2 * (Sin(4 * gm) + Sin(em4 - gs) + Sin(gm2 - em))

plon_moon = lm + atr * l

l = 6910 * Sin(gs) + 72 * Sin(2 * gs) - 17 * t * Sin(gs)
l = l - 7 * Cos(gs - g5) + 6 * Sin(lm - ls) + 5 * Sin(4 * gs - 8 * g4 + 3 * g5)
l = l - 5 * Cos(2 * (gs - g2)) - 4 * (Sin(gs - g2) - Cos(4 * gs - 8 * g4 + 3 * g5))
l = l + 3 * (Sin(2 * (gs - g2)) - Sin(g5) - Sin(2 * (gs - g5)))

plon_sun = ls + atr * (l - 17.0 * Sin(rm))

End Sub

''''''''''''''''''''''
''''''''''''''''''''''

Function r2r(x) As float

' revolutions to radians function

' input

'  x = argument (revolutions; 0 <= x <= 1)

' output

'  r2r = equivalent x (radians; 0 <= y <= 2 pi)

'''''''''''''''''''''''''''''''

r2r = pi2 * (x - Fix(x))

End Function

''''''''''''''''''''''''''''''''''''''''''''''''''
''''''''''''''''''''''''''''''''''''''''''''''''''

Sub broot(x1in, x2in, factor, dxmax, x1out, x2out)

' bracket a single root of a nonlinear equation

' input

'  x1in   = initial guess for first bracketing x value
'  x2in   = initial guess for second bracketing x value
'  factor = acceleration factor (non-dimensional)
'  dxmax  = rectification interval

' output

'  x1out = final value for first bracketing x value
'  x2out = final value for second bracketing x value

' Celestial Computing with MMBASIC

''''''''''''''''''''''''''''''''''

' evaluate objective function at initial value

phase_func(x1in, f1)

' save initial value

x3 = x1in

' save initial delta-x

dx = x2in - x1in

' perform bracketing until the product of the
' two function values is negative

Do

' geometrically accelerate the second point

x2in = x2in + factor * (x2in - x3)

' evaluate objective function at x2

phase_func(x2in, f2)

' check to see if rectification is required

If (Abs(x2in - x3) > dxmax) Then

  x3 = x2in - dx

EndIf

' is the root bracketed?

Loop Until ((f1 * f2) < 0.0)

x1out = x1in

x2out = x2in

End Sub

''''''''''''''''''''''''''''''''''''''
''''''''''''''''''''''''''''''''''''''

Sub realroot1(x1, x2, tol, xroot, froot)

' real root of a single non-linear function subroutine

' input

'  x1  = lower bound of search interval
'  x2  = upper bound of search interval
'  tol = convergence criter%ia

' output

'  xroot = real root of f(x) = 0
'  froot = function value

' note: requires sub phase_func

'''''''''''''''''''''''''''''''

Local eps, a, b, c, d, e, fa, fb, fcc, tol1

Local xm, p, q, r, s, xmin, tmp

eps = 2.23e-16

e = 0.0

a = x1

b = x2

phase_func(a, fa)

phase_func(b, fb)

fcc = fb

For iter% = 1 To 50

If (fb * fcc > 0.0) Then

  c = a

  fcc = fa

  d = b - a

  e = d

EndIf

If (Abs(fcc) < Abs(fb)) Then

  a = b

  b = c

  c = a

  fa = fb

  fb = fcc

  fcc = fa

EndIf

tol1 = 2.0 * eps * Abs(b) + 0.5 * tol

xm = 0.5 * (c - b)

If (Abs(xm) <= tol1 Or fb = 0.0) Then Exit For

If (Abs(e) >= tol1 And Abs(fa) > Abs(fb)) Then

  s = fb / fa

  If (a = c) Then

 p = 2.0 * xm * s

 q = 1.0 - s

  Else

 q = fa / fcc

 r = fb / fcc

 p = s * (2.0 * xm * q * (q - r) - (b - a) * (r - 1.0))

 q = (q - 1.0) * (r - 1.0) * (s - 1.0)

  EndIf

  If (p > 0.0) Then q = -q

  p = Abs(p)

  min = Abs(e * q)

  tmp = 3.0 * xm * q - Abs(tol1 * q)

  If (min < tmp) Then min = tmp

  If (2.0 * p < min) Then

 e = d

 d = p / q

  Else

 d = xm

 e = d

  EndIf

Else

  d = xm

  e = d

EndIf

a = b

fa = fb

If (Abs(d) > tol1) Then

  b = b + d

Else

  b = b + Sgn(xm) * tol1

EndIf

phase_func(b, fb)

Next iter%

froot = fb

xroot = b

End Sub

''''''''''''''''''''''''''''''''''''''
''''''''''''''''''''''''''''''''''''''

Sub realroot2(x1, x2, tol, xroot, froot)

' real root of a single non-linear function subroutine

' input

'  x1  = lower bound of search interval
'  x2  = upper bound of search interval
'  tol = convergence criter%ia

' output

'  xroot = real root of f(x) = 0
'  froot = function value

' note: requires sub jdfunc

'''''''''''''''''''''''''''

Local eps, a, b, c, d, e, fa, fb, fcc, tol1

Local xm, p, q, r, s, xmin, tmp

eps = 2.23e-16

e = 0.0

a = x1

b = x2

jdfunc(a, fa)

jdfunc(b, fb)

fcc = fb

For iter% = 1 To 50

If (fb * fcc > 0.0) Then

  c = a

  fcc = fa

  d = b - a

  e = d

EndIf

If (Abs(fcc) < Abs(fb)) Then

  a = b

  b = c

  c = a

  fa = fb

  fb = fcc

  fcc = fa

EndIf

tol1 = 2.0 * eps * Abs(b) + 0.5 * tol

xm = 0.5 * (c - b)

If (Abs(xm) <= tol1 Or fb = 0) Then Exit For

If (Abs(e) >= tol1 And Abs(fa) > Abs(fb)) Then

  s = fb / fa

  If (a = c) Then

 p = 2.0 * xm * s

 q = 1.0 - s

  Else

 q = fa / fcc

 r = fb / fcc

 p = s * (2.0 * xm * q * (q - r) - (b - a) * (r - 1.0))

 q = (q - 1.0) * (r - 1.0) * (s - 1.0)

  EndIf

  If (p > 0) Then q = -q

  p = Abs(p)

  xmin = Abs(e * q)

  tmp = 3.0 * xm * q - Abs(tol1 * q)

  If (xmin < tmp) Then xmin = tmp

  If (2.0 * p < xmin) Then

 e = d

 d = p / q

  Else

 d = xm

 e = d

  EndIf

Else

  d = xm

  e = d

EndIf

a = b

fa = fb

If (Abs(d) > tol1) Then

  b = b + d

Else

  b = b + Sgn(xm) * tol1

EndIf

jdfunc(b, fb)

Next iter%

froot = fb

xroot = b

End Sub

''''''''''''''''''''''''''''''''
''''''''''''''''''''''''''''''''

Sub julian(month, day, year, jday)

' Gregorian date to julian day subroutine

' input

'  month = calendar month
'  day   = calendar day
'  year  = calendar year (all four digits)

' output

'  jday = julian day

' special notes

'  (1) calendar year must include all digits

'  (2) will report October 5, 1582 to October 14, 1582
'      as invalid calendar dates and exit

' Celestial Computing with MMBASIC

''''''''''''''''''''''''''''''''''

Local a, b, c, m, y

y = year

m = month

b = 0.0

c = 0.0

If (m <= 2.0) Then

y = y - 1.0

m = m + 12.0

EndIf

If (y < 0.0) Then c = -0.75

If (year < 1582.0) Then

' null

ElseIf (year > 1582.0) Then

a = Fix(y / 100.0)

b = 2.0 - a + Fix(a / 4.0)

ElseIf (month < 10.0) Then

' null

ElseIf (month > 10.0) Then

a = Fix(y / 100.0)

b = 2.0 - a + Fix(a / 4.0)

ElseIf (day <= 4.0) Then

' null

ElseIf (day > 14.0) Then

a = Fix(y / 100.0)

b = 2.0 - a + Fix(a / 4.0)

Else

Print "this date does not exist!!"

Exit

EndIf

jday = Fix(365.25 * y + c) + Fix(30.6001 * (m + 1.0)) + day + b + 1720994.5

End Sub

''''''''''''''''''''''''''''''''
''''''''''''''''''''''''''''''''

Sub gdate(jday, month, day, year)

' Julian day to Gregorian date subroutine

' input

'  jday = julian day

' output

'  month = calendar month
'  day   = calendar day
'  year  = calendar year

' Celestial Computing with MMBASIC

''''''''''''''''''''''''''''''''''

Local a, b, c, d, e, f, z, alpha

z = Fix(jday + 0.5)

f = jday + 0.5 - z

If (z < 2299161) Then

a = z

Else

alpha = Fix((z - 1867216.25) / 36524.25)

a = z + 1.0 + alpha - Fix(alpha / 4.0)

EndIf

b = a + 1524.0

c = Fix((b - 122.1) / 365.25)

d = Fix(365.25 * c)

e = Fix((b - d) / 30.6001)

day = b - d - Fix(30.6001 * e) + f

If (e < 13.5) Then

month = e - 1.0

Else

month = e - 13.0

EndIf

If (month > 2.5) Then

year = c - 4716.0

Else

year = c - 4715.0

EndIf

End Sub

'''''''''''''''''''''''''''''''
'''''''''''''''''''''''''''''''

Sub jd2str(jdutc, cdate$, utc$)

' convert julian date to date and time strings

Local thr, tmin0, tmin, tsec

'''''''''''''''''''''''''''''

gdate(jdutc, cmonth, day, year)

lmonth = Len(month$(cmonth))

cdate$ = Left$(month$(cmonth), lmonth) + " "

cdate$ = cdate$ + Str$(Int(day)) + ", " + Str$(year)

thours = 24.0 * (day - Int(day))

thr = Fix(thours)

tmin0 = 60.0 * (thours - thr)

tmin = Fix(tmin0)

tsec = 60.0 * (tmin0 - tmin)

' fix seconds and minutes for rollover

If (tsec >= 60.0) Then

  tsec = 0.0

  tmin = tmin + 1.0

EndIf

' fix minutes for rollover

If (tmin >= 60.0) Then

 tmin = 0.0

 thr = thr + 1

EndIf

thour$ = Str$(thr)

If (thr < 10) Then

  thour$ = "0" + thour$

EndIf

tminute$ = Str$(tmin)

If (tmin < 10) Then

  tminute$ = "0" + tminute$

EndIf

tsecond$ = Str$(Int(tsec))

If (tsec < 10) Then

  tsecond$ = "0" + tsecond$

EndIf

utc$ = thour$ + ":" + tminute$ + ":" + tsecond$

End Sub

'''''''''''''''''''''''''''''
'''''''''''''''''''''''''''''

Sub find_leap(jday, leapsecond)

' find number of leap seconds for utc julian day

' input

'  jday = utc julian day

' input via global

'  jdleap  = array of utc julian dates
'  leapsec = array of leap seconds

' output

'  leapsecond = number of leap seconds

' Celestial Computing with MMBASIC

''''''''''''''''''''''''''''''''''

If (jday <= jdleap(1)) Then

' date is <= 1972; set to first data element

leapsecond = leapsec(1)

Exit Sub

EndIf

If (jday >= jdleap(28)) Then

' date is >= end of current data
' set to last data element

leapsecond = leapsec(28)

Exit Sub

EndIf

' find data within table

For i% = 1 To 27

If (jday >= jdleap(i%) And jday < jdleap(i% + 1)) Then

  leapsecond = leapsec(i%)

  Exit Sub

EndIf

Next i%

End Sub

'''''''''''''''''''''''''
'''''''''''''''''''''''''

Function modulo(x) As float

' modulo 2 pi function

' Celestial Computing with MMBASIC

''''''''''''''''''''''''''''''''''

Local a

a = x - pi2 * Fix(x / pi2)

If (a < 0.0) Then

a = a + pi2

EndIf

modulo = a

End Function

'''''''''''''''''''''''''''
'''''''''''''''''''''''''''

Function atan3(a, b) As float

' four quadrant inverse tangent function

' input

'  a = sine of angle
'  b = cosine of angle

' output

'  atan3 = angle (0 <= atan3 <= 2 * pi; radians)

' Celestial Computing with MMBASIC

''''''''''''''''''''''''''''''''''

Local c

If (Abs(a) < 1.0e-10) Then

atan3 = (1.0 - Sgn(b)) * pidiv2

Exit Function

Else

c = (2.0 - Sgn(a)) * pidiv2

EndIf

If (Abs(b) < 1.0e-10) Then

atan3 = c

Exit Function

Else

atan3 = c + Sgn(a) * Sgn(b) * (Abs(Atn(a / b)) - pidiv2)

EndIf

End Function

'''''''''''''''
'''''''''''''''

Sub check4touch

' check for user touching screen

Font 1

Text 50, 225, "touch screen to continue"

Font 4

Do

   If (Touch(down) <> 0) Then

     Exit

   EndIf

Loop

End Sub
