// Includes ------------------------------------------------------------------//
#include "GFX.h"
#include "fonts.h"

#define POLY_Y(Z)          ((int32_t)((Points + Z)->X))
#define POLY_X(Z)          ((int32_t)((Points + Z)->Y))   

#define ABS(X)  ((X) > 0 ? (X) : -(X))    

// This variable are for compatibility issue
/* Global variables to set the written text color */
uint16_t CurrentTextColor   = 0x0000;
uint16_t CurrentBackColor   = 0xFFFF;
/* Default LCD configuration with LCD Layer 1 */
uint32_t CurrentFrameBuffer = LCD_FRAME_BUFFER;
uint32_t CurrentLayer = LCD_BACKGROUND_LAYER;
//---------------------------------------------------


static sFONT *GFX_Currentfonts;

// GFX Private FunctionPrototypes
static void GFX_PolyLineRelativeClosed(pPoint Points, uint16_t PointCount, uint16_t Closed, uint16_t Color, uint8_t BuffNum, uint8_t SDRAM_Buff);
static void GFX_AF_GPIOConfig(void);

// Video Buffers as pointer array
uint16_t *VideoBuff = (uint16_t *)GFX_FRAME_BUFFER;
uint16_t *VideoBuffExt = (uint16_t *)GFX_EXT_FRAME_BUFFER;
// Sprites and Map Buffer as pointer array
uint16_t *SpriteMemory = (uint16_t *)GFX_SPRITE_START;
uint16_t *MapMemory = (uint16_t *)GFX_MAP_START;
sprinfo	 SpriteShow[NbSpriteShow];

// Game Map
uint16_t Map_Width  = 0;
uint16_t Map_Height = 0;

// DeInitializes the GFX.
void GFX_DeInit(void)
{
/*	
 +------------------------+-----------------------+----------------+
 +                       GFX pins assignment                       +
 +------------------------+-----------------------+----------------+
 |  GFX R2 <-> PC.12  |  GFX G2 <-> PA.06 |  GFX B2 <-> PD.06      |
 |  GFX R3 <-> PB.00  |  GFX G3 <-> PG.10 |  GFX B3 <-> PG.11      |
 |  GFX R4 <-> PA.11  |  GFX G4 <-> PB.10 |  GFX B4 <-> PG.12      |
 |  GFX R5 <-> PA.12  |  GFX G5 <-> PB.11 |  GFX B5 <-> PA.03      |
 |  GFX R6 <-> PB.01  |  GFX G6 <-> PC.07 |  GFX B6 <-> PB.08      |
 |  GFX R7 <-> PG.06  |  GFX G7 <-> PD.03 |  GFX B7 <-> PB.09      |
 -------------------------------------------------------------------
				 |  GFX HSYNC <-> PC.06  | GFX VSYNC <->  PA.04 |
				 ------------------------------------------------
*/

GPIO_InitTypeDef GFX_InitStructure;

  // GPIOA configuration //
  GPIO_PinAFConfig(GPIOA, GPIO_PinSource4, GPIO_AF_MCO);
  GPIO_PinAFConfig(GPIOA, GPIO_PinSource6, GPIO_AF_MCO);
  GPIO_PinAFConfig(GPIOA, GPIO_PinSource11, GPIO_AF_MCO);
  GPIO_PinAFConfig(GPIOA, GPIO_PinSource12, GPIO_AF_MCO);
  
  GFX_InitStructure.GPIO_Pin = GPIO_Pin_4 | GPIO_Pin_6 |
                                GPIO_Pin_11 | GPIO_Pin_12;
  GFX_InitStructure.GPIO_Mode = GPIO_Mode_AF;
  GFX_InitStructure.GPIO_Speed = GPIO_Speed_50MHz;
  GFX_InitStructure.GPIO_OType = GPIO_OType_PP;
  GFX_InitStructure.GPIO_PuPd  = GPIO_PuPd_NOPULL;
  GPIO_Init(GPIOA, &GFX_InitStructure);

  // GPIOB configuration //
  GPIO_PinAFConfig(GPIOB, GPIO_PinSource0, GPIO_AF_MCO);
  GPIO_PinAFConfig(GPIOB, GPIO_PinSource1, GPIO_AF_MCO);
  GPIO_PinAFConfig(GPIOB, GPIO_PinSource8, GPIO_AF_MCO);
  GPIO_PinAFConfig(GPIOB, GPIO_PinSource9, GPIO_AF_MCO);
  GPIO_PinAFConfig(GPIOB, GPIO_PinSource10, GPIO_AF_MCO);
  GPIO_PinAFConfig(GPIOB, GPIO_PinSource11, GPIO_AF_MCO);
  
  GFX_InitStructure.GPIO_Pin = GPIO_Pin_0  | GPIO_Pin_1  | GPIO_Pin_8    |
                                GPIO_Pin_9  |  GPIO_Pin_10 | GPIO_Pin_11;
  GFX_InitStructure.GPIO_Mode = GPIO_Mode_AF;
  GFX_InitStructure.GPIO_Speed = GPIO_Speed_50MHz;
  GFX_InitStructure.GPIO_OType = GPIO_OType_PP;
  GFX_InitStructure.GPIO_PuPd  = GPIO_PuPd_NOPULL;
  GPIO_Init(GPIOB, &GFX_InitStructure);
  
  // GPIOC configuration //
  GPIO_PinAFConfig(GPIOC, GPIO_PinSource6, GPIO_AF_MCO);
  GPIO_PinAFConfig(GPIOC, GPIO_PinSource7, GPIO_AF_MCO);
  GPIO_PinAFConfig(GPIOC, GPIO_PinSource12, GPIO_AF_MCO);
  
  GFX_InitStructure.GPIO_Pin = GPIO_Pin_6  | GPIO_Pin_7  | GPIO_Pin_12;
  GFX_InitStructure.GPIO_Mode = GPIO_Mode_AF;
  GFX_InitStructure.GPIO_Speed = GPIO_Speed_50MHz;
  GFX_InitStructure.GPIO_OType = GPIO_OType_PP;
  GFX_InitStructure.GPIO_PuPd  = GPIO_PuPd_NOPULL;
  GPIO_Init(GPIOC, &GFX_InitStructure);
  
  // GPIOD configuration //
  GPIO_PinAFConfig(GPIOD, GPIO_PinSource3, GPIO_AF_MCO);
  GPIO_PinAFConfig(GPIOD, GPIO_PinSource6, GPIO_AF_MCO);
  
  GFX_InitStructure.GPIO_Pin = GPIO_Pin_3  | GPIO_Pin_6; 
  GFX_InitStructure.GPIO_Mode = GPIO_Mode_AF;
  GFX_InitStructure.GPIO_Speed = GPIO_Speed_50MHz;
  GFX_InitStructure.GPIO_OType = GPIO_OType_PP;
  GFX_InitStructure.GPIO_PuPd  = GPIO_PuPd_NOPULL;
  GPIO_Init(GPIOD, &GFX_InitStructure);  

  // GPIOG configuration //
  GPIO_PinAFConfig(GPIOG, GPIO_PinSource6, GPIO_AF_MCO);
  GPIO_PinAFConfig(GPIOG, GPIO_PinSource10, GPIO_AF_MCO);
  GPIO_PinAFConfig(GPIOG, GPIO_PinSource11, GPIO_AF_MCO);
  GPIO_PinAFConfig(GPIOG, GPIO_PinSource12, GPIO_AF_MCO);
  
  GFX_InitStructure.GPIO_Pin = GPIO_Pin_6  | GPIO_Pin_10    |
                                GPIO_Pin_11 | GPIO_Pin_12;
  GFX_InitStructure.GPIO_Mode = GPIO_Mode_AF;
  GFX_InitStructure.GPIO_Speed = GPIO_Speed_50MHz;
  GFX_InitStructure.GPIO_OType = GPIO_OType_PP;
  GFX_InitStructure.GPIO_PuPd  = GPIO_PuPd_NOPULL;
  GPIO_Init(GPIOG, &GFX_InitStructure);
}

// Initializes the GFX.
void GFX_Init(void)
{ 
  LTDC_InitTypeDef       LTDC_InitStruct;
  
  // Enable the LTDC Clock //
  RCC_APB2PeriphClockCmd(RCC_APB2Periph_LTDC, ENABLE);
  
  // Enable the DMA2D Clock //
  RCC_AHB1PeriphClockCmd(RCC_AHB1Periph_DMA2D, ENABLE); 
  
  // Configure the LCD Control pins //
  GFX_AF_GPIOConfig();  
  
  // Configure the FMC Parallel interface : SDRAM is used as Frame Buffer for GFX //
  SDRAM_Init();
	delay_ms(100);
  
  // ***************************** LTDC Configuration ****************************//  
  // Polarity configuration //
	if(HSync_Level == 0)
	{
		// Initialize the horizontal synchronization polarity as active low //
		LTDC_InitStruct.LTDC_HSPolarity = LTDC_HSPolarity_AL;
	}
	else
	{
		// Initialize the horizontal synchronization polarity as active high //
		LTDC_InitStruct.LTDC_HSPolarity = LTDC_HSPolarity_AH;
	}
	if(VSync_Level == 0)
	{
		// Initialize the vertical synchronization polarity as active low //  
		LTDC_InitStruct.LTDC_VSPolarity = LTDC_VSPolarity_AL;     
	}
	else
	{
 		// Initialize the vertical synchronization polarity as active high //  
		LTDC_InitStruct.LTDC_VSPolarity = LTDC_VSPolarity_AH;     
	}
  // Initialize the data enable polarity as active low //
  LTDC_InitStruct.LTDC_DEPolarity = LTDC_DEPolarity_AL;     
  // Initialize the pixel clock polarity as input pixel clock // 
  LTDC_InitStruct.LTDC_PCPolarity = LTDC_PCPolarity_IPC;
  
  // Configure R,G,B component values for GFX background color //                   
  LTDC_InitStruct.LTDC_BackgroundRedValue = 0;            
  LTDC_InitStruct.LTDC_BackgroundGreenValue = 0;          
  LTDC_InitStruct.LTDC_BackgroundBlueValue = 0;  
  
  // Configure PLLSAI prescalers for GFX //
  // Enable Pixel Clock //
  // PLLSAI_VCO Input = HSE_VALUE/PLL_M = 1 Mhz //
  // PLLSAI_VCO Output = PLLSAI_VCO Input * PLLSAI_N = 192 Mhz //
  // PLLLCDCLK = PLLSAI_VCO Output/PLLSAI_R = 192/4 = 48 Mhz //
  // LTDC clock frequency = PLLLCDCLK / RCC_PLLSAIDivR = 48/8 = 6 Mhz //
  RCC_PLLSAIConfig(Freq_PLLSAI_VCO, Freq_PLLSAIDivR, Freq_PLLSAI_R);
  RCC_LTDCCLKDivConfig(RCC_PLLSAIDivR_Div8);
  
  // Enable PLLSAI Clock //
  RCC_PLLSAICmd(ENABLE);
  // Wait for PLLSAI activation //
  while(RCC_GetFlagStatus(RCC_FLAG_PLLSAIRDY) == RESET)
  {
  }
  
  // Timing configuration //  
  // Configure horizontal synchronization width //     
  LTDC_InitStruct.LTDC_HorizontalSync = GFX_H_Sync;
  // Configure vertical synchronization height //
  LTDC_InitStruct.LTDC_VerticalSync = GFX_V_Sync;
  // Configure accumulated horizontal back porch //
  LTDC_InitStruct.LTDC_AccumulatedHBP = GFX_H_Sync + GFX_H_Back_Porch + GFX_H_Right_Border;
  // Configure accumulated vertical back porch //
  LTDC_InitStruct.LTDC_AccumulatedVBP = GFX_V_Sync + GFX_V_Back_Porch + GFX_V_Bottom_Border;
  // Configure accumulated active width //  
  LTDC_InitStruct.LTDC_AccumulatedActiveW = GFX_H_Sync + GFX_H_Back_Porch + GFX_H_Right_Border + GFX_H_Video_Line;
  // Configure accumulated active height //
  LTDC_InitStruct.LTDC_AccumulatedActiveH = GFX_V_Sync + GFX_V_Back_Porch + GFX_V_Bottom_Border + GFX_V_Video_Height;
  // Configure total width //
  LTDC_InitStruct.LTDC_TotalWidth = GFX_Total_line_length;
  // Configure total height //
  LTDC_InitStruct.LTDC_TotalHeigh = GFX_Total_Height;
  
  LTDC_Init(&LTDC_InitStruct);
}  

// Initializes the LCD Layers.
void GFX_LayerInit(void)
{
  LTDC_Layer_InitTypeDef LTDC_Layer_InitStruct; 
  
  /* Windowing configuration */
  /* In this case all the active display area is used to display a picture then :
  Horizontal start = horizontal synchronization + Horizontal back porch = 30 
  Horizontal stop = Horizontal start + window width -1 = 30 + 240 -1
  Vertical start   = vertical synchronization + vertical back porch     = 4
  Vertical stop   = Vertical start + window height -1  = 4 + 320 -1      */      
  LTDC_Layer_InitStruct.LTDC_HorizontalStart = GFX_H_Sync + GFX_H_Back_Porch + GFX_H_Right_Border;
  LTDC_Layer_InitStruct.LTDC_HorizontalStop = (GFX_PIXEL_WIDTH + GFX_H_Sync + GFX_H_Back_Porch + GFX_H_Right_Border - 1); 
  LTDC_Layer_InitStruct.LTDC_VerticalStart = GFX_V_Sync + GFX_V_Back_Porch + GFX_V_Bottom_Border;
  LTDC_Layer_InitStruct.LTDC_VerticalStop = (GFX_PIXEL_HEIGHT + GFX_V_Sync + GFX_V_Back_Porch + GFX_V_Bottom_Border - 1);
  
  /* Pixel Format configuration*/
  LTDC_Layer_InitStruct.LTDC_PixelFormat = LTDC_Pixelformat_RGB565;
  /* Alpha constant (255 totally opaque) */
  LTDC_Layer_InitStruct.LTDC_ConstantAlpha = 255; 
  /* Default Color configuration (configure A,R,G,B component values) */          
  LTDC_Layer_InitStruct.LTDC_DefaultColorBlue = 0;        
  LTDC_Layer_InitStruct.LTDC_DefaultColorGreen = 0;       
  LTDC_Layer_InitStruct.LTDC_DefaultColorRed = 0;         
  LTDC_Layer_InitStruct.LTDC_DefaultColorAlpha = 0;
  /* Configure blending factors */       
  LTDC_Layer_InitStruct.LTDC_BlendingFactor_1 = LTDC_BlendingFactor1_CA;    
  LTDC_Layer_InitStruct.LTDC_BlendingFactor_2 = LTDC_BlendingFactor2_CA;
  
  /* the length of one line of pixels in bytes + 3 then :
  Line Lenth = Active high width x number of bytes per pixel + 3 
  Active high width         = GFX_PIXEL_WIDTH 
  number of bytes per pixel = 2    (pixel_format : RGB565) 
  */
  LTDC_Layer_InitStruct.LTDC_CFBLineLength = ((GFX_PIXEL_WIDTH * 2) + 3);
  /* the pitch is the increment from the start of one line of pixels to the 
  start of the next line in bytes, then :
  Pitch = Active high width x number of bytes per pixel */ 
  LTDC_Layer_InitStruct.LTDC_CFBPitch = (GFX_PIXEL_WIDTH * 2);
  
  /* Configure the number of lines */  
  LTDC_Layer_InitStruct.LTDC_CFBLineNumber = GFX_PIXEL_HEIGHT;
  
  /* Start Address configuration : the LCD Frame buffer is defined on SDRAM */    
  LTDC_Layer_InitStruct.LTDC_CFBStartAdress = GFX_FRAME_BUFFER;
  
  /* Initialize LTDC layer 1 */
  LTDC_LayerInit(LTDC_Layer1, &LTDC_Layer_InitStruct);
  
  /* Configure Layer2 */
  /* Start Address configuration : the LCD Frame buffer is defined on SDRAM w/ Offset */     
  LTDC_Layer_InitStruct.LTDC_CFBStartAdress = GFX_FRAME_BUFFER + BUFFER_OFFSET;
  
  /* Configure blending factors */       
  LTDC_Layer_InitStruct.LTDC_BlendingFactor_1 = LTDC_BlendingFactor1_PAxCA;    
  LTDC_Layer_InitStruct.LTDC_BlendingFactor_2 = LTDC_BlendingFactor2_PAxCA;
  
  /* Initialize LTDC layer 2 */
  LTDC_LayerInit(LTDC_Layer2, &LTDC_Layer_InitStruct);
  
  /* LTDC configuration reload */  
  LTDC_ReloadConfig(LTDC_IMReload);
  
  /* Enable foreground & background Layers */
  LTDC_LayerCmd(LTDC_Layer1, ENABLE); 
  LTDC_LayerCmd(LTDC_Layer2, ENABLE);
  
  /* LTDC configuration reload */  
  LTDC_ReloadConfig(LTDC_IMReload);
  
  /* dithering activation */
  LTDC_DitherCmd(ENABLE);
}

// Functions in use for compatibility --------------------------
/**
  * @brief  Sets the LCD Layer.
  * @param  Layerx: specifies the Layer foreground or background.
  * @retval None
  */
void LCD_SetLayer(__IO uint32_t Layerx)
{
  if (Layerx == LCD_BACKGROUND_LAYER)
  {
    CurrentFrameBuffer = LCD_FRAME_BUFFER; 
    CurrentLayer = LCD_BACKGROUND_LAYER;
  }
  else
  {
    CurrentFrameBuffer = LCD_FRAME_BUFFER + BUFFER_OFFSET;
    CurrentLayer = LCD_FOREGROUND_LAYER;
  }
}  

/**
  * @brief  Config and Sets the color Keying.
  * @param  RGBValue: Specifies the Color reference. 
  * @retval None
  */
void LCD_SetColorKeying(uint32_t RGBValue)
{  
  LTDC_ColorKeying_InitTypeDef   LTDC_colorkeying_InitStruct;
  
  /* configure the color Keying */
  LTDC_colorkeying_InitStruct.LTDC_ColorKeyBlue = 0x0000FF & RGBValue;
  LTDC_colorkeying_InitStruct.LTDC_ColorKeyGreen = (0x00FF00 & RGBValue) >> 8;
  LTDC_colorkeying_InitStruct.LTDC_ColorKeyRed = (0xFF0000 & RGBValue) >> 16;  

  if (CurrentLayer == LCD_BACKGROUND_LAYER)
  {   
    /* Enable the color Keying for Layer1 */
    LTDC_ColorKeyingConfig(LTDC_Layer1, &LTDC_colorkeying_InitStruct, ENABLE);
    LTDC_ReloadConfig(LTDC_IMReload);
  }
  else
  {
    /* Enable the color Keying for Layer2 */
    LTDC_ColorKeyingConfig(LTDC_Layer2, &LTDC_colorkeying_InitStruct, ENABLE);
    LTDC_ReloadConfig(LTDC_IMReload);
  }
}

/**
  * @brief  Clears the hole LCD.
  * @param  Color: the color of the background.
  * @retval None
  */
void LCD_Clear(uint16_t Color)
{
  uint32_t index = 0;
  
  /* erase memory */
  for (index = 0x00; index < BUFFER_OFFSET; index++)
  {
    *(__IO uint16_t*)(CurrentFrameBuffer + (2*index)) = Color;
  } 
}

/**
  * @brief  Sets a display window
  * @param  Xpos: specifies the X bottom left position from 0 to 240.
  * @param  Ypos: specifies the Y bottom left position from 0 to 320.
  * @param  Height: display window height, can be a value from 0 to 320.
  * @param  Width: display window width, can be a value from 0 to 240.
  * @retval None
  */
void LCD_DrawLine(uint16_t Xpos, uint16_t Ypos, uint16_t Length, uint8_t Direction)
{
  DMA2D_InitTypeDef      DMA2D_InitStruct;
  
  uint32_t  Xaddress = 0;
  uint16_t Red_Value = 0, Green_Value = 0, Blue_Value = 0;
  
  Xaddress = CurrentFrameBuffer + 2*(LCD_PIXEL_WIDTH*Ypos + Xpos);
 
  Red_Value = (0xF800 & CurrentTextColor) >> 11;
  Blue_Value = 0x001F & CurrentTextColor;
  Green_Value = (0x07E0 & CurrentTextColor) >> 5;

  /* Configure DMA2D */    
  DMA2D_DeInit();  
  DMA2D_InitStruct.DMA2D_Mode = DMA2D_R2M;       
  DMA2D_InitStruct.DMA2D_CMode = DMA2D_RGB565;      
  DMA2D_InitStruct.DMA2D_OutputGreen = Green_Value;      
  DMA2D_InitStruct.DMA2D_OutputBlue = Blue_Value;     
  DMA2D_InitStruct.DMA2D_OutputRed = Red_Value;                
  DMA2D_InitStruct.DMA2D_OutputAlpha = 0x0F;                  
  DMA2D_InitStruct.DMA2D_OutputMemoryAdd = Xaddress;                  
  
  if(Direction == LCD_DIR_HORIZONTAL)
  {                                                      
    DMA2D_InitStruct.DMA2D_OutputOffset = 0;                
    DMA2D_InitStruct.DMA2D_NumberOfLine = 1;            
    DMA2D_InitStruct.DMA2D_PixelPerLine = Length; 
  }
  else
  {                                                            
    DMA2D_InitStruct.DMA2D_OutputOffset = LCD_PIXEL_WIDTH - 1;                
    DMA2D_InitStruct.DMA2D_NumberOfLine = Length;            
    DMA2D_InitStruct.DMA2D_PixelPerLine = 1;  
  }
  
  DMA2D_Init(&DMA2D_InitStruct);  
  /* Start Transfer */ 
  DMA2D_StartTransfer();  
  /* Wait for CTC Flag activation */
  while(DMA2D_GetFlagStatus(DMA2D_FLAG_TC) == RESET)
  {
  }
  
}


// End Functions in use for compatibility ----------------------

//--------------------------------------------------------------
// Draw a line between 2 points
//--------------------------------------------------------------
void GFX_DrawUniLine(int16_t x1, int16_t y1, int16_t x2, int16_t y2 , uint16_t color , uint8_t BuffNum, uint8_t SDRAM_Buff )
{
  uint8_t yLonger = 0;
  int incrementVal, endVal;
  int shortLen = y2-y1;
  int longLen = x2-x1;
	int decInc;
	int j = 0, i = 0;

  if(ABS(shortLen) > ABS(longLen)) {
    int swap = shortLen;
    shortLen = longLen;
    longLen = swap;
    yLonger = 1;
  }

  endVal = longLen;

  if(longLen < 0) {
    incrementVal = -1;
    longLen = -longLen;
    endVal--;
  } else {
    incrementVal = 1;
    endVal++;
  }

  if(longLen == 0)
    decInc = 0;
  else
    decInc = (shortLen << 16) / longLen;

  if(yLonger) {
    for(i = 0;i != endVal;i += incrementVal) {
      GFX_SetPixel(x1 + (j >> 16),y1 + i,color , BuffNum, SDRAM_Buff);
      j += decInc;
    }
  } else {
    for(i = 0;i != endVal;i += incrementVal) {
      GFX_SetPixel(x1 + i,y1 + (j >> 16),color , BuffNum, SDRAM_Buff);
      j += decInc;
    }
  }
} 

//--------------------------------------------------------------
// Draw a circle.
// x, y - center of circle.
// r - radius.
// color - color of the circle.
//--------------------------------------------------------------
void GFX_DrawCircle(int16_t x, int16_t y, uint16_t radius, uint16_t color , uint8_t BuffNum, uint8_t SDRAM_Buff) 
{
   int16_t a, b, P;

   a = 0;
   b = radius;
   P = 1 - radius;

   do {
				if(((a+x) >= 0) && ((b+x) >= 0))
         GFX_SetPixel(a+x, b+y, color , BuffNum, SDRAM_Buff);
				if(((b+x) >= 0) && ((a+y) >= 0))
         GFX_SetPixel(b+x, a+y, color , BuffNum, SDRAM_Buff);
				if(((x-a) >= 0) && ((b+y) >= 0))
         GFX_SetPixel(x-a, b+y, color , BuffNum, SDRAM_Buff);
				if(((x-b) >= 0) && ((a+y) >= 0))
         GFX_SetPixel(x-b, a+y, color , BuffNum, SDRAM_Buff);
				if(((b+x) >= 0) && ((y-a) >= 0))
         GFX_SetPixel(b+x, y-a, color , BuffNum, SDRAM_Buff);
				if(((a+x) >= 0) && ((y-b) >= 0))
         GFX_SetPixel(a+x, y-b, color , BuffNum, SDRAM_Buff);
				if(((x-a) >= 0) && ((y-b) >= 0))
         GFX_SetPixel(x-a, y-b, color , BuffNum, SDRAM_Buff);
				if(((x-b) >= 0) && ((y-a) >= 0))
         GFX_SetPixel(x-b, y-a, color , BuffNum, SDRAM_Buff);

      if(P < 0)
         P+= 3 + 2*a++;
      else
         P+= 5 + 2*(a++ - b--);
    } while(a <= b);
}

//--------------------------------------------------------------
// Draw a Triangle.
// Triangle	- the triangle points.
// color		- color of the triangle.
//--------------------------------------------------------------
void GFX_Draw_Triangle(TriPoint Triangle, uint16_t color, uint8_t BuffNum, uint8_t SDRAM_Buff)
{
	GFX_DrawUniLine((int16_t)Triangle.ax , (int16_t)Triangle.ay , (int16_t)Triangle.bx , (int16_t)Triangle.by , color , BuffNum, SDRAM_Buff);
	GFX_DrawUniLine((int16_t)Triangle.bx , (int16_t)Triangle.by , (int16_t)Triangle.cx , (int16_t)Triangle.cy , color , BuffNum, SDRAM_Buff);
	GFX_DrawUniLine((int16_t)Triangle.cx , (int16_t)Triangle.cy , (int16_t)Triangle.ax , (int16_t)Triangle.ay , color , BuffNum, SDRAM_Buff);
}

//--------------------------------------------------------------
// Draw a Filled Triangle.
// Triangle	- the triangle points.
// incolor	- Fill color of the triangle.
// outcolor	- Border color of the triangle.
//--------------------------------------------------------------
void GFX_Draw_Full_Triangle(TriPoint Triangle, uint16_t incolor, uint16_t outcolor, uint8_t BuffNum, uint8_t SDRAM_Buff)
{
  float ma, mb, mc    ; //'gradient of the lines
  float start, finish ; //'draw a line from start to finish!
  float tempspace     ; //'temporary storage for swapping values...
  double x1,x2,x3      ;
  double y1,y2,y3      ;
  int16_t n           ;

  //' need to sort out ay, by and cy into order.. highest to lowest
  //'
  if(Triangle.ay < Triangle.by) 
    {
      //'swap x's
      tempspace = Triangle.ax;
      Triangle.ax = Triangle.bx;
      Triangle.bx = tempspace;

      //'swap y's
      tempspace = Triangle.ay;
      Triangle.ay = Triangle.by;
      Triangle.by = tempspace;
    }

  if(Triangle.ay < Triangle.cy)
		{
      //'swap x's
      tempspace = Triangle.ax;
      Triangle.ax = Triangle.cx;
      Triangle.cx = tempspace;

      //'swap y's
      tempspace = Triangle.ay;
      Triangle.ay = Triangle.cy;
      Triangle.cy = tempspace;
		}

  if(Triangle.by < Triangle.cy)
		{
      //'swap x's
      tempspace = Triangle.bx;
      Triangle.bx = Triangle.cx;
      Triangle.cx = tempspace;

      //'swap y's
      tempspace = Triangle.by;
      Triangle.by = Triangle.cy;
      Triangle.cy = tempspace;
		}

  //' Finally - copy the values in order...

  x1 = Triangle.ax; x2 = Triangle.bx; x3 = Triangle.cx;
  y1 = Triangle.ay; y2 = Triangle.by; y3 = Triangle.cy;

  //'bodge if y coordinates are the same
  if(y1 == y2)  y2 = y2 + 0.01;
  if(y2 == y3)  y3 = y3 + 0.01;
  if(y1 == y3)  y3 = y3 + 0.01;

  ma = (x1 - x2) / (y1 - y2);
  mb = (x3 - x2) / (y2 - y3);
  mc = (x3 - x1) / (y1 - y3);

  //'from y1 to y2
  for(n = 0;n >= (y2 - y1);n--)
    {
     start = n * mc;
     finish = n * ma;
     GFX_DrawUniLine((int16_t)(x1 - start), (int16_t)(n + y1), (int16_t)(x1 + finish), (int16_t)(n + y1), incolor , BuffNum, SDRAM_Buff);
    }


  //'and from y2 to y3

  for(n = 0;n >= (y3 - y2);n--) 
    {
     start = n * mc;
     finish = n * mb;
     GFX_DrawUniLine((int16_t)(x1 - start - ((y2 - y1) * mc)), (int16_t)(n + y2), (int16_t)(x2 - finish), (int16_t)(n + y2), incolor , BuffNum, SDRAM_Buff);
    }
	
	// draw the border color triangle
	GFX_Draw_Triangle(Triangle,outcolor,BuffNum, SDRAM_Buff);
}

//--------------------------------------------------------------
// Draw an image in format RGB565.
// x, y - position, where to start displaying.
// x_res, y_res - resolution in pixels.
// *ptr_image - pointer to image array.
//--------------------------------------------------------------
void VGA_Draw_Image(int16_t x, int16_t y, uint16_t x_res, uint16_t y_res,const uint8_t *ptr_image , uint8_t BuffNum, uint8_t SDRAM_Buff)
{
  int16_t cnt =0;
	int16_t h_line = 0;
	
  for(h_line = y; h_line < (y + y_res) ; h_line++)
		for(cnt = x;cnt < (x + x_res);cnt++)
		{
			GFX_SetPixel(cnt,h_line,*(ptr_image++) , BuffNum, SDRAM_Buff);
		}
}


// ----------------------------------------------------------------------------------------------------------------

// Configure the transparency.
// transparency: specifies the transparency, 
// This parameter must range from 0 to 255.
void GFX_SetTransparency(uint8_t transparency, uint8_t LayerNum)
{
  if (LayerNum == GFX_BACKGROUND_LAYER)
  {
    LTDC_LayerAlpha(LTDC_Layer1, transparency);
  }
  else
  {     
    LTDC_LayerAlpha(LTDC_Layer2, transparency);
  }
  LTDC_ReloadConfig(LTDC_IMReload);
}

// Gets the Text Font.
// retval the used font.
sFONT *GFX_GetFont(void)
{
  return GFX_Currentfonts;
}

// Sets the Text Font.
// fonts: specifies the font to be used.
void GFX_SetFont(sFONT *fonts)
{
  GFX_Currentfonts = fonts;
}

// Clears the whole Screen.
// Color: the color of the background.
// None
void GFX_Clear(uint16_t Color, uint8_t BuffNum, uint8_t SDRAM_Buff)
{
	// LTDC Method
	GFX_DrawFullRect(0,0,GFX_MAX_X + 1,GFX_MAX_Y + 1,Color,Color,BuffNum, SDRAM_Buff);
}

// Config and Sets the color Keying.
// RGBValue: Specifies the Color reference. 
// BuffNum : Video Buffer to use
void GFX_SetColorKeying(uint32_t RGBValue, uint8_t BuffNum, uint8_t SDRAM_Buff)
{  
  LTDC_ColorKeying_InitTypeDef   LTDC_colorkeying_InitStruct;
  
  /* configure the color Keying */
  LTDC_colorkeying_InitStruct.LTDC_ColorKeyBlue = 0x0000FF & RGBValue;
  LTDC_colorkeying_InitStruct.LTDC_ColorKeyGreen = (0x00FF00 & RGBValue) >> 8;
  LTDC_colorkeying_InitStruct.LTDC_ColorKeyRed = (0xFF0000 & RGBValue) >> 16;  

  if (BuffNum == GFX_BACKGROUND_LAYER)
  {   
    /* Enable the color Keying for Layer1 */
    LTDC_ColorKeyingConfig(LTDC_Layer1, &LTDC_colorkeying_InitStruct, ENABLE);
    LTDC_ReloadConfig(LTDC_IMReload);
  }
  else
  {
    /* Enable the color Keying for Layer2 */
    LTDC_ColorKeyingConfig(LTDC_Layer2, &LTDC_colorkeying_InitStruct, ENABLE);
    LTDC_ReloadConfig(LTDC_IMReload);
  }
}

// Disable the color Keying.
// BuffNum : Video Buffer to use
void GFX_ReSetColorKeying(uint8_t BuffNum, uint8_t SDRAM_Buff)
{
  LTDC_ColorKeying_InitTypeDef   LTDC_colorkeying_InitStruct;
  
  if (BuffNum == GFX_BACKGROUND_LAYER)
  {   
    /* Disable the color Keying for Layer1 */
    LTDC_ColorKeyingConfig(LTDC_Layer1, &LTDC_colorkeying_InitStruct, DISABLE);
    LTDC_ReloadConfig(LTDC_IMReload);
  }
  else
  {
    /* Disable the color Keying for Layer2 */
    LTDC_ColorKeyingConfig(LTDC_Layer2, &LTDC_colorkeying_InitStruct, DISABLE);
    LTDC_ReloadConfig(LTDC_IMReload);
  }
} 

//--------------------------------------------------------------
// Return Y for LineNumber
//--------------------------------------------------------------
uint16_t GFX_StringLine(uint16_t LineNumber)
{
	return (LineNumber * (GFX_Currentfonts->Height));
}

//--------------------------------------------------------------
// Return X for Column Number
//--------------------------------------------------------------
uint16_t GFX_StringColumn(uint16_t ColumnNumber)
{
	return (ColumnNumber * (GFX_Currentfonts->Width));
}

//--------------------------------------------------------------
// Draw a character.
// x, y - position
// color - character's color
// *c - pointer to character data
// Fcolor : character ForeColor
// Bcolor : character BackColor
// BuffNum : Video Buffer to use
//--------------------------------------------------------------
void GFX_DrawChar(int16_t x, int16_t y, const uint16_t *c,uint16_t Fcolor ,uint16_t Bcolor , uint8_t BuffNum, uint8_t SDRAM_Buff)
{
  uint32_t index = 0, i = 0;

  for(index = 0; index < GFX_Currentfonts->Height; index++)
  {
    for(i = 0; i < GFX_Currentfonts->Width; i++)
    {
      if( ((((c[index] & ((0x80 << ((GFX_Currentfonts->Width / 12 ) * 8 ) ) >> i)) == 0x00) && (GFX_Currentfonts->Width <= 12)) ||
          (((c[index] & (0x1 << i)) == 0x00)&&(GFX_Currentfonts->Width > 12 )))  == 0x00)
      {
        GFX_SetPixel( x+i,y+index, Fcolor , BuffNum, SDRAM_Buff);
			}
			else
      {
        GFX_SetPixel( x+i,y+index, Bcolor , BuffNum, SDRAM_Buff);
			}
    }
    //x++;
  }
}

//--------------------------------------------------------------
// Display a character.
// x, y - position
// c - character in ASCII
// Fcolor : character ForeColor
// Bcolor : character BackColor
// BuffNum : Video Buffer to use
//--------------------------------------------------------------
void GFX_DisplayChar(int16_t x, int16_t y, uint8_t c,uint16_t Fcolor ,uint16_t Bcolor, uint8_t BuffNum, uint8_t SDRAM_Buff)
{
  c -= 32;
  GFX_DrawChar(x, y, &GFX_Currentfonts->table[c * GFX_Currentfonts->Height], Fcolor, Bcolor , BuffNum, SDRAM_Buff);
}

//--------------------------------------------------------------
// Display a string.
// x - line. Next line have to respect the high of used font.
// y - vertical position
// *ptr - pointer to string.
// Fcolor : String ForeColor
// Bcolor : String BackColor
// BuffNum : Video Buffer to use
//--------------------------------------------------------------
void GFX_DisplayString(int16_t x, int16_t y, char *ptr,uint16_t Fcolor ,uint16_t Bcolor , uint8_t BuffNum, uint8_t SDRAM_Buff)
{
  uint16_t refcolumn = x;
  /* Send the string character by character on LCD */
  while (*ptr != 0)
  {
    /* Display one character on the Screen */
    GFX_DisplayChar(x,y, *ptr, Fcolor, Bcolor , BuffNum, SDRAM_Buff);
    /* Decrement the column position by 16 */
    refcolumn += GFX_Currentfonts->Width;
		x += GFX_Currentfonts->Width;
    /* Point on the next character */
    ptr++;
  }
}

// Displays a line.
// Xpos: specifies the X position, can be a value from 0 to 240.
// Ypos: specifies the Y position, can be a value from 0 to 320.
// Length: line length.
// Direction: line direction.
// This parameter can be one of the following values: GFX_DIR_HORIZONTAL or GFX_DIR_VERTICAL.
// retval None
void GFX_DrawLine(int16_t Xpos, int16_t Ypos, int16_t Length, uint8_t Direction, uint16_t Color, uint8_t BuffNum, uint8_t SDRAM_Buff)
{
#ifndef USE_320x240_60_DOUBLE
  DMA2D_InitTypeDef      DMA2D_InitStruct;
  uint32_t  Xaddress = 0;
  uint16_t Red_Value = 0, Green_Value = 0, Blue_Value = 0;
	ClipScreen LineClip;
#endif

#ifdef USE_320x240_60_DOUBLE
	if(Direction == GFX_DIR_HORIZONTAL)
	{
		GFX_DrawFullRect(Xpos,Ypos,Xpos + Length,Ypos + 1,Color,Color,BuffNum, SDRAM_Buff);
	}
	else
	{
		GFX_DrawFullRect(Xpos,Ypos,Xpos + 1,Ypos + Length,Color,Color,BuffNum, SDRAM_Buff);
	}
	return;
#endif

#ifndef USE_320x240_60_DOUBLE
	LineClip = GFX_Line_Clip(Xpos,Ypos,Length,Direction);

	// Test if we can draw the line
	if(Direction == GFX_DIR_HORIZONTAL)
	{
		if((LineClip.x1 == 0) && (LineClip.y1 == 0) && (LineClip.w == 0)){return;}
	}
	else
	{
		if((LineClip.x1 == 0) && (LineClip.y1 == 0) && (LineClip.h == 0)){return;}
	}
	if(SDRAM_Buff==2)
		Xaddress = (GFX_FRAME_BUFFER + (BuffNum * BUFFER_OFFSET)) + 2*(GFX_PIXEL_WIDTH*LineClip.y1 + LineClip.x1);
	else
		Xaddress = (GFX_EXT_FRAME_BUFFER + (BuffNum * BUFFER_OFFSET)) + 2*(GFX_PIXEL_WIDTH*LineClip.y1 + LineClip.x1);

 
  Red_Value = (0xF800 & Color) >> 11;
  Blue_Value = 0x001F & Color;
  Green_Value = (0x07E0 & Color) >> 5;

  /* Configure DMA2D */    
  DMA2D_DeInit();  
  DMA2D_InitStruct.DMA2D_Mode = DMA2D_R2M;       
  DMA2D_InitStruct.DMA2D_CMode = DMA2D_RGB565;      
  DMA2D_InitStruct.DMA2D_OutputGreen = Green_Value;      
  DMA2D_InitStruct.DMA2D_OutputBlue = Blue_Value;     
  DMA2D_InitStruct.DMA2D_OutputRed = Red_Value;                
  DMA2D_InitStruct.DMA2D_OutputAlpha = 0x0F;                  
  DMA2D_InitStruct.DMA2D_OutputMemoryAdd = Xaddress;                  
  
  if(Direction == GFX_DIR_HORIZONTAL)
  {                                                      
    DMA2D_InitStruct.DMA2D_OutputOffset = 0;                
    DMA2D_InitStruct.DMA2D_NumberOfLine = 1;            
    DMA2D_InitStruct.DMA2D_PixelPerLine = LineClip.w; 
  }
  else
  {                                                            
    DMA2D_InitStruct.DMA2D_OutputOffset = GFX_PIXEL_WIDTH - 1;                
    DMA2D_InitStruct.DMA2D_NumberOfLine = LineClip.h;            
    DMA2D_InitStruct.DMA2D_PixelPerLine = 1;  
  }
  
  DMA2D_Init(&DMA2D_InitStruct);  
  /* Start Transfer */ 
  DMA2D_StartTransfer();  
  /* Wait for CTC Flag activation */
  while(DMA2D_GetFlagStatus(DMA2D_FLAG_TC) == RESET)
  {
  }  
#endif
}

//  Displays a rectangle.
//  Xpos1,Ypos1: specifies the Upper Left Corner from the Rectangle.
//  Xpos2,Ypos2: specifies the Lower Right Corner from the Rectangle.
//  Color: specifies the Color to use.
//  BuffNum: specifies the Video Buffer to use.
//void GFX_DrawRect(int16_t Xpos1, int16_t Ypos1, int16_t Xpos2, int16_t Ypos2, uint16_t Color, uint8_t BuffNum)
void GFX_DrawRect(int16_t Xpos, int16_t Ypos, int16_t Width, int16_t Height, uint16_t Color, uint8_t BuffNum, uint8_t SDRAM_Buff)
{
  /* draw horizontal lines */
  GFX_DrawLine(Xpos, Ypos, Width, GFX_DIR_HORIZONTAL,Color,BuffNum, SDRAM_Buff);
  GFX_DrawLine(Xpos, (Ypos+ Height), Width, GFX_DIR_HORIZONTAL,Color,BuffNum, SDRAM_Buff);
  
  /* draw vertical lines */
  GFX_DrawLine(Xpos, Ypos, Height, GFX_DIR_VERTICAL,Color,BuffNum, SDRAM_Buff);
  GFX_DrawLine((Xpos + Width), Ypos, Height, GFX_DIR_VERTICAL,Color,BuffNum, SDRAM_Buff);
}

// Draw a full ellipse.
// Xpos: specifies the X position, can be a value from 0 to 240.
// Ypos: specifies the Y position, can be a value from 0 to 320.
// Radius: minor radius of ellipse.
// Radius2: major radius of ellipse.  
// FillColor  : specifies the Color to use for Fill the Ellipse.
// BorderColor: specifies the Color to use for draw the Border from the Ellipse.
// BuffNum: specifies the Video Buffer to use.
void GFX_DrawFullEllipse(int16_t Xpos, int16_t Ypos, uint16_t Radius, uint16_t Radius2, uint16_t FillColor, uint16_t BorderColor, uint8_t BuffNum, uint8_t SDRAM_Buff)
{
  int x = -Radius, y = 0, err = 2-2*Radius, e2;
  float K = 0, rad1 = 0, rad2 = 0;
  
  rad1 = Radius;
  rad2 = Radius2;
  
  if (Radius > Radius2)
  { 
    do 
    {
      K = (float)(rad1/rad2);
      GFX_DrawLine((Xpos+x), (Ypos-(uint16_t)(y/K)), (2*(uint16_t)(y/K) + 1), GFX_DIR_VERTICAL,FillColor,BuffNum, SDRAM_Buff);
      GFX_DrawLine((Xpos-x), (Ypos-(uint16_t)(y/K)), (2*(uint16_t)(y/K) + 1), GFX_DIR_VERTICAL,FillColor,BuffNum, SDRAM_Buff);
      
      e2 = err;
      if (e2 <= y) 
      {
        err += ++y*2+1;
        if (-x == y && e2 <= x) e2 = 0;
      }
      if (e2 > x) err += ++x*2+1;
      
    }
    while (x <= 0);
  }
  else
  {
    y = -Radius2; 
    x = 0;
    do 
    { 
      K = (float)(rad2/rad1);       
      GFX_DrawLine((Xpos-(uint16_t)(x/K)), (Ypos+y), (2*(uint16_t)(x/K) + 1), GFX_DIR_HORIZONTAL,FillColor,BuffNum, SDRAM_Buff);
      GFX_DrawLine((Xpos-(uint16_t)(x/K)), (Ypos-y), (2*(uint16_t)(x/K) + 1), GFX_DIR_HORIZONTAL,FillColor,BuffNum, SDRAM_Buff);
      
      e2 = err;
      if (e2 <= x) 
      {
        err += ++x*2+1;
        if (-y == x && e2 <= y) e2 = 0;
      }
      if (e2 > y) err += ++y*2+1;
    }
    while (y <= 0);
  }
	GFX_DrawEllipse(Xpos,Ypos,Radius,Radius2,BorderColor,BuffNum, SDRAM_Buff);
}

// Displays an Ellipse.
// Xpos: specifies the X position, can be a value from 0 to 240.
// Ypos: specifies the Y position, can be a value from 0 to 320.
// Radius: specifies Radius.
// Radius2: specifies Radius2.
// Color: specifies the Color to use for draw the Border from the Ellipse.
// BuffNum: specifies the Video Buffer to use.
void GFX_DrawEllipse(int16_t Xpos, int16_t Ypos, uint16_t Radius, uint16_t Radius2, uint16_t Color, uint8_t BuffNum, uint8_t SDRAM_Buff)
{
  int x = -Radius, y = 0, err = 2-2*Radius, e2;
  float K = 0, rad1 = 0, rad2 = 0;
   
  rad1 = Radius;
  rad2 = Radius2;
  
  if (Radius > Radius2)
  { 
    do {
      K = (float)(rad1/rad2);
			GFX_SetPixel(Xpos-x,Ypos+(uint16_t)(y/K),Color,BuffNum, SDRAM_Buff);
			GFX_SetPixel(Xpos+x,Ypos+(uint16_t)(y/K),Color,BuffNum, SDRAM_Buff);
			GFX_SetPixel(Xpos+x,Ypos-(uint16_t)(y/K),Color,BuffNum, SDRAM_Buff);
			GFX_SetPixel(Xpos-x,Ypos-(uint16_t)(y/K),Color,BuffNum, SDRAM_Buff);
            
      e2 = err;
      if (e2 <= y) {
        err += ++y*2+1;
        if (-x == y && e2 <= x) e2 = 0;
      }
      if (e2 > x) err += ++x*2+1;
    }
    while (x <= 0);
  }
  else
  {
    y = -Radius2; 
    x = 0;
    do { 
      K = (float)(rad2/rad1);
			GFX_SetPixel(Xpos-(uint16_t)(x/K),Ypos+y,Color,BuffNum, SDRAM_Buff);
			GFX_SetPixel(Xpos+(uint16_t)(x/K),Ypos+y,Color,BuffNum, SDRAM_Buff);
			GFX_SetPixel(Xpos+(uint16_t)(x/K),Ypos-y,Color,BuffNum, SDRAM_Buff);
			GFX_SetPixel(Xpos-(uint16_t)(x/K),Ypos-y,Color,BuffNum, SDRAM_Buff);
      
      e2 = err;
      if (e2 <= x) {
        err += ++x*2+1;
        if (-y == x && e2 <= y) e2 = 0;
      }
      if (e2 > y) err += ++y*2+1;     
    }
    while (y <= 0);
  }
}

//  Displays a filled Rectangle.
//  Xpos1,Ypos1: specifies the Upper Left Corner from the Rectangle.
//  Xpos2,Ypos2: specifies the Lower Right Corner from the Rectangle.
//  FillColor  : specifies the Color to use for Fill the Rectangle.
//  BorderColor: specifies the Color to use for draw the Border from the Rectangle.
//  BuffNum: specifies the Video Buffer to use.
void GFX_DrawFullRect(int16_t Xpos1, int16_t Ypos1, int16_t Xpos2, int16_t Ypos2, uint16_t FillColor, uint16_t BorderColor, uint8_t BuffNum, uint8_t SDRAM_Buff)
{
  DMA2D_InitTypeDef      DMA2D_InitStruct;
  
  uint32_t  Xaddress = 0; 
  uint16_t Red_Value = 0, Green_Value = 0, Blue_Value = 0;
#ifndef USE_320x240_60_DOUBLE
	uint16_t dl , dh;
#endif
	ClipScreen RectClip;
	
	// Test Screen Clipping
#ifdef USE_320x240_60_DOUBLE
	RectClip = GFX_Rect_Clip(Xpos1 * 2,Ypos1 * 2,Xpos2 * 2,Ypos2 * 2);
#else
	RectClip = GFX_Rect_Clip(Xpos1,Ypos1,Xpos2,Ypos2);
#endif
	if((RectClip.w == 0) || (RectClip.h == 0)) return;
		
  Red_Value = (0xF800 & FillColor) >> 11;
  Blue_Value = 0x001F & FillColor;
  Green_Value = (0x07E0 & FillColor) >> 5;
  
	if(SDRAM_Buff==2)
		Xaddress = (GFX_FRAME_BUFFER + (BuffNum * BUFFER_OFFSET)) + 2*(GFX_PIXEL_WIDTH * RectClip.y1 + RectClip.x1);
	else
		Xaddress = (GFX_EXT_FRAME_BUFFER + (BuffNum * BUFFER_OFFSET)) + 2*(GFX_PIXEL_WIDTH * RectClip.y1 + RectClip.x1);
		
  
  /* configure DMA2D */
  DMA2D_DeInit();
  DMA2D_InitStruct.DMA2D_Mode = DMA2D_R2M;       
  DMA2D_InitStruct.DMA2D_CMode = DMA2D_RGB565;      
  DMA2D_InitStruct.DMA2D_OutputGreen = Green_Value;      
  DMA2D_InitStruct.DMA2D_OutputBlue = Blue_Value;     
  DMA2D_InitStruct.DMA2D_OutputRed = Red_Value;                
  DMA2D_InitStruct.DMA2D_OutputAlpha = 0x0F;                  
  DMA2D_InitStruct.DMA2D_OutputMemoryAdd = Xaddress;                
  DMA2D_InitStruct.DMA2D_OutputOffset = (GFX_PIXEL_WIDTH - RectClip.w);                
  DMA2D_InitStruct.DMA2D_NumberOfLine = RectClip.h;            
  DMA2D_InitStruct.DMA2D_PixelPerLine = RectClip.w;
  DMA2D_Init(&DMA2D_InitStruct); 
  
  /* Start Transfer */ 
  DMA2D_StartTransfer();
  
  /* Wait for CTC Flag activation */
  while(DMA2D_GetFlagStatus(DMA2D_FLAG_TC) == RESET)
  {
  } 
	
#ifdef USE_320x240_60_DOUBLE
	GFX_DrawRectBorder(RectClip.x1,RectClip.y1,RectClip.x2,RectClip.y1+1,BorderColor,BuffNum, SDRAM_Buff);
	GFX_DrawRectBorder(RectClip.x1,RectClip.y2-1,RectClip.x2,RectClip.y2,BorderColor,BuffNum, SDRAM_Buff);
	GFX_DrawRectBorder(RectClip.x1,RectClip.y1,RectClip.x1+1,RectClip.y2,BorderColor,BuffNum, SDRAM_Buff);
	GFX_DrawRectBorder(RectClip.x2-1,RectClip.y1,RectClip.x2,RectClip.y2,BorderColor,BuffNum, SDRAM_Buff);
#else
	dl = RectClip.x2-RectClip.x1;
	dh = RectClip.y2-RectClip.y1;
	GFX_DrawRect(RectClip.x1,RectClip.y1,dl,dh,BorderColor,BuffNum, SDRAM_Buff);
#endif
}

//  Displays Rectangle border in of USE_320x240_60_DOUBLE mode.
//  Xpos1,Ypos1: specifies the Upper Left Corner from the Rectangle.
//  Xpos2,Ypos2: specifies the Lower Right Corner from the Rectangle.
//  BorderColor: specifies the Color to use for draw the Border from the Rectangle.
//  BuffNum: specifies the Video Buffer to use.
void GFX_DrawRectBorder(int16_t Xpos1, int16_t Ypos1, int16_t Xpos2, int16_t Ypos2, uint16_t BorderColor, uint8_t BuffNum, uint8_t SDRAM_Buff)
{
  DMA2D_InitTypeDef      DMA2D_InitStruct;
  
  uint32_t  Xaddress = 0; 
  uint16_t Red_Value = 0, Green_Value = 0, Blue_Value = 0;
	ClipScreen BorderClip;
	
	BorderClip = GFX_Rect_Clip(Xpos1,Ypos1,Xpos2,Ypos2);
	if((BorderClip.w == 0) || (BorderClip.h == 0)) return;
		
  Red_Value = (0xF800 & BorderColor) >> 11;
  Blue_Value = 0x001F & BorderColor;
  Green_Value = (0x07E0 & BorderColor) >> 5;
  
	if(SDRAM_Buff==2)
		Xaddress = (GFX_FRAME_BUFFER + (BuffNum * BUFFER_OFFSET)) + 2*(GFX_PIXEL_WIDTH * BorderClip.y1 + BorderClip.x1);
  else
		Xaddress = (GFX_EXT_FRAME_BUFFER + (BuffNum * BUFFER_OFFSET)) + 2*(GFX_PIXEL_WIDTH * BorderClip.y1 + BorderClip.x1);

  /* configure DMA2D */
  DMA2D_DeInit();
  DMA2D_InitStruct.DMA2D_Mode = DMA2D_R2M;       
  DMA2D_InitStruct.DMA2D_CMode = DMA2D_RGB565;      
  DMA2D_InitStruct.DMA2D_OutputGreen = Green_Value;      
  DMA2D_InitStruct.DMA2D_OutputBlue = Blue_Value;     
  DMA2D_InitStruct.DMA2D_OutputRed = Red_Value;                
  DMA2D_InitStruct.DMA2D_OutputAlpha = 0x0F;                  
  DMA2D_InitStruct.DMA2D_OutputMemoryAdd = Xaddress;                
  DMA2D_InitStruct.DMA2D_OutputOffset = (GFX_PIXEL_WIDTH - BorderClip.w);                
  DMA2D_InitStruct.DMA2D_NumberOfLine = BorderClip.h;            
  DMA2D_InitStruct.DMA2D_PixelPerLine = BorderClip.w;
  DMA2D_Init(&DMA2D_InitStruct); 
  
  /* Start Transfer */ 
  DMA2D_StartTransfer();
  
  /* Wait for CTC Flag activation */
  while(DMA2D_GetFlagStatus(DMA2D_FLAG_TC) == RESET)
  {
  } 
}


// Displays a full circle.
// Xpos       : specifies the X position, can be a value from 0 to 240.
// Ypos       : specifies the Y position, can be a value from 0 to 320.
// Radius     : specifies the Circle Radius
// FillColor  : specifies the Circle Fill Color
// BorderColor: specifies the Circle Border Color
// BuffNum    : specifies the Video Buffer number
void GFX_DrawFullCircle(int16_t Xpos, int16_t Ypos, int16_t Radius, uint16_t FillColor, uint16_t BorderColor, uint8_t BuffNum, uint8_t SDRAM_Buff)
{
  int32_t  D;    /* Decision Variable */ 
  uint32_t  CurX;/* Current X Value */
  uint32_t  CurY;/* Current Y Value */ 
  
  D = 3 - (Radius << 1);
  
  CurX = 0;
  CurY = Radius;
  
  while (CurX <= CurY)
  {
    if(CurY > 0) 
    {
      GFX_DrawLine(Xpos - CurX, Ypos - CurY, 2*CurY, GFX_DIR_VERTICAL,FillColor,BuffNum, SDRAM_Buff);
      GFX_DrawLine(Xpos + CurX, Ypos - CurY, 2*CurY, GFX_DIR_VERTICAL,FillColor,BuffNum, SDRAM_Buff);
    }
    
    if(CurX > 0) 
    {
      GFX_DrawLine(Xpos - CurY, Ypos - CurX, 2*CurX, GFX_DIR_VERTICAL,FillColor,BuffNum, SDRAM_Buff);
      GFX_DrawLine(Xpos + CurY, Ypos - CurX, 2*CurX, GFX_DIR_VERTICAL,FillColor,BuffNum, SDRAM_Buff);
    }
    if (D < 0)
    { 
      D += (CurX << 2) + 6;
    }
    else
    {
      D += ((CurX - CurY) << 2) + 10;
      CurY--;
    }
    CurX++;
  }
  
  GFX_DrawCircle(Xpos, Ypos, Radius,BorderColor,BuffNum, SDRAM_Buff);  
}

/**
  * @brief  Displays an triangle.
  * @param  Points: pointer to the points array.
  * @retval None
  */
void GFX_Triangle(pPoint Points, uint16_t PointCount, uint16_t Color, uint8_t BuffNum, uint8_t SDRAM_Buff)
{
  int16_t X = 0, Y = 0;
  pPoint First = Points;

  if(PointCount != 3)
  {
    return;
  }

  while(--PointCount)
  {
    X = Points->X;
    Y = Points->Y;
    Points++;
    GFX_DrawUniLine(X, Y, Points->X, Points->Y,Color,BuffNum, SDRAM_Buff);
  }
  GFX_DrawUniLine(First->X, First->Y, Points->X, Points->Y,Color,BuffNum, SDRAM_Buff);
}

/**
  * @brief  Fill an triangle (between 3 points).
  * @param  x1..3: x position of triangle point 1..3.
  * @param  y1..3: y position of triangle point 1..3.
  * @retval None
  */
void GFX_FillTriangle(int16_t x1, int16_t x2, int16_t x3, int16_t y1, int16_t y2, int16_t y3, uint16_t FillColor, uint16_t BorderColor, uint8_t BuffNum, uint8_t SDRAM_Buff)
{ 
  
  int16_t deltax = 0, deltay = 0, x = 0, y = 0, xinc1 = 0, xinc2 = 0, 
  yinc1 = 0, yinc2 = 0, den = 0, num = 0, numadd = 0, numpixels = 0, 
  curpixel = 0;
  
  deltax = ABS(x2 - x1);        /* The difference between the x's */
  deltay = ABS(y2 - y1);        /* The difference between the y's */
  x = x1;                       /* Start x off at the first pixel */
  y = y1;                       /* Start y off at the first pixel */
  
  if (x2 >= x1)                 /* The x-values are increasing */
  {
    xinc1 = 1;
    xinc2 = 1;
  }
  else                          /* The x-values are decreasing */
  {
    xinc1 = -1;
    xinc2 = -1;
  }
  
  if (y2 >= y1)                 /* The y-values are increasing */
  {
    yinc1 = 1;
    yinc2 = 1;
  }
  else                          /* The y-values are decreasing */
  {
    yinc1 = -1;
    yinc2 = -1;
  }
  
  if (deltax >= deltay)         /* There is at least one x-value for every y-value */
  {
    xinc1 = 0;                  /* Don't change the x when numerator >= denominator */
    yinc2 = 0;                  /* Don't change the y for every iteration */
    den = deltax;
    num = deltax / 2;
    numadd = deltay;
    numpixels = deltax;         /* There are more x-values than y-values */
  }
  else                          /* There is at least one y-value for every x-value */
  {
    xinc2 = 0;                  /* Don't change the x for every iteration */
    yinc1 = 0;                  /* Don't change the y when numerator >= denominator */
    den = deltay;
    num = deltay / 2;
    numadd = deltax;
    numpixels = deltay;         /* There are more y-values than x-values */
  }
  
  for (curpixel = 0; curpixel <= numpixels; curpixel++)
  {
    GFX_DrawUniLine(x, y, x3, y3,FillColor,BuffNum, SDRAM_Buff);
    
    num += numadd;              /* Increase the numerator by the top of the fraction */
    if (num >= den)             /* Check if numerator >= denominator */
    {
      num -= den;               /* Calculate the new numerator value */
      x += xinc1;               /* Change the x as appropriate */
      y += yinc1;               /* Change the y as appropriate */
    }
    x += xinc2;                 /* Change the x as appropriate */
    y += yinc2;                 /* Change the y as appropriate */
  }  
  
  
}
/**
  * @brief  Displays an poly-line (between many points).
  * @param  Points: pointer to the points array.
  * @param  PointCount: Number of points.
  * @retval None
  */
void GFX_PolyLine(pPoint Points, uint16_t PointCount, uint16_t Color, uint8_t BuffNum, uint8_t SDRAM_Buff)
{
  int16_t X = 0, Y = 0;

  if(PointCount < 2)
  {
    return;
  }

  while(--PointCount)
  {
    X = Points->X;
    Y = Points->Y;
    Points++;
    GFX_DrawUniLine(X, Y, Points->X, Points->Y,Color,BuffNum, SDRAM_Buff);
  }
}

/**
  * @brief  Displays an relative poly-line (between many points).
  * @param  Points: pointer to the points array.
  * @param  PointCount: Number of points.
  * @param  Closed: specifies if the draw is closed or not.
  *           1: closed, 0 : not closed.
  * @retval None
  */
static void GFX_PolyLineRelativeClosed(pPoint Points, uint16_t PointCount, uint16_t Closed, uint16_t Color, uint8_t BuffNum, uint8_t SDRAM_Buff)
{
  int16_t X = 0, Y = 0;
  pPoint First = Points;

  if(PointCount < 2)
  {
    return;
  }  
  X = Points->X;
  Y = Points->Y;
  while(--PointCount)
  {
    Points++;
    GFX_DrawUniLine(X, Y, X + Points->X, Y + Points->Y,Color,BuffNum, SDRAM_Buff);
    X = X + Points->X;
    Y = Y + Points->Y;
  }
  if(Closed)
  {
    GFX_DrawUniLine(First->X, First->Y, X, Y,Color,BuffNum, SDRAM_Buff);
  }  
}

/**
  * @brief  Displays a closed poly-line (between many points).
  * @param  Points: pointer to the points array.
  * @param  PointCount: Number of points.
  * @retval None
  */
void GFX_ClosedPolyLine(pPoint Points, uint16_t PointCount, uint16_t Color, uint8_t BuffNum, uint8_t SDRAM_Buff)
{
  GFX_PolyLine(Points, PointCount,Color,BuffNum, SDRAM_Buff);
  GFX_DrawUniLine(Points->X, Points->Y, (Points+PointCount-1)->X, (Points+PointCount-1)->Y,Color,BuffNum, SDRAM_Buff);
}

/**
  * @brief  Displays a relative poly-line (between many points).
  * @param  Points: pointer to the points array.
  * @param  PointCount: Number of points.
  * @retval None
  */
void GFX_PolyLineRelative(pPoint Points, uint16_t PointCount, uint16_t Color, uint8_t BuffNum, uint8_t SDRAM_Buff)
{
  GFX_PolyLineRelativeClosed(Points, PointCount, 0,Color,BuffNum, SDRAM_Buff);
}

/**
  * @brief  Displays a closed relative poly-line (between many points).
  * @param  Points: pointer to the points array.
  * @param  PointCount: Number of points.
  * @retval None
  */
void GFX_ClosedPolyLineRelative(pPoint Points, uint16_t PointCount, uint16_t Color, uint8_t BuffNum, uint8_t SDRAM_Buff)
{
  GFX_PolyLineRelativeClosed(Points, PointCount, 1,Color,BuffNum, SDRAM_Buff);
}

/**
  * @brief  Displays a  full poly-line (between many points).
  * @param  Points: pointer to the points array.
  * @param  PointCount: Number of points.
  * @retval None
  */
void GFX_FillPolyLine(pPoint Points, uint16_t PointCount, uint16_t FillColor, uint16_t BorderColor, uint8_t BuffNum, uint8_t SDRAM_Buff)
{
 
  int16_t X = 0, Y = 0, X2 = 0, Y2 = 0, X_center = 0, Y_center = 0, X_first = 0, Y_first = 0, pixelX = 0, pixelY = 0, counter = 0;
  uint16_t  IMAGE_LEFT = 0, IMAGE_RIGHT = 0, IMAGE_TOP = 0, IMAGE_BOTTOM = 0;  

  IMAGE_LEFT = IMAGE_RIGHT = Points->X;
  IMAGE_TOP= IMAGE_BOTTOM = Points->Y;

  for(counter = 1; counter < PointCount; counter++)
  {
    pixelX = POLY_X(counter);
    if(pixelX < IMAGE_LEFT)
    {
      IMAGE_LEFT = pixelX;
    }
    if(pixelX > IMAGE_RIGHT)
    {
      IMAGE_RIGHT = pixelX;
    }
    
    pixelY = POLY_Y(counter);
    if(pixelY < IMAGE_TOP)
    { 
      IMAGE_TOP = pixelY;
    }
    if(pixelY > IMAGE_BOTTOM)
    {
      IMAGE_BOTTOM = pixelY;
    }
  }  
  
  if(PointCount < 2)
  {
    return;
  }
  
  X_center = (IMAGE_LEFT + IMAGE_RIGHT)/2;
  Y_center = (IMAGE_BOTTOM + IMAGE_TOP)/2;
 
  X_first = Points->X;
  Y_first = Points->Y;
  
  while(--PointCount)
  {
    X = Points->X;
    Y = Points->Y;
    Points++;
    X2 = Points->X;
    Y2 = Points->Y;    
  
    GFX_FillTriangle(X, X2, X_center, Y, Y2, Y_center,FillColor,BorderColor,BuffNum, SDRAM_Buff);
    GFX_FillTriangle(X, X_center, X2, Y, Y_center, Y2,FillColor,BorderColor,BuffNum, SDRAM_Buff);
    GFX_FillTriangle(X_center, X2, X, Y_center, Y2, Y,FillColor,BorderColor,BuffNum, SDRAM_Buff);   
  }
  
  GFX_FillTriangle(X_first, X2, X_center, Y_first, Y2, Y_center,FillColor,BorderColor,BuffNum, SDRAM_Buff);
  GFX_FillTriangle(X_first, X_center, X2, Y_first, Y_center, Y2,FillColor,BorderColor,BuffNum, SDRAM_Buff);
  GFX_FillTriangle(X_center, X2, X_first, Y_center, Y2, Y_first,FillColor,BorderColor,BuffNum, SDRAM_Buff); 
}

/**
  * @brief GPIO config for LTDC.
  * @retval
  *  None
  */
static void GFX_AF_GPIOConfig(void)
{
  GPIO_InitTypeDef GFX_InitStruct;
  
  /* Enable GPIOI, GPIOJ, GPIOG, GPIOF, GPIOH AHB Clocks */
  RCC_AHB1PeriphClockCmd(RCC_AHB1Periph_GPIOA | RCC_AHB1Periph_GPIOB | \
                         RCC_AHB1Periph_GPIOC | RCC_AHB1Periph_GPIOD | \
                         RCC_AHB1Periph_GPIOG, ENABLE);

/* GPIOs Configuration */
/*
 +------------------------+-----------------------+----------------------------+
 +                       GFX pins assignment                                   +
 +------------------------+-----------------------+----------------------------+
 |  GFX_TFT R2 <-> PC.12  |  GFX_TFT G2 <-> PA.06 |  GFX_TFT B2 <-> PD.06      |
 |  GFX_TFT R3 <-> PB.00  |  GFX_TFT G3 <-> PG.10 |  GFX_TFT B3 <-> PG.11      |
 |  GFX_TFT R4 <-> PA.11  |  GFX_TFT G4 <-> PB.10 |  GFX_TFT B4 <-> PG.12      |
 |  GFX_TFT R5 <-> PA.12  |  GFX_TFT G5 <-> PB.11 |  GFX_TFT B5 <-> PA.03      |
 |  GFX_TFT R6 <-> PB.01  |  GFX_TFT G6 <-> PC.07 |  GFX_TFT B6 <-> PB.08      |
 |  GFX_TFT R7 <-> PG.06  |  GFX_TFT G7 <-> PD.03 |  GFX_TFT B7 <-> PB.09      |
 -------------------------------------------------------------------------------
          |  GFX_TFT HSYNC <-> PC.06  | LCDTFT VSYNC <->  PA.04 |
           -----------------------------------------------------

*/

 /* GPIOA configuration */
  GPIO_PinAFConfig(GPIOA, GPIO_PinSource3, GPIO_AF_LTDC);
  GPIO_PinAFConfig(GPIOA, GPIO_PinSource4, GPIO_AF_LTDC);
  GPIO_PinAFConfig(GPIOA, GPIO_PinSource6, GPIO_AF_LTDC);
  GPIO_PinAFConfig(GPIOA, GPIO_PinSource11, GPIO_AF_LTDC);
  GPIO_PinAFConfig(GPIOA, GPIO_PinSource12, GPIO_AF_LTDC);

  GFX_InitStruct.GPIO_Pin = GPIO_Pin_3 | GPIO_Pin_4 | GPIO_Pin_6 | \
                             GPIO_Pin_11 | GPIO_Pin_12;
                             
  GFX_InitStruct.GPIO_Speed = GPIO_Speed_50MHz;
  GFX_InitStruct.GPIO_Mode = GPIO_Mode_AF;
  GFX_InitStruct.GPIO_OType = GPIO_OType_PP;
  GFX_InitStruct.GPIO_PuPd = GPIO_PuPd_NOPULL;
  GPIO_Init(GPIOA, &GFX_InitStruct);
  
 /* GPIOB configuration */  
  GPIO_PinAFConfig(GPIOB, GPIO_PinSource0, 0x09);
  GPIO_PinAFConfig(GPIOB, GPIO_PinSource1, 0x09);
  GPIO_PinAFConfig(GPIOB, GPIO_PinSource8, GPIO_AF_LTDC);
  GPIO_PinAFConfig(GPIOB, GPIO_PinSource9, GPIO_AF_LTDC);
  GPIO_PinAFConfig(GPIOB, GPIO_PinSource10, GPIO_AF_LTDC);
  GPIO_PinAFConfig(GPIOB, GPIO_PinSource11, GPIO_AF_LTDC);

  GFX_InitStruct.GPIO_Pin = GPIO_Pin_0 | GPIO_Pin_1 | GPIO_Pin_8 | \
                             GPIO_Pin_9 | GPIO_Pin_10 | GPIO_Pin_11;
  
  GPIO_Init(GPIOB, &GFX_InitStruct);

 /* GPIOC configuration */
  GPIO_PinAFConfig(GPIOC, GPIO_PinSource6, GPIO_AF_LTDC);
  GPIO_PinAFConfig(GPIOC, GPIO_PinSource7, GPIO_AF_LTDC);
  GPIO_PinAFConfig(GPIOC, GPIO_PinSource10, GPIO_AF_LTDC);
  
  GFX_InitStruct.GPIO_Pin = GPIO_Pin_6 | GPIO_Pin_7 | GPIO_Pin_10;
                             
  GPIO_Init(GPIOC, &GFX_InitStruct);

 /* GPIOD configuration */
  GPIO_PinAFConfig(GPIOD, GPIO_PinSource3, GPIO_AF_LTDC);
  GPIO_PinAFConfig(GPIOD, GPIO_PinSource6, GPIO_AF_LTDC);
  
  GFX_InitStruct.GPIO_Pin = GPIO_Pin_3 | GPIO_Pin_6;
                             
  GPIO_Init(GPIOD, &GFX_InitStruct);
  
 /* GPIOG configuration */  
  GPIO_PinAFConfig(GPIOG, GPIO_PinSource6, GPIO_AF_LTDC);
  GPIO_PinAFConfig(GPIOG, GPIO_PinSource10, 0x09);
  GPIO_PinAFConfig(GPIOG, GPIO_PinSource11, GPIO_AF_LTDC);
  GPIO_PinAFConfig(GPIOG, GPIO_PinSource12, 0x09);

  GFX_InitStruct.GPIO_Pin = GPIO_Pin_6 | GPIO_Pin_10 | \
                             GPIO_Pin_11 | GPIO_Pin_12;
  
  GPIO_Init(GPIOG, &GFX_InitStruct);
 
}

// Display a Pixel.
// @param  Xpos: pixel x.
// @param  Ypos: pixel y.  
// @param  Color: Pixel Color  
// @retval None
void GFX_SetPixel(int16_t Xpos, int16_t Ypos, uint16_t Color, uint8_t BuffNum, uint8_t SDRAM_Buff)
{
#ifndef USE_320x240_60_DOUBLE
	uint32_t PixelAdress = (uint32_t)(((GFX_PIXEL_WIDTH * Ypos) + Xpos) + (PIXEL_COUNT * BuffNum));
#else
	// We have to set 4 Pixels in Double mode
	int16_t x1 = Xpos * 2;
	int16_t x2 = Xpos * 2 + 1;
	int16_t y1 = Ypos * 2;
	int16_t y2 = Ypos * 2 + 1;
	uint32_t pix1addr = 0;
	uint32_t pix2addr = 0;
	uint32_t pix3addr = 0;
	uint32_t pix4addr = 0;
#endif
  
  if(Xpos < 0 || Xpos > GFX_MAX_X || Ypos < 0 || Ypos > GFX_MAX_Y){return;}

#ifdef USE_320x240_60_DOUBLE
	// Compute the adress from the 4 Pixels to set in double mode
	pix1addr = (uint32_t)(((GFX_PIXEL_WIDTH * y1) + x1) + (PIXEL_COUNT * BuffNum));
	pix2addr = (uint32_t)(((GFX_PIXEL_WIDTH * y2) + x1) + (PIXEL_COUNT * BuffNum));
	pix3addr = (uint32_t)(((GFX_PIXEL_WIDTH * y1) + x2) + (PIXEL_COUNT * BuffNum));
	pix4addr = (uint32_t)(((GFX_PIXEL_WIDTH * y2) + x2) + (PIXEL_COUNT * BuffNum));
#endif	

#ifdef USE_320x240_60_DOUBLE
	if(SDRAM_Buff == 2)
	{
		VideoBuff[pix1addr] = Color;
		VideoBuff[pix2addr] = Color;
		VideoBuff[pix3addr] = Color;
		VideoBuff[pix4addr] = Color;
	}
	else
	{
		VideoBuffExt[pix1addr] = Color;
		VideoBuffExt[pix2addr] = Color;
		VideoBuffExt[pix3addr] = Color;
		VideoBuffExt[pix4addr] = Color;
	}		
#else
	if(SDRAM_Buff == 2)
		VideoBuff[PixelAdress] = Color;			// st429 disco default  SDRAM
	else
		VideoBuffExt[PixelAdress] = Color;	// st429 disco external SDRAM
#endif
}

// Display a Pixel.
// Xpos				: Coord X.
// Ypos				: Coord Y.  
// BuffNum		: Layer number  
// SDRAM_Buff	: SDRAM Bank
// Returned		: Color value
uint16_t GFX_GetPixel(int16_t Xpos, int16_t Ypos, uint8_t BuffNum, uint8_t SDRAM_Buff)
{
#ifndef USE_320x240_60_DOUBLE
	uint32_t PixelAdress = (uint32_t)(((GFX_PIXEL_WIDTH * Ypos) + Xpos) + (PIXEL_COUNT * BuffNum));
#else
	// We have to set 4 Pixels in Double mode
	int16_t x1 = Xpos * 2;
	int16_t y1 = Ypos * 2;
	uint32_t pix1addr = 0;
#endif
  
  if(Xpos < 0 || Xpos > GFX_MAX_X || Ypos < 0 || Ypos > GFX_MAX_Y){return 0;}

#ifdef USE_320x240_60_DOUBLE
	// Compute the adress from the 4 Pixels to set in double mode
	pix1addr = (uint32_t)(((GFX_PIXEL_WIDTH * y1) + x1) + (PIXEL_COUNT * BuffNum));
#endif	

#ifdef USE_320x240_60_DOUBLE
	if(SDRAM_Buff == 2)
	{
		return VideoBuff[pix1addr];					// st429 disco default  SDRAM
	}
	else
	{
		return VideoBuffExt[pix1addr];			// st429 disco external SDRAM
	}		
#else
	if(SDRAM_Buff == 2)
		return VideoBuff[PixelAdress];			// st429 disco default  SDRAM
	else
		return VideoBuffExt[PixelAdress];		// st429 disco external SDRAM
#endif
}

// Copy one video buffer to another one.
// Buff_From	: From where we get the data.
// Buff_To		: Place to put the data.  
// retval  None
void GFX_Copy_Buffer_to_Buffer(uint8_t Buff_From, uint8_t Buff_To, uint8_t SDRAM_Buff_From, uint8_t SDRAM_Buff_To)
{
  uint32_t BuffAdressFrom = PIXEL_COUNT * Buff_From;
  uint32_t BuffAdressTo = PIXEL_COUNT * Buff_To;
  uint32_t index;
	
//	if((Buff_From == Buff_To) || (Buff_From > 2) || (Buff_To > 2)){return;}
	if(Buff_From == Buff_To){return;}
	if((SDRAM_Buff_From==2) && (SDRAM_Buff_To==2))
		for (index = 0 ; index < PIXEL_COUNT;index++)
		{
			VideoBuff[BuffAdressTo + index] = VideoBuff[BuffAdressFrom + index];
		}
	if((SDRAM_Buff_From==2) && (SDRAM_Buff_To==1))
		for (index = 0 ; index < PIXEL_COUNT;index++)
		{
			VideoBuffExt[BuffAdressTo + index] = VideoBuff[BuffAdressFrom + index];
		}
	if((SDRAM_Buff_From==1) && (SDRAM_Buff_To==2))
		for (index = 0 ; index < PIXEL_COUNT;index++)
		{
			VideoBuff[BuffAdressTo + index] = VideoBuffExt[BuffAdressFrom + index];
		}
	if((SDRAM_Buff_From==1) && (SDRAM_Buff_To==1))
		for (index = 0 ; index < PIXEL_COUNT;index++)
		{
			VideoBuffExt[BuffAdressTo + index] = VideoBuffExt[BuffAdressFrom + index];
		}
}

ClipScreen GFX_Rect_Clip(int16_t Xpos1, int16_t Ypos1, int16_t Xpos2, int16_t Ypos2)
{
	// Clip a rectangle into the screen
	int16_t x1 = Xpos1;
	int16_t y1 = Ypos1;
	int16_t x2 = Xpos2;
	int16_t y2 = Ypos2;
	uint16_t tmp = 0;
	
	ClipScreen CRes;
	if(x1 < 0) x1 = 0;
	if(x1 > (GFX_PIXEL_WIDTH - 1)) x1 = (GFX_PIXEL_WIDTH - 1);
	if(x2 < 0) x2 = 0;
	if(x2 > (GFX_PIXEL_WIDTH - 1)) x2 = (GFX_PIXEL_WIDTH - 1);
	if(y1 < 0) y1 = 0;
	if(y1 > (GFX_PIXEL_HEIGHT - 1)) y1 = (GFX_PIXEL_HEIGHT - 1);
	if(y2 < 0) y2 = 0;
	if(y2 > (GFX_PIXEL_HEIGHT - 1)) y2 = (GFX_PIXEL_HEIGHT - 1);
	// x1 must be < than x2 and y1 must be < than y2
	if(x1 > x2){tmp = x1;x1 = x2;x2 = tmp;}
	if(y1 > y2){tmp = y1;y1 = y2;y2 = tmp;}
	CRes.x1 = x1 ; CRes.y1 = y1 ;CRes.x2 = x2 ; CRes.y2 = y2 ;
	// get the width and height
	CRes.w = x2-x1+1 ;CRes.h = y2-y1+1 ;
	return CRes;
}

ClipMapScreen GFX_Map_Clip(int16_t StartX, int16_t StartY, int16_t Xpos1, int16_t Ypos1, int16_t Xpos2, int16_t Ypos2)
{
	// Clip a rectangle into the screen
	int16_t x1 = Xpos1;
	int16_t y1 = Ypos1;
	int16_t x2 = Xpos2;
	int16_t y2 = Ypos2;
	int16_t lx =0;
	int16_t ly =0;
	uint16_t tmp = 0;
	
	ClipMapScreen CRes;
	
	// x1 must be < than x2 and y1 must be < than y2
	if(x1 > x2){tmp = x1;x1 = x2;x2 = tmp;}
	if(y1 > y2){tmp = y1;y1 = y2;y2 = tmp;}
	
	// Check if we are on screen
	if(x2<0)
	{
		CRes.Dest_w = 0 ;CRes.Dest_h = 0;
		return CRes;
	}
	if(y2<0)
	{
		CRes.Dest_w = 0 ;CRes.Dest_h = 0;
		return CRes;
	}
	if(x1>(GFX_PIXEL_WIDTH - 1))
	{
		CRes.Dest_w = 0 ;CRes.Dest_h = 0;
		return CRes;
	}
	if(y1>(GFX_PIXEL_HEIGHT - 1))
	{
		CRes.Dest_w = 0 ;CRes.Dest_h = 0;
		return CRes;
	}
	// Here we are sure that some pixels are on screen
	
	// Width and Height
	if(x1<0)
		lx = x2 + abs(x1);
	else
		lx = x2 - x1;

	if(y1<0)
		ly = y2 + abs(y1);
	else
		ly = y2 - y1;
	
	CRes.Src_x1 = StartX;
	CRes.Src_y1 = StartY;
	
	if(x1 < 0)
	{
		CRes.Src_x1 = StartX + abs(x1);
		lx -= abs(x1);
		x1 = 0;
	}
	if(y1 < 0)
	{
		CRes.Src_y1 = StartY + abs(y1);
		ly -= abs(y1);
		y1 = 0;
	}
	if(x2 > (GFX_PIXEL_WIDTH - 1))
	{
		lx -= (x2 - (GFX_PIXEL_WIDTH - 1));
		x2 = (GFX_PIXEL_WIDTH - 1);
	}
	if(y2 > (GFX_PIXEL_HEIGHT - 1))
	{
		ly -= (y2 - (GFX_PIXEL_HEIGHT - 1));
		y2 = (GFX_PIXEL_HEIGHT - 1);
	}
	CRes.Dest_x1 = x1 ; CRes.Dest_y1 = y1 ;CRes.Dest_x2 = x2 ; CRes.Dest_y2 = y2 ;
	// get the width and height
	CRes.Dest_w = lx ;CRes.Dest_h = ly ;
	return CRes;
}

ClipScreen GFX_Line_Clip(int16_t Xpos, int16_t Ypos, int16_t Length, uint8_t Direction)
{
	// Clip Horizontal or Vartical Line into screen
	int16_t x1 = Xpos;
	int16_t y1 = Ypos;
	int16_t x2 = 0;
	int16_t y2 = 0;
	uint16_t tmp = 0;
	ClipScreen CRes;
	
	if (Direction == GFX_DIR_HORIZONTAL)
	{
		if((y1 < 0) || (y1 > (GFX_PIXEL_HEIGHT - 1)))
		{
			// This will not allow the line drawing
			CRes.x1 = 0;CRes.y1 = 0;CRes.w = 0;CRes.h = 0;
			return CRes;
		}
		x2 = x1 + Length;
		if(x1 < 0) x1 = 0;
		if(x1 > (GFX_PIXEL_WIDTH - 1)) x1 = (GFX_PIXEL_WIDTH - 1);
		if(x2 < 0) x2 = 0;
		if(x2 > (GFX_PIXEL_WIDTH - 1)) x2 = (GFX_PIXEL_WIDTH - 1);
		// x1 must be < than x2
		if(x1 > x2){tmp = x1;x1 = x2;x2 = tmp;}
		CRes.x1 = x1;CRes.y1 = y1;CRes.w = x2-x1+1;CRes.h = 0;
		return CRes;
	}
	else
	{
		if((x1 < 0) || (x1 > (GFX_PIXEL_WIDTH - 1)))
		{
			// This will not allow the line drawing
			CRes.x1 = 0;CRes.y1 = 0;CRes.w = 0;CRes.h = 0;
			return CRes;
		}
		y2 = y1 + Length;
		if(y1 < 0) y1 = 0;
		if(y1 > (GFX_PIXEL_HEIGHT - 1)) y1 = (GFX_PIXEL_HEIGHT - 1);
		if(y2 < 0) y2 = 0;
		if(y2 > (GFX_PIXEL_HEIGHT - 1)) y2 = (GFX_PIXEL_HEIGHT - 1);
		// y1 must be < than y2
		if(y1 > y2){tmp = y1;y1 = y2;y2 = tmp;}
		CRes.x1 = x1;CRes.y1 = y1;CRes.h = y2-y1+1;CRes.w = 0;
		return CRes;
	}
	
}

// Copy from one to other Video buffer using the DMA2D
// StartX			: Upper left source Corner.
// StartY			:   
// Width			: Width in Pixels.
// Height			: Height in Pixels.  
// DestX			: Upper left Destination Corner.
// DestY			:   
// Buff_From	: From where we get the data.
// Buff_To		: Place to put the data.  
void GFX_D2D_Copy(int16_t StartX, int16_t StartY,uint16_t Width,uint16_t Height,int16_t DestX, int16_t DestY,uint8_t Buff_From, uint8_t Buff_To, uint8_t SDRAM_Buff_From, uint8_t SDRAM_Buff_To)
{
  DMA2D_InitTypeDef      DMA2D_InitStruct;
  DMA2D_FG_InitTypeDef   DMA2D_FG_InitStruct;

  uint32_t  dest_address = 0;
  uint32_t  source_address = 0;

  // check for Screen Limit
  if(Width==0) return;
  if(Height==0) return;
  if(((StartX + Width) <= 0) || ((StartY + Height) <= 0)) return;
  if(((StartX + Width) > GFX_MAX_X) || ((StartY + Height) > GFX_MAX_Y)) return;
  if(((DestX + Width) <= 0) || ((DestY + Height) <= 0)) return;
  if(((DestX + Width) > GFX_MAX_X) || ((DestY + Height) > GFX_MAX_Y)) return;

  // Compute the Source and Destination Adress
	if(SDRAM_Buff_From==2)
		source_address 	= GFX_FRAME_BUFFER + (BUFFER_OFFSET * Buff_From) + 2*(GFX_PIXEL_WIDTH * StartY + StartX);
	else
		source_address 	= GFX_EXT_FRAME_BUFFER + (BUFFER_OFFSET * Buff_From) + 2*(GFX_PIXEL_WIDTH * StartY + StartX);
		
	if(SDRAM_Buff_To==2)
		dest_address 		= GFX_FRAME_BUFFER + (BUFFER_OFFSET * Buff_To)   + 2*(GFX_PIXEL_WIDTH * DestY + DestX);
	else
		dest_address 		= GFX_EXT_FRAME_BUFFER + (BUFFER_OFFSET * Buff_To)   + 2*(GFX_PIXEL_WIDTH * DestY + DestX);

  DMA2D_DeInit();
  DMA2D_InitStruct.DMA2D_Mode = DMA2D_M2M;
  DMA2D_InitStruct.DMA2D_CMode = DMA2D_RGB565;
  DMA2D_InitStruct.DMA2D_OutputMemoryAdd = dest_address;
  DMA2D_InitStruct.DMA2D_OutputGreen = 0;
  DMA2D_InitStruct.DMA2D_OutputBlue = 0;
  DMA2D_InitStruct.DMA2D_OutputRed = 0;
  DMA2D_InitStruct.DMA2D_OutputAlpha = 0;
  DMA2D_InitStruct.DMA2D_OutputOffset = GFX_PIXEL_WIDTH-Width;
  DMA2D_InitStruct.DMA2D_NumberOfLine = Height;
  DMA2D_InitStruct.DMA2D_PixelPerLine = Width;
  DMA2D_Init(&DMA2D_InitStruct);

  DMA2D_FG_StructInit(&DMA2D_FG_InitStruct);
  DMA2D_FG_InitStruct.DMA2D_FGMA = source_address;
  DMA2D_FG_InitStruct.DMA2D_FGCM = CM_RGB565;
  DMA2D_FG_InitStruct.DMA2D_FGPFC_ALPHA_MODE = NO_MODIF_ALPHA_VALUE;
  DMA2D_FG_InitStruct.DMA2D_FGPFC_ALPHA_VALUE = 0;
  DMA2D_FG_InitStruct.DMA2D_FGO=GFX_PIXEL_WIDTH-Width;
  DMA2D_FGConfig(&DMA2D_FG_InitStruct);

  DMA2D_StartTransfer();

  while(DMA2D_GetFlagStatus(DMA2D_FLAG_TC) == RESET);
}

// Open a Sprite file and record all data to SDRAM Buffer
// Datafile				: File handler
// name						: name of the file with the full path , ex: "0:/file.ext"
// SpriteStart		: First Sprite number (so you can load more than one sprite file)
// Returned Values: 0 = no error
//									1 = Card not present
//									2 = Card not mounted
//									3 = File open error
//									4 = Fist and Last Sprite Read error
//									5 = Sprite Wide Read error
//									6 = Sprite Height Read error
//									7 = Pixels color Read error
uint8_t GFX_Load_Sprite(FIL DataFile, const char* name,uint16_t SpriteStart)
{
	// Open and read a file with Sprites
	// 
  // Sprite file format:
  // H,L                      : Hi and Low byte from First Sprite number
  // H,L                      : Hi and Low byte from Last Sprite number
	//
  // H,L                      : First Sprite Wide
  // H,L                      : First Sprite Height
  // H,L,H,L,H ...       			: Hi and Low byte from each pixel color
	//
  // H,L                      : 2nd Sprite Wide
  // H,L                      : 2nd Sprite Height
  // H,L,H,L,H ...       			: Hi and Low byte from each pixel color
	// ....
	// Hi and Low have to be inverted (LSB first)
  uint32_t  spr_address = 0;	// Sprites address in memory
	uint16_t	spr_w = 0;				// Width and Height from sprites
	uint16_t	spr_h = 0;
	uint32_t Counter = 0;
	uint16_t ActSprite = 0;
	uint16_t FirstSprite = 0;
	uint16_t LastSprite = 0;
	uint8_t  px,py;
	uint32_t PixAddr = 0;
	unsigned char ReadChar[1] = {0};
	unsigned char WordToRead[4] = {0};
	unsigned char PixelRead[2] = {0};
	uint8_t Close_Act_File = 0;
	
  // Check if SDCard is present
  if(Fatfs_CheckMedia(MMC_0)==FATFS_OK) 
	{		
		// Mount SD Card
    if(Fatfs_Mount(MMC_0)==FATFS_OK) 
		{
      // File to read
      if(Fatfs_OpenFile(&DataFile, name, F_RD)==FATFS_OK) 
			{
				// Get the First and Last Sprite number
				for(Counter = 0 ; Counter < 4 ; Counter++)
				{
					if((Fatfs_ReadUChar(&DataFile,ReadChar))==FATFS_OK)
					{
						WordToRead[Counter] = ReadChar[0];
					}
					else
					{ // First / Last Sprite Read error
						return 4;
					}
				}
				// ************* Get the number of the First sprite (16 bits) *************
				FirstSprite = (WordToRead[1] << 8) | WordToRead[0];
				// ************* Get the number of the Last sprite (16 bits) **************
				LastSprite = (WordToRead[3] << 8) | WordToRead[2];
				
				//Here we go for decode all Sprites data
				for(ActSprite = FirstSprite;ActSprite <= LastSprite;ActSprite++)
				{
					// Compute the Start Address where we Transfer the Sprite informations
					spr_address = ActSprite * (GFX_SPRITE_LENGTH / 2) ;
					
					// ************* Get the Sprite Wide ************* 
					for(Counter = 0 ; Counter < 2 ; Counter++)
					{
						if((Fatfs_ReadUChar(&DataFile,ReadChar))==FATFS_OK)
						{
							PixelRead[Counter] = ReadChar[0];
						}
						else
						{ // Sprite Wide Read error
							return 5;
						}
					}
					spr_w = (uint16_t)((PixelRead[1] << 8) | PixelRead[0]);
					SpriteMemory[spr_address] = spr_w;
					spr_address++;
					// ************* Get the Sprite Height ************* 
					for(Counter = 0 ; Counter < 2 ; Counter++)
					{
						if((Fatfs_ReadUChar(&DataFile,ReadChar))==FATFS_OK)
						{
							PixelRead[Counter] = ReadChar[0];
						}
						else
						{ // Sprite Height Read error
							return 6;
						}
					}
					spr_h = (uint16_t)((PixelRead[1] << 8) | PixelRead[0]);
					SpriteMemory[spr_address] = spr_h;
					spr_address++;
					//Get all the Sprite pixels
					for(py = 0;py < spr_h;py++)
						for(px = 0;px < spr_w;px++)
						{
							PixAddr = spr_address + ((py * spr_w) + px);
							// get the 2 bytes from the pixel
							for(Counter = 0 ; Counter < 2 ; Counter++)
							{
								if((Fatfs_ReadUChar(&DataFile,ReadChar))==FATFS_OK)
								{
									PixelRead[Counter] = ReadChar[0];
								}
								else
								{ // Sprite pixels Read error
									return 7;
								}
							}
							// put the pixel color value to SDRAM buffer
							SpriteMemory[PixAddr] = (uint16_t)((PixelRead[1] << 8) | PixelRead[0]);
						}
				}
			}
			else
			{ // Sprite File open error
				return 3;
			}
 			// Close the Sprite file
			Close_Act_File = Fatfs_CloseFile(&DataFile);
			if(Close_Act_File != 0)
			{
				GFX_DisplayString(GFX_StringColumn(0),GFX_StringLine(10),"  I can't close the Sprite file !!!  ",
				GFX_COLOR_RED,GFX_COLOR_BLACK,GFX_FOREGROUND_LAYER, 2);
			}
     // Unmount SD Card
   	  Fatfs_UnMount(MMC_0);
		}
		else
		{ // Card not mounted
			return 2;
		}
	}
	else
	{ // Card not present
		return 1;
	}
	return 0;
}

// Open a Map file and record all data to SDRAM Buffer
// Datafile				: File handler
// name						: name of the file with the full path , ex: "0:/file.ext"
// MapNum					: Map Number
// Returned Values: 0 = no error
//									1 = Card not present
//									2 = Card not mounted
//									8 = File open error
//									9 = Width and Height Read error
//									10= Pixels Read error
//									11= Map size is too big for the allocated Buffer
uint8_t GFX_Load_Map(FIL DataFile, const char* name,uint8_t MapNum)
{
  // Map file format:
  // H,L                      : Hi and Low byte from the Map Width
  // H,L                      : Hi and Low byte from the Map Height
  // H,L,H,L,H ...       			: Hi and Low byte from each pixel color
	// Hi and Low have to be inverted (LSB first)
  uint32_t  map_address = 0;	// Map address in memory
	uint32_t Counter = 0;
//	uint16_t px,py;
	unsigned char ReadChar[1] = {0};
	unsigned char WordToRead[4] = {0};
//	unsigned char PixelRead[2] = {0};
	uint8_t Close_Act_File = 0;
	FATFS_t File_Status = FATFS_OK;
	uint32_t Save_Counter = 0;
	uint32_t cnt = 0;
	unsigned char Char_Buff[512];
	
	Map_Width  = 0;
	Map_Height = 0;
	
  // Check if SDCard is present
  if(Fatfs_CheckMedia(MMC_0)==FATFS_OK) 
	{		
		// Mount SD Card
    if(Fatfs_Mount(MMC_0)==FATFS_OK) 
		{
      // File to read
      if(Fatfs_OpenFile(&DataFile, name, F_RD)==FATFS_OK) 
			{
				// We read the Width and Height from the map
				for(Counter = 0 ; Counter < 4 ; Counter++)
				{
					if((Fatfs_ReadUChar(&DataFile,ReadChar))==FATFS_OK)
					{
						WordToRead[Counter] = ReadChar[0];
					}
					else
					{ // Map Width / Height Read error
						return 9;
					}
				}
				// ************* Map Width  *************
				Map_Width = ((WordToRead[1] << 8) | WordToRead[0]) + 1;
				// ************* Map Height *************
				Map_Height = ((WordToRead[3] << 8) | WordToRead[2]) + 1;
				
				// Test if Map is not too big for the memory buffer
				if((Map_Width * Map_Height) > (GFX_MAP_LENGTH / 2))
					return 11;
					
				// record this values to Map Buffer in SDRAM
				map_address = 0;
				MapMemory[map_address] = (uint16_t)(Map_Width);
				map_address++;
				MapMemory[map_address] = (uint16_t)(Map_Height);
				map_address++;
				
				// Reading Map pixels with 512 bytes block
				// And convert each pixels in 16bits
				while(File_Status == FATFS_OK)
				{
					File_Status = Fatfs_ReadBlock(&DataFile , Char_Buff , 512 , &cnt);
					if(File_Status == FATFS_OK)
					{
						for (Save_Counter = 0;Save_Counter < 512; Save_Counter+=2)
						{
							MapMemory[map_address] = (uint16_t)(Char_Buff[Save_Counter + 1] << 8 | Char_Buff[Save_Counter]);
							map_address++;
						}
					}
					else if (File_Status == FATFS_EOF)
					{
						for (Save_Counter = 0;Save_Counter < cnt; Save_Counter+=2)
						{
							MapMemory[map_address] = (uint16_t)(Char_Buff[Save_Counter + 1] << 8 | Char_Buff[Save_Counter]);
							map_address++;
						}
					}
				}
			}
			else
			{ // Map File open error
				return 8;
			}
			// Close the Map file
			Close_Act_File = Fatfs_CloseFile(&DataFile);
			if(Close_Act_File != 0)
			{
				GFX_DisplayString(GFX_StringColumn(0),GFX_StringLine(10),"  I can't close the Map file !!!  ",
				GFX_COLOR_RED,GFX_COLOR_BLACK,GFX_FOREGROUND_LAYER, 2);
			}
      // Unmount SD Card
   	  Fatfs_UnMount(MMC_0);
		}
		else
		{ // Card not mounted
			return 2;
		}
	}
	else
	{ // Card not present
		return 1;
	}
	return 0;
}


// Copy the Game Map to one of the video buffer
// This function dont use DMA2D
// StartX			: Upper left Corner from where we start to copy the map.
// StartY			:   
// Width			: Width in Pixels.
// Height			: Height in Pixels.  
// DestX			: Upper left Destination Corner.
// DestY			:   
// showtrans	: 0 = Show the transparent color , other = Don't show the transparent color
// transcol		: Number of the transparent color
// SDRAM_Buff	: Internal or External SDRAM Buffer. 
// GFX_Layer	: Layer number where we copy the Map data

void GFX_Show_Map(int16_t StartX, int16_t StartY,uint16_t Width,uint16_t Height,int16_t DestX, int16_t DestY, uint8_t showtrans , uint16_t transcol , uint8_t SDRAM_Buff, uint8_t GFX_Layer)
{
	int16_t	  x,y;
	int16_t	  dX,dY;
	uint32_t  PixAddr = 0;
	
	ClipMapScreen MapClip;
	
	MapClip = GFX_Map_Clip(StartX, StartY, DestX , DestY , DestX + Width , DestY + Height);
	if((MapClip.Dest_w == 0) || (MapClip.Dest_h == 0)) return;
	
  
	// Copy Map pixels to Video Buffer
	dX = MapClip.Dest_x1 ;
	dY = MapClip.Dest_y1 ;
	for (y = MapClip.Src_y1;y < (MapClip.Src_y1 + MapClip.Dest_h);y++)
	{
		for(x = MapClip.Src_x1;x < (MapClip.Src_x1 + MapClip.Dest_w);x++)
		{
			PixAddr = Map_Width * y + x;
			if(showtrans == 0)
			{ // We show the transparent color
				GFX_SetPixel(dX,dY,MapMemory[PixAddr], GFX_Layer , SDRAM_Buff);
			}
			else
			{ // Here we hide the transparent color
				if(MapMemory[PixAddr] != transcol)
				{
					GFX_SetPixel(dX,dY,MapMemory[PixAddr], GFX_Layer , SDRAM_Buff);
				}
			}
			dX++;
		}
		dX = MapClip.Dest_x1;
		dY++;
	}
}

// Copy the Game Map to one of the video buffer
// This function use DMA2D
// StartX			: Upper left Corner from where we start to copy the map.
// StartY			:   
// Width			: Width in Pixels.
// Height			: Height in Pixels.  
// DestX			: Upper left Destination Corner.
// DestY			:   
// SDRAM_Buff	: Internal or External SDRAM Buffer. 
// GFX_Layer	: Layer number where we copy the Map data
void GFX_Show_Map_DMA2D(int16_t StartX, int16_t StartY,uint16_t Width,uint16_t Height,int16_t DestX, int16_t DestY, uint8_t SDRAM_Buff, uint8_t GFX_Layer)
{
  DMA2D_InitTypeDef      DMA2D_InitStruct;
  DMA2D_FG_InitTypeDef   DMA2D_FG_InitStruct;

  uint32_t  dest_address = 0;
  uint32_t  source_address = 0;
	
	ClipMapScreen MapClip;
	
	MapClip = GFX_Map_Clip(StartX, StartY, DestX , DestY , DestX + Width , DestY + Height);
	if((MapClip.Dest_w == 0) || (MapClip.Dest_h == 0)) return;
  
  // Compute the Source and Destination Adress
	source_address 	= MAP_PIX_START + (2 * (Map_Width * MapClip.Src_y1 + MapClip.Src_x1));
		
	if(SDRAM_Buff==2)
		dest_address 		= GFX_FRAME_BUFFER + (BUFFER_OFFSET * GFX_Layer) + (2 * (GFX_PIXEL_WIDTH * MapClip.Dest_y1 + MapClip.Dest_x1));
	else
		dest_address 		= GFX_EXT_FRAME_BUFFER + (BUFFER_OFFSET * GFX_Layer) + (2 * (GFX_PIXEL_WIDTH * MapClip.Dest_y1 + MapClip.Dest_x1));

  DMA2D_DeInit();
  DMA2D_InitStruct.DMA2D_Mode = DMA2D_M2M;
  DMA2D_InitStruct.DMA2D_CMode = DMA2D_RGB565;
  DMA2D_InitStruct.DMA2D_OutputMemoryAdd = dest_address;
  DMA2D_InitStruct.DMA2D_OutputGreen = 0;
  DMA2D_InitStruct.DMA2D_OutputBlue = 0;
  DMA2D_InitStruct.DMA2D_OutputRed = 0;
  DMA2D_InitStruct.DMA2D_OutputAlpha = 0;
  DMA2D_InitStruct.DMA2D_OutputOffset = GFX_PIXEL_WIDTH - MapClip.Dest_w;
  DMA2D_InitStruct.DMA2D_NumberOfLine = MapClip.Dest_h;
  DMA2D_InitStruct.DMA2D_PixelPerLine = MapClip.Dest_w;
  DMA2D_Init(&DMA2D_InitStruct);

  DMA2D_FG_StructInit(&DMA2D_FG_InitStruct);
  DMA2D_FG_InitStruct.DMA2D_FGMA = source_address;
  DMA2D_FG_InitStruct.DMA2D_FGCM = CM_RGB565;
  DMA2D_FG_InitStruct.DMA2D_FGPFC_ALPHA_MODE = NO_MODIF_ALPHA_VALUE;
  DMA2D_FG_InitStruct.DMA2D_FGPFC_ALPHA_VALUE = 0;
  DMA2D_FG_InitStruct.DMA2D_FGO=Map_Width - MapClip.Dest_w;
  DMA2D_FGConfig(&DMA2D_FG_InitStruct);

  DMA2D_StartTransfer();

  while(DMA2D_GetFlagStatus(DMA2D_FLAG_TC) == RESET);
}


//--------------------------------------------------------------
// Show a Sprite on Screen of copy it in Video Buffer
// num     			- Sprite number to Show
// x       			- X coordinate from the Left Upper corner of the Sprite.
// y       			- Y coordinate from the Left Upper corner of the Sprite.
// show    			- 0 = Show the transparent color , other = Don't show the transparent color
// transparent	- Number of the transparent color
// GFX_Layer		- Screen buffer number where we put the Sprite.
// SDRAM_Buff		- Internal or External SDRAM Video Buffer.
//--------------------------------------------------------------
void GFX_Sprite_Show(uint16_t Sprite_Num , int16_t x , int16_t y , uint8_t show , uint16_t transparent , uint8_t GFX_Layer, uint8_t SDRAM_Buff)
{
	// Show the Sprite
	uint16_t	x1;
	uint16_t	y1;
	uint16_t	w = 0;
	uint16_t	h = 0;
	uint32_t  addr = 0;
	
	//Starting memory offset for this Sprite
	addr = Sprite_Num * (GFX_SPRITE_LENGTH / 2);
	// Sprite Width
	w = SpriteMemory[addr];
	addr++;
	// Sprite Height
	h = SpriteMemory[addr];
	addr++;

	// Show the sprite
	for (y1 = 0;y1 < h;y1++)
		for (x1 = 0;x1 < w;x1++)
		{
			// Show each Sprite pixel
			if(show == 0)
			{
				//Put the pixel in the VideoBuffer
				GFX_SetPixel(x + x1,y + y1,SpriteMemory[addr],GFX_Layer, SDRAM_Buff);
			}
			else
			{
				// Here we hide the transparent color
				if(SpriteMemory[addr] != transparent)
				{
					//Put the pixel in the VideoBuffer
					GFX_SetPixel(x + x1,y + y1,SpriteMemory[addr],GFX_Layer, SDRAM_Buff);
				}
			}
			addr++;
		}
}

