OPTION BASE 0
OPTION EXPLICIT
OPTION DEFAULT NONE

'******************************************************************************************
' Filename: sub-backup-program.inc
' Author  : elk1984
' Version : 1.1
' Date    : 27 October 2020
' 
' Description:
' Automatic Backup Routine to prevent file loss, the dreaded "line too long" error or
' to protect against bad edits.
'
' Usage:  
' Add into your main program
' #Include "sub-backup-program.inc"
' DoBackup(<NumVersion>, <ShowDetail>)
'
' Where <NumVersion> is an integer of the number of versions to keep
' And <ShowDetail> is an integer; 0 = silent, 1 = show debug information
'
' Change History:
' 1.0 - 22 Oct 2020 - Initial Version
' 1.1 - 27 Oct 2020 - Update to only create a new backup copy if the source file was modified and no backup exists
'
'
'******************************************************************************************

SUB DoBackup(versions As Integer, showDetail As Integer)
  LOCAL thisDate As String = IsoDate(date$)
  LOCAL thisTime As String = ReplaceText(time$,":","-")
  LOCAL fullPath As String = MM.INFO$(CURRENT)
  LOCAL directory As String = LEFT$(fullPath,LastIndexOf(fullPath,"/"))
  LOCAL fileName As String = RIGHT$(fullPath,LEN(fullPath)-LEN(directory))
  LOCAL extension As String = RIGHT$(fileName,LEN(fileName)-LastIndexOf(fileName,"."))
  LOCAL fileOnly As String = LEFT$(fileName,LEN(fileName)-LEN(extension)-1)
  LOCAL backupFileName As String = fileOnly+"."+ReplaceText(ReplaceText(MM.INFO(MODIFIED fullPath),":","-")," ",".")+"."+extension
  LOCAL cleanupSpec As String = directory + fileOnly+".*.*."+extension

  LOCAL currentWorking As String = CWD$

  CHDIR directory

  IF ShowDetail>=1 THEN  
    PRINT "Backing Up Program "
    PRINT "Full Path        : " + fullPath
    PRINT "Directory        : " + directory
    PRINT "File Name        : " + fileName
    PRINT "File Only        : " + fileOnly
    PRINT "Extension        : " + extension
    PRINT "Backup Name      : " + backupFileName
    PRINT "Date             : " + thisDate 
    PRINT "Time             : " + thisTime
    PRINT "Versions To Keep : " + STR$(Versions)
    PRINT "Cleanup Spec     : " + cleanupSpec
  END IF

  IF showDetail>=2 THEN PRINT "Creating backup of " + fileName + " called " + backupFileName

  'Create a copy if there's not already one for that modified time
  IF MM.INFO(FILESIZE backupFileName) = -1 THEN
    IF versions > 0 THEN
      COPY fileName TO backupFileName
    END IF
  END IF
  
  IF showDetail>=2 THEN PRINT "Clearing Up Old Versions.  Retaining " + str$(versions) + " of files matching " + cleanupSpec
  ManageVersions(cleanupSpec, versions, showDetail)

  CHDIR currentWorking
END SUB

'******************************************************************************************

SUB ManageVersions(fileSpec As String, versionCount As Integer, showDetail as Integer)
  LOCAL backupFile As String = DIR$(fileSpec,FILE)
  LOCAL backupFilesFound As Integer = 0
  LOCAL i As Integer = 0
  LOCAL currentWorking As String = CWD$

  'Find the number of matching files to generate a correctly sized array 
  DO WHILE backupFile <> ""
    backupFilesFound = backupFilesFound + 1
    backupFile = DIR$()
  LOOP 

  IF backupFilesFound > versionCount THEN

    'Create and fill array of filenames
    LOCAL backupFiles(backupFilesFound) As String  
  
    'Change into the directory of the fileSpec
    CHDIR LEFT$(fileSpec,LastIndexOf(fileSpec,"/"))
    
    backupFile = DIR$(fileSpec,FILE)
    
    IF backupFile <> "" THEN
      DO 
        backupFiles(i) = backupFile
        backupFile = DIR$()
        i = i + 1
      LOOP UNTIL backupFile = ""
    END IF
    
    'Sort array ascending
    SORT backupFiles()
    
    'Loop backwards through the file collection to remove versions > version count
    i = backupFilesFound-1
    DO 
      IF backupFilesFound-i > versionCount-1 THEN
        IF showDetail>=2 THEN PRINT "Removing File " + backupFiles(i)
        IF backupFiles(i) <> "" THEN
          KILL backupFiles(i)
        END IF
      END IF
  
      IF showDetail>=3 THEN
        PRINT backupFilesFound-i,versionCount,backupFiles(i)
      END IF
      i=i-1
    LOOP UNTIL i < 0
  
    'Cleanup    
    CHDIR currentWorking
    ERASE backupFiles
  END IF

  'Cleanup
  ERASE backupFile
  ERASE backupFilesFound
  ERASE i
END SUB

'******************************************************************************************

FUNCTION IsoDate(inputDate As String) As String
  IsoDate = RIGHT$(inputDate,4) + "-" + MID$(inputDate,4,2) + "-" + LEFT$(inputDate,2)
END FUNCTION

'******************************************************************************************

FUNCTION ReplaceText(inputString As String, textToReplace As String, replacementString As String) As String
  LOCAL foundPosition As Integer = 1
  LOCAL lastFoundPosition As Integer = 1
  LOCAL newString As String = ""
  LOCAL preFindString As String = ""
  LOCAL postFindString As String = ""

  lastFoundPosition=foundPosition
  foundPosition=INSTR(inputString, textToReplace)
  
  IF foundPosition > 0 THEN
    'Replace The Found Text
    preFindString=MID$(inputString,lastFoundPosition,foundPosition-1)
    postFindString = MID$(inputString,foundPosition+LEN(textToReplace),LEN(inputString)-foundPosition+LEN(textToReplace))
    newString=preFindString + replacementString + postFindString

    'Search for the next instance by recursing the string
    newString=ReplaceText(newString, textToReplace, replacementString)
  ELSE
    newString=inputString
  END IF

  ReplaceText=newString
END FUNCTION

'******************************************************************************************

FUNCTION LastIndexOf(inputString As String, textToFind As String) As Integer
  LOCAL foundPosition As Integer = 1
  LOCAL lastFoundPosition As Integer = 1

  DO
    lastFoundPosition=foundPosition
    foundPosition=INSTR(foundPosition+1,inputString, textToFind)
  LOOP UNTIL foundPosition=0
  
  LastIndexOf=lastFoundPosition
END FUNCTION

'******************************************************************************************

                     