import serial
import keyboard
import re
import pdb
import getopt
import sys

#xprof.py acquires and analyzes statistical profiling data generated by xprof.inc on CMM2.
#Author: Epsilon.

XPROF_VERSION = "0.1"

#Width of one bin in the histogram (unit: MMBasic line numbers)
DEFAULT_BIN_WIDTH=10

#Sometimes certain bins consume so much CPU attention that they completely squash the other entries
#in the histogram. When you want more resolution on the other entries, you can temporarly list the 
#CPU hogging entries here and they will be filtered out from the results.
IGNORE_LIST = ["GameManager.inc:[60-69]"]

pat = re.compile('P\:\[(\S*)\:(\d*)\]')

def xprof(port, baudrate, binwidth):
    "Acquire profiling data from given serial port at given baudrate."

    datapoints = 0
    acquiredLines = []
    hist = {}

    ser = serial.Serial(port, baudrate)

    print("Press 'q' to quit data acquisition.")

    while not keyboard.is_pressed('q'):
        l = ser.readline().decode()
        if l[0:2] == "P:":
            acquiredLines.append(l)
            datapoints += 1
            print("{}\r".format(datapoints), end='')

    for l in acquiredLines:
        match = pat.match(l)
        if match:
            modname = match.group(1)
            if modname == '':
                modname = '<main>.bas'

            binnr = int(match.group(2)) // binwidth

            if modname not in hist:
                hist[modname] = {}
            
            if binnr not in hist[modname]:
                hist[modname][binnr] = 0
            else:
                hist[modname][binnr] += 1

    maxval=0

    for modname in hist:
        for binnr in hist[modname]:
            entryStr = "{}:[{}-{}]".format(modname, binnr*binwidth, binnr*binwidth+binwidth-1)
            if entryStr not in IGNORE_LIST:
                v = hist[modname][binnr]
                maxval = max(v,maxval)

    scalefactor = 20/maxval

    print("Acquired {} datapoints.".format(datapoints))
    print()

    modnames = list(hist.keys())
    modnames.sort()
    for modname in modnames:
        bins = list(hist[modname].keys())
        bins.sort()
        for binnr in bins:
            entryStr = "{}:[{}-{}]".format(modname, binnr*binwidth, binnr*binwidth+binwidth-1)
            if entryStr not in IGNORE_LIST:
                entryStr += ": {}".format(hist[modname][binnr])
                entryStr += "{}:{}".format(" "*max(0, 50-len(entryStr)),"*"*int(hist[modname][binnr]*scalefactor))
                print(entryStr)

def usage():
    print()
    print("Usage:")
    print("python xprof.py -p <serial port> -b <baudrate> [-w <binwidth>]")
    print("Default binwidth value is {} lines.".format(DEFAULT_BIN_WIDTH))
    print()
    print("Example:")
    print("python xprof.py -p COM8 -b 115200")
    print()

if __name__ == "__main__":
    print("xprof.py V{} by Epsilon.".format(XPROF_VERSION))

    try:
        opts, args = getopt.getopt(sys.argv[1:], "hp:b:w:", ["help", "port=", "baudrate=", "binwidth="])
    except getopt.GetoptError as err:
        # print help information and exit:
        print(err)  # will print something like "option -a not recognized"
        usage()
        sys.exit(2)

    port = None
    baudrate = None
    binwidth = DEFAULT_BIN_WIDTH

    for o, a in opts:
        if o in ("-h", "--help"):
            usage()
            sys.exit()
        elif o in ("-p", "--port"):
            port = a
        elif o in ("-b", "--baudrate"):
            baudrate = int(a)
        elif o in ("-w", "--binwidth"):
            binwidth = int(a)
        else:
            assert False, "unhandled option"

    if port and baudrate:
        xprof(port, baudrate, binwidth)
    else:
        usage()
