  '-------------------------------------------------------------------------------'
  '       MM Console+ Library                         								            '
  '                                                                               '
  '       This file contains Library routines and a special Font                  '
  '       required for use with the MM Console+ program.                          '
  '                                                                               '      
  '       Author:       Philip Erb                                                '
  '                                                                               '
      DIM Lib_Title$ =  "MM Console+ Lib"                                         '
  '                                                                               '
      DIM Lib_Ver$   =  "v1.0.01"                                                 '
  '                                                                               '
      DIM Lib_Date$  =  "22-Jun-2019"                                             '
  '-------------------------------------------------------------------------------'


  '================================================================================================
  '
  '   S T R I N G    A R R A Y    S O R T    R O U T I N E    
  '
  '================================================================================================
  
  'Handles sorting the entries in a String Array into ascending alphanumeric order.  

  Sub SortStrArray(spArray$(),ipMaxIdx%)
    'Sorts the entries in the spArray$() array into ascending order (not case sensitive).
    'Assumes OPTION BASE 0 (the default setting).
    'ipMaxIdx specifies the maximum valid Index into the spArray$() array.
    'If ipMaxIdx is less than 1, the subroutine returns immediately as there is
    'either 0 or only 1 entry in the array and therefore doesn't need sorting.
    
    If ipMaxIdx < 1 then Exit Sub
    
    Local Integer Idx,Sorted
    Local Temp$
    
    Sorted=0
    
    Do While Sorted=0
      Sorted=1
      
      For Idx=0 to (ipMaxIdx-1)
        If UCASE$(spArray$(Idx)) > UCASE$(spArray$(Idx+1)) Then
          Temp$=spArray$(Idx+1)
          spArray$(Idx+1)=spArray$(Idx)
          spArray$(Idx)=Temp$
          Sorted=0
        Endif
      Next Idx
      
    Loop
    
  end sub
  

  '================================================================================================
  '
  '   2 0 ,   1 6 ,   1 2   &   4    B I T    I N T E G E R    S T O R A G E    R O U T I N E S    
  '
  '================================================================================================
  
  'Supports storing 1 x unsigned integer value between 0 and 1048575 (&HFFFFF) as a 20-bit value  
  'plus 1 x unsigned integer value between 0 and 65535 (&HFFFF) as a 16-bit value
  'plus 2 x unsigned integer values between 0 and 4095 (&HFFF) as 2 x 12-bit value groups 
  'and 1 x unsigned integer value between 0 and 15 (&HF) as 1 x 4-bit value 
  'all stored within a standard MM 64-bit Integer variable.
  'This is specifically intended for use with Integer Arrays in the MicroMite Console+ project
  'for managing the linked list as used by the Lite Editor whilst minimising the RAM usage.
  '
  '   The 20-bit value will be referred to as Int20.
  '   The 16-bit value will be referred to as Int16. 
  '   The 12-bit values will be referred to as Int12.
  '   The 4-bit value will be referred to as Bit4.
  '
  '   The 2 x Int12's occupy the Least Significant 24 bits (0-23) of the Int64 variable.
  '   The 1 x Int16 occupies the next 16 bits (24-39) of the Int64 variable.
  '   The 1 x Int20 occupies the next 20 bits (40-59) of the Int64 variable.
  '   The 1 x Bit4 (4-bit value) occupies the Most Significant 4 bits (60-63) of the Int64 variable.
  '
  '     E.G.  DIM Integer SampleArray(5)  
  '
  '           So to retrieve the 2 12-bit values in index 3 of SampleArray use:-
  '                 Value0% = GetInt12(SampleArray(3),0)  'Int12 position 0
  '                 Value1% = GetInt12(SampleArray(3),1)  'Int12 position 1  
  '
  '           To retrieve the 16-bit value from this same array element (index 3) use:-
  '                 Value2% = GetInt16(SampleArray(3))    'Int16
  '
  '           To retrieve the 20-bit value from this same array element (index 3) use:-
  '                 Value3% = GetInt20(SampleArray(3))    'Int20
  ' 
  '           and to retrieve the 4-bit value from this same array element (index 3) use:-
  '                 Value4% = GetBit4(SampleArray(3))     'Bit4 
  '
  '           To update the value stored in Int12 position 1 in index (4) use:-
  '                 PutInt12(SampleArray(4),1,NewValue%)
  '
  '           To update the value stored in Int16 in this array element (index 4) use:-
  '                 PutInt16(SampleArray(4),NewValue%)
  '
  '           To update the value stored in Int20 in this array element (index 4) use:-
  '                 PutInt20(SampleArray(4),NewValue%)
  ' 
  '           or to update the 4-bit value in this array element (index 4) use:-
  '                 PutBit4(SampleArray(4),NewValue%)
  '
  'Use ONLY positive values for all data types - Int20, Int16, Int12 and Bit4. 
  'The values may be specified as Hex values. The allowed ranges are:-
  '
  '           For the Int20     0 - 1048575           '&H0 -> &HFFFFF
  '           For the Int16     0 - 65535             '&H0 -> &HFFFF
  '           For the Int12's   0 - 4095              '&H0 -> &HFFF
  '           For the Bit4      0 - 15                '&H0 -> &HF
  '
  'NOTE: All values are passed into or out of the Getxxxx and Putxxxx functions via
  '----- standard Int64 variables.
  '
  'If either the Int12 Position number (ipPos) or the Int20, Int16, Int12 or Bit4 
  'value (ipVal) arguments are outside the allowed range, an appropriate error 
  'message will be displayed on the console and the program terminated.
  '  
  Function GetInt20(ipInt64%) AS Integer
    'Retrieves the 20 bit group from a 64 bit integer
    '(setup as packed 1 x 4-bit + 2 x Int12 + 1 x Int16 + 1 x Int20 values)
    
    GetInt20=(ipInt64 AND (&HFFFFF << 40)) >> 40
    
  end function
  
  Sub PutInt20(ipInt64%,ipVal%)
    'Stores a 20 bit Int20 value within a 64 bit integer
    '(setup as packed 1 x 4-bit + 2 x Int12 + 1 x Int16 + 1 x Int20 values)
    Local Integer i_Mask,i_Int20Bits
    
    If ipVal < 0 or ipVal > 1048575 Then
      'value too big to store in 20 bit Int20
      ERROR GetStr(250,STR$(ipVal))                   'display error and terminate program
      Exit Sub
    Endif
    
    i_Mask=&HF00000FFFFFFFFFF
    i_Int20Bits=(ipVal AND &HFFFFF) << 40
    ipInt64=(ipInt64 AND i_Mask) OR i_Int20Bits       'store Int20 value

  end SUB
  
  Function GetInt16(ipInt64%) as integer
    'Retrieves the 16 bit group from a 64 bit integer
    '(setup as packed 1 x 4-bit + 2 x Int12 + 1 x Int16 + 1 x Int20 values)
    
    GetInt16=(ipInt64 AND (&HFFFF << 24)) >> 24
    
  end function
  
  Sub PutInt16(ipInt64%,ipVal%)
    'Stores a 16 bit Int16 value within a 64 bit integer
    '(setup as packed 1 x 4-bit + 2 x Int12 + 1 x Int16 + 1 x Int20 values)
    Local Integer i_Mask,i_Int16Bits
    
    If ipVal < 0 or ipVal > 65535 Then
      'value too big to store in 16 bit Int16
      ERROR GetStr(251,STR$(ipVal))                   'display error and terminate program
      Exit Sub
    Endif
    
    i_Mask=&HFFFFFF0000FFFFFF
    i_Int16Bits=(ipVal AND &HFFFF) << 24
    ipInt64=(ipInt64 AND i_Mask) OR i_Int16Bits       'store Int16 value
    
  end SUB
  
  Function GetInt12(ipInt64%,ipPos%) as Integer
    'Retrieves a specified 12 bit group from a 64 bit integer
    '(setup as packed 1 x 4-bit + 2 x Int12 + 1 x Int16 + 1 x Int20 values)
    
    If ipPos < 0 or ipPos > 1 then
      'valid Int12 Position range is 0 -> 1
      ERROR GetStr(252)                               'display error and terminate program
      Exit Sub
    Endif
    
    GetInt12=(ipInt64 AND (&HFFF << (ipPos * 12))) >> (ipPos * 12)
    
  end function
  
  Sub PutInt12(ipInt64%,ipPos%,ipVal%)
    'Stores a 12 bit Int12 value in a specified Int12 position (0, 1) 
    'within a 64 bit integer (setup as packed 1 x 4-bit + 2 x Int12 
    '+ 1 x Int16 + 1 x Int20 values)
    Local Integer i_Mask,i_Int12Mask,i_Int12Bits 
    local Integer i_Mask_0=&HFFFFFFFFFFFFF000
    Local Integer i_Mask_1=&HFFFFFFFFFF000FFF
    
    If ipPos < 0 or ipPos > 1 then
      'valid Int12 Position range is 0 -> 1
      Error GetStr(252)                               'display error and terminate program
      Exit Sub
    Endif
    
    If ipVal < 0 or ipVal > 4095 Then
      'value too big to store in 12 bit Int12
      ERROR GetStr(253,STR$(ipVal))                   'display error and terminate program
      Exit Sub
    Endif
    
    i_Int12Mask=&HFFF
    Select Case ipPos
      Case 0
        i_Mask=i_Mask_0
        i_Int12Bits=ipVal AND i_Int12Mask
        
      Case 1
        i_Mask=i_Mask_1
        i_Int12Bits=(ipVal AND i_Int12Mask) << 12
        
    end Select
    
    ipInt64=(ipInt64 AND i_Mask) OR i_Int12Bits       'store Int12 in selected position
    
  end SUB
  
  Function GetBit4(ipInt64%) as Integer
    'Retrieves the 4-bit value from a 64 bit integer 
    '(setup as packed 1 x 4-bit + 2 x Int12 + 1 x Int16 + 1 x Int20 values)
    
    GetBit4=(ipInt64 AND (&HF << 60)) >> 60
    
  end function
  
  Sub PutBit4(ipInt64%,ipVal%)
    'Stores a specified 4-bit value in a 64 bit integer
    '(setup as packed 1 x 4-bit + 2 x Int12 + 1 x Int16 + 1 x Int20 values)
    Local Integer i_Mask,i_4Bits
    
    If ipVal < 0 or ipVal > 15 Then
      'value too big to store in 4 bits
      ERROR GetStr(254,STR$(ipVal))                   'display error and terminate program
      Exit Sub
    Endif
    
    i_Mask=&H0FFFFFFFFFFFFFFF
    i_4Bits=(ipVal AND &HF) << 60
    ipInt64=(ipInt64 AND i_Mask) OR i_4Bits           'store Bit4 value in Most Significant 4 bits
    
  end SUB



  '===========================================================================================
  '     E M B E D D E D    F O N T    D E C L A R A T I O N S
  '===========================================================================================

  ' IBM_VGA10x16_96 (Modified)
  ' Font type    : Full (96 characters)
  ' Font start   : CHR$(32)
  ' Font size    : 10x16 pixels
  ' Memory usage : 1924 bytes
  ' Font adapted from: http://www.rinkydinkelectronics.com/r_fonts.php

  DefineFont #16
    6020100A 00000000 00000000 00000000 00000000 00000000 C0000000 80071E78
    000C30C0 0030C000 00000000 30C30C00 000012CC 00000000 00000000 00000000
    00000000 C31F36D8 1F36D860 00D860C3 00000000 E103030C 0318618C 10030CE0
    0CF830C6 00000003 00000000 C0186100 0C183060 000C3106 00000000 C0010000
    030736D8 1966B8B1 00EC6086 00000000 C0000300 00001830 00000000 00000000
    00000000 60000000 01061830 06186080 0018C000 00000000 80010000 80010630
    01061860 0060C080 00000000 00000000 C10C0000 0C1EFEE1 000000C0 00000000
    00000000 00030000 030CFCC0 00000000 00000000 00000000 00000000 03000000
    1830C000 00000000 00000000 00000000 0000FC01 00000000 00000000 00000000
    00000000 00000000 0030C000 00000000 00000000 C0000100 0C183060 00000106
    00000000 E0010000 669861CC 9861B6D9 00783063 00000000 C0000000 00033C70
    030C30C0 00FCC000 00000000 E1030000 8001038C 183060C0 00FC3106 00000000
    E1030000 C100038C 00030CE0 00F830C6 00000000 60000000 860D1E38 0106FC61
    003C6080 00000000 F1070000 07186080 00030CE0 00F830C6 00000000 C0010000
    071860C0 18638CE1 00F830C6 00000000 F1070000 C000038C 06183060 00608001
    00000000 E1030000 C318638C 18638CE1 00F830C6 00000000 E1030000 C318638C
    00030CF0 00F060C0 00000000 00000000 00030C00 03000000 0000C000 00000000
    00000000 00030C00 03000000 0060C000 00000000 00000000 0103060C 0318C080
    000C6000 00000000 00000000 C00F0000 003F0000 00000000 00000000 00000000
    000318C0 03060C60 00C08001 00000000 E1030000 8001638C 000C30C0 0030C000
    00000000 00000000 C61863F8 1B6FBCF1 00F80086 00000000 80000000 C6183670
    1863FC31 008C31C6 00000000 E0070000 C30C33CC 0C33CCE0 00F831C3 00000000
    E0010000 061861CC 18608001 00783043 00000000 C0070000 C30C33D8 0C33CC30
    00F061C3 00000000 F0070000 030D31CC 0C30D0C0 00FC3143 00000000 F0070000
    030D31CC 0C30D0C0 00E00103 00000000 E0010000 061861CC 1863BC01 007430C3
    00000000 31060000 C718638C 18638CF1 008C31C6 00000000 E0010000 00030C30
    030C30C0 0078C000 00000000 F0000000 80010618 19661860 00F06086 00000000
    30070000 830D33CC 0C36F0C0 00CC31C3 00000000 80070000 030C30C0 0C30C000
    00FC3143 00000000 19060000 E69F7FCE 986186D9 00861966 00000000 31060000
    C61F7BCC 18639CF1 008C31C6 00000000 E1030000 C618638C 18638C31 00F830C6
    00000000 E0070000 C30C33CC 0C30C0E0 00E00103 00000000 E1030000 C618638C
    1A638C31 06F8F0C6 0000C001 E0070000 C30C33CC 0C33D8E0 00CC31C3 00000000
    E1030000 010C638C 180318C0 00F830C6 00000000 F9070000 00834CB6 030C30C0
    0078C000 00000000 31060000 C618638C 18638C31 00F830C6 00000000 19060000
    66986186 8C618619 0030E0C1 00000000 19060000 66986186 9F6DB619 00CC30E3
    00000000 19060000 80073386 0C1E30C0 008619C6 00000000 19060000 C18C6186
    030C30E0 0078C000 00000000 F9070000 80014386 183060C0 00FE1926 00000000
    E0010000 01061860 06186080 00788001 00000000 01000000 031C6000 010E7080
    000430C0 00000000 E0010000 80010618 01061860 00786080 00000000 80000000
    C0183670 00000000 00000000 00000000 00000000 00000000 00000000 00000000
    0000E01F 80010600 00000030 00000000 00000000 00000000 00000000 000F0000
    1966F860 00EC6086 00000000 00070000 030F30C0 0C33CC60 00F830C3 00000000
    00000000 860F0000 18608031 00F83006 00000000 E0000000 83070618 19669861
    00EC6086 00000000 00000000 860F0000 1860FC31 00F83006 00000000 E0000000
    0306196C 061860C0 00F08001 00000000 00000000 C60E0000 19669861 06F86086
    00C08319 00070000 830D30C0 0C33CCB0 00CC31C3 00000000 C0000000 00070030
    030C30C0 0078C000 00000000 30000000 C001000C 00030C30 330C30C0 00E0C10C
    00070000 C30C30C0 0D3CF060 00CC3183 00000000 C0010000 00030C30 030C30C0
    0078C000 00000000 00000000 C71C0000 9B6DB6F9 00B6D966 00000000 00000000
    831B0000 0C33CC30 00CC30C3 00000000 00000000 860F0000 18638C31 00F830C6
    00000000 00000000 831B0000 0C33CC30 30F830C3 0080070C 00000000 C60E0000
    19669861 06F86086 00F08001 00000000 831B0000 0C30CCB0 00E00103 00000000
    00000000 860F0000 011CC030 00F83086 00000000 80000000 811F1860 06186080
    0038B001 00000000 00000000 86190000 19669861 00EC6086 00000000 00000000
    66180000 8C618619 0030E0C1 00000000 00000000 66180000 9B6D8619 00CCF867
    00000000 00000000 63180000 070C7830 00863183 00000000 00000000 C6180000
    18638C31 03FC30C6 00C08701 00000000 C61F0000 0C183060 00FC3106 00000000
    70000000 03030C30 030C3080 001CC000 00000000 C0000000 00030C30 030C3000
    0030C000 00000000 80030000 00030C30 030C3070 00E0C000 00000000 B1030000
    000000B8 00000000 00000000 00000000 00020000 030F38C0 0F3EFCE0 20C08003
    00000000 00000000
  End DefineFont