  '-------------------------------------------------------------------------------'
  '       Font Viewer Utility Program                                             '
  '                                                                               '
  '       Author:   Philip Erb                                                    '
  '                                                                               '
      Dim ProgTitle$ = "FontView"                                                 '
  '                                                                               '
      Dim ProgVer$ =  "v1.0.08"                                                   '
  '                                                                               '
      Dim ProgDate$ = "09-Mar-2020"                                               '
  '-------------------------------------------------------------------------------'
  
  OPTION EXPLICIT
  OPTION DEFAULT INTEGER
  
  'CONSTANTS
  '=========

  Const FONTPATH$ = "A:/FONTS"                                  'Directory for the Font files on the SD Card
  Const MAXBUTTONS% = 13                                        'Max buttons displayed at the same time
  Const ICON_FONT%  = 9                                         'Font# for PgUp/PgDn Icon symbols
  
  'Message Box / Special Use Colours
  Const FCERR% = RGB(WHITE)
  Const BCERR% = RGB(RED)
  Const FCMSG% = RGB(CYAN)
  Const BCMSG% = RGB(BLACK)
  
  'Screen Colours
  Const FC% = RGB(WHITE)
  Const BC% = RGB(BLUE)

  'Debug Mode Control
  Const DEBUG=0                                                 '0 = No Debug, >0 = Debug

  
  'VARIABLES
  '=========
  
  'screen size to use - only 320x240 and 800x480 are used.
  'below 320x240 is NOT supported.
  'sizes between 320x240 and 800x480 will be formatted as 320x240,
  'except for the Character Map which will use the actual screen
  'size if the Character Map grid is large enough.  
  Dim Hres,Vres                                                 'in-use Hres & Vres
  
  'Font size to use - 320x240 uses Font #1, 800x480 uses Font #2.
  Dim Fnt                                                       'in-use Font (NOT Font being viewed!!)

  'size of Font being used (in pixels): Font #1 = 8x13, Font #2 = 12x20
  Dim fW,fH                                                     'in-use Font size (pixel Width & Height)
  
  'for tracking text positioning on screens, message boxes, etc.
  Dim TxtX,TxtY,SaveTxtY

  'for managing File Lists
  Dim FileCnt                                                   'total # of files in the list
  Dim FN_MaxLen                                                 'length of largest file name
  Dim flMaxIdx                                                  'max index into flNames$ array (list)
  Dim flMaxCh                                                   'max file name chars to be displayed
  Dim flMaxLn                                                   'max lines visible in the list box
  Dim flStrtPtr                                                 'index of first entry visible in list box
  Dim flEndPtr                                                  'index of last entry visible in list box
  Dim flX,flY,flW,flH                                           'list box location / size tracking

  'for processing the selected font file and font definition
  Dim Ln
  Dim TokPtr,TokStrt,TxtLen,EOL
  Dim F_Strt,F_Attr,F_NbrChars,F_StrtChar,F_Width,F_Height,F_PixelsPerChar,F_PixelCount
  Dim DWord,QWord,QWordByteIdx
  Dim QWordByteMask(7)                                          '8 element array (0 -> 7)
  Dim Err
  Dim EOJ = 0
  Dim ScaleFactor = 1                                           'scaling factor (default = 1)

  Dim FontFile$, LoadedFont$
  Dim Txt$, Token$ 
  Dim Work$, ErrMsg$
  
  'Array and associated variables for storing the pixel representation of each character
  'defined in the font file.
  Dim CharIdx, MaxCharIdx, FirstCharASCIIVal, LastCharASCIIVal
  Dim QWordIdx, MaxQWordIdx
  'Following is the Font Chars Array. The Dim statement here is a dummy only. The "real"
  'declaration of the array is done in DeclareFontCharsArray() subroutine, where values
  'from the Font Attributes are used to correctly size each array dimension.
  Dim FontChars(5,1)

  
  'Button Table
  'B_Coords array uses 10 elements - used as follows:
  '   0 - Left
  '   1 - Top
  '   2 - Right
  '   3 - Bottom
  '   4 - Enabled (1) / Disabled (0)
  '   5 - File Name List entry index / Icon Chr$ value
  '   6 - Text / Icon Left
  '   7 - Text / Icon Top
  '   8 - Forecolour (RGB value)
  '   9 - Backcolour (RGB value)
  
  Dim B_Coords((MAXBUTTONS%-1), 9)                              'Button Table array
  Dim MaxButtonIdx                                              'Max button index in use on current screen

  'Special variables for requesting Screen Image dump to SDCard via Console Input
  Dim In_Count 
  Dim Flag
  Dim In_Txt$
  Dim Img$

  'Program State tracking
  Dim State                                                     'tracks the current program action

  ' Program States   Program Activity
  ' ==============   ================
  '       0          Not Used
  '      10          Validate SDCard is configured and present
  '      20          Display Main Screen  (List of Font files on SDCard)
  '      25          Wait input on Main Screen (Font File select, PgUp/PgDn or Prog Exit)
  '      30          Load selected Font File
  '      40          Display Character Map for selected Font File
  '      45          Wait input on Char Map Screen (Close or Change Scaling Factor)
  '      50          Display Font Scaling Factor Selection Screen
  '      55          Wait input on Font Scaling Factor Selection Screen (1x, 2x or 3x)
  '      70          Display Info Message Box and wait for OK button to be pressed
  '      75          Display Info Message Box, wait for OK button press, then go to EOJ
  '      80          Display Error Message Box and wait for OK button to be pressed
  '      99          Exit Program

  '================================================================================================
  '     C O D E    I N I T I A L I Z A T I O N 
  '================================================================================================

  Print "Starting " + ProgTitle$ + ": " + ProgVer$ + " - " + ProgDate$
  
  Colour FC%, BC%                                                                
  CLS
  
  If UCase$(MM.DEVICE$) = "MICROMITE MKII" Then
    Print "This program cannot run on a Micromite MkII (MX170)"
    Text MM.HRES\2,MM.VRES\2-16,"This program cannot run on",CT,1,1
    Text MM.HRES\2,MM.VRES\2+3,"a Micromite MkII (MX170)",CT,1,1
    END                                                         'terminate if on MM2
  EndIf
  
  If MM.HRES < 320 OR MM.VRES < 240 Then
    If MM.HRES=240 AND MM.VRES=320 Then
      'LCD in PORTRAIT mode
      Print "The LCD Screen needs to be configured in LANDSCAPE ";
      Print "orientation for use with this program."
      Text 120,280,"The LCD screen needs to be",CT,1,1
      Text 120,300,"configured in LANDSCAPE",CT,1,1
      Text 120,320,"orientation for use with",CT,1,1
      Text 120,340,"this program.",CT,1,1
    Else
      Print "This program requires a LCD touch screen with minimum "; 
      Print "resolution of 320x240 in LANDSCAPE orientation."
      Text MM.HRES\2,MM.VRES\2-20,"Min LCD res of",CT,1,1
      Text MM.HRES\2,MM.VRES\2,"320x240 required.",CT,1,1
    EndIf

    END                                                        'terminate the program
  EndIf

  If MM.HRES=480 AND MM.VRES=800 Then
    '800x480 screen in PORTRAIT orientation
    Print "The LCD screen needs to be configured for LANDSCAPE ";
    Print "orientation for use with this program."
    Text 240,340,"The LCD screen needs to be",CT,2,1
    Text 240,370,"configured in LANDSCAPE",CT,2,1
    Text 240,400,"orientation for use with",CT,2,1
    Text 240,430,"this program.",CT,2,1
    END                                                         'terminate program
  EndIf

  If MM.HRES < 800 OR MM.VRES < 480 Then
    'use 320x240 res with Font #1
    Hres=320
    Vres=240
    Fnt=1
    fW=8
    fH=13

  Else
    'use 800x480 res with Font #2
    Hres=800
    Vres=480
    Fnt=2
    fW=12
    fH=20
  EndIf    
  
  'other initialization routines
  InitQWordByteMask()
  State=10                                                      'start @ verify SDCard access

  '================================================================================================
  '     M A I N    P R O G R A M    L O O P
  '================================================================================================
  
  Do
    Watchdog 40000                                              'reset after 40 secs on hang

    'check for Console input
    In_Count = LOC(#0)                                          'count of chars in Console buffer

    If In_Count > 0 Then                                        'console input present
      In_Txt$ = In_Txt$ + UCase$(Input$(In_Count, #0))          'grab data from buffer (in upper case)

      If Instr(In_Txt$,"#IMG#") Then
        'request to Save Screen Image to SDCard received
        '- perform Save Image function.
        Img$="A:/IMG/"+Mid$(Date$,7,4)+Mid$(Date$,4,2)+Left$(Date$,2)
        Img$=Img$+"_"+Left$(Time$,2)+Mid$(Time$,4,2)+Mid$(Time$,7,2)
        Print "Processing save screen image request - please wait."

        ON ERROR SKIP 1                                         'trap error if any
        Save Image Img$                                         'Save screen image to SDCard

        If MM.ERRNO <> 0 Then                                   'Save Image failed.
           Print "Save Image attempt failed: " + MM.ERRMSG$     'report failed attempt & error msg
           ON ERROR CLEAR

        Else
          Print "Screen image saved to " + Img$                 'Save Image successful
        EndIf

        In_Txt$ = ""                                            'clear input buffer

      Else
        Flag = 0                                                'reset partial rqst flag
        'check for partial #IMG# request at end of buffer
        If Len(In_TxT$) > 3 Then 
          If Right$(In_Txt$,4) = "#IMG" Then Flag = 1           'set flag                                  
        ElseIf Len(In_Txt$) > 2 Then
          If Right$(In_Txt$,3) = "#IM" Then Flag = 1            'set flag
        ElseIf Len(In_Txt$) > 1 Then
          If Right$(In_Txt$,2) = "#I" Then Flag = 1             'set flag
        ElseIf Len(In_Txt$) > 0 Then
          If Right$(In_Txt$,1) = "#" Then Flag = 1              'set flag
        EndIf

        If Flag = 0 Then                                        'no partial rqst in buffer
          In_Txt$ = ""                                          'clear buffer
        EndIf

        'if the Flag was set, leave the buffer intact and wait for more
        'data to arrive to see if it completes the Save Image request.
      EndIf
    EndIf
      
    'check if display is touched
    If Touch(x) <> -1 Then
      'display is touched - get input and process
      HandleTouch()
    EndIf
    
    Select Case State
      Case 0                                                    'General Wait State
        'do nothing here

      Case 10                                                    'verify SDCard access
        'validate that a SDCard is configured and present.
        If NOT SDCardPresent() Then
          'SD card not accessible - report error
          If Fnt = 1 Then
            'use abridged error messages to fit small screen
            ShowFileError(ErrMsg$,"SDCard MUST be present.")
          Else
            ShowFileError("SDCard Status: " + ErrMsg$,"SDCard is required for this program to run.")
          EndIf

        Else                                                    'successful
          State=20                                              'move on to next action
        EndIf
        
      Case 20                                                   'build and display Font File list
        'display the Main Screen
        DisplayMainScreen()

      Case 25                                                   'Main Screen
        'waiting for Font File selection (or PgUp/PgDn) - nothing to do right now

      Case 30                                                   'load selected Font
        CLS

        Work$=FontFile$

        If Fnt=1 AND Len(FontFile$) > 23 Then
          'full file name will not fit, truncate the file name after char 23.
          Work$=Left$(FontFile$,23)
        EndIf

        Text Hres\2,10,"Font File Name: "+ Work$,CT,Fnt,1
        Text Hres\2,50,"Please Wait - Font File being loaded",CT,Fnt,1       
        LoadFontFile()
        ON ERROR SKIP 1
        Close #1                                                'close the Font File now
        ON ERROR CLEAR

      Case 40                                                   'Font Char Map
        DisplayCharMap()                                        'display Char Map screen

      Case 45                                                   'Char Map screen
        'waiting for 'Close' button press - nothing to do right now.

      Case 50                                                   'Font Scaling screen
        DisplayFontScalingSelect()

      Case 55                                                   'Scaling Factor Select
        'waiting for Font Scaling Factor selection - nothing to do right now

      Case 70                                                   'info message box
        'waiting for 'OK' button press - nothing to do right now

      Case 80                                                   'error message box
        'waiting for 'OK' button press - nothing to do right now

      Case 99                                                   'go to EOJ
        Watchdog Off                                            'disable the Watchdog timeout      
        Exit Do                                                 'exit the Main Loop
    End Select

  Loop

  On Error Skip 1
  CLOSE #1                                                      'attempt to close Font file
  
  CLS RGB(Black)
  
  Text MM.HRES\2,MM.VRES\2,"Program Terminated.",CM,Fnt,1,RGB(WHITE),RGB(BLACK)
  Print "Program terminated."
  
  Pause 500

  END                                                           'terminate program


  '================================================================================================
  '     B U T T O N    H A N D L I N G    P R O C E D U R E S
  '================================================================================================

  Sub DrawButton(which, Pressed)
    'controls drawing/re-drawing any button on any screen / message box
    Local X, Y, W, H, BFC, BBC, Border
    Local Caption$, Icon
    
    If DEBUG = 2 Then Print "DrawButton: which=" + STR$(which) + ", Pressed=" + Str$(Pressed)
    
    X = B_Coords(which,0)
    Y = B_Coords(which,1)
    W = ((B_Coords(which,2) - B_Coords(which,0)) + 1)
    H = ((B_Coords(which,3) - B_Coords(which,1)) + 1)
    Border = 1                                                  'default is border 1px wide
    
    If B_Coords(which,4) = 0 Then
      'disabled button - use Black Forecolour and Gray Backcolour
      BFC = RGB(BLACK)
      BBC = RGB(GRAY)
      
    Else
      If Pressed = 0 Then
        'not pressed - use 'normal' colours as specified for this button
        BFC = B_Coords(which,8)                                 'Forecolour
        BBC = B_Coords(which,9)                                 'Backcolour
      Else
        'pressed - use 'reverse video' - i.e. reverse the normal colours
        BFC = B_Coords(which,9)                                 'Forecolour (reversed)
        BBC = B_Coords(which,8)                                 'Backcolour (reversed)
      EndIf
    EndIf
    
    Select Case State
    
      Case 20, 25                                             'Main Screen
        Select Case which
          Case 0 To 9                                           '10 x File Name "buttons"         
            Border = 0                                          'no button border
            If B_Coords(which,4) = 1 Then                       'only setup Caption text for Enabled buttons in the File List
              Work$ = flNames$(B_Coords(which,5))               'get File Name List text for this button 
              If Len(Work$) > flMaxCh Then
                Work$=Left$(flNames$(B_Coords(which,5))         'truncate name if necessary
              EndIf
              Caption$ = Work$
            Endif
            
          Case 10,11                                            'PgUp / PgDn buttons
            Icon = B_Coords(which,5)

          Case 12                                               'Program Exit button
            If Fnt = 1 Then
              Border = 0                                        'no button border
              Caption$ = Chr$(B_Coords(which,5))                'get text (upper case 'X')

            Else    'Font #2                                                
              Caption$ = "Exit"                                 ''Exit' button
            EndIf

        End Select
            
      Case 40, 45                                             'Font Character Map
        Select Case which
          Case 0                                                'Close button
            Caption$ = "Close"

          Case 1                                                'Change Scaling button
            Caption$ = "Change Scaling"
        End Select

      Case 50, 55                                             'Font Scaling Select
        Border = 0                                              'no borders on "buttons"

        Select Case which
          Case 0                                                '1x Scaling
            Caption$ = "1x Scaling"

          Case 1                                                '2x Scaling
            Caption$ = "2x Scaling"

          Case 2                                                '3x Scaling
            Caption$ = "3x Scaling"

        End Select

      Case 70, 75                                             'Info Message Box
        Caption$ = "OK"
  
      Case 80                                                 'Error Message Box
        Caption$ = "OK"
    
    End Select
    
    If Border > 0 Then                                          'Outline is visible
      'draw the filled button outline (with rounded corners)
      RBox X, Y, W, H, 10, BFC, BBC
    Else
      If B_Coords(which,4)=1 Then
        'if button is enabled, draw rectangle box without an outline
        Box X, Y, W, H, 0, 0, BBC
      EndIf
    EndIf  
    
    If Caption$ <> "" Then
     'add the caption
      Text B_Coords(which,6),B_Coords(which,7),Caption$,LT,Fnt,1,BFC,BBC
    ElseIf Icon <> 0 Then
      'display the specified Icon (Symbol) - uses special Font #9
      Text B_Coords(which,6),B_Coords(which,7),Chr$(Icon),LT,ICON_FONT%,1,BFC,BBC
    EndIf 
    
  End Sub

  '================================================================================================
  '     T O U C H    H A N D L I N G    P R O C E D U R E S
  '================================================================================================

  Sub HandleTouch()
    'see which button has been pressed and process accordingly
    Local which
    
    If DEBUG = 2 Then Print "HandleTouch()"    
    
    which = CheckButtonPress(0,MaxButtonIdx)
    
    If DEBUG = 1 Then Print "CheckButtonPress return value:" + Str$(which)
    
    If which > -1 Then
      'a defined button was pressed - action it
      Select Case State

        Case 0        
      
        Case 25                                                 'Main Screen
          Select Case which
            Case 0 to 9                                         'File Name select "buttons"
              FontFile$=flNames$(flStrtPtr+which)               'get selected File Name

              If FontFile$=LoadedFont$ Then
                'selected Font is already loaded -
                'don't need to re-load, just go to Display Char Map
                State=40                                        'move to next state (Display Font)

              Else
                State=30                                        'move to next State (Load selected Font)
              EndIf
            
            Case 10                                             'PgUp Button
              'scroll list down 1 page (10 lines) or to the start of the list
              ShowFL(Max(flStrtPtr-flMaxLn,0))

            Case 11                                             'PgDn Button
              'scroll list up 1 page (10 lines)
              ShowFL(flEndPtr+1)

            Case 12                                             'Program Exit Button
              State=99                                          'go to EOJ
          End Select

        Case 45                                                 'Font Character Map
          Select Case which
            Case 0                                              'Close button
              ScaleFactor=1                                     'reset Scaling to 1x
              CLS                                               'chear screen then
              State=20                                          'go to Main Screen

            Case 1                                              'Change Scaling button
              State=50                                          'go to Scaling Select screen
          End Select

        Case 55                                                 'Select Scaling Factor
          Select Case which
            Case 0                                              '1x Scaling
              ScaleFactor=1

            Case 1                                              '2x Scaling
              ScaleFactor=2

            Case 2                                              '3x Scaling
              ScaleFactor=3                                     
          End Select

          State=40                                              're-display Font Char Map

        Case 70                                                 'Info Message Box
          'only 1 button - "OK' - redisplay Main Screen
          State=20

        Case 75                                                 'Info Msg Box (go to EOJ)
          'only 1 button - 'OK' - terminate the program
          State=99                                              'go to EOJ

        Case 80                                                 'Error Msg Box
          'only 1 button - 'OK' - terminate the program
          State=99                                              'go to EOJ State

      End Select
      
    Else
      If DEBUG > 0 Then
        Print "Invalid Button pressed - State = " + Str$(State) + ", Button Index = " + Str$(which)
      EndIf
    EndIf
    
  End Sub

  Function CheckButtonPress(startn, endn) As Integer
    Local keydwn, xt, yt, n, saveTimer
    
    If DEBUG = 2 Then Print "CheckButtonPress(): startn=" + Str$(startn) + ", endn=" + Str$(endn)
    
    keydwn = 0
    saveTimer = Timer                                 'save timer value on entry
    
    'check which button has been pressed, then loop until it is released again
    Do
      If Touch(x) <> -1 AND NOT keydwn Then
        'we have a touch - get touch coordinates
        xt = Touch(x)
        yt = Touch(y)
        
        'check the button coordinates to see if the touch was within the boundaries of a button
        For n = startn To endn
          'only consider 'Enabled' buttons
          If B_Coords(n,4) > 0 Then                             'button enabled
            If xt > B_Coords(n, 0) And xt < B_Coords(n, 2) And yt > B_Coords(n, 1) And yt < B_Coords(n, 3) Then
              'we have a defined button press
              'draw the button as pressed
              DrawButton n, 1
              keydwn = 1
              Exit For
            EndIf
          EndIf
        Next n
        
        If DEBUG = 1 Then
          Print "Touch at: " + Str$(xt) + "," + Str$(yt) + " n = " + Str$(n) + " keydwn = " + Str$(keydwn)
        EndIf
      EndIf
      
      'check when touch has been released or if button has been down > 30 secs
      If (Touch(x) = -1 OR (Timer - saveTimer) > 30000) Then
        If keydwn Then
          'button was down - has now been released
          'draw the button as normal (ie, not pressed)
          DrawButton n, 0
          'return value is the button's index
          CheckButtonPress = n
        Else
          'touch did not correspond to any defined button
          'return -1 (invalid button)
          CheckButtonPress = -1
        EndIf
        Exit Function
      EndIf
    Loop
    
  End Function
  
  
  
'==================================================================================================
'       M A I N    S C R E E N    F O R M A T T I N G    A N D    C O N T R O L
'==================================================================================================

  Sub DisplayMainScreen()
    'display the main screen format, then build and display the Font File list
    'State = 20
    
    If Fnt=1 then
      BuildMainScreen_Font1()
    Else
      BuildMainScreen_Font2()
    EndIf
    
    'set Working Directory to Fonts directory
    Chdir FONTPATH$                                               

    'Get info on the Font files in the working directory (count & max file name len)
    GetFileInfo() 

    'handle condition where Font directory is empty
    If FileCnt = 0 Then
      InfoMsgBox("No Font Files found in the", FONTPATH$ + " folder.",1)
      Exit Sub
    EndIf

    'create and populate the File Names array containing a list of all Font files found.
    BuildFileList()
    
    'display the Font File Name list starting from the first entry.
    ShowFL(0)
    
    'wait for file selection
    State=25                                                    

  End Sub
  
  Sub BuildMainScreen_Font1()
    'builds the Main Screen format as used on 320x240 res LCD screen with Font #1
    Local i,ulX,ulY,ulW,Title$
    
    Title$=ProgTitle$ + " - " + ProgVer$
    Text Hres\2,5,Title$,CT,Fnt,1                               'screen title
    ulW=Len(Title$)*fW
    ulX=(Hres\2)-(ulW\2)
    ulY=fH+7
    Line ulX,ulY,(ulX+ulW),ulY,1                                'title underline
    
    TxtX=fW
    TxtY=fH+10
    
    Text Hres\2,TxtY,"Touch the Font File name to be viewed:",CT,Fnt,1
    TxtY=TxtY+(fH+6)
    SaveTxtY=TxtY                                               'save for Fonts Count caption later
    'this text line is not displayed yet since FilCnt hasn't been determined
    'the line (commented out) is included here to assist in tracking the screen format.
    'Text TxtX,TxtY,"Fonts=" + Str$(FileCnt),LT,Fnt,1
    TxtY=TxtY+(fH+3)
    Text TxtX,TxtY,"Use the",LT,Fnt,1
    TxtY=TxtY+(fH+3)
    Text TxtX,TxtY,"buttons",LT,Fnt,1
    TxtY=TxtY+(fH+3)
    Text TxtX,TxtY,"below to",LT,Fnt,1
    TxtY=TxtY+(fH+3)
    Text TxtX,TxtY,"scroll the",LT,Fnt,1
    TxtY=TxtY+(fH+3)
    Text TxtX,TxtY,"list.",LT,Fnt,1
    
    'setup all common values for the 10 File NameList "buttons"
    For i=0 to 9
      B_Coords(i,0)=104                                         'Left
      B_Coords(i,1)=(i*20)+39                                   'Top
      B_Coords(i,2)=319                                         'Right - max, will be adjusted dynamically
      B_Coords(i,3)=B_Coords(i,1)+20                            'Bottom
      B_Coords(i,4)=1                                           'enabled (will be modified dynamically)
      B_Coords(i,5)=0                                           'File List entry index (modified dynamically)
      B_Coords(i,6)=112                                         'Text Left (104+8)
      B_Coords(i,7)=B_Coords(i,1)+3                             'Text Top (3px below button top)
      B_Coords(i,8)=FC%                                         'Forecolour
      B_Coords(i,9)=BC%                                         'Backcolour
    Next i

    'setup the Button Table for the PgUp, PgDn and Prog Exit buttons
    B_Coords(10,0)=23                                           'Left
    B_Coords(10,1)=146                                          'Top
    B_Coords(10,2)=73                                           'Right
    B_Coords(10,3)=190                                          'Bottom
    B_Coords(10,4)=0                                            'disabled - dynamicaly managed
    B_Coords(10,5)=1                                            'PgUp Icon value
    B_Coords(10,6)=34                                           'Icon Left
    B_Coords(10,7)=155                                          'Icon Top
    B_Coords(10,8)=FC%                                          'Forecolour
    B_Coords(10,9)=BC%                                          'Backcolour

    B_Coords(11,0)=23                                           'Left
    B_Coords(11,1)=194                                          'Top
    B_Coords(11,2)=73                                           'Right
    B_Coords(11,3)=234                                          'Bottom
    B_Coords(11,4)=1                                            'enabled - dynamically managed
    B_Coords(11,5)=2                                            'PgDn Icon value
    B_Coords(11,6)=34                                           'Icon Left
    B_Coords(11,7)=203                                          'Icon Top
    B_Coords(11,8)=FC%                                          'Forecolour
    B_Coords(11,9)=BC%                                          'Backcolour

    B_Coords(12,0)=299                                          'Left
    B_Coords(12,1)=0                                            'Top
    B_Coords(12,2)=319                                          'Right
    B_Coords(12,3)=20                                           'Bottom
    B_Coords(12,4)=1                                            'enabled (always)
    B_Coords(12,5)=88                                           'upper case 'X' char value
    B_Coords(12,6)=307                                          'Text Left
    B_Coords(12,7)=4                                            'Text Top
    B_Coords(12,8)=RGB(WHITE)                                   'Forecolour
    B_Coords(12,9)=RGB(RED)                                     'Backcolour

    'set Max Button Index for this screen
    'NOTE: This value remains set even if some of the File Name List entries are not used.
    '===== Those buttons will be disabled and will therefore not show or be 'touch activated'.
    MaxButtonIdx=12                                             'max 13 buttons on the Main Screen

    'setup parameters for the File Name List box
    flMaxCh=25
    flMaxLn=10                                                  'allow 20px line spacing with Font #1 (7px space)
    flX=104
    flY=SaveTxtY
    flW=Hres-flX
    flH=240-flY                                                 
   
  End Sub
    
  
  Sub BuildMainScreen_Font2()
    'builds the Main Screen format as used on 800x480 res LCD screen with Font #2
    Local i,ulX,ulY,ulW,Title$
    
    Title$=ProgTitle$ + " - " + ProgVer$
    Text Hres\2,5,Title$,CT,Fnt,1                               'screen title
    ulW=Len(Title$)*fW
    ulX=(Hres\2)-(ulW\2)
    ulY=fH+7
    Line ulX,ulY,(ulX+ulW),ulY,1                                'title underline
    
    TxtX=fW
    TxtY=fH+13
    
    Text Hres\2,TxtY,"Select the Font File to view from the list (touch file name):",CT,Fnt,1
    TxtY=TxtY+(fH+6)
    SaveTxtY=TxtY                                               'save for top of File Name list
    TxtY=TxtY+6
    Text TxtX,TxtY,"Use the buttons",LT,Fnt,1
    TxtY=TxtY+(fH+6)
    Text TxtX,TxtY,"below to scroll",LT,Fnt,1
    TxtY=TxtY+(fH+6)
    Text TxtX,TxtY,"the list up or",LT,Fnt,1
    TxtY=TxtY+(fH+6)
    Text TxtX,TxtY,"down.",LT,Fnt,1
    
    'setup all common values for the 10 File Name List "buttons"
    For i=0 to 9
      B_Coords(i,0)=228                                         'Left
      B_Coords(i,1)=(i*42)+59                                   'Top  
      B_Coords(i,2)=612                                         'Right  - max, will be adjusted dynamically
      B_Coords(i,3)=B_Coords(i,1)+32                            'Bottom - allows 10px space between "buttons"
      B_Coords(i,4)=1                                           'enabled (willl be modified dynamically)
      B_Coords(i,5)=0                                           'File List entry index (modified dynamically)
      B_Coords(i,6)=240                                         'Text Left (228+12)
      B_Coords(i,7)=(i*42)+65                                   'Text Top  (6px below button top)
      B_Coords(i,8)=FC%                                         'Forecolour
      B_Coords(i,9)=BC%                                         'Backcolour
    Next i
    
    'setup the Button Table for the PgUp, PgDn and Prog Exit buttons
    B_Coords(10,0)=77                                           'Left
    B_Coords(10,1)=200                                          'Top  
    B_Coords(10,2)=127                                          'Right
    B_Coords(10,3)=250                                          'Bottom
    B_Coords(10,4)=0                                            'disabled - dynamically managed
    B_Coords(10,5)=1                                            'PgUp Icon value
    B_Coords(10,6)=88                                           'Icon Left
    B_Coords(10,7)=214                                          'Icon Top
    B_Coords(10,8)=FC%                                          'Forecolour
    B_Coords(10,9)=BC%                                          'Backcolour

    B_Coords(11,0)=77                                           'Left
    B_Coords(11,1)=280                                          'Top  
    B_Coords(11,2)=127                                          'Right
    B_Coords(11,3)=330                                          'Bottom
    B_Coords(11,4)=1                                            'enabled - dynamically managed
    B_Coords(11,5)=2                                            'PgDn Icon value
    B_Coords(11,6)=88                                           'Icon Left
    B_Coords(11,7)=294                                          'Icon Top
    B_Coords(11,8)=FC%                                          'Forecolour
    B_Coords(11,9)=BC%                                          'Backcolour

    B_Coords(12,0)=640                                          'Left
    B_Coords(12,1)=437                                          'Top
    B_Coords(12,2)=B_Coords(12,0)+fW*8                          'Right
    B_Coords(12,3)=B_Coords(12,1)+fH+12                         'Bottom
    B_Coords(12,4)=1                                            'enabled (always)
    B_Coords(12,5)=0                                            'not used
    B_Coords(12,6)=B_Coords(12,0)+fW*2                          'Text Left
    B_Coords(12,7)=B_Coords(12,1)+6                             'Text Top
    B_Coords(12,8)=BC%                                          'Forecolour (reverse video)
    B_Coords(12,9)=FC%                                          'Backcolour (reverse video)
    
    'set Max Button Index for this screen
    'NOTE: This value remains set even if some of the File Name List entries are not used.
    '===== Those buttons will be disabled and will therefor not show or be "touch activated".
    MaxButtonIdx=12                                             'max 13 buttons on the Main Screen
    
    'setup parameters for the File Name List box
    flMaxCh=30                                                  'max file name chars visible in list box
    flMaxLn=10                                                  'allows 42px line spacing with Font #2 (22px space)
    flX=228
    flY=SaveTxtY
    flW=384                                                     'max needed for 30 char file name + 2 spaces (32*12=384)  
    flH=420                                                     '10 lines @ 42px each
    
  End Sub
   
  Sub ShowFL(ipStrtPtr)                                     
    'setup the button coordinates to display the File Name List from
    'the entry specified via the start pointer value.
    If DEBUG=2 Then PRINT "ShowFL: ipStrtPtr=" + Str$(ipStrtPtr)

    LOCAL i,flDispLines,FNLen           
        
    Box flX,flY,flW,flH,0,0,BC%                                 'clear the File List area

    flDispLines=Min((flMaxIdx-ipStrtPtr),(flMaxLn-1))           'calc actual number of lines that will be displayed  
    flStrtPtr=ipStrtPtr                                         'save the current list start pointer
    flEndPtr=flStrtPtr+flDispLines                              'calc current list end pointer

    'update the dynamic values in the Button Table to match the data to be displayed
    For i = 0 to 9
      FNLen=0
      B_Coords(i,5)=ipStrtPtr+i                                 'File Name List entry index for this button
      B_Coords(i,4)=(B_Coords(i,5)<=flMaxIdx)                   'set Enabled/Disabled flag based on button used or not
      If B_Coords(i,4)=1 Then
        'enabled button - in use
        FNLen=Len(flNames$(ipStrtPtr+i))                        'get length of File Name for this entry.
      EndIf
      B_Coords(i,2)=B_Coords(i,0)+((Min(FNLen,flMaxCh)+2)*fW)   'adjust Right coord to text length of entry (+2 spaces)
    Next i
    
    'determine which of the PgUp/PgDn buttons should be enabled / disabled based on 
    'the current position of the displayed File Name List entries in the overall list.
    If flStrtPtr=0 Then
      'at start of list - PgUp not valid at this point.
      B_Coords(10,4)=0                                          'PgUp disabled
    Else
      B_Coords(10,4)=1                                          'PgUp enabled
    EndIf
    
    If flEndPtr=flMaxIdx Then
      'at end of list - PgDn not valid at this point.
      B_Coords(11,4)=0                                          'PgDn disabled
    Else
      B_Coords(11,4)=1                                          'PgDn enabled
    EndIf
    
    For i=0 to MaxButtonIdx                                     'draw/redraw all the buttons on the screen
      DrawButton(i,0)
    Next i
    
    'For Font #2, also show the details of which entries in the list are being displayed
    If Fnt=2 Then
      'clear area first
      Box 0,360,fW*14,119,0,0,BC%
      TxtX=fW
      TxtY=365
      Text TxtX,TxtY,"Displaying",LT,Fnt,1
      TxtY=TxtY+(fH+6)
      Text TxtX,TxtY,"#" + Str$(flStrtPtr+1) + " to #" + Str$(flEndPtr+1),LT,Fnt,1
      TxtY=TxtY+(fH+6)
      Text TxtX,TxtY,"of " + Str$(FileCnt) + " total",LT,Fnt,1
      TxtY=TxtY+(fH+6)
      Text TxtX,TxtY,"Font Files.",LT,Fnt,1

    Else
      'for Font #1, just show the count of Font Files found
      Box (TxtX-3),(SaveTxtY-2),(((Len(Str$(Filecnt))+6)*fW)+6),(fH+4),0,0,BCMSG%
      Text TxtX,SaveTxtY,"Fonts=" + Str$(FileCnt),LT,Fnt,1,FCMSG%,BCMSG%
    EndIf

  End Sub
 
'==================================================================================================
'       L O A D    S E L E C T E D    F O N T    F I L E
'==================================================================================================

  Sub ClearFontLoadVars()
    'clears all global vars associated with the LoadFontFile() subroutine to ensure
    'correct load results are obtained on a subsequent Font Load action.

    LoadedFont$=""
    Err=0
    Ln=0
    TokPtr=0
    TokStrt=0
    F_Strt=0
    F_Attr=0
    F_NbrChars=0
    F_StrtChar=0
    F_Width=0
    F_Height=0
    F_PixelsPerChar=0
    F_PixelCount=0
    CharIdx=0                                                   'starting index to FontChars(..) array
    QWordByteIdx=7

  End Sub

  Sub LoadFontFile()
    'controls the reading of the Font File data and creating the FontChars() array.
    If DEBUG=2 Then PRINT "LoadFontFile() - FontFile=" + FontFile$

    Local Done
    Local ErrMsg1$,ErrMsg2$,ErrMsg3$
    
    ErrMsg3$="The program will be terminated."
    
    ON ERROR SKIP 1
    OPEN FontFile$ FOR INPUT AS #1
      
    If MM.ERRNO<>0 THEN
      'show the Error Message Box which will wait for the 'OK' button, then terminate the program. 
      ShowFileError("SDCard Access Error -", MM.ERRMSG$) 
      ON ERROR CLEAR 
      Exit Sub
    EndIf
    
    ClearFontLoadVars()                                         'clear global variables

    Do
      Line Input #1, Txt$
      Ln = Ln + 1

      If DEBUG = 1 Then
        PRINT "Line #" + STR$(Ln,3) + ": " + Txt$
      EndIf

      TxtLen = Len(Txt$)
      TokPtr = 1
      EOL = 0

      Do
        Token$ = GetToken$()

        If Token$ <> "" Then                                    'not empty Token - check content
          If F_Strt = 0 Then
            If Token$ = "DEFINEFONT" Then
              F_Strt = 1                                        'found DefineFont keyword
              F_Attr = 0                                        'need to process Font Attributes Word
              EOL = 1                                           'ignore remainder of this line
            EndIf

          ElseIf F_Attr = 0 Then
            If Not ValidToken() Then
              ErrMsg1$="Error processing Font Attributes -"
              If Err=1 Then
                ErrMsg2$="DWord is not 8 digits long."
              ElseIf Err=2 Then
                ErrMsg2$= "Current Token (" + Token$ + ") is not a valid DWord."
              EndIf
              ErrorMsg(Hres\2,Vres\2,ErrMsg1$,ErrMsg2$,ErrMsg3$)
              Exit Sub
               
            Else
              GetFontAttributes()
              F_Attr=1                                          'Font Attributes processed
              If DEBUG = 1 Then
                PRINT "Font (" + FontFile$ + ")  Attributes:"
                PRINT "     Nbr Chars = " + STR$(F_NbrChars) + " (" + Hex$(F_NbrChars,2) + ")"
                PRINT "    Start Char = " + STR$(F_StrtChar) + " (" + Hex$(F_StrtChar,2) + ")"
                PRINT "    Font Width = " + STR$(F_Width)    + " (" + Hex$(F_Width,2) + ")"
                PRINT "   Font Height = " + STR$(F_Height)   + " (" + Hex$(F_Height,2) + ")"
              EndIf
     
              DeclareFontCharsArray()  
            EndIf

          Else
            If Token$ = "END" Then                              'end of Font Definition
              If F_PixelCount => F_PixelsPerChar Then
                'need to save last QWord for the last character in the font file
                FontChars(CharIdx,QWordIdx) = QWord

              ElseIf F_PixelCount > 0 Then 
                If CharIdx <= MaxCharIdx Then
                  'oops - partial character definition at the end of the font file
                  ErrMsg1$="Error processing Font Dword -"
                  ErrMsg2$="Partial character at end of Font Definition"
                  ErrMsg2$=ErrMsg2$ + "Extra Pixel Count = " + Str$(F_PixelCount)
                  ErrorMsg(Hres\2,Vres\2,ErrMsg1$,ErrMsg2$,ErrMsg3$)
                  Exit Sub
                EndIf
              EndIf
              
              If DEBUG = 1 Then PRINT "End of Font Definition."
              Done = 1                                          'exit processing now

            ElseIf Not ValidToken() Then
              ErrMsg1$="Error processing Font Dword -"
              If Err=1 Then
                ErrMsg2$="DWord is not 8 digits long."
              ElseIf Err=2 Then
                ErrMsg2$= "Current Token (" + Token$ + ") is not a valid DWord."
              EndIf
              ErrorMsg(Hres\2,Vres\2,ErrMsg1$,ErrMsg2$,ErrMsg3$)
              Exit Sub

            Else
              BuildFontCharsArray()                             'add token data to FontChars array
            EndIf
          EndIf
        
        Else
          'reached end of current line (or start of comment text) 
          '- need to read next line from file
          EOL = 1
        EndIf

      Loop Until EOL OR Done

      If NOT Done Then
        If EOF(#1) Then                                         'reached end of Font File
          If DEBUG = 1 Then PRINT "End Of File: LOC=" + STR$(LOC(#1)) + ", LOF=" + STR$(LOF(#1))
          Exit Do                                               'exit program now
        EndIf
      EndIf

    Loop Until Done

    'got here so load was successful - save loaded font name
    LoadedFont$ = FontFile$

    State=40                                                    'move to next action - show Char Map for Font
    
  End Sub

  Sub InitQWordByteMask()
    'initializes the 8 entries in the QWordByteMask array to contain the correct mask
    'values for inserting a byte (8 bits) into any of the 8 byte positions within the
    '64-bit QWord used to store the pixel representation for each font character.

    QWordByteMask(0) = &hFFFFFFFFFFFFFF00
    QWordByteMask(1) = &hFFFFFFFFFFFF00FF
    QWordByteMask(2) = &hFFFFFFFFFF00FFFF
    QWordByteMask(3) = &hFFFFFFFF00FFFFFF
    QWordByteMask(4) = &hFFFFFF00FFFFFFFF
    QWordByteMask(5) = &hFFFF00FFFFFFFFFF
    QWordByteMask(6) = &hFF00FFFFFFFFFFFF
    QWordByteMask(7) = &h00FFFFFFFFFFFFFF

    If DEBUG > 1 Then
      Local Idx
      For Idx = 0 To 7
        PRINT Hex$(QWordByteMask(Idx),16)
      Next Idx
    EndIf

  End Sub

  Sub DeclareFontCharsArray()
    'uses the Font Attributes to declare an integer array of suitable dimensions to
    'store the pixel definition for each character included in the font.
    'Each font character uses 1 or more 64-bit integers (QWords) to store the pixel 
    'representation. Not all bits of the last QWord for each character will necessarily 
    'be used. 

    Local CharPixels, Q

    F_PixelsPerChar = F_Width * F_Height                        '# of pixels in each char
    Q = F_PixelsPerChar \ 64                                    '# of whole QWords reqd
    If (F_PixelsPerChar MOD 64) > 0 Then Q = Q +1               'additional QWord reqd

    'Erase the dummy FontChars array and re-declare it
    'using the correct sizes for each dimension.
    On Error Skip 1                                             'in case the array doesn't already exist
    Erase FontChars()
 
    'calculate Max array index value for each array dimension
    MaxCharIdx = F_NbrChars - 1                                 'array BASE is 0
    MaxQWordIdx = Q - 1                                         'array BASE is 0

    'Declare the new array instance - NOTE: ensure the size of the second dimension
    '(QWordIdx) is a minimum of 1 (actually 2 - index 0 and 1) as attempting to use
    '0 (i.e. just index 0) for this value will result in a run-time error!! 
    Dim FontChars(MaxCharIdx,MAX(MaxQWordIdx,1))                'declare new array instance

    FirstCharASCIIVal = F_StrtChar
    LastCharASCIIVal = (FirstCharASCIIVal + F_NbrChars) - 1
    QWordIdx=0

  End Sub

  Sub BuildFontCharsArray()
    'uses the data from the current DWord to build QWord entries in the FontChars array.
    
    'NOTE: QWords are built from the MSB end - i.e. the first DWord Byte to be added to a QWord
    '===== will use QWordByteIdx = 7, and the 8th Byte added (if there is one) would use QWordByteIdx = 0.
    '      Consequently when the QWordByteIdx is "reset" it will be set to value 7 not 0.    

    Local Byte,ByteCount
    
    Do
      If F_PixelCount => F_PixelsPerChar Then
        'previous character has been fully loaded, save final QWord of prev char
        FontChars(CharIdx,QWordIdx) = QWord

        'then increment CharIdx to next character 
        CharIdx = CharIdx + 1
        'reset PixelCount, QWordIdx, QWord (all to 0) and QWordByteIdx (to 7 - see above for explanation)
        F_PixelCount = 0
        QWordIdx = 0
        QWord = 0
        QWordByteIdx = 7
      Else
        'continuing the current character definition,
        'check if current QWord is full.
        If QWordByteIdx < 0 Then
          'need to move to next QWord, save current QWord first
          FontChars(CharIdx,QWordIdx) = QWord

          'increment to next QWordIdx, reset QWordByteIdx (to 7) and clear QWord
          QWordIdx = QWordIdx + 1
          QWordByteIdx = 7
          QWord = 0
        EndIf
      EndIf

      'extract the current ByteCount Byte from the DWord.
      Byte = (DWord >> (ByteCount *8)) AND &hFF

      'manipulate the Byte value then insert it at the appropriate Byte position in the QWord
      Byte = Byte << (QWordByteIdx * 8)
      QWord = (QWord AND QWordByteMask(QWordByteIdx)) OR Byte

      'increment pointers and counters
      F_PixelCount = F_PixelCount + 8                           'added 8 pixel bits to QWord
      ByteCount = ByteCount + 1                                 'increment ByteCount.
      
      'decrement QWordByteIdx
      QWordByteIdx = QWordByteIdx - 1                           'this one counts DOWN !!
    Loop Until ByteCount => 4

  End Sub

  Sub GetFontAttributes()
    'parses the Font Attributes DWord to extract the 4 Attribute values

    F_NbrChars = Val("&h" + Left$(Token$,2))
    F_StrtChar = Val("&h" + Mid$(Token$,3,2))
    F_Height   = Val("&h" + Mid$(Token$,5,2))
    F_Width    = Val("&h" + Mid$(Token$,7))

  End Sub

  Function ValidToken()
    'verfies that Token$ contains a valid 32-bit DWord.
    'returns 0 - invalid and sets Err to error number,
    '        1 - valid

    ValidToken = 0                                              'default return value (invalid)

    If Len(Token$) <> 8 Then
      'Token$ MUST contain 8 hexadecimal digits to be valid
      Err = 1

    Else
      On Error Skip 1
      DWord = Val("&h" + Token$)                                'get value of Token

      If MM.ErrNo = 0 Then
        ValidToken = 1
      Else
        Err = 2
        On Error Clear
      EndIf
    EndIf

  End Function  

  Function GetToken$()
    'return the first token (word, number or comment char) in the Txt$ string
    'returns an empty string if Txt$ is blank, there is no further characters in the string,
    'or the string (whole or remainder) is a comment.

    GetToken$ = ""                                              'empty string - default return value
    SkipSpaces()

    If TokPtr = -1 Then Exit Function                           'string or rest of string is blank
    If Mid$(Txt$,TokPtr,1) = "'" Then Exit Function             'string or rest of string is a comment

    TokStrt = TokPtr
    SkipNonSpaces()
    GetToken$ = UCase$(Mid$(Txt$,TokStrt,TokPtr-TokStrt))       'return token string

  End Function

  Sub SkipSpaces()
    'scans from the current position in the Txt$ string until the first non-space character
    'expects TokPtr and TxtLen to be setup on entry.
    'returns TokPtr updated to point to the next non-space char (or -1 if no more characters).

    Do
      If Mid$(Txt$,TokPtr,1) <> " " Then Exit Do
      TokPtr = TokPtr + 1
      If TokPtr > TxtLen Then TokPtr = -1
    Loop Until TokPtr = -1

  End Sub

  Sub SkipNonSpaces()
    'scans from the current position in the Txt$ string until the next space character 
    'or end of the string is found.
    'returns TokPtr updated to point to the character beyond the non-spaces string.

    Do
      If Mid$(Txt$,TokPtr,1) = " " Then Exit Sub
      TokPtr = TokPtr + 1
      If TokPtr > TxtLen Then Exit Sub
    Loop

  End Sub

'========================================================================================
'       D I S P L A Y    A S C I I    C H A R A C T E R S
'========================================================================================

  Sub DisplayASCIIChar(ASCIIVal,X,Y,Scale)
    'displays the specified ASCII character at the X, Y co-ordinates (Top Left).

    If ASCIIVal < FirstCharASCIIVal OR ASCIIVal > LastCharASCIIVal Then
      ErrMsg1$="Invalid character value - "
      ErrMsg2$="Char value ("+ Str$(ASCIIVal) + ") must be in the range "
      ErrMsg2$=ErrMsg2$ + Str$(FirstCharASCIIVal) + " to " + Str$(LastCharASCIIVal)
      ErrorMsg(Hres\2,Vres\2,ErrMsg1$,ErrMsg2$,ErrMsg3$)
      Exit Sub
    EndIf

    Local pIdx,pMaxIdx,pX,pXInc,pY,pYInc,pCol,pByte,pBit

    pMaxIdx = F_PixelsPerChar - 1
    pX = X
    pXInc=Scale
    pY = Y
    pYInc=Scale
    pCol = 0
    QWordIdx = -1
    QWordByteIdx = -1

    'calculate index into the Font array for the requested character
    CharIdx = ASCIIVal - FirstCharASCIIVal

    For pIdx = 0 To pMaxIdx
      If (pIdx MOD 8) = 0 Then
        'time to retrieve next byte from the QWord

        If QWordByteIdx =< 0 Then
          'need to get the next QWord
          QWordIdx = QWordIdx + 1
          QWord = FontChars(CharIdx,QWordIdx)
          'set QWordByteIdx for the new QWord (to 8)
          QWordByteIdx = 8
        EndIf

        'decrement QWordByteIdx to point to the next byte in the current (or new) QWord
        QWordByteIdx = QWordByteIdx - 1
        'retrieve the next byte from the QWord, then bit shift to the right so that
        'the byte occupies the Least Significant 8 bits within the pByte Integer
        pByte = QWord AND (&hFF << (QWordByteIdx * 8))
        pByte = pByte >> (QWordByteIdx * 8)
      EndIf
  
      'retrieve next bit from the current pByte.
      'Note: a single bit is retrieved from the MSB end and the remaining bits
      '----  shifted left (towards MSB end) by 1 bit position.
      '      The bit retrieved is then shifted right by 7 bit positions so that
      '      it occupies the LSB position within the pBit Integer.

      pBit = pByte AND &h80
      pByte = pByte << 1
      pBit = pBit >> 7

      'paint the next pixel in the character if pBit = 1
      If pBit = 1 Then
        If Scale = 1 Then                                       '1x Scaling
          Pixel pX, pY, FC%                                     'paint the single pixel

        Else                                                    '2x or 3x Scaling
          Box px, pY, Scale, Scale, 0, 0, FC%                   'paint the 2x2 or 3x3 box
        EndIf
      Endif

      pCol = pCol  + 1

      If pCol = F_Width Then
        'time to move to the next pixel row for the character
        pY = pY + pYInc                                         'account for Scaling Factor
        'reset pX back to the specified X position for the character
        pX = X
        'reset pCol back to 0
        pCol = 0

      Else
        'just move pX coordinate pXInc pixels to the right
        pX = pX + pXInc
      EndIf

    Next pIdx

  End Sub

  Sub DisplayFontAttr()
    'displays font file name and relevant font attributes
    Local Temp$

    Work$=FontFile$
    If Fnt=1 AND Len(FontFile$) > 22 Then Work$=Left$(FontFile$,22)
    Text 12,5,"Font File Name: " + Work$,LT,Fnt,1
    Text 12,(fH+8)," Starting Char: " + Chr$(F_StrtChar) + " (&H" + Hex$(F_StrtChar,2) + ")",LT,Fnt,1
    Text 12,((fH+3)*2)+5,"  Nbr of Chars: " + Str$(F_NbrChars),LT,Fnt,1
    Text 12,((fH+3)*3)+5,"     Font Size: " + Str$(F_Width) + " x " + Str$(F_Height) + " (W x H)",LT,Fnt,1

    If Fnt = 2 Then
      'build Scaling information display
      Temp$="Scaling Factor = " + Str$(ScaleFactor) + "x ("
      Temp$=Temp$+Str$(F_Width*ScaleFactor) + " x " 
      Temp$=Temp$+Str$(F_Height*ScaleFactor) + ")"
      Text (fW*35)+12,((fH+3)*3)+5,Temp$,LT,Fnt,1
    EndIf

  End Sub

  Sub DisplayCharMap()
    'displays the Character Map showing all characters available in the font

    Local c,cW,r,rH,rMax,rTop
    Local hlc,i,m,n,x,y
    Local Char

    CLS

    'display the font file name and attributes
    DisplayFontAttr()

    'setup button coordinates for the 'Close' Button
    If Fnt = 1 Then
      B_Coords(0,0)=(fW*27)+12                                  'Left - Font #1
      B_Coords(0,1)=fH+10                                       'Top
    Else
      B_Coords(0,0)=(fw*53)+12                                  'Left - Font #2
      B_Coords(0,1)=fH+10                                       'Top
    EndIf

    B_Coords(0,2)=B_Coords(0,0)+fW*9                            'Right
    B_Coords(0,3)=B_Coords(0,1)+fH+12                           'Bottom
    B_Coords(0,4)=1                                             'enabled (always)
    B_Coords(0,5)=0                                             'not used
    B_Coords(0,6)=B_Coords(0,0)+fW*2                            'Text Left
    B_Coords(0,7)=B_Coords(0,1)+6                               'Text Top
    B_Coords(0,8)=FCMSG%                                        'Forecolour
    B_Coords(0,9)=BCMSG%                                        'BackColour

    If Fnt = 2 Then
      'display Change Scaling button
      B_Coords(1,0)=(fW*35)+12                                  'Left
      B_Coords(1,1)=fH+10                                       'Top
      B_Coords(1,2)=B_Coords(1,0)+fW*16                         'Right
      B_Coords(1,3)=B_Coords(1,1)+fH+12                         'Bottom
      B_Coords(1,4)=1                                           'enabled (always)
      B_Coords(1,5)=0                                           'not used
      B_Coords(1,6)=B_Coords(1,0)+fW                            'Text Left
      B_Coords(1,7)=B_Coords(1,1)+6                             'Text Top
      B_Coords(1,8)=FCMSG%                                      'Forecolour
      B_Coords(1,9)=BCMSG%                                      'Backcolour
    EndIf

    'set Max Button Index for this screen
    If Fnt = 1 Then
      MaxButtonIdx=0                                            'max 1 button on this screen
    Else
      MaxButtonIdx=1                                            'max 2 buttons on this screen
    EndIf

    'calculate the grid line spacing for the font being displayed
    cW = (F_Width * ScaleFactor) + 1
    rH = (F_Height * ScaleFactor) + 1

    'calculate the number of rows needed
    m = FirstCharASCIIVal \ 16
    n = LastCharASCIIVal \ 16
    rMax = n - m

    'calculate the number of horizontal grid lines needed
    'this is 1 more than rMax as the bottom row needs top and bottom grid lines.
    hlc = rMax + 1

    'calculate Y position of top border of the grid
    rTop = ((fH+3)*4)+fH\2+5                                    '75 (Font #1) / 107 (Font #2)

    'draw the grid - rMax rows x 16 columns
    'draw horizontal lines first
    For i = 0 to hlc
      Line 12,((rH * i) + rTop),((cW * 16) + 12),((rH * i) + rTop),1,RGB(MAGENTA)
    Next i

    'next draw 17 vertical lines
    For i = 0 to 16
      Line ((cW * i) + 12),rTop,((cW * i) + 12),((rH * hlc) + rTop),1,RGB(MAGENTA)
    Next i

    'setup to insert the characters into the grid
    'calculate the starting character position in the first row of the grid,
    'based on the ASCII value of the starting character in the font.
    c = F_StrtChar MOD 16                                       'calculate column index
    x = (cW * c) + 13                                           'calc X coordinate
    r = 0                                                       'set starting row index
    y = rTop + 1                                                'set Y coordinate for top row text

    For CharIdx = 0 to MaxCharIdx
      'calc ASCII value of the character to display
      Char = FirstCharASCIIVal + CharIdx
      'display the character in the appropriate grid position
      DisplayASCIIChar(Char,x,y,ScaleFactor)

      'if not at the last character to be displayed
      If CharIdx < MaxCharIdx Then
        'increment column 
        c = c + 1
        'check if reached end of the row
        If c > 15 Then
          'need to move to the next row and reset column index
          r = r + 1
          c = 0
          Watchdog 40000                                        'to prevent possible Watchdog T/O 
        EndIf
      
        'calculate the X and Y coordinates for the next character
        x = ((cW * c) + 13)
        y = ((rH * r) + (rTop + 1))
      EndIf
    Next CharIdx
 
    'draw the 'Close' button (details setup earlier)
    'also the Change Scaling button when using Font #2 (800 x 480 res)
    For i=0 to MaxButtonIdx
      DrawButton(i,0)                                           'not presssed
    Next i

    State=45                                                    'wait for 'Close' button

  End Sub


  '===========================================================================================
  '     F O N T    S C A L I N G    F A C T O R    S E L E C T I O N
  '===========================================================================================

  Sub DisplayFontScalingSelect()
    'displays the Font Scaling Selection screen
    If DEBUG=2 Then PRINT "DisplayFontScalingSelection():"

    Local i,sX,sY,sW,sH
    Local Caption$

    'calculate the location and size of the screen
    Caption$="Select Font Scaling:"
    sW=fW*28                                                    'screen Width
    sH=210                                                      'screen Height
    sX=Hres\2 - sW\2                                            'screen Left
    sY=Vres\2 - sH\2                                            'screen Top

    'draw screen outline + filled box
    Box sX,sY,sW,sH,1,FCMSG%,BCMSG%

    'place screen caption and underline
    Text sX+fW*2,sY+20,Caption$,LT,Fnt,1,FCMSG%,BCMSG%
    Line sx+fW*2,sY+42,sX+(Len(Caption$)+2)*fW,sY+42,1,FCMSG%

    'place instruction text
    Text sx+fW*4,sY+50,"Touch required setting.",LT,Fnt,1,FCMSG%,BCMSG%

    'build Button Table for the Scaling Factor selections
                                                                'Button #0 - 1x Scaling
    B_Coords(0,0)=Hres\2-fW*6                                   'Left
    B_Coords(0,1)=sY+80                                         'Top
    B_Coords(0,2)=B_Coords(0,0)+fW*12                           'Right
    B_Coords(0,3)=B_Coords(0,1)+30                              'Bottom
    B_Coords(0,4)=1                                             'enabled (always)
    B_Coords(0,5)=0                                             'not used
    B_Coords(0,6)=B_Coords(0,0)+fW                              'Text Left
    B_Coords(0,7)=sY+85                                         'Text Top
    B_Coords(0,8)=FCMSG%                                        'Forecolour
    B_Coords(0,9)=BCMSG%                                        'Backcolour

                                                                'Button #1 - 2x Scaling
    B_Coords(1,0)=B_Coords(0,0)                                 'Left
    B_Coords(1,1)=sY+120                                        'Top
    B_Coords(1,2)=B_Coords(1,0)+fW*12                           'Right
    B_Coords(1,3)=B_Coords(1,1)+30                              'Bottom
    B_Coords(1,4)=1                                             'enabled (always)
    B_Coords(1,5)=0                                             'not used
    B_Coords(1,6)=B_Coords(1,0)+fW                              'Text Left
    B_Coords(1,7)=sY+125                                        'Text Top
    B_Coords(1,8)=FCMSG%                                        'Forecolour
    B_Coords(1,9)=BCMSG%                                        'Backcolour

                                                                'Button #2 - 3x Scaling
    B_Coords(2,0)=B_Coords(0,0)                                 'Left
    B_Coords(2,1)=sY+160                                        'Top
    B_Coords(2,2)=B_Coords(2,0)+fW*12                           'Right
    B_Coords(2,3)=B_Coords(2,1)+30                              'Bottom
    B_Coords(2,4)=1                                             'enabled (always)
    B_Coords(2,5)=0                                             'not used
    B_Coords(2,6)=B_Coords(2,0)+fW                              'Text Left
    B_Coords(2,7)=sY+165                                        'Text Top
    B_Coords(2,8)=FCMSG%                                        'Forecolour
    B_Coords(2,9)=BCMSG%                                        'Backcolour

    'Set Max Buttion Index for this screen
    MaxButtonIdx=2                                              '3 buttons on this screen

    'draw all buttons
    For i=0 to MaxButtonIdx
      DrawButton(i,0)                                           'draw buttons - not pressed
    Next i

    'set new State value
    State=55                                                    'waiting for selection

  End Sub

  '===========================================================================================
  '     F I L E    H A N D L I N G     F U N C T I O N S    &    S U B R O U T I N E S
  '===========================================================================================

  Function SDCardPresent()
    'checks whether the SD Card is configured and present
    '
    'Returns: 1 - Success
    '         0 - Failed (Error Message will be saved in ErrMsg$)
    
    LOCAL FN$          
    
    SDCardPresent=1                                             'default return value - success
    
    If UCase$(Left$(MM.DEVICE$,7)) = "ARMMITE" Then             'ARMmite models
      FN$ = MM.INFO$(SDCard)
      If UCase$(FN$) <> "READY" Then  
        SDCardPresent=0
        ErrMsg$=FN$
      EndIf  
        
    Else                                                        'MM+ or MMX
      ON ERROR SKIP 1
      FN$=DIR$("*", DIR)                                        'attempt to list directories
      If MM.ERRNO <> 0 Then                                     'error occurred
        SDCardPresent=0                                         'return value - failed
        ErrMsg$=MM.ERRMSG$                                      'save error msg
        ON ERROR CLEAR                                          'clear error indicators
      EndIf
    EndIf  
    
  End Function
  
  Sub ShowFileError(spErrMsg1$,spErrMsg2$)
    'displays a error message box to report a file handling related error
        
    State=80                                                    'Error MsgBox State

    MessageBox(Hres\2,Vres\2,"SDCard Error:",spErrMsg1$,spErrMsg2$,"The program will terminate.",FCERR%,BCERR%)
    
  End Sub
  
  Sub GetFileInfo() 
    'scans the current directory counting the number of Font files it contains
    'and determining the longest file name in the list.
    'Assumes current working directory is set to the FONTS directory
    
    LOCAL FN$,FNLen           
    
    FileCnt=0                                                   'reset file count
    FN_MaxLen=0                                                 'and maximum file name length
    
    FN$=Dir$("*.bas",file)                                      'start looking for font (.BAS) files
    
    Do While FN$ <> "" 
      FNLen=Len(FN$)                                            'get length of this file name
      FN_MaxLen=Max(FN_MaxLen,FNLen)                            'save max length so far
      FileCnt=FileCnt+1                                         'increment file count
      FN$=Dir$()                                                'get next file name from the list
    Loop
    
  End Sub
  
  Sub BuildFileList()
    'build a list of Font File Names
    LOCAL FN$,idx           

    'declare a tailored array sufficient to contain the full File List
    'NOTE:  This is a global declaration (DIM) as the array needs to be accessible across multiple subroutines
    '=====  It is therefore necessary to ERASE the old array before attempting to declare it again.
    '       The ERASE is done following an ON ERROR SKIP, so that if the array is not found
    '       the resulting error doesn't cause a crash!
    ON ERROR SKIP 1
    Erase flNames$()                                            'kill the old array, if it exists
    ON ERROR CLEAR
    
    DIM flNames$(Max(FileCnt,1)) Length Max(FN_MaxLen,1)        'declare new array for the File List
    
    'scans the current directory populating the list of Font (.BAS) files.   
    idx=0
    
    FN$=Dir$("*.bas",File)                                      'start looking for files
    Do While FN$ <> ""
      flNames$(idx)=FN$                                         'put file name into the array entry
      idx=idx+1                                                 'increment array entry pointer
      FN$=Dir$()                                                'get next file name from the list
    Loop
    
    flMaxIdx=idx-1                                              'max valid index into the file name array - will be
    '                                                           '-1 if NO files found matching the SearchParams$  
    SortStrArray(flNames$(),flMaxIdx)                           'sort file name list  
    
  End Sub
  

  '================================================================================================
  '     S T R I N G    A R R A Y    S O R T    R O U T I N E    
  '================================================================================================
  
  'Handles sorting the entries in a String Array into ascending alphanumeric order.  

  Sub SortStrArray(spArray$(),ipMaxIdx%)
    'Sorts the entries in the spArray$() array into ascending order (not case sensitive).
    'Assumes OPTION BASE 0 (the default setting).
    'ipMaxIdx specifies the maximum valid Index into the spArray$() array.
    'If ipMaxIdx is less than 1, the subroutine returns immediately as there is
    'either 0 or only 1 entry in the array and therefore doesn't need sorting.
    
    If ipMaxIdx < 1 Then EXIT Sub
    
    LOCAL Integer Idx,Sorted
    LOCAL Temp$
    
    Sorted=0
    
    DO WHILE Sorted=0
      Sorted=1
      
      FOR Idx=0 TO (ipMaxIdx-1)
        If UCASE$(spArray$(Idx)) > UCASE$(spArray$(Idx+1)) Then
          Temp$=spArray$(Idx+1)
          spArray$(Idx+1)=spArray$(Idx)
          spArray$(Idx)=Temp$
          Sorted=0
        EndIf
      NEXT Idx
      
    LOOP
    
  End Sub


  '===========================================================================================
  '     G E N E R A L    P U R P O S E     F U N C T I O N S    &    S U B R O U T I N E S
  '===========================================================================================

  Sub MessageBox(Xc,Yc,spCaption$,spMsg1$,spMsg2$,spMsg3$,msgFC,msgBC)
    'display the specified message(s)
    
    Local H,Ln,X,Y,W,Txt$
    Local i,LCap,L1,L1a,L1b,L2,L2a,L2b,L3,Wc
    Local Msg1a$,Msg1b$,Msg2a$,Msg2b$

    If DEBUG=2 Then 
      PRINT "MessageBox:"

    ElseIf DEBUG=1 Then
      Print "MessageBox: Caption$=" + spCaption$ 
      Print "            Msg1$=" + spMsg1$
      If Len(spMsg2$) > 0 Then PRINT "            Msg2$=" + spMsg2$
      If Len(spMsg3$) > 0 Then PRINT "            Msg3$=" + spMsg3$
    EndIf

    'get lengths of the argument strings
    LCap=Len(spCaption$)
    L1=Len(spMsg1$)
    L2=Len(spMsg2$)
    L3=Len(spMsg3$)

    If Fnt=1 Then
      'ensure the Msg Box does not exceed the screen width for the 320x240 screens
      'and the # of lines required.
      Ln=1                                                      '1 line reqd for Caption

      If L1 > 35 Then
        'need to split this message into 2 parts.
        'scan in reverse from char pos 35 until previous space (word separator).
        For i=35 To 1 Step -1
          If Mid$(spMsg1$,i,1) = " " Then Exit For              'found previous space
        Next i

        'split msg into 2 parts on word boundary pointed to by i
        L1a=i-1
        L1b=L1-L1a
        Msg1a$=Left$(spMsg1$,L1a)
        Msg1b$=Mid$(spMsg1$,(i+1))
        L1=0                                                    'clear L1 - msg has been split
        Ln=Ln+2

      Else
        Ln=Ln+1
      EndIf
 
      'calculate reqd width (in chars) to this point
      Wc=Max(LCap,L1,L1a,L1b)

      If L2 > 35 Then
        'need to split this message into 2 parts.
        'scan in reverse from char pos 35 until previous space (word separator).
        For i=35 To 1 Step -1
          If Mid$(spMsg2$,i,1) = " " Then Exit For              'found previous space
        Next i

        'split msg into 2 parts on word boundary pointed to by i
        L2a=i-1
        L2b=L2-L2a
        Msg2a$=Left$(spMsg2$,L2a)
        Msg2b$=Mid$(spMsg2$,(i+1))
        L2=0                                                    'clear L2 - msg has been split
        Ln=Ln+2

      Else
        Ln=Ln+1
      EndIf

      'calculate final reqd width (in chars and in pixels)
      Wc=Max(Wc,L2,L2a,L2b,L3)
      W=fW*(Wc+4)
    
      'calculate # of lines required - check if spMsg3$ is used
      If L3 > 0 Then Ln=Ln+1 

    Else   'for Font #2 (600x480 screen)
      'calculate # of lines reqd
      If spMsg2$="" Then Ln=2 Else Ln=3
      If spMsg3$<>"" Then Ln=Ln+1

      'calculate total width (in pixels)
      W=fW*(Max(Len(spCaption$),Len(spMsg1$),Len(spMsg2$),Len(spMsg3$))+4)
    EndIf

    H=((fH+3)*Ln)+(fH*3)+10
    X=Xc-(W\2)
    Y=Yc-(H\2)
    
    TxtX=X+(fW*2)
    TxtY=Y+(fH\2)
    
    RBox X,Y,W,H,10,msgFC,msgBC
    
    Txt$=spCaption$
    
    For Ln=0 to 5
      If Txt$ <> "" Then Text TxtX,TxtY,Txt$,LT,,1,msgFC,msgBC
      
      Txt$=""
      
      SELECT CASE Ln
        CASE 0
          TxtY=TxtY+(fH+2)
          Line TxtX,TxtY,TxtX+(fW*LCap),TxtY,1,msgFC
          TxtY=TxtY+8
          If L1 > 0 Then
            Txt$=spMsg1$
          Else
            Txt$=Msg1a$
          EndIf
        CASE 1
          If L1b > 0 Then
            TxtY=TxtY+(fH+3)
            Txt$=Msg1b$
          EndIf 
        CASE 2
          If L2 > 0 Then
            TxtY=TxtY+(fH+3)
            Txt$=spMsg2$
          ElseIf L2a > 0 Then
            TxtY=TxtY+(fH+3)
            Txt$=Msg2a$
          EndIf
        CASE 3
          If L2b > 0 Then
            TxtY=TxtY+(fH+3)
            Txt$=Msg2b$
          EndIf
        CASE 4  
          If L3 > 0 Then
            TxtY=TxtY+(fH+3)
            Txt$=spMsg3$
          EndIf 
      END SELECT
    Next Ln
    
    'draw the 'OK' button - centered below the text
    'setup button coords for this button - this also gives the drawing coords
    B_Coords(0,0)=Xc-fW*3                                       'Left
    B_Coords(0,1)=TxtY+fH+5                                     'Top
    B_Coords(0,2)=Xc+fW*3                                       'Right
    B_Coords(0,3)=B_Coords(0,1)+fH*2                            'Bottom
    B_Coords(0,4)=1                                             'Enabled
    B_Coords(0,6)=Xc-fW                                         'Text Left
    B_Coords(0,7)=B_Coords(0,1)+fH\2                            'Text Top
    
    'set the 'normal' Forecolour and Backcolour values for the 'OK' button
    'based on the current program State value.
    'NOTE: The button colours are reversed compared to the message box itself -
    '===== i.e. button Forecolour is MsgBox Backcolour and Button Backcolour is MsgBox Forecolour.
    
    If State = 80 Then                                          'ERROR message box
      B_Coords(0,8)=BCERR%                                      'button Forecolour
      B_Coords(0,9)=FCERR%                                      'button Backcolour 
    Else                                                        'INFO message box
      B_Coords(0,8)=BCMSG%                                      'button Forecolour 
      B_Coords(0,9)=FCMSG%                                      'button Backcolour 
    EndIf
    
    'set max button index value - single button only
    MaxButtonIdx = 0
    'draw the button, not pressed
    DrawButton(0,0)                                     
    
  End Sub
  
  Sub ErrorMsg(Xc,Yc,spErr1$,spErr2$,spErr3$)
    'display the specified error messages
    
    State=80                                                    'error message box
    MessageBox(Xc,Yc,"ERROR:",spErr1$,spErr2$,spErr3$,FCERR%,BCERR%)
    
  End Sub

  Sub InfoMsgBox(spInfo1$,spInfo2$,ipFatal)
    'display the specified Info message.
    
    If ipFatal Then
      State = 75
      Work$ = "The program will terminate."
    Else
      State = 70                                                'Info Message Box
      Work$ = ""
    Endif

    MessageBox(Hres\2,Vres\2,"ATTENTION:",spInfo1$,spInfo2$,Work$,FCMSG%,BCMSG%)

  End Sub

  '===========================================================================================
  '     F O N T    S P E C I F I C A T I O N S
  '===========================================================================================

' PE_Symbol1.bas
' Font type    : Sub-set (2 characters)
' Font start   : CHR$(1)
' Font size    : 28x22 pixels
' Memory usage : 158 bytes

DefineFont #9
  0201161C 00000400 0000E000 0300001B 60000018 060C00C0 30800100 80013000
  000C0406 1860E0C0 0303031B 60601818 060CCCC0 30800106 80013000 000C0006
  186000C0 00030300 00601800 0000CC00 0000C006 0C000066 81010030 0C300080
  60000600 0003C000 03180018 60C0C000 0C0C0666 81813130 0C30B081 6000060E
  0003C040 03180018 60C00000 000C0600 00803100 0000B001 0000000E 00400000
End DefineFont
