option explicit

font 7

mode 1,16

const tileSize=80
const RNGseed=15

dim gridX,gridY,tileNo as integer
dim originX,originY as integer
dim sx,sy,x,y,i,f,t as integer
dim roomOffset as integer

'32 different "room" types (0-31)
dim room$(31)

'tile vals holds every nth calculated tileVal RNG
'to speed up tile calcs in game
const storedVals=200
const skip=300
dim tileVals(storedVals) as integer
tileVals(0)=getTileVal(0)
for f=1 to storedVals
  tileVals(f)=getRelTile(skip,tileVals(f-1))
next f

dim prevTile as integer =999999
dim prevTileVal as integer
dim tileVal,reltile as integer
dim offsetX,offsetY as integer =0
dim playerX,playerY as integer =0
dim playerScreenX,playerScreenY
dim useLights=1
'max light dist is in pixels
dim maxLightDist=96

'for future implementation
dim monster,chest,key as integer

dim tileBin$
dim roomString$,roomStringE$,roomStringS$

'room code is 0-31 and is the top 5 bits of the 20-bit Lehmer tile code
dim roomCode

'some colours
const solid=rgb(80,20,20)
const floor=rgb(128,128,128)
const wall=rgb(white)
const door=rgb(128,50,50)

'pre-calc inverse square lighting values to speed up lighting in game
dim solidColour(10) as integer
dim floorColour(10) as integer
dim wallColour(10) as integer
dim doorColour(10) as integer

for f=0 to 10
  solidColour(f)=getLitColour(solid,1-f/10)
  floorColour(f)=getLitColour(floor,1-f/10)
  wallColour(f)=getLitColour(wall,1-f/10)
  doorColour(f)=getLitColour(door,1-f/10)
next f

'labyrinth grid is dran in the top left of the screen
originX=mm.hres/2-80
originY=mm.vres/2-50

'init room types
initRooms

drawWholeScreen

do

  'get a keypress from the player

  i=getKey()

  'parse the keypress
  if i=130 then
    if checkBlocked("W") then goto skip_moves:
    blankPlayer
    inc playerX,-1
    if playerX<-2 then 
      playerX=2
      inc offsetX,-1
      if not uselights then drawLeft
    endif
    if uselights then drawWholeScreen
  endif

  if i=131 then 
    if checkBlocked("E") then goto skip_moves:
    blankPlayer
    inc playerX
    if playerX>2 then
      playerX=-2
      inc offsetX
      if not uselights then drawRight
    endif
    if uselights then drawWholeScreen
  endif

  if i=128 then
    if checkBlocked("N") then goto skip_moves:
    blankPlayer
    inc playerY,-1
    if playerY<-2 then 
      playerY=2
      inc offsetY,-1
      if not uselights then drawTop
    endif
    if uselights then drawWholeScreen
  endif

  if i=129 then
    if checkBlocked("S") then goto skip_moves:
    blankPlayer
    inc playerY
    if playerY>2 then
      playerY=-2
      inc offsetY
      if not uselights then drawBottom
    endif
    if uselights then drawWholeScreen
  endif

skip_moves:

  if i=asc("=") then
    inc maxLightDist,8
    if maxLightDist>200 then maxLightDist=200
    drawWholeScreen
  endif

  if i=asc("-") then
    inc maxLightDist,-8
    if maxLightDist<1 then maxLightDist=1
    drawWholeScreen
  endif

  playerScreenX=originX+40+(playerX*16*not uselights)
  playerScreenY=originY+40+(playerY*16*not uselights)

  if i=asc("l") then
    if useLights=1 then useLights=0 else useLights=1
    drawWholeScreen
  endif

  'put the player on the screen
  circle playerScreenX,playerScreenY,4,0,,0,rgb(yellow)

  'clamp x and y to not exceed 62,500 tile limit
  if offsetX<-125 then offsetX=-125
  if offsetX>125 then offsetX=125
  if offsetY<-125 then offsetY=-125
  if offsetY>125 then offsetY=125
  
loop

end


'send either "N", "E", "S" or "W" to this routine
'to see if the way is blocked in that direction
function checkBlocked(dir$)
  local checkString$,checkRow,checkCol,checkChar$
  'start by assuming the way is not blocked
  checkBlocked=0
  'start by assuming the square to check is on this room tile
  checkString$=roomString$
  'start by assuming checkRow/checkCol are in line with player
  checkRow=playerY
  checkCol=playerX  

  if dir$="S" then
    checkRow=playerY+1
    if playerY=2 then checkString$=roomStringS$ : checkRow=-2
  endif
  if dir$="E" then
    checkCol=playerX+1
    if playerX=2 then checkString$=roomStringE$ : checkCol=-2
  endif

  checkchar$=mid$(checkString$,getCharPos(checkCol,checkRow),1)

  if asc(checkChar$)>asc("D") then
    checkChar$=chr$(asc(checkChar$)-21)
  endif
  
  if dir$="N" or dir$="S" then
    if checkChar$="2" or checkChar$="4" or checkChar$="8" or checkChar$="9" or checkChar$="B" then 
      checkBlocked=1
    endif
  endif
  
  if (dir$="E" or dir$="W") then
    if checkChar$="3" or checkChar$="4" or checkChar$="7" or checkChar$="A" or checkChar$="B" then
      checkBlocked=1
    endif
  endif
  
  
end function


function getCharPos(px,py)
  getCharPos=13+px+py*5
end function

sub blankPlayer
  if uselights=0 then circle playerScreenX,playerScreenY,4,0,,0,floor
end sub


'NON-LIGHTING DRAWING ROUTINES
'blit the left 7 tiles one to the right
'and draw in the new left hand column
sub drawLeft
  local x,y
  blit 0,10,80,10,560,560
  box 0,10,80,560,0,0,solid
  x=-4+offsetX
  for y=-3+offsetY to 3+offsetY
    drawTile(x,y)
  next y
    
end sub

'blit the right 7 tiles one to the left
'and draw in the new right hand column
sub drawRight
  local x,y
  blit 80,10,0,10,560,560
  box 560,10,80,560,0,0,solid
  x=3+offsetX
  for y=-3+offsetY to 3+offsetY
    drawTile(x,y)
  next y
end sub

'blit the top 6 tiles one down
'and draw in the new top row
sub drawTop
  local x,y
  blit 0,10,0,90,640,480
  box 0,10,640,80,0,0,solid
  y=-3+offsetY
  for x=-4+offsetX to 3+offsetX
    drawTile(x,y)
  next x
end sub

'blit the bottom 6 tiles one row up
'and draw in the new bottom row
sub drawBottom
  local x,y
  blit 0,90,0,10,640,480
  box 0,490,640,80,0,0,solid
  y=3+offsetY
  for x=-4+offsetX to 3+offsetX
    drawtile(x,y)
  next x
end sub


sub drawWholeScreen

  timer=0
  
  'draw dungeon "solid" background
  if uselights then 
    page write 1
    box 0,10,640,560,0,0,0
  else
    box 0,10,640,560,0,0,solid 
  endif    

  'draw whole dungeon map
  for y=-3+offsetY to 3+offsetY
    for x=-4+offsetX to 3+offsetX
      drawTile(x,y)
    next x
  next y

  'shift the map so the player is always dead centre of the labyrinth draw area
  if useLights then
    page scroll 1,playerX*-16,playerY*16,-1
    page copy 1 to 0
    page write 0
  endif

  text 650,0,"redraw time="+str$(timer)+"ms",,,,rgb(green)
  text 650,10,"Room tile X="+str$(offsetX)+" Y="+str$(offsetY)
  text 650,30,"Torch Dist="+str$(maxLightDist)
  text 650,40,"- / + to change"


end sub


sub drawTile(x,y)
  local lightDist

  'update the player's screen position
  playerScreenX=originX+40+playerX*16
  playerScreenY=originY+40+playerY*16

  'get the screen origin (top left) of this 5x5 tile
  sX=screenX(x) : sY=screenY(y)

  'if it's out of reach of the light, ignore it
  if useLights then
    lightDist=getDist2Player(sX,sY)
    if lightDist>maxLightDist+80 then goto unlit_tile:
  endif

  'get the Lehmer procedural tile no from the (x,y)
  tileNo=getTileNo(x,y)

  'get highest previously calculated tile value
  if (tileNo-prevTile)>0 and (tileNo-prevTile)<(tileNo mod skip) then
    relTile=tileNo-prevTile
    goto calc_tileVal:
  endif

  'use a pre-calculated Lehmer tile value from the tileVals() array  
  if tileNo>skip then 
    t=int(tileNo/skip)
    prevtile=t*skip
    prevTileVal=tileVals(t)
    relTile=tileNo-prevTile
    goto calc_tileVal:
  endif

  relTile=tileNo
  prevTileVal=tileVals(0)

calc_tileVal:
  'calculated tile value
  tileVal=getRelTile(relTile,prevTileVal)  

  'get 20-bit binary code of current Lehmer tile value
  tileBin$=bin$(tileVal,20)
  roomCode=getRoomCode(tileBin$)

  'draw this room
  drawRoom(sX,sY,roomCode)

  'make a note of the room string of the tile the player is on
  if x-offsetX=0 and y-offsetY=0 then
    roomString$=room$(roomCode)
  endif

  'make a note of the room string of the tile to the east of the player
  if x-offSetX=1 and y-offSetY=0 then
    roomStringE$=room$(roomCode)
  endif

  'make a note of the room string of the tile to the south of the player
  if x-offsetX=0 and y-offsetY=1 then
    roomStringS$=room$(roomCode)
  endif

  prevTile=tileNo
  prevTileVal=tileVal

unlit_tile:

end sub

'wait for user to press a key and then let go of it
function getKey()

  getKey=keydown(1)

  do 
  loop while keydown(0)<>0

end function

'draw a room tile
sub drawRoom(posX,posY,code)
  local char$
  local x,y,subtile,wallThick,doorSize,doorWall
  local localX,localY,lightDist,light
  local litWallColour,litFloorColour,litDoorColour,litSolidcolour
  local monsterSquare
  subTile=16
  wallThick=2
  doorSize=6
  doorWall=6
  for y=0 to 4
    localY=posY+y*subTile
    for x=0 to 4
      localX=posX+x*subTile
      char$=mid$(room$(code),y*5+x+1,1)

      'assign unlit colours by default
      litFloorColour=floor
      litWallColour=wall
      litDoorColour=door

      if useLights then
        lightDist=getDist2Player(localX,localY)
        if lightDist>maxLightDist then goto dont_draw:

        light=lightDist/maxLightDist*10
        litWallColour=wallColour(light)
        litFloorColour=floorColour(light)
        litDoorColour=doorColour(light)
        litSolidColour=solidColour(light)
        'solid
        box localX,localY,subTile,subTile,0,0,litSolidColour
      endif

      if asc(char$)>asc("D") then
        monsterSquare=1
        char$=chr$(asc(char$)-21)
      endif

      'floor
      if char$<>"9" and char$<>"A" and char$<>"B" and char$<>"C" and char$<>"D" then
        box localX,localY,subtile,subTile,0,0,litFloorColour      
      endif
      
      'corner wall piece
      if char$="1" or char$="D" then
        box localX,localY,wallThick,wallThick,0,0,litWallColour
      endif

      'top wall
      if char$="2" or char$="9" or char$="4" or char$="8" or char$="B" then
        box localX,localY,subTile,wallThick,0,0,litWallColour
      endif

      'left wall
      if char$="3" or char$="4" or char$="7" or char$="A" or char$="B" then
        box localX,localY,wallThick,subTile,0,0,litWallColour
      endif

      'top door
      if char$="5" or char$="7" then
        box localX,localY,doorWall,wallThick,0,0,litWallColour
        box localX+doorWall,localY,doorSize,1,0,0,litDoorColour
        box localX+doorWall+doorSize,localY,doorWall,wallThick,0,0,litWallColour
      endif

      'left door
      if char$="6" or char$="8" then
        box localX,localY,wallThick,doorWall,0,0,litWallColour
        box localX,localY+doorWall,1,doorSize,0,0,litDoorColour
        box localX,localY+doorWall+doorSize,wallThick,doorWall,0,0,litWallColour
      endif

dont_draw:
    next x
  next y
end sub

'used to pre-calculate lit colours
function getLitColour(inputColour,amount)
  local red,green,blue, binary$
  'split the input colour into RGB (8 bits, 8 bits, 8 bits)
  binary$=bin$(inputColour,24)
  binary$=mid$(binary$,5,24)
  red=bin2dec(mid$(binary$,1,8))
  green=bin2dec(mid$(binary$,9,8))
  blue=bin2dec(mid$(binary$,17,8))

  red=red*amount*amount
  blue=blue*amount*amount
  green=green*amount*amount

  getLitColour=rgb(red,green,blue)  
end function

'how far away from the player (light source) is the currently drawn square?
function getDist2Player(x,y)
  local relX,relY
  relX=x-playerScreenX
  relY=y-playerScreenY
  getDist2Player=sqr(relX*relX+relY*relY)
end function

function getRoomCode(b$)
  local tot,l
  'b$ is the tile val, 20-bit binary number
  'room code is first 5 bits of the tile val
  getRoomCode=bin2dec(mid$(b$,1,5))
end function

function bin2dec(b$)
  local l,tot
  for l=0 to len(b$)-1
    if mid$(b$,len(b$)-l,1)="1" then tot=tot+2^l
  next l
  bin2dec=tot
end function

'these functions convert a grid x, y into a screen pixel position
function screenX(x)
  screenX=originX+tileSize*(x-offsetX)
end function

function screenY(y)
  screenY=originY+tileSize*(y-offsetY)
end function

'this fn returns the start tile number of the layer number supplied
function getLayerStart(n)
  if n=0 then getLayerStart=0 : goto skip_layer:
  local rootLayerStart
  rootLayerStart=2*(n-1)+1
  getLayerStart=rootLayerStart*rootLayerStart
skip_layer:
end function

'this fn returns the tile no for given x and y values
function getTileNo(x,y)
  'n is the layer number
  local n=abs(x)
  local layerStart
  if abs(y)>abs(x) then n=abs(y)
  layerStart=getLayerStart(n)
  if y=-n then getTileNo=layerStart+x+n : goto skip_tile:
  if x=n then getTileNo=layerStart+n*2+y+n : goto skip_tile:
  if y=n then getTileNo=layerStart+n*4-x+n : goto skip_tile:
  getTileNo=layerStart+n*6-y+n  
skip_tile:
end function

'this sub returns a Lehmer generated RNG for an absolute tile no.
function getTileVal(tileNo)
  local f, rand
  rand=RNGseed
  for f=0 to tileNo
    rand=nextRand(rand)
  next f
  getTileVal=rand
end function

'this function returns a Lehmer generated RNG for a relative tile increment
function getRelTile(incTile,currentTileVal)
  local f
  for f=1 to incTile
    currentTileVal=nextRand(currentTileVal)
  next f
  getRelTile=currentTileVal
end function

'this function returns a procedural RND in the range 1-1048261 (largest prime under 20 bits)
function nextRand(value)
  nextRand=value*8753 mod 1048261
end function



sub waitkey
  do 
  loop while inkey$<>""
  do 
  loop while inkey$=""
end sub


'this sub assigns values to the global vars gridX and gridY depending on supplied tile no.
sub getXY(tileNo)
  'n is the layer number
  local n
  local layerStart,layerOffset as integer
  n=fix((1+sqr(tileNo))/2)
  layerStart=getLayerstart(layer)
  layerOffset=tileNo-layerStart

  if layerOffset<n*2 then
    gridX=originX+(layerOffset-n)*tileSize
    gridY=originY-n*tileSize
    goto skip_xy:
  endif
  if layerOffset<n*4 then
    layerOffset=layerOffset-n*2
    gridX=originX+n*tileSize
    gridY=originY+(layerOffset-n)*tileSize
    goto skip_xy:
  endif
  if layerOffset<n*6 then
    layerOffset=layerOffset-n*4
    gridX=originX+(n-layerOffset)*tileSize
    gridY=originY+n*tileSize
    goto skip_xy:
  layerOffset=layerOffset-n*6
  gridX=originX-n*tileSize
  gridY=originY-(n-layerOffset)*tileSize
skip_xy:
end sub


sub initRooms
  local i$
  for f=0 to 31
    read i$
    room$(f)=i$
  next f
  
  'room layout data - must be 32 in total (codes 0-31)

  'one single large 5x5 room
  data "I212230000100003000030000"
  '3 rooms
  data "421423003052G603003030030"
  'crossroads with door
  data "B93B9AC3AC2G122B97B9AC3AC"
  data "B93B9AC3AC2G122B97B9AC3AC"
  'corner to corner corridors
  data "I21B93B9DC1A422BD3B9AC3AC"
  data "I21B93B9DC1A422BD3B9AC3AC"
  'boss room
  data "421223452316E063300332521"
  'jail cells
  data "44344333335515577377333H3"
  'vertical passage
  data "B93B9AC3ACAC3AIAC3A9AC3AC"
  data "B93B9AC3ACAC3AIAC3A9AC3AC"
  data "B93B9AC3ACAC3AIAC3A9AC3AC"
  'vertical passage with joining passages
  data "426B9HB3AC1A382BD3B9AC3AC"
  'horizontal passage
  data "B9999ACCCC22222B9999AC4AC"
  data "B9999ACCCC22222B9999AC4AC"
  data "B9999ACCCC22222B9999AC4AC"
  'horizontal passage with joining passages
  data "B9322AC99322225B97B9AC3AC"
  'mazey passages with 2x2 in centre
  data "421423426316E334221232240"
  'corner to corner corridors with 1x1 room in middle
  data "B93B9AC32A22M62B32B9A93AC"
  '4X3 room with antechambers
  data "4254B3003A600383007B7221A"
  't-junction going down
  data "B9999ACCCC22222B97B9AC3AC"
  'columns room
  data "4212231111111113111131111"
  't-junction to right
  data "B93B9AC3ACAC322AC3B9AC3AC"
  'circle room
  data "B412B41B321BD9332A41B321A"
'----23 TO HERE

  'horizontal passage with joining passages
  data "B9322AC99322225B97B9AC3AC"
  'mazey passages with 2x2 in centre
  data "421423426316E334221232240"
  data "421423426316E334221232240"
  'corner to corner corridors with 1x1 room in middle
  data "B93B9AC32A22M62B32B9A93AC"
  '4X3 room with antechambers
  data "4254B3003A600383007B7221A"
  't-junctin going down
  data "B9999ACCCC22222B97B9AC3AC"
  'columns room
  data "4212231111111113111131111"
  't-junction to right
  data "B93B9AC3ACAC322AC3B9AC3AC"
  'circle room
  data "B412B41B321BD9332A41B321A"


end sub

