' V1.0
' Monochrome Port of Adafruit graphics library with drivers for i2c SSD1306 display
'
' drawing primitives assume a graphics display where top left is point 0,0
'
' set screen orientation: sub setrot(x) ;0=normal,1=invert,2=rotate right,3-rotate left
' draw circle: sub dcirc(x0,y0,r) ;x,y coordinates of centre,radius
' draw filled circle: sub fcirc(x0,y0,r) ;x,y coordinates of centre,radius
' draw triangle:  sub dtri(x0,y0,x1,y1,	x2,y2) ;three coordinate pairs
' draw filled triangle sub ftri (x0,y0,x1,y1,	x2,y2) ;three coordinate pairs
' print string: sub pstring(xx0,yy0,text$,sz,orientation) ;x,y coordinates of top left of first character,text string,size,orientation
' draw line: sub dLine(xx0,yy0,xx1,yy1) ;x,y of start line,x,y of end line
' draw rectangle: sub dRect(x,y,w,h) ;x,y coordinate of top left,width,height
' draw filled rectangle: sub frect(x,y,w,h) ;x,y coordinate of top left,width,height
' clear rectangle: sub crect(x,y,w,h) ;x,y coordinate of top left,width,height
' draw rounded rectangle: sub drndRect(x,y,w,h,r) ;x,y coordinates of top left,width,height,radius of corner
' draw filled rounded rectangle: sub frndRect(x,y,w,h,r) ;x,y coordinates of top left,width,height,radius of corner
'
' These routines just update the memory image of the display
' Use the refresh command to write to the display as required
'
cpu 48
OPTION EXPLICIT
OPTION DEFAULT integer
'option autorun on
const normal=0
const inverse=1
const rt_right=2
const rt_left=3
const ON=&B1
const OFF=&B0
const pix=0
const ffr=1
const drawchar=2
const drawline=3
'
' SSD1306 Display specific constants
const S_wi=128
const S_ht=64
const S_nr=S_ht\8 'memory is arranged as rows 8 bits high
const S_i2caddr=&H3C
' Global variables
dim  wi,ht,nr,i2caddr,rot_ht,rot_wi,sz,rot,AddrOfFont
dim  sc(128),dd(128) 'set up arrays for screen map to mimic display
init:  
  print "Starting initialisation"  
  print "Using SSD1306"  
  initSdisplay 'initialise the display  
  AddrOfFont=PEEK(CFunAddr FONT)  ' 
  print "starting main"  
' Variables for use in main
  dim  i,xfs,yfs,x0=0,y0=0,x1,y1=63
  dim float tn
Main:  
' demo fast line draw  
  setrot(normal)
  S_clearsc  
  for i=0 to 45 step 5
    tn=tan(rad(i))
    x1=wi*tn
    dline(x0,y0,x1,y1)
  next i
  x1=127
  for i=0 to 40 step 5
    tn=tan(rad(i))
    y1=ht*tn
    dline(x0,y0,x1,y1)
  next i
  s_refresh
  pause 4000
  S_clearsc 
  s_refresh
'demo fast text write
  for i=0 to 56 step 8
    pstring(0,i,"Hello world:Bye world",1,normal)
  next i
  s_refresh
  pause 4000
  S_clearsc  
  for i=63 to 15 step -16
    pstring(127,i,"Micromite2",2,inverse)
  next i
  s_refresh
  pause 4000
  S_clearsc  
  for i=16 to 128 step 16
    pstring(i,0,"RIGHT",2,rt_right)
  next i
  s_refresh
  pause 4000
  S_clearsc  
  for i=104 to 0 step -24
    pstring(i,63,"LEFT",3,rt_left)
  next i
  s_refresh
  pause 4000
 S_clearsc  
  ' test drawing rectangles 
  S_clearsc  
  for i=0 to 3 
     setrot(i)  
     frect(5,5,10,15) 
     drect(2,2,16,21) 
  next i 
  s_refresh
  pause 2000 
' test drawing rounded rectangles 
  S_clearsc  
  for i=0 to 3 
     setrot(i)  
     frndrect(5,5,10,15,4) 
     drndrect(2,2,16,21,6) 
  next i 
  s_refresh
  pause 2000 
' test drawing circles 
  S_clearsc  
  for i=0 to 3 
     setrot(i)  
     fcirc(10,10,7) 
     dcirc(10,10,9) 
  next i 
  s_refresh
  pause 2000 
' test drawing triangles and line drawing 
  S_clearsc  
  for i=0 to 3 
     setrot(i)  
     ftri(4,4,20,6,12,14) 
     dtri(1,2,24,4,12,17) 
  next i 
  s_refresh
  pause 2000 
' test character output and string output at all orientations 
  for sz=1 to 3 
  S_clearsc  
  setrot(normal)   
  xfs=0 
  yfs=0 
  pstring(xfs,yfs,"Nr",sz,normal) 
  pstring(xfs,yfs,"g",sz,normal) 
  pstring(rot_wi-1,rot_ht-1,"Iv",sz,inverse) 
  pstring(rot_wi-1,0,"Rr",sz,rt_right) 
  pstring(0,rot_ht-1,"Rl",sz,rt_left) 
  s_refresh
  pause 2000 
   S_clearsc  
   setrot(inverse)  
  xfs=0 
  yfs=0 
  pstring(xfs,yfs,"Nr",sz,normal) 
  pstring(xfs,yfs,"g",sz,normal) 
  pstring(rot_wi-1,rot_ht-1,"Iv",sz,inverse) 
  pstring(rot_wi-1,0,"Rr",sz,rt_right) 
  pstring(0,rot_ht-1,"Rl",sz,rt_left) 
  s_refresh
  pause 2000 
   S_clearsc  
   setrot(rt_right)  
  xfs=0 
  yfs=0 
  pstring(xfs,yfs,"Nr",sz,normal) 
  pstring(xfs,yfs,"g",sz,normal) 
  pstring(rot_wi-1,rot_ht-1,"Iv",sz,inverse) 
  pstring(rot_wi-1,0,"Rr",sz,rt_right) 
  pstring(0,rot_ht-1,"Rl",sz,rt_left) 
  s_refresh
  pause 2000 
   S_clearsc  
   setrot(rt_left)  
  xfs=0 
  yfs=0 
  pstring(xfs,yfs,"Nr",sz,normal) 
  pstring(xfs,yfs,"g",sz,normal) 
  pstring(rot_wi-1,rot_ht-1,"Iv",sz,inverse) 
  pstring(rot_wi-1,0,"Rr",sz,rt_right) 
  pstring(0,rot_ht-1,"Rl",sz,rt_left) 
  s_refresh
  pause 2000 
  next sz  
  goto main  
end  
'
' SSD1306 Display specific routines
'
sub initSdisplay 'i2c oled
  local i 
  wi=S_wi 
  ht=S_ht 
  nr=S_nr 
  i2caddr=S_i2caddr 
  I2C OPEN 400,1000
  s_cmd(&HAE)'DISPLAYOFF)
  s_cmd(&HD5)'DISPLAYCLOCKDIV
  s_cmd(&H80)'the suggested ratio &H80
  s_cmd(&HA8)'MULTIPLEX
  s_cmd(&H3F)'
  s_cmd(&HD3)'DISPLAYOFFSET
  s_cmd(&H0)'no offset
  s_cmd(&H40)'STARTLINE
  s_cmd(&H8D)'CHARGEPUMP
  s_cmd(&H14)
  s_cmd(&H20)'MEMORYMODE
  s_cmd(&H00)'&H0 act like ks0108
  s_cmd(&HA1)'SEGREMAP OR 1
  s_cmd(&HC8)'COMSCANDEC
  s_cmd(&HDA)'COMPINS
  s_cmd(&H12)
  s_cmd(&H81)'SETCONTRAST
  s_cmd(&HCF)
  s_cmd(&Hd9)'SETPRECHARGE
  s_cmd(&HF1)
  s_cmd(&HDB)'VCOMDETECT
  s_cmd(&H40)
  s_cmd(&HA4)'DISPLAYALLON_RESUME
  s_cmd(&HA6)'NORMALDISPLAY
  s_cmd(&HAF)'DISPLAYON
  for i=0 to 127 'force the whole screen to be re-written at the next refresh
    dd(i)= -1'&HFFFFFFFFFFFFFFFF
    sc(i)=0
  next i
  s_refresh
end sub
'
Sub s_cmd(command )
  I2C WRITE i2caddr,0,2,&H00,command
end sub
 '
sub S_refresh
  local startrow,startcol,rst
  local outchar$ length 130
  S_cz(0,0)
  do 
    rst=Sgetnextupdate(outchar$,startcol,startrow,wi,nr,sc(),dd())
    if rst<>0 then
  S_cz(startcol,startrow)
  i2c write i2caddr,0,len(outchar$),outchar$
    endif
  loop while rst<>0
end sub
'
Sub S_Clearsc
  local i 
  for i=0 to 127
    sc(i)=0
  next i
  S_refresh
end sub
 '
sub S_cz(x ,y )
  s_cmd(&HB0+y)        'set page address
  s_cmd(&H10+(x>>4 and &H0F))  'set high col address
  s_cmd(&H00+(x AND &H0f))     'set low col address
end sub
'
' Generic drawing routines
'
sub dcirc(x0 ,y0 ,r ) 'x,y coordinates of centre,radius
  local  f=1-r,ddF_x=1,ddF_y=-2 * r,x=0,y=r,dp
  dp=draw(pix,sc(),ON,x0,y0+r,rot,wi,ht)
  dp=draw(pix,sc(),ON,x0,y0-r,rot,wi,ht)
  dp=draw(pix,sc(),ON,x0+r,y0,rot,wi,ht)
  dp=draw(pix,sc(),ON,x0-r,y0,rot,wi,ht)
  do while (x<y)
    if f>=0 then
  y=y-1
  ddF_y=ddF_y+2
  f=f+ddF_y
    endif
    x=x+1
    ddF_x=ddF_x+2
    f=f+ddF_x
    dp=draw(pix,sc(),ON,x0+x,y0+y,rot,wi,ht)
    dp=draw(pix,sc(),ON,x0-x,y0+y,rot,wi,ht)
    dp=draw(pix,sc(),ON,x0+x,y0-y,rot,wi,ht)
    dp=draw(pix,sc(),ON,x0-x,y0-y,rot,wi,ht)
    dp=draw(pix,sc(),ON,x0+y,y0+x,rot,wi,ht)
    dp=draw(pix,sc(),ON,x0-y,y0+x,rot,wi,ht)
    dp=draw(pix,sc(),ON,x0+y,y0-x,rot,wi,ht)
    dp=draw(pix,sc(),ON,x0-y,y0-x,rot,wi,ht)
  loop
end sub
 '
sub dtri(x0 ,y0 ,x1 ,y1 ,	x2 ,y2 ) 'three coordinate pairs
  dLine(x0,y0,x1,y1)
  dLine(x1,y1,x2,y2)
  dLine(x2,y2,x0,y0)
end sub
 '
sub ftri (x0 ,y0 ,x1 ,y1 ,	x2 ,y2 ) 'three coordinate pairs
  local  a,b,y,dp,last
  if (y0>y1) then
    swap(y0,y1)
    swap(x0,x1)
  endif
  if (y1>y2) then
    swap(y2,y1)
    swap(x2,x1)
  endif
  if (y0>y1) then
    swap(y0,y1)
    swap(x0,x1)
  endif
  if(y0=y2) then ' Handle awkward all-on-same-line case as its own thing
    a=x0
    b=x0
    if(x1<a) then
  a=x1
    else
  if(x1>b) then b=x1
    endif
    if(x2<a) then
  a=x2
    else
  if(x2>b) then b=x2
    endif
    dp=draw(ffr,sc(),ON,a,y0,1,b-a+1,rot,wi,ht) 
    exit sub
  endif
    local   dx01=x1-x0, dy01=y1-y0, dx02=x2-x0, dy02=y2-y0,dx12=x2-x1, dy12=y2-y1, sa=0,sb=0
  if(y1=y2) then
    last=y1 'Include y1 scanline
  else
     last=y1-1 ' Skip it
  endif
  for y=y0 to last
    a=x0+sa \ dy01
    b=x0+sb \ dy02
    sa=sa+dx01
    sb=sb+dx02
    a=x0+(x1-x0) * (y-y0) \ (y1-y0)
    b=x0+(x2-x0) * (y-y0) \ (y2-y0)
    if(a>b) then swap(a,b)
    dp=draw(ffr,sc(),ON,a,y,b-a+1,1,rot,wi,ht) 
  next y
  sa=dx12 * (y-y1)
  sb=dx02 * (y-y0)
  do while y<=y2
    a=x1+sa \ dy12
    b=x0+sb \ dy02
    sa=sa+dx12
    sb=sb+dx02
    a=x1+(x2-x1) * (y-y1) \ (y2-y1)
    b=x0+(x2-x0) * (y-y0) \ (y2-y0)
    if(a>b) then swap(a,b)
    dp=draw(ffr,sc(),ON,a,y,b-a+1,1,rot,wi,ht) 
    y=y+1
  loop
end sub
'
sub pstring(xx0 ,yy0 ,text$ as string,sz ,orientation ) 'x,y coordinates of top left of first character,text string,size,orientation
  local  i,c,x=xx0,y=yy0,chr0,dp
  for i=1 to len(text$)
    c=asc(mid$(text$,i,1))
    chr0=getchar(c) 'loads the character
    dp=draw(drawchar,sc(),chr0,x,y,sz,orientation,rot,wi,ht)
    on orientation+1 goto pnormal,pinvert,prtright,prtleft
pnormal:
    x=x+(6*sz)
    goto pcontinue
pinvert:
    x=x-(6*sz)
    goto pcontinue
prtright:
    y=y+(6*sz)
    goto pcontinue
prtleft:
    y=y-(6*sz)
pcontinue:
  next i
  xx0=x
  yy0=y
end sub
'
sub dLine(xx0 ,yy0 ,xx1 ,yy1 ) ' x,y of start line,x,y of end line
   local dp=draw(drawline,sc(),ON,xx0,yy0,xx1,yy1,rot,wi,ht)
end sub
 '
sub swap (a ,b )
  local  t
  t=b
  b=a
  a=t
end sub
 '
sub dRect(x ,y ,w ,h ) 'x,y coordinate of top left,wi,ht
  local  a,dp
  dp=draw(ffr,sc(),ON,x,y,w,1,rot,wi,ht) 
  a=y+h-1
  dp=draw(ffr,sc(),ON,x,a,w,1,rot,wi,ht) 
  dp=draw(ffr,sc(),ON,x,y,1,h,rot,wi,ht) 
  a=x+w-1
  dp=draw(ffr,sc(),ON,a,y,1,h,rot,wi,ht) 
end sub
'
sub frect(x ,y ,w ,h ) 'x,y coordinate of top left,width,height
  local  dp
  dp=draw(ffr,sc(),ON,x,y,w,h,rot,wi,ht) 
end sub
'
sub crect(x ,y ,w ,h ) 'x,y coordinate of top left,width,height
  local  dp
  dp=draw(ffr,sc(),OFF,x,y,w,h,rot,wi,ht) 
end sub
'
FUNCTION Getchar(char ) 
  local  i=char<<3
  Getchar=PEEK(WORD AddrOfFont+i)+(PEEK(WORD AddrOfFont+i+4)<<32) 
END FUNCTION
'
sub fcirc(x0 ,y0 ,r )  'x,y coordinates of centre,radius
  local  dp
  dp=draw(ffr,sc(),ON,x0,y0-r,1,2*r+1,rot,wi,ht)
  fcircH(x0,y0,r,3,0);
end sub
 '
sub drndRect(x ,y ,w ,h ,r ) 'x,y coordinates of top left,width,height,radius of corner
  local  dp
  dp=draw(ffr,sc(),ON,x+r,y,w-2*r,1,rot,wi,ht) 
  dp=draw(ffr,sc(),ON,x+r,y+h-1,w-2*r,1,rot,wi,ht) 
  dp=draw(ffr,sc(),ON,x,y+r,1,h-2*r,rot,wi,ht) 
  dp=draw(ffr,sc(),ON,x+w-1,y+r,1,h-2*r,rot,wi,ht) 
  dcircH(x+r,y+r,r,1)
  dcircH(x+w-r-1,y+r,r,2)
  dcircH(x+w-r-1,y+h-r-1,r,4)
  dcircH(x+r,y+h-r-1,r,8)
end sub
 '
sub dcircH( x0 ,y0 ,r ,cn )
  local  f=1-r,ddF_x=1,ddF_y=-2 * r,x=0,y=r,dp
  do while (x<y)
    if (f>=0) then
  y=y-1
  ddF_y=ddF_y+2
  f=f+ddF_y
    endif
    x=x+1
    ddF_x=ddF_x+2
    f=f+ddF_x
    if (cn and 4) then
  dp=draw(pix,sc(),ON,x0+x,y0+y,rot,wi,ht)
  dp=draw(pix,sc(),ON,x0+y,y0+x,rot,wi,ht)
    endif
    if (cn and 2) then
  dp=draw(pix,sc(),ON,x0+x,y0-y,rot,wi,ht)
  dp=draw(pix,sc(),ON,x0+y,y0-x,rot,wi,ht)
    endif
    if (cn and 8) then
  dp=draw(pix,sc(),ON,x0-y,y0+x,rot,wi,ht)
  dp=draw(pix,sc(),ON,x0-x,y0+y,rot,wi,ht)
    endif
    if (cn and 1) then
  dp=draw(pix,sc(),ON,x0-y,y0-x,rot,wi,ht)
  dp=draw(pix,sc(),ON,x0-x,y0-y,rot,wi,ht)
    endif
  loop
end sub
 '
sub fcircH(x0 ,y0 ,r ,cn ,delta )
  local  dp,f=1-r,ddF_x=1,ddF_y=-2 * r,x=0,y=r
  do while x<y
    if f>=0 then
  y=y-1
  ddF_y=ddF_y+2
  f=f+ddF_y
    endif
    x=x+1
    ddF_x=ddF_x+2
    f=f+ddF_x
    if (cn and 1) then
  dp=draw(ffr,sc(),ON,x0+x,y0-y,1,2*y+1+delta,rot,wi,ht) 
  dp=draw(ffr,sc(),ON,x0+y,y0-x,1,2*x+1+delta,rot,wi,ht) 
    endif
    if (cn and 2) then
  dp=draw(ffr,sc(),ON,x0-x,y0-y,1,2*y+1+delta,rot,wi,ht) 
  dp=draw(ffr,sc(),ON,x0-y,y0-x,1,2*x+1+delta,rot,wi,ht) 
    endif
  loop
end sub
'
sub frndRect(x ,y ,w ,h ,r ) 'x,y coordinates of top left,width,height,radius of corner
  local  dp
  dp=draw(ffr,sc(),ON,x+r,y,w-2*r,h,rot,wi,ht) 
  fcircH(x+w-r-1,y+r,r,1,h-2*r-1)
  fcircH(x+r,y+r,r,2,h-2*r-1)
end sub
 '
sub setrot(x ) 'sets screen rotation: 0=normal,1=invert,2=rotate right,3-rotate left
  rot=(x and 3)
  rot_wi=wi
  rot_ht=ht
  if x>=2 then
    rot_wi=ht
    rot_ht=wi
  endif
end sub
'
' CFunctions
'
CFunction Sgetnextupdate
	00000000
	27bdffa8 afb00054 8fb80068 8faf006c 8fae0070 24020040 a0820001 8f090000 
	1120000b 00001021 00094880 2403ffff 27aa0010 01424021 ad030000 27b00030 
	02024021 24420004 1449fffa ad030000 8ce20000 1040002a 00004021 8f190000 
	00005821 240dffff 000840c0 01e81021 8c430004 8c420000 afa20000 afa30004 
	01c84021 8d020000 8d030004 afa20008 13200016 afa3000c 8f0c0000 27a30010 
	27a80030 00001021 03a25021 27b00008 02024821 914a0000 91290000 51490007 
	24420001 8c690000 552d0003 ad0b0000 ac6b0000 ad0b0000 24420001 24630004 
	004c482b 1520fff0 25080004 256b0001 8ce20000 0162102b 1440ffdb 01604021 
	8f020000 2442ffff 00004021 04400036 00004821 00021880 27a70010 00e31821 
	240bffff 2408ffff 8c670000 00e83826 0047580b 2442ffff 0441fffb 2463fffc 
	00004021 2402ffff 11620027 00004821 000b1080 03a21021 8c4a0010 acaa0000 
	8c4d0030 01aa102a 1440001b 24070001 000a28c0 01e57821 01c52821 03ab6021 
	8de20000 8de30004 afa20000 afa30004 8ca20000 8ca30004 afa20008 afa3000c 
	24e70001 91820000 a1820008 00871821 a0620000 8fa20008 8fa3000c aca20000 
	aca30004 254a0001 25ef0008 01aa102a 1040ffeb 24a50008 accb0000 a0870000 
	00e04021 00074fc3 01001021 01201821 8fb00054 03e00008 27bd0058 
End CFunction
'
CFunction draw
'localfastfillrect
	000006B4
	27bdffa0 afbe005c afb70058 afb60054 afb50050 afb4004c afb30048 afb20044 
	afb10040 afb0003c 8fb80070 8fae0074 8fb10078 8fb9007c 8fb40080 8fb50084 
	8fa90088 afa90028 8fb6008c 8fad0090 8faf0094 8fb20098 8fa9009c 8fb300a0 
	8fb000a4 2417ffff a3b70008 241efffe a3be0009 241efffc a3be000a 241efff8 
	a3be000b 241efff0 a3be000c 241effe0 a3be000d 241effc0 a3be000e 241eff80 
	a3be000f 241e0001 a3be0010 241e0003 a3be0011 241e0007 a3be0012 241e000f 
	a3be0013 241e001f a3be0014 241e003f a3be0015 241e007f a3be0016 a3b70017 
	01afb825 56e0003d 24170001 01c9102a 14400007 2702ffff 152e01d3 00001021 
	0312102b 104001d0 00001021 2702ffff 0058282b 25c3ffff 00a32821 00541821 
	0062102b 00b52821 00451021 00402821 0049102a 1440000a 00606021 55250005 
	264cffff 0072182b 14600006 0330102a 264cffff 0192282b 2522ffff 00a22821 
	0330102a 14400007 2622ffff 161901b6 00001021 0233102b 104001b3 00001021 
	2622ffff 0051402b 2723ffff 01034021 8faa0028 004a1821 0062102b 01164021 
	00481021 00404021 0050102a 144001a8 afa30020 56080005 266cffff 0073182b 
	546001a4 02201021 266cffff 0193282b 2602ffff 1000019e 00a22821 55b70035 
	24170002 55e00033 24170002 0018c027 000e7027 03129021 0258c02b 01c94821 
	03094821 02406021 0520018e 01202821 00118827 0019c827 02339821 0271882b 
	03308021 02308021 afb30020 06000185 02004021 24020001 0054a023 0054102b 
	01224823 02929021 0254a02b 0135a823 02955021 000a1fc3 00037827 000fc7c3 
	02581024 01581824 00405821 00605021 24020001 8fad0028 004d4823 0049102b 
	02028023 01339821 0269482b 0216b023 01366821 000d7fc3 000fc827 001987c3 
	02707024 01b07824 01c01021 1000008e 01e01821 55b7004f 24170003 55e0004d 
	24170003 00111827 00192827 00721021 0043182b 00a92821 00651821 00602821 
	0069182a 1460000a 00406021 55250005 264cffff 0052102b 14400006 24020001 
	264cffff 0192282b 2522ffff 00a22821 24020001 8faa0028 004a4023 0048102b 
	00a21023 010c1821 0068402b 00561023 00605821 01025021 000a47c3 000a1827 
	afa30030 00084027 afa80034 00081fc3 afa30020 afa30024 01631024 01431824 
	00405821 01d0102a 14400006 00605021 160e0135 00001021 0313102b 10400132 
	00001021 2702ffff 0058402b 25c3ffff 01034021 00541821 0062102b 01154021 
	00481021 00404021 0050102a 1440000e afa30020 16080005 2662ffff 0073182b 
	5460000a 03001021 2662ffff 0053402b 2603ffff afa20020 01034021 03001021 
	10000003 01c01821 03001021 01c01821 24170003 55b7003d 0003c7c3 55e0003b 
	0003c7c3 0329102a 14400007 2622ffff 1539010d 00001021 0232102b 1040010a 
	00001021 2622ffff 0051282b 2723ffff 00a32821 8faa0028 004a1821 0062102b 
	00b6b021 00561021 00402821 0049102a 1440000a 00606021 55250005 264cffff 
	0072182b 54600006 0018c027 264cffff 0192282b 2529ffff 00a92821 0018c027 
	000e7027 03139821 0278c02b 01d08021 03108021 afb30020 060000ea 02004021 
	24020001 0054a023 0054102b 02028023 02939821 0274a02b 0215a823 02954821 
	02601021 05200004 01201821 02205821 10000005 03205021 02205821 03205021 
	00001021 00001821 0003c7c3 33180007 0302a021 0298c02b 0303c021 00184f40 
	0014a0c2 0134a025 0018c0c3 0280c821 00086fc3 31ad0007 8faf0020 01af4821 
	012d682b 01a84021 00086f40 000948c2 01a94825 0008b0c3 01208821 02c0f021 
	0014a0c0 0054a023 000948c0 01e94823 17310038 afa90020 17160037 27350001 
	00aa102a 14400035 02b9b82b 15450004 000b68c0 018b102b 14400031 02f8b821 
	008d6821 01607021 01407821 00c7b825 03b98021 8fa20020 03a2a821 03b49821 
	01a09021 8da20000 8da30004 afa20000 12e00008 afa30004 92a30010 92620008 
	00621024 92030000 00431025 10000008 a2020000 92a30010 92620008 00621024 
	00021027 92030000 00431024 a2020000 8fa80000 8fa90004 ae480000 ae490004 
	25c20001 004e702b 01cf1821 00407021 00607821 00a3182a 14600006 25ad0008 
	15e5ffe0 01a09021 0182102b 5040ffde 8da20000 27350001 02b9b82b 02f8b821 
	02a06821 02d7102a 14400041 02e0c021 16fe0005 00aa102a 0235102b 5440003d 
	02f6b02a 00aa102a 5440003a 02f6b02a 15450004 000b18c0 018b102b 54400035 
	02f6b02a 00831821 01607021 01407821 00c79025 03b91021 03b4a021 03b1c821 
	8fa80020 03a89821 00608021 8c680000 8c690004 afa80000 1240000a afa90004 
	92890008 90480000 01284025 a0480000 92690010 93280000 01284025 1000000b 
	a3280000 92880008 00084027 90490000 01094024 a0480000 92680010 00084027 
	93290000 01094024 a3280000 8fa80000 8fa90004 ae080000 ae090004 25c80001 
	010e702b 01cf4821 01007021 01207821 00a9482a 15200006 24630008 15e5ffdb 
	00608021 0188402b 5100ffd9 8c680000 02f6b02a 56c00006 03b5a821 57d80042 
	8fbe005c 02b1102b 10400034 03b5a821 00aa782a 000b10c0 00822021 00c73025 
	2407ffff 10000023 018bc82b 01008021 8d020000 8d030004 afa20000 10c00003 
	afa30004 10000002 a2a70000 a2a00000 8fa20000 8fa30004 ae020000 ae030004 
	25220001 0049482b 012e1821 00404821 00607021 00a3182a 14600006 25080008 
	15c5ffeb 01008021 0182102b 5040ffe9 8d020000 25a20001 004d182b 00406821 
	0078c021 14510003 26b50001 531e0016 8fbe005c 15e0fff8 25a20001 55450004 
	00804021 1720fff5 004d182b 00804021 01604821 1000ffd5 01407021 1000000a 
	8fbe005c 00001021 10000006 00001821 02201021 03201821 03005821 1000fee4 
	01c05021 8fbe005c 8fb70058 8fb60054 8fb50050 8fb4004c 8fb30048 8fb20044 
	8fb10040 8fb0003c 03e00008 27bd0060 

'localdrawPixel
27bdfff8 afb10004 afb00000 8fb80018 
	8faf001c 8fae0020 8fad0024 8fa30028 8fab002c 8fb00030 8faa0034 8fb10038 
	006bc825 17200006 8fac003c 03004821 01e01021 01c04021 10000020 01a02821 
	24190001 14790010 24190002 1560000e 00000000 0018c027 000f7827 03104821 
	0138c02b 01ea7821 030f1021 000e7027 000d6827 01d14021 010e702b 01ac6821 
	1000001a 01cd2821 1479000d 24190003 1560000b 00000000 03004021 01e02821 
	000e7027 000d6827 01d04821 012e702b 01aa6821 1000000d 01cd1021 24190003 
	5479000b 004a582b 15600008 0018c027 000f7827 03114021 0118c02b 01ec7821 
	030f2821 01c04821 01a01021 004a582b 15600007 24030001 55420005 00001821 
	0130802b 16000003 306300ff 00001821 306300ff 1060003a 00ac182b 14600007 
	24020001 55850005 00001021 0111882b 16200003 304200ff 00001021 304200ff 
	1040002f 00c73025 10c00015 00001021 14a00008 00001821 14a00004 24050001 
	2d050020 10a00004 24050001 10000003 01051004 24050001 01051804 000928c0 
	00852021 8c850000 00a22825 ac850000 8c850004 00a32825 10000016 ac850004 
	54a00009 24050001 14a00003 24050001 2d020020 10400005 01052804 00051027 
	10000004 2403ffff 01052804 00051827 2402ffff 000928c0 00852021 8c850000 
	00a22824 ac850000 8c850004 00a32824 ac850004 01202021 10000003 01002821 
	00002021 00002821 00801021 00a01821 8fb10004 8fb00000 03e00008 27bd0008 

'drawline
	27bdffa0 afbf005c afbe0058 afb70054 afb60050 afb5004c afb40048 afb30044 
	afb20040 afb1003c afb00038 0080a821 afa60068 afa7006c 8fa20070 8fa30078 
	8fa40080 8fa50088 00408021 00609021 00809821 0065a02a 00a33023 00653823 
	00f4300a 00c0a021 0044402a 00823823 00443023 00e8300b 00d4a02a 12800005 
	00a0b021 00a09821 0080b021 00608021 00409021 0270102a 10400008 0256102a 
	02401021 02c09021 0040b021 02001021 02608021 00409821 0256102a 10400007 
	0270b823 02d2b023 00178fc2 02378821 00118843 10000006 241e0001 0256b023 
	00178fc2 02378821 00118843 241effff 0270102a 1440003a 8fbf005c 5280001a 
	afb00010 afb20010 001217c3 afa20014 afb00018 001017c3 afa2001c 8fa20090 
	8fa30094 afa20020 afa30024 8fa60098 8fa7009c afa60028 afa7002c 8fa200a0 
	8fa300a4 afa20030 afa30034 02a02021 8fa60068 8fa7006c 0411FF1D 00000000 
	10000018 02368823 001017c3 afa20014 afb20018 001217c3 afa2001c 8fa60090 
	8fa70094 afa60020 afa70024 8fa20098 8fa3009c afa20028 afa3002c 8fa600a0 
	8fa700a4 afa60030 afa70034 02a02021 8fa60068 8fa7006c 0411FF05 00000000 
	02368823 06210003 26100001 025e9021 02378821 0270102a 1040ffc8 8fbf005c 
	8fbe0058 8fb70054 8fb60050 8fb5004c 8fb40048 8fb30044 8fb20040 8fb1003c 
	8fb00038 03e00008 27bd0060 

'drawchar
27bdff40 afbf00bc afbe00b8 afb700b4 afb600b0 
	afb500ac afb400a8 afb300a4 afb200a0 afb1009c afb00098 afa400c0 8fa200d0 
	afa20068 8fa300d4 afa30070 8fa400d8 afa40074 8fa500dc afa50078 8fb000e0 
	8fb400e4 8fa200e8 afa20080 8fa300ec afa30088 afa60048 00431025 144000b7 
	afa7004c 8fa50068 24a20005 0045182b afa2005c 8fa60070 00661821 afa30064 
	27b50048 24020005 afa20060 0200b021 0280b821 0200f021 10000093 afb4006c 
	92a20000 02021007 30420001 1040003b 24030001 17c3001d 8fa30054 8fa4006c 
	5480001b afa30010 8fa5005c afa50010 8fa60064 afa60014 afb20018 afb4001c 
	8fa200f0 8fa300f4 afa20020 afa30024 8fa400f8 8fa500fc afa40028 afa5002c 
	8fa20100 8fa30104 afa20030 afa30034 8fa400c0 24060001 00003821 0411FEA4 
	00000000 10000056 26100001 afa30010 8fa40058 afa40014 afb10018 afb3001c 
	afb60020 afb70024 afb60028 afb7002c 8fa200f0 8fa300f4 afa20030 afa30034 
	8fa400f8 8fa500fc afa40038 afa5003c 8fa20100 8fa30104 afa20040 afa30044 
	8fa400c0 24060001 00003821 0411FC64 00000000 1000003a 26100001 17c3001d 
	8fa30054 8fa4006c 5480001b afa30010 8fa5005c afa50010 8fa60064 afa60014 
	afb20018 afb4001c 8fa200f0 8fa300f4 afa20020 afa30024 8fa400f8 8fa500fc 
	afa40028 afa5002c 8fa20100 8fa30104 afa20030 afa30034 8fa400c0 00003021 
	00003821 0411FE6A 00000000 1000001c 26100001 afa30010 8fa40058 afa40014 
	afb10018 afb3001c afb60020 afb70024 afb60028 afb7002c 8fa200f0 8fa300f4 
	afa20030 afa30034 8fa400f8 8fa500fc afa40038 afa5003c 8fa20100 8fa30104 
	afa20040 afa30044 8fa400c0 00003021 00003821 0411FC2A 00000000 26100001 
	02361021 0051182b 02779821 00408821 00739821 26420001 0052182b 0074a021 
	24030008 1603ff7e 00409021 8fa40060 2484ffff afa40060 8fa5005c 24a2ffff 
	0045202b 8fa60064 24c3ffff afa2005c 00831821 afa30064 8fa20060 2403ffff 
	104300f6 26b50001 8fa40060 009e0019 00001012 00001810 8fa5006c 70a43002 
	00c31821 8fa50068 00452021 0082282b 8fa60070 00661021 afa40054 00a21021 
	afa20058 8fb20074 8fb40078 02408821 02809821 1000ff5a 00008021 24020001 
	8fa30080 146200e1 24020002 8fa40088 148000de 00142023 00101023 0002182b 
	afa2008c 00832023 afa40090 8fa50068 24a20001 0045202b 8fa60070 00862021 
	00501823 0043102b 00942023 afa3006c 00822023 afa4007c afa50060 afa60064 
	27b6004d 27a20047 afa20050 8fa30074 2463fff9 afa30084 8fa40074 24820001 
	0044182b 8fa50078 00651821 0050a823 0055102b 00741823 00621823 afa3005c 
	8fbe00c0 100000a7 02009021 92c20000 02021007 30420001 10400044 00000000 
	1657001c 8fa50054 5680001b afa50010 8fa60060 afa60010 8fa20064 afa20014 
	afb10018 afb3001c 8fa400f0 8fa500f4 afa40020 afa50024 8fa200f8 8fa300fc 
	afa20028 afa3002c 8fa40100 8fa50104 afa40030 afa50034 03c02021 24060001 
	00003821 0411FDD2 00000000 10000069 2610ffff afa50010 8fa60058 afa60014 
	02120019 00001012 00001810 72902002 00831821 02a22023 02a4282b 8fa6005c 
	00c31023 00452823 afa40018 afa5001c afb20020 afb40024 afb20028 afb4002c 
	8fa200f0 8fa300f4 afa20030 afa30034 8fa400f8 8fa500fc afa40038 afa5003c 
	8fa20100 8fa30104 afa20040 afa30044 03c02021 24060001 00003821 0411FB88 
	00000000 10000043 2610ffff 1657001c 8fa30054 5680001b afa30010 8fa30060 
	afa30010 8fa40064 afa40014 afb10018 afb3001c 8fa200f0 8fa300f4 afa20020 
	afa30024 8fa400f8 8fa500fc afa40028 afa5002c 8fa20100 8fa30104 afa20030 
	afa30034 03c02021 00003021 00003821 0411FD8F 00000000 10000026 2610ffff 
	afa30010 8fa40058 afa40014 02120019 00001012 00001810 72902002 00831821 
	02a22023 02a4282b 8fa6005c 00c31023 00452823 afa40018 afa5001c afb20020 
	afb40024 afb20028 afb4002c 8fa200f0 8fa300f4 afa20030 afa30034 8fa400f8 
	8fa500fc afa40038 afa5003c 8fa20100 8fa30104 afa20040 afa30044 03c02021 
	00003021 00003821 0411FB45 00000000 2610ffff 26220001 0051182b 00408821 
	2402ffff 1602ff71 00739821 8fa3006c 8fa4008c 00641021 0043182b 8fa5007c 
	8fa60090 00a62021 afa2006c 00642021 afa4007c 8fa30060 2462ffff 0043182b 
	8fa50064 24a4ffff afa20060 00642021 afa40064 26d6ffff 8fa60050 52c600d9 
	02408021 8fa2006c afa20054 8fa3007c afa30058 8fa40084 8fa50074 0085982b 
	8fa60078 24c2ffff 00808821 02629821 24100007 1000ff4d 24170001 03c08021 
	8fb4006c 24020002 8fa30080 146200c6 24020003 8fa40088 148000c4 8fa30080 
	00101023 0002182b 00142023 afa2007c 00832023 afa40080 8fa50068 24a20001 
	0045202b 8fa60070 00862021 00501823 0043102b 00942023 afa30060 00822023 
	afa40064 afa0005c 0200b021 0280b821 27a20047 afa20050 0200f021 1000009c 
	afb4006c 92020000 02a21024 1040003b 24030001 17c3001d 8fa30054 8fa4006c 
	5480001b afa30010 8fa50068 afa50010 8fa60070 afa60014 afb20018 afb4001c 
	8fa200f0 8fa300f4 afa20020 afa30024 8fa400f8 8fa500fc afa40028 afa5002c 
	8fa20100 8fa30104 afa20030 afa30034 8fa400c0 24060001 00003821 0411FCFC 
	00000000 10000056 2610ffff afa30010 8fa40058 afa40014 afb10018 afb3001c 
	afb60020 afb70024 afb60028 afb7002c 8fa200f0 8fa300f4 afa20030 afa30034 
	8fa400f8 8fa500fc afa40038 afa5003c 8fa20100 8fa30104 afa20040 afa30044 
	8fa400c0 24060001 00003821 0411FABC 00000000 1000003a 2610ffff 17c3001d 
	8fa30054 8fa4006c 5480001b afa30010 8fa50068 afa50010 8fa60070 afa60014 
	afb20018 afb4001c 8fa200f0 8fa300f4 afa20020 afa30024 8fa400f8 8fa500fc 
	afa40028 afa5002c 8fa20100 8fa30104 afa20030 afa30034 8fa400c0 00003021 
	00003821 0411FCC2 00000000 1000001c 2610ffff afa30010 8fa40058 afa40014 
	afb10018 afb3001c afb60020 afb70024 afb60028 afb7002c 8fa200f0 8fa300f4 
	afa20030 afa30034 8fa400f8 8fa500fc afa40038 afa5003c 8fa20100 8fa30104 
	afa20040 afa30044 8fa400c0 00003021 00003821 0411FA82 00000000 2610ffff 
	02361021 0051182b 02779821 00408821 00739821 26420001 0052182b 0074a021 
	8fa30050 1603ff7f 00409021 8fa4005c 24840001 afa4005c 8fa50060 8fa6007c 
	00a61021 0045202b 8fa50064 8fa60080 00a61821 afa20060 00831821 afa30064 
	8fa30068 2462ffff 0043202b 8fa50070 24a3ffff afa20068 00831821 afa30070 
	8fa6005c 24020008 10c20104 8fa20048 24030001 8fa4005c 0083a804 8fa50060 
	afa50054 8fa60064 afa60058 27b0004d 8fb20074 8fb40078 02408821 1000ff59 
	02809821 24020003 8fa30080 146200f2 8fa40088 148000f0 00101023 0002182b 
	00142023 afa20084 00832023 afa40088 00101742 001418c0 00431825 001020c0 
	00901023 0082202b 00741823 00641823 8fa50068 00452021 0082102b 8fa60070 
	00661821 afa40064 00431821 afa3006c 24a20007 0045182b afa2005c 00661821 
	afa30070 24020007 afa20060 8fa30074 24620001 0043282b 8fa40078 00a42821 
	00101823 0003202b 00143023 00c42023 00033782 00042080 00c43025 00032080 
	00901821 0064202b 00d43021 00862021 000337c2 00042040 00c42025 00031840 
	00431821 afa3007c 0062102b 00a41821 00431821 afa3008c 8fa50074 24a5fffb 
	afa50080 8fa60074 00a6182b 8fa40078 2482ffff 00621021 afa20074 0280b021 
	1000009c 0200a021 92020000 02e21024 1040003b 24030001 24050001 1685001c 
	8fa50058 56c0001b afa50010 8fa6005c afa60010 8fa20070 afa20014 afb20018 
	afb5001c 8fa400f0 8fa500f4 afa40020 afa50024 8fa200f8 8fa300fc afa20028 
	afa3002c 8fa40100 8fa50104 afa40030 afa50034 8fa400c0 24060001 00003821 
	0411FC0B 00000000 10000055 26100001 afa50010 8fa60068 afa60014 afb10018 
	afb3001c afb40020 afb60024 afb40028 afb6002c 8fa200f0 8fa300f4 afa20030 
	afa30034 8fa400f8 8fa500fc afa40038 afa5003c 8fa20100 8fa30104 afa20040 
	afa30044 8fa400c0 24060001 00003821 0411F9CB 00000000 10000039 26100001 
	1683001c 8fa30058 56c0001b afa30010 8fa4005c afa40010 8fa50070 afa50014 
	afb20018 afb5001c 8fa200f0 8fa300f4 afa20020 afa30024 8fa400f8 8fa500fc 
	afa40028 afa5002c 8fa20100 8fa30104 afa20030 afa30034 8fa400c0 00003021 
	00003821 0411FBD2 00000000 1000001c 26100001 afa30010 8fa40068 afa40014 
	afb10018 afb3001c afb40020 afb60024 afb40028 afb6002c 8fa200f0 8fa300f4 
	afa20030 afa30034 8fa400f8 8fa500fc afa40038 afa5003c 8fa20100 8fa30104 
	afa20040 afa30044 8fa400c0 00003021 00003821 0411F992 00000000 26100001 
	023e1021 0051182b 8fa40054 02649821 00408821 00739821 26420001 0052182b 
	00409021 8fa50050 1605ff7f 0075a821 8fa60060 24c6ffff afa60060 8fa30064 
	8fa40084 00641021 0043202b 8fa5006c 8fa60088 00a61821 afa20064 00831821 
	afa3006c 8fa3005c 2462ffff 0043202b 8fa50070 24a3ffff afa2005c 00831821 
	afa30070 2402ffff 8fa60060 10c20013 8fa20048 24170001 8fa20060 0057b804 
	8fa30064 afa30058 8fa4006c afa40068 27b00048 0280f021 afb60054 8fb1007c 
	8fb3008c 8fb20080 8fb50074 27a5004e 1000ff55 afa50050 8fa20048 8fa3004c 
	8fbf00bc 8fbe00b8 8fb700b4 8fb600b0 8fb500ac 8fb400a8 8fb300a4 8fb200a0 
	8fb1009c 8fb00098 03e00008 27bd00c0 

'main
27bdffb0 afbf004c 00e06021 8fab0060 
	8faa0064 8fa90068 8fa8006c 8fae0070 8c820000 8c830004 00432025 1480001d 
	8fad0074 8cc20000 8ce30000 afa30010 00031fc3 afa30014 8d630000 afa30018 
	00031fc3 afa3001c 8d430000 afa30020 00031fc3 afa30024 8d230000 afa30028 
	00031fc3 afa3002c 8d030000 afa30030 00031fc3 afa30034 00a02021 00403021 
	00023fc3 0411FB4A 00000000 10000076 8fbf004c 24040001 14440027 24040002 
	14600025 00000000 8cc20000 8ce30000 afa30010 00031fc3 afa30014 8d630000 
	afa30018 00031fc3 afa3001c 8d430000 afa30020 00031fc3 afa30024 8d230000 
	afa30028 00031fc3 afa3002c 8d030000 afa30030 00031fc3 afa30034 8dc30000 
	afa30038 00031fc3 afa3003c 8da30000 afa30040 00031fc3 afa30044 00a02021 
	00403021 00023fc3 0411F8FD 00000000 1000004d 8fbf004c 14440026 24040003 
	14600024 00000000 8cc70004 8cc60000 8d820000 afa20010 000217c3 afa20014 
	8d620000 afa20018 000217c3 afa2001c 8d420000 afa20020 000217c3 afa20024 
	8d220000 afa20028 000217c3 afa2002c 8d020000 afa20030 000217c3 afa20034 
	8dc20000 afa20038 000217c3 afa2003c 8da20000 afa20040 000217c3 afa20044 
	00a02021 0411FC09 00000000 10000026 8fbf004c 14440024 8fbf004c 14600022 
	00a02021 8cc70004 8cc60000 8d820000 afa20010 000217c3 afa20014 8d620000 
	afa20018 000217c3 afa2001c 8d420000 afa20020 000217c3 afa20024 8d220000 
	afa20028 000217c3 afa2002c 8d020000 afa20030 000217c3 afa20034 8dc20000 
	afa20038 000217c3 afa2003c 8da20000 afa20040 000217c3 afa20044 0411FB60 
	00000000 8fbf004c 03e00008 27bd0050 
End CFunction
'
CFunction font
	00000000
	00000000 00000000 4f5b3e00 00003e5b 4f6b3e00 00003e6b 7c3e1c00 00001c3e 
	7e3c1800 0000183c 7d571c00 00001c57 7f5e1c00 00001c5e 3c180000 00000018 
	c3e7ff00 0000ffe7 24180000 00000018 dbe7ff00 0000ffe7 3a060e00 00003048 
	79292600 00002629 05050700 0000407f 05253f00 0000407f e73c5a00 00005a3c 
	1c1c0800 00007f3e 1c3e7f00 0000081c 7f221400 00001422 005f5f00 00005f5f 
	7f017f00 00000609 89956a00 00000066 60606000 00006060 ffa29400 000094a2 
	7e040800 00000804 7e201000 00001020 2a1c0800 00000808 2a080800 0000081c 
	10101000 00001e10 0c1e0c00 00000c1e 3e383000 00003038 3e0e0600 0000060e 
	00000000 00000000 5f000000 00000000 00070000 00000007 147f1400 0000147f 
	7f2a1200 0000242a 08646200 00002313 56205000 00003649 07030000 00000008 
	22410000 0000001c 221c0000 00000041 7f1c2a00 00002a1c 3e080800 00000808 
	70300000 00000080 08080800 00000808 60600000 00000000 08040200 00002010 
	49453e00 00003e51 7f400000 00000042 49494600 00007249 494d3300 00002141 
	127f1000 00001814 45453900 00002745 49493100 00003c4a 11090700 00004121 
	49493600 00003649 49291e00 00004649 14000000 00000000 34000000 00000040 
	14224100 00000008 14141400 00001414 22140800 00000041 59090600 00000201 
	5d594e00 00003e41 11127c00 00007c12 49493600 00007f49 41412200 00003e41 
	41413e00 00007f41 49494100 00007f49 09090100 00007f09 41517300 00003e41 
	08087f00 00007f08 7f410000 00000041 413f0100 00002040 14224100 00007f08 
	40404000 00007f40 1c027f00 00007f02 08107f00 00007f04 41413e00 00003e41 
	09090600 00007f09 51215e00 00003e41 19294600 00007f09 49493200 00002649 
	7f010300 00000301 40403f00 00003f40 40201f00 00001f20 38403f00 00003f40 
	08146300 00006314 78040300 00000304 494d4300 00006159 41414100 0000007f 
	08102000 00000204 41417f00 00000041 01020400 00000402 40404000 00004040 
	07080000 00000003 54784000 00002054 44443800 00007f28 44442800 00003844 
	44287f00 00003844 54541800 00003854 7e090200 00000008 a49c7800 000018a4 
	04047800 00007f08 7d400000 00000044 403d0000 00002040 28440000 00007f10 
	7f400000 00000041 78047800 00007c04 04047800 00007c08 44443800 00003844 
	24241800 0000fc18 2418fc00 00001824 04040800 00007c08 54542400 00004854 
	3f442400 00000404 40207c00 00003c40 40201c00 00001c20 30403c00 00003c40 
	10284400 00004428 90907c00 00004c90 544c4400 00004464 36410000 00000008 
	77000000 00000000 36080000 00000041 02040200 00000201 23263c00 00003c26 
	a1611200 00001ea1 40207a00 00003a40 54555900 00003854 55794100 00002155 
	54784200 00002254 54784000 00002155 55794000 00002054 52721200 00000c1e 
	55555900 00003955 54545900 00003954 54545800 00003955 457c4100 00000000 
	457d4200 00000002 457c4000 00000001 11127d00 00007d12 2528f000 0000f028 
	55450000 00007c54 547c5400 00002054 097f4900 00007c0a 49493200 00003249 
	44443a00 00003a44 48483000 0000324a 41217a00 00003a41 40207800 00003a42 
	a0a07d00 0000009d 42423d00 00003d42 40403d00 00003d40 ff242400 00003c24 
	49436600 0000487e fc2f2b00 00002b2f 29f62000 0000ff09 7e090300 0000c088 
	54794100 00002054 447d4100 00000000 484a3200 00003048 40227a00 00003840 
	0a0a7200 0000007a 19317d00 00007d0d 292f2800 00002629 29292600 00002629 
	4d402000 00003048 08080800 00003808 08083800 00000808 c8acba00 00002f10 
	2834fa00 00002f10 7b000000 00000000 2a142200 00000814 2a140800 00002214 
	5500aa00 0000aa00 aa55aa00 0000aa55 00ff0000 00000000 10ff0000 00001010 
	14ff0000 00001414 ff00ff00 00001010 f010f000 00001010 14fc0000 00001414 
	f700ff00 00001414 ff00ff00 00000000 f404fc00 00001414 17101f00 00001414 
	1f101f00 00001010 141f0000 00001414 10f00000 00001010 001f1000 00000000 
	101f1000 00001010 10f01000 00001010 00ff1000 00000000 10101000 00001010 
	10ff1000 00001010 00ff1400 00000000 ff00ff00 00000000 1f101700 00000000 
	fc04f400 00000000 17101700 00001414 f404f400 00001414 ff00f700 00000000 
	14141400 00001414 f700f700 00001414 14171400 00001414 1f101f00 00001010 
	14f41400 00001414 f010f000 00001010 1f101f00 00000000 001f1400 00000000 
	00fc1400 00000000 f010f000 00000000 ff10ff00 00001010 14ff1400 00001414 
	101f0000 00001010 00f01000 00000000 ffffff00 0000ffff f0f0f000 0000f0f0 
	ff000000 0000ffff 00ffff00 00000000 0f0f0f00 00000f0f 44384400 00003844 
	4a4a3400 0000fc4a 02060600 00007e02 027e0200 0000027e 49416300 00006355 
	443c0400 00003844 201e2000 0000407e 7e020200 00000602 e7a59900 000099a5 
	492a1c00 00001c2a 01724c00 00004c72 4d4d3000 0000304a 78483000 00003048 
	5a463d00 0000bc62 49490000 00003e49 01017e00 00007e01 2a2a2a00 00002a2a 
	5f444400 00004444 4a444000 00004051 4a514000 00004044 ff010300 00000000 
	ff000000 0000e080 6b6b0800 00000808 36243600 00003612 090f0600 0000060f 
	18180000 00000000 10100000 00000000 ff010100 00003040 01011e00 0000001f 
	1d171200 00000019 3c3c3c00 0000003c 00000000 00000000 
End CFunction
