'************************************************************************************* 
' 
' Peter Mather
' 
' 2.2 
' 2015-24-6 
' 
' Graphics library with drivers for SSD1289 display 
' 16-bit parallel output from Micromite MX170 - flexible pinout
'
' drawing primitives assume a graphics display where top left is point 0, 0
'
' set screen orientation: sub SSD.setrot(x) portrait, portrait180, landscape, landscape180
' plot PIXEL: sub SSD_PIXEL(x, y, colour) x, y coordinates of PIXEL, colour
' draw circle: sub SSD.dcirc(x0, y0, r, colour) x, y coordinates of centre, radius, colour
' draw filled circle: sub SSD.fcirc(x0, y0, r, colour) x, y coordinates of centre, radius, colour
' draw triangle:  sub SSD.dtri(x0, y0, x1, y1, 	x2, y2, colour) three coordinate pairs, colour
' draw filled triangle sub SSD.ftri (x0, y0, x1, y1, x2, y2, colour) three coordinate pairs, colour
' print string: sub SSD.pstring(font, xx0, yy0, text$, sz, orientation, forecolour, backcolour) font number, x, y coordinates of top left of first character, text string, size, character orientation,  foreground colour,  background colour
' print bitmap: sub SSD.pbitmap(xx0, yy0, width, height, scale, forecolour, backcolour) x, y coordinates of top left of first character, bitmap, size, character orientation,  foreground colour,  background colour
' draw line: sub SSD.LINE(xx0, yy0, xx1, yy1, colour) x, y of start line, x, y of end line, colour
' draw rectangle: sub SSD.BOX(x, y, w, h, colour) x, y coordinate of top left, width, height, colour
' draw filled rectangle: sub SSD.frect(x, y, w, h, colour) x, y coordinate of top left, width, height, colour
' draw rounded rectangle: sub SSD.drnBOX(x, y, w, h, r, colour) x, y coordinates of top left, width, height, radius of corner, colour
' draw filled rounded rectangle: sub SSD.frnBOX(x, y, w, h, r, colour) x, y coordinates of top left, width, height, radius of corner, colour
' draw bitmap: sub SSD.bitmap(x, y , w , h , size, forecolour, backcolour, bitmap()) x, y coordinates of top left, width, height, scale, foreground colour, background colour, bitmap as integer array
'
option explicit
option default NONE
'
' Define the pins to use for all connections to the display
'
Const RS=12 ' define the pin to use for RS
const RD=13 ' set to zero if tied high
const WR=23 ' define the pin to use for WR
const DB0=25 'etc.
const DB1=26
const DB2=27
CONST DB3=36
CONST DB4=37
CONST DB5=38
CONST DB6=2
CONST DB7=3
CONST DB8=4
const DB9=5  
CONST DB10=30
const db11=31
const db12=21
const db13=22
const db14=11
const db15=24
const Master_reset=32 'define the pin to use for REST
' Connect CS to GND
'
const t_CS=42 'chip select pin for XPT2046 touch controller
const T_IRQ=43 'interrupt pin for XPT2046 touch controller
' Connect T_DO to SPIIN
' Connect T_CLK to SPICLK
' Connect T_DIN to SPIOUT
'
const NSAMPLES=5 'number of times to sample each touch point
const screen_width=240 'width of display in use
const screen_height=320 'height of display in use
CONST default_rotation=0 'default screen orientation for this display type, in this case portrait
dim integer width=screen_width, height=screen_height 'set active values based on default orientation
DIM INTEGER XLEFT0, XRIGHT0, YTOP0, YBOTTOM0 ' global variables to hold calibration values
DIM INTEGER XLEFT, XRIGHT, YTOP, YBOTTOM ' global variables to hold active values
DIM INTEGER xvalues(NSAMPLES-1),yvalues(NSAMPLES-1)' arrays to hold raw sample values
DIM integer XSAMPLE0=&H90 'parameters to hold XPT commands, these may be reversed by the calibration routine
'for a particular application these values can be saved and set as constants
DIM integer YSAMPLE0=&HD0
DIM INTEGER XSAMPLE, YSAMPLE, calibrated=0
dim integer rbuf(320)
pin(t_CS)=1
setpin t_irq,din
setpin t_cs,dout
spi OPEN 1000000,3,8
cpu 48
  print "start Init"
  SSD.initdisplay
  print "Start main"

Do
   SSD_test
loop

end
'
' 
' Display specific routines
'
sub SSD.initdisplay
'
  ' Global variables
  dim integer SSD.pix=0, SSD.ffr=1, SSD.filltri=2, SSD.drawline=3, SSD.writecommand=4, SSD.writeregister=5
  dim integer SSD.fpstring=7, SSD.fcirch=8, SSD.dcirch=9, SSD.Bits=10, SSD.scroll=11, SSD.init=12
  dim integer landscape=2, landscape180=3, portrait=0, portrait180=1
  dim integer SSD_params(27) ' Global variables
  DIM integer SSD_wi=0,SSD_ht=1,SSD_rot=2,SSD_nr=3,SSD_X=4,SSD_Y=5,SSD_seed=6
  dim integer SSD_RS=7, SSD_WR=8
  dim integer SSD_DB0=9, SSD_DB1=10, SSD_DB2=11, SSD_DB3=12, SSD_DB4=13, SSD_DB5=14, SSD_DB6=15, SSD_DB7=16
  DIM INTEGER SSD_DB8=17, SSD_DB9=18, SSD_DB10=19, SSD_DB11=20, SSD_DB12=21, SSD_DB13=22, SSD_DB14=23, SSD_DB15=24
  dim integer SSD_RD=25,SSD_PORTS=26,SSD_OPTIMISE=27
'
  SSD_params(SSD_RS)=RS 'store pin allocation for use in Cfunction
  SSD_params(SSD_WR)=WR
  SSD_params(SSD_RD)=RD
  SSD_params(SSD_DB0)=DB0
  SSD_params(SSD_DB1)=DB1
  SSD_params(SSD_DB2)=DB2
  SSD_params(SSD_DB3)=DB3
  SSD_params(SSD_DB4)=DB4
  SSD_params(SSD_DB5)=DB5
  SSD_params(SSD_DB6)=DB6
  SSD_params(SSD_DB7)=DB7
  SSD_params(SSD_DB8)=DB8
  SSD_PARAMS(SSD_DB9)=DB9
  SSD_PARAMS(SSD_DB10)=DB10
  SSD_PARAMS(SSD_DB11)=DB11
  SSD_PARAMS(SSD_DB12)=DB12
  SSD_PARAMS(SSD_DB13)=DB13
  SSD_PARAMS(SSD_DB14)=DB14
  SSD_PARAMS(SSD_DB15)=DB15
  SSD_params(SSD_wi)=screen_width :  SSD_params(SSD_ht)=screen_height :SSD_params(SSD_optimise)=0
  SSD_PARAMS(SSD_PORTS)=SSDdraw(SSD.init,SSD_params())
  ' if we are using the portC pins in order then the Cfunction can use some optimisations
  if DB0=25 and DB1=26 and DB2=27 and DB3=36 and DB4=37 and DB5=38 and DB6=2 and DB7=3 and DB8=4 and DB9=5 then SSD_params(SSD_optimise)=1
  pin(Master_reset)=1:  SETPIN Master_reset,dout
  pause 100
  pin(Master_RESET)=0
  pause 100
  pin(Master_RESET)=1
  pause 100
	SSD.wreg(&H00,&H0001)
  SSD.wreg(&H03,&HA8A4)
	SSD.wreg(&H0C,&H0000)
	SSD.wreg(&H0D,&H080C)
	SSD.wreg(&H0E,&H2B00)
	SSD.wreg(&H1E,&H00B7)
	SSD.wreg(&H01,&H2B3F)
	SSD.wreg(&H02,&H0600)
	SSD.wreg(&H10,&H0000)
	SSD.wreg(&H11,&H6070)
	SSD.wreg(&H05,&H0000)
	SSD.wreg(&H06,&H0000)
	SSD.wreg(&H16,&HEF1C)
	SSD.wreg(&H17,&H0003)
	SSD.wreg(&H07,&H0233)
	SSD.wreg(&H0B,&H0000)
	SSD.wreg(&H0F,&H0000)
	SSD.wreg(&H41,&H0000)
	SSD.wreg(&H42,&H0000)
	SSD.wreg(&H48,&H0000)
	SSD.wreg(&H49,&H013F)
	SSD.wreg(&H4A,&H0000)
	SSD.wreg(&H4B,&H0000)
	SSD.wreg(&H30,&H0707)
	SSD.wreg(&H31,&H0204)
	SSD.wreg(&H32,&H0204)
	SSD.wreg(&H33,&H0502)
	SSD.wreg(&H34,&H0507)
	SSD.wreg(&H35,&H0204)
	SSD.wreg(&H36,&H0204)
	SSD.wreg(&H37,&H0502)
	SSD.wreg(&H3A,&H0302)
	SSD.wreg(&H3B,&H0302)
	SSD.wreg(&H23,&H0000)
	SSD.wreg(&H24,&H0000)
	SSD.wreg(&H25,&H8000)
  pause 50
  SSD.setrot(portrait)
end sub
'
sub SSD.wreg(register as integer, databyte as integer)
  local integer i
  i=SSDdraw(SSD.writeregister, register,databyte,SSD_params())
end sub
'
'
sub SSD.fillscreen(scol as integer)
  local integer i=SSD_params(SSD_rot)
  SSD_params(SSD_rot)=portrait
  local integer j=ssddraw(SSD.ffr,scol, 0,0,SSD_params(SSD_wi),SSD_params(SSD_ht), SSD_params())
  SSD_params(SSD_rot)=i
end sub
'
' Generic drawing routines
'
sub SSD.dcirc(x0 as integer, y0 as integer, r as integer, colour as integer) 'x, y coordinates of centre, radius
  local integer i=ssddraw(SSD.dcirch, colour,x0,y0,r,31,SSD_params())

end sub
 '
sub SSD.dtri(x0 as integer, y0 as integer, x1 as integer, y1 as integer, x2 as integer, y2 as integer, colour as integer) 'three coordinate pairs
  SSD.dline(x0, y0, x1, y1, colour)
  SSD.dline(x1, y1, x2, y2, colour)
  SSD.dline(x2, y2, x0, y0, colour)
end sub
'
sub SSD.pixel(x0 as integer, y0 as integer,  colour as integer) 'three coordinate pairs
  local integer i=ssddraw(SSD.pix, colour,x0,y0,SSD_params())
end sub
 '
sub SSD.ftri(x0 as integer, y0 as integer, x1 as integer, y1 as integer, x2 as integer, y2 as integer, colour as integer) 'three coordinate pairs
  local integer i=ssddraw(SSD.filltri, x0,y0,x1,y1,x2,y2,colour,SSD_params())
end sub
'
sub SSD.pstring(font as integer, xx0 as integer, yy0 as integer, text$, sz as integer, fontcolour as integer, backgroundcolour as integer) 'x, y coordinates of top left of first character, text string, size, orientation
  local integer j=font and 1
  local integer i=ssddraw(SSD.fpstring,j,xx0,yy0,sz,fontcolour,backgroundcolour,SSD_params(),text$)
end sub
'
sub SSD.dline(xx0 as integer, yy0 as integer, xx1 as integer, yy1 as integer, colour as integer) ' x, y of start line, x, y of end line
   local integer dp=ssddraw(SSD.drawline, colour, xx0, yy0, xx1, yy1,  SSD_params())
end sub
'
sub SSD.drect(x as integer, y as integer, w as integer, h as integer, rcolour as integer) 'x, y coordinate of top left, wi, ht
  local integer a, dp
  dp=ssddraw(SSD.ffr, rcolour, x, y, w, 1,  SSD_params()) 
  a=y+h-1
  dp=ssddraw(SSD.ffr, rcolour, x, a, w, 1,  SSD_params()) 
  dp=ssddraw(SSD.ffr, rcolour, x, y, 1, h,  SSD_params()) 
  a=x+w-1
  dp=ssddraw(SSD.ffr, rcolour, a, y, 1, h,  SSD_params()) 
end sub
'
sub SSD.bitmap(x as integer, y as integer, w as integer, h as integer, size as integer, fcolour as integer, bcolour as integer, array() as integer) 'x, y coordinate of top left, wi, ht
  local integer  j=ssddraw(SSD.Bits, x,y,w,h,size,fcolour,bcolour,array(),SSD_params())  
end sub
'
sub SSD.frect(x as integer, y as integer, w as integer, h as integer, rcolour as integer) 'x, y coordinate of top left, width, height
  local integer dp
  dp=ssddraw(SSD.ffr, rcolour, x, y, w, h,  SSD_params()) 
end sub
'
sub SSD.fcirc(x0 as integer, y0 as integer, r as integer, ccolour as integer)  'x, y coordinates of centre, radius
  local integer dp
  dp=ssddraw(SSD.ffr, ccolour, x0, y0-r, 1, 2*r+1,  SSD_params())
  dp=ssddraw(SSD.fcircH,x0, y0, r, 3, 0, ccolour, SSD_params())
end sub
 '
sub SSD.drndrect(x as integer, y as integer, w as integer, h as integer, r as integer, rcolour as integer) 'x, y coordinates of top left, width, height, radius of corner
  local integer dp
  dp=ssddraw(SSD.ffr, rcolour, x+r, y, w-2*r, 1,  SSD_params()) 
  dp=ssddraw(SSD.ffr, rcolour, x+r, y+h-1, w-2*r, 1,  SSD_params()) 
  dp=ssddraw(SSD.ffr, rcolour, x, y+r, 1, h-2*r,  SSD_params()) 
  dp=ssddraw(SSD.ffr, rcolour, x+w-1, y+r, 1, h-2*r,  SSD_params()) 
  dp=ssddraw(SSD.dcirch, rcolour, x+r, y+r, r, 1,SSD_params())
  dp=ssddraw(SSD.dcirch,  rcolour, x+w-r-1, y+r, r, 2,SSD_params())
  dp=ssddraw(SSD.dcirch,  rcolour, x+w-r-1, y+h-r-1, r, 4,SSD_params())
  dp=ssddraw(SSD.dcirch,  rcolour, x+r, y+h-r-1, r, 8,SSD_params())
end sub
'
sub SSD.frndrect(x as integer, y as integer, w as integer, h as integer, r as integer, colour as integer) 'x, y coordinates of top left, width, height, radius of corner
  local integer dp
  dp=ssddraw(SSD.ffr, colour, x+r, y, w-2*r, h,  SSD_params()) 
  dp=ssddraw(SSD.fcircH,x+w-r-1, y+r, r, 1, h-2*r-1, colour, SSD_params())
  dp=ssddraw(SSD.fcircH,x+r, y+r, r, 2, h-2*r-1, colour, SSD_params())
end sub
'
sub SSD.bitmap(x as integer, y as integer, w as integer, h as integer, r as integer, colour as integer) 'x, y coordinates of top left, width, height, radius of corner
  local integer dp
  dp=ssddraw(SSD.ffr, colour, x+r, y, w-2*r, h,  SSD_params()) 
  dp=ssddraw(SSD.fcircH,x+w-r-1, y+r, r, 1, h-2*r-1, colour, SSD_params())
  dp=ssddraw(SSD.fcircH,x+r, y+r, r, 2, h-2*r-1, colour, SSD_params())
end sub
 '
sub SSD.setrot(x%) 'sets screen rotation: 0=portrait, 1=portrait180, 2=landscape, 3-landscape180
    SSD_params(SSD_rot)=(x% and 3)
    local integer orient = &H6070
    select case x%
        case portrait
        case portrait180
            orient = &H6040
        case landscape
          orient = &H6068
         case landscape180
            orient = &H6058
    end select
    SSD.wreg(&H11,orient) ' set address mode
end sub
'
sub T.rotate(r as integer)
  local integer temp
  ssd.setrot(r)'set the screen rotation
  if default_rotation\2=r\2 then
    XSAMPLE=XSAMPLE0
    YSAMPLE=YSAMPLE0
    width=screen_width
    height=screen_height
    if r=default_rotation then
      xleft=xleft0: xright=xright0 : ytop=ytop0 : ybottom=ybottom0
    else
      xleft=xright0: xright=xleft0 : ytop=ybottom0 : ybottom=ytop0
    endif
  else
    XSAMPLE=YSAMPLE0
    YSAMPLE=XSAMPLE0
    width=screen_height
    height=screen_width
    if ((r mod 2) = (default_rotation mod 2)) then
      if default_rotation=2 then
         xleft=ybottom0
         xright=ytop0
         ytop=xleft0
         ybottom=xright0
       else
         xleft=ytop0
         xright=ybottom0
         ytop=xright0
         ybottom=xleft0
       endif
    else
      if default_rotation=0 then
         xleft=ybottom0
         xright=ytop0
         ytop=xleft0
         ybottom=xright0
       else
         xleft=ytop0
         xright=ybottom0
         ytop=xright0
         ybottom=xleft0
       endif
     endif
   endif
end sub
    '
sub T.calibrate 'gets the orientation correct and then sets global variables XLEFT0, XRIGHT0, YTOP0, YBOTTOM0
  LOCAL integer x,y,refsize=(height+width)\100, refoffset=refsize\2, charsize=refsize\3, testreading
  print ""
  width=screen_width: height=screen_height 
  ssd.setrot(default_rotation)
  XSAMPLE=XSAMPLE0
  YSAMPLE=YSAMPLE0
  if (refsize mod 2) then refsize=refsize+1
  if charsize=0 then charsize=1
  SWITCHXY:
  SSD.fillscreen(0) 
  SSD.frect(0,height\2-refoffset,refsize,refsize,rgb(white))
  SSD.pstring(0,width\2-charsize*4*8,height\2-charsize*4,"Touch box",charsize,rgb(green),0)
  T.gettouch(xleft0,y)
  if xleft0>1000 and xleft0<3000 then 'Switching orientation
    XSAMPLE0=&HD0:XSAMPLE=XSAMPLE0
    YSAMPLE0=&H90:YSAMPLE=YSAMPLE0
    GOTO SWITCHXY
  endif
  PRINT "XSAMPLE0=",hex$(XSAMPLE0)
  PRINT "YSAMPLE0=",HEX$(YSAMPLE0)
  print "XLEFT0=",xleft0
  SSD.fillscreen(0) : pause 1000
  SSD.frect(width-refsize,height\2-refoffset,refsize,refsize,rgb(white))
  SSD.pstring(0,width\2-charsize*4*8,height\2-charsize*4,"Touch box",charsize,rgb(green),0)
  T.gettouch(xright0,y)
  print "XRIGHT0=",xright0
  SSD.fillscreen(0) : pause 1000
  SSD.frect(width\2-refoffset,0,refsize,refsize,rgb(white))
  SSD.pstring(0,width\2-charsize*4*8,height\2-charsize*4,"Touch box",charsize,rgb(green),0)
  T.gettouch(x,ytop0)
  print "YTOP0=",ytop0
  SSD.fillscreen(0) : pause 1000
  SSD.frect(width\2-refoffset,height-refsize,refsize,refsize,rgb(white))
  SSD.pstring(0,width\2-charsize*4*8,height\2-charsize*4,"Touch box",charsize,rgb(green),0)
  T.gettouch(x,ybottom0)
  print "YBOTTOM0=",ybottom0
  SSD.fillscreen(0) : pause 1000
  calibrated=1
end sub
    '
function T.screenpos(x as integer, y as integer, timeout as integer) as integer
    local integer j
    IF T.getxy(xvalues(),yvalues(),timeout,NSAMPLES)<>0 then
      '      j=shellsortARM(xvalues(),NSAMPLES) 'uncomment to use sort as required
      j=shellsortMX(xvalues(),NSAMPLES) 'uncomment to use sort as required
      '      j=shellsortARM(yvalues(),NSAMPLES) 'uncomment to use sort as required
      j=shellsortMX(yvalues(),NSAMPLES) 'uncomment to use sort as required
      x=xvalues(NSAMPLES\2) 'get the median value
      y=yvalues(NSAMPLES\2) 'get the median value
      if xleft>xright then 'x coordinates reversed
        if x<xright then x=xright
        if x>xleft then x=xleft
        x=width-((x-xright)*width/(xleft-xright))
      else 'x coordinates normal
        if x<xleft then x=xleft
        if x>xright then x=xright
        x=(x-xleft)*width/(xright-xleft)
      endif
      if ytop>ybottom then ' y coordinates reversed
        if y>ytop then y=ytop
        if y<ybottom then y=ybottom
        y=height-((y-ybottom)*height/(ytop-ybottom))
      else 'y coordinates normal
        if y<ytop then y=ytop
        if y>ybottom then y=ybottom
        y=(y-ytop)*height/(ybottom-ytop)
      endif
      T.screenpos=1
    else
      T.screenpos=0
    endif
end function
    
sub T.gettouch(x as integer, y as integer)
    local integer j, n=51,xvalues(n-1),yvalues(n-1)
loophere:
    x=-500
    IF T.getxy(xvalues(),yvalues(),1000,n)<>0 then
      j=shellsortMX(xvalues(),n) 'change to use sort as required
      j=shellsortMX(yvalues(),n) 'change to use sort as required
      x=xvalues(n\2) 'get the median value
      y=yvalues(n\2) 'get the median value
    endif
    if x=-500 then goto loophere
end sub
    '
function T.Getdata(command as integer) as integer 'get data from the touch controller
    local integer i,j,k
    pin(T_CS)=0
    i=spi(command)
    j=spi(0)
    k=spi(0)
    pin(T_CS)=1
    T.Getdata= (j*256+k)>>4
end function
    '
function T.getxy(x() as integer, y() as integer, timeout as integer, samples as integer) as integer
    local integer i
    timer=0
    T.getxy=1
    for i=0 to samples-1
      do while timer<timeout and pin(T_IRQ)=1:loop
      if timer>=timeout then
        T.getxy=0
      else
        y(i)=T.getdata(YSAMPLE)
        x(i)=T.getdata(XSAMPLE)
      endif
    next i
end function
'
CFunction SSDdraw
	00000773
	'psend
	8ca200d0 8ca30030 00831826 aca40030 000437c3 10600069 aca60034 8ca600d8 
	8ca500dc 00c53025 50c00005 30640001 3c05bf88 aca46230 1000003d 30640400 
	10800005 30640002 8c440024 8c450028 ac850000 30640002 10800005 30640004 
	8c44002c 8c450030 ac850000 30640004 10800005 30640008 8c440034 8c450038 
	ac850000 30640008 10800005 30640010 8c44003c 8c450040 ac850000 30640010 
	10800005 30640020 8c440044 8c450048 ac850000 30640020 10800005 30640040 
	8c44004c 8c450050 ac850000 30640040 10800005 30640080 8c440054 8c450058 
	ac850000 30640080 10800005 30640100 8c44005c 8c450060 ac850000 30640100 
	10800005 30640200 8c440064 8c450068 ac850000 30640200 10800005 30640400 
	8c44006c 8c450070 ac850000 30640400 10800005 30640800 8c440074 8c450078 
	ac850000 30640800 10800005 30641000 8c44007c 8c450080 ac850000 30641000 
	10800005 30642000 8c440084 8c450088 ac850000 30642000 10800005 30644000 
	8c44008c 8c450090 ac850000 30644000 50800005 30638000 8c440094 8c450098 
	ac850000 30638000 50600005 8c430010 8c43009c 8c4400a0 ac640000 8c430010 
	8c440014 ac640000 8c43000c 8c420014 ac620000 03e00008 00000000 

	'writeRegister
	27bdffe0 afbf001c afb20018 afb10014 afb00010 00a09021 00c08821 8cd000d0 
	8e020004 8e030008 ac430000 00c02821 0411FF7C 00000000 8e020000 8e030008 
	ac430000 02402021 02202821 0411FF75 00000000 8fbf001c 8fb20018 8fb10014 
	8fb00010 03e00008 27bd0020 

	'initdisplay
	27bdffd8 afbf0024 afb30020 afb2001c afb10018 afb00014 00809021 3c119d00 
	8e22003c 0040f809 24040100 00408021 8e220024 8e440038 0040f809 24050006 
	ae020000 8e220024 8e440038 0040f809 24050005 ae020004 8e220028 0040f809 
	8e440038 24130001 00531004 ae020008 8e220024 8e440040 0040f809 24050006 
	ae02000c 8e220024 8e440040 0040f809 24050005 ae020010 8e220028 0040f809 
	8e440040 00531004 ae020014 8e220024 8e440048 0040f809 24050007 ae020024 
	8e220028 0040f809 8e440048 00531004 ae020028 8e220024 8e440050 0040f809 
	24050007 ae02002c 8e220028 0040f809 8e440050 00531004 ae020030 8e220024 
	8e440058 0040f809 24050007 ae020034 8e220028 0040f809 8e440058 00531004 
	ae020038 8e220024 8e440060 0040f809 24050007 ae02003c 8e220028 0040f809 
	8e440060 00531004 ae020040 8e220024 8e440068 0040f809 24050007 ae020044 
	8e220028 0040f809 8e440068 00531004 ae020048 8e220024 8e440070 0040f809 
	24050007 ae02004c 8e220028 0040f809 8e440070 00531004 ae020050 8e220024 
	8e440078 0040f809 24050007 ae020054 8e220028 0040f809 8e440078 00531004 
	ae020058 8e220024 8e440080 0040f809 24050007 ae02005c 8e220028 0040f809 
	8e440080 00531004 ae020060 8e220024 8e440088 0040f809 24050007 ae020064 
	8e220028 0040f809 8e440088 00531004 ae020068 8e220024 8e440090 0040f809 
	24050007 ae02006c 8e220028 0040f809 8e440090 00531004 ae020070 8e220024 
	8e440098 0040f809 24050007 ae020074 8e220028 0040f809 8e440098 00531004 
	ae020078 8e220024 8e4400a0 0040f809 24050007 ae02007c 8e220028 0040f809 
	8e4400a0 00531004 ae020080 8e220024 8e4400a8 0040f809 24050007 ae020084 
	8e220028 0040f809 8e4400a8 00531004 ae020088 8e220024 8e4400b0 0040f809 
	24050007 ae02008c 8e220028 0040f809 8e4400b0 00531004 ae020090 8e220024 
	8e4400b8 0040f809 24050007 ae020094 8e220028 0040f809 8e4400b8 00531004 
	ae020098 8e220024 8e4400c0 0040f809 24050007 ae02009c 8e220028 0040f809 
	8e4400c0 00531004 ae0200a0 8e220010 8e440038 24050008 0040f809 00003021 
	8e220010 8e440040 24050008 0040f809 00003021 8e220010 8e440048 24050008 
	0040f809 00003021 8e220010 8e440050 24050008 0040f809 00003021 8e220010 
	8e440058 24050008 0040f809 00003021 8e220010 8e440060 24050008 0040f809 
	00003021 8e220010 8e440068 24050008 0040f809 00003021 8e220010 8e440070 
	24050008 0040f809 00003021 8e220010 8e440078 24050008 0040f809 00003021 
	8e220010 8e440080 24050008 0040f809 00003021 8e220010 8e440088 24050008 
	0040f809 00003021 8e220010 8e440090 24050008 0040f809 00003021 8e220010 
	8e440098 24050008 0040f809 00003021 8e220010 8e4400a0 24050008 0040f809 
	00003021 8e220010 8e4400a8 24050008 0040f809 00003021 8e220010 8e4400b0 
	24050008 0040f809 00003021 8e220010 8e4400b8 24050008 0040f809 00003021 
	8e220010 8e4400c0 24050008 0040f809 00003021 8e22001c 8e440038 0040f809 
	24050006 8e22001c 8e440040 0040f809 24050006 8e22001c 8e440048 0040f809 
	24050005 8e22001c 8e440050 0040f809 24050005 8e22001c 8e440058 0040f809 
	24050005 8e22001c 8e440060 0040f809 24050005 8e22001c 8e440068 0040f809 
	24050005 8e22001c 8e440070 0040f809 24050005 8e22001c 8e440078 0040f809 
	24050005 8e22001c 8e440080 0040f809 24050005 8e22001c 8e440088 0040f809 
	24050005 8e22001c 8e440090 0040f809 24050005 8e22001c 8e440098 0040f809 
	24050005 8e22001c 8e4400a0 0040f809 24050005 8e22001c 8e4400a8 0040f809 
	24050005 8e22001c 8e4400b0 0040f809 24050005 8e22001c 8e4400b8 0040f809 
	24050005 8e22001c 8e4400c0 0040f809 24050005 00001021 00001821 ae420030 
	ae430034 8e4200c8 8e4300cc 00431025 1040001b 02001021 8e220024 8e440040 
	0040f809 24050006 ae0200a4 8e220024 8e440040 0040f809 24050005 ae0200a8 
	8e220028 0040f809 8e440040 24030001 00431004 ae0200ac 8e220010 8e4400c8 
	24050008 0040f809 00003021 8e22001c 8e4400c8 0040f809 24050006 02001021 
	00001821 8fbf0024 8fb30020 8fb2001c 8fb10018 8fb00014 03e00008 27bd0028 

	'defineregion
	27bdffd0 afbf002c afb50028 afb40024 afb30020 afb2001c afb10018 afb00014 
	8fb00050 2488ffff 8fa50040 01054021 24c7ffff 8fa50048 00e53821 8e050010 
	8e0a0000 8e090008 14a00008 8e1100d0 00801821 00c09821 00c0a821 0080a021 
	00e09021 1000001a 01001021 240b0001 14ab000c 240b0002 00081827 006a1821 
	00041027 004a1021 00079827 02699821 00069027 02499021 0240a821 10000017 
	0040a021 54ab000b 240a0003 01471823 2463ffff 00061027 004a1021 00809821 
	0080a821 0040a021 1000000c 01009021 240a0003 14aa000a 00021200 00c01821 
	00089827 02699821 00049027 02499021 0240a821 00c0a021 00e01021 00021200 
	24040044 00432821 02003021 0411FE09 00000000 24040045 02602821 02003021 
	0411FE04 00000000 24040046 02402821 02003021 0411FDFF 00000000 2404004e 
	02802821 02003021 0411FDFA 00000000 2404004f 02a02821 02003021 0411FDF5 
	00000000 8e220004 8e230008 ac430000 24040022 02002821 0411FD77 00000000 
	8e220000 8e230008 ac430000 8fbf002c 8fb50028 8fb40024 8fb30020 8fb2001c 
	8fb10018 8fb00014 03e00008 27bd0030 

	'localfastfillrect
	27bdffc0 afbf003c afb30038 afb20034 afb10030 afb0002c 00808021 8fa30058 
	8fa20060 8fb10068 70439802 afa30010 8fa3005c afa30014 afa20018 8fa20064 
	afa2001c afb10020 00c02021 00e02821 8fa60050 8fa70054 0411FF7D 00000000 
	7e0220c0 00109203 3252f800 00529025 00108143 321007e0 1260000a 02509025 
	2670ffff 2413ffff 02402021 02202821 0411FD45 00000000 2610ffff 1613fffb 
	02402021 24020001 00001821 8fbf003c 8fb30038 8fb20034 8fb10030 8fb0002c 
	03e00008 27bd0040 

	'DBitmap
	27bdff78 afbf0084 afbe0080 afb7007c afb60078 afb50074 afb40070 afb3006c 
	afb20068 afb10064 afb00060 afa40088 00a08021 afa60090 afa70094 8fb20098 
	8fa4009c 8fa500a8 8ca20010 14400005 8fa300a0 8cb30000 8ca50008 10000010 
	afa50044 24050001 14450006 24050002 8fa200a8 8c530000 8c450008 1000000f 
	afa50044 14450007 24050003 8fa200a8 8c530008 8c450000 10000008 afa50044 
	24050003 14450006 7c8520c0 8fa200a8 8c530008 8c450000 afa50044 7c8520c0 
	00041203 3042f800 00a21025 00042143 308407e0 00442025 afa40038 7c6420c0 
	00031203 3042f800 00821025 00031943 306307e0 00431825 afa3003c 8fa30090 
	72431002 afa20010 000217c3 afa20014 8fa30094 72431002 afa20018 000217c3 
	afa2001c 8fa500a8 afa50020 8fa40088 00042fc3 02003021 00103fc3 0411FF12 
	00000000 8fa20094 18400068 8fbf0084 afb20030 afb00048 8fa30090 afa30054 
	00032823 afa50058 00430018 00001012 afa20050 afa0002c 10000058 afa0004c 
	06000015 0213102b 50400014 26310001 06a20012 26310001 52e00010 26310001 
	93c20000 8fa30028 00431024 10400007 8fa4003c 8fa40038 8fa500a8 0411FCC8 
	00000000 10000005 26310001 8fa500a8 0411FCC3 00000000 26310001 1632ffe8 
	26100001 26940001 8fa50030 8fa20090 12820016 02c5b021 8fa3002c 02831021 
	24430007 285e0000 007e100b 0002f0c3 8fa500a4 00bef021 8fa20034 00541823 
	000317c3 00021742 00621821 30630007 00621023 24030001 00431004 afa20028 
	02c08021 1000ffce 00008821 8fa50040 24a50001 afa50040 14b20007 26b50001 
	1000000c 8fa2004c afa00040 8fa20050 2442ffff afa20034 8fa30090 1860fff3 
	8fb60088 0000a021 8fa50044 1000ffda 02a5b82b 24420001 afa2004c 8fa30048 
	8fa50030 00651821 afa30048 8fa2002c 8fa30054 00431021 afa2002c 8fa50050 
	8fa20058 00a22821 afa50050 8fa3004c 8fa50094 10650005 8fbf0084 1e40ffe2 
	8fb50048 1000ffeb 8fa2004c 8fbe0080 8fb7007c 8fb60078 8fb50074 8fb40070 
	8fb3006c 8fb20068 8fb10064 8fb00060 03e00008 27bd0088 

	'swap
	8c820000 8ca30000 ac830000 03e00008 aca20000 

	'drawline
	27bdffa8 afbf0054 afbe0050 afb7004c afb60048 afb50044 afb40040 afb3003c 
	afb20038 afb10034 afb00030 00809821 afa5005c afa60060 afa70064 8fa20068 
	14c2001f 8fb6006c 00e5102a 10400006 8fa7005c 27a4005c 27a50064 0411FFE3 
	00000000 8fa7005c 8fa20060 afa20010 000217c3 afa20014 8fa20064 24420001 
	00471023 afa20018 000217c3 afa2001c 24020001 00001821 afa20020 afa30024 
	afb60028 02602021 00e03021 00073fc3 0411FED6 00000000 1000005b 8fbf0054 
	8fb7005c 8fa30064 16e3001f 00779023 0046102a 10400006 8fa40060 27a40060 
	27a50068 0411FFC1 00000000 8fa40060 8fa7005c afa40010 000417c3 afa20014 
	24020001 00001821 afa20018 afa3001c 8fa20068 24420001 00442023 afa40020 
	000427c3 afa40024 afb60028 02602021 00e03021 00073fc3 0411FEB4 00000000 
	10000039 8fbf0054 001227c3 00929026 02449023 02e3b82a 24040001 2405ffff 
	00b7200a 0080b821 00468823 00111fc3 00718826 00718823 00c2f02a 24030001 
	2404ffff 009e180a 0060f021 02328021 24140001 0000a821 8fa7005c 8fa20060 
	afa20010 000217c3 afa20014 afb40018 afb5001c afb40020 afb50024 afb60028 
	02602021 00e03021 00073fc3 0411FE8F 00000000 00101040 0051182a 54600008 
	0242102a 8fa3005c 8fa40064 1064000d 02118021 02e31821 afa3005c 0242102a 
	1440ffe6 8fa7005c 8fa20060 8fa30068 10430004 02128021 03c21021 1000ffde 
	afa20060 8fbf0054 8fbe0050 8fb7004c 8fb60048 8fb50044 8fb40040 8fb3003c 
	8fb20038 8fb10034 8fb00030 03e00008 27bd0058 

	'drawchar
	27bdffc8 afbf0034 8fa20048 8fa9004c 8fa80050 8fab0054 7c042420 3c039d00 
	8c630054 00052880 00a31821 8c650000 90a30001 90ac0002 008c682a 15a00019 
	90aa0000 90ad0003 01ac6821 008d682a 51a00015 afa90010 afa80010 afa60014 
	afa70018 706a3002 000630c3 008c6023 70cc2002 24840004 00a42821 afa5001c 
	afab0020 00402021 01202821 01403021 00603821 0411FE7A 00000000 10000013 
	8fbf0034 afa90010 00094fc3 afa90014 71485002 afaa0018 000a57c3 afaa001c 
	70681802 afa30020 00031fc3 afa30024 afab0028 00e02021 00403021 00023fc3 
	0411FE35 00000000 8fbf0034 03e00008 27bd0038 

	'testchar
	8fa30010 8c680010 8c690000 70e61002 00442021 2d020002 10400009 8c630008 
	04800012 00001021 04a00010 0124302a 14c0000e 0065302a 14c0000c 00000000 
	2508fffe 2d080002 11000008 24020001 04800006 00001021 04a00004 0064182a 
	14600002 0125282a 38a20001 03e00008 00000000 

	'pstring
	27bdffa8 afbf0054 afbe0050 afb7004c afb60048 afb50044 afb40040 afb3003c 
	afb20038 afb10034 afb00030 afa40020 8fa20068 afa2002c 8fb00070 8fb30088 
	8fb7008c 00c01821 0040a821 8fa50078 afa50024 8fa20080 afa20028 3c029d00 
	8c420054 00042080 00821021 8c420000 905e0000 92e20000 18400021 0200a021 
	73d08002 0200b021 00d08021 24110001 02169023 afb30010 02402021 02a02821 
	02803021 03c03821 0411FFB8 00000000 1040001c 02f11021 80440000 afb20010 
	afb50014 afb40018 afb3001c 8fa50020 8fa60024 8fa70028 0411FF6F 00000000 
	00404021 00604821 02001821 26310001 92e20000 0051102a 1040ffe5 02168021 
	ae630020 00031fc3 ae630024 8fa2002c ae620028 8fa3002c 000317c3 10000003 
	ae62002c 00004021 00004821 01001021 01201821 8fbf0054 8fbe0050 8fb7004c 
	8fb60048 8fb50044 8fb40040 8fb3003c 8fb20038 8fb10034 8fb00030 03e00008 
	27bd0058 

	'dcirch
	27bdff88 afbf0074 afbe0070 afb7006c afb60068 afb50064 afb40060 afb3005c 
	afb20058 afb10054 afb00050 00808021 00c0b021 8fb50088 8fb20090 8fa20098 
	8fb700a0 02408821 afa60038 afb5003c afa20034 30420010 1040003f 0080f021 
	00069fc3 02551021 afa20010 000217c3 afa20014 24020001 00001821 afa20018 
	afa3001c afa20020 afa30024 afb70028 02603821 0411FD95 00000000 02b21023 
	afa20010 000217c3 afa20014 24020001 00001821 afa20018 afa3001c afa20020 
	afa30024 afb70028 02002021 02c03021 02603821 0411FD85 00000000 00159fc3 
	02563821 afb50010 afb30014 24020001 00001821 afa20018 afa3001c afa20020 
	afa30024 afb70028 02002021 00e03021 00073fc3 0411FD75 00000000 02d23823 
	afb50010 afb30014 24020001 00001821 afa20018 afa3001c afa20020 afa30024 
	afb70028 02002021 00e03021 00073fc3 0411FD66 00000000 1a2000bb 24110001 
	02328823 00121023 00021040 afa20030 26b3ffff 26d4ffff 26b50001 26d60001 
	02408021 00009021 8fa30034 30630004 afa30040 8fa20034 30420002 afa20044 
	8fa30034 30630008 afa30048 8fa20034 30420001 afa20034 06200005 8fa30030 
	2610ffff 24630002 afa30030 02238821 8fa20040 10400023 26520001 8fa3003c 
	02031021 afa20010 000217c3 afa20014 24020001 00001821 afa20018 afa3001c 
	afa20020 afa30024 afb70028 03c02021 02c03021 00163fc3 0411FD34 00000000 
	8fa30038 02033821 afb50010 001517c3 afa20014 24020001 00001821 afa20018 
	afa3001c afa20020 afa30024 afb70028 03c02021 00e03021 00073fc3 0411FD23 
	00000000 8fa30044 10600024 8fa30048 8fa3003c 00701023 afa20010 000217c3 
	afa20014 24020001 00001821 afa20018 afa3001c afa20020 afa30024 afb70028 
	03c02021 02c03021 00163fc3 0411FD0F 00000000 8fa30038 02033821 afb30010 
	001317c3 afa20014 24020001 00001821 afa20018 afa3001c afa20020 afa30024 
	afb70028 03c02021 00e03021 00073fc3 0411FCFE 00000000 8fa30048 10600024 
	8fa30034 8fa20038 00503823 afb50010 001517c3 afa20014 24020001 00001821 
	afa20018 afa3001c afa20020 afa30024 afb70028 03c02021 00e03021 00073fc3 
	0411FCEA 00000000 8fa3003c 02031021 afa20010 000217c3 afa20014 24020001 
	00001821 afa20018 afa3001c afa20020 afa30024 afb70028 03c02021 02803021 
	00143fc3 0411FCD9 00000000 8fa30034 50600024 2673ffff 8fa20038 00503823 
	afb30010 001317c3 afa20014 24020001 00001821 afa20018 afa3001c afa20020 
	afa30024 afb70028 03c02021 00e03021 00073fc3 0411FCC5 00000000 8fa3003c 
	00701023 afa20010 000217c3 afa20014 24020001 00001821 afa20018 afa3001c 
	afa20020 afa30024 afb70028 03c02021 02803021 00143fc3 0411FCB4 00000000 
	2673ffff 2694ffff 26b50001 0250102a 10400005 26d60001 00121040 24420001 
	1000ff5d 02228821 8fbf0074 8fbe0070 8fb7006c 8fb60068 8fb50064 8fb40060 
	8fb3005c 8fb20058 8fb10054 8fb00050 03e00008 27bd0078 

	'fcirch
	27bdff90 afbf006c afbe0068 afb70064 afb60060 afb5005c afb40058 afb30054 
	afb20050 afb1004c afb00048 8fb10080 8fb40090 8fbe0098 afa40038 afa6003c 
	8fa20088 1a200076 02801821 24100001 02118023 0011b823 0017b840 2495ffff 
	24d3ffff 26940003 24960001 00009021 30440001 afa40030 24630001 afa30040 
	30420002 afa20034 06000004 8fa20030 2631ffff 26f70002 02178021 1040002a 
	26520001 8fa3003c 00711023 afa20010 000217c3 afa20014 24020001 00001821 
	afa20018 afa3001c 00111040 8fa30040 00621021 afa20020 000217c3 afa20024 
	8fa400a0 afa40028 03c02021 02c03021 00163fc3 0411FC5F 00000000 8fa20038 
	02223821 afb30010 001317c3 afa20014 24020001 00001821 afa20018 afa3001c 
	afb40020 001417c3 afa20024 8fa300a0 afa30028 03c02021 00e03021 00073fc3 
	0411FC4C 00000000 8fa40034 5080002b 26b5ffff 8fa3003c 00711023 afa20010 
	000217c3 afa20014 24020001 00001821 afa20018 afa3001c 00111040 8fa30040 
	00621021 afa20020 000217c3 afa20024 8fa400a0 afa40028 03c02021 02a03021 
	00153fc3 0411FC33 00000000 8fa20038 00513823 afb30010 001317c3 afa20014 
	24020001 00001821 afa20018 afa3001c afb40020 001417c3 afa20024 8fa300a0 
	afa30028 03c02021 00e03021 00073fc3 0411FC20 00000000 26b5ffff 2673ffff 
	26940002 0251102a 10400005 26d60001 00121040 24420001 1000ff9b 02028021 
	8fbf006c 8fbe0068 8fb70064 8fb60060 8fb5005c 8fb40058 8fb30054 8fb20050 
	8fb1004c 8fb00048 03e00008 27bd0070 

	'ftri
	27bdff90 afbf006c afb50068 afb40064 afb30060 afb2005c afb10058 afb00054 
	8fa20088 8fb100a0 8fb200a8 afa40038 afa6003c 8fa30080 afa30040 afa20044 
	8fa30090 afa30048 8fa30098 0046102a 10400009 afa3004c 27a4003c 27a50044 
	0411FCE8 00000000 27a40038 27a50040 0411FCE4 00000000 8fa2004c 8fa30044 
	0043102a 1040000a 8fa20044 27a4004c 27a50044 0411FCDB 00000000 27a40048 
	27a50040 0411FCD7 00000000 8fa20044 8fa3003c 0043102a 1040000a 8fb0003c 
	27a4003c 27a50044 0411FCCE 00000000 27a40038 27a50040 0411FCCA 00000000 
	8fb0003c 8fa2004c 1602002d 8fb30044 8fa20038 afa20030 afa20034 8fa30040 
	0062202a 10800003 0043102a 10000003 afa30030 54400001 afa30034 8fa20048 
	8fa30030 0043182a 10600003 8fa30034 10000004 afa20030 0062182a 54600001 
	afa20034 8fa70030 8fa2003c afa20010 000217c3 afa20014 24020001 00001821 
	afa20018 afa3001c 8fa20034 24420001 00471023 afa20020 000217c3 afa20024 
	afb20028 02202021 00e03021 00073fc3 0411FBA4 00000000 10000077 8fbf006c 
	00531026 0002102b 02629823 0270102a 14400036 24140001 0000a821 8fa20038 
	8fa3003c 02032823 8fa60040 00c23023 70a63002 8fa40044 00832023 00c4001a 
	008001f4 00002012 00822021 afa40030 8fa60048 00c23023 70a62802 8fa6004c 
	00c31823 00a3001a 006001f4 00001812 00621021 afa20034 0044102a 10400006 
	8fa70030 27a40030 27a50034 0411FC75 00000000 8fa70030 afb00010 001017c3 
	afa20014 8fa20034 24420001 00471023 afa20018 000217c3 afa2001c afb40020 
	afb50024 afb20028 02202021 00e03021 00073fc3 0411FB6B 00000000 26100001 
	0270102a 1040ffce 8fa20038 8fa2004c 0050182a 14600037 24140001 0000a821 
	8fa60040 8fa40038 8fa50048 8fa80044 02083823 00a61823 70e33802 00484023 
	00e8001a 010001f4 00001812 00661821 afa30030 8fa7003c 02073023 00a42823 
	70c53002 00471023 00c2001a 004001f4 00001012 00442021 afa40034 0083202a 
	10800006 8fa70030 27a40030 27a50034 0411FC3C 00000000 8fa70030 afb00010 
	001017c3 afa20014 8fa20034 24420001 00471023 afa20018 000217c3 afa2001c 
	afb40020 afb50024 afb20028 02202021 00e03021 00073fc3 0411FB32 00000000 
	26100001 8fa2004c 0050182a 1060ffcd 8fa60040 8fbf006c 8fb50068 8fb40064 
	8fb30060 8fb2005c 8fb10058 8fb00054 03e00008 27bd0070 

	'main
	27bdffb8 afbf0044 afb00040 00808021 00e05821 8fa90058 8faa005c 8fa80060 
	8fac0064 8c820000 8c830004 00432025 14800014 8fad0068 8ca40000 8cc70004 
	8cc60000 8d620000 8d630004 afa20010 afa30014 24020001 00001821 afa20018 
	afa3001c afa20020 afa30024 afa90028 0411FB06 00000000 00402021 100000d4 
	00602821 24040001 14440018 24040002 14600016 00000000 8ca40000 8cc70004 
	8cc60000 8d620000 8d630004 afa20010 afa30014 8d220000 8d230004 afa20018 
	afa3001c 8d420000 8d430004 afa20020 afa30024 afa80028 0411FAEC 00000000 
	00402021 100000ba 00602821 14440021 24040003 1460001f 00000000 8ca40000 
	8ca50004 8cc70004 8cc60000 8d620000 8d630004 afa20010 afa30014 8d220000 
	8d230004 afa20018 afa3001c 8d420000 8d430004 afa20020 afa30024 8d020000 
	8d030004 afa20028 afa3002c 8d820000 8d830004 afa20030 afa30034 afad0038 
	0411FEC1 00000000 8e040000 10000098 8e050004 1444000f 24040005 1460000d 
	00000000 8ca40000 8cc50000 8d660000 8d270000 8d420000 afa20010 afa80014 
	0411FBB7 00000000 8e040000 10000088 8e050004 1444000b 24040007 14600009 
	00000000 8ca40000 8cc50000 01603021 0411F88F 00000000 8e040000 1000007c 
	8e050004 14440020 24040008 1460001e 00000000 8ca40000 8ca50004 8cc20000 
	8cc30004 8d660000 8d670004 afa60010 afa70014 8d260000 8d270004 afa60018 
	afa7001c 8d460000 8d470004 afa60020 afa70024 8d060000 8d070004 afa60028 
	afa7002c afac0030 afad0034 00403021 00603821 0411FC79 00000000 00402021 
	1000005b 00602821 1444001f 24040009 1460001d 00000000 8ca40000 8ca50004 
	8cc20000 8cc30004 8d660000 8d670004 afa60010 afa70014 8d260000 8d270004 
	afa60018 afa7001c 8d460000 8d470004 afa60020 afa70024 8d060000 8d070004 
	afa60028 afa7002c afac0030 00403021 00603821 0411FDD0 00000000 8e040000 
	1000003b 8e050004 1444001b 2404000a 14600019 00000000 8ca40000 8ca50004 
	8cc20000 8cc30004 8d660000 8d670004 afa60010 afa70014 8d260000 8d270004 
	afa60018 afa7001c 8d460000 8d470004 afa60020 afa70024 afa80028 00403021 
	00603821 0411FC96 00000000 8e040000 1000001f 8e050004 14440015 2404000c 
	14600013 00000000 8ca40000 8cc50000 8d660000 8d270000 8d420000 afa20010 
	8d020000 afa20014 8d820000 afa20018 afad001c 8fa2006c afa20020 0411FA6D 
	00000000 00002021 10000009 00002821 1444000a 8fbf0044 14600008 00a02021 
	0411F82E 00000000 00402021 00602821 00801021 00a01821 8fbf0044 8fb00040 
	03e00008 27bd0048 
End CFunction   'MIPS32 M4K
'
CFunction shellsortMX
  00000000
  8ca20000 27bdfff8 afb10004 afb00000 00401821 00032fc2 00a31821 00031843
  5060003d 00001021 0062282a 10a0fffa 00032fc2 00036823 000370c0 000d60c0
  008e7021 0080c021 008c8021 00607821 0000c821 01e3282a 01c03021 8dd10000
  14a0001e 8dcb0004 8f070004 03202821 0167402a 11000034 8f090000 01e04021
  000840c0 00884021 00a3502a 02003021 ad090000 1540000f ad070004 8cc70004
  8cc90000 0167402a 00ad5021 1100001f 00cc3021 00a04021 01402821 000840c0
  00884021 00a3502a ad090000 1140fff3 ad070004 000528c0 00853021 25ef0001
  01e2282a acd10000 accb0004 27390001 25ce0008 27180008 14a0ffd6 26100008
  00032fc2 00a31821 00031843 1460ffc7 0062282a 00001021 00001821 8fb10004
  8fb00000 03e00008 27bd0008 14ebffe9 0229402b 5100ffe8 000528c0 00a04021
  1000ffde 01402821 54ebffe5 25ef0001 0229402b 5500ffca 01e04021 1000ffe0
  25ef0001
End CFunction
'
sub SSD_test
  local integer k,x,y 'main program variables
  local integer i, j, x0, y0, x1, y1, wi, ht, sz, bitmap(1)=(&HFFC1A191898583FF,0)
  local float tn
  SSD.setrot(landscape)
  wi= SSD_params(SSD_wi)
  ht= SSD_params(SSD_ht)
  SSD.fillscreen(rgb(black))
  SSD.drect(0, 0,ht,wi, rgb(red))
  for i=10 to 0 step -1
    SSD.pstring(1,ht/2-48, 70, str$(i,2), 2, rgb(green), rgb(black))
    pause 1000
  next i
  SSD.pstring(1,ht/2-48, 70, "  ", 2, rgb(green), rgb(black))
  for i=0 to 3
     SSD.setrot(i)
     SSD.Bitmap(2,2,8,8,3,rgb(white),rgb(black),bitmap())
  next i  
  SSD.setrot(landscape)
  SSD.pstring(0,ht/2-112, 60, "Welcome to the", 2, rgb(green), rgb(black))
  SSD.pstring(0,ht/2-80, 100,  "Micromite MKII 4.7b8", 1, rgb(red), rgb(black))
  SSD.pstring(0,ht/2-108, 130, "in colour", 3, rgb(yellow), rgb(black))
  SSD.pstring(0,ht/2-92, 180,  "16-bit parallel SSD1289", 1, rgb(cyan), rgb(black))
  pause 4000
' demo fast line draw
  SSD.fillscreen(rgb(black))
  SSD.setrot(portrait180)
  x0=0:y0=0:y1=ht-1
  for i=0 to 44 step 2
    tn=tan(rad(i))
    x1=wi*tn
    SSD.dline(x0, y0, x1, y1, rgb(red))
  next i
  x1=wi-1
  for i=0 to 44 step 2
    tn=tan(rad(i))
    y1=ht*tn
    SSD.dline(x0, y0, x1, y1, rgb(red))
  next i
  SSD.drect(0, 0, wi, ht, rgb(blue))
  pause 2000

  SSD.fillscreen(rgb(black))
'demo fast text write
  SSD.setrot(landscape)
  for i=0 to wi-13 step 13
    SSD.pstring(0,0, i, "Hello world:Bye world:Hello world:Bye world:Hello world:Bye world:Hello world:Bye world", 1, rgb(white), rgb(black))
  next i
  pause 2000
  SSD.setrot(landscape180)
  SSD.fillscreen(rgb(black))
  for i=0 to wi-26 step 26
    SSD.pstring(0,0, i, "Micromite MKII:Invert:Micromite MKII:Invert", 2, rgb(red), rgb(black))
  next i
  pause 2000
  SSD.setrot(portrait180)
  SSD.fillscreen(rgb(black))
  for i=0 to ht-39 step 39
    SSD.pstring(0,0,i, "Rotate right:Rotate right:Rotate right", 3, rgb(yellow), rgb(black))
  next i
  pause 2000
  SSD.setrot(portrait)
  SSD.fillscreen(rgb(black))
  for i=0 to ht-39 step 39
    SSD.pstring(0,0,i, "Rotate Left:Rotate Left:Rotate Left", 3, rgb(blue), rgb(black))
  next i
  pause 2000
  SSD.fillscreen(rgb(black))
  ' test drawing rectangles
  for i=0 to 3
     SSD.setrot(i)
     SSD.frect(5, 5, 80, 90, rgb(blue))
     SSD.drect(2, 2, 86, 96, rgb(red))
  next i
  pause 2000
' test drawing rounded rectangles
  SSD.fillscreen(rgb(black))
  for i=0 to 3
     SSD.setrot(i)
     SSD.frndrect(5, 5, 80, 45, 10, rgb(blue))
     SSD.drndrect(2, 2, 86, 51, 12, rgb(red))
  next i
  pause 2000
  SSD.fillscreen(rgb(black))
' test drawing circles
  for i=0 to 3
     SSD.setrot(i)
     SSD.fcirc(40, 40, 35, rgb(blue))
     SSD.dcirc(40, 40, 39, rgb(red))
  next i
  pause 2000
' test drawing triangles and line drawing
  SSD.fillscreen(rgb(black))
  for i=0 to 3
     SSD.setrot(i)
     SSD.ftri(4, 4, 100, 6, 22, 104, rgb(blue))
     SSD.dtri(1, 2, 104, 4, 22, 107, rgb(red))
  next i
  pause 2000
  if(NOT calibrated) then T.calibrate 'calibrate the touch sensor
  xleft=xleft0: xright=xright0 : ytop=ytop0 : ybottom=ybottom0 ' set the parameters for the default orientation
  'for a particular application these values can be saved and set as constants
  for j=0 to 3
     T.rotate(j)
     SSD.fillscreen(0)
     SSD.pstring(0,0,0,"Top left",2,rgb(green),0)
     k=0
main:
     if t.screenpos(x,y,1000) then
       if x>3 and x<width-3 and y>3 and y<height-3 then SSD.fcirc(x,y,3,rgb(white))
      else
        k=k+1
     endif
     if k<15 goto main
  next j
end SUB
