'************************************************************************************* 
' 
' Peter Mather
' 
' 2.2 
' 2015-24-6 
' 
' Graphics library with drivers for SSD1983 display 
' 9 bit parallel output from Micromite MX170 - flexible pinout
'
' drawing primitives assume a graphics display where top left is point 0, 0
'
' set screen orientation: sub SSD.setrot(x) portrait, portrait180, landscape, landscape180
' plot PIXEL: sub SSD_PIXEL(x, y, colour) x, y coordinates of PIXEL, colour
' draw circle: sub SSD.dcirc(x0, y0, r, colour) x, y coordinates of centre, radius, colour
' draw filled circle: sub SSD.fcirc(x0, y0, r, colour) x, y coordinates of centre, radius, colour
' draw triangle:  sub SSD.dtri(x0, y0, x1, y1, 	x2, y2, colour) three coordinate pairs, colour
' draw filled triangle sub SSD.ftri (x0, y0, x1, y1, x2, y2, colour) three coordinate pairs, colour
' print string: sub SSD.pstring(font, xx0, yy0, text$, sz, orientation, forecolour, backcolour) font number, x, y coordinates of top left of first character, text string, size, character orientation,  foreground colour,  background colour
' print bitmap: sub SSD.pbitmap(xx0, yy0, width, height, scale, forecolour, backcolour) x, y coordinates of top left of first character, bitmap, size, character orientation,  foreground colour,  background colour
' draw line: sub SSD.LINE(xx0, yy0, xx1, yy1, colour) x, y of start line, x, y of end line, colour
' draw rectangle: sub SSD.BOX(x, y, w, h, colour) x, y coordinate of top left, width, height, colour
' draw filled rectangle: sub SSD.frect(x, y, w, h, colour) x, y coordinate of top left, width, height, colour
' draw rounded rectangle: sub SSD.drnBOX(x, y, w, h, r, colour) x, y coordinates of top left, width, height, radius of corner, colour
' draw filled rounded rectangle: sub SSD.frnBOX(x, y, w, h, r, colour) x, y coordinates of top left, width, height, radius of corner, colour
' draw bitmap: sub SSD.bitmap(x, y , w , h , size, forecolour, backcolour, bitmap()) x, y coordinates of top left, width, height, scale, foreground colour, background colour, bitmap as integer array
' scroll region: sub SSD.scroll(x ,y, w, h) x, y coordinate of top left, width, height of scroll area
'
option explicit
option default NONE
const displaysize=4.3 'choose the size of display 4.3 or 5 or 7
'
' Define the pins to use for all connections to the display
'
Const RS=12   ' define the pin to use for RS, any valid pin
const RD=13   ' any valid pin, set to zero if tied high, scroll functionality not available if zero
const WR=23   ' define the pin to use for WR, any valid pin
const DB0=25  ' Use 25 for optimised update and scroll functionality, otherwise any valid pin
const DB1=26  ' Use 26 for optimised update and scroll functionality, otherwise any valid pin
const DB2=27  ' Use 27 for optimised update and scroll functionality, otherwise any valid pin
CONST DB3=36  ' Use 36 for optimised update and scroll functionality, otherwise any valid pin
CONST DB4=37  ' Use 37 for optimised update and scroll functionality, otherwise any valid pin
CONST DB5=38  ' Use 38 for optimised update and scroll functionality, otherwise any valid pin
CONST DB6=2   ' Use 2 for optimised update and scroll functionality, otherwise any valid pin
CONST DB7=3   ' Use 3 for optimised update and scroll functionality, otherwise any valid pin
CONST DB8=4   ' Use 4 for optimised update and scroll functionality, otherwise any valid pin
const DB9=5   ' Use 5 for optimised update and scroll functionality, set to zero if not using PortC pins for databus
const Master_reset=32 ' define the pin to use for REST, any valid pin
' Connect CS to GND
'
const t_CS=42  ' chip select pin for XPT2046 touch controller, any valid pin
const T_IRQ=43 ' interrupt pin for XPT2046 touch controller, any valid pin
' Connect T_DO to SPIIN
' Connect T_CLK to SPICLK
' Connect T_DIN to SPIOUT
'
const NSAMPLES=5 'number of times to sample each touch point
CONST default_rotation=2 'default screen orientation for this display type, in this case landscape
DIM INTEGER XLEFT0, XRIGHT0, YTOP0, YBOTTOM0 ' global variables to hold calibration values
DIM INTEGER XLEFT, XRIGHT, YTOP, YBOTTOM ' global variables to hold active values
DIM INTEGER xvalues(NSAMPLES-1),yvalues(NSAMPLES-1)' arrays to hold raw sample values
DIM integer XSAMPLE0=&H90 'parameters to hold XPT commands, these may be reversed by the calibration routine
'for a particular application these values can be saved and set as constants
DIM integer YSAMPLE0=&HD0
DIM INTEGER XSAMPLE, YSAMPLE
pin(t_CS)=1
setpin t_irq,din
setpin t_cs,dout
spi OPEN 1000000,3,8
cpu 48
  print "start Init"
  SSD.initdisplay
  print "Start main"

Do
   SSD_test
loop

end
'
' 
' Display specific routines
'
sub SSD.initdisplay
'
  ' Global variables
  dim integer SSD.pix=0, SSD.ffr=1, SSD.filltri=2, SSD.drawline=3, SSD.writecommand=4, SSD.writedata=5
  dim integer SSD.fpstring=7, SSD.fcirch=8, SSD.dcirch=9, SSD.Bits=10, SSD.scrolled=11, SSD.init=12
  dim integer landscape=2, landscape180=3, portrait=0, portrait180=1
  dim integer screen_width, screen_height
  dim integer  SSD1963_PAG_COL_ADDR_ORDER = &H20
  dim integer  SSD1963_COL_ADDR_ORDER     = &H40
  dim integer  SSD1963_PAGE_ADDR_ORDER    = &H80
  dim integer SSD_params(20) ' Global variables
  DIM integer SSD_wi=0,SSD_ht=1,SSD_rot=2,SSD_nr=3,SSD_X=4,SSD_Y=5,SSD_seed=6
  dim integer SSD_RS=7, SSD_WR=8, SSD_DB0=9, SSD_DB1=10, SSD_DB2=11, SSD_DB3=12, SSD_DB4=13, SSD_DB5=14, SSD_DB6=15, SSD_DB7=16, SSD_DB8=17, SSD_RD=18
  dim integer SSD_PORTS=19,SSD_OPTIMISE=20
  dim integer rbuf(800)
'
  SSD_params(SSD_RS)=RS 'store pin allocation for use in Cfunction
  SSD_params(SSD_WR)=wr
  SSD_params(SSD_RD)=rd
  SSD_params(SSD_DB0)=DB0
  SSD_params(SSD_DB1)=DB1
  SSD_params(SSD_DB2)=DB2
  SSD_params(SSD_DB3)=DB3
  SSD_params(SSD_DB4)=DB4
  SSD_params(SSD_DB5)=DB5
  SSD_params(SSD_DB6)=DB6
  SSD_params(SSD_DB7)=DB7
  SSD_params(SSD_DB8)=DB8
  SSD_params(SSD_optimise)=0
  if displaysize=4.3 then
    screen_width=480: screen_height=272
  else
    screen_width=800: screen_height=480
  endif
  dim integer t_width=screen_width, t_height=screen_height 'set active values based on default orientation
  if DB0=25 and DB1=26 and DB2=27 and DB3=36 and DB4=37 and DB5=38 and DB6=2 and DB7=3 and DB8=4 and DB9=5 then SSD_params(SSD_optimise)=1
  SSD_params(SSD_wi)=screen_width :  SSD_params(SSD_ht)=screen_height
  SSD_PARAMS(SSD_PORTS)=SSDdraw(SSD.init,SSD_params())
  pin(Master_reset)=1:  SETPIN Master_reset,dout
  pause 100
  pin(Master_RESET)=0
  pause 100
  pin(Master_RESET)=1
  pause 100
	if displaysize=4.3 then
    SSDsendconfig(&HE2,3,&H23,&H02,&H54)	   
  else
    SSDsendconfig(&HE2,3,&H1E,&H02,&H54)	   
  endif
	SSDsendconfig(&HE0,1,&H01)		' PLL enable
	pause 10
	SSDsendconfig(&HE0,1,&H03)
	pause 10
	SSDwritecommand(&H01)		' software reset
	pause 100
	if displaysize=4.3 then
	  SSDsendconfig(&HE6,3,&H01,&H1F,&HFF)		'PLL setting for PCLK, depends on resolution
  else
	  SSDsendconfig(&HE6,3,&H03,&HFF,&HFF)		'PLL setting for PCLK, depends on resolution
  endif
	if displaysize=4.3 then
	  SSDsendconfig(&HB0,7,&H20,&H00,&H01,&HDF,&H01,&H0F,&H00)		'LCD SPECIFICATION
    SSDsendconfig(&HB4,8,&H02,&H13,&H00,&H08,&H2B,&H00,&H02,&H00)	
    SSDsendconfig(&HB6,7,&H01,&H20,&H00,&H04,&H0C,&H00,&H02)	
  else
	  SSDsendconfig(&HB0,7,&H24,&H00,&H03,&H1F,&H01,&HDF,&H00)		'LCD SPECIFICATION
    SSDsendconfig(&HB4,8,&H03,&HA0,&H00,&H2E,&H30,&H00,&H0F,&H00)
    SSDsendconfig(&HB6,7,&H02,&H0D,&H00,&H10,&H10,&H00,&H08)	
  endif
	SSDsendconfig(&HBA,1,&H0F)
	SSDsendconfig(&HB8,2,&H07,&H01)
	SSDsendconfig(&H36,1,&H00)		'rotation
	SSDsendconfig(&HF0,1,&H06)		'PIXEL data interface
	SSDsendconfig(&H3A,1,&B01100000)		'PIXEL data interface
	SSDsendconfig(&H26,1,&H01)		'gamma curve
	pause 1
	SSDwritecommand(&H29)		'display on
	SSDsendconfig(&HBE,6,&H06,&HF0,&H01,&HF0,&H00,&H00)		'set PWM for B/L
	SSDsendconfig(&Hd0,1,&H0D) 
end sub
'
sub SSDwritecommand(d as integer)
  local integer i=ssddraw(SSD.writecommand, d, SSD_params())
end sub
'
sub SSDsendconfig(command as integer, n as integer, d1%, d2%, d3%, d4%, d5%, d6%, d7%, d8%)
local integer i
   i=ssddraw(SSD.writecommand, command, SSD_params())
   i=ssddraw(SSD.writedata, d1%, SSD_params())
   if n>1 then i=ssddraw(SSD.writedata, d2%, SSD_params())
   if n>2 then i=ssddraw(SSD.writedata, d3%, SSD_params())
   if n>3 then i=ssddraw(SSD.writedata, d4%, SSD_params())
   if n>4 then i=ssddraw(SSD.writedata, d5%, SSD_params())
   if n>5 then i=ssddraw(SSD.writedata, d6%, SSD_params())
   if n>6 then i=ssddraw(SSD.writedata, d7%, SSD_params())
   if n>7 then i=ssddraw(SSD.writedata, d8%, SSD_params())
end sub
'
sub SSD.fillscreen(scol as integer)
  local integer i=SSD_params(SSD_rot)
  SSD_params(SSD_rot)=landscape
  local integer j=ssddraw(SSD.ffr,scol, 0,0,SSD_params(SSD_wi),SSD_params(SSD_ht), SSD_params())
  SSD_params(SSD_rot)=i
end sub
'
' Generic drawing routines
'
sub SSD.dcirc(x0 as integer, y0 as integer, r as integer, colour as integer) 'x, y coordinates of centre, radius
  local integer i=ssddraw(SSD.dcirch, colour,x0,y0,r,31,SSD_params())

end sub
 '
sub SSD.dtri(x0 as integer, y0 as integer, x1 as integer, y1 as integer, x2 as integer, y2 as integer, colour as integer) 'three coordinate pairs
  SSD.dline(x0, y0, x1, y1, colour)
  SSD.dline(x1, y1, x2, y2, colour)
  SSD.dline(x2, y2, x0, y0, colour)
end sub
'
sub SSD.pixel(x0 as integer, y0 as integer,  colour as integer) 'three coordinate pairs
  local integer i=ssddraw(SSD.pix, colour,x0,y0,SSD_params())
end sub
 '
sub SSD.ftri(x0 as integer, y0 as integer, x1 as integer, y1 as integer, x2 as integer, y2 as integer, colour as integer) 'three coordinate pairs
  local integer i=ssddraw(SSD.filltri, x0,y0,x1,y1,x2,y2,colour,SSD_params())
end sub
'
sub SSD.pstring(font as integer, xx0 as integer, yy0 as integer, text$, sz as integer, fontcolour as integer, backgroundcolour as integer) 'x, y coordinates of top left of first character, text string, size, orientation
  local integer j=font and 1
  local integer i=ssddraw(SSD.fpstring,j,xx0,yy0,sz,fontcolour,backgroundcolour,SSD_params(),text$)
end sub
'
sub SSD.dline(xx0 as integer, yy0 as integer, xx1 as integer, yy1 as integer, colour as integer) ' x, y of start line, x, y of end line
   local integer dp=ssddraw(SSD.drawline, colour, xx0, yy0, xx1, yy1,  SSD_params())
end sub
'
sub SSD.drect(x as integer, y as integer, w as integer, h as integer, rcolour as integer) 'x, y coordinate of top left, wi, ht
  local integer a, dp
  dp=ssddraw(SSD.ffr, rcolour, x, y, w, 1,  SSD_params()) 
  a=y+h-1
  dp=ssddraw(SSD.ffr, rcolour, x, a, w, 1,  SSD_params()) 
  dp=ssddraw(SSD.ffr, rcolour, x, y, 1, h,  SSD_params()) 
  a=x+w-1
  dp=ssddraw(SSD.ffr, rcolour, a, y, 1, h,  SSD_params()) 
end sub
'
sub SSD.bitmap(x as integer, y as integer, w as integer, h as integer, size as integer, fcolour as integer, bcolour as integer, array() as integer) 'x, y coordinate of top left, wi, ht
  local integer  j=ssddraw(SSD.Bits, x,y,w,h,size,fcolour,bcolour,array(),SSD_params())  
end sub
'
sub SSD.frect(x as integer, y as integer, w as integer, h as integer, rcolour as integer) 'x, y coordinate of top left, width, height
  local integer dp
  dp=ssddraw(SSD.ffr, rcolour, x, y, w, h,  SSD_params()) 
end sub
'
sub SSD.fcirc(x0 as integer, y0 as integer, r as integer, ccolour as integer)  'x, y coordinates of centre, radius
  local integer dp
  dp=ssddraw(SSD.ffr, ccolour, x0, y0-r, 1, 2*r+1,  SSD_params())
  dp=ssddraw(SSD.fcircH,x0, y0, r, 3, 0, ccolour, SSD_params())
end sub
 '
sub SSD.drndrect(x as integer, y as integer, w as integer, h as integer, r as integer, rcolour as integer) 'x, y coordinates of top left, width, height, radius of corner
  local integer dp
  dp=ssddraw(SSD.ffr, rcolour, x+r, y, w-2*r, 1,  SSD_params()) 
  dp=ssddraw(SSD.ffr, rcolour, x+r, y+h-1, w-2*r, 1,  SSD_params()) 
  dp=ssddraw(SSD.ffr, rcolour, x, y+r, 1, h-2*r,  SSD_params()) 
  dp=ssddraw(SSD.ffr, rcolour, x+w-1, y+r, 1, h-2*r,  SSD_params()) 
  dp=ssddraw(SSD.dcirch, rcolour, x+r, y+r, r, 1,SSD_params())
  dp=ssddraw(SSD.dcirch,  rcolour, x+w-r-1, y+r, r, 2,SSD_params())
  dp=ssddraw(SSD.dcirch,  rcolour, x+w-r-1, y+h-r-1, r, 4,SSD_params())
  dp=ssddraw(SSD.dcirch,  rcolour, x+r, y+h-r-1, r, 8,SSD_params())
end sub
'
sub SSD.frndrect(x as integer, y as integer, w as integer, h as integer, r as integer, colour as integer) 'x, y coordinates of top left, width, height, radius of corner
  local integer dp
  dp=ssddraw(SSD.ffr, colour, x+r, y, w-2*r, h,  SSD_params()) 
  dp=ssddraw(SSD.fcircH,x+w-r-1, y+r, r, 1, h-2*r-1, colour, SSD_params())
  dp=ssddraw(SSD.fcircH,x+r, y+r, r, 2, h-2*r-1, colour, SSD_params())
end sub
'
sub SSD.bitmap(x as integer, y as integer, w as integer, h as integer, r as integer, colour as integer) 'x, y coordinates of top left, width, height, radius of corner
  local integer dp
  dp=ssddraw(SSD.ffr, colour, x+r, y, w-2*r, h,  SSD_params()) 
  dp=ssddraw(SSD.fcircH,x+w-r-1, y+r, r, 1, h-2*r-1, colour, SSD_params())
  dp=ssddraw(SSD.fcircH,x+r, y+r, r, 2, h-2*r-1, colour, SSD_params())
end sub
 '
sub SSD.SCROLL(x as integer, y as integer, w as integer, h as integer)
   if ((not SSD_params(SSD_optimise)) or (not RD)) then
      print "Cannot use Scroll functionality with this pin configuration"
      end
   endif
   local integer i=ssddraw(ssd.scrolled,x,y,w,h,0,rbuf(),SSD_params())
end sub
'
sub SSD.setrot(x%) 'sets screen rotation: 0=portrait, 1=portrait180, 2=landscape, 3-landscape180
    SSD_params(SSD_rot)=(x% and 3)
    local integer orient = 0
    select case x%
        case portrait
           orient = SSD1963_PAGE_ADDR_ORDER OR SSD1963_PAG_COL_ADDR_ORDER
        case portrait180
            orient = SSD1963_COL_ADDR_ORDER OR SSD1963_PAG_COL_ADDR_ORDER
        case landscape
        case landscape180
            orient = SSD1963_PAGE_ADDR_ORDER OR  SSD1963_COL_ADDR_ORDER
    end select
    SSDsendconfig(&H36,1,orient) ' set addresss mode
end sub
'
sub T.rotate(r as integer)
  local integer temp
  ssd.setrot(r)'set the screen rotation
  if default_rotation\2=r\2 then
    XSAMPLE=XSAMPLE0
    YSAMPLE=YSAMPLE0
    t_width=screen_width
    t_height=screen_height
    if r=default_rotation then
      xleft=xleft0: xright=xright0 : ytop=ytop0 : ybottom=ybottom0
    else
      xleft=xright0: xright=xleft0 : ytop=ybottom0 : ybottom=ytop0
    endif
  else
    XSAMPLE=YSAMPLE0
    YSAMPLE=XSAMPLE0
    t_width=screen_height
    t_height=screen_width
    if ((r mod 2) = (default_rotation mod 2)) then
      if default_rotation=2 then
         xleft=ybottom0
         xright=ytop0
         ytop=xleft0
         ybottom=xright0
       else
         xleft=ytop0
         xright=ybottom0
         ytop=xright0
         ybottom=xleft0
       endif
    else
      if default_rotation=0 then
         xleft=ybottom0
         xright=ytop0
         ytop=xleft0
         ybottom=xright0
       else
         xleft=ytop0
         xright=ybottom0
         ytop=xright0
         ybottom=xleft0
       endif
     endif
   endif
end sub
    '
sub T.calibrate 'gets the orientation correct and then sets global variables XLEFT0, XRIGHT0, YTOP0, YBOTTOM0
  LOCAL integer x,y,refsize=(T_height+t_width)\100, refoffset=refsize\2, charsize=refsize\3, testreading
  print ""
  ssd.setrot(default_rotation)
  XSAMPLE=XSAMPLE0
  YSAMPLE=YSAMPLE0
  if (refsize mod 2) then refsize=refsize+1
  if charsize=0 then charsize=1
  SWITCHXY:
  SSD.fillscreen(0) 
  SSD.frect(0,T_height\2-refoffset,refsize,refsize,rgb(white))
  SSD.pstring(0,T_width\2-charsize*4*8,T_height\2-charsize*4,"Touch box",charsize,rgb(green),0)
  T.gettouch(xleft0,y)
  if xleft0>1000 and xleft0<3000 then 'Switching orientation
    XSAMPLE0=&HD0:XSAMPLE=XSAMPLE0
    YSAMPLE0=&H90:YSAMPLE=YSAMPLE0
    GOTO SWITCHXY
  endif
  PRINT "XSAMPLE0=",hex$(XSAMPLE0)
  PRINT "YSAMPLE0=",HEX$(YSAMPLE0)
  print "XLEFT0=",xleft0
  SSD.fillscreen(0) : pause 1000
  SSD.frect(T_width-refsize,T_height\2-refoffset,refsize,refsize,rgb(white))
  SSD.pstring(0,T_width\2-charsize*4*8,T_height\2-charsize*4,"Touch box",charsize,rgb(green),0)
  T.gettouch(xright0,y)
  print "XRIGHT0=",xright0
  SSD.fillscreen(0) : pause 1000
  SSD.frect(T_width\2-refoffset,0,refsize,refsize,rgb(white))
  SSD.pstring(0,T_width\2-charsize*4*8,T_height\2-charsize*4,"Touch box",charsize,rgb(green),0)
  T.gettouch(x,ytop0)
  print "YTOP0=",ytop0
  SSD.fillscreen(0) : pause 1000
  SSD.frect(T_width\2-refoffset,T_height-refsize,refsize,refsize,rgb(white))
  SSD.pstring(0,T_width\2-charsize*4*8,T_height\2-charsize*4,"Touch box",charsize,rgb(green),0)
  T.gettouch(x,ybottom0)
  print "YBOTTOM0=",ybottom0
  SSD.fillscreen(0) : pause 1000
end sub
    '
function T.screenpos(x as integer, y as integer, timeout as integer) as integer
    local integer j
    IF T.getxy(xvalues(),yvalues(),timeout,NSAMPLES)<>0 then
      '      j=shellsortARM(xvalues(),NSAMPLES) 'uncomment to use sort as required
      j=shellsortMX(xvalues(),NSAMPLES) 'uncomment to use sort as required
      '      j=shellsortARM(yvalues(),NSAMPLES) 'uncomment to use sort as required
      j=shellsortMX(yvalues(),NSAMPLES) 'uncomment to use sort as required
      x=xvalues(NSAMPLES\2) 'get the median value
      y=yvalues(NSAMPLES\2) 'get the median value
      if xleft>xright then 'x coordinates reversed
        if x<xright then x=xright
        if x>xleft then x=xleft
        x=T_width-((x-xright)*T_width/(xleft-xright))
      else 'x coordinates normal
        if x<xleft then x=xleft
        if x>xright then x=xright
        x=(x-xleft)*T_width/(xright-xleft)
      endif
      if ytop>ybottom then ' y coordinates reversed
        if y>ytop then y=ytop
        if y<ybottom then y=ybottom
        y=T_height-((y-ybottom)*T_height/(ytop-ybottom))
      else 'y coordinates normal
        if y<ytop then y=ytop
        if y>ybottom then y=ybottom
        y=(y-ytop)*T_height/(ybottom-ytop)
      endif
      T.screenpos=1
    else
      T.screenpos=0
    endif
end function
    
sub T.gettouch(x as integer, y as integer)
    local integer j, n=51,xvalues(n-1),yvalues(n-1)
loophere:
    x=-500
    IF T.getxy(xvalues(),yvalues(),1000,n)<>0 then
      j=shellsortMX(xvalues(),n) 'change to use sort as required
      j=shellsortMX(yvalues(),n) 'change to use sort as required
      x=xvalues(n\2) 'get the median value
      y=yvalues(n\2) 'get the median value
    endif
    if x=-500 then goto loophere
end sub
    '
function T.Getdata(command as integer) as integer 'get data from the touch controller
    local integer i,j,k
    pin(T_CS)=0
    i=spi(command)
    j=spi(0)
    k=spi(0)
    pin(T_CS)=1
    T.Getdata= (j*256+k)>>4
end function
    '
function T.getxy(x() as integer, y() as integer, timeout as integer, samples as integer) as integer
    local integer i
    timer=0
    T.getxy=1
    for i=0 to samples-1
      do while timer<timeout and pin(T_IRQ)=1:loop
      if timer>=timeout then
        T.getxy=0
      else
        y(i)=T.getdata(YSAMPLE)
        x(i)=T.getdata(XSAMPLE)
      endif
    next i
end function
'
CFunction SSDdraw
	000006D1
	'psend
	8ca300a0 8ca600a4 00661825 10600005 8ca20098 3c03bf88 ac646230 1000003d 
	8c430010 8ca30030 00831826 aca40030 000427c3 aca40034 30640001 10800005 
	30640002 8c440024 8c450028 ac850000 30640002 10800005 30640004 8c44002c 
	8c450030 ac850000 30640004 10800005 30640008 8c440034 8c450038 ac850000 
	30640008 10800005 30640010 8c44003c 8c450040 ac850000 30640010 10800005 
	30640020 8c440044 8c450048 ac850000 30640020 10800005 30640040 8c44004c 
	8c450050 ac850000 30640040 10800005 30640080 8c440054 8c450058 ac850000 
	30640080 50800005 30630100 8c44005c 8c450060 ac850000 30630100 50600005 
	8c430010 8c430064 8c440068 ac640000 8c430010 8c440014 ac640000 8c43000c 
	8c420014 ac620000 03e00008 00000000 

	'writeCommand
	27bdffe8 afbf0014 8ca20098 8c430004 8c420008 ac620000 0411FFAD 00000000 
	8fbf0014 03e00008 27bd0018 

	'writeData
	27bdffe8 afbf0014 8ca20098 8c430000 8c420008 ac620000 0411FFA2 00000000 
	8fbf0014 03e00008 27bd0018 

	'initdisplay
	27bdffd8 afbf0024 afb30020 afb2001c afb10018 afb00014 00808021 3c119d00 
	8e22003c 0040f809 24040080 00409021 8e220024 8e040038 0040f809 24050006 
	ae420000 8e220024 8e040038 0040f809 24050005 ae420004 8e220028 0040f809 
	8e040038 24130001 00531004 ae420008 8e220024 8e040040 0040f809 24050006 
	ae42000c 8e220024 8e040040 0040f809 24050005 ae420010 8e220028 0040f809 
	8e040040 00531004 ae420014 8e220024 8e040048 0040f809 24050007 ae420024 
	8e220028 0040f809 8e040048 00531004 ae420028 8e220024 8e040050 0040f809 
	24050007 ae42002c 8e220028 0040f809 8e040050 00531004 ae420030 8e220024 
	8e040058 0040f809 24050007 ae420034 8e220028 0040f809 8e040058 00531004 
	ae420038 8e220024 8e040060 0040f809 24050007 ae42003c 8e220028 0040f809 
	8e040060 00531004 ae420040 8e220024 8e040068 0040f809 24050007 ae420044 
	8e220028 0040f809 8e040068 00531004 ae420048 8e220024 8e040070 0040f809 
	24050007 ae42004c 8e220028 0040f809 8e040070 00531004 ae420050 8e220024 
	8e040078 0040f809 24050007 ae420054 8e220028 0040f809 8e040078 00531004 
	ae420058 8e220024 8e040080 0040f809 24050007 ae42005c 8e220028 0040f809 
	8e040080 00531004 ae420060 8e220024 8e040088 0040f809 24050007 ae420064 
	8e220028 0040f809 8e040088 00531004 ae420068 8e220010 8e040038 24050008 
	0040f809 00003021 8e220010 8e040040 24050008 0040f809 00003021 8e220010 
	8e040048 24050008 0040f809 00003021 8e220010 8e040050 24050008 0040f809 
	00003021 8e220010 8e040058 24050008 0040f809 00003021 8e220010 8e040060 
	24050008 0040f809 00003021 8e220010 8e040068 24050008 0040f809 00003021 
	8e220010 8e040070 24050008 0040f809 00003021 8e220010 8e040078 24050008 
	0040f809 00003021 8e220010 8e040080 24050008 0040f809 00003021 8e220010 
	8e040088 24050008 0040f809 00003021 8e22001c 8e040038 0040f809 24050006 
	8e22001c 8e040040 0040f809 24050006 8e22001c 8e040048 0040f809 24050005 
	8e22001c 8e040050 0040f809 24050005 8e22001c 8e040058 0040f809 24050005 
	8e22001c 8e040060 0040f809 24050005 8e22001c 8e040068 0040f809 24050005 
	8e22001c 8e040070 0040f809 24050005 8e22001c 8e040078 0040f809 24050005 
	8e22001c 8e040080 0040f809 24050005 8e22001c 8e040088 0040f809 24050005 
	8e040090 8e020094 00821025 5040001a 8e0200a0 8e220024 0040f809 24050006 
	ae420018 8e220024 8e040090 0040f809 24050005 ae42001c 8e220028 0040f809 
	8e040090 24030001 00431004 ae420020 8e220010 8e040090 24050008 0040f809 
	00003021 8e22001c 8e040090 0040f809 24050006 8e0200a0 8e0300a4 00431025 
	1040000c 02401021 3c109d00 8e020010 24040005 24050008 0040f809 00003021 
	8e02001c 24040005 0040f809 24050005 02401021 00001821 8fbf0024 8fb30020 
	8fb2001c 8fb10018 8fb00014 03e00008 27bd0028 

	'defineregion
	27bdffd0 afbf002c afb50028 afb40024 afb30020 afb2001c afb10018 afb00014 
	8fb10044 2482ffff 00461021 24a3ffff 00671821 8e260010 8e270000 8e280008 
	14c00008 8e300098 00a09021 00609821 0002a027 0288a021 0004a827 10000012 
	02a8a821 24090001 14c90008 24090002 00031827 00679021 00059827 02679821 
	0080a021 10000012 0040a821 14c90007 24090003 00809021 00409821 00a0a021 
	1000000b 0060a821 24090003 14c90008 00021027 00479021 00049827 02679821 
	00031827 0068a021 0005a827 02a8a821 8e020004 8e030008 ac430000 2404002a 
	02202821 0411FE47 00000000 8e020000 8e030008 ac430000 00122202 02202821 
	0411FE40 00000000 02402021 02202821 0411FE3C 00000000 00132202 02202821 
	0411FE38 00000000 02602021 02202821 0411FE34 00000000 8e020004 8e030008 
	ac430000 2404002b 02202821 0411FE2D 00000000 8e020000 8e030008 ac430000 
	00142202 02202821 0411FE26 00000000 02802021 02202821 0411FE22 00000000 
	00152202 02202821 0411FE1E 00000000 02a02021 02202821 0411FE1A 00000000 
	8e020004 8e030008 ac430000 8fa20040 10400007 2404002e 2404002c 02202821 
	0411FE10 00000000 10000005 8e020000 02202821 0411FE0B 00000000 8e020000 
	8e030008 ac430000 8fbf002c 8fb50028 8fb40024 8fb30020 8fb2001c 8fb10018 
	8fb00014 03e00008 27bd0030 

	'localfastfillrect
	27bdffc8 afbf0034 afb50030 afb4002c afb30028 afb20024 afb10020 afb0001c 
	0080a021 8fa20048 8fb1004c 8e300098 7047a802 24030001 afa30010 afb10014 
	00a02021 00c02821 00e03021 00403821 0411FF68 00000000 7e821340 001493c3 
	325201f8 00529025 7e932880 0014a103 329401c0 02749825 8e020000 8e030008 
	ac430000 12530006 02402021 26b0ffff 16a0001f 2414ffff 10000028 24020001 
	02202821 0411FDD4 00000000 02402021 02202821 0411FDD0 00000000 26b5ffff 
	12a0001d 26b5ffff 2404ffff 8e020010 8e030014 ac430000 8e02000c 8e030014 
	ac430000 8e020010 8e030014 ac430000 8e02000c 8e030014 ac430000 26b5ffff 
	56a4fff3 8e020010 1000000c 24020001 02202821 0411FDB8 00000000 02602021 
	02202821 0411FDB4 00000000 2610ffff 1614fff7 02402021 24020001 00001821 
	8fbf0034 8fb50030 8fb4002c 8fb30028 8fb20024 8fb10020 8fb0001c 03e00008 
	27bd0038 

	'DBitmap
	27bdff80 afbf007c afbe0078 afb70074 afb60070 afb5006c afb40068 afb30064 
	afb20060 afb1005c afb00058 afa40080 00a08021 afa60088 afa7008c 8fb20090 
	8fa30094 8fbe00a0 8fc40010 2c840002 10800005 8fa20098 8fd30008 8fc40000 
	10000004 afa40040 8fd30000 8fc40008 afa40040 7c651340 000323c3 308401f8 
	00a42025 afa4002c 7c642880 00031903 306301c0 00831825 afa30030 7c441340 
	00021bc3 306301f8 00831825 afa30034 7c432880 00021143 304201c0 00621025 
	afa20038 24020001 afa20010 afbe0014 8fa40080 02002821 8fa20088 72423002 
	8fa2008c 72423802 0411FEE9 00000000 3c029d00 8c42001c 8fc40038 0040f809 
	24050006 8fa3008c 18600072 8fbf007c afb20024 afb00044 8fa40088 afa40050 
	00041023 afa20054 00640018 00001012 afa2004c afa00020 10000062 afa00048 
	0600001e 0213102b 5040001d 26310001 06a2001b 26310001 52e00019 26310001 
	8fa3001c 90620000 8fa40018 00441024 1040000b 8fa40034 8fa4002c 03c02821 
	0411FD44 00000000 8fa40030 03c02821 0411FD40 00000000 10000009 26310001 
	03c02821 0411FD3B 00000000 8fa40038 03c02821 0411FD37 00000000 26310001 
	1632ffdf 26100001 26940001 8fa20024 8fa30088 12830017 02c2b021 8fa40020 
	02841021 24430007 28440000 0064100b 000210c3 8fa3009c 00621021 afa2001c 
	8fa40028 00941823 000317c3 00021742 00621821 30630007 00621023 24030001 
	00431004 afa20018 02c08021 1000ffc4 00008821 8fa4003c 24840001 afa4003c 
	14920007 26b50001 1000000c 8fa20048 afa0003c 8fa2004c 2442ffff afa20028 
	8fa30088 1860fff3 8fb60080 0000a021 8fa40040 1000ffd9 02a4b82b 24420001 
	afa20048 8fa30044 8fa40024 00641821 afa30044 8fa20020 8fa30050 00431021 
	afa20020 8fa4004c 8fa20054 00822021 afa4004c 8fa30048 8fa4008c 10640005 
	8fbf007c 1e40ffe2 8fb50044 1000ffeb 8fa20048 8fbe0078 8fb70074 8fb60070 
	8fb5006c 8fb40068 8fb30064 8fb20060 8fb1005c 8fb00058 03e00008 27bd0080 

	'swap
	8c820000 8ca30000 ac830000 03e00008 aca20000 

	'drawline
	27bdffc0 afbf003c afb70038 afb60034 afb50030 afb4002c afb30028 afb20024 
	afb10020 afb0001c 0080a021 afa50044 afa60048 afa7004c 8fa20050 14c20016 
	8fb50054 00e5102a 10400006 8fa20044 27a40044 27a5004c 0411FFE4 00000000 
	8fa20044 8fa7004c 24e70001 24030001 afa30010 afb50014 02802021 00402821 
	8fa60048 00e23823 0411FEBF 00000000 10000049 8fbf003c 8fb60044 8fa3004c 
	16c30015 00769823 0046102a 10400006 8fa60048 27a40048 27a50050 0411FFCB 
	00000000 8fa60048 8fa20050 24420001 00461023 afa20010 afb50014 02802021 
	8fa50044 24070001 0411FEA7 00000000 10000031 8fbf003c 001327c3 00939826 
	02649823 02c3b02a 24040001 2405ffff 00b6200a 0080b021 00469023 00121fc3 
	00729026 00729023 00c2b82a 24030001 2404ffff 0097180a 0060b821 02538021 
	24110001 afb10010 afb50014 02802021 8fa50044 8fa60048 02203821 0411FE8A 
	00000000 00101040 0052182a 54600008 0262102a 8fa30044 8fa4004c 1064000d 
	02128021 02c31821 afa30044 0262102a 5440ffed afb10010 8fa20048 8fa30050 
	10430004 02138021 02e21021 1000ffe5 afa20048 8fbf003c 8fb70038 8fb60034 
	8fb50030 8fb4002c 8fb30028 8fb20024 8fb10020 8fb0001c 03e00008 27bd0040 

	'drawchar
	27bdffd0 afbf002c 8faa0040 8fa90044 8fa20048 8fab004c 7c042420 3c039d00 
	8c630054 00052880 00a31821 8c650000 90a80001 90ac0002 008c682a 15a00019 
	90a30000 90ad0003 01ac6821 008d682a 51a00015 71024002 afa20010 afa60014 
	afa70018 71031002 000210c3 008c6023 704c2002 24840004 00a42821 afa5001c 
	afab0020 01402021 01202821 00603021 01003821 0411FE9D 00000000 1000000b 
	8fbf002c 71024002 afa80010 afab0014 00e02021 01402821 01203021 70623802 
	0411FE39 00000000 8fbf002c 03e00008 27bd0030 

	'testchar
	8fa30010 8c680010 8c690000 70e61002 00442021 2502fffe 2c420002 10400009 
	8c630008 04800011 00001021 04a0000f 0124302a 14c0000d 0065302a 14c0000b 
	00000000 2d080002 11000008 24020001 04800006 00001021 04a00004 0064182a 
	14600002 0125282a 38a20001 03e00008 00000000 

	'pstring
	27bdffa8 afbf0054 afbe0050 afb7004c afb60048 afb50044 afb40040 afb3003c 
	afb20038 afb10034 afb00030 afa40020 8fa20068 afa2002c 8fb00070 8fb30088 
	8fb7008c 00c01821 0040a821 8fa50078 afa50024 8fa20080 afa20028 3c029d00 
	8c420054 00042080 00821021 8c420000 905e0000 92e20000 18400021 0200a021 
	73d08002 0200b021 00d08021 24110001 02169023 afb30010 02402021 02a02821 
	02803021 03c03821 0411FFB8 00000000 1040001c 02f11021 80440000 afb20010 
	afb50014 afb40018 afb3001c 8fa50020 8fa60024 8fa70028 0411FF77 00000000 
	00404021 00604821 02001821 26310001 92e20000 0051102a 1040ffe5 02168021 
	ae630020 00031fc3 ae630024 8fa2002c ae620028 8fa3002c 000317c3 10000003 
	ae62002c 00004021 00004821 01001021 01201821 8fbf0054 8fbe0050 8fb7004c 
	8fb60048 8fb50044 8fb40040 8fb3003c 8fb20038 8fb10034 8fb00030 03e00008 
	27bd0058 

	'dcirch
	27bdffa0 afbf005c afbe0058 afb70054 afb60050 afb5004c afb40048 afb30044 
	afb20040 afb1003c afb00038 00808021 00c0b821 8fb60070 8fb10078 8fa20080 
	8fbe0088 02209021 afa60024 afb60028 afa20020 30420010 10400021 afa40018 
	24130001 afb30010 afbe0014 00c02821 02363021 24070001 0411FDA0 00000000 
	afb30010 afbe0014 02002021 02e02821 02d13023 24070001 0411FD98 00000000 
	afb30010 afbe0014 02002021 02372821 02c03021 24070001 0411FD90 00000000 
	afb30010 afbe0014 02002021 02f12823 02c03021 24070001 0411FD88 00000000 
	1a40007c 24120001 02519023 00111023 00021040 afa2001c 26d4ffff 26f5ffff 
	26d60001 26f70001 02208021 00009821 8fa20020 30420004 afa2002c 24110001 
	8fa20020 30420002 afa20030 8fa20020 30420008 afa20034 8fa20020 30420001 
	afa20020 06400005 8fa2001c 2610ffff 24420002 afa2001c 02429021 8fa2002c 
	10400013 26730001 afb10010 afbe0014 8fa40018 02e02821 8fa20028 02023021 
	02203821 0411FD5D 00000000 afb10010 afbe0014 8fa40018 8fa20024 02022821 
	02c03021 02203821 0411FD54 00000000 8fa20030 10400014 8fa20034 afb10010 
	afbe0014 8fa40018 02e02821 8fa20028 00503023 02203821 0411FD48 00000000 
	afb10010 afbe0014 8fa40018 8fa20024 02022821 02803021 02203821 0411FD3F 
	00000000 8fa20034 10400014 8fa20020 afb10010 afbe0014 8fa40018 8fa20024 
	00502823 02c03021 02203821 0411FD33 00000000 afb10010 afbe0014 8fa40018 
	02a02821 8fa20028 02023021 02203821 0411FD2A 00000000 8fa20020 50400014 
	2694ffff afb10010 afbe0014 8fa40018 8fa20024 00502823 02803021 02203821 
	0411FD1E 00000000 afb10010 afbe0014 8fa40018 02a02821 8fa20028 00503023 
	02203821 0411FD15 00000000 2694ffff 26b5ffff 26d60001 0270102a 10400005 
	26f70001 00131040 24420001 1000ff9d 02429021 8fbf005c 8fbe0058 8fb70054 
	8fb60050 8fb5004c 8fb40048 8fb30044 8fb20040 8fb1003c 8fb00038 03e00008 
	27bd0060 

	'fcirch
	27bdffa8 afbf0054 afbe0050 afb7004c afb60048 afb50044 afb40040 afb3003c 
	afb20038 afb10034 afb00030 8fb10068 8fb40078 8fbe0080 afa40020 afa60024 
	8fa20070 1a200052 02801821 24100001 02118023 0011b823 0017b840 2495ffff 
	24d3ffff 26940003 24960001 00009021 30440001 afa40018 24630001 afa30028 
	30420002 afa2001c 06000004 8fa20018 2631ffff 26f70002 02178021 10400018 
	26520001 00111040 8fa30028 00621021 afa20010 8fa40088 afa40014 03c02021 
	02c02821 8fa20024 00513023 24070001 0411FCC9 00000000 afb40010 8fa30088 
	afa30014 03c02021 8fa20020 02222821 02603021 24070001 0411FCBF 00000000 
	8fa3001c 50600019 26b5ffff 00111040 8fa40028 00821021 afa20010 8fa20088 
	afa20014 03c02021 02a02821 8fa30024 00713023 24070001 0411FCAF 00000000 
	afb40010 8fa40088 afa40014 03c02021 8fa20020 00512823 02603021 24070001 
	0411FCA5 00000000 26b5ffff 2673ffff 26940002 0251102a 10400005 26d60001 
	00121040 24420001 1000ffbf 02028021 8fbf0054 8fbe0050 8fb7004c 8fb60048 
	8fb50044 8fb40040 8fb3003c 8fb20038 8fb10034 8fb00030 03e00008 27bd0058 

	'ftri
	27bdffb0 afbf004c afb40048 afb30044 afb20040 afb1003c afb00038 8fa20068 
	8fb10080 8fb20088 afa40020 afa60024 8fa30060 afa30028 afa2002c 8fa30070 
	afa30030 8fa30078 0046102a 10400009 afa30034 27a40024 27a5002c 0411FD8F 
	00000000 27a40020 27a50028 0411FD8B 00000000 8fa20034 8fa3002c 0043102a 
	1040000a 8fa2002c 27a40034 27a5002c 0411FD82 00000000 27a40030 27a50028 
	0411FD7E 00000000 8fa2002c 8fa30024 0043102a 1040000a 8fb00024 27a40024 
	27a5002c 0411FD75 00000000 27a40020 27a50028 0411FD71 00000000 8fb00024 
	8fa20034 16020023 8fb3002c 8fa20020 afa20018 afa2001c 8fa30028 0062202a 
	10800003 0043102a 10000003 afa30018 54400001 afa3001c 8fa20030 8fa30018 
	0043182a 10600003 8fa3001c 10000004 afa20018 0062182a 54600001 afa2001c 
	8fa50018 8fa2001c 24420001 00451023 afa20010 afb20014 02202021 8fa60024 
	24070001 0411FC34 00000000 10000068 8fbf004c 00531026 0002102b 02629823 
	0270102a 14400030 8fa20034 24140001 8fa20020 8fa30024 02032823 8fa60028 
	00c23023 70a63002 8fa4002c 00832023 00c4001a 008001f4 00002012 00822021 
	afa40018 8fa60030 00c23023 70a62802 8fa60034 00c31823 00a3001a 006001f4 
	00001812 00621021 afa2001c 0044102a 10400006 8fa20018 27a40018 27a5001c 
	0411FD26 00000000 8fa20018 8fa7001c 24e70001 afb40010 afb20014 02202021 
	00402821 02003021 00e23823 0411FC02 00000000 26100001 0270102a 1040ffd5 
	8fa20020 8fa20034 0050182a 1460002f 24130001 8fa60028 8fa40020 8fa50030 
	8fa8002c 02083823 00a61823 70e33802 00484023 00e8001a 010001f4 00001812 
	00661821 afa30018 8fa70024 02073023 00a42823 70c53002 00471023 00c2001a 
	004001f4 00001012 00442021 afa4001c 0083202a 10800006 8fa20018 27a40018 
	27a5001c 0411FCF5 00000000 8fa20018 8fa7001c 24e70001 afb30010 afb20014 
	02202021 00402821 02003021 00e23823 0411FBD1 00000000 26100001 8fa20034 
	0050182a 1060ffd4 8fa60028 8fbf004c 8fb40048 8fb30044 8fb20040 8fb1003c 
	8fb00038 03e00008 27bd0050 

	'scroll
	27bdffb0 afbf004c afbe0048 afb70044 afb60040 afb5003c afb40038 afb30034 
	afb20030 afb1002c afb00028 00808821 afa50054 afa7005c 8fb40068 24b5ffff 
	02a7a821 2482ffff 00461021 afa2001c 3c109d00 8e020028 0040f809 8e840090 
	24120001 00529004 8e020024 8e840090 0040f809 24050005 0040b821 8e020024 
	8e840090 0040f809 24050006 0040b021 26310001 afb10018 8fa3001c 0071102a 
	1440004b 3c10bf88 8fa40054 0095202a afa40020 3c1ebf81 afa00010 afb40014 
	8fa40018 8fa50054 24060001 8fa7005c 0411FB0B 00000000 240201ff ae026210 
	3c039d00 8c62001c 8e840038 0040f809 24050006 8fa40020 10800041 8fa30054 
	8fb10064 02201021 aef20000 8e046210 8fc4f220 aed20000 8e046220 ac440000 
	aef20000 8e046210 8fc4f220 aed20000 8e046220 ac440004 24630001 1475fff2 
	24420008 ae006210 24020001 afa20010 afb40014 8fa20018 2444ffff 8fa50054 
	24060001 8fa7005c 0411FAE5 00000000 3c039d00 8c62001c 8e840038 0040f809 
	24050006 8fb30054 8e240000 02802821 0411F95C 00000000 8e240004 02802821 
	0411F958 00000000 26730001 1675fff6 26310008 8fa40018 24840001 afa40018 
	8fa3001c 0064102a 5040ffbc afa00010 8fbf004c 8fbe0048 8fb70044 8fb60040 
	8fb5003c 8fb40038 8fb30034 8fb20030 8fb1002c 8fb00028 03e00008 27bd0050 
	ae006210 24020001 afa20010 afb40014 8fa20018 2444ffff 8fa50054 24060001 
	8fa7005c 0411FAB6 00000000 3c039d00 8c62001c 8e840038 0040f809 24050006 
	1000ffdd 8fa40018 

	'main
	27bdffb8 afbf0044 afb00040 00808021 00e01021 8fac0058 8fab005c 8faa0060 
	8fa80064 8c830000 8c840004 00643825 14e0000d 8fa90068 8ca40000 8cc50000 
	8c460000 24020001 afa20010 afac0014 24070001 0411FB1B 00000000 00402021 
	100000e0 00602821 24070001 1467000f 24070002 1480000d 00000000 8ca40000 
	8cc50000 8c460000 8d870000 8d620000 afa20010 afaa0014 0411FB0A 00000000 
	00402021 100000cf 00602821 14670021 24070003 1480001f 00000000 8ca40000 
	8ca50004 8cc70004 8cc60000 8c430004 8c420000 afa20010 afa30014 8d820000 
	8d830004 afa20018 afa3001c 8d620000 8d630004 afa20020 afa30024 8d420000 
	8d430004 afa20028 afa3002c 8d020000 8d030004 afa20030 afa30034 afa90038 
	0411FE5A 00000000 8e040000 100000ad 8e050004 1467000f 24070004 1480000d 
	00000000 8ca40000 8cc50000 8c460000 8d870000 8d620000 afa20010 afaa0014 
	0411FBF6 00000000 8e040000 1000009d 8e050004 1467000a 24070005 14800008 
	00000000 8ca40000 00c02821 0411F917 00000000 8e040000 10000092 8e050004 
	1467000a 24070007 14800008 00000000 8ca40000 00c02821 0411F917 00000000 
	8e040000 10000087 8e050004 1467001e 24070008 1480001c 00000000 8ca40000 
	8ca50004 8cc70004 8cc60000 8c430004 8c420000 afa20010 afa30014 8d820000 
	8d830004 afa20018 afa3001c 8d620000 8d630004 afa20020 afa30024 8d420000 
	8d430004 afa20028 afa3002c afa80030 afa90034 0411FC8B 00000000 00402021 
	10000068 00602821 1467001d 24070009 1480001b 00000000 8ca40000 8ca50004 
	8cc70004 8cc60000 8c430004 8c420000 afa20010 afa30014 8d820000 8d830004 
	afa20018 afa3001c 8d620000 8d630004 afa20020 afa30024 8d420000 8d430004 
	afa20028 afa3002c afa80030 0411FD87 00000000 8e040000 1000004a 8e050004 
	14670019 2407000a 14800017 00000000 8ca40000 8ca50004 8cc70004 8cc60000 
	8c430004 8c420000 afa20010 afa30014 8d820000 8d830004 afa20018 afa3001c 
	8d620000 8d630004 afa20020 afa30024 afaa0028 0411FCAC 00000000 8e040000 
	10000030 8e050004 14670015 2407000b 14800013 00000000 8ca40000 8cc50000 
	8c460000 8d870000 8d620000 afa20010 8d420000 afa20014 8d020000 afa20018 
	afa9001c 8fa2006c afa20020 0411FAAE 00000000 00002021 1000001a 00002821 
	54670010 2402000c 5480000e 2402000c 8ca40000 8cc50000 8c460000 8d870000 
	8d620000 afa20010 afaa0014 afa80018 0411FE81 00000000 00402021 10000009 
	00602821 1462000a 8fbf0044 14800008 00a02021 0411F89B 00000000 00402021 
	00602821 00801021 00a01821 8fbf0044 8fb00040 03e00008 27bd0048 
End CFunction   'MIPS32 M4K
'
CFunction shellsortMX
  00000000
  8ca20000 27bdfff8 afb10004 afb00000 00401821 00032fc2 00a31821 00031843
  5060003d 00001021 0062282a 10a0fffa 00032fc2 00036823 000370c0 000d60c0
  008e7021 0080c021 008c8021 00607821 0000c821 01e3282a 01c03021 8dd10000
  14a0001e 8dcb0004 8f070004 03202821 0167402a 11000034 8f090000 01e04021
  000840c0 00884021 00a3502a 02003021 ad090000 1540000f ad070004 8cc70004
  8cc90000 0167402a 00ad5021 1100001f 00cc3021 00a04021 01402821 000840c0
  00884021 00a3502a ad090000 1140fff3 ad070004 000528c0 00853021 25ef0001
  01e2282a acd10000 accb0004 27390001 25ce0008 27180008 14a0ffd6 26100008
  00032fc2 00a31821 00031843 1460ffc7 0062282a 00001021 00001821 8fb10004
  8fb00000 03e00008 27bd0008 14ebffe9 0229402b 5100ffe8 000528c0 00a04021
  1000ffde 01402821 54ebffe5 25ef0001 0229402b 5500ffca 01e04021 1000ffe0
  25ef0001
End CFunction
'
sub SSD_test
  local integer k,x,y 'main program variables
  local integer i, j, x0, y0, x1, y1, wi, ht, sz, bitmap(1)=(&HFFC1A191898583FF,0)
  local float tn
  const scrollxstart=100, scrollxwidth=280,newpoint=scrollxstart+scrollxwidth-1
  const scrollystart=20,scrollyheight=232, scrolldirection=0
  local integer rp=130, bp=100, yp=160, wp=190, cp=70, xp=40, zp=220
  SSD.setrot(landscape)
  wi= SSD_params(SSD_wi)
  ht= SSD_params(SSD_ht)
  SSD.fillscreen(rgb(black))
  SSD.drect(0, 0,wi,ht, rgb(red))
  for i=10 to 0 step -1
    SSD.pstring(1,wi/2-48, 70, str$(i,2), 2, rgb(green), rgb(black))
    pause 1000
  next i
  SSD.pstring(1,wi/2-48, 70, "  ", 2, rgb(green), rgb(black))
  for i=0 to 3
     SSD.setrot(i)
     SSD.Bitmap(10,10,8,8,3,rgb(blue),rgb(black),bitmap())
  next i  
  SSD.setrot(landscape)
  SSD.pstring(0,wi/2-112, 60, "Welcome to the", 2, rgb(green), rgb(black))
  SSD.pstring(0,wi/2-168, 100,  "Micromite MKII 4.7b8", 2, rgb(red), rgb(black))
  SSD.pstring(0,wi/2-108, 150, "in colour", 3, rgb(yellow), rgb(black))
  SSD.pstring(0,wi/2-176, 220,  "9-bit parallel SSD1963", 2, rgb(cyan), rgb(black))
  pause 4000
' demo fast line draw
  SSD.fillscreen(rgb(black))
  SSD.setrot(landscape180)
  x0=0:y0=0:y1=ht-1
  for i=0 to 44 step 2
    tn=tan(rad(i))
    x1=wi*tn
    SSD.dline(x0, y0, x1, y1, rgb(red))
  next i
  x1=wi-1
  for i=0 to 44 step 2
    tn=tan(rad(i))
    y1=ht*tn
    SSD.dline(x0, y0, x1, y1, rgb(red))
  next i
  SSD.drect(0, 0, wi, ht, rgb(blue))
  pause 2000

  SSD.fillscreen(rgb(black))
'demo fast text write
  SSD.setrot(landscape)

  for i=0 to ht-13 step 13
    SSD.pstring(0,0, i, "Hello world:Bye world:Hello world:Bye world:Hello world:Bye world:Hello world:Bye world", 1, rgb(white), rgb(black))
  next i
  pause 2000
  SSD.setrot(landscape180)
  SSD.fillscreen(rgb(black))
  for i=0 to ht-26 step 26
    SSD.pstring(0,0, i, "Micromite MK2:Invert:Micromite MK2:Invert", 2, rgb(red), rgb(black))
  next i
  pause 2000
  SSD.setrot(portrait180)
  SSD.fillscreen(rgb(black))
  for i=0 to wi-39 step 39
    SSD.pstring(0,0,i, "Rotate right:Rotate right:Rotate right", 3, rgb(yellow), rgb(black))
  next i
  pause 2000
  SSD.setrot(portrait)
  SSD.fillscreen(rgb(black))
  for i=0 to wi-52 step 52
    SSD.pstring(0,0,i, "Rotate Left:Rotate Left:Rotate Left", 4, rgb(blue), rgb(black))
  next i
  pause 2000
  SSD.fillscreen(rgb(black))
  ' test drawing rectangles
  for i=0 to 3
     SSD.setrot(i)
     SSD.frect(5, 5, 80, 90, rgb(blue))
     SSD.drect(2, 2, 86, 96, rgb(red))
  next i
  pause 2000
' test drawing rounded rectangles
  SSD.fillscreen(rgb(black))
  for i=0 to 3
     SSD.setrot(i)
     SSD.frndrect(5, 5, 80, 45, 10, rgb(blue))
     SSD.drndrect(2, 2, 86, 51, 12, rgb(red))
  next i
  pause 2000
  SSD.fillscreen(rgb(black))
' test drawing circles
  for i=0 to 3
     SSD.setrot(i)
     SSD.fcirc(40, 40, 35, rgb(blue))
     SSD.dcirc(40, 40, 39, rgb(red))
  next i
  pause 2000
' test drawing triangles and line drawing
  SSD.fillscreen(rgb(black))
  for i=0 to 3
     SSD.setrot(i)
     SSD.ftri(4, 4, 140, 6, 22, 124, rgb(blue))
     SSD.dtri(1, 2, 144, 4, 22, 127, rgb(red))
  next i
  pause 2000
  SSD.setrot(landscape180)
  SSD.fillscreen(rgb(black))
  ssd.drect(scrollxstart-1,scrollystart-1,scrollxwidth+2,scrollyheight+2,rgb(red))
  ssd.pstring(0,scrollxstart+8,0,"Micromite MKII 4.7b8 with SSD1963",1,rgb(white),rgb(black))
  ssd.pstring(0,newpoint+10,scrollystart-13+(scrollyheight\2),"Fixed",2,rgb(green),rgb(black))
  ssd.pstring(0,newpoint+10,scrollystart+13+(scrollyheight\2),"Area",2,rgb(green),rgb(black))
  ssd.pstring(0,10,scrollystart-13+(scrollyheight\2),"Fixed",2,rgb(green),rgb(black))
  ssd.pstring(0,10,scrollystart+13+(scrollyheight\2),"Area",2,rgb(green),rgb(black))
  ssd.pstring(0,scrollxstart+12,scrollystart+scrollyheight+5,"Demonstration of scrolling data",1,rgb(white),rgb(black))
 timer=0
 do
   ' scroll an area defined by the x-y coordinates of the top left point
   ' the width of the area to be scrolled
   ' the height of the are to be scrolled
   ' the scroll driection (NOT YET IMPLEMENTED)
   ' a buffer used to store the data as it is read and written back to the display
   ' the array containing display specific information set up during initialisation
   ssd.scroll(scrollxstart,scrollystart,scrollxwidth,scrollyheight)
   ssd.pixel(newpoint,xp,rgb(black)) ' clear the old data after the scroll
   ssd.pixel(newpoint,rp,rgb(black))
   ssd.pixel(newpoint,yp,rgb(black))
   ssd.pixel(newpoint,bp,rgb(black))
   ssd.pixel(newpoint,wp,rgb(black))
   ssd.pixel(newpoint,cp,rgb(black))
   ssd.pixel(newpoint,zp,rgb(black))
   rp=rp+(rnd()*4)-2 'get the new data points
   bp=bp+(rnd()*4)-2
   yp=yp+(rnd()*4)-2
   xp=xp+(rnd()*2)-1
   zp=zp+(rnd()*4)-2
   wp=wp+(rnd()*4)-2
   cp=cp+(rnd()*4)-2
   ssd.pixel(newpoint,xp,rgb(yellow)) 'write them to the space left
   ssd.pixel(newpoint,rp,rgb(blue))
   ssd.pixel(newpoint,yp,rgb(red))
   ssd.pixel(newpoint,bp,rgb(green))
   ssd.pixel(newpoint,wp,rgb(magenta))
   ssd.pixel(newpoint,cp,rgb(cyan))
   ssd.pixel(newpoint,zp,rgb(white))
loop UNTIL timer>60000
  T.calibrate 'calibrate the touch sensor
  xleft=xleft0: xright=xright0 : ytop=ytop0 : ybottom=ybottom0 ' set the parameters for the default orientation
  'for a particular application these values can be saved and set as constants
  for j=0 to 3
     T.rotate(j)
     SSD.fillscreen(0)
     SSD.pstring(0,0,0,"Top left",2,rgb(green),0)
     k=0
main:
     if t.screenpos(x,y,1000) then
       if x>3 and x<T_width-3 and y>3 and y<T_height-3 then SSD.fcirc(x,y,3,rgb(white))
     else 
        k=k+1
     endif
     if k<15 goto main
  next j
end SUB
