/*******************************************************************************
 *
 * Driver for SSD1306 Display written as CFunctions
 *
 * Author: Peter Mather 2015 with acknowledgements to Peter Carnegie & Geoff Graham
 * 
 *
 * This CFunction MUST be compiled with Optimization Level 1, -O1
 * -O2,-O3,-Os will compile successfully, but generate exceptions at runtime.
 *
 * When Generating the CFunction, use MERGE CFunction mode, and name the CFunction
 * SSD1306
 *
 * Entry point is function long long main(long long *MyAddress,
 *                                        long long *DC,
 *                                        long long *RST
 *                                        long long *CS
 *                                        long long *size //zero for 0.96", non-zero for 1.3"
 *                                        long long *orientation)
 *
 * V1.0     2015-10-15 Peter Mather
 * 
 ******************************************************************************/
#include <stdarg.h>

#define Version 100     //Version 1.00
#define _SUPPRESS_PLIB_WARNING                                      // required for XC1.33  Later compiler versions will need PLIB to be installed
#include <plib.h>                                                   // the pre Harmony peripheral libraries
#include "cfunctions.h"

#define LANDSCAPE       1
#define PORTRAIT        2
#define RLANDSCAPE      3
#define RPORTRAIT       4

#define screenwidth 128
#define screenheight 64
#define screenrows screenheight/8
#define update screenwidth*screenrows //location for status to force screen write
#define displaysize update+1 //location for screen size 

// SPI pin numbers and registers
#define SPI_INP_PIN         (HAS_44PINS ? 41 : 14)
#define SPI_OUT_PIN         (HAS_44PINS ? 20 :  3)
#define SPI_CLK_PIN         (HAS_44PINS ? 14 : 25)
#define SPI_PPS_OPEN        PPSInput(2, SDI1, RPB5); PPSOutput(2, RPA1, SDO1)
#define SPI_PPS_CLOSE       PPSOutput(2, RPA1, NULL)
#define SPICON *(volatile unsigned int *)(0xbf805800)               //SPI status register
#define SPISTAT *(volatile unsigned int *)(0xbf805810)              //SPI status register
#define SPIBUF *(volatile unsigned int *)(0xbf805820)               //SPI output buffer
#define SPIBRG *(volatile unsigned int *)(0xbf805830)               //SPI output buffer
#define SPICON2 *(volatile unsigned int *)(0xbf805840)              //SPI status register

#define SPIsend(a) {int j;SPIBUF=a; while((SPISTAT & 0x80)==0); j=SPIBUF;}

void spi_write_command(unsigned char data){
    PinSetBit(Option->LCD_CD, LATCLR);
    PinSetBit(Option->LCD_CS, LATCLR);
    SPIsend(data);
    PinSetBit(Option->LCD_CS, LATSET);
}
void OLED_setxy(int x,int y){
  unsigned char* p=(void *)(unsigned int)(StartOfCFuncRam);
  if(p[displaysize])x+=2;
  spi_write_command(0xB0+y)        ; //set page address 
  spi_write_command(0x10 | ((x>>4) & 0x0F))  ; //set high col address 
  spi_write_command(x & 0x0f)     ; //set low col address 
}

void updatedisplay(void){
    char* p=(void *)(unsigned int)(StartOfCFuncRam);
    if(p[update]){
        int m,n;
        unsigned int consave=0,brgsave=0,con2save;
        p[update]=0;
        brgsave=SPIBRG; //save any user SPI setup
        consave=SPICON;
        con2save=SPICON2;
        SPICON=0x8060;
        SPIBRG=2;
        SPICON2=0xC00;
        for(n=0;n<screenrows;n++){
            OLED_setxy(0,n);
            PinSetBit(Option->LCD_CS, LATCLR);
            PinSetBit(Option->LCD_CD, LATSET);
            for(m=n*screenwidth;m<(n+1)*screenwidth;m++){
                SPIsend(p[m]);
            }
            PinSetBit(Option->LCD_CS, LATCLR);
        } 
        SPIBRG=brgsave;  //restore user (or my) setup
        SPICON=consave;
        SPICON2=con2save;
    }
}
long long DrawRectangleMEM(int x1, int y1, int x2, int y2, int c){
    unsigned char* p=(void *)(unsigned int)(StartOfCFuncRam);
    int i,j,loc,t;
    unsigned char mask;
    if(Option->DISPLAY_ORIENTATION==PORTRAIT){
        t=x1;
        x1=screenwidth-y2-1;
        y2=t;
        t=x2;
        x2=screenwidth-y1-1;
        y1=t;
    }
     if(Option->DISPLAY_ORIENTATION==RLANDSCAPE){
        x1=screenwidth-x1-1;
        x2=screenwidth-x2-1;
        y1=screenheight-y1-1;
        y2=screenheight-y2-1;
    }
   if(Option->DISPLAY_ORIENTATION==RPORTRAIT){
        t=y1;
        y1=screenheight-x1-1;
        x1=t;
        t=y2;
        y2=screenheight-x2-1;
        x2=t;
    }
    if(x2 <= x1) { t = x1; x1 = x2; x2 = t; }
    if(y2 <= y1) { t = y1; y1 = y2; y2 = t; }
    if(x1 < 0) x1 = 0; if(x1 >= screenwidth) x1 = screenwidth - 1;
    if(x2 < 0) x2 = 0; if(x2 >= screenwidth) x2 = screenwidth - 1;
    if(y1 < 0) y1 = 0; if(y1 >= screenheight) y1 = screenheight - 1;
    if(y2 < 0) y2 = 0; if(y2 >= screenheight) y2 = screenheight - 1;
        
     for(i=x1;i<=x2;i++){
        for(j=y1;j<=y2;j++){
           loc=i+(j/8)*screenwidth; //get the byte address for this bit
           mask=1<<(j % 8); //get the bit position for this bit
           if(c){
               p[loc]|=mask;
           } else {
               p[loc]&=(~mask);
           }
        }
    }
    p[update]=1;
}
void DrawBitmapMEM(int x1, int y1, int width, int height, int scale, int fc, int bc, unsigned char *bitmap){
    int i, j, k, m, x, y,t, loc;
    unsigned char omask, amask;
    unsigned char* p=(void *)(unsigned int)(StartOfCFuncRam);
    for(i = 0; i < height; i++) {                                   // step thru the font scan line by line
        for(j = 0; j < scale; j++) {                                // repeat lines to scale the font
            for(k = 0; k < width; k++) {                            // step through each bit in a scan line
                for(m = 0; m < scale; m++) {                        // repeat pixels to scale in the x axis
                    x=x1 + k * scale + m ;
                    y=y1 + i * scale + j ;
                    if(Option->DISPLAY_ORIENTATION==PORTRAIT){
                        t=x;
                        x=VRes-y-1;
                        y=t;
                    }
                    if(Option->DISPLAY_ORIENTATION==RLANDSCAPE){
                        x=HRes-x-1;
                        y=VRes-y-1;
                    }
                    if(Option->DISPLAY_ORIENTATION==RPORTRAIT){
                        t=y;
                        y=HRes-x-1;
                        x=t;
                    }
                    loc=x+(y/8)*screenwidth; //get the byte address for this bit
                    omask=1<<(y % 8); //get the bit position for this bit
                    amask=~omask;
                    if(x >= 0 && x < screenwidth && y >= 0 && y < screenheight) {  // if the coordinates are valid
                        if((bitmap[((i * width) + k)/8] >> (((height * width) - ((i * width) + k) - 1) %8)) & 1) {
                            if(fc){
                                p[loc]|=omask;
                             } else {
                                p[loc]&=amask;
                             }
                       } else {
                            if(bc){
                                p[loc]|=omask;
                             } else {
                                p[loc]&=amask;
                             }
                        }
                   }
                }
            }
        }
    }
    p[update]=1;
}

//CFunction Driver_SSD1306
long long main(long long *MyAddress, long long *CD, long long *RST, long long *CS,long long *size, long long *orientation){
    int i,DrawRectangleVectorOffset,DrawBitmapVectorOffset,updatedisplayVectorOffset,HorizontalRes=screenwidth,VerticalRes=screenheight;
    unsigned int consave=0,brgsave=0,con2save;
    Option->LCD_Reset=*RST;
    Option->LCD_CD=*CD;
    Option->LCD_CS=*CS;
    Option->DISPLAY_ORIENTATION=*orientation;
    //Get some persistent memory
    unsigned char * p = GetMemory(HorizontalRes*VerticalRes/8+256);
    //Save the address for other CFunctions
    StartOfCFuncRam=(unsigned int)(p);
    p[displaysize]=*size;

    ExtCfg(Option->LCD_Reset,EXT_DIG_OUT,0);ExtCfg(Option->LCD_Reset,EXT_BOOT_RESERVED,0);
    PinSetBit(Option->LCD_Reset, LATSET);
    ExtCfg(Option->LCD_CD,EXT_DIG_OUT,0);ExtCfg(Option->LCD_CD,EXT_BOOT_RESERVED,0);
    PinSetBit(Option->LCD_CD, LATSET);
    ExtCfg(Option->LCD_CS,EXT_DIG_OUT,0);ExtCfg(Option->LCD_CS,EXT_BOOT_RESERVED,0);
    PinSetBit(Option->LCD_CS, LATSET);
    if(ExtCurrentConfig[SPI_OUT_PIN] == EXT_RESERVED) { //already open
        brgsave=SPIBRG;
        consave=SPICON;
        con2save=SPICON2;
    }
    ExtCfg(SPI_OUT_PIN, EXT_DIG_OUT, 0); ExtCfg(SPI_OUT_PIN, EXT_BOOT_RESERVED, 0);
    ExtCfg(SPI_INP_PIN, EXT_DIG_IN, 0); ExtCfg(SPI_INP_PIN, EXT_BOOT_RESERVED, 0);
    ExtCfg(SPI_CLK_PIN, EXT_DIG_OUT, 0); ExtCfg(SPI_CLK_PIN, EXT_BOOT_RESERVED, 0);
    SPI_PPS_OPEN; 
    SPICON=0x8060;
    SPIBRG=2;
    SPICON2=0xC00;// this is defined in IOPorts.h
    if(!brgsave){ //save my settings
        brgsave=SPIBRG;
        consave=SPICON;
        con2save=SPICON2;
    }    


   //Calculate the address vectors
    if ((unsigned int)&DrawRectangleMEM < (unsigned int)&main){
        DrawRectangleVectorOffset=*MyAddress - ((unsigned int)&main - (unsigned int)&DrawRectangleMEM);
    }else{
        DrawRectangleVectorOffset=*MyAddress + ((unsigned int)&DrawRectangleMEM - (unsigned int)&main);
    }
    
    if ((unsigned int)&DrawBitmapMEM < (unsigned int)&main){
        DrawBitmapVectorOffset=*MyAddress - ((unsigned int)&main - (unsigned int)&DrawBitmapMEM);
    }else{
        DrawBitmapVectorOffset=*MyAddress + ((unsigned int)&DrawBitmapMEM - (unsigned int)&main);
    }
    if ((unsigned int)&updatedisplay < (unsigned int)&main){
        updatedisplayVectorOffset=*MyAddress - ((unsigned int)&main - (unsigned int)&updatedisplay);
    }else{
        updatedisplayVectorOffset=*MyAddress + ((unsigned int)&updatedisplay - (unsigned int)&main);
    }

   //Reset the SSD1963
    PinSetBit(Option->LCD_Reset,LATSET);
    uSec(10000);
    PinSetBit(Option->LCD_Reset,LATCLR);
    uSec(10000);
    PinSetBit(Option->LCD_Reset,LATSET);
    uSec(10000);
    spi_write_command(0xAE); //DISPLAYOFF) 
    spi_write_command(0xD5); //DISPLAYCLOCKDIV 
    spi_write_command(0x80); //the suggested ratio 0x80 
    spi_write_command(0xA8); //MULTIPLEX 
    spi_write_command(0x3F); // 
    spi_write_command(0xD3); //DISPLAYOFFSET 
    spi_write_command(0x0); //no offset 
    spi_write_command(0x40); //STARTLINE 
    spi_write_command(0x8D); //CHARGEPUMP 
    spi_write_command(0x14) ;
    spi_write_command(0x20); //MEMORYMODE 
    spi_write_command(0x00); //0x0 act like ks0108 
    spi_write_command(0xA1); //SEGREMAP OR 1 
    spi_write_command(0xC8); //COMSCANDEC 
    spi_write_command(0xDA); //COMPINS 
    spi_write_command(0x12) ;
    spi_write_command(0x81); //SETCONTRAST 
    spi_write_command(0xCF) ;
    spi_write_command(0xd9); //SETPRECHARGE 
    spi_write_command(0xF1) ;
    spi_write_command(0xDB); //VCOMDETECT 
    spi_write_command(0x40) ;
    spi_write_command(0xA4); //DISPLAYALLON_RESUME 
    spi_write_command(0xA6); //NORMALDISPLAY 
    spi_write_command(0xAF); //DISPLAYON

    if(Option->DISPLAY_ORIENTATION&1){
        HRes=screenwidth;
        VRes=screenheight;
    } else {
        VRes=screenwidth;
        HRes=screenheight;
    }

   //Set the DrawRectangle vector to point to our function
    DrawRectangleVector=DrawRectangleVectorOffset;


    //Set the DrawBitmap vector to point to our function
    DrawBitmapVector=DrawBitmapVectorOffset;
	
	//Set the end of Basic statement vector to point to our routine
    
    CFuncmInt=updatedisplayVectorOffset;
    
    //CLS
    DrawRectangle(0,0,HRes-1,VRes-1,0);
    SPIBRG=brgsave; //restore user (or my) setup
    SPICON=consave;
    SPICON2=con2save;

    return (unsigned int)p;

}