/*******************************************************************************
 *
 * Driver for ILI9325 Display written as CFunctions
 *
 * (c) Peter Mather 2015 with acknowledgements to Peter Carnegie & Geoff Graham
 * 
 *
 * This CFunction MUST be compiled with Optimization Level 1, -O1
 * -O2,-O3,-Os will compile successfully, but generates Exceptions at runtime.
 *
 * When Generating the CFunction, use MERGE CFunction mode, and name the CFunction
 * Driver_ILI9325
 *
 * Entry point is function long long main(long long *MyAddress,
 *                                        long long *reset,
 *                                        long long *orientation)
 *
 * V1.0     2015-09-7 Peter Mather
 * V1.01   Improved performance of CLS for black and white
 * 
 ******************************************************************************/

#define Version 101     //Version 1.01

#include "../APIDefs.h"

#define cport *(volatile unsigned int *)(0xbf886230) //latch registers
#define cclrport *(volatile unsigned int *)(0xbf886234) //latch registers
#define csetport *(volatile unsigned int *)(0xbf886238) //latch registers

//Offsets in PinDefs of Control Signals
#define RS_Pin 0x100
#define WR_Pin 0x200
#define Both 0x300

//Offsets into the persistent RAM of variables

#define RSLo    cclrport=RS_Pin
#define RSHi    csetport=RS_Pin

#define WRLo    cclrport=WR_Pin
#define WRHi    csetport=WR_Pin


#define LANDSCAPE       1
#define PORTRAIT        2
#define RLANDSCAPE      3
#define RPORTRAIT       4

/*******************************************************************************
 *
 * Write Data to a register on the Chip
 *
 ******************************************************************************/

void writeRegister(unsigned int command, int data){
    cport=(((command>>8) & 0x2FF) | WR_Pin);
    WRLo;
    WRHi;
    cport=(((command) & 0x2FF) | WR_Pin);
    WRLo;
    WRHi;
    if(data!=-1){
    cport=(((data>>8) & 0x3FF) | Both);
    WRLo;
    WRHi;
    cport=(((data) & 0x3FF) | Both);
    WRLo;
    WRHi;
    }
    return;
}
/*******************************************************************************
 *
 * defines start/end coordinates for memory access from host to ILI9325
 * also maps the start and end points to suit the orientation
 *
 * This function is a modified version of the function inside the MMBasic Interpreter
 * for MM+ on 'MX470 chips
 *
*******************************************************************************/
void defineregion(long x, long y, long width,long height){ //ILI9325
    long x1=x,x2=x+width-1,y1=y,y2=y+height-1;
    unsigned long xstart,xend,ystart,yend,xp,yp,Vertical,Horizontal;
    if(HRes<VRes){
        Vertical=VRes;
        Horizontal=HRes;
    }
    else {
        Vertical=HRes;
        Horizontal=VRes;
    }
     if(Option->DISPLAY_ORIENTATION==PORTRAIT){
              xstart = x1;
              xend = x2;
              ystart = y1;
              yend = y2;
              xp=xstart; yp=ystart;
      }
     if(Option->DISPLAY_ORIENTATION==RPORTRAIT){
              xstart = (Horizontal - 1) - x2;
              xend = (Horizontal - 1) - x1;
              ystart = (Vertical - 1) - y2;
              yend = (Vertical - 1) - y1;
              xp=xend; yp=yend;
      }
    if(Option->DISPLAY_ORIENTATION==LANDSCAPE){
              xstart = (Horizontal - 1) - y2;
              xend = (Horizontal - 1) - y1;
              ystart = x1;
              yend = x2;
              xp=xend; yp=ystart;
    }
     if(Option->DISPLAY_ORIENTATION==RLANDSCAPE){
              xstart = y1;
              xend = y2;
              ystart = (Vertical - 1) - x2;
              yend = (Vertical - 1) - x1;
              xp=xstart; yp=yend;
       }
    writeRegister(0x0020,xstart);
    writeRegister(0x0021,ystart);
    writeRegister(0x0050,xstart);
    writeRegister(0x0052,ystart);
    writeRegister(0x0051,xend);
    writeRegister(0x0053,yend);
    writeRegister(0x0022,-1);
}



/***********************************************************************************************
 * Display the bitmap of a char on the TFT panel
 *
 * This function is NEVER called by MMBasic programs
 *
 * This is optimised for the
 *      x, y - the top left of the char
 *      width, height - size of the char's bitmap
 *      scale - how much to scale the bitmap
 * 	fg, bg - foreground and background colour
 *      bitmap - pointer to the butmap
 *
 * This function is a modified version of the function inside the MMBasic Interpreter
 * for MM+ on 'MX470 chips
***********************************************************************************************/
void DrawBitmap_ILI9325(int x1, int y1, int width, int height, int scale, int fc, int bc, unsigned char *bitmap ){
    long i, j, k, m;
    unsigned int fch,fcm,bch,bcm,xtest, ytest;
    xtest=HRes;
    ytest=VRes;
    fch = ((fc >> 16) & 0b11111000) | ((fc >> 13) & 0b00000111)| Both;
    fcm = ((fc >> 5) & 0b11100000) | ((fc >> 3) & 0b00011111)| Both;
    bch = ((bc >> 16) & 0b11111000) | ((bc >> 13) & 0b00000111)| Both;
    bcm = ((bc >> 5) & 0b11100000) | ((bc >> 3) & 0b00011111)| Both;

    defineregion(x1,y1,width*scale,height*scale);
    for(i = 0; i < height; i++) {                                   // step thru the font scan line by line
        for(j = 0; j < scale; j++) {                                // repeat lines to scale the font
            for(k = 0; k < width; k++) {                            // step through each bit in a scan line
                for(m = 0; m < scale; m++) {                        // repeat pixels to scale in the x axis
                    if(x1 + k * scale + m >= 0 && x1 + k * scale + m < xtest && y1 + i * scale + j >= 0 && y1 + i * scale + j < ytest) {  // if the coordinates are valid
                        if((bitmap[((i * width) + k)/8] >> (((height * width) - ((i * width) + k) - 1) %8)) & 1) {
                            cport=fch;WRLo;WRHi;
                            cport=fcm;WRLo;WRHi;
                        } else {
                            cport=bch;WRLo;WRHi;
                            cport=bcm;WRLo;WRHi;
                        }
                    }
                }
            }
        }
    }
}

/*******************************************************************************
 *
 * Called by MMBasic Interpreter to draw a rectangle
 * on the ILI9325 panel
 *
 * This function is NEVER called by MMBasic programs
 *
 *
 ******************************************************************************/

void DrawRectangle_ILI9325(int x1, int y1, int x2, int y2, int c){
    int t;
    long width, height;
    unsigned long i,hb,lb;
    // make sure the coordinates are kept within the display area
    if(x2 <= x1) { t = x1; x1 = x2; x2 = t; }
    if(y2 <= y1) { t = y1; y1 = y2; y2 = t; }
    if(x1 < 0) x1 = 0; if(x1 >= HRes) x1 = HRes - 1;
    if(x2 < 0) x2 = 0; if(x2 >= HRes) x2 = HRes - 1;
    if(y1 < 0) y1 = 0; if(y1 >= VRes) y1 = VRes - 1;
    if(y2 < 0) y2 = 0; if(y2 >= VRes) y2 = VRes - 1;
    width=x2-x1+1;height=y2-y1+1;
    i=width*height;
    defineregion(x1,y1,width,height);
    // convert the colours to 565 format
    hb = ((c >> 16) & 0b11111000) | ((c >> 13) & 0b00000111)| Both;
    lb = ((c >> 5) & 0b11100000) | ((c >> 3) & 0b00011111)| Both;
    if(hb==lb){
       cport=hb;    WRLo;    WRHi;
       WRLo;    WRHi;
       i--;
       while (i--){
          WRLo;
          WRHi;
          WRLo;
          WRHi;
       }
    }else {
        while (i--) {
            cport=hb;    WRLo;    WRHi;
            cport=lb;    WRLo;    WRHi;
        }
    }
}

/*******************************************************************************
 *
 * Driver_ILI9325 : Initialise the CFunction Driver Sub-System
 *
 * Function called to initialise the driver SubSystem
 *
 * OpenDriver_ILI9325 is ALWAYS called from an MMBasic program
 * On exit, vectors DrawRectangleVector, and DrawBitmapVector will
 * be set to point to the CFunctions DrawRectangle_ILI9325 and
 * DrawBitmap_ILI9325 respectively
 *
 * Input Arguments
 * MyAddress    The Address of the CFunction, use PEEK(CFUNADDR Driver_ILI9325)
 *
 * func         0=Open Driver, 1=Close Driver, 2=Return Version Number
 *
 * PinDefs[] contains the pin definitions for the I/O to be used to interface to the ILI9325 display
 *  PinDefs[0 .. 8] pin numbers of DB0-DB8
 *  PinDefs[8] pin number of RS
 *  PinDefs[9] pin number of WR
 *  PinDefs[10] pin number of RESET
 *
 * Rotation
 *   is 0 for Portrait orientation, 1 for reverse portrait, 2 for landscape, 3 for reverse landscape
 *
 * Exit Value   Version
 *
 * Note, For added performance, it may be desirable to create a
 * separate WriteData function which doen't asser/de-asser CS for each byte and
 * instead use block CSLo/CSHi around data transfers. These "block" CSLo/CSHi
 * statements have been left in the code as comments.
 * 
 ******************************************************************************/
//CFunction Driver_ILI9325
long long main(long long *MyAddress, long long *RST,long long *orientation){
    //Close Driver Request ?
   int t,DrawRectangleVectorOffset,DrawBitmapVectorOffset,HorizontalRes,VerticalRes;


    int i;
    unsigned int p[8];

    Option->LCD_Reset=*RST;
    Option->DISPLAY_ORIENTATION=*orientation;
    //DataBus
    p[0]=25;
    p[1]=26;
    p[2]=27;
    p[3]=36;
    p[4]=37;
    p[5]=38;
    p[6]=2;
    p[7]=3;
    for(i=0;i<+8;i++){
        ExtCfg(p[i],EXT_DIG_OUT,0);ExtCfg(p[i],EXT_BOOT_RESERVED,0);
        PinSetBit(p[i],LATCLR);
    }

    //Control Signals
    ExtCfg(4,EXT_DIG_OUT,0);ExtCfg(4,EXT_BOOT_RESERVED,0);
    RSHi;

    ExtCfg(5,EXT_DIG_OUT,0);ExtCfg(5,EXT_BOOT_RESERVED,0);
    WRHi;

    ExtCfg(Option->LCD_Reset,EXT_DIG_OUT,0);ExtCfg(Option->LCD_Reset,EXT_BOOT_RESERVED,0);
    PinSetBit(Option->LCD_Reset, LATSET);
    if(Option->LCD_CS) PinSetBit(Option->LCD_CS, LATSET);
    //Calculate the address vectors
    if ((unsigned int)&DrawRectangle_ILI9325 < (unsigned int)&main){
        DrawRectangleVectorOffset=*MyAddress - ((unsigned int)&main - (unsigned int)&DrawRectangle_ILI9325);
    }else{
        DrawRectangleVectorOffset=*MyAddress + ((unsigned int)&DrawRectangle_ILI9325 - (unsigned int)&main);
    }

    if ((unsigned int)&DrawBitmap_ILI9325 < (unsigned int)&main){
        DrawBitmapVectorOffset=*MyAddress - ((unsigned int)&main - (unsigned int)&DrawBitmap_ILI9325);
    }else{
        DrawBitmapVectorOffset=*MyAddress + ((unsigned int)&DrawBitmap_ILI9325 - (unsigned int)&main);
    }


    //Save DisplayMetrics
    HorizontalRes=240;
    VerticalRes=320;

    /************************************************
     * 
     * Initialise the ILI9325
     * 
     ***********************************************/

    //Reset the ILI9325
    PinSetBit(Option->LCD_Reset,LATSET);
    uSec(50000);
    PinSetBit(Option->LCD_Reset,LATCLR);
    uSec(50000);
    PinSetBit(Option->LCD_Reset,LATSET);
    uSec(50000);
    writeRegister(0x00E5,0x78F0); // set SRAM internal timing
    writeRegister(0x0001,0x0100); // set SS and SM bit
    writeRegister(0x0002,0x0700); // set 1 line inversion
    writeRegister(0x0003,0x1030); // set GRAM write direction and BGR=1.
    writeRegister(0x0004,0x0000); // Resize register
    writeRegister(0x0008,0x0207); // set the back porch and front porch
    writeRegister(0x0009,0x0000); // set non-display area refresh cycle ISC[3:0]
    writeRegister(0x000A,0x0000); // FMARK function
    writeRegister(0x000C,0x0000); // RGB interface setting
    writeRegister(0x000D,0x0000); // Frame marker Position
    writeRegister(0x000F,0x0000); // RGB interface polarity
    //*************Power On sequence ****************//
    writeRegister(0x0010,0x0000); // SAP, BT[3:0], AP, DSTB, SLP, STB
    writeRegister(0x0011,0x0007); // DC1[2:0], DC0[2:0], VC[2:0]
    writeRegister(0x0012,0x0000); // VREG1OUT voltage
    writeRegister(0x0013,0x0000); // VDV[4:0] for VCOM amplitude
    writeRegister(0x0007,0x0001);
    uSec(50000); // Dis-charge capacitor power voltage
    writeRegister(0x0010,0x1090); // 1490//SAP, BT[3:0], AP, DSTB, SLP, STB
    writeRegister(0x0011,0x0227); // DC1[2:0], DC0[2:0], VC[2:0]
    uSec(50000); // Delay 50ms
    writeRegister(0x0012,0x001F); //001C// Internal reference voltage= Vci;
    uSec(50000); // Delay 50ms
    writeRegister(0x0013,0x1500); //0x1000//1400   Set VDV[4:0] for VCOM amplitude  1A00
    writeRegister(0x0029,0x0027); //0x0012 //001a  Set VCM[5:0] for VCOMH  //0x0025  0034
    writeRegister(0x002B,0x000D); // Set Frame Rate   000C
    uSec(50000); // Delay 50ms
    writeRegister(0x0020,0x0000); // GRAM horizontal Address
    writeRegister(0x0021,0x0000); // GRAM Vertical Address
    // ----------- Adjust the Gamma Curve ----------//
    writeRegister(0x0030,0x0000);
    writeRegister(0x0031,0x0707);
    writeRegister(0x0032,0x0307);
    writeRegister(0x0035,0x0200);
    writeRegister(0x0036,0x0008);//0207
    writeRegister(0x0037,0x0004);//0306
    writeRegister(0x0038,0x0000);//0102
    writeRegister(0x0039,0x0707);//0707
    writeRegister(0x003C,0x0002);//0702
    writeRegister(0x003D,0x1D04);//1604
    //------------------ Set GRAM area ---------------//
    writeRegister(0x0050,0x0000); // Horizontal GRAM Start Address
    writeRegister(0x0051,0x00EF); // Horizontal GRAM End Address
    writeRegister(0x0052,0x0000); // Vertical GRAM Start Address
    writeRegister(0x0053,0x013F); // Vertical GRAM Start Address
    writeRegister(0x0060,0xA700); // Gate Scan Line
    writeRegister(0x0061,0x0001); // NDL,VLE, REV
    writeRegister(0x006A,0x0000); // set scrolling line
    //-------------- Partial Display Control ---------//
    writeRegister(0x0080,0x0000);
    writeRegister(0x0081,0x0000);
    writeRegister(0x0082,0x0000);
    writeRegister(0x0083,0x0000);
    writeRegister(0x0084,0x0000);
    writeRegister(0x0085,0x0000);
    //-------------- Panel Control -------------------//
    writeRegister(0x0090,0x0010);
    writeRegister(0x0092,0x0600);
    writeRegister(0x0007,0x0133); // 262K color and display ON    //Set Hres and VRes
    if(Option->DISPLAY_ORIENTATION==PORTRAIT){
              i=0x1030;
    }
    if(Option->DISPLAY_ORIENTATION==RPORTRAIT){
              i=0x1000;
    }
    if(Option->DISPLAY_ORIENTATION==LANDSCAPE){
              i=0x1028;
    }
    if(Option->DISPLAY_ORIENTATION==RLANDSCAPE){
              i=0x1018;
    }
    writeRegister(0x03,i);
       if(Option->DISPLAY_ORIENTATION&1){ //landscape is not default
                 VRes=HorizontalRes;
                 HRes=VerticalRes;
       } else {
                 HRes=HorizontalRes;
                 VRes=VerticalRes;
       }

    //Save the current values of the Vectors

    //Hook the DrawRectangle vector to point to our function
    DrawRectangleVector=DrawRectangleVectorOffset;

    //Draw a black box  = CLS
    DrawRectangle(0,0,HRes-1,VRes-1,0x000000);

    //Hook the DrawRectangle vector to point to our function
    DrawBitmapVector=DrawBitmapVectorOffset;

    return 0;

}
