/*******************************************************************************
 *
 * Driver for SSD1963 Display written as CFunctions
 *
 * (c) Peter Mather 2015 with acknowledgements to Peter Carnegie & Geoff Graham
 * 
 *
 * This CFunction MUST be compiled with Optimization Level 1, -O1
 * -O2,-O3,-Os will compile successfully, but generate exceptions at runtime.
 *
 * When Generating the CFunction, use MERGE CFunction mode, and name the CFunction
 * SSD1963_V44
 *
 * Entry point is function long long main(long long *MyAddress,
 *                                        long long *RST,
 *                                        long long *orientation
 *                                        long long *size)
 *
 * V1.0     2015-07-21 Peter Mather
 * 1.01     Change orientation codes to 1:landscape, 2:portrait, 3:rlandscape, 4:rportrait
 * 1.02     Use fixed PortC pinout for maximum performance
 * 1.03     Stripped routines for use as loadable LIBRARY
 * 1.04     Conditional compilation for 44 and 64-pin parts
 * 1.1      Convert to Nathan approach
 * 1.11     Minor change for compatibility with touch and restrincting page usage
 * 
 ******************************************************************************/
#define Version 111     //Version 1.11
#include <stdarg.h>
#include <xc.h>			   // Required for SFR defs
#include <sys/attribs.h>   // Required to use __longramfunc__
#include "../CFunctions.h"
//#define MX470 //Comment out for compilation for 44-pin part
#define bclrport *(volatile unsigned int *)(0xbf886134) //latch registers
#define bsetport *(volatile unsigned int *)(0xbf886138) //latch registers
#define cport *(volatile unsigned int *)(0xbf886230) //latch registers
#define cread *(volatile unsigned int *)(0xbf886220) //latch registers
#define ctrisinv *(volatile unsigned int *)(0xbf88621C) //latch registers
#define cclrport *(volatile unsigned int *)(0xbf886234) //latch registers
#define csetport *(volatile unsigned int *)(0xbf886238) //latch registers
#define eport *(volatile unsigned int *)(0xbf886430) //latch registers
#define eread *(volatile unsigned int *)(0xbf886420) //latch registers
#define etrisinv *(volatile unsigned int *)(0xbf88641C) //latch registers
#define eclrport *(volatile unsigned int *)(0xbf886434) //latch registers
#define esetport *(volatile unsigned int *)(0xbf886438) //latch registers
#define DEVID (*(volatile unsigned int *)0xBF80F220)

//Offsets into the persistent RAM of variables
#ifdef MX470
    #define RS_Pin_No 27
    #define WR_Pin_No 24
    #define RS_Pin 0x1000
    #define WR_Pin 0x0800
    #define clrport bclrport
    #define setport bsetport
    #define port eport
    #define read eread
    #define trisinv etrisinv
    #define RSLo    {clrport=RS_Pin;}
    #define RSHi    {setport=RS_Pin;}
    #define WRLo    {clrport=WR_Pin;}
    #define WRHi    {setport=WR_Pin;}
#else
    #define RS_Pin_No 4
    #define WR_Pin_No 5
    #define RS_Pin 0x100
    #define WR_Pin 0x200
    #define clrport cclrport
    #define setport csetport
    #define port cport
    #define read cread
    #define trisinv ctrisinv
    #define RSLo    clrport=RS_Pin
    #define RSHi    setport=RS_Pin
    #define WRLo    clrport=WR_Pin
    #define WRHi    setport=WR_Pin
#endif
#define Both WR_Pin | RS_Pin
#define RDLo    (*(volatile unsigned int *)RDclrport)=RDpin
#define RDHi    (*(volatile unsigned int *)RDsetport)=RDpin
#define LANDSCAPE       1
#define PORTRAIT        2
#define RLANDSCAPE      3
#define RPORTRAIT       4

/*******************************************************************************
 *
 * Write Data to a register on the Chip
 *
 ******************************************************************************/
void write_command_data(unsigned int command, int data, ...){
   int i;
#ifdef MX470
        RSLo;
#endif
   va_list ap;
   va_start(ap, data);
   port=(command & 0x00FF) | WR_Pin;  WRLo;    WRHi; // RS low
#ifdef MX470
        RSHi;
#endif
   for(i = 0; i < data; i++) {
       port= (char)va_arg(ap, int) | Both;  WRLo;    WRHi; //RS high
   }
   va_end(ap);
}
/*******************************************************************************
 *
 * defines start/end coordinates for memory access from host to SSD1963
 * also maps the start and end points to suit the orientation
 *
 * This function is a modified version of the function inside the MMBasic Interpreter
 * for MM+ on 'MX470 chips
 *
*******************************************************************************/
void defineregion(long x, long y, long width,long height){ //SSD1963
    long x1=x,x2=x+width-1,y1=y,y2=y+height-1;
    unsigned long xstart,xend,ystart,yend,Vertical,Horizontal;
#ifdef MX470
        RSLo;
#endif
        if(HRes>VRes){
        Vertical=VRes;
        Horizontal=HRes;
    }
    else {
        Vertical=HRes;
        Horizontal=VRes;
    }
     if(Option->DISPLAY_ORIENTATION!=LANDSCAPE)goto isP;
              xstart = x1;
              xend = x2;
              ystart = y1;
              yend = y2;
              if(Option->LCD_CD>6){ //reverse for 7" displays
                xstart = (Horizontal - 1) - x2;
                xend = (Horizontal - 1) - x1;
              }
              goto setreg;
     isP:         
     if(Option->DISPLAY_ORIENTATION!=PORTRAIT)goto isRL;
              xstart = y1;
              xend = y2;
              ystart = (Vertical - 1) - x2;
              yend = (Vertical - 1) - x1;
              goto setreg;
    isRL:
    if(Option->DISPLAY_ORIENTATION!=RLANDSCAPE)goto isRP;
              xstart = (Horizontal - 1) - x2;
              xend = (Horizontal - 1) - x1;
              ystart = (Vertical - 1) - y2;
              yend = (Vertical - 1) - y1;
              if(Option->LCD_CD>6){//reverse for 7" displays
                xstart = x1;
                xend = x2;
              }
              goto setreg;
     isRP:
              xstart = (Horizontal - 1) - y2;
              xend = (Horizontal - 1) - y1;
              ystart = x1;
              yend = x2;
    setreg:
    port=0x22A ;WRLo;    WRHi; // RS low
    #ifdef MX470
        RSHi;
    #endif
    port=(xstart>>8 ) | Both;  WRLo;    WRHi; // RS HIGH
    port=(xstart) | Both;  WRLo;    WRHi; // RS HIGH
    port=(xend>>8 ) | Both;  WRLo;    WRHi; // RS HIGH
    port=(xend) | Both;  WRLo;    WRHi; // RS HIGH
    #ifdef MX470
        RSLo;
    #endif
    port=0x22B ;  WRLo;    WRHi; // RS low
    #ifdef MX470
        RSHi;
    #endif
    port=(ystart>>8 ) | Both;  WRLo;    WRHi; // RS HIGH
    port=(ystart) | Both;  WRLo;    WRHi; // RS HIGH
    port=(yend>>8 ) | Both;  WRLo;    WRHi; // RS HIGH
    port=(yend) | Both;  WRLo;    WRHi; // RS HIGH    RSHi;
}



/***********************************************************************************************
 * Display the bitmap of a char on the TFT panel
 *
 * This function is NEVER called by MMBasic programs
 *
 * This is optimised for the
 *      x, y - the top left of the char
 *      width, height - size of the char's bitmap
 *      scale - how much to scale the bitmap
 * 	fg, bg - foreground and background colour
 *      bitmap - pointer to the butmap
 *
 * This function is a modified version of the function inside the MMBasic Interpreter
 * for MM+ on 'MX470 chips
***********************************************************************************************/
void DrawBitmap_SSD1963(int x1, int y1, int width, int height, int scale, int fc, int bc, unsigned char *bitmap ){
    long i, j, k, m, n, RDpin, RDsetport, RDclrport;
    unsigned int fhb,fmb, flb, bhb, bmb, blb, xtest, ytest;
    char *p;
    xtest=HRes;
    ytest=VRes;
    if(bc==1 && Option->LCD_CS!=0){ //special case of overlay text
        RDpin=1<<GetPinBit(Option->LCD_CS);
        RDsetport=(unsigned int)GetPortAddr(Option->LCD_CS,LATSET);
        RDclrport=(unsigned int)GetPortAddr(Option->LCD_CS,LATCLR);
        i=0;
        j=width*height*scale*scale*3;
        p=GetMemory(j); //allocate some temporary memory
        defineregion(x1,y1,width*scale,height*scale);
        #ifdef MX470
            RSLo;
        #endif
        port=0x22E ;  WRLo;    WRHi; // RS low
        #ifdef MX470
            RSHi;
        #endif
        trisinv=0xFF; //set pins to read
        do { //read in the screen area to be overlayed
            RDLo;
            n=DEVID;n=DEVID;
            RDHi;
            p[i++]=(read & 0xFF);
            RDLo;
            n=DEVID;n=DEVID;
            RDHi;
            p[i++]=(read & 0xFF);
            RDLo;
            n=DEVID;n=DEVID;
            RDHi;
            p[i++]=(read & 0xFF);
        } while(i<j);
        trisinv=0xFF; //set pins to write
    }

    // convert the colours to 888 format
    fhb = (fc >> 16) | Both;
    fmb = (fc >> 8) | Both;
    flb = fc  | Both;
    bhb = (bc >> 16) | Both;
    bmb = (bc >> 8) | Both;
    blb = bc | Both;

    n=0;
    defineregion(x1,y1,width*scale,height*scale);
    #ifdef MX470
        RSLo;
    #endif
    port=0x22C ;  WRLo;    WRHi; // RS low
    #ifdef MX470
        RSHi;
    #endif
    for(i = 0; i < height; i++) {                                   // step thru the font scan line by line
        for(j = 0; j < scale; j++) {                                // repeat lines to scale the font
            for(k = 0; k < width; k++) {                            // step through each bit in a scan line
                for(m = 0; m < scale; m++) {                        // repeat pixels to scale in the x axis
                    if(x1 + k * scale + m >= 0 && x1 + k * scale + m < xtest && y1 + i * scale + j >= 0 && y1 + i * scale + j < ytest) {  // if the coordinates are valid
                        if((bitmap[((i * width) + k)/8] >> (((height * width) - ((i * width) + k) - 1) %8)) & 1) {
                                port=fhb;    WRLo;    WRHi;
                                port=fmb;    WRLo;    WRHi;
                                port=flb;    WRLo;    WRHi;
                                n+=3;
                        } else {
                            if(bc==1 && Option->LCD_CS!=0){
                                port=p[n++] | Both;    WRLo;    WRHi;
                                port=p[n++] | Both;    WRLo;    WRHi;
                                port=p[n++] | Both;    WRLo;    WRHi;
                            } else {
                                port=bhb;    WRLo;    WRHi;
                                port=bmb;    WRLo;    WRHi;
                                port=blb;    WRLo;    WRHi;
                            }
                        }
                    }
                }
            }
        }
    }
    if(bc==1 && Option->LCD_CS!=0) FreeMemory(p);
}

/*******************************************************************************
 *
 * Called by MMBasic Interpreter to draw a rectangle
 * on the SSD1963 panel
 *
 * This function is NEVER called by MMBasic programs
 *
 *
 ******************************************************************************/

void DrawRectangle_SSD1963(int x1, int y1, int x2, int y2, int fc){
    int t;
    long width, height;
    unsigned long i,ch,cm,cl;
    // make sure the coordinates are kept within the display area
    if(x2 <= x1) { t = x1; x1 = x2; x2 = t; }
    if(y2 <= y1) { t = y1; y1 = y2; y2 = t; }
    if(x1 < 0) x1 = 0; if(x1 >= HRes) x1 = HRes - 1;
    if(x2 < 0) x2 = 0; if(x2 >= HRes) x2 = HRes - 1;
    if(y1 < 0) y1 = 0; if(y1 >= VRes) y1 = VRes - 1;
    if(y2 < 0) y2 = 0; if(y2 >= VRes) y2 = VRes - 1;
    width=x2-x1+1;height=y2-y1+1;
    i=width*height;
    defineregion(x1,y1,width,height);
    #ifdef MX470
        RSLo;
    #endif
    port=0x22C ;  WRLo;    WRHi; // RS low
    #ifdef MX470
        RSHi;
    #endif
    // convert the colours to 888 format
    ch = ((fc >> 16)) & 0xFF | Both; // make 8 bits to allow the optimisation to work
    cm = ((fc >> 8)) & 0xFF | Both;
    cl = fc & 0xFF | Both;
    if((ch==cl) && (ch==cm)){
       port=ch;    WRLo;    WRHi;
       WRLo;    WRHi;
       WRLo;    WRHi;
       i--;
       while (i--){
          WRLo;
          WRHi;
          WRLo;
          WRHi;
          WRLo;
          WRHi;
       } 
    }else {
        while (i--) {
            port=ch;    WRLo;    WRHi;
            port=cm;    WRLo;    WRHi;
            port=cl;    WRLo;    WRHi;
        }
    }
}

__attribute__((noinline)) void getFPC(void *a, void *b, volatile unsigned int *c) 
     { 
         *c = (unsigned int) (__builtin_return_address (0) - (b -a)) ;      
     } 
void pstring(const char *s){
    volatile unsigned int libAddr ; 
    getFPC(NULL,&&getFPCLab,&libAddr) ; // warning can be ignored, stupid editor 
    getFPCLab: { } 
    unsigned char  * testData    = (unsigned char *)((void *)s + libAddr );
    MMPrintString(testData);
}

/*******************************************************************************
 *
 * SSD1963_V44 : Initialise the CFunction Driver Sub-System
 *
 * Function called to initialise the driver SubSystem
 *
 * SSD1963_V44 is ALWAYS called from an MMBasic program
 * On exit, vectors DrawRectangleVector, and DrawBitmapVector will
 * be set to point to the CFunctions DrawRectangle_SSD1963 and
 * DrawBitmap_SSD1963 respectively
 *
 * Input Arguments
 *    MyAddress:    The Address of the CFunction, use PEEK(CFUNADDR SSD1963_V44)
 *    RST:  The pin connected to the Reset input on the display
 *    orientation:
 *       is 0 for Portrait orientation, 1 for reverse portrait, 2 for landscape, 3 for reverse landscape
 *    size: The display size 4, 5, or 7"
 *
 * Exit Value   Version
 *
 * Note, For added performance, it may be desirable to create a
 * separate WriteData function which doen't asser/de-asser CS for each byte and
 * instead use block CSLo/CSHi around data transfers. These "block" CSLo/CSHi
 * statements have been left in the code as comments.
 * 
 ******************************************************************************/
//CFunction Driver_SSD1963
#ifdef MX470
void main(long long *page, long long *RD, long long *orientation, long long *size){
#else
void main(long long *page, long long *RD, long long *orientation, long long *size, long long *RST){
#endif
    int t,HorizontalRes,VerticalRes;
    unsigned int p[8];
    int j;

    volatile unsigned int libAddr ; 
    getFPC(NULL,&&getFPCLab,&libAddr) ; // warning can be ignored, stupid editor 
    getFPCLab: { } 
    
    if(*page==0){
#ifndef MX470
    Option->LCD_Reset=*RST;
#endif
    Option->LCD_CS=*RD;
    Option->DISPLAY_ORIENTATION=*orientation;
    Option->LCD_CD=*size; //save the display size as the display type for use in the driver routines

#ifdef MX470
    Option->LCD_Reset=28;
    p[0]=60;
    p[1]=61;
    p[2]=62;
    p[3]=63;
    p[4]=64;
    p[5]=1;
    p[6]=2;
    p[7]=3;

#else    
    Option->LCD_Reset=*RST;
    p[0]=25;
    p[1]=26;
    p[2]=27;
    p[3]=36;
    p[4]=37;
    p[5]=38;
    p[6]=2;
    p[7]=3;
#endif    
    for(j=0;j<8;j++){
        ExtCfg(p[j],EXT_DIG_OUT,0);ExtCfg(p[j],EXT_BOOT_RESERVED,0);
        PinSetBit(p[j],LATCLR);
    }

    //Control Signals
    ExtCfg(RS_Pin_No,EXT_DIG_OUT,0);ExtCfg(RS_Pin_No,EXT_BOOT_RESERVED,0);
    RSHi;

    ExtCfg(WR_Pin_No,EXT_DIG_OUT,0);ExtCfg(WR_Pin_No,EXT_BOOT_RESERVED,0);
    WRHi;

    ExtCfg(Option->LCD_Reset,EXT_DIG_OUT,0);ExtCfg(Option->LCD_Reset,EXT_BOOT_RESERVED,0);
    PinSetBit(Option->LCD_Reset, LATSET);

    if(Option->LCD_CS){
        ExtCfg(Option->LCD_CS, EXT_DIG_OUT,0);
        ExtCfg(Option->LCD_CS, EXT_BOOT_RESERVED,0);
        PinSetBit(Option->LCD_CS, LATSET);
    }

    //Save DisplayMetrics
    if(Option->LCD_CD<5){
       HorizontalRes=480;
//       if(Option->DISPLAY_ORIENTATION==1) VerticalRes=272*3; //3 frame buffers
       /*else*/ VerticalRes=272;
    } else {
      HorizontalRes=800;
      VerticalRes=480;
    }

    //Reset the SSD1963
    PinSetBit(Option->LCD_Reset,LATSET);
    uSec(10000);
    PinSetBit(Option->LCD_Reset,LATCLR);
    uSec(10000);
    PinSetBit(Option->LCD_Reset,LATSET);
    uSec(10000);

    if(Option->LCD_CD<5) {
        write_command_data(0xE2,3,0x23,0x02,0x54);
        write_command_data(0x33,6,0,0,1,16,0,0); //set scroll area
    } else {
	if(Option->LCD_CD<6){
             write_command_data(0xE2,3,0x1E,0x02,0x54);
        } else {
             write_command_data(0xE2,3,0x23,0x02,0x04);
        }
    }

    write_command_data(0xE0,1,0x01); uSec( 10000); 		// PLL enable
    write_command_data(0xE0,1,0x03);	uSec( 10000);
    write_command_data(0x01,0); uSec( 100000);		// software reset

    if(Option->LCD_CD<5) {
	write_command_data(0xE6,3,0x01,0x1F,0xFF);
	write_command_data(0xB0,7,0x20,0x00,0x01,0xDF,0x01,0x0F,0x00);		//LCD SPECIFICATION
	write_command_data(0xB4,8,0x02,0x13,0x00,0x08,0x2B,0x00,0x02,0x00);
	write_command_data(0xB6,7,0x01,0x20,0x00,0x04,0x0C,0x00,0x02);
    } else {
        if (Option->LCD_CD<6){
            write_command_data(0xE6,3,0x03,0xFF,0xFF);
	    write_command_data(0xB0,7,0x24,0x00,0x03,0x1F,0x01,0xDF,0x00);		//LCD SPECIFICATION
        } else {
	    write_command_data(0xE6,3,0x04,0x93,0xE0);
	    write_command_data(0xB0,7,0x00,0x00,0x03,0x1F,0x01,0xDF,0x00);		//LCD SPECIFICATION
        }
	write_command_data(0xB4,8,0x03,0xA0,0x00,0x2E,0x30,0x00,0x0F,0x00);
	write_command_data(0xB6,7,0x02,0x0D,0x00,0x10,0x10,0x00,0x08);
    }

    write_command_data(0xBA,1,0x0F);
    write_command_data(0xB8,2,0x07,0x01);
    write_command_data(0x36,1,0x00);		//rotation
    write_command_data(0xF0,1,0x00);		//PIXEL data interface - 8-bit
    write_command_data(0x3A,1,0b01110000);		//PIXEL data interface 24-bit
    write_command_data(0x26,1,0x01);  uSec( 10000);		//gamma curve
    write_command_data(0x29,0);		//display on
    write_command_data(0xBE,6,0x06,0xF0,0x01,0xF0,0x00,0x00);		//set PWM for B/L
    write_command_data(0xd0,1,0x0D);
    //Set Hres and VRes
    if(Option->DISPLAY_ORIENTATION==PORTRAIT){
        if (Option->LCD_CD<6) t=0xA0;
        else t=0xE0;
    }
    if(Option->DISPLAY_ORIENTATION==RPORTRAIT){
        if (Option->LCD_CD<6) t=0x60;
        else t=0x20;
    }
    if(Option->DISPLAY_ORIENTATION==LANDSCAPE){
        if (Option->LCD_CD<6) t=0x00;
        else t=0x40;
    }
    if(Option->DISPLAY_ORIENTATION==RLANDSCAPE){
        if (Option->LCD_CD<6) t=0xC0;
        else t=0x80;
    }
    write_command_data(0x36,1,t);
    if(Option->DISPLAY_ORIENTATION&1){
        HRes=HorizontalRes;
        VRes=VerticalRes;
    } else {
        VRes=HorizontalRes;
        HRes=VerticalRes;
    }

    //Save the current values of the Vectors

    //Set the DrawRectangle vector to point to our function
    DrawRectangleVector= (unsigned int)&DrawRectangle_SSD1963 + libAddr;

    //Set the DrawBitmap vector to point to our function
    DrawBitmapVector=(unsigned int)&DrawBitmap_SSD1963 + libAddr;

    //CLS
    DrawRectangle(0,0,HRes-1,VRes-1,0x000000);
#ifdef MX470
      static const char startup[]="SSD1963_64 driver loaded\r\n";
#else    
       static const char startup[]="SSD1963_44 driver loaded\r\n";
#endif
    pstring(startup);
    } else {
       if(Option->DISPLAY_ORIENTATION==1){
           VRes=272*3;
           j=(*page-1) * 272; 
           write_command_data(0x37,2,j>>8,j&0xFF); 
       }
    }
}
