/*******************************************************************************
 *
 * Driver for ILI9325 and R61505V displays written as CFunction
 *
 * (c) Peter Mather 2015 with acknowledgements to Peter Carnegie & Geoff Graham
 * 
 *
 * This CFunction MUST be compiled with Optimization Level 1, -O1
 * -O2,-O3,-Os will compile successfully, but generates Exceptions at runtime.
 *
 * When Generating the CFunction, use MERGE CFunction mode, and name the CFunction
 * Driver_ILI9325
 *
 *                                        long long *orientation)
 *
 * V1.0     2015-09-7 Peter Mather
 * V1.01   Improved performance of CLS for black and white
 * V1.03   Support 64-pin chip
 * V1.04   Support for R61505V controller
 * V1.1    Remove address passing, use Nathan's enhancements
 * V1.2    Support overlay text and store/restore functionality
 * 
 ******************************************************************************/

#define Version 110     //Version 1.1
#include <stdarg.h>
#include <xc.h>			   // Required for SFR defs
#include <sys/attribs.h>   // Required to use __longramfunc__
#include "../CFunctions.h"

#define MX470 //Comment out for compilation for 44-pin part
#define R61505V // Comment out for ili9325
#define bclrport *(volatile unsigned int *)(0xbf886134) //latch registers
#define bsetport *(volatile unsigned int *)(0xbf886138) //latch registers
#define cport *(volatile unsigned int *)(0xbf886230) //latch registers
#define cread *(volatile unsigned int *)(0xbf886220) //latch registers
#define ctrisinv *(volatile unsigned int *)(0xbf88621C) //latch registers
#define cclrport *(volatile unsigned int *)(0xbf886234) //latch registers
#define csetport *(volatile unsigned int *)(0xbf886238) //latch registers
#define eport *(volatile unsigned int *)(0xbf886430) //latch registers
#define eread *(volatile unsigned int *)(0xbf886420) //latch registers
#define etrisinv *(volatile unsigned int *)(0xbf88641C) //latch registers
#define eclrport *(volatile unsigned int *)(0xbf886434) //latch registers
#define esetport *(volatile unsigned int *)(0xbf886438) //latch registers
#define DEVID (*(volatile unsigned int *)0xBF80F220)
#define swap(a, b) {int t; t = a; a = b; b = t; }


//Offsets into the persistent RAM of variables
#ifdef MX470
    #define RS_Pin_No 27
    #define WR_Pin_No 24
    #define RS_Pin 0x1000
    #define WR_Pin 0x0800
    #define clrport bclrport
    #define setport bsetport
    #define port eport
    #define read eread
    #define trisinv etrisinv
    #define RSLo    {clrport=RS_Pin;}
    #define RSHi    {setport=RS_Pin;}
    #define WRLo    {clrport=WR_Pin;}
    #define WRHi    {setport=WR_Pin;}
#else
    #define RS_Pin_No 4
    #define WR_Pin_No 5
    #define RS_Pin 0x100
    #define WR_Pin 0x200
    #define clrport cclrport
    #define setport csetport
    #define port cport
    #define read cread
    #define trisinv ctrisinv
    #define RSLo    clrport=RS_Pin
    #define RSHi    setport=RS_Pin
    #define WRLo    clrport=WR_Pin
    #define WRHi    setport=WR_Pin
#endif
#define Both WR_Pin | RS_Pin
#define RDLo    (*(volatile unsigned int *)RDclrport)=RDpin
#define RDHi    (*(volatile unsigned int *)RDsetport)=RDpin


#define LANDSCAPE       1
#define PORTRAIT        2
#define RLANDSCAPE      3
#define RPORTRAIT       4

/*******************************************************************************
 *
 * Write Data to a register on the Chip
 *
 ******************************************************************************/

void writeRegister(unsigned int command, int data){
#ifdef MX470
        RSLo;
#endif
    port=(((command>>8) & 0x2FF) | WR_Pin);
    WRLo;
    WRHi;
    port=(((command) & 0x2FF) | WR_Pin);
    WRLo;
    WRHi;
    if(data!=-1){
#ifdef MX470
        RSHi;
#endif
    port=(((data>>8) & 0x3FF) | Both);
    WRLo;
    WRHi;
    port=(((data) & 0x3FF) | Both);
    WRLo;
    WRHi;
    }
}

/*******************************************************************************
 *
 * defines start/end coordinates for memory access from host to ILI9325
 * also maps the start and end points to suit the orientation
 *
 * This function is a modified version of the function inside the MMBasic Interpreter
 * for MM+ on 'MX470 chips
 *
*******************************************************************************/
void defineregion(long x, long y, long width,long height){ //ILI9325
    long x1=x,x2=x+width-1,y1=y,y2=y+height-1;
    unsigned long xstart,xend,ystart,yend,xp,yp,Vertical,Horizontal;
    if(HRes<VRes){
        Vertical=VRes;
        Horizontal=HRes;
    }
    else {
        Vertical=HRes;
        Horizontal=VRes;
    }
    switch(Option->DISPLAY_ORIENTATION){
        case PORTRAIT:
              xstart = x1;
              xend = x2;
              ystart = y1;
              yend = y2;
              xp=xstart; yp=ystart;
#ifdef R61505V
              xstart=xstart+10;
              xend=xend+10;
              xp=xp+10;
#endif              
              break;
        case RPORTRAIT:
              xstart = (Horizontal - 1) - x2;
              xend = (Horizontal - 1) - x1;
              ystart = (Vertical - 1) - y2;
              yend = (Vertical - 1) - y1;
              xp=xend; yp=yend;
#ifdef R61505V
              ystart=ystart-10;
              yend=yend-10;
              yp=yp-10;
#endif              
              break;
        case LANDSCAPE:
              xstart = (Horizontal - 1) - y2;
              xend = (Horizontal - 1) - y1;
              ystart = x1;
              yend = x2;
              xp=xend; yp=ystart;
              break;
        default:
              xstart = y1;
              xend = y2;
              ystart = (Vertical - 1) - x2;
              yend = (Vertical - 1) - x1;
              xp=xstart; yp=yend;
#ifdef R61505V
              xstart=xstart+10;
              xend=xend+10;
              xp=xp+10;
              ystart=ystart-10;
              yend=yend-10;
              yp=yp-10;
#endif              
       }
    writeRegister(0x0020,xp);
    writeRegister(0x0021,yp);
    writeRegister(0x0050,xstart);
    writeRegister(0x0052,ystart);
    writeRegister(0x0051,xend);
    writeRegister(0x0053,yend);
    writeRegister(0x0022,-1);
}

int ReadRectangle(int x1, int y1, int width, int height, int pos, char *buff ){
    int i,j,n, RDpin, RDsetport, RDclrport;
    RDpin=1<<GetPinBit(Option->LCD_CS);
    RDsetport=(unsigned int)GetPortAddr(Option->LCD_CS,LATSET);
    RDclrport=(unsigned int)GetPortAddr(Option->LCD_CS,LATCLR);
    for(j=y1;j<y1+height;j++){
        for(i=x1;i<x1+width;i++){
            defineregion(i,j,1,1);

            trisinv=0xFF; //set pins to read
#ifndef MX470
            port=Both;
            RDLo;n=DEVID;n=DEVID;RDHi;n=DEVID;n=DEVID;
            RDLo;n=DEVID;n=DEVID;RDHi;buff[pos++]=(read & 0xFF);n=DEVID;n=DEVID;
            RDLo;n=DEVID;n=DEVID;RDHi;buff[pos++]=(read & 0xFF);n=DEVID;n=DEVID;
            trisinv=0xFF; //set pins to write
#else
            WRHi;
            RSHi;n=DEVID;n=DEVID;
            RDLo;n=DEVID;n=DEVID;n=DEVID;n=DEVID;RDHi;n=DEVID;n=DEVID;n=DEVID;n=DEVID;
            RDLo;n=DEVID;n=DEVID;n=DEVID;n=DEVID;RDHi;buff[pos++]=(read & 0xFF);n=DEVID;n=DEVID;n=DEVID;n=DEVID;
            RDLo;n=DEVID;n=DEVID;n=DEVID;n=DEVID;RDHi;buff[pos++]=(read & 0xFF);n=DEVID;n=DEVID;n=DEVID;n=DEVID;
            trisinv=0xFF; //set pins to write
#endif
        }
    }
    return pos;
}

int RestoreRectangle(int x1, int y1, int width, int height, int pos, char p[]){
    int i;
    defineregion(x1,y1,width,height);
    i=width*height;
#ifdef MX470
    RSHi;
#endif
    while(i--){
        port=p[pos++] | Both;    WRLo;    WRHi;
        port=p[pos++] | Both;    WRLo;    WRHi;
    }
    return pos;
}

/***********************************************************************************************
 * Display the bitmap of a char on the TFT panel
 *
 * This function is NEVER called by MMBasic programs
 *
 * This is optimised for the
 *      x, y - the top left of the char
 *      width, height - size of the char's bitmap
 *      scale - how much to scale the bitmap
 * 	fg, bg - foreground and background colour
 *      bitmap - pointer to the butmap
 *
 * This function is a modified version of the function inside the MMBasic Interpreter
 * for MM+ on 'MX470 chips
***********************************************************************************************/
void DrawBitmap_ILI9325(int x1, int y1, int width, int height, int scale, int fc, int bc, unsigned char *bitmap ){
    long i, j, k, m, n, RDpin, RDsetport, RDclrport;
    unsigned int fch,fcm,bch,bcm,xtest, ytest;
    char *p;
    xtest=HRes;
    ytest=VRes;

    if(bc==1 && Option->LCD_CS!=0){ //special case of overlay text
        RDpin=1<<GetPinBit(Option->LCD_CS);
        RDsetport=(unsigned int)GetPortAddr(Option->LCD_CS,LATSET);
        RDclrport=(unsigned int)GetPortAddr(Option->LCD_CS,LATCLR);
        i=0;
        j=width*height*scale*scale*2;
        p=GetMemory(j); //allocate some temporary memory
        ReadRectangle(x1, y1, width*scale, height*scale, 0, p);
    }

    fch = ((fc >> 16) & 0b11111000) | ((fc >> 13) & 0b00000111)| Both;
    fcm = ((fc >> 5) & 0b11100000) | ((fc >> 3) & 0b00011111)| Both;
    bch = ((bc >> 16) & 0b11111000) | ((bc >> 13) & 0b00000111)| Both;
    bcm = ((bc >> 5) & 0b11100000) | ((bc >> 3) & 0b00011111)| Both;
    n=0;
    defineregion(x1,y1,width*scale,height*scale);
#ifdef MX470
    RSHi;
#endif
    for(i = 0; i < height; i++) {                                   // step thru the font scan line by line
        for(j = 0; j < scale; j++) {                                // repeat lines to scale the font
            for(k = 0; k < width; k++) {                            // step through each bit in a scan line
                for(m = 0; m < scale; m++) {                        // repeat pixels to scale in the x axis
                    if(x1 + k * scale + m >= 0 && x1 + k * scale + m < xtest && y1 + i * scale + j >= 0 && y1 + i * scale + j < ytest) {  // if the coordinates are valid
                        if((bitmap[((i * width) + k)/8] >> (((height * width) - ((i * width) + k) - 1) %8)) & 1) {
                                port=fch;    WRLo;    WRHi;
                                port=fcm;    WRLo;    WRHi;
                                n+=2;
                        } else {
                            if(bc==1 && Option->LCD_CS!=0){
                                port=p[n++] | Both;    WRLo;    WRHi;
                                port=p[n++] | Both;    WRLo;    WRHi;
                            } else {
                                port=bch;    WRLo;    WRHi;
                                port=bcm;    WRLo;    WRHi;
                            }
                        }
                    }
                }
            }
        }
    }
}

/*******************************************************************************
 *
 * Called by MMBasic Interpreter to draw a rectangle
 * on the ILI9325 panel
 *
 * This function is NEVER called by MMBasic programs
 *
 *
 ******************************************************************************/

void DrawRectangle_ILI9325(int x1, int y1, int x2, int y2, int c){
    int t;
    long width, height;
    unsigned long i,hb,lb;
    if(c==1)    writeRegister(0x0007,0x0103); // 262K color and display OFF 
    if(c==2)    writeRegister(0x0007,0x0133); // 262K color and display ON 

    // make sure the coordinates are kept within the display area
    if(x2 <= x1) { t = x1; x1 = x2; x2 = t; }
    if(y2 <= y1) { t = y1; y1 = y2; y2 = t; }
    if(x1 < 0) x1 = 0; if(x1 >= HRes) x1 = HRes - 1;
    if(x2 < 0) x2 = 0; if(x2 >= HRes) x2 = HRes - 1;
    if(y1 < 0) y1 = 0; if(y1 >= VRes) y1 = VRes - 1;
    if(y2 < 0) y2 = 0; if(y2 >= VRes) y2 = VRes - 1;
    width=x2-x1+1;height=y2-y1+1;
    i=width*height;
    defineregion(x1,y1,width,height);
#ifdef MX470
    RSHi;
#endif
    // convert the colours to 565 format
    hb = ((c >> 16) & 0b11111000) | ((c >> 13) & 0b00000111)| Both;
    lb = ((c >> 5) & 0b11100000) | ((c >> 3) & 0b00011111)| Both;
#ifndef MX470 //Don't optimise if MX470 as it is too fast for the controller
    if(hb==lb){
       port=hb;    WRLo;    WRHi;
       WRLo;    WRHi;
       i--;
       while (i--){
          WRLo;
          WRHi;
          WRLo;
          WRHi;
       }
    }else {
#endif       
        while (i--) {
            port=hb;    WRLo;    WRHi;
            port=lb;    WRLo;    WRHi;
        }
#ifndef MX470
    }
#endif
}

long triangles(int mode, int buffpos, char *buff, long  col, long x0, long y0, long x1, long y1,long x2, long y2) {//Cfunction
  long a, b, y, last;
  long  dx01,  dy01,  dx02,  dy02, dx12,  dy12,  sa, sb;
      
      if (y0>y1) {
        swap(y0, y1);
        swap(x0, x1);
      }
      if (y1>y2) {
        swap(y2, y1);
        swap(x2, x1);
      }
      if (y0>y1) {
        swap(y0, y1);
        swap(x0, x1);
      }
      if(y0==y2) { // Handle awkward all-on-same-line case as its own thing
        a=x0;
        b=x0;
        if(x1<a) {
          a=x1;
        } else {
          if(x1>b) b=x1;
        }
        if(x2<a) {
          a=x2;
        } else {
          if(x2>b)  b=x2;
        }
        if(mode==0){
            buffpos=ReadRectangle(a,y0,1,b-a+1,buffpos,buff);
        } else if(mode==1) {
            buffpos=RestoreRectangle(a,y0,1,b-a+1,buffpos,buff);
        } else {
            DrawRectangle_ILI9325(a,y0,a,b-a+y0,col);
        }
      } else {
        dx01=x1-x0;  dy01=y1-y0;  dx02=x2-x0;  dy02=y2-y0; dx12=x2-x1;  dy12=y2-y1;  sa=0; sb=0;
        if(y1==y2) {
            last=y1; //Include y1 scanline
        } else {
            last=y1-1; // Skip it
        }
        for (y=y0;y<=last;y++){
            a=x0+sa / dy01;
            b=x0+sb / dy02;
            sa=sa+dx01;
            sb=sb+dx02;
            a=x0+(x1-x0) * (y-y0) / (y1-y0);
            b=x0+(x2-x0) * (y-y0) / (y2-y0);
            if(a>b)swap(a,b);
            if(mode==0){
                buffpos=ReadRectangle(a,y,b-a+1,1,buffpos,buff);
            } else if(mode==1) {
                buffpos=RestoreRectangle(a,y,b-a+1,1,buffpos,buff);
            } else {
                DrawRectangle_ILI9325(a,y,b,y,col);
            }
        }
        sa=dx12 * (y-y1);
        sb=dx02 * (y-y0);
        while (y<=y2){
            a=x1+sa / dy12;
            b=x0+sb / dy02;
            sa=sa+dx12;
            sb=sb+dx02;
            a=x1+(x2-x1) * (y-y1) / (y2-y1);
            b=x0+(x2-x0) * (y-y0) / (y2-y0);
            if(a>b) swap(a,b);
            if(mode==0){
                buffpos=ReadRectangle(a,y,b-a+1,1,buffpos,buff);
            } else if(mode==1) {
                buffpos=RestoreRectangle(a,y,b-a+1,1,buffpos,buff);
            } else {
                DrawRectangle_ILI9325(a,y,b,y,col);
            }
            y=y+1;
            }
        }

  return buffpos;
}

__attribute__((noinline)) void getFPC(void *a, void *b, volatile unsigned int *c) 
     { 
         *c = (unsigned int) (__builtin_return_address (0) - (b -a)) ;      
     } 
void pstring(const char *s){
    volatile unsigned int libAddr ; 
    getFPC(NULL,&&getFPCLab,&libAddr) ; // warning can be ignored, stupid editor 
    getFPCLab: { } 
    unsigned char  * testData    = (unsigned char *)((void *)s + libAddr );
    MMPrintString(testData);
}
/*******************************************************************************
 *
 * Driver_ILI9325 : Initialise the CFunction Driver Sub-System
 *
 * Function called to initialise the driver SubSystem
 *
 * OpenDriver_ILI9325 is ALWAYS called from an MMBasic program
 * On exit, vectors DrawRectangleVector, and DrawBitmapVector will
 * be set to point to the CFunctions DrawRectangle_ILI9325 and
 * DrawBitmap_ILI9325 respectively
 *
 * Input Arguments
 * MyAddress    The Address of the CFunction, use PEEK(CFUNADDR Driver_ILI9325)
 *
 * func         0=Open Driver, 1=Close Driver, 2=Return Version Number
 *
 * PinDefs[] contains the pin definitions for the I/O to be used to interface to the ILI9325 display
 *  PinDefs[0 .. 8] pin numbers of DB0-DB8
 *  PinDefs[8] pin number of RS
 *  PinDefs[9] pin number of WR
 *  PinDefs[10] pin number of RESET
 *
 * Rotation
 *   is 0 for Portrait orientation, 1 for reverse portrait, 2 for landscape, 3 for reverse landscape
 *
 * Exit Value   Version
 *
 * Note, For added performance, it may be desirable to create a
 * separate WriteData function which doen't asser/de-asser CS for each byte and
 * instead use block CSLo/CSHi around data transfers. These "block" CSLo/CSHi
 * statements have been left in the code as comments.
 * 
 ******************************************************************************/
//CFunction Driver_ILI9325

long long main(char *orientation, unsigned char *p1, long long p2[], long long p3[], long long p4[], long long p5[], long long p6[],long long p7[], long long p8[]){
  
    int i,HorizontalRes,VerticalRes;
    unsigned int p[8];
    volatile unsigned int libAddr ; 
    getFPC(NULL,&&getFPCLab,&libAddr) ; // warning can be ignored, stupid editor 
    getFPCLab: { } 
    if(*orientation>=0 && *orientation<4){
    Option->DISPLAY_ORIENTATION=*orientation;
    Option->LCD_CS=0;
#ifndef MX470
    if(p2[0])Option->LCD_CS=p2[0];
#else
    if(*p1)Option->LCD_CS=*p1;
#endif
#ifdef MX470
    Option->LCD_Reset=28;
    p[0]=60;
    p[1]=61;
    p[2]=62;
    p[3]=63;
    p[4]=64;
    p[5]=1;
    p[6]=2;
    p[7]=3;

#else    
    Option->LCD_Reset=*p1;
    p[0]=25;
    p[1]=26;
    p[2]=27;
    p[3]=36;
    p[4]=37;
    p[5]=38;
    p[6]=2;
    p[7]=3;
#endif    

    
    //DataBus
    for(i=0;i<+8;i++){
        ExtCfg(p[i],EXT_DIG_OUT,0);ExtCfg(p[i],EXT_BOOT_RESERVED,0);
        PinSetBit(p[i],LATCLR);
    }

    //Control Signals
    ExtCfg(RS_Pin_No,EXT_DIG_OUT,0);ExtCfg(RS_Pin_No,EXT_BOOT_RESERVED,0);
    RSHi;

    ExtCfg(WR_Pin_No,EXT_DIG_OUT,0);ExtCfg(WR_Pin_No,EXT_BOOT_RESERVED,0);
    WRHi;

    ExtCfg(Option->LCD_Reset,EXT_DIG_OUT,0);ExtCfg(Option->LCD_Reset,EXT_BOOT_RESERVED,0);
    PinSetBit(Option->LCD_Reset, LATSET);

    if(Option->LCD_CS){
        ExtCfg(Option->LCD_CS,EXT_DIG_OUT,0);ExtCfg(Option->LCD_CS,EXT_BOOT_RESERVED,0);
        PinSetBit(Option->LCD_CS, LATSET);
    }


    //Save DisplayMetrics
#ifndef R61505V
    HorizontalRes=240;
    VerticalRes=320;
#else
    HorizontalRes=230;
    VerticalRes=230;
#endif

    /************************************************
     * 
     * Initialise the ILI9325
     * 
     ***********************************************/

    //Reset the ILI9325
    PinSetBit(Option->LCD_Reset,LATSET);
    uSec(50000);
    PinSetBit(Option->LCD_Reset,LATCLR);
    uSec(50000);
    PinSetBit(Option->LCD_Reset,LATSET);
    uSec(50000);
#ifndef R61505V
    writeRegister(0x00E5,0x78F0); // set SRAM internal timing
    writeRegister(0x0001,0x0100); // set SS and SM bit
    writeRegister(0x0002,0x0700); // set 1 line inversion
    writeRegister(0x0003,0x1030); // set GRAM write direction and BGR=1.
    writeRegister(0x0004,0x0000); // Resize register
    writeRegister(0x0008,0x0207); // set the back porch and front porch
    writeRegister(0x0009,0x0000); // set non-display area refresh cycle ISC[3:0]
    writeRegister(0x000A,0x0000); // FMARK function
    writeRegister(0x000C,0x0000); // RGB interface setting
    writeRegister(0x000D,0x0000); // Frame marker Position
    writeRegister(0x000F,0x0000); // RGB interface polarity
    //*************Power On sequence ****************//
    writeRegister(0x0010,0x0000); // SAP, BT[3:0], AP, DSTB, SLP, STB
    writeRegister(0x0011,0x0007); // DC1[2:0], DC0[2:0], VC[2:0]
    writeRegister(0x0012,0x0000); // VREG1OUT voltage
    writeRegister(0x0013,0x0000); // VDV[4:0] for VCOM amplitude
    writeRegister(0x0007,0x0001);
    uSec(50000); // Dis-charge capacitor power voltage
    writeRegister(0x0010,0x1090); // 1490//SAP, BT[3:0], AP, DSTB, SLP, STB
    writeRegister(0x0011,0x0227); // DC1[2:0], DC0[2:0], VC[2:0]
    uSec(50000); // Delay 50ms
    writeRegister(0x0012,0x001F); //001C// Internal reference voltage= Vci;
    uSec(50000); // Delay 50ms
    writeRegister(0x0013,0x1500); //0x1000//1400   Set VDV[4:0] for VCOM amplitude  1A00
    writeRegister(0x0029,0x0027); //0x0012 //001a  Set VCM[5:0] for VCOMH  //0x0025  0034
    writeRegister(0x002B,0x000D); // Set Frame Rate   000C
    uSec(50000); // Delay 50ms
    writeRegister(0x0020,0x0000); // GRAM horizontal Address
    writeRegister(0x0021,0x0000); // GRAM Vertical Address
    // ----------- Adjust the Gamma Curve ----------//
    writeRegister(0x0030,0x0000);
    writeRegister(0x0031,0x0707);
    writeRegister(0x0032,0x0307);
    writeRegister(0x0035,0x0200);
    writeRegister(0x0036,0x0008);//0207
    writeRegister(0x0037,0x0004);//0306
    writeRegister(0x0038,0x0000);//0102
    writeRegister(0x0039,0x0707);//0707
    writeRegister(0x003C,0x0002);//0702
    writeRegister(0x003D,0x1D04);//1604
    //------------------ Set GRAM area ---------------//
    writeRegister(0x0050,0x0000); // Horizontal GRAM Start Address
    writeRegister(0x0051,0x00EF); // Horizontal GRAM End Address
    writeRegister(0x0052,0x0000); // Vertical GRAM Start Address
    writeRegister(0x0053,0x013F); // Vertical GRAM Start Address
    writeRegister(0x0060,0xA700); // Gate Scan Line
    writeRegister(0x0061,0x0001); // NDL,VLE, REV
    writeRegister(0x006A,0x0000); // set scrolling line
    //-------------- Partial Display Control ---------//
    writeRegister(0x0080,0x0000);
    writeRegister(0x0081,0x0000);
    writeRegister(0x0082,0x0000);
    writeRegister(0x0083,0x0000);
    writeRegister(0x0084,0x0000);
    writeRegister(0x0085,0x0000);
    //-------------- Panel Control -------------------//
    writeRegister(0x0090,0x0010);
    writeRegister(0x0092,0x0600);
    writeRegister(0x0007,0x0133); // 262K color and display ON 
#else
    writeRegister(0xe5, 0x8000);         
    writeRegister(0x00, 0x0001);         
    writeRegister(0x01, 0x0100); 
    writeRegister(0x02, 0x0000); // writeRegister(0x02, 0x0700); 
    writeRegister(0x03, 0x1030); // writeRegister(0x03, 0x1000);  
    writeRegister(0x04, 0x0000); 
    writeRegister(0x08, 0x0202);         
    writeRegister(0x09, 0x002F); // writeRegister(0x09, 0x0000);         
    writeRegister(0x0A, 0x0000); 
    writeRegister(0x0C, 0x0000);         
    writeRegister(0x0D, 0x0000);         
    writeRegister(0x0F, 0x0000);         
//-----Power On sequence-----------------------         
    writeRegister(0x10, 0x0000);         
    writeRegister(0x11, 0x0007);         
    writeRegister(0x12, 0x0000);         
    writeRegister(0x13, 0x0000);         
    uSec(50000); 
    writeRegister(0x10, 0x0730); // writeRegister(0x10, 0x17B0);         
    writeRegister(0x11, 0x0007);         
    uSec(10000); 
    writeRegister(0x12, 0x0138); // writeRegister(0x12, 0x013A);         
    uSec(10000); 
    writeRegister(0x13, 0x1A00);         
    writeRegister(0x29, 0x000c);                 
    uSec(10000); 
//-----Gamma control-----------------------         
    writeRegister(0x30, 0x0000);         
    writeRegister(0x31, 0x0505);         
    writeRegister(0x32, 0x0004);         
    writeRegister(0x35, 0x0006);         
    writeRegister(0x36, 0x0707);         
    writeRegister(0x37, 0x0105);         
    writeRegister(0x38, 0x0002);         
    writeRegister(0x39, 0x0707);         
    writeRegister(0x3C, 0x0704);         
    writeRegister(0x3D, 0x0807);         
//-----Set RAM area-----------------------         
    writeRegister(0x50, 0x0000); 
    writeRegister(0x51, 0x00EF); 
    writeRegister(0x52, 0x0000); 
    writeRegister(0x53, 0x013F); 
    writeRegister(0x60, 0x2700); 
    writeRegister(0x61, 0x0001); 
    writeRegister(0x6A, 0x0000); 
    writeRegister(0x21, 0x0000);         
    writeRegister(0x20, 0x0000);         
//-----Partial Display Control------------         
    writeRegister(0x80, 0x0000);         
    writeRegister(0x81, 0x0000); 
    writeRegister(0x82, 0x0000); 
    writeRegister(0x83, 0x0000); 
    writeRegister(0x84, 0x0000);         
    writeRegister(0x85, 0x0000); 
//-----Panel Control---------------------- 
    writeRegister(0x90, 0x0010);         
    writeRegister(0x92, 0x0000); 
    writeRegister(0x93, 0x0003); 
    writeRegister(0x95, 0x0110); 
    writeRegister(0x97, 0x0000);         
    writeRegister(0x98, 0x0000); 
//-----Display on-----------------------         
    writeRegister(0x07, 0x0173);         
    uSec(50000);
//    IntToStr(x,readRegister(0),16);
//    MMPrintString(x);
//    putConsole(13);
#endif
    switch(Option->DISPLAY_ORIENTATION){
        case PORTRAIT:
              i=0x1030;
              break;
        case RPORTRAIT:
              i=0x1000;
              break;
        case LANDSCAPE:
              i=0x1028;
              break;
        default:
              i=0x1018;
              break;
    }
    writeRegister(0x03,i);
       if(Option->DISPLAY_ORIENTATION&1){ //landscape is not default
                 VRes=HorizontalRes;
                 HRes=VerticalRes;
       } else {
                 HRes=HorizontalRes;
                 VRes=VerticalRes;
       }

    //Save the current values of the Vectors

    //Set the DrawRectangle vector to point to our function
    DrawRectangleVector= (unsigned int)&DrawRectangle_ILI9325 + libAddr;

    //Set the DrawBitmap vector to point to our function
    DrawBitmapVector=(unsigned int)&DrawBitmap_ILI9325 + libAddr;

    //Draw a black box  = CLS
    DrawRectangle(0,0,HRes-1,VRes-1,0x000000);
    #ifndef R61505V
    static const char startup[]="ILI9325 driver loaded\r\n";
    #else
    static const char startup[]="R61505V driver loaded\r\n";
    #endif
    pstring(startup);
    return 0;
    } 

    else if(*orientation==4){
        defineregion(*p1, *p2, *p3, *p4);
        return 4;
    } else if(*orientation==5){
        int num=p1[0],a=1,fc;
        while(a<=num){  
            fc=(p1[a]<<16 ) | (p1[a+1]<<8 ) | p1[a+2];
            port = ((fc >> 16) & 0b11111000) | ((fc >> 13) & 0b00000111)| Both;WRLo;    WRHi;
            port = ((fc >> 5) & 0b11100000) | ((fc >> 3) & 0b00011111)| Both;WRLo;    WRHi;
            a+=3;
        }
        return a-3;
    } else if(*orientation==6){
            return ReadRectangle(*p2, *p3, *p4, *p5, *p6, p1 );
    } else if(*orientation==7){
            return RestoreRectangle(p2[0], p3[0], p4[0], p5[0], p6[0], p1 );
    } else { //must be triangles
        int count=*orientation & 0x7F;
        int i=0,j,buffpos=0;
        while(i<count && p2[i]<0){ //repeat for any restores
            buffpos=triangles(1,buffpos,p1,p2[i],p3[i],p4[i],p5[i],p6[i],p7[i],p8[i]);
            i++;
        }  
        buffpos=0;
        j=i;
        while(i<count && p2[i]>=0){ //repeat for any store and write commands
            buffpos=triangles(0,buffpos,p1,p2[i],p3[i],p4[i],p5[i],p6[i],p7[i],p8[i]);
            i++;
        }
        i=j;
        while(i<count && p2[i]>=0){ //repeat for any store and write commands
            buffpos=triangles(2,buffpos,p1,p2[i],p3[i],p4[i],p5[i],p6[i],p7[i],p8[i]);
            i++;
        }
       return buffpos>>3;        
    }

}
