/*******************************************************************************
 *
 * Driver for ILI9341 Display written as CFunctions
 *
 * (c) Peter Mather 2015 with acknowledgements to Peter Carnegie & Geoff Graham
 * 
 *
 * This CFunction MUST be compiled with Optimization Level 1, -O1
 * -O2,-O3,-Os will compile successfully, but generate exceptions at runtime.
 *
 * When Generating the CFunction, use MERGE CFunction mode, and name the CFunction
 * SSD1963_V44
 *
 * Entry point is function long long main(long long *DC,
 *                                        long long *RST,
 *                                        long long *CS
 *                                        long long *orientation)
 *
 * V1.0     2015-09-23 Peter Mather
 * V2.0     2016-08-07 Add overlay text and area read capability
 * 
 ******************************************************************************/
#include <stdarg.h>

#define Version 120     //Version 1.20
#define _SUPPRESS_PLIB_WARNING                                      // required for XC1.33  Later compiler versions will need PLIB to be installed
#include <plib.h>                                                   // the pre Harmony peripheral libraries
#define MX470

#include "../cfunctions.h"

#define ILI9341_SOFTRESET       0x01
#define ILI9341_SLEEPIN         0x10
#define ILI9341_SLEEPOUT        0x11
#define ILI9341_NORMALDISP      0x13
#define ILI9341_INVERTOFF       0x20
#define ILI9341_INVERTON        0x21
#define ILI9341_GAMMASET        0x26
#define ILI9341_DISPLAYOFF      0x28
#define ILI9341_DISPLAYON       0x29
#define ILI9341_COLADDRSET      0x2A
#define ILI9341_PAGEADDRSET     0x2B
#define ILI9341_MEMORYWRITE     0x2C
#define ILI9341_RAMRD           0x2E
#define ILI9341_PIXELFORMAT     0x3A
#define ILI9341_FRAMECONTROL    0xB1
#define ILI9341_DISPLAYFUNC     0xB6
#define ILI9341_ENTRYMODE       0xB7
#define ILI9341_POWERCONTROL1   0xC0
#define ILI9341_POWERCONTROL2   0xC1
#define ILI9341_VCOMCONTROL1    0xC5
#define ILI9341_VCOMCONTROL2    0xC7
#define ILI9341_MEMCONTROL 	0x36
#define ILI9341_MADCTL_MY  	0x80
#define ILI9341_MADCTL_MX  	0x40
#define ILI9341_MADCTL_MV  	0x20
#define ILI9341_MADCTL_ML  	0x10
#define ILI9341_MADCTL_RGB 	0x00
#define ILI9341_MADCTL_BGR 	0x08
#define ILI9341_MADCTL_MH  	0x04

#define ILI9341_Portrait        ILI9341_MADCTL_MX | ILI9341_MADCTL_BGR
#define ILI9341_Portrait180     ILI9341_MADCTL_MY | ILI9341_MADCTL_BGR
#define ILI9341_Landscape       ILI9341_MADCTL_MV | ILI9341_MADCTL_BGR
#define ILI9341_Landscape180    ILI9341_MADCTL_MY | ILI9341_MADCTL_MX | ILI9341_MADCTL_MV | ILI9341_MADCTL_BGR
//

#define LANDSCAPE       1
#define PORTRAIT        2
#define RLANDSCAPE      3
#define RPORTRAIT       4
#ifdef MX470
// SPI pin numbers and registers
#define SPI_INP_PIN        (HAS_100PINS ?  11 : 47)
#define SPI_OUT_PIN        (HAS_100PINS ? 12 :  5)
#define SPI_CLK_PIN        (HAS_100PINS ? 10 :  4)
#define SPI_PPS_OPEN       { if(HAS_100PINS) {PPSInput(2, SDI2, RPG7); PPSOutput(1, RPG8, SDO2); } else {PPSInput(2, SDI2, RPC13); PPSOutput(2, RPG7, SDO2);}}
#define SPICON *(volatile unsigned int *)(0xbf805A00)               //SPI status register
#define SPISTAT *(volatile unsigned int *)(0xbf805A10)              //SPI status register
#define SPIBUF *(volatile unsigned int *)(0xbf805A20)               //SPI output buffer
#define SPIBRG *(volatile unsigned int *)(0xbf805A30)               //SPI output buffer
#define SPICON2 *(volatile unsigned int *)(0xbf805A40)              //SPI status register
#else
// SPI pin numbers and registers
#define SPI_INP_PIN         (HAS_44PINS ? 41 : 14)
#define SPI_OUT_PIN         (HAS_44PINS ? 20 :  3)
#define SPI_CLK_PIN         (HAS_44PINS ? 14 : 25)
#define SPI_PPS_OPEN        PPSInput(2, SDI1, RPB5); PPSOutput(2, RPA1, SDO1)
#define SPI_PPS_CLOSE       PPSOutput(2, RPA1, NULL)
#define SPICON *(volatile unsigned int *)(0xbf805800)               //SPI status register
#define SPISTAT *(volatile unsigned int *)(0xbf805810)              //SPI status register
#define SPIBUF *(volatile unsigned int *)(0xbf805820)               //SPI output buffer
#define SPIBRG *(volatile unsigned int *)(0xbf805830)               //SPI output buffer
#define SPICON2 *(volatile unsigned int *)(0xbf805840)              //SPI status register
#endif
#define SPIsend(a) {int j;SPIBUF=a; while((SPISTAT & 0x80)==0); j=SPIBUF;}
#define SPIqueue(a) {while(SPISTAT & 0x02){};SPIBUF=a;}
#define swap(a, b) {int t; t = a; a = b; b = t; }


void spi_write_data(unsigned char data){
    PinSetBit(Option->LCD_CD, LATSET);
    PinSetBit(Option->LCD_CS, LATCLR);
    SPIsend(data);
    PinSetBit(Option->LCD_CS, LATSET);
}
void spi_write_command(unsigned char data){
    PinSetBit(Option->LCD_CD, LATCLR);
    PinSetBit(Option->LCD_CS, LATCLR);
    SPIsend(data);
    PinSetBit(Option->LCD_CS, LATSET);
}
void spi_write_cd(unsigned char command, int data, ...){
   int i;
   va_list ap;
   va_start(ap, data);
   spi_write_command(command);
   for(i = 0; i < data; i++) spi_write_data((char)va_arg(ap, int));
   va_end(ap);
}
/*******************************************************************************
 *
 * defines start/end coordinates for memory access from host to SSD1963
 * also maps the start and end points to suit the orientation
 *
 * This function is a modified version of the function inside the MMBasic Interpreter
 * for MM+ on 'MX470 chips
 *
*******************************************************************************/
void DefineRegion(int xstart, int ystart, int xend, int yend, int rw) {
    PinSetBit(Option->LCD_CD, LATCLR);
    SPIsend(ILI9341_COLADDRSET);
    PinSetBit(Option->LCD_CD, LATSET);
    SPIqueue(xstart >> 8);
    SPIqueue(xstart);
    SPIqueue(xend >> 8);
    SPIsend(xend);
    PinSetBit(Option->LCD_CD, LATCLR);
    SPIsend(ILI9341_PAGEADDRSET);
    PinSetBit(Option->LCD_CD, LATSET);
    SPIqueue(ystart >> 8);
    SPIqueue(ystart);
    SPIqueue(yend >> 8);
    SPIsend(yend);
    PinSetBit(Option->LCD_CD, LATCLR);
    if(rw){
        SPIsend(ILI9341_MEMORYWRITE);
    } else {
        SPIsend(ILI9341_RAMRD);
    }
    PinSetBit(Option->LCD_CD, LATSET);                               //set CD high
}
int ReadRectangle(int x1, int y1, int x2, int y2, int pos, char *p ){
    int r, N;
    N=(x2- x1+1) * (y2- y1+1);
    DefineRegion(x1, y1, x2, y2, 0);
    SPIBRG=0;
    SPIsend(0); //dummy read
    while(!(SPISTAT & 0x20)){r=SPIBUF;}   // clean up rx fifo if not empty
    while(N--) {                                 
        SPIBUF = 0;                       
        SPIBUF = 0;
        SPIBUF = 0;           // set up three reads
        while((SPISTAT & 0x80)==0);
        p[pos++] = SPIBUF;
        p[pos++] = SPIBUF;
        p[pos++] = SPIBUF;     
    } 
    PinSetBit(Option->LCD_CD, LATCLR); 
    PinSetBit(Option->LCD_CS, LATSET); 
    SPIBRG=0;
    PinSetBit(Option->LCD_CS, LATCLR); 
    return pos;
}
int RestoreRectangle(int x1, int y1, int x2, int y2,  int pos, char p[]){
    int N;
    N=(x2- x1+1) * (y2- y1+1);
    DefineRegion(x1, y1, x2, y2, 1);
    while(N--){
        SPIqueue(p[pos++]);
        SPIqueue(p[pos++]);
        SPIqueue(p[pos++]);
    }
    while((SPISTAT & 0x80)==0); //wait for all writes to complete
    while(!(SPISTAT & 0x20)){N=SPIBUF;}   // clean up rx fifo if not empty
#ifdef MX470
    SPI2STATCLR=0x40;
#else
    SPI1STATCLR=0x40;
#endif
    return pos;
}
//Print the bitmap of a char on the video output
//    x, y - the top left of the char
//    width, height - size of the char's bitmap
//    scale - how much to scale the bitmap
//	  fc, bc - foreground and background colour
//    bitmap - pointer to the butmap
void DrawBitmapSPI(int x1, int y1, int width, int height, int scale, int fc, int bc, unsigned char *bitmap){
    int i, j, k, m, n;
    unsigned char fhb, fmb, flb, bhb, bmb, blb;
    unsigned int consave=0,brgsave=0,con2save;
    int vertCoord, horizCoord, XStart, XEnd, YEnd;
    char *p=0;
    brgsave=SPIBRG; //save any user SPI setup
    consave=SPICON;
    con2save=SPICON2;
    SPICON=0x0;
    SPICON=0x018260;
    SPIBRG=0;
    SPICON2=0xC00;
    PinSetBit(Option->LCD_CS, LATCLR);
    // adjust when part of the bitmap is outside the displayable coordinates
    vertCoord = y1; if(y1 < 0) y1 = 0;                                 // the y coord is above the top of the screen
    XStart = x1; if(XStart < 0) XStart = 0;                            // the x coord is to the left of the left marginn
    XEnd = x1 + (width * scale) - 1; if(XEnd >= HRes) XEnd = HRes - 1; // the width of the bitmap will extend beyond the right margin
    YEnd = y1 + (height * scale) - 1; if(YEnd >= VRes) YEnd = VRes - 1;// the height of the bitmap will extend beyond the bottom margin
    if(bc==1){ //special case of overlay text
        i=0;
        j=width*height*scale*scale*3;
        p=GetMemory(j); //allocate some temporary memory
        ReadRectangle(XStart, y1, XEnd, YEnd, 0, p);
    }

    // convert the colours to 565 format
    // convert the colours to 666 format
    fhb= (fc>>16) & 0xFF;
    fmb= (fc>>8) & 0xFF;
    flb = fc & 0xFF;
    // convert the colours to 666 format
    bhb= (bc>>16) & 0xFF;
    bmb= (bc>>8) & 0xFF;
    blb = bc & 0xFF;
    DefineRegion(XStart, y1, XEnd, YEnd, 1);
    n=0;
    for(i = 0; i < height; i++) {                                   // step thru the font scan line by line
        for(j = 0; j < scale; j++) {                                // repeat lines to scale the font
            if(vertCoord++ < 0) continue;                           // we are above the top of the screen
            if(vertCoord > VRes) return;                            // we have extended beyond the bottom of the screen
            horizCoord = x1;
            for(k = 0; k < width; k++) {                            // step through each bit in a scan line
                for(m = 0; m < scale; m++) {                        // repeat pixels to scale in the x axis
                    if(horizCoord++ < 0) continue;                  // we have not reached the left margin
                    if(horizCoord > HRes) continue;                 // we are beyond the right margin
                    if((bitmap[((i * width) + k)/8] >> (((height * width) - ((i * width) + k) - 1) %8)) & 1) {
                       SPIqueue(fhb);
                       SPIqueue(fmb);
                       SPIqueue(flb);
                       n+=3;
                    } else {
                        if(bc==1){
                            SPIqueue(p[n++]);
                            SPIqueue(p[n++]);
                            SPIqueue(p[n++]);
                        } else {
                            SPIqueue(bhb);
                            SPIqueue(bmb);
                            SPIqueue(blb);
                        }
                    }
                }
            }
        }
    }
    while((SPISTAT & 0x80)==0); //wait for all writes to complete
    while(!(SPISTAT & 0x20)){i=SPIBUF;}   // clean up rx fifo if not empty
    SPICON=0;
#ifdef MX470
    SPI2STATCLR=0x40;
#else
    SPI1STATCLR=0x40;
#endif
    
    SPIBRG=brgsave;  //restore user (or my) setup
    SPICON=consave;
    SPICON2=con2save;
    if(p)FreeMemory(p);
    PinSetBit(Option->LCD_CS, LATSET);
}
// Draw a rectangle
// this is the basic drawing primitive used by most drawing routines
//    x1, y1, x2, y2 - the coordinates
//    c - the colour
void DrawRectangleSPI(int x1, int y1, int x2, int y2, int c){
    unsigned int consave=0,brgsave=0,con2save;
	int N,t;
    unsigned char hb, mb, lb;
    brgsave=SPIBRG; //save any user SPI setup
    consave=SPICON;
    con2save=SPICON2;
    SPICON=0x0;
    SPICON=0x018260;
    SPIBRG=0;
    SPICON2=0xC00;
    PinSetBit(Option->LCD_CS, LATCLR);
    // make sure the coordinates are kept within the display area
    if(x2 <= x1) { t = x1; x1 = x2; x2 = t; }
    if(y2 <= y1) { t = y1; y1 = y2; y2 = t; }
    if(x1 < 0) x1 = 0; if(x1 >= HRes) x1 = HRes - 1;
    if(x2 < 0) x2 = 0; if(x2 >= HRes) x2 = HRes - 1;
    if(y1 < 0) y1 = 0; if(y1 >= VRes) y1 = VRes - 1;
    if(y2 < 0) y2 = 0; if(y2 >= VRes) y2 = VRes - 1;
    // convert the colours to 666 format
    hb= (c>>16) & 0xFF;
    mb= (c>>8) & 0xFF;
    lb = c & 0xFF;
    DefineRegion(x1, y1, x2, y2, 1);
    N=(x2- x1+1) * (y2- y1+1);
    while(N--){
        SPIqueue(hb);
        SPIqueue(mb);
        SPIqueue(lb);
    }
    while((SPISTAT & 0x80)==0); //wait for all writes to complete
    while(!(SPISTAT & 0x20)){N=SPIBUF;}   // clean up rx fifo if not empty
    SPICON=0;
#ifdef MX470
    SPI2STATCLR=0x40;
#else
    SPI1STATCLR=0x40;
#endif

    SPIBRG=brgsave; //restore user (or my) setup
    SPICON=consave;
    SPICON2=con2save;
    PinSetBit(Option->LCD_CS, LATSET);
}
long triangles(int mode, int buffpos, char *buff, long  col, long x0, long y0, long x1, long y1,long x2, long y2) {//Cfunction
  long a, b, y, last;
  long  dx01,  dy01,  dx02,  dy02, dx12,  dy12,  sa, sb;
      if (y0>y1) {
        swap(y0, y1);
        swap(x0, x1);
      }
      if (y1>y2) {
        swap(y2, y1);
        swap(x2, x1);
      }
      if (y0>y1) {
        swap(y0, y1);
        swap(x0, x1);
      }
      if(y0==y2) { // Handle awkward all-on-same-line case as its own thing
        a=x0;
        b=x0;
        if(x1<a) {
          a=x1;
        } else {
          if(x1>b) b=x1;
        }
        if(x2<a) {
          a=x2;
        } else {
          if(x2>b)  b=x2;
        }
        if(mode==0){
            buffpos=ReadRectangle(a,y0,a,b-a+y0,buffpos,buff);
        } else if(mode==1) {
            buffpos=RestoreRectangle(a,y0,a,b-a+y0,buffpos,buff);
        } else {
            DrawRectangleSPI(a,y0,a,b-a+y0,col);
        }
      } else {
        dx01=x1-x0;  dy01=y1-y0;  dx02=x2-x0;  dy02=y2-y0; dx12=x2-x1;  dy12=y2-y1;  sa=0; sb=0;
        if(y1==y2) {
            last=y1; //Include y1 scanline
        } else {
            last=y1-1; // Skip it
        }
        for (y=y0;y<=last;y++){
            a=x0+sa / dy01;
            b=x0+sb / dy02;
            sa=sa+dx01;
            sb=sb+dx02;
            a=x0+(x1-x0) * (y-y0) / (y1-y0);
            b=x0+(x2-x0) * (y-y0) / (y2-y0);
            if(a>b)swap(a,b);
            if(mode==0){
                buffpos=ReadRectangle(a,y,b,y,buffpos,buff);
            } else if(mode==1) {
                buffpos=RestoreRectangle(a,y,b,y,buffpos,buff);
            } else {
                DrawRectangleSPI(a,y,b,y,col);
            }
        }
        sa=dx12 * (y-y1);
        sb=dx02 * (y-y0);
        while (y<=y2){
            a=x1+sa / dy12;
            b=x0+sb / dy02;
            sa=sa+dx12;
            sb=sb+dx02;
            a=x1+(x2-x1) * (y-y1) / (y2-y1);
            b=x0+(x2-x0) * (y-y0) / (y2-y0);
            if(a>b) swap(a,b);
            if(mode==0){
                buffpos=ReadRectangle(a,y,b,y,buffpos,buff);
            } else if(mode==1) {
                buffpos=RestoreRectangle(a,y,b,y,buffpos,buff);
            } else {
                DrawRectangleSPI(a,y,b,y,col);
            }
            y=y+1;
            }
        }

  return buffpos;
}
__attribute__((noinline)) void getFPC(void *a, void *b, volatile unsigned int *c) 
     { 
         *c = (unsigned int) (__builtin_return_address (0) - (b -a)) ;      
     } 
void pstring(const char *s){
    volatile unsigned int libAddr ; 
    getFPC(NULL,&&getFPCLab,&libAddr) ; // warning can be ignored, stupid editor 
    getFPCLab: { } 
    unsigned char  * testData    = (unsigned char *)((void *)s + libAddr );
    MMPrintString(testData);
}
/*******************************************************************************
 *
 * ILI9341 : Initialise the CFunction Driver Sub-System
 *
 * Function called to initialise the driver SubSystem
 *
 * ILI9341 is ALWAYS called from an MMBasic program
 * On exit, vectors DrawRectangleVector, and DrawBitmapVector will
 * be set to point to the CFunctions DrawRectangleSPI and
 * DrawBitmapSPI respectively
 *
 * Entry point is function long long main(long long *MyAddress,
 *                                        long long *DC,
 *                                        long long *RST,
 *                                        long long *CS
 *                                        long long *orientation)
 *                                        long long *size)
 * 
 ******************************************************************************/
//CFunction Driver_ILI9341
long long main(int *orientation, unsigned char *p1, long long p2[], long long p3[], long long p4[], long long p5[], long long p6[],long long p7[], long long p8[]){
    volatile unsigned int libAddr ; 
    getFPC(NULL,&&getFPCLab,&libAddr) ; // warning can be ignored, stupid editor 
    getFPCLab: { } 
    int HorizontalRes=240,VerticalRes=320;
    unsigned int i,consave=0,brgsave=0,con2save,ret=-1;
    brgsave=SPIBRG;
    consave=SPICON;
    con2save=SPICON2;
    if(*orientation>4){
        SPICON=0x0;
        SPICON=0x018260;
        SPIBRG=0;
        SPICON2=0xC00;
        PinSetBit(Option->LCD_CS, LATCLR);
    }
    if(*orientation>=0 && *orientation<=4){
        Option->LCD_Reset=p2[0];
        Option->LCD_CD=*p1;
        Option->LCD_CS=p3[0];
        Option->DISPLAY_ORIENTATION=*orientation;
        if(ExtCurrentConfig[Option->LCD_Reset] != EXT_BOOT_RESERVED){ExtCfg(Option->LCD_Reset,EXT_DIG_OUT,0);ExtCfg(Option->LCD_Reset,EXT_BOOT_RESERVED,0);}
        PinSetBit(Option->LCD_Reset, LATSET);
        if(ExtCurrentConfig[Option->LCD_CD] != EXT_BOOT_RESERVED){ExtCfg(Option->LCD_CD,EXT_DIG_OUT,0);ExtCfg(Option->LCD_CD,EXT_BOOT_RESERVED,0);}
        PinSetBit(Option->LCD_CD, LATSET);
        if(ExtCurrentConfig[Option->LCD_CS] != EXT_BOOT_RESERVED){ExtCfg(Option->LCD_CS,EXT_DIG_OUT,0);ExtCfg(Option->LCD_CS,EXT_BOOT_RESERVED,0);}
        PinSetBit(Option->LCD_CS, LATSET);

        if(ExtCurrentConfig[SPI_OUT_PIN] != EXT_BOOT_RESERVED) { //already open
        ExtCfg(SPI_OUT_PIN, EXT_DIG_OUT, 0); ExtCfg(SPI_OUT_PIN, EXT_BOOT_RESERVED, 0);
        ExtCfg(SPI_INP_PIN, EXT_DIG_IN, 0); ExtCfg(SPI_INP_PIN, EXT_BOOT_RESERVED, 0);
        ExtCfg(SPI_CLK_PIN, EXT_DIG_OUT, 0); ExtCfg(SPI_CLK_PIN, EXT_BOOT_RESERVED, 0);
        }
        SPI_PPS_OPEN; 
        SPICON=0x060;
        SPICON=0x018260;
        SPIBRG=1;
        SPICON2=0xC00;// this is defined in IOPorts.h
        //Reset the SSD1963
        PinSetBit(Option->LCD_Reset,LATSET);
        uSec(10000);
        PinSetBit(Option->LCD_Reset,LATCLR);
        uSec(10000);
        PinSetBit(Option->LCD_Reset,LATSET);
        uSec(10000);

        spi_write_command(ILI9341_DISPLAYOFF);
        spi_write_cd(ILI9341_POWERCONTROL1,1,0x23);
        spi_write_cd(ILI9341_POWERCONTROL2,1,0x10);
        spi_write_cd(ILI9341_VCOMCONTROL1,2,0x2B,0x2B);
        spi_write_cd(ILI9341_VCOMCONTROL2,1,0xC0);
        spi_write_cd(ILI9341_PIXELFORMAT,1,0x66);
        spi_write_cd(ILI9341_FRAMECONTROL,2,0x00,0x1B);
        spi_write_cd(ILI9341_ENTRYMODE,1,0x07);
        spi_write_cd(ILI9341_SLEEPOUT,1,0);
        uSec(50000);
        spi_write_command(ILI9341_NORMALDISP);
        spi_write_command(ILI9341_DISPLAYON);
        uSec(100000);
        if(Option->DISPLAY_ORIENTATION==LANDSCAPE)     spi_write_cd(ILI9341_MEMCONTROL,1,ILI9341_Landscape);
        if(Option->DISPLAY_ORIENTATION==PORTRAIT)      spi_write_cd(ILI9341_MEMCONTROL,1,ILI9341_Portrait); 
        if(Option->DISPLAY_ORIENTATION==RLANDSCAPE)    spi_write_cd(ILI9341_MEMCONTROL,1,ILI9341_Landscape180);
        if(Option->DISPLAY_ORIENTATION==RPORTRAIT)     spi_write_cd(ILI9341_MEMCONTROL,1,ILI9341_Portrait180);

        if(Option->DISPLAY_ORIENTATION&1){
            VRes=HorizontalRes;
            HRes=VerticalRes;
        } else {
            HRes=HorizontalRes;
            VRes=VerticalRes;
        }

        //Set the DrawRectangle vector to point to our function
        DrawRectangleVector= (unsigned int)&DrawRectangleSPI + libAddr;

        //Set the DrawBitmap vector to point to our function
        DrawBitmapVector=(unsigned int)&DrawBitmapSPI + libAddr;

        //CLS
        DrawRectangle(0,0,HRes-1,VRes-1,0x000000);

        static const char startup[]="ILI9341 driver loaded\r\n";
        pstring(startup);
        ret=0;
    } else if(*orientation==5){
        int num=p1[0],a=1;
        PinSetBit(Option->LCD_CD, LATSET);                               //set CD high
        while(a<=num){  
            SPIqueue(p1[a++]);
            SPIqueue(p1[a++]);
            SPIqueue(p1[a++]);
        }
        ret= a;
    } else if(*orientation==6){
            ret=ReadRectangle(*p2, *p3, *p2+*p4-1, *p3+*p5-1, *p6, p1 );

    } else if(*orientation==7){
            ret=RestoreRectangle(*p2, *p3, *p2+*p4-1, *p3+*p5-1, *p6, p1 );
    } else if(*orientation==8){
        DefineRegion(*p2, *p3, *p2+*p4-1, *p3+*p5-1, *p1);
        ret=8;
    } else if(*orientation==9){
        triangles(2,0,0,p2[0],p3[0],p4[0],p5[0],p6[0],p7[0],p8[0]);
        ret=9;
    } else { //must be triangles
        int count=*orientation & 0x7F;
        int i=0,j,buffpos=0;
        while(i<count && p2[i]<0){ //repeat for any restores
            buffpos=triangles(1,buffpos,p1,p2[i],p3[i],p4[i],p5[i],p6[i],p7[i],p8[i]);
            i++;
        }  
        
        j=i;
        while(i<count && p2[i]>=0){ //repeat for any store and write commands
            if(j==i)buffpos=0;
            buffpos=triangles(0,buffpos,p1,p2[i],p3[i],p4[i],p5[i],p6[i],p7[i],p8[i]);
            i++;
        }
        i=j;
        while(i<count && p2[i]>=0){ //repeat for any store and write commands
            buffpos=triangles(2,buffpos,p1,p2[i],p3[i],p4[i],p5[i],p6[i],p7[i],p8[i]);
            PinSetBit(Option->LCD_CS, LATCLR);        
            i++;
        }
       ret= buffpos;        
    }
    while((SPISTAT & 0x80)==0); //wait for all writes to complete
    while(!(SPISTAT & 0x20)){i=SPIBUF;}   // clean up rx fifo if not empty
    SPICON=0;
#ifdef MX470
    SPI2STATCLR=0x40;
#else
    SPI1STATCLR=0x40;
#endif
    
    SPIBRG=brgsave;  //restore user (or my) setup
    SPICON=consave;
    SPICON2=con2save;
    PinSetBit(Option->LCD_CS, LATSET);
    return ret;
}
