/*******************************************************************************
 *
 * Driver for ST7735 Display written as CFunctions
 *
 * (c) Peter Mather 2015 with acknowledgements to Peter Carnegie & Geoff Graham
 * 
 *
 * This CFunction MUST be compiled with Optimization Level 1, -O1
 * -O2,-O3,-Os will compile successfully, but generate exceptions at runtime.
 *
 * When Generating the CFunction, use MERGE CFunction mode, and name the CFunction
 * SSD1963_V44
 *
 * Entry point is function long long main(long long *MyAddress,
 *                                        long long *DC,
 *                                        long long *RST,
 *                                        long long *CS
 *                                        long long *orientation)
 *
 * V1.0     2015-09-23 Peter Mather
 * 
 ******************************************************************************/
#include <stdarg.h>

#define Version 100     //Version 1.00
#define _SUPPRESS_PLIB_WARNING                                      // required for XC1.33  Later compiler versions will need PLIB to be installed
#include <plib.h>                                                   // the pre Harmony peripheral libraries


#include "../cfunctions.h"
#define ST7735_NOP              0x0
#define ST7735_SWRESET          0x01
#define ST7735_RDDID            0x04
#define ST7735_RDDST            0x09
#define ST7735_SLPIN            0x10
#define ST7735_SLPOUT           0x11
#define ST7735_PTLON            0x12
#define ST7735_NORON            0x13
#define ST7735_INVOFF           0x20
#define ST7735_INVON            0x21
#define ST7735_DISPOFF          0x28
#define ST7735_DISPON           0x29
#define ST7735_CASET            0x2A
#define ST7735_RASET            0x2B
#define ST7735_RAMWR            0x2C
#define ST7735_RAMRD            0x2E
#define ST7735_PTLAR            0x30
#define ST7735_MADCTL           0x36
#define ST7735_COLMOD           0x3A
#define ST7735_FRMCTR1          0xB1
#define ST7735_FRMCTR2          0xB2
#define ST7735_FRMCTR3          0xB3
#define ST7735_INVCTR           0xB4
#define ST7735_DISSET5          0xB6
#define ST7735_PWCTR1           0xC0
#define ST7735_PWCTR2           0xC1
#define ST7735_PWCTR3           0xC2
#define ST7735_PWCTR4           0xC3
#define ST7735_PWCTR5           0xC4
#define ST7735_VMCTR1           0xC5
#define ST7735_RDID1            0xDA
#define ST7735_RDID2            0xDB
#define ST7735_RDID3            0xDC
#define ST7735_RDID4            0xDD
#define ST7735_PWCTR6           0xFC
#define ST7735_GMCTRP1          0xE0
#define ST7735_GMCTRN1          0xE1
#define ST7735_Portrait         0xC0
#define ST7735_Portrait180      0
#define ST7735_Landscape        0xA0
#define ST7735_Landscape180     0x60
#define ST77XX_SWRESET    0x01
#define ST77XX_DISPON     0x29
#define ST77XX_CASET      0x2A
#define ST77XX_RASET      0x2B
#define ST77XX_INVON      0x21
#define ST77XX_NORON      0x13
#define ST77XX_SLPOUT     0x11
#define ST77XX_COLMOD     0x3A

#define LANDSCAPE       1
#define PORTRAIT        2
#define RLANDSCAPE      3
#define RPORTRAIT       4
#define MX170
#if defined(MX170)
// SPI pin numbers and registers
#define SPI_INP_PIN         (HAS_44PINS ? 41 : 14)
#define SPI_OUT_PIN         (HAS_44PINS ? 20 :  3)
#define SPI_CLK_PIN         (HAS_44PINS ? 14 : 25)
#define SPI_PPS_OPEN        PPSInput(2, SDI1, RPB5); PPSOutput(2, RPA1, SDO1)
#define SPI_PPS_CLOSE       PPSOutput(2, RPA1, NULL)
#define SPICON *(volatile unsigned int *)(0xbf805800)               //SPI status register
#define SPISTAT *(volatile unsigned int *)(0xbf805810)              //SPI status register
#define SPIBUF *(volatile unsigned int *)(0xbf805820)               //SPI output buffer
#define SPIBRG *(volatile unsigned int *)(0xbf805830)               //SPI output buffer
#define SPICON2 *(volatile unsigned int *)(0xbf805840)              //SPI status register
#define SPISTATCLR *(volatile unsigned int *)(0xbf805814)           //SPI status clear register
#define ILImode 0x18520
#define ILIbrg 0
#elif defined(MX470)
// SPI pin numbers and registers
#define SPI_INP_PIN        (HAS_100PINS ?  11 : 47)
#define SPI_OUT_PIN        (HAS_100PINS ? 12 :  5)
#define SPI_CLK_PIN        (HAS_100PINS ? 10 :  4)
#define SPI_PPS_OPEN        { if(HAS_100PINS) {PPSInput(2, SDI2, RPG7); PPSOutput(1, RPG8, SDO2); } else {PPSInput(2, SDI2, RPC13); PPSOutput(2, RPG7, SDO2);}}
#define SPI_PPS_CLOSE       { if(HAS_100PINS) PPSOutput(1, RPG8, NULL); else PPSOutput(2, RPG7, NULL); }
#define SPICON *(volatile unsigned int *)(0xbf805A00)               //SPI status register
#define SPISTAT *(volatile unsigned int *)(0xbf805A10)              //SPI status register
#define SPIBUF *(volatile unsigned int *)(0xbf805A20)               //SPI output buffer
#define SPIBRG *(volatile unsigned int *)(0xbf805A30)               //SPI output buffer
#define SPICON2 *(volatile unsigned int *)(0xbf805A40)              //SPI status register
#define SPISTATCLR *(volatile unsigned int *)(0xbf805A14)           //SPI status clear register
#endif
#define SPIsend(a) {int j;SPIBUF=a; while((SPISTAT & 0x80)==0); j=SPIBUF;}
#define SPIqueue(a) {while(SPISTAT & 0x02){};SPIBUF=a;}


#define SPIsend(a) {int j;SPIBUF=a; while((SPISTAT & 0x80)==0); j=SPIBUF;}
// set the chip select for the SPI to low (enabled)
// if the SPI is currently set to a different mode or baudrate this will change it accordingly
// also, it checks if the chip select pin needs to be changed
// set the chip select for SPI1 to high (disabled)
void spi_write_data(unsigned char data){
    PinSetBit(Option->LCD_CD, LATSET);
    PinSetBit(Option->LCD_CS, LATCLR);
    SPIsend(data);
    PinSetBit(Option->LCD_CS, LATSET);
}
void spi_write_command(unsigned char data){
    PinSetBit(Option->LCD_CD, LATCLR);
    PinSetBit(Option->LCD_CS, LATCLR);
    SPIsend(data);
    PinSetBit(Option->LCD_CS, LATSET);
}
void spi_write_cd(unsigned char command, int data, ...){
   int i;
   va_list ap;
   va_start(ap, data);
   spi_write_command(command);
   for(i = 0; i < data; i++) spi_write_data((char)va_arg(ap, int));
   va_end(ap);
}
/*******************************************************************************
 *
 * defines start/end coordinates for memory access from host to SSD1963
 * also maps the start and end points to suit the orientation
 *
 * This function is a modified version of the function inside the MMBasic Interpreter
 * for MM+ on 'MX470 chips
 *
*******************************************************************************/
void DefineRegion(int xstart, int ystart, int xend, int yend) {
    if(Option->DISPLAY_ORIENTATION==2){
        ystart+=80;
        yend+=80;
    }
    if(Option->DISPLAY_ORIENTATION==1){
        xstart+=80;
        xend+=80;
    }
    PinSetBit(Option->LCD_CD, LATCLR);
    PinSetBit(Option->LCD_CS, LATCLR);
    SPIsend(ST7735_CASET);
    PinSetBit(Option->LCD_CD, LATSET);
    SPIsend(xstart >> 8);
    SPIsend(xstart);
    SPIsend(xend >> 8);
    SPIsend(xend);
    PinSetBit(Option->LCD_CD, LATCLR);
    SPIsend(ST7735_RASET);
    PinSetBit(Option->LCD_CD, LATSET);
    SPIsend(ystart >> 8);
    SPIsend(ystart);
    SPIsend(yend >> 8);
    SPIsend(yend);
    PinSetBit(Option->LCD_CD, LATCLR);
    SPIsend(ST7735_RAMWR);
    PinSetBit(Option->LCD_CS, LATSET);
}


//Print the bitmap of a char on the video output
//    x, y - the top left of the char
//    width, height - size of the char's bitmap
//    scale - how much to scale the bitmap
//	  fc, bc - foreground and background colour
//    bitmap - pointer to the bitmap
void DrawBitmapSPI(int x1, int y1, int width, int height, int scale, int fc, int bc, unsigned char *bitmap){
    int i, j, k, m;
    unsigned char fhb, flb, bhb, blb;
    unsigned int consave=0,brgsave=0,con2save;
    int vertCoord, horizCoord, XStart, XEnd, YEnd;
    if(x1>=HRes || y1>=VRes || x1+width*scale<0 || y1+height*scale<0)return;

    // adjust when part of the bitmap is outside the displayable coordinates
    vertCoord = y1; if(y1 < 0) y1 = 0;                                 // the y coord is above the top of the screen
    XStart = x1; if(XStart < 0) XStart = 0;                            // the x coord is to the left of the left marginn
    XEnd = x1 + (width * scale) - 1; if(XEnd >= HRes) XEnd = HRes - 1; // the width of the bitmap will extend beyond the right margin
    YEnd = y1 + (height * scale) - 1; if(YEnd >= VRes) YEnd = VRes - 1;// the height of the bitmap will extend beyond the bottom margin
    brgsave=SPIBRG; //save any user SPI setup
    consave=SPICON;
    con2save=SPICON2;
    SPICON=0x0;
    SPICON=0x018260;
#ifdef MX470
    SPIBRG=0;
#else
    SPIBRG=0;
#endif
    SPICON2=0xC00;
     // convert the colours to 565 format
    fhb = ((fc >> 16) & 0b11111000) | ((fc >> 13) & 0b00000111);
    flb = ((fc >>  5) & 0b11100000) | ((fc >>  3) & 0b00011111);
    bhb = ((bc >> 16) & 0b11111000) | ((bc >> 13) & 0b00000111);
    blb = ((bc >>  5) & 0b11100000) | ((bc >>  3) & 0b00011111);

    DefineRegion(x1, y1, x1 + (width * scale) - 1, y1 + (height * scale) -1);
    PinSetBit(Option->LCD_CD, LATSET);                               //set CD high
    PinSetBit(Option->LCD_CS, LATCLR);
    for(i = 0; i < height; i++) {                                   // step thru the font scan line by line
        for(j = 0; j < scale; j++) {                                // repeat lines to scale the font
            if(vertCoord++ < 0) continue;                           // we are above the top of the screen
            if(vertCoord > VRes) return;                            // we have extended beyond the bottom of the screen
            horizCoord = x1;
            for(k = 0; k < width; k++) {                            // step through each bit in a scan line
                for(m = 0; m < scale; m++) {                        // repeat pixels to scale in the x axis
                    if(horizCoord++ < 0) continue;                  // we have not reached the left margin
                    if(horizCoord > HRes) continue;                 // we are beyond the right margin
                    if((bitmap[((i * width) + k)/8] >> (((height * width) - ((i * width) + k) - 1) %8)) & 1) {
                        SPIqueue(fhb); SPIqueue(flb);
                    } else {
                        SPIqueue(bhb); SPIqueue(blb);
                    }
                }
            }
        }
    }
    while((SPISTAT & 0x80)==0); //wait for all writes to complete
    while(!(SPISTAT & 0x20)){i=SPIBUF;}   // clean up rx fifo if not empty
    SPICON=0;
#ifdef MX470
    SPI2STATCLR=0x40;
#else
    SPI1STATCLR=0x40;
#endif
    
    SPIBRG=brgsave;  //restore user (or my) setup
    SPICON=consave;
    SPICON2=con2save;
    PinSetBit(Option->LCD_CS, LATSET);
}
// Draw a rectangle
// this is the basic drawing primitive used by most drawing routines
//    x1, y1, x2, y2 - the coordinates
//    c - the colour
void DrawRectangleSPI(int x1, int y1, int x2, int y2, int c){
    unsigned int consave=0,brgsave=0,con2save;
	int i, t;
    unsigned char hb, lb;
    brgsave=SPIBRG; //save any user SPI setup
    consave=SPICON;
    con2save=SPICON2;
    SPICON=0x0;
    SPICON=0x018260;
#ifdef MX470
    SPIBRG=0;
#else
    SPIBRG=0;
#endif
    SPICON2=0xC00;
    // make sure the coordinates are kept within the display area
    if(x2 <= x1) { t = x1; x1 = x2; x2 = t; }
    if(y2 <= y1) { t = y1; y1 = y2; y2 = t; }
    if(x1 < 0) x1 = 0; if(x1 >= HRes) x1 = HRes - 1;
    if(x2 < 0) x2 = 0; if(x2 >= HRes) x2 = HRes - 1;
    if(y1 < 0) y1 = 0; if(y1 >= VRes) y1 = VRes - 1;
    if(y2 < 0) y2 = 0; if(y2 >= VRes) y2 = VRes - 1;

    // convert the colours to 565 format
    hb = ((c >> 16) & 0b11111000) | ((c >> 13) & 0b00000111);
    lb = ((c >> 5) & 0b11100000) | ((c >> 3) & 0b00011111);

    DefineRegion(x1, y1, x2, y2);
    PinSetBit(Option->LCD_CD, LATSET);                               //set CD high
    PinSetBit(Option->LCD_CS, LATCLR);
	i = x2 - x1 + 1;
	i *= (y2 - y1 + 1);
    while(i--){
        SPIqueue(hb);
        SPIqueue(lb);
     }
    while((SPISTAT & 0x80)==0); //wait for all writes to complete
    while(!(SPISTAT & 0x20)){i=SPIBUF;}   // clean up rx fifo if not empty
    SPICON=0;
#ifdef MX470
    SPI2STATCLR=0x40;
#else
    SPI1STATCLR=0x40;
#endif

    SPIBRG=brgsave; //restore user (or my) setup
    SPICON=consave;
    SPICON2=con2save;
    PinSetBit(Option->LCD_CS, LATSET);
}
#ifdef MX470
void DrawBufferSPI(int x1, int y1, int x2, int y2, char* p) {
    unsigned int i, consave=0,brgsave=0,con2save;
    brgsave=SPIBRG; //save any user SPI setup
    consave=SPICON;
    con2save=SPICON2;
    SPICON=0x0;
    SPICON=0x018260;
#ifdef MX470
    SPIBRG=0;
#else
    SPIBRG=0;
#endif
    SPICON2=0xC00;
    unsigned char hb, lb;
    union colourmap
    {
    char rgbbytes[4];
    unsigned int rgb;
    } c;
    int xx1=x1, yy1=y1, xx2=x2, yy2=y2, x, y, t ;
    // make sure the coordinates are kept within the display area
    if(x2 <= x1) { t = x1; x1 = x2; x2 = t; }
    if(y2 <= y1) { t = y1; y1 = y2; y2 = t; }
    if(x1 < 0) xx1 = 0; if(x1 >= HRes) xx1 = HRes - 1;
    if(x2 < 0) xx2 = 0; if(x2 >= HRes) xx2 = HRes - 1;
    if(y1 < 0) yy1 = 0; if(y1 >= VRes) yy1 = VRes - 1;
    if(y2 < 0) yy2 = 0; if(y2 >= VRes) yy2 = VRes - 1;


    DefineRegion(xx1, yy1, xx2, yy2);
    PinSetBit(Option->LCD_CD, LATSET);                               //set CD high
    PinSetBit(Option->LCD_CS, LATCLR);


    for(y=y1;y<=y2;y++){
        for(x=x1;x<=x2;x++){
            if(x>=0 && x<HRes && y>=0 && y<VRes){
                c.rgbbytes[0]=*p++; //this order swaps the bytes to match the .BMP file
                c.rgbbytes[1]=*p++;
                    c.rgbbytes[2]=*p++;
            // convert the colours to 565 format
                hb = ((c.rgb >> 16) & 0b11111000) | ((c.rgb >> 13) & 0b00000111);
                lb = ((c.rgb >> 5) & 0b11100000) | ((c.rgb >> 3) & 0b00011111);
                SPIqueue(hb);
                SPIqueue(lb);
            } else p+=3;
        }
    }      

    while((SPISTAT & 0x80)==0); //wait for all writes to complete
    while(!(SPISTAT & 0x20)){i=SPIBUF;}   // clean up rx fifo if not empty
    SPICON=0;
#ifdef MX470
    SPI2STATCLR=0x40;
#else
    SPI1STATCLR=0x40;
#endif

    SPIBRG=brgsave; //restore user (or my) setup
    SPICON=consave;
    SPICON2=con2save;
    PinSetBit(Option->LCD_CS, LATSET);
}
#endif

__attribute__((noinline)) void getFPC(void *a, void *b, volatile unsigned int *c) 
     { 
         *c = (unsigned int) (__builtin_return_address (0) - (b -a)) ;      
     } 
void pstring(const char *s){
    volatile unsigned int libAddr ; 
    getFPC(NULL,&&getFPCLab,&libAddr) ; // warning can be ignored, stupid editor 
    getFPCLab: { } 
    unsigned char  * testData    = (unsigned char *)((void *)s + libAddr );
    MMPrintString(testData);
}

/*******************************************************************************
 *
 * ST7735 : Initialise the CFunction Driver Sub-System
 *
 * Function called to initialise the driver SubSystem
 *
 * ST7735 is ALWAYS called from an MMBasic program
 * On exit, vectors DrawRectangleVector, and DrawBitmapVector will
 * be set to point to the CFunctions DrawRectangleSPI and
 * DrawBitmapSPI respectively
 *
 * Entry point is function long long main(long long *MyAddress,
 *                                        long long *DC,
 *                                        long long *RST,
 *                                        long long *CS
 *                                        long long *orientation)
 *                                        long long *size)
 * 
 ******************************************************************************/
//CFunction Driver_ST7735
void main(long long *CD, long long *RST, long long *CS,long long *orientation){
    volatile unsigned int libAddr ; 
    getFPC(NULL,&&getFPCLab,&libAddr) ; // warning can be ignored, stupid editor 
    getFPCLab: { } 
    int HorizontalRes=240,VerticalRes=240;
    unsigned int consave=0,brgsave=0,con2save;
    Option->LCD_Reset=*RST;
    Option->LCD_CD=*CD;
    Option->LCD_CS=*CS;
    Option->DISPLAY_ORIENTATION=*orientation + 2;
    while(Option->DISPLAY_ORIENTATION>3)Option->DISPLAY_ORIENTATION-=4;
    Option->DISPLAY_TYPE=20;
    
    ExtCfg(Option->LCD_Reset,EXT_DIG_OUT,0);ExtCfg(Option->LCD_Reset,EXT_BOOT_RESERVED,0);
    PinSetBit(Option->LCD_Reset, LATSET);
    ExtCfg(Option->LCD_CD,EXT_DIG_OUT,0);ExtCfg(Option->LCD_CD,EXT_BOOT_RESERVED,0);
    PinSetBit(Option->LCD_CD, LATSET);
    ExtCfg(Option->LCD_CS,EXT_DIG_OUT,0);ExtCfg(Option->LCD_CS,EXT_BOOT_RESERVED,0);
    PinSetBit(Option->LCD_CS, LATSET);
    if(ExtCurrentConfig[SPI_OUT_PIN] == EXT_RESERVED) { //already open
        brgsave=SPIBRG;
        consave=SPICON;
        con2save=SPICON2;
    }
    if(ExtCurrentConfig[SPI_OUT_PIN]!=EXT_BOOT_RESERVED){
        ExtCfg(SPI_OUT_PIN, EXT_DIG_OUT, 0); ExtCfg(SPI_OUT_PIN, EXT_BOOT_RESERVED, 0);
        ExtCfg(SPI_INP_PIN, EXT_DIG_IN, 0); ExtCfg(SPI_INP_PIN, EXT_BOOT_RESERVED, 0);
        ExtCfg(SPI_CLK_PIN, EXT_DIG_OUT, 0); ExtCfg(SPI_CLK_PIN, EXT_BOOT_RESERVED, 0);
        SPI_PPS_OPEN; 
    }
    SPICON=0x8060;
    SPIBRG=3;
    SPICON2=0xC00;// this is defined in IOPorts.h
    if(!brgsave){ //save my settings
        brgsave=SPIBRG;
        consave=SPICON;
        con2save=SPICON2;
    }    


   //Reset the ST7735
    PinSetBit(Option->LCD_Reset,LATSET);
    uSec(10000);
    PinSetBit(Option->LCD_Reset,LATCLR);
    uSec(10000);
    PinSetBit(Option->LCD_Reset,LATSET);
    spi_write_command(ST77XX_SWRESET);    uSec(150000);
    spi_write_command(ST77XX_SLPOUT);    uSec(500000);
    spi_write_command(ST77XX_COLMOD);    spi_write_data(0x55);uSec(10000);
    spi_write_command(ST77XX_CASET); spi_write_data(0x0); spi_write_data(0x0); spi_write_data(0x0); spi_write_data(240);
    spi_write_command(ST77XX_RASET); spi_write_data(0x0); spi_write_data(0); spi_write_data(0); spi_write_data(240);
    spi_write_command(ST77XX_INVON);    uSec(10000);
    spi_write_command(ST77XX_NORON);    uSec(10000);
    spi_write_command(ST77XX_DISPON);    uSec(500000);
    if(Option->DISPLAY_ORIENTATION==LANDSCAPE)     spi_write_cd(ST7735_MADCTL, 1, ST7735_Landscape);
    if(Option->DISPLAY_ORIENTATION==PORTRAIT)      spi_write_cd(ST7735_MADCTL, 1, ST7735_Portrait); 
    if(Option->DISPLAY_ORIENTATION==RLANDSCAPE)    spi_write_cd(ST7735_MADCTL, 1, ST7735_Landscape180);
    if(Option->DISPLAY_ORIENTATION==RPORTRAIT)     spi_write_cd(ST7735_MADCTL, 1, ST7735_Portrait180);

    if(Option->DISPLAY_ORIENTATION&1){
        VRes=HorizontalRes;
        HRes=VerticalRes;
    } else {
        HRes=HorizontalRes;
        VRes=VerticalRes;
    }

    //Set the DrawRectangle vector to point to our function
    DrawRectangleVector= (unsigned int)&DrawRectangleSPI + libAddr;

    //Set the DrawBitmap vector to point to our function
    DrawBitmapVector=(unsigned int)&DrawBitmapSPI + libAddr;

#ifdef MX470
    DrawBufferVector=(unsigned int)&DrawBufferSPI + libAddr;
#endif
    //CLS
    DrawRectangle(0,0,HRes-1,VRes-1,0x000000);
    SPIBRG=brgsave; //restore user (or my) setup
    SPICON=consave;
    SPICON2=con2save;
    static const char startup[]="ST7789 driver loaded\r\n";
    pstring(startup);
}
