' WaterSort Game
' Rev 1.0.0 William M Leue 14-Feb-2022

option default integer
option base 1

' constants
const NUM_SEGMENTS = 4
const MAX_COLORS   = 14
const MAX_TUBES    = 17
const MAX_MOVES    = 200
const STSIZE       = MAX_MOVES
const NLEVELS      = 4
const MAX_TUBES_PER_ROW = 7

const TUBE_HEIGHT = 180
const TUBE_WIDTH  = 40
const TUBE_SEP    = 20
const WMARGIN     = 20
const STYOFF      = 20

const WEMPTY   = 0

const MCHAN    = 2

const UP    = 128
const DOWN  = 129
const LEFT  = 130
const RIGHT = 131
const ENTER = 10
const ESC   = 27

const BUTTON_X   = 698
const BUTTON_Y   = 200
const BUTTON_W   = 80
const BUTTON_H   = 30
const BUTTON_SEP = 50
const NBUTTONS   = 7

const BUTTON_LOAD    = 1
const BUTTON_RESTART = 2
const BUTTON_UNDO    = 3
const BUTTON_HINT    = 4
const BUTTON_MASK    = 5
const BUTTON_HELP    = 6
const BUTTON_QUIT    = 7

const MAX_PPC = 34
const MAX_PUZZLES = MAX_PPC*3

const SCORE_X  = 655
const SCORE_Y  =  80
const SCORE_W  = 120
const SCORE_H  = 100

' globals
dim ntubes = 0
dim ncolors = 0
dim colors(MAX_COLORS)
dim tubes(NUM_SEGMENTS, MAX_TUBES)
dim start_tubes(NUM_SEGMENTS, MAX_TUBES)
dim nfilled(MAX_TUBES)
dim tubelocs(2, MAX_TUBES)

dim state = 0
dim source = 0
dim destination = 0
dim running = 0
dim level = 1
dim masked = 0

dim nsolsteps = 0
dim hintstep = 0
dim solution(3, MAX_MOVES)
dim nsolstepsm = 0
dim solutionm(3, MAX_MOVES)
dim slog(3, MAX_MOVES)

dim button_labels$(NBUTTONS)
dim button_states(NBUTTONS)

dim stack(3, STSIZE)
dim sp = 0
dim lp = 0

dim chosen_puz = 0
dim list_col = 1
dim list_row = 1
dim list_npuz = 0
dim listing = 0
dim list_pnums(MAX_PUZZLES)
dim list_col_np(3)

' main program
'open "debug.txt" for output as #1
mode 1,8
font 1
ReadColors
ReadButtonData
DrawPuzzleList
InitMouse
HandleKeyboardEvents
end

' Read the water colors
sub ReadColors()
  local i
  for i = 1 to MAX_COLORS
    read colors(i)
  next i
end sub

' Read the button data
sub ReadButtonData
  local i
  for i = 1 to NBUTTONS
    read button_labels$(i)
  next i
  for i = 1 to NBUTTONS
    read button_states(i)
  next i
end sub

' Initialize mouse and cursor
sub InitMouse
'  on error skip 1
'  controller mouse open MCHAN, LClick
'  if mm.errno <> 0 then
'    cls
'    print "Cannot open mouse on channel ";MCHAN
'    end
'  end if
'  gui cursor on 1
'  gui cursor show
  mouse LClick
  settick 20, UpdateCursor
end sub

' Make cursor track the mouse
sub UpdateCursor
'  gui cursor mouse(X), mouse(Y)
end sub

' Handle all user keyboard inputs
' These are mainly for cursor handling in the
' Puzzle List page.
sub HandleKeyboardEvents
  local z$
  local cmd, pindex
  z$ = INKEY$
  do
    do
      z$ = INKEY$
    loop until z$ <> ""
    cmd = asc(UCASE$(z$))
    select case cmd
      case UP
        if not listing then continue do
        if list_row > 1 then list_row = list_row-1
        SetListCursor
      case DOWN
        if not listing then continue do
        if list_row < list_col_np(list_col) then list_row = list_row+1
        SetListCursor
      case LEFT
        if not listing then continue do
        if list_col > 1 then list_col = list_col-1
        SetListCursor
      case RIGHT
        if not listing then continue do
        if list_col < 3 then
          if list_col_np(list_col+1) > 0 then
            list_col = list_col+1
          end if
        end if
        SetListCursor
      case ENTER
        if not listing then continue do
        pindex = list_row + (list_col-1)*MAX_PPC
        chosen_puz = list_pnums(pindex)
        listing = 0
        cls
        StartChosenPuzzle
      case ESC
        end
    end select
  loop
end sub

' Manipulate the cursor on the puzzle list
sub SetListCursor
  local x, y, cy
  static py = 7
  static px = 80
  x = 7 + (list_col-1)*mm.hres\3
  y = 80 + (list_row-1)*15
  if x > 0 and y > 0 then
'    gui cursor x, y
  end if
  box px+4, py-6, mm.hres\3-18, 15,, rgb(black)
  box x+4, y-6, mm.hres\3-18, 15,, rgb(yellow)
  px = x
  py = y
end sub

' Quit the program
sub Quit
  settick 0, 0
  gui cursor hide
  controller mouse close
  cls
  mode 1,8
  end
end sub

' After the puzzle has been chosen, start it.
sub StartChosenPuzzle
  ClearGame
  ReadPuzzle "./Puzzles/p" + str$(chosen_puz) + ".wpz"
  ReadSolution "./Puzzles/p" + str$(chosen_puz) + ".sol"
  ReadSolution "./Puzzles/p" + str$(chosen_puz) + ".solm"
  InitGame
  DrawPuzzle
  DrawButtons
  running = 1
end sub

' Handle Virtual button presses
sub HandleButtonPresses which
  select case which
    case BUTTON_LOAD
      masked = 0
      DrawPuzzleList
    case BUTTON_RESTART
      cls
      source = 0
      destination = 0
      InitGame
      DrawPuzzle
      button_states(BUTTON_RESTART) = 0
      button_states(BUTTON_UNDO) = 0
      if nsolsteps > 0 then
        button_states(BUTTON_HINT) = 1
        hintstep = 0
      end if
      DrawButtons
      running = 1
      lp = 0
    case BUTTON_UNDO
      if sp > 0 then
        button_states(BUTTON_HINT) = 0
        DrawButtons
        UndoMove
      end if
    case BUTTON_HINT
      if nsolsteps > 0 then
        if hintstep = 0 then
          InitGame
          DrawPuzzle
          running = 1
        end if
        ShowHints
        if hintstep = nsolsteps then
          button_states(BUTTON_HINT) = 0
        end if
        DrawButtons
      end if
    case BUTTON_MASK
      masked = 1-masked
      if masked then 
        if nsolstepsm = 0 then
          button_states(BUTTON_HINT) = 0
        end if
      else
        if nsolsteps > 0 then
          button_states(BUTTON_HINT) = 1
        end if
      end if      
      DrawButtons
      InitGame
      DrawPuzzle
    case BUTTON_HELP
      running = 0
      DrawIntroPage
      InitGame
      DrawPuzzle  
    case BUTTON_QUIT
      settick 0, 0
'      gui cursor hide
'      controller mouse close
      mouse close
      cls
      mode 1,8
      end
  end select
end sub

' Completely clear out a game
sub ClearGame
  local t, w
  InitGame
  for t = 1 to ntubes - nempties
    for w = NUM_SEGMENTS to 1 step -1
      start_tubes(w, t) = 0
    next w
    nfilled(t) = 0
  next t
  button_states(BUTTON_RESTART) = 0
  button_states(BUTTON_UNDO) = 0
  button_states(BUTTON_HINT) = 0
end sub

' Init Game by copying starting tube data
' to game tubes
sub InitGame
  local t, w
  for t = 1 to ntubes
    for w = NUM_SEGMENTS to 1 step -1
      tubes(w, t) = 0
    next w
    nfilled(t) = 0
  next t
  for t = 1 to ntubes - nempties
    for w = NUM_SEGMENTS to 1 step -1
      if masked and w < NUM_SEGMENTS then
        tubes(w, t) = -start_tubes(w, t)
      else
        tubes(w, t) = start_tubes(w, t)
      end if
      if start_tubes(w, t) > 0 then inc nfilled(t)
    next w
  next t
  sp = 0
  moves = 0
  running = 1
end sub

' Handle left mouse clicks
sub LClick
  local i, x1, x2, y2, mx, my, nstate
  local m$
  mx = mouse(X) : my = mouse(Y)
  if listing then
    SelectPuzzle mx, my
  end if
  button_index = GetButtonIndex(mx, my)
  if button_index > 0 then
    HandleButtonPresses button_index
    exit sub
  end if
  if not running then exit sub
  for i = 1 to ntubes
    x1 = tubelocs(1, i)
    y1 = tubelocs(2, i)
    x2 = x1 + TUBE_WIDTH
    y2 = y1 + TUBE_HEIGHT
    if mx >= x1 and mx <= x2 and my >= y1 and my <= y2 then
      select case state
        case 0
          nstate = PickSourceTube(i)
        case 1
          nstate = PickDestinationTube(i)
      end select
      state = nstate
      select case state
        case 0 ' nothing
          source = 0
          destination = 0
          DrawPuzzle
        case 1
          DrawPuzzle
        case 2
          button_states(BUTTON_HINT) = 0
          PourWater source, destination
      end select
      exit for
    end if
  next i
end sub

' Select a listed puzzle using a mouse click.
' (The arrow keys work too.)
sub SelectPuzzle mx, my
  local flag = 0
  SetListCursor
  list_row = (my-61)\15
  list_col = (mx-20)\(mm.hres\3) + 1
  SetListCursor
end sub
    
' Find out if the mouse click is in an enabled button
function GetButtonIndex(mx, my)
  local i, x, y
  x = BUTTON_X
  for i = 1 to NBUTTONS
    if button_states(i) = 1 then
      y = BUTTON_Y + (i-1)*BUTTON_SEP
      if mx >= x and mx <= x+BUTTON_W then
        if my >= y and my <= y+BUTTON_H then
          GetButtonIndex = i
          exit function
        end if
      end if
    end if
  next i
  GetButtonIndex = 0
end function

' Return the 1 if the source tube is legal, or
' zero if there is a problem.
' 'source' global is set to the source tube index
' as a side effect.
' Empty tubes and full tubes with a single color
' are not qualified to be sources.
function PickSourceTube(which)
  local nw = nfilled(which)
  if nw = 0 then
    source = 0
    PickSourceTube = 0
    exit function
  end if
  if NumSameWater(which) = NUM_SEGMENTS then
    source = 0
    PickSourceTube = 0
    exit function
  else
    source = which
    PickSourceTube = 1
  end if
end function

' Returns the number of water segments starting from
' the topmost in a tube that are the same color
function NumSameWater(which)
  local wt, i, nw, nsw
  nw = nfilled(which)
  if nw = 0 then
    NumSameWater = 0
    exit function
  end if
  wt = tubes(nw, which)
  nsw = 0
  if wt > 0 then
    for i = nw to 1 step -1
      if tubes(i, which) <> wt then
        exit for
      else
        inc nsw
      end if
    next i
  end if
  NumSameWater = nsw
end function
      
' Pick the destination tube
' returns state 2 (ready to pour),
' or zero if the pour cannot be done.
' 'destination' global is set to the destination
' tube index as a side effect.
' The source and destination cannot be the same.
' Tubes that are full cannot be destinations.
' If the destination is not empty, its top water
' color must match the top water color of the source.
function PickDestinationTube(which)
  local swater, dwater, nww, nsw
  if which = source then
    PickDestinationTube = 0
    source = 0
    destination = 0
    exit function
  end if
  nww = nfilled(which)
  if nww = NUM_SEGMENTS then
    destination = 0
    PickDestinationTube = 0
    exit function
  else
    nsw = nfilled(source)
    swater = tubes(nsw, source)
    if nww = 0 then
      dwater = 0
    else
      dwater = tubes(nww, which)
    end if
    if (dwater > 0) and dwater <> swater then
      destination = 0
      PickDestinationTube = 0
      exit function
    else
      destination = which
      PickDestinationTube = 2
    end if
  end if
end function

' Pour water from source to destination tube
sub PourWater ps, pd
  local nsw, ndw, i, wcnt, swt
  local w = TUBE_WIDTH-1
  nsw = nfilled(ps)
  ndw = nfilled(pd)
  swt = tubes(nsw, ps)
  wcnt = 0
  for i = nsw to 1 step -1
    if tubes(i, ps) = swt then
      inc wcnt
    else
      exit for
    end if
  next i
  wcnt = min(wcnt, NUM_SEGMENTS-ndw)
  for i = 1 to wcnt
    ndw = ndw+1
    tubes(ndw, pd) = tubes(nsw, ps)
    tubes(nsw, ps) = 0
    nsw = nsw-1
  next i
  if nsw > 0 then
    if tubes(nsw, ps) < 0 then 
      tubes(nsw, ps) = -tubes(nsw, ps)
    end if
  end if
  inc sp
  stack(1, sp) = ps
  stack(2, sp) = pd
  stack(3, sp) = wcnt
  nfilled(ps) = nsw
  nfilled(pd) = ndw
  inc lp
  slog(1, lp) = ps
  slog(2, lp) = pd
  slog(3, lp) = wcnt
  inc moves
  source = 0
  state = 0
  button_states(BUTTON_UNDO) = 1
  button_states(BUTTON_RESTART) = 1
  if IsSolved() then
    running = 0
    button_states(BUTTON_UNDO) = 0
    button_states(BUTTON_RESTART) = 0
    saveSolution
    sp = 0
  end if
  DrawPuzzle
  DrawButtons
end sub

' Undo a move
sub UndoMove
  local ps, pd, nsw, ndw, i, wcnt, swt
  ps = stack(1, sp)
  pd = stack(2, sp)
  wcnt = stack(3, sp)
  sp = sp-1
  if sp = 0 then button_states(BUTTON_UNDO) = 0
  nsw = nfilled(pd)
  ndw = nfilled(ps)
  swt = tubes(nsw, pd)
  for i = 1 to wcnt
    ndw = ndw+1
    tubes(ndw, ps) = tubes(nsw, pd)
    tubes(nsw, pd) = 0
    nsw = nsw-1
  next i
  nfilled(pd) = nsw
  nfilled(ps) = ndw
  if masked then  
    if nfilled(ps) > 1 then
      for i = nfilled(ps)-1 to 1 step -1
        if tubes(i, ps) > 0 then tubes(i, ps) = -tubes(i, ps)
      next i
    end if
  end if
  source = 0
  state = 0
  if lp > 0 then lp = lp-1
  DrawPuzzle
end sub

' Check for being stuck
function HaveNoMoves()
  local src, dst
  for src = 1 to ntubes
    for dst = 1 to ntubes
      if dst <> src then
        if IsLegalMove(src, dst) then
          HaveNoMoves = 0
          exit function
        end if
      end if
    next dst
  next src
  HaveNoMoves = 1
end function

' Check for a legal move. Returns 1 if legal, 0 if not.
' Things that make a move illegal:
'  1. Source tube empty
'  2. Source tube full with all the same water.
'  3. Number of segments with same water at top of source tube
'      greater than number of spaces in destination tube.
'  4. Destination tube full.
'  5. Source and destination waters don't match.
function IsLegalMove(s, d)
  local ns, nd, nw, nsw, swt, dwt, nde
  ns = nfilled(s)
  nd = nfilled(d)
  if ns = 0 then
    IsLegalMove = 0
    exit function
  end if
  if nd = NUM_SEGMENTS then
    IsLegalMove = 0
    exit function
  end if
  nsw = NumSameWater(s)
  if nsw = NUM_SEGMENTS then
    IsLegalMove = 0
    exit function
  end if
  nde = NUM_SEGMENTS - nd
  if nsw > nde then
    IsLegalMove = 0
    exit function
  end if
  if nd > 0 then
    swt = tubes(ns, s)
    dwt = tubes(nd, d)
    if dwt <> swt then
      IsLegalMove = 0
      exit function
    end if
  end if
  IsLegalMove = 1
end function

' Check for a won game
function IsSolved()
  local i, wt, w, solved, nw
  solved = 1
  for i = 1 to ntubes
    nw = nfilled(i)
    if (nw <> NUM_SEGMENTS) and (nw <> 0) then
      solved = 0
      exit for
    end if
    if nw = NUM_SEGMENTS then
      wt = tubes(NUM_SEGMENTS, i)
      for w = NUM_SEGMENTS to 1 step -1
        if tubes(w, i) <> wt then
          solved = 0
          exit for
        end if
      next w
    end if
  next i
  IsSolved = solved
end function

' Save the solution back to the puzzle file if one does
' not already exist. Will not overwrite an existing file!
sub SaveSolution
  local i, e
  local path$
  if masked then
    path$ = "./Puzzles/p" + str$(chosen_puz) + ".solm"
  else
    path$ = "./Puzzles/p" + str$(chosen_puz) + ".sol"
  end if
  on error skip 1
  open path$ for input as #3
  if mm.errno = 0 then
    close #3
    exit sub
  end if
  on error skip 1
  open path$ for output as #3
  e = mm.errno
  if e <> 0 then
    'print #1, "Error ";e;" opening solution file '";path$;"' for output"
    exit sub
  end if
  print #3, lp
  for i = 1 to lp
    print #3, slog(1, i);",";slog(2,i);",";slog(3,i)
  next i
  close #3
end sub

' Show hints for solution
sub ShowHints
  if masked then
    if hintstep >= nsolstepsm then exit sub
    inc hintstep
    PourWater solutionm(1, hintstep), solutionm(2, hintstep)
  else
    if hintstep >= nsolsteps then exit sub
    inc hintstep
    PourWater solution(1, hintstep), solution(2, hintstep)
  end if
end sub

' Render the Puzzle
sub DrawPuzzle
  DrawTitle
  DrawTubes
end sub

' Draw a Title
sub DrawTitle
  local m$
  text mm.hres\2, 10, "Water Sort Puzzle", "CT", 5
  if IsSolved() then
    text mm.hres\2, 40, "SOLVED!", "CT", 5,, rgb(green)
    button_states(BUTTON_UNDO) = 0
  end if
  if chosen_puz > 0 then
    box SCORE_X, SCORE_Y, SCORE_W, SCORE_H
    m$ = "Puzzle " + str$(chosen_puz)
    text SCORE_X+SCORE_W\2, SCORE_Y+12, m$, "CT", 4,, rgb(green)
    m$ = "Colors: " + str$(ncolors)
    text SCORE_X+SCORE_W\2, SCORE_Y+32, m$, "CT", 4,, rgb(blue)
    m$ = "Tubes: " + str$(ntubes)
    text SCORE_X+SCORE_W\2, SCORE_Y+52, m$, "CT", 4,, rgb(blue)
    m$ = "Moves: " + str$(moves)
    text SCORE_X+SCORE_W\2, SCORE_Y+72, m$, "CT", 4,, rgb(yellow)
  end if
end sub

' Draw the tubes for the current state
sub DrawTubes
  local i, x, t, sy, nrows, row, rnt, rnt2
  local nt(2), xs, y(2)
  nrows = 1
  if ntubes > MAX_TUBES_PER_ROW then
    nrows = 2
  end if
  sy = 0
  if nrows = 1 then
    nt(1) = ntubes
    nt(2) = 0
    y(1) = 130
    y(2) = 0
  else
    nt(1) = ntubes\2+1
    nt(2) = ntubes - nt(1)
    y(1) = 130
    y(2) = 340
  end if
  rnt = nt(1) : rnt2 = rnt\2
  xs = mm.hres\2 - 60 - rnt2*TUBE_WIDTH - (rnt2-1)*TUBE_SEP - (rnt mod 2)*TUBE_WIDTH
  box xs-5, y(1)-30, rnt*(TUBE_WIDTH+TUBE_SEP), TUBE_HEIGHT+30,, rgb(black), rgb(black)
  if nrows = 2 then
    box xs-5, y(2)-30, rnt*(TUBE_WIDTH+TUBE_SEP), TUBE_HEIGHT+30,, rgb(black), rgb(black)
  end if
  t = 0
  for row = 1 to nrows
    rnt = nt(row)
    rnt2 = rnt\2
    xs = mm.hres\2 - 60 - rnt2*TUBE_WIDTH - (rnt2-1)*TUBE_SEP - (rnt mod 2)*TUBE_WIDTH
    x = xs
    for i = 1 to nt(row)
      inc t
      if t = source then
        sy = -30
      else
        sy = 0
      end if
      DrawTube t, x, y(row)+sy
      x = x + TUBE_WIDTH + TUBE_SEP
    next i
  next row
end sub

' Draw a single tube
sub DrawTube index, x, y
  local i, wy, cx, nw
  local tw2 = TUBE_WIDTH\2
  local y2 = TUBE_HEIGHT-20
  local tbx = x+tw2
  local tby = y+y2
  local wh = (y2-WMARGIN\2)\NUM_SEGMENTS
  tubelocs(1, index) = x
  tubelocs(2, index) = y
  line x, y, x, y+y2
  line x+TUBE_WIDTH, y, x+TUBE_WIDTH, y+y2
  line x, y, x+TUBE_WIDTH, y
  arc tbx-1, tby, tw2-1,,90, 270
  wy = y+14
  nw = nfilled(index)
  for i = NUM_SEGMENTS to 1 step -1
    cx = tubes(i, index)
    if cx > 0 then
      box x+1, wy, w+TUBE_WIDTH-1, wh,, colors(cx), colors(cx)
    else if cx < 0 then
      text x+TUBE_WIDTH\2, wy+wh\2, "?", "CM"
    else if cx = 0 then
      box x+1, wy, w+TUBE_WIDTH-1, wh,, rgb(black), rgb(black)
    end if
    wy = wy + wh
  next i
  if nw > 0 and cx > 0 then
    cx = tubes(1, index)
    arc tbx-1, tby, 0, w+TUBE_WIDTH\2-2, 90, 270, colors(cx)
  end if
  if NumSameWater(index) = NUM_SEGMENTS then
    DrawStar index, x, y
  end if
end sub

' Draw a star over a completely filled tube
sub DrawStar index, x, y
  local xv(10), yv(10)
  local cx, cy
  cx = x + TUBE_WIDTH\2
  cy = y - STYOFF
  xv(1) = cx        : yv(1) = cy
  xv(2) = cx+2      : yv(2) = cy+3
  xv(3) = cx+5      : yv(3) = yv(2)
  xv(4) = cx+3      : yv(4) = yv(2)+2
  xv(5) = cx+4      : yv(5) = yv(4)+3
  xv(6) = cx        : yv(6) = yv(2)+3
  xv(7) = cx-4      : yv(7) = yv(5)
  xv(8) = cx-3      : yv(8) = yv(4)
  xv(9) = cx-5      : yv(9) = yv(2)
  xv(10) = xv(9)+3  : yv(10) = yv(2)
  polygon 10, xv(), yv(), rgb(yellow), rgb(yellow)
end sub

' Draw the command buttons
sub DrawButtons
  local i, x, y, c
  x = BUTTON_X
  y = BUTTON_Y
  for i = 1 to nbuttons
    c = rgb(white)
    if button_states(i) = 0 then c = rgb(gray)
    rbox x, y, BUTTON_W, BUTTON_H, 10, c
    text x+BUTTON_W\2, y+BUTTON_H\2, button_labels$(i), "CM",,, c
    inc y, BUTTON_SEP
  next i
end sub

' Read a puzzle file
sub ReadPuzzle path$
  local buf$, f1$, f2$
  local tube, seg, sstep, i, wl
  on error skip 1
  open path$ for input as #2
  if mm.errno <> 0 then
    cls
    mode 1,8
    print "Error reading puzzle file '";path$;"': errno: ";mm.errno
    end
  end if
  nsolsteps = 0
  line input #2, buf$
  inc puz_offset, len(buf$)
  cat buf$, ","
  f1$ = field$(buf$, 1, ",")
  f2$ = field$(buf$, 2, ",")
  f3$ = field$(buf$, 3, ",")
  f4$ = field$(buf$, 4, ",")
  ncolors = val(f1$)
  nempties = val(f2$)
  ntubes = ncolors + nempties
  level = val(f3$)
  masked = val(f4$)
  for tube = 1 to ntubes
    line input #2, buf$
    inc puz_offset, len(buf$)
    cat buf$, ","
    for seg = 1 to NUM_SEGMENTS
      wl = NUM_SEGMENTS-seg+1
      start_tubes(wl, tube) = val(field$(buf$, seg, ","))
      if start_tubes(wl, tube) > 0 then
        inc nfilled(tube)
      end if
      if masked then
        if wl < NUM_SEGMENTS then
          start_tubes(wl, tube) = -start_tubes(wl, tube)
        end if
      end if
    next seg
  next tube
  close #2
  for tube = ncolors+1 to ncolors+nempties-1
    for seg = 1 to NUM_SEGMENTS
      start_tubes(seg, tube) = 0
    next seg
    nfilled(tube) = 0
  next tube
  button_states(BUTTON_RESTART) = 0
  button_states(BUTTON_MASK) = 1
  lp = 0
end sub

' Read a solution file
sub ReadSolution path$
  local buf$, f1$, f2$
  local tube, seg, sstep, i, wl, mflag
  mflag = 0
  if RIGHT$(path$, 1) = "m" then mflag = 1
  on error skip 1
  open path$ for input as #3
  if mm.errno <> 0 then
    exit sub
  end if
  line input #3, buf$
  if mflag then
    nsolstepsm = val(buf$)
    if nsolstepsm > 0 then
      for sstep = 1 to nsolstepsm
        line input #3, buf$
        cat buf$, ","
        for i = 1 to 3
          solutionm(i, sstep) = val(field$(buf$, i, ","))
        next i
      next sstep
    end if
  else
    nsolsteps = val(buf$)
    if nsolsteps > 0 then
      for sstep = 1 to nsolsteps
        line input #3, buf$
        cat buf$, ","
        for i = 1 to 3
          solution(i, sstep) = val(field$(buf$, i, ","))
        next i
      next sstep
    end if
  end if
  button_states(BUTTON_HINT) = 1
  hintstep = 0

end sub

' (debug)
sub PrintTube which
  local seg, ns
  ns = nfilled(which)
  for seg = 1 to ns
    print #1, tubes(seg, which);" ";
  next seg
  print #1, ""
end sub

' Draw the Intro Page
sub DrawIntroPage
  cls
  text mm.hres\2, 20, "Water Sort Puzzle", "CT", 4, 1, rgb(green)
  text 0, 50, ""
  print "Welcome to the Water Sort Puzzle. You will see a set of test tubes that are each"
  print "filled with several different kinds of colored water. Your job is to sort the water"
  print "so that each tube is either empty, or contains only a single color."
  print ""
  print "To move water from one tube to another, use the mouse to click on the tube you want"
  print "to move water FROM, and then the tube you want to move water TO. If the receiving tube"
  print "is empty, then the pour will work. But if the receiving tube has some water in it,"
  print "the color of its top water must match the top water color of the 'FROM' tube."
  print "If you regret a move, use the UNDO button at right to take it back. You can undo any"
  print "number of moves. You can also press the RESTART button to start anew at the beginning."
  print ""
  print "When the program starts, you will be presented with a list of all available"
  print "puzzles. Use the mouse or arrow keys to navigate to the puzzle you want to"
  print "work on and press the ENTER key to start. The puzzles are presented in no"
  print "particular order and higher-numbered puzzles are not necessarily more difficult."
  print "Difficulty is determined by the number of colors, the number of tubes, the number"
  print "of empty tubes, and details of how the water layers are arranged."
  print ""
  print "When you first start using the program, only puzzle #1 has a stored solution. You can"
  print "press the HINT button repeatedly to watch the stored solution be replayed. If you want,"
  print "you can press HINT only a few times to get started and then continue with your own moves"
  print "using the mouse. As you solve puzzles, your solutions will be stored and available for"
  print "replay any time later.
  print ""
  print "Press the LOAD button at any time to return to the puzzle listing screen where you can"
  print "choose another puzzle. All the puzzles bundled with the program are solvable, but some"
  print "are harder than others. For a real challenge, press the MASK button to hide all but the"
  print "top layers of water in each tube. When you solve a masked puzzle, the solution will be"
  print "stored in a different file than the unmasked solution.
  print ""
  print "The key to solving a difficult puzzle is to try to plan moves so that after each set of"
  print "a few moves, you have one tube empty. This will give you the flexibility to continue on"
  print "to victory!
  print ""
  print "Press any Key to Continue
  z$ = INKEY$
  do
    z$ = INKEY$
  loop until z$ <> ""
  cls
  DrawButtons
end sub

' Draw the list of available puzzles in './Puzzles'
sub DrawPuzzleList
  local x, y, nl, px
  local buf$, buf2$, snum$, m$, nc$, ne$
  local pindex(MAX_PUZZLES)
  local pnc$(MAX_PUZZLES), pne$(MAX_PUZZLES), pslv$(MAX_PUZZLES)
  local hdr$ = "Pnum  Ncolors  Nempties  Solved"
  cls
  box 0, 0, mm.hres-1, mm.vres-1
  text mm.hres\2, 8, "Puzzles", "CT", 5, 1, rgb(green)
  text 30, 20, "Use arrow Keys or Mouse to navigate",,7
  text 600, 20, "Press ENTER to choose",, 7
  x = 0
  y = 40
  line x, y, mm.hres-1, y
  text x+10, y+9, hdr$
  line mm.hres\3, y, mm.hres\3, mm.vres-1
  text mm.hres\3+10, y+9, hdr$
  line 2*mm.hres\3, y, 2*mm.hres\3, mm.vres-1
  text 2*mm.hres\3+10, y+9, hdr$
  y = 50
  y = 70
  line x, y, mm.hres-1, y
  chdir "Puzzles"
  buf$ = dir$("*.wpz", FILE)
  list_npuz = 0 
  do while buf$ <> ""
    inc list_npuz
    nl = len(buf$)
    snum$ = mid$(buf$, 2, nl-5)
    list_pnums(list_npuz) = val(snum$)  
    open buf$ for input as #3
    line input #3, buf2$
    close #3
    cat buf2$, ","
    pnc$(list_npuz) = field$(buf2$, 1, ",")
    pne$(list_npuz) = field$(buf2$, 2, ",")
    buf$ = "p" + snum$ + ".sol"
    on error skip 1
    open buf$ for input as #3
    if mm.errno = 0 then 
      pslv$(list_npuz) = "Y"
      close #3
    else  
      pslv$(list_npuz) = "N"
    end if
    if npuz >= MAX_PUZZLES then
      exit do
    end if 
    buf$ = dir$()
  loop
  sort list_pnums(), pindex(),,, list_npuz
  list_row = 0
  list_col = 1
  x = 20 + (list_col-1)*mm.hres\3
  y = 61 + list_row*15
  for i = 1 to 3
    list_col_np(i) = 0
  next i
  for i = 1 to list_npuz
    px = pindex(i)
    pn$  = str$(list_pnums(i))
    m$ = pn$ + space$(7-len(pn$)) + pnc$(px)
      cat m$, space$(9-len(pnc$(px))) + pne$(px)
      cat m$, space$(10-len(pne$(px))) + pslv$(px)
    inc list_row
    if list_row > MAX_PPC then
      inc list_col
      list_row = 1
    end  if
    x = 20 + (list_col-1)*mm.hres\3
    y = 61 + list_row*15
    inc list_col_np(list_col)
    text x, y, m$
    inc y, 15
  next i
  chdir "../"
  list_row = 1
  list_col = 1
  SetListCursor
  listing = 1
end sub

' Create uniformly distributed random integer in the specified closed interval.
function RandomIntegerInRange(a as integer, b as integer) as integer
  local v, c
  c = b - a + 1
  do
    v = a + (b-a+2)*rnd()
    if v > 1 and v-1 <= c then exit do
  loop
  RandomIntegerInRange = v-1
end function
    
' 14 Water Colors
data rgb(204,   0,   0)' Red
data rgb(204, 118,   0) ' Orange
data rgb(194, 204,   0) ' Yellow
data rgb( 93, 204,   0) ' Lime
data rgb(  0, 177, 179) ' Cyan
data rgb(  0,  70, 179) ' Blue
data rgb( 94,   0, 179) ' Violet
data rgb(179,   0, 164) ' Magenta
data rgb(250, 136, 135) ' Salmon
data rgb(151, 240, 153) ' Lt Green
data rgb(122, 166, 204) ' Lt Blue
data rgb(100,  67,  26) ' Brown
data rgb(179, 179, 179) ' Lt Gray
data rgb( 77,  77,  77) ' Gray

' Button labels
data "Load", "Restart", "Undo", "Hint", "Mask", "Help", "Quit"

' Button states
data 1, 0, 0, 0, 0, 1, 1

