'*************************************************************
' Caravan System Monitor for 12V Battery System              '
' Written by Doug Pankhurst May 2017, based on coding        '
' ideas from Tom Pankhurst TZAdvantage, CaptainBoing and     '
' various other Back Shed Forum members - thanks.            '                                    '
' Written for MMBasic V5.4.08  , copyright Geoff Graham 2012 '
' Program runs on PIC32MX470 with an SSD1930 7" touch screen '
' This program is free for anyone to use or modify as they   '
'  choose however, if you wish to re-distribute, please      '
'  acknowledge authors and contributors above.               '
''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
' File Version csmbe417-gui.bas  19 April 2018
' - this version moves all display function to GUI
' last known good version text 4.13  (live in caravan)
' For a detailed explanation of the program, it's
' construction and rationale, please read the
' CSM4READ.TXT file.
'*************************************************************

' Initial Entry to Program
EntryPoint:
  SetTick 0,0        ' Disable timer interrupt for now
  Option explicit
  Option autorun on  

  ' Define constants for gui controls
  const BGround_Box = 20
  Const Head_Box  = 51
  Const Time_Box  = 52
  const Hint_1 = 53
  const Back_Arrow = 54  ' back arrow box
  const Help_Button = 55 ' question mark box
  const DB_Hdr_1 = 30    ' 4 small box display boxes
  const DB_Hdr_2 = 31
  const DB_Hdr_3 = 32
  const DB_Hdr_4 = 33
  const DB_Val_5 = 34
  const DB_Val_6 = 35
  const DB_Val_7 = 36
  const DB_Val_8 = 37
  const Menu_1 = 38  ' 4 lower menu selection boxes
  const Menu_2= 39   ' - note these have touch areas overlayed
  const Menu_3= 40
  const Menu_4= 41
  const Help_Box= 22  ' used for help messages
  const LDB_Hdr_1= 43 ' 2 large display boxes
  const LDB_Hdr_2= 44
  const LDB_Val_1= 45
  const LDB_Val_2= 46
  ' touch areas for display boxes
  const Back_Arrow_Touch = 1
  const Help_Button_Touch = 2
  const Hint_Touch = 4
  const Led_1 = 9
  const Menu_1_Touch = 5
  const Menu_2_Touch= 6
  const Menu_3_Touch= 7
  const Menu_4_Touch= 8
  const Detail_1_Touch = 10
  const Detail_2_Touch = 11
  const Detail_3_Touch = 12
  const Detail_4_Touch = 13
  
'Define variables
  ' Global Variables - initialised to default values
  Dim F_Ver = 4.17    ' File Version csmbe417-gui.bas  19 April 2018
  ' Voltage and current of caravan battery
  Dim BVolts = 12.8  '  Battery Voltage
  Dim BVArray(29)    ' Battary Volts 30 element array - 1 each 2 secs
  Dim AverageBV=12.8 ' Average battery voltage over last 60 seconds
  Dim PreviousABV=12.8 '
  Dim OldABV = 13    ' so we can calculate ABV correctly
  Dim DeltaBV = 0.1  ' Battery voltage variation over 60 seconds. The
  ' difference between the first and the last used to
  ' calculate the "Time to 30% discharge point"
  Dim Time2Discharge = 20 ' Time to 30% discharge in minutes
  Dim BatCharge = 100' Battery state of charge as a percentage where
  ' 12.75V or higher is 100%, down to 12.05V being 35% - danger level
  ' Voltage and current from Solar Panels
  Dim SVolts = 20    ' Solar Array voltage
  Dim SVArray(29)    ' Solar Volts 30 element array - 1 each two secs
  Dim AverageSV = 21 ' Average Solar Panel volts over last minute
  Dim SAmps = 0 ' current from Solar Array into D250
  Dim SAArray(29)    ' solar panel current array - 1 each two secs
  Dim AverageSC = 0  ' average solar panerl current last minute
  ' Voltage and current from towing vehicle
  Dim CVolts = 13.8  ' Vehicle battery volts
  Dim CVArray(29)    ' Car volts 30 element array  - 1 each two secs
  Dim AverageCV = 13 ' Average vehicle volts ove rlast 30 60 secs
  Dim CAmps = 0 ' current from vehicle into D250
  Dim CAArray(29)    ' car current array  - 1 each two secs
  Dim AverageCC = 0  ' average car current last minute
  ' Input (charging), output (load) and battery charge/discharge currents.
  ' Battery charge/discharge is calculated from input minus load currents.
  ' - instantaneous values, minute averaged value currents
  ' - daily accumulating input, output and battery amp hour useage
  ' - last 4 days battery charge/discharge currents and rolling total
  Dim ACAmps = 0 ' Charging current from XPS25000 240VAC charger
  Dim DCAmps = 0 ' Charging current from D250 Solar Charger
  Dim ACAArray(29)   ' XPS25000 240VAC minute current array - 1 each two secs
  Dim DCAArray(29)   ' D250 minute current array  - 1 each 2 secs
  Dim AverageACC = 0 ' average D250 output current last minute
  Dim AverageDCC = 0 ' average X25000 output current last minute
  ' Input current -  a calculated sum of AC Charger and DC charger currents
  Dim IAmps = 18  ' Charging current into battery/load
  Dim AverageIC = 18 ' Average input current
  Dim LAmps = 12 ' Load current - supplied from batteries/charging systems
  Dim LAArray(29)    ' Load Current 60 element array
  Dim AverageLC = 12 ' Average load current
  ' Battery current - a calculated value; if positive, = a charging current to
  ' the battery; if negative, equals a discharge current from the battery.
  Dim BAmps = 6     ' Input charging current minus load current
  Dim AverageBC = 6      ' Average charge/discharge current
  ' daily accumulating values
  Dim InputWH = 0        ' Input KiloWatt Hours
  Dim LoadWH = 0         ' Load KiloWatt Hours
  ' previous days values
  Dim PrevIWH = 12 ' yesterdays Input Current kwhs
  Dim PrevLWH = 10 ' yesterdays Load Current kwhs
  Dim BatWH = 0    ' Bat charge (+ value) or discharge (- value) in Whs
  Dim PrevBWH = 2  ' yesterdays Bat Current KWh
  Dim Day2BWH = 20 ' last 4 days battery KWh usage
  Dim Day3BWH = 30
  Dim Day4BWH = 40
  Dim TotalBWH = 90   ' rolling 4 day kwatt hour total usage
  dim BV3HrArray(179) ' 3 hour average battery volts updated
                      ' every minute to give continuous
                      ' last 3 hours values
  dim BV3HrPointer = 0 ' index into latest average value
  dim BA3HrArray(179) ' 3 hour average battery amps updated
                      ' every minute to give continuous
                      ' last 3 hours values
  dim BA3HrPointer = 0 ' index into latest average value
  dim BP3HrArray(179) ' 3 hour average battery power updated
                      ' every minute to give continuous
                      ' last 3 hours values
  dim BP3HrPointer = 0 ' index into latest average value
  dim BC3HrArray(179) ' 3 hour average battery charge updated
                      ' every minute to give continuous
                      ' last 3 hours values
  dim BC3HrPointer = 0 ' index into latest average value
    
  Dim hrs = 6
  Dim mins = 0
  Dim secs = 0
  Dim DataVals$ length 64 ' front end running values
  Dim DataValid = 1  ' 0 = bad data, anything else is good data
  ' misc variables
  dim Help$          ' string to display in help box
  dim ActivePage = 1 'default to main GUI page
  dim LastPage = 1   '  remember where we were last
  dim ReEnter = 0    ' flag to control re-entry into GUI display
  Dim Ax             'loop counter 
  dim S_Bright = 25  ' initial display brightness
  dim delim$ = ","   ' delimiter for field extract
  dim n_field = 1    ' number of field in front end data stream

  ' some variables to simulate a 6Amp discharge rate
  Dim cf1 = 14      ' AverageBC  - offset of x from zero
  DIM cf2 = 0.15    ' shape of curve
  DIM cf3 = 11.3    ' 0% charge point
  Dim tbv = 13
  Dim twosecled = 1 ' LED flashes on 2 sec interrupt
  
  ' Flags
  Dim NewDataFlag = 0   ' Two second flag - set every
  ' 2 seconds by the interrupt routine when new voltage
  ' and current values are collected.
  Dim MinFlag = 0 ' Minute flag - set once every min at 00 seconds
  ' by sec interrupt process and cleared by min processing code.
  dim EODFlag = 0
  dim Simulate = 1  ' for testing
'''''''''''''''''
' Carry out initialisation routines here - enable interrupt last
StartPoint:
  BackLight S_Bright  ' set initially for 25%, tap heading to increase
  Colour RGB(White),RGB(white)   'fg and bg - needed for page switch
  GUI interrupt TouchDown ' subroutine to handle touch int
  RTC GETTIME        ' date and time from DS3231
  ' Data values come in from front end via COM1
  ' Note: Front end action could be included in this
  '  program if the system is close enough to read
  '  values directly
  Open "COM1:9600,128,GetVals, 1" As #1
  
  ' set up any GUI controls
  InitGUI ' routine to set GUI controls
  ' preset some display boxes
  CtrlVal(Head_Box) = "Caravan System Monitor"
  CtrlVal(Time_Box)  = left$(Time$,5)+"     "+Date$
  ctrlval(Back_Arrow) = "<" ' left arrow
  ctrlval(Help_Button) = "?" ' home
  ' Fill the the minute average arrays
  ' just so initial values make sense
  For Ax=0 To 29
    BVArray(Ax) = 12.8
    SVArray(Ax) = 19
    CVArray(Ax) = 13
    SAArray(Ax) = 10
    CAArray(Ax) = 6
    ACAArray(Ax) = 0
    DCAArray(Ax) = 16
    LAArray(Ax) = 12
  Next Ax
  ' preset some variables
  Help$ = "Caravan System Monitor v4.17 19 April 2018~"
  Help$ = Help$ + "Created by D.Pankhurst~"
  Help$ = Help$ + "Select action from menu boxes or < to restore main page~~"
  Help$ = Help$ + "Repeatedly tap header to increase brightness"
  ctrlval(Help_Box) = Help$
  ctrlval(Hint_1) = ".... press < for large display"
  ' Display all static data
  ActivePage = 1
  ReEnter = 1
  UpdateGUIScreen  ' initial update GUI screen then only if
                   ' ReEnter is set which is only done when
                   ' a touch detected or at 2 sec processing
  
  ' Wait until 00 seconds to start for clean entry.
  Do While Val(Mid$(Time$,7,2)) < 58
  Loop
  
  ' Enable timer tick interrupts to start data collection at 00 seconds
  SetTick 2000,RequestData  ' Call every other second, send ACK to
  ' front end that initiates a data dump into COM2 from the front end.
  '
  ' end of initialisation routines
  ''''''''''''''''''''
  
Main:
  Do    'Main program loop, for every 2 sec interrupt, get the data, do
    '  2 second, minute and end of day processing as appropriate
    Do While NewDataFlag = 0   ' wait until data request sent
      ' from back end to front end which then generates fresh data
      ' and sends this to back end
    Loop
    ' fresh data received so now process
    TwoSecProcess  ' real time data
    If MinFlag = 1 Then
      MinProcess   ' minute average data
    EndIf
    If EODFlag = 1 Then
      EODProcess   ' end of day tidy up
    EndIf
  Loop
  ' end of main process loop
  
Sub TwoSecProcess
  ' entered with the following instantaneous values from front end
  ' battery volts, solar volts, solar amps, vehicle amps
  ' 240VAC charger amps, solar/car charger amps
  ' total input amps (solar/car + 240VAC), load amps
  ' battery amps (display as green charging, red discharging)
  Local Ax2  'array counter
  ' 2 second processing code - every interrupt sets 2 sec flag
  ' Push most recent values into arrays for minute averaging
  ' then set up for minute averaging
  For Ax2 = 29 To 1 Step -1  ' shuffle everyone down one spot
    BVArray(Ax2) = BVArray(Ax2-1)
    SVArray(Ax2) = SVArray(Ax2-1)
    CVArray(Ax2) = CVArray(Ax2-1)
    SAArray(Ax2) = SAArray(Ax2-1)
    CAArray(Ax2) = CAArray(Ax2-1)
    ACAArray(Ax2) = ACAArray(Ax2-1)
    DCAArray(Ax2) = DCAArray(Ax2-1)
    LAArray(Ax2) = LAArray(Ax2-1)
  Next Ax2
  BVArray(0) = BVolts ' then plug in new values to first spot
  SVArray(0) = SVolts
  CVArray(0) = CVolts
  SAArray(0) = SAmps
  CAArray(0) = CAmps
  ACAArray(0) = ACAmps
  DCAArray(0) = DCAmps
  LAArray(0) = LAmps
  ' Battery Current is IAmps-LAmps - if positive
  '  it is a charging current into battery, if negative, it is a
  '  discharging current out of battery
  IAmps = ACAmps + DCAmps  ' from both mains and solar/car chargers
  BAmps = IAmps-LAmps
  
  ' Convert both input, output & battery instantaneous
  '  currents into WHrs
  InputWH = InputWH + ((IAmps/1800)*BVolts) ' WHrs
  LoadWH = LoadWH + ((LAmps/1800)*BVolts)   ' WHrs
  BatWH = InputWH-LoadWH
  ' Note: All AH accumulating totals are plugged into
  ' last day values then reset to zero at 6AM
  ' - now update terminal display with all values
  BatChargeUpdate  ' sub to calculate display battery charge
  UpDateGUIVals  ' display current data
  NewDataFlag = 0  ' clear new data flag ready for next interrupt
  ' End of two second processing code
End Sub
  
  '''''''''''''''''''''''''''''''''''
Sub MinProcess
  Local Ax1m
  ' Check for minute flag, calculate the average of the last 60 seconds of
  ' readings, plug them into the log file array at the log file array pointer
  ' calculate averages for last minute
  PreviousABV = AverageBV   ' save for discharge calculation purposes
  ' clear ready for averaging
  AverageBV = 0
  AverageSV = 0
  AverageCV = 0
  AverageSC = 0
  AverageCC = 0
  AverageACC = 0
  AverageDCC = 0
  AverageLC = 0
  For Ax1m = 0 To 29   'don't forget only sampled every 2 secs
    AverageBV = AverageBV + BVArray(Ax1m)
    AverageSV = AverageSV + SVArray(Ax1m)
    AverageCV = AverageCV + CVArray(Ax1m)
    AverageSC = AverageSC + SAArray(Ax1m)
    AverageCC = AverageCC + CAArray(Ax1m)
    AverageACC = AverageACC + ACAArray(Ax1m)
    AverageDCC = AverageDCC + DCAArray(Ax1m)
    AverageLC = AverageLC + LAArray(Ax1m)
  Next Ax1m
  ' now average
  AverageBV = AverageBV / 30
  AverageSV = AverageSV / 30
  AverageCV = AverageCV / 30
  AverageSC = AverageSC / 30
  AverageCC = AverageCC / 30
  AverageACC = AverageACC / 30
  AverageDCC = AverageDCC / 30
  AverageLC = AverageLC / 30
  AverageIC = AverageACC + AverageDCC  ' sum X250 car/solar
                                       ' plus X25000 AC chargers
  AverageBC = AverageIC - AverageLC ' sum of amps in(+) & out(-)
  ' now update 3 hr array with latest minute average bat volts
  BV3HrArray(BV3HrPointer) = AverageBV
  BV3HrPointer = BV3HrPointer + 1  
  If BV3HrPointer = 180 then
    BV3HrPointer = 0
  endif
  ' now update 3 hr array with latest minute average bat amps
  BA3HrArray(BA3HrPointer) = AverageBC
  BA3HrPointer = BA3HrPointer + 1  
  If BA3HrPointer = 180 then
    BA3HrPointer = 0
  endif
  ' now update 3 hr array with latest minute average bat WHrs
  BP3HrArray(BP3HrPointer) = BatWH
  BP3HrPointer = BP3HrPointer + 1  
  If BP3HrPointer = 180 then
    BP3HrPointer = 0
  endif
  ' now update 3 hr array with latest minute average bat charge
  BC3HrArray(BC3HrPointer) = BatCharge
  BC3HrPointer = BC3HrPointer + 1  
  If BC3HrPointer = 180 then
    BC3HrPointer = 0
  endif
  MinFlag = 0   ' reset the min flag
End Sub  ' End of minute code

  ''''''''''''''''''''''''''''''''''''''
  ' End of Day process - save values for historical records,
  ' tidy up and reset. Note EOD currently 0600 Hrs.
Sub EODProcess  ' Check for end of day
  ' - for new day starting at 6AM
  RTC GETTIME        ' date and time from DS3231
  PrevIWH = InputWH
  InputWH = 0
  PrevLWH = LoadWH
  LoadWH = 0
  Day3BWH = Day2BWH
  Day2BWH = PrevBWH
  PrevBWH = BatWH
  BatWH = 0
  TotalBWH = PrevBWH + Day2BWH + Day3BWH
  EODFlag = 0    ' clear end of day flag until next 6AM
End Sub  ' End of EOD code
  '''''''''''''''''
  
  ''''''''''''''''''''''''''
  '  2 Second Interrupt Routine Handling to send data requestto front end
Sub RequestData      ' Timer Interrupt handler
  ReEnter = 1  ' allow GUI screen update
  ' timer = 0
  ' Send data request to front end
  Print #1,Chr$(6);     ' the ACK character
'  Print "Data request sent at ",time$ 'debug use
  CtrlVal(Time_Box)  = left$(Time$,5)+"     "+Date$
  DataValid = 0  ' reset ready for new data
  'flip twosecled on-off - just an 'I'm alive' indicator
  if twosecled = 1 then
    ctrlval(Led_1) = 1
    twosecled = 0
  else
    ctrlval(Led_1) = 0
    twosecled = 1
  endif
  ' grab time into component variables
  hrs = Val(left$(Time$,2))
  mins = Val(Mid$(Time$,4,2))
  secs = Val(right$(Time$,2))
  ' Get the current time, extract the minutes component
  If secs = 0 Then
    MinFlag = 1
  Else
    MinFlag = 0
  EndIf
  ' Get the current time, test for 06:00
  If hrs = 6 And mins = 0 And secs = 0 Then
    EODFlag = 1
  Else
    EODFlag = 0
  EndIf

End Sub
  
  '   Interrupt to receive data string from front end with all data values
Sub GetVals  ' Receive character interrupt handler
  Local BytesReady
  ' structure of DataVal$ -
  ' battery volts vv.vv, solar volts vv.v, car volts vv.vv,
  ' solar amps ii.i, car amps ii.i,
  ' ac charge amps ii.i, dc charge amps ii.i,
  ' load amps ii.i, data valid zz.z  in the form
  ' vv.vv,vv.v,vv.vv,ii.i,ii.i,ii.i,ii.i,ii.i,zz.z
  ' total 46 bytes
  pause 50    ' wait for data in buffer to stabilise
  BytesReady = Loc(#1)
'  Print BytesReady,    " characters waiting"
  DataVals$ = Input$(BytesReady,#1)
  ' if the back end Tx Rx are looped, plug in data values to simulate
  ' data from the front end
  if asc(left$(DataVals$,1)) = 6 then  ' look for looped ACK
  ' print "ASCII 6 loopback character received" ' debug use
  ' plug in simulated values for standalone testing
    DataVals$ = "12.52,19.5,2.85,11.5,0.2,0.1,10.1,4.8,zz.z"
  else ' extract the 9 fields and plug into variables
  
  endif
  ' Print DataVals$   ' debug use
  ' so now extract the front end data
  BVolts = Val(Field$(DataVals$,1,delim$))
  SVolts = Val(Field$(DataVals$,2,delim$))
  CVolts = Val(Field$(DataVals$,3,delim$))
  SAmps  = Val(Field$(DataVals$,4,delim$))
  CAmps  = Val(Field$(DataVals$,5,delim$))
  ACAmps = Val(Field$(DataVals$,6,delim$))
  DCAmps = Val(Field$(DataVals$,7,delim$))
  LAmps  = Val(Field$(DataVals$,8,delim$))
  If Field$(DataVals$,9,delim$) = "zz.z" Then
    ' very simple checksum test
    ' print "Data valid" ' debug use
    DataValid = 1
  Else
    DataValid = 0
  EndIf
'  print BVolts,SVolts,CVolts,SAmps,CAmps,ACAmps,DCAmps,LAmps,DataValid
  IAmps = DCAmps + ACAmps
  BAmps = IAmps - LAmps
  NewDataFlag = 1 ' set to indicate fresh data
End Sub ' Returns with the following values:-
  ' BVolts = 0 to 15 equalling battery volts
  ' SVolts = 0 to 25 equalling solar Array volts
  ' CVolts = 0 to 25 equalling solar Array volts
  ' SAmps = 0 to 50 equalling 0 to 50Amps
  ' CAmps = 0 to 50 equalling 0 to 50Amps
  ' ACAmps=0 to 50 equalling 0 to 50Amps
  ' DCAmps=0 to 50 equalling 0 to 50Amps
  ' LAmps=0 to 50 equalling 0 to 50Amps
  ' plus flags set/clear
  'End of Interrupt code
  '''''''''''''''''''''
  
  ' Defined Subroutines
  '''''''''''''''''''''
  sub UpdateGUIScreen ' update GUI display
    CtrlVal(Time_Box)  = left$(Time$,5)+"     "+Date$
    if ReEnter = 1 then
      If ActivePage = 1 then     'startup page with help
        colour rgb(white),rgb(white)
        rbox 10,130,780,220,10,rgb(white),rgb(white)
        page 1,3 
      elseif ActivePage = 2 then ' battery volts, current, time to 40%
        colour rgb(white),rgb(white)
        rbox 10,130,780,220,10,rgb(white),rgb(white)
        page 1,2
        gui show Detail_1_Touch
        gui show Detail_2_Touch
        gui show Detail_3_Touch
        gui show Detail_4_Touch
        CtrlVal(Hint_1) = "Touch battery label for history graph"
        ctrlval(DB_Hdr_1) = "Battery~Volts"
        ctrlval(DB_Hdr_2) = "Battery~Amps"
        ctrlval(DB_Hdr_3) = "Battery~Power"
        ctrlval(DB_Hdr_4) = "Battery~Charge %"
      elseif ActivePage = 3 then ' running input and load watt hours
        colour rgb(white),rgb(white)
        rbox 10,130,780,220,10,rgb(white),rgb(white)
        page 1,2      
        gui hide Detail_1_Touch
        gui hide Detail_2_Touch
        gui hide Detail_3_Touch
        gui hide Detail_4_Touch
        CtrlVal(Hint_1) = "Power Details"
        ctrlval(DB_Hdr_1) = "Input~WHrs Today"
        ctrlval(DB_Hdr_2) = "Load~WHrs Today"
        ctrlval(DB_Hdr_3) = "Battery~WHrs Today"
        ctrlval(DB_Hdr_4) = "Time to~Battery 40%"
      elseif ActivePage = 4 then ' watt hours over previous 4 days
        colour rgb(white),rgb(white)
        rbox 10,130,780,220,10,rgb(white),rgb(white)
        page 1,2
        gui hide Detail_1_Touch
        gui hide Detail_2_Touch
        gui hide Detail_3_Touch
        gui hide Detail_4_Touch
        CtrlVal(Hint_1) = "Power History"
        ctrlval(DB_Hdr_1) = "WattHours~Yesterday"
        ctrlval(DB_Hdr_2) = " -2 Days~WattHours"
        ctrlval(DB_Hdr_3) = " -3 Days~WattHours"
        ctrlval(DB_Hdr_4) = " 3Day Total~WattHours"
      elseif ActivePage = 5 then ' input voltage and current summary
        colour rgb(white),rgb(white)
        rbox 10,130,780,220,10,rgb(white),rgb(white)
        page 1,2
        gui hide Detail_1_Touch
        gui hide Detail_2_Touch
        gui hide Detail_3_Touch
        gui hide Detail_4_Touch
        CtrlVal(Hint_1) = "Input Summary"
        ctrlval(DB_Hdr_1) = "Solar~Volts"
        ctrlval(DB_Hdr_2) = "Car~Volts"
        ctrlval(DB_Hdr_3) = "Solar~Amps"
        ctrlval(DB_Hdr_4) = "Car~Amps"
      elseif ActivePage = 6 then ' help display
        page 1,3
        gui hide Detail_1_Touch
        gui hide Detail_2_Touch
        gui hide Detail_3_Touch
        gui hide Detail_4_Touch
        ctrlval(Help_Box) = Help$
      elseif ActivePage = 7 then ' large display battery volts and charge
        colour rgb(white),rgb(white)
        colour rgb(white),rgb(white)
        rbox 10,130,780,220,10,rgb(white),rgb(white)
        page 1,7
        gui hide Detail_1_Touch
        gui hide Detail_2_Touch
        gui hide Detail_3_Touch
        gui hide Detail_4_Touch
        ctrlval(Hint_1) = " 2 second values "
        ctrlval(LDB_Hdr_1) = "Battery~Volts"
        ctrlval(LDB_Hdr_2) = "Battery~Charge %"
      elseif ActivePage = 8 then ' hint box
        colour rgb(white),rgb(white)
        rbox 10,130,780,220,10,rgb(white),rgb(white)
        page 1,3
      elseif ActivePage = 9 then ' battery volts graph
        page 1
        gui hide Detail_1_Touch
        gui hide Detail_2_Touch
        gui hide Detail_3_Touch
        gui hide Detail_4_Touch
        colour rgb(white),rgb(white)
        rbox 10,130,780,220,10,rgb(white),rgb(white)
        colour rgb(black),rgb(white)
        rbox 10,130,780,220,10,rgb(black),rgb(white)
        font 2
        BVGraph
        text 450,180,"Battery voltage for"
        text 450,196,"the last 3 hours."
        text 450,240,"Press < to return to"
        text 450,256,"battery details page."
      elseif ActivePage = 10 then ' battery amps graph
        page 1
        gui hide Detail_1_Touch
        gui hide Detail_2_Touch
        gui hide Detail_3_Touch
        gui hide Detail_4_Touch
        colour rgb(white),rgb(white)
        rbox 10,130,780,220,10,rgb(white),rgb(white)
        colour rgb(black),rgb(white)
        rbox 10,130,780,220,10,rgb(black),rgb(white)
        font 2
        BAGraph ' current in amps in or out
        text 450,180,"Battery current in (green)"
        text 450,196,"or out (red) for the"
        text 450,212,"last 3 hours."
        text 450,240,"Press < to return to"
        text 450,256,"battery main page."
      elseif ActivePage = 11 then ' battery power graph
        page 1
        gui hide Detail_1_Touch
        gui hide Detail_2_Touch
        gui hide Detail_3_Touch
        gui hide Detail_4_Touch
        colour rgb(white),rgb(white)
        rbox 10,130,780,220,10,rgb(white),rgb(white)
        colour rgb(black),rgb(white)
        rbox 10,130,780,220,10,rgb(black),rgb(white)
        font 2
        BPGraph ' power in watt hours in or out
        text 450,180,"Battery charge (green) or"
        text 450,196,"discharge (red) in WHrs"
        text 450,212,"for the last 3 hours."
        text 450,240,"Press < to return to"
        text 450,256,"battery main page."
      elseif ActivePage = 12 then ' battery charge graph
        page 1
        gui hide Detail_1_Touch
        gui hide Detail_2_Touch
        gui hide Detail_3_Touch
        gui hide Detail_4_Touch
        colour rgb(white),rgb(white)
        rbox 10,130,780,220,10,rgb(white),rgb(white)
        colour rgb(black),rgb(white)
        rbox 10,130,780,220,10,rgb(black),rgb(white)
        font 2
        BCGraph ' charge in percent
        text 450,180,"Battery state of charge"
        text 450,196,"in % for the last 3 hours."
        text 450,240,"Press < to return to"
        text 450,256,"battery main page."
      endif
    endif
    ReEnter = 0
end sub

  sub UpdateGUIVals ' update GUI display values
    if ReEnter = 1 then
      CtrlVal(Time_Box)  = left$(Time$,5)+"     "+Date$
      If ActivePage = 1 then
         ' do nothing
      elseif ActivePage = 2 then ' battery details
        gui fColour rgb(black),DB_Val_5
        gui BColour rgb(80,80,255),DB_Val_5
        CtrlVal(DB_Val_5) = str$(BVolts,2,2)
        if BAmps < 0 then
          gui fColour rgb(black),DB_Val_6
          gui BColour rgb(red),DB_Val_6          
          CtrlVal(DB_Val_6) = str$(BAmps,2,1)        
        else
          gui fColour rgb(black),DB_Val_6
          gui BColour rgb(GREEN),DB_Val_6
          CtrlVal(DB_Val_6) = str$(BAmps,2,1)        
        endif
        if BatWH < 0 then
          gui fColour rgb(black),DB_Val_7
          gui BColour rgb(red),DB_Val_7          
          CtrlVal(DB_Val_7) = str$(BatWH,3,0)        
        else
          gui fColour rgb(black),DB_Val_7
          gui BColour rgb(GREEN),DB_Val_7
          CtrlVal(DB_Val_7) = str$(BatWH,3,0)        
        endif
          gui fColour rgb(black),DB_Val_8
          gui BColour rgb(255,255,100),DB_Val_8
        ctrlval(DB_Val_8) = str$(BatCharge)
      elseif ActivePage = 3 then ' power details
          gui fColour rgb(black),DB_Val_5
          gui BColour rgb(GREEN),DB_Val_5
        CtrlVal(DB_Val_5) = str$(InputWH,3,0)
          gui fColour rgb(black),DB_Val_6
          gui BColour rgb(red),DB_Val_6          
        CtrlVal(DB_Val_6) = str$(LoadWH,3,0)
        if BatWH < 0 then
          gui fColour rgb(black),DB_Val_7
          gui BColour rgb(red),DB_Val_7          
          CtrlVal(DB_Val_7) = str$(BatWH,3,0)        
        else
          gui fColour rgb(black),DB_Val_7
          gui BColour rgb(GREEN),DB_Val_7
          CtrlVal(DB_Val_7) = str$(BatWH,3,0)        
        endif
          gui fColour rgb(black),DB_Val_8
          gui BColour rgb(255,255,100),DB_Val_8          
        ctrlval(DB_Val_8) = str$(Time2Discharge)
      elseif ActivePage = 4 then ' power history
        if PrevBWH < 0 then
          gui fColour rgb(black),DB_Val_5
          gui BColour rgb(red),DB_Val_5         
        CtrlVal(DB_Val_5) = str$(PrevBWH,3,0)
        else
          gui fColour rgb(black),DB_Val_5
          gui BColour rgb(GREEN),DB_Val_5
        CtrlVal(DB_Val_5) = str$(PrevBWH,3,0)
        endif
        if Day2BWH < 0 then
          gui fColour rgb(black),DB_Val_6
          gui BColour rgb(red),DB_Val_6         
        CtrlVal(DB_Val_6) = str$(Day2BWH,3,0)
        else
          gui fColour rgb(black),DB_Val_6
          gui BColour rgb(GREEN),DB_Val_6
        CtrlVal(DB_Val_6) = str$(Day2BWH,3,0)
        endif
        if Day3BWH < 0 then
          gui fColour rgb(black),DB_Val_7
          gui BColour rgb(red),DB_Val_7         
        CtrlVal(DB_Val_7) = str$(Day3BWH,3,0)
        else
          gui fColour rgb(black),DB_Val_7
          gui BColour rgb(GREEN),DB_Val_7
        CtrlVal(DB_Val_7) = str$(Day3BWH,3,0)
        endif
        if Day4BWH < 0 then
          gui fColour rgb(black),DB_Val_8
          gui BColour rgb(red),DB_Val_8         
        CtrlVal(DB_Val_8) = str$(TotalBWH,3,0)
        else
          gui fColour rgb(black),DB_Val_8
          gui BColour rgb(GREEN),DB_Val_8
        CtrlVal(DB_Val_8) = str$(TotalBWH,3,0)
        endif
     elseif ActivePage = 5 then 'input summary
          gui fColour rgb(black),DB_Val_5
          gui BColour rgb(cyan),DB_Val_5
        CtrlVal(DB_Val_5) = str$(SVolts,2,2)
          gui fColour rgb(black),DB_Val_6
          gui BColour rgb(magenta),DB_Val_6
        CtrlVal(DB_Val_6) = str$(CVolts,2,2)
          gui fColour rgb(black),DB_Val_7
          gui BColour rgb(GREEN),DB_Val_7
        CtrlVal(DB_Val_7) = str$(SAmps,2,2)
          gui fColour rgb(black),DB_Val_8
          gui BColour rgb(green),DB_Val_8
        ctrlval(DB_Val_8) = str$(CAmps,2,2)
      elseif ActivePage = 6 then ' help box
        ctrlval (Hint_1) = ".... press < for large display"
        ctrlval(Help_Box) = Help$
      elseif ActivePage = 7 then ' large character Battery Volts
        ctrlval (Hint_1) = "Battery Voltage and State of Charge"
        CtrlVal(LDB_Val_1) = str$(BVolts,2,2)        
        ctrlval(LDB_Val_2) = str$(BatCharge)
      elseif ActivePage = 8 then ' hint box
        ctrlval (Hint_1) = "Be kind to your mother!"
      elseif ActivePage = 9 then ' battery  volts graph
        ctrlval (Hint_1) = " Last 3 hours of battery volts"
      elseif ActivePage = 10 then ' battery amps graph
        ctrlval (Hint_1) = " Last 3 hours of battery amps"
      elseif ActivePage = 11 then ' battery power graph
        ctrlval (Hint_1) = " Last 3 hours of battery power"
      elseif ActivePage = 12 then ' battery charge graph
        ctrlval (Hint_1) = " Last 3 hours of battery charge"
     endif
    endif
    ReEnter = 0
end sub

  '---------------------------------------------------
  ' screen touch routines for GUI interface
Sub TouchDown
  ReEnter = 1
  LastPage = ActivePage ' where were we last
  Select Case Touch(REF)
    Case Menu_1_Touch
      ActivePage = 2
      Print "Battery details Touch detected "
    Case Menu_2_Touch
      ActivePage = 3
      Print "Power Details Touch detected "
    Case Menu_3_Touch
      ActivePage = 4
      Print "Power History Touch detected "
    Case Menu_4_Touch
      ActivePage = 5
      Print "Input Summary Touch detected "
    Case Help_Button_Touch
      ActivePage = 6
      Print "? symbol Touch detected "
    Case Back_Arrow_Touch
      if ActivePage = LastPage then
        ActivePage = 7
      else
        ActivePage = LastPage
      endif
      Print "< symbol Touch detected "
    Case Hint_Touch
      ActivePage = 8
      Print "Hint area Touch detected "
    Case Detail_1_Touch
      ActivePage = 9
      Print "Detail 1 Touch detected "
    Case Detail_2_Touch
      ActivePage = 10
      Print "Detail 2 Touch detected "
    Case Detail_3_Touch
      ActivePage = 11
      Print "Detail 3 Touch detected "
    Case Detail_4_Touch
      ActivePage = 12
      Print "Detail 4 Touch detected "
    Case Else
      ReEnter = 0
      Print "?? Touch detected "
      If S_Bright < 96 then
        S_Bright = S_Bright + 5
        backlight S_Bright
      else
        S_Bright = 0
        backlight S_Bright
      endif
  End Select
  UpdateGUIScreen
End Sub
'''''''''''''
'
Sub BatChargeUpdate
  ' Calculate battery change as percentage and fill charge state graph
  ' now calculate charge and paint capacity graph
  ' 12.1V equates to 30% charge - note this is a rough calculation
  '  - it should be done under no load.
  ' *** work needed here ***
  '  - this code attemps to simulate a discharge curve for the purposes
  '    of calculating state of charge under load
  local RemainBC
  local range = 29       ' shape of discharge curve
  local offset = 10.9   ' offset to position range of values
  local curve = 6       ' shape determined by discharge current
  ' ... some rule of thumb values to position values in the
  ' correct range and with a discharge curve approximating a discharge
  ' rate of between 4 and 10 amps
'  curve = 10-(AverageLC/2.1)
  BatCharge = (((BVolts-offset)^2)*range)+ curve
  ' now set up to display charge graphically
  BatCharge = Int(BatCharge)
  If BatCharge < 1 Then BatCharge = 1
  If BatCharge > 100 Then BatCharge = 100
  ' Calculate time to 30% discharge of battery
  RemainBC = BatCharge - 30 ' rough convert from 30% - 100%
  ' to 0Ahrs to 70AHrs remaining charge
  If RemainBC < 1 Then
    RemainBC = 1
  EndIf
  AverageBC = 10
  Time2Discharge = RemainBC/AverageBC  'if we have 70 AHrs and we
  ' discharge at 10AHrs
  ' then we have 7 hours of usage till 30%
  ' - again these are only rough approximations
  
End Sub
  
  '------------------------------------------------------
  ' GUI Setup routine - only called once but safer in a subroutine
sub InitGUI  ' initialise graphics
  gui setup 1 ' same for all pages
    GUI displaybox BGround_Box,0,0,800,480,RGB(white),RGB(white)
    GUI led Led_1,"",770,30,10,RGB(green) ' flash every 2 secs
    Font 4,2
    GUI displaybox Head_Box,10,1,780,54,RGB(black),RGB(white)
    font 6
    GUI Displaybox Time_Box,10,60,780,60,RGB(black),RGB(green)
    font 4,2
    GUI Displaybox Back_Arrow,10,420,70,54,RGB(black),RGB(white)
    GUI Displaybox Help_Button,720,420,70,54,RGB(black),RGB(white)
    font 4    
    GUI Displaybox Menu_1,10,356,180,54,RGB(black),RGB(cyan)
    GUI Displaybox Menu_2,210,356,180,54,RGB(black),RGB(cyan)
    GUI Displaybox Menu_3,410,356,180,54,RGB(black),RGB(cyan)
    GUI Displaybox Menu_4,610,356,180,54,RGB(black),RGB(cyan)
    GUI displaybox Hint_1,100,420,600,54,RGB(black),RGB(white)
    CtrlVal(Head_Box) = "Caravan System Monitor"
    ctrlval(Menu_1) = "Battery~Details"
    ctrlval(Menu_2) = "Power~Details"
    ctrlval(Menu_3) = "Power~History"
    ctrlval(Menu_4) = "Input~Summary"
    ctrlval(Hint_1) = " .. waiting for minute rollover"
    ' areas for boxes above
    GUI area Back_Arrow_Touch,10,420,70,54 '
    GUI area Help_Button_Touch,720,420,70,54 '
    gui area Hint_Touch,100,420,600,54 '
    gui area Menu_1_Touch,10,356,180,54
    GUI area Menu_2_Touch,210,356,180,54
    GUI area Menu_3_Touch,410,356,180,54
    GUI area Menu_4_Touch,610,356,180,54
    gui area Detail_1_Touch,10,130,180,130
    gui area Detail_2_Touch,210,130,180,130
    gui area Detail_3_Touch,410,130,180,130
    gui area Detail_4_Touch,610,130,180,130
  gui setup 2 ' small display battery information
    font 4
    GUI Displaybox DB_Hdr_1,10,130,180,60,RGB(black),RGB(220,220,220)
    GUI Displaybox DB_Hdr_2,210,130,180,60,RGB(black),RGB(220,220,220)
    GUI Displaybox DB_Hdr_3,410,130,180,60,RGB(black),RGB(220,220,220)
    GUI Displaybox DB_Hdr_4,610,130,180,60,RGB(black),RGB(220,220,220)
    font 6
    GUI Displaybox DB_Val_5,10,200,180,60,RGB(black),RGB(magenta)
    GUI Displaybox DB_Val_6,210,200,180,60,RGB(black),RGB(green)
    GUI Displaybox DB_Val_7,410,200,180,60,RGB(black),RGB(red)
    GUI Displaybox DB_Val_8,610,200,180,60,RGB(black),RGB(yellow)
    'areas for boxes above
  gui setup 3 ' help screen
    font 2  
    GUI Displaybox Help_Box,10,130,782,210,RGB(black),RGB(220,220,220)
  gui setup 4 '
  gui setup 5
  gui setup 6
  gui setup 7 ' large battery volts and battery charge
    font 4
    GUI Displaybox LDB_Hdr_1,10,130,380,60,RGB(black),RGB(220,220,220)
    GUI Displaybox LDB_Hdr_2,410,130,380,60,RGB(black),RGB(220,220,220)
    font 6,2
    GUI Displaybox LDB_Val_1,10,200,380,150,RGB(black),RGB(220,220,255)
    GUI Displaybox LDB_Val_2,410,200,380,150,RGB(black),RGB(255,230,50)
  gui setup 8 '
  ' end of GUI setups
end sub
'-------------------------------------------------------------
' graphs that overlay GUI areas
'
sub BVGraph ' graph of battery voltage over last 3 hrs
  ' draw out running graph frame, ticks etc.
  local volts,yrg,xrg,rg_abv,BV3HrPtr
  Colour RGB(black),RGB(white)  
  volts=15
  For yrg =130 To 330 Step 50
    Text 20,yrg,Str$(volts,2,0)+"V",LT,1,1,RGB(blue)
    volts=volts-1
  Next yrg
  Line 70,130,70,330,2,RGB(blue)
  For yrg = 130 To 330 Step 5 'minor ticks
    Pixel 68,yrg,RGB(black)
    Pixel 69,yrg,RGB(black)
  Next yrg
  For yrg = 130 To 330 Step 50  'major ticks
    Pixel 67,yrg,RGB(black)
    Pixel 68,yrg,RGB(black)
    Pixel 69,yrg,RGB(black)
  Next yrg
  Line 70,330,430,330,2,RGB(blue) 'now for the bottom
  For xrg = 70 To 430 Step 30 ' 15 min ticks
    pixel xrg,331,RGB(black)
    pixel xrg,332,RGB(black)
  Next xrg
  For xrg = 70 To 430 Step 60 ' half hour ticks
    pixel xrg,331,RGB(black)
    pixel xrg,332,RGB(black)
    pixel xrg,333,RGB(black)
  Next xrg
  For xrg = 70 To 430 Step 120 ' hour ticks
    pixel xrg,331,RGB(black)
    pixel xrg,332,RGB(black)
    pixel xrg,333,RGB(black)
    pixel xrg,334,RGB(black)
  Next xrg
  Text 80,335,"Battery voltage for the last 3 hours",LT,1,1,RGB(blue)
  ' now fill graph
  BV3HrPtr = BV3HrPointer - 1
  if BV3HrPtr < 0 then
    BV3HrPtr = 179
  endif
    ' paint out pixels into graph for running values
  if BV3HrArray(BV3HrPtr) = 0 then
    BV3HrArray(BV3HrPtr) = 12.0
  endif
  For xrg = 71 To 431 Step 2
    rg_abv = 330 - Int((BV3HrArray(BV3HrPtr)-11) * 50)
    If rg_abv > 330 Then
      rg_abv = 330
    EndIf
    if rg_abv  < 260 then
      line xrg,rg_abv,xrg,329,2,RGB(green)
    elseif rg_abv > 260 and rg_abv < 270 then
      line xrg,rg_abv,xrg,329,2,RGB(yellow)
    elseif rg_abv > 270 then
      line xrg,rg_abv,xrg,329,2,RGB(red)
    endif
    BV3HrPtr = BV3HrPtr + 1
    if BV3HrPtr =180 then
      BV3HrPtr = 0
    endif
  Next xrg
End Sub

sub BAGraph ' graph of battery current over last 3 hrs
  ' draw out running graph frame, ticks etc.
  local amps,yrg,xrg,rg_aba,BA3HrPtr
  Colour RGB(black),RGB(white)    
  amps=20
  For yrg =130 To 330 Step 25
    Text 20,yrg,Str$(amps,2,0)+"A",LT,1,1,RGB(blue)
    amps=amps-5
  Next yrg
  Line 70,130,70,330,2,RGB(blue)
  For yrg = 130 To 330 Step 5 'minor ticks
    Pixel 68,yrg,RGB(black)
    Pixel 69,yrg,RGB(black)
  Next yrg
  For yrg = 130 To 330 Step 10  'major ticks
    Pixel 67,yrg,RGB(black)
    Pixel 68,yrg,RGB(black)
    Pixel 69,yrg,RGB(black)
  Next yrg
  Line 70,330,430,330,2,RGB(blue) 'now for the bottom
  For xrg = 70 To 430 Step 30 ' 15 min ticks
    pixel xrg,331,RGB(black)
    pixel xrg,332,RGB(black)
  Next xrg
  For xrg = 70 To 430 Step 60 ' half hour ticks
    pixel xrg,331,RGB(black)
    pixel xrg,332,RGB(black)
    pixel xrg,333,RGB(black)
  Next xrg
  For xrg = 70 To 430 Step 120 ' hour ticks
    pixel xrg,331,RGB(black)
    pixel xrg,332,RGB(black)
    pixel xrg,333,RGB(black)
    pixel xrg,334,RGB(black)
  Next xrg
  Line 70,230,430,230,1,RGB(blue) 'now the middle
  Text 80,335,"Battery current for the last 3 hours",LT,1,1,RGB(blue)
  ' now fill graph
  BA3HrPtr = BA3HrPointer - 1
  if BA3HrPtr < 0 then
    BA3HrPtr = 179
  endif
    ' paint out pixels into graph for running values
'  if BA3HrArray(BA3HrPtr) < 0 then
'    baneg =1
'  endif
  For xrg = 71 To 431 Step 2
    rg_aba = 230 - Int((BA3HrArray(BA3HrPtr)) * 5)
    If rg_aba > 330 Then
      rg_aba = 330
    elseif rg_aba < 130 then
      rg_aba = 130
    EndIf
    if rg_aba  < 230 then
      line xrg,rg_aba,xrg,230,2,RGB(green)
    elseif rg_aba > 230 then
      line xrg,230,xrg,rg_aba,2,RGB(red)
    endif
    BA3HrPtr = BA3HrPtr + 1
    if BA3HrPtr =180 then
      BA3HrPtr = 0
    endif
  Next xrg
End Sub

sub BPGraph ' graph of power usage over last 3 hrs
  ' draw out running graph frame, ticks etc.
  local power,yrg,xrg,rg_abp,BP3HrPtr
  Colour RGB(black),RGB(white)    
  power=200
  For yrg =130 To 330 Step 25
    Text 12,yrg,Str$(power,3,0)+"WHrs",LT,1,1,RGB(blue)
    power=power-50
  Next yrg
  Line 90,130,90,330,2,RGB(blue)
  For yrg = 130 To 330 Step 5 'minor ticks
    Pixel 88,yrg,RGB(black)
    Pixel 89,yrg,RGB(black)
  Next yrg
  For yrg = 130 To 330 Step 25  'major ticks
    Pixel 87,yrg,RGB(black)
    Pixel 88,yrg,RGB(black)
    Pixel 89,yrg,RGB(black)
  Next yrg    
  Line 90,330,450,330,2,RGB(blue) 'now for the bottom
  For xrg = 90 To 450 Step 30 ' 15 min ticks
    pixel xrg,331,RGB(black)
    pixel xrg,332,RGB(black)
  Next xrg
  For xrg = 90 To 450 Step 60 ' half hour ticks
    pixel xrg,331,RGB(black)
    pixel xrg,332,RGB(black)
    pixel xrg,333,RGB(black)
  Next xrg
  For xrg = 90 To 450 Step 120 ' hour ticks
    pixel xrg,331,RGB(black)
    pixel xrg,332,RGB(black)
    pixel xrg,333,RGB(black)
    pixel xrg,334,RGB(black)
  Next xrg
  Line 90,230,450,230,1,RGB(blue) 'now the middle
  Text 100,335,"Battery WHrs for the last 3 hours",LT,1,1,RGB(blue)
  ' now fill graph
  BP3HrPtr = BP3HrPointer - 1
print BP3HrPtr
  if BP3HrPtr < 0 then
    BP3HrPtr = 179
  endif
    ' paint out pixels into graph for running values
  For xrg = 91 To 451 Step 2 ' step across x axis
    rg_abp = 230 - Int((BP3HrArray(BP3HrPtr)))
    If rg_abp > 330 Then
      rg_abp = 330
    elseif rg_abp < 130 then
      rg_abp = 130
    EndIf
    if rg_abp  < 230 then
      line xrg,rg_abp,xrg,230,2,RGB(green)
    elseif rg_abp > 230 then
      line xrg,230,xrg,rg_abp,2,RGB(red)
    endif
    BP3HrPtr = BP3HrPtr + 1
    if BP3HrPtr =180 then
      BP3HrPtr = 0
    endif
  Next xrg
End Sub

sub BCGraph ' graph of battery charge left over last 3 hrs
  ' draw out running graph frame, ticks etc.
  local charge,yrg,xrg,rg_abc,BC3HrPtr
  Colour RGB(black),RGB(white)  
  charge=100
  For yrg =130 To 330 Step 20
    Text 20,yrg,Str$(charge,2,0)+"%",LT,1,1,RGB(blue)
    charge=charge-10
  Next yrg
  Line 70,130,70,330,2,RGB(blue)
  For yrg = 130 To 330 Step 10 'minor ticks
    Pixel 68,yrg,RGB(black)
    Pixel 69,yrg,RGB(black)
  Next yrg
  For yrg = 130 To 330 Step 20  'major ticks
    Pixel 67,yrg,RGB(black)
    Pixel 68,yrg,RGB(black)
    Pixel 69,yrg,RGB(black)
  Next yrg
  Line 70,330,430,330,2,RGB(blue) 'now for the bottom
  For xrg = 70 To 430 Step 30 ' 15 min ticks
    pixel xrg,331,RGB(black)
    pixel xrg,332,RGB(black)
  Next xrg
  For xrg = 70 To 430 Step 60 ' half hour ticks
    pixel xrg,331,RGB(black)
    pixel xrg,332,RGB(black)
    pixel xrg,333,RGB(black)
  Next xrg
  For xrg = 70 To 430 Step 120 ' hour ticks
    pixel xrg,331,RGB(black)
    pixel xrg,332,RGB(black)
    pixel xrg,333,RGB(black)
    pixel xrg,334,RGB(black)
  Next xrg
  Text 80,335,"Battery state of charge for the 3 hours",LT,1,1,RGB(blue)
  ' now fill graph
  BC3HrPtr = BC3HrPointer - 1
  if BC3HrPtr < 0 then
    BC3HrPtr = 179
  endif
    ' paint out pixels into graph for running values
  if BC3HrArray(BC3HrPtr) = 0 then
    BC3HrArray(BC3HrPtr) = 12.0
  endif
  For xrg = 71 To 431 Step 2
    rg_abc = 330 - Int((BC3HrArray(BC3HrPtr)) * 2)
    If rg_abc > 330 Then
      rg_abc = 330
    EndIf
    if rg_abc  < 220 then
      line xrg,rg_abc,xrg,329,2,RGB(green)
    elseif rg_abc > 220 and rg_abc < 250 then
      line xrg,rg_abc,xrg,329,2,RGB(yellow)
    elseif rg_abc > 250 then
      line xrg,rg_abc,xrg,329,2,RGB(red)
    endif
    BC3HrPtr = BC3HrPtr + 1
    if BC3HrPtr =180 then
      BC3HrPtr = 0
    endif
  Next xrg
End Sub

'-----------------------------------------------------
' Geoff's Message Field$ Routine (with one amendment
'   by CaptainBoing)
'-----------------------------------------------------
Function Field$(DataVals$,n_field,delim$)
  Local Integer i, StartIdx = 1, EndIdx
  local dv$ length 64
  ' enter with DataVals$ with 9 fields in the form
  ' BVolts,SVolts,CVolts,SAmps,CAmps,ACAmps,DCAmps,LAmps,DataValid
  ' note always 9 fields but field length may vary
  dv$ = DataVals$
  ' get the start of the field in StartIdx
  For i = 2 To n_field
    StartIdx = Instr(StartIdx, dv$, delim$) + 1
    If StartIdx = 1 Or StartIdx > Len(dv$) Then
      Exit Function
    endif
  Next i
  ' get the end of the field in EndIdx and extract the field
  EndIdx = Instr(StartIdx, dv$, delim$)
  If EndIdx = 0 Then
    EndIdx = Len(dv$)+1 ' makes it smaller than 255
  endif
  Field$ = Mid$(dv$, StartIdx, EndIdx - StartIdx)
End Function
  
'''''''''
' End program.
''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''  
