'*************************************************************
' Caravan System Monitor for 12V Battery System              '
' Written by Doug Pankhurst May 2015, based on coding        '
' ideas from Tom Pankhurst TZAdvantage, CaptainBoing and     '
' various other Back Shed Forum members - thanks.            '
' Hardware is a Peter Mather designed 470 Backpack back end &'
' custom PIC32MX170 front end.                               '
' Written for MMBasic V5.5.02B6  copyright Geoff Graham 2012 '
' Program runs on PIC32MX470 with an SSD1930 7in touch screen'
' This program is free for anyone to use or modify as they   '
'  choose however, if you wish to re-distribute, please      '
'  acknowledge authors and contributors above.               '
''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
' File Version:
' 3.5b1 6 Mar 2025 cbe-35b1-MMB4W.bas
' Notes:
' This version adapted for MMB4W running under wine
' Current stable version - cbe-35b2-MMB4W.bas 2 AUG 2025
' running on MMBasic V5.05.05
' - last known good version live in caravan cbe-34c-live.bas
' Updates: 
' 22 APR 2024 -upgrade to display date/time large
' 12 Mar 2023 - add clock; move graph select
' Note: caravan back end running MMBasic 5.05.05
' For a detailed explanation of the program, it's
' construction and rationale, please read the
' Caravan Battery Monitor doco AUG2025-3.pdf file.
'************************************************************
' Notes:
' 1.  To run in Demonstrate Mode, link COM1 Tx to COM1 Rx.
' This will return some simulated values for demonstration
' purposes. Program automatically detects and sets/clears
' simulation mode if COM1 loopback in place.
' 2. Option list
' OPTION LCDPANEL SSD1963,_7,RL,,30
' GUI TEST LCDPANEL
' OPTION TOUCH 51,33,42
' GUI CALIBRATE
' OPTION SDCARD 52
' OPTION RTC 43,44
'*************************************************************
' Initial Entry to Program
EntryPoint:
  SetTick 0,0        ' Disable timer interrupt for now
  Option explicit
  ' Option autorun on  ' enable as required
  ' Colours
  Const BLK = RGB(BLACK)
  Const WHT = RGB(white)
  Const RED = RGB(RED)
  Const YEL = RGB(YELLOW)
  Const GRN = RGB(GREEN)
  Const MAG = RGB(MAGENTA)
  Const CYN = RGB(CYAN)
  Const LGR = RGB(200,200,200)
  Const DGR = RGB(130,130,130)

  ' Define constants for gui controls - as the display box GUI does not
  ' support touch, I used the AREA GUI control to overlay display boxes 
  ' for the menu system.
  ' Touch areas for display boxes - these must have a lower control
  ' value than any thing they overlay.

  ' gui controls
  Const Back_Arrow_Touch  = 1
  Const Help_Button_Touch = 2
  Const Hint_Touch    = 4
  Const Menu_1_Touch  = 5  ' 4 lower menu box touch areas
  Const Menu_2_Touch  = 6  ' to select display pages
  Const Menu_3_Touch  = 7
  Const Menu_4_Touch  = 8
  Const Frame_1_Touch = 9 ' touch area overlaying Battery Volts/Amps frame.
'  const Frame_2_Touch = 10 ' Note that touch frames 2,3 and 4 are
'  const Frame_3_Touch = 11 ' not used yet.
'  const Frame_4_Touch = 12 '
  const BigTime_Touch = 13 ' touch area to switch to large time display
  Const Graph_Touch    = 14 '  - touch to toggle graphs
  Const Graph_BG      = 16  ' background for graphs
  Const Whiteboard    = 17  ' the display area
  Const Led_1         = 18
  Const Frame_1    = 20  ' 4 display boxes as frames enclosing boxes
  Const Frame_2    = 21  ' below used as labels for values in boxes below
  Const Frame_3    = 22  ' -note frame 1 also has a touch area overlayed
  Const Frame_4    = 23
  Const DB_Val_5   = 34    ' 12 boxes for plugging in values
  Const DB_Val_6   = 35
  Const DB_Val_7   = 36
  Const DB_Val_8   = 37
  Const DB_Val_9   = 38
  Const DB_Val_10  = 39
  Const DB_Val_11  = 40
  Const DB_Val_12  = 41
  Const DB_Val_13  = 42
  Const DB_Val_14  = 43
  Const DB_Val_15  = 44
  Const DB_Val_16  = 45
  Const Graph_Text = 56
  Const Menu_1     = 60  ' 4 lower menu selection boxes
  Const Menu_2     = 61  ' - note these have touch areas overlayed
  Const Menu_3     = 62
  Const Menu_4     = 63
  Const Help_Box   = 64 ' used for help messages
  Const LDB_Val_1  = 65
  Const LDB_Val_2  = 66
  Const LDB_Hdr_1  = 67 ' 2 large display boxes for labels
  Const LDB_Hdr_2  = 68
  Const BCBarGraph = 69
  Const BVBarGraph = 70
  const BigTime_Box = 71 ' takes up whole LDB area for big time display
  Const Head_Box   = 80
  Const Time_Box   = 81 ' the heading box with date and time and led
  Const Hint_1     = 82
  Const Back_Arrow = 83 ' back arrow box
  Const Help_Button= 84 ' question mark box
  ' other constants
  Const NUL = &H00 ' ASCII null or zero value
  Const ENQ = &H05 ' ASCII enquiry
  Const ACK = &H06 ' ASCII acknowledge
  Const BEL = &H07 ' ASCII bell sound
  Const NAK = &H15 ' ASCII negative acknowledge
  Const ESC = &H1B ' ASCII escape character
'Define variables
  ' Global Variables - initialised to default values
  Dim F_Ver$ = "3.4d"   ' File Version cbe-34d-live.bas 22 April 2024
  ' Voltage and current of caravan battery
  Dim BVolts         ' Battery Voltage
  Dim BVArray(29)    ' Battary Volts 30 element array - 1 each 2 secs
  Dim AverageBV      ' Average battery voltage over last 60 seconds
  Dim PreviousABV    '
  Dim Time2Discharge ' Time to 30% discharge in minutes
  Dim BatCharge      ' Battery state of charge as a percentage where
  ' 12.75V or higher is 100%, down to 12.05V being 35% - danger level
  ' Voltage and current from Solar Panels
  Dim SVolts         ' Solar Array voltage
  Dim SVArray(29)    ' Solar Volts 30 element array - 1 each two secs
  Dim AverageSV      ' Average Solar Panel volts over last minute
  Dim SAmps          ' current from Solar Array into D250
  Dim SAArray(29)    ' solar panel current array - 1 each two secs
  Dim AverageSC      ' average solar panerl current last minute
  ' Voltage and current from towing vehicle
  Dim CVolts         ' Vehicle battery volts
  Dim CVArray(29)    ' Car volts 30 element array  - 1 each two secs
  Dim AverageCV      ' Average vehicle volts ove rlast 30 60 secs
  Dim CAmps          ' current from vehicle into D250
  Dim CAArray(29)    ' car current array  - 1 each two secs
  Dim AverageCC      ' average car current last minute
  ' Input (charging), output (load) and battery charge/discharge currents.
  ' Battery charge/discharge is calculated from input minus load currents.
  ' - instantaneous values, minute averaged value currents
  ' - daily accumulating input, output and battery amp hour useage
  ' - last 4 days battery charge/discharge currents and rolling total
  Dim ACAmps         ' Charging current from XPS25000 240VAC charger
  Dim DCAmps         ' Charging current from D250 Solar Charger
  Dim DCAArray(29)   ' D250 minute current array  - 1 each 2 secs
  Dim ACAArray(29)   ' XPS25000 240VAC minute current array - 1 each two secs
  Dim AverageDCC     ' average D250 output current last minute
  Dim AverageACC     ' average X25000 output current last minute
  ' Input current -  a calculated sum of AC Charger and DC charger currents
  Dim IAmps          ' Charging current into battery/load
  Dim AverageIC      ' Average input current
  Dim LAmps          ' Load current - supplied from batteries/charging systems
  Dim LAArray(29)    ' Load Current 60 element array
  Dim AverageLC      ' Average load current
  ' Battery current - a calculated value; if positive, = a charging current to
  ' the battery; if negative, equals a discharge current from the battery.
  Dim BAmps          ' Input charging current minus load current
  Dim AverageBC      ' Average charge/discharge current
  ' daily accumulating values
  Dim InputAH        ' daily running input Amp Hours
  Dim LoadAH         ' daily running load Amp Hours
  Dim BatAH          ' daily running battery Amp Hours
  Dim InputWH        ' daily running input Watt Hours
  Dim LoadWH         ' daily running load Watt Hours
  Dim BatWH          ' daily running battery Watt Hours
                     ' as battery charge (+) or discharge (-) in Whs
  ' previous days values
  Dim PrevBWH        ' yesterdays Bat Current Wh
  Dim PrevIWH        ' yesterdays Input whs
  Dim PrevLWH        ' yesterdays Load whs
  Dim Day2BWH        ' -2 and -3 days input, load and battery Wh usage
  Dim Day2IWH
  Dim Day2LWH
  Dim Day3BWH
  Dim Day3IWH
  Dim Day3LWH
  Dim TotalBWH       ' rolling 4 day KWHr total usage
  Dim TotalIWH
  Dim TotalLWH
  ' These 4 arrays are for storing 3 hours worth of data
  ' for use in the graphs selected from the Battery Details page
  Dim BV3HrArray(179)  ' 3 hour average battery volts updated
                       ' every minute to give continuous
                       ' last 3 hours values
  Dim BV3HrPointer     ' index into latest average value
  Dim BA3HrArray(179)  ' 3 hour average battery amps updated
                       ' every minute to give continuous
                       ' last 3 hours values
  Dim BA3HrPointer     ' index into latest average value
  Dim BP3HrArray(179)  ' 3 hour average battery power updated
                       ' every minute to give continuous
                       ' last 3 hours values
  Dim BP3HrPointer     ' index into latest average value
  Dim BC3HrArray(179)  ' 3 hour average battery charge updated
                       ' every minute to give continuous
                       ' last 3 hours values
  Dim BC3HrPointer     ' index into latest average value
  Dim SV3HrArray(179)  ' 3 hour average solar volts updated
                       ' every minute to give continuous
                       ' last 3 hours values
  Dim SV3HrPointer     ' index into latest average value

  ' misc variables
  Dim DataVals$ length 64 ' front end running values
  Dim Help$          ' string to display in help box
  Dim ActiveScreen = 1 'default to main GUI page
  Dim LastScreen = 1   '  remember where we were last
  Dim Buzzer = 29    ' pin D8 on shiled to buzzer
  '  dim ReEnter = 0    ' flag to control re-entry into GUI display
  Dim Ax             'loop counter
  Dim S_Bright = 95  ' initial display brightness
  Dim delim$ = ","   ' delimiter for field extract
  Dim n_field = 1    ' number of field in front end data stream
  Dim hrs,mins,secs  ' used in calculating time flags
  Dim twosecled = 1  ' LED flashes on 2 sec interrupt
  Dim bdcount = 0    ' bad data counter
  Dim LinkDown = 0   ' link failure counter
  Dim LinkError$     ' holds link error messages
  Dim GText$         ' use with graph display box
  ' some variables to simulate a 6Amp discharge rate
  Dim cf1 = 14       ' AverageBC  - offset of x from zero
  Dim cf2 = 0.15     ' shape of curve
  Dim cf3 = 11.3     ' 0% charge point
  Dim tbv = 13
  ' Flags
  Dim FEUpFlag = 0   ' front end link up flag
  Dim NewDataFlag = 0   ' Two second flag - set every
  ' 2 seconds by the interrupt routine when new voltage
  ' and current values are collected.
  Dim TwoSecEvenFlag = 0 ' data request to front end on odd 2 secs,
  ' - front end response checked on even 2 secs
  Dim MinFlag = 0 ' Minute flag - set once every min at 00 seconds
  ' by sec interrupt process and cleared by min processing code.
  Dim EODFlag = 0 ' end of day at 6AM to reset accumulating WHrs
  Dim RTCFlag = 0 ' reset internal clock from RTC at midnight
  Dim TouchDetected = 0 ' flag touch
  Dim GraphToggle = 1  ' flag to toggle through graph displays
  Dim SimFlag = 1  ' 1 for testing, will flip back to 0 if front end
                   '   replies with valid data
  Dim debug = 0  ' flag to indicate diagnostic mode errors to console
  Dim null$ = "" ' holder for rubbish
  Dim ReqChr$ = "R"
  Dim as Integer MBLU,LTGRY,MGRY,LTGRN,LTBLU
  MBLU = RGB(190,190,190)
  LTGRY = RGB(220,220,220)
  MGRY = RGB(200,200,200)
  LTGRN = RGB(210,230,210)
  LTBLU = RGB(220,220,240)
''''''''''''''''''''''''''''''''''''''''''''''
' Carry out initialisation routines here - enable interrupt last
StartPoint:
'  BackLight S_Bright  ' set initially for 20%, tap heading to increase
'  SetPin Buzzer,DOUT
  GUI interrupt TouchDown ' subroutine to handle touch int
'  RTC GETTIME  ' disabled in MMB4W - enabled in live version for date and
'   time from DS3231 

  ' set up all GUI controls - only called once here but safer in a subroutine
  InitGUI ' routine to set GUI controls
  ' preset some display boxes
  CtrlVal(Head_Box) = "Caravan System Monitor"
  CtrlVal(Time_Box)  = Time$+"     "+Date$
  CtrlVal(Back_Arrow) = "<" ' left arrow
  CtrlVal(Help_Button) = "?" ' home
  ' Fill the the minute average arrays
  ' just so initial values make sense
  For Ax=0 To 29
    BVArray(Ax) = 12.8
    SVArray(Ax) = 19
    CVArray(Ax) = 13
    SAArray(Ax) = 10
    CAArray(Ax) = 6
    DCAArray(Ax) = 16
    ACAArray(Ax) = 0
    LAArray(Ax) = 12
  Next Ax
  ' preset some variables
  Help$ = "Caravan System Monitor v"+F_ver$+"  "+Date$+"~"
  Help$ = Help$+ "Created by D.Pankhurst. MMBasic Copyright Geoff Graham~"
  Help$ = Help$+ "Select action from menu boxes or < to restore main page~~"
  Help$ = Help$+ "Repeatedly tap header to increase brightness"
  CtrlVal(Help_Box) = Help$
  CtrlVal(Hint_1) = "Press < for large battery volts"
  ' Display all static data
  ActiveScreen = 1

  UpdateGUIScreen  ' paint initial GUI screen then only
                   ' done when a touch detected
  ''''''''''''''''''''''
  ' Initialise interrupts
  ' Data values come in from front end via COM1
  ' Note: Front end action could be included in this
  '  program if the system is close enough to read
  '  values directly
  Open "COM4:9600,256,GetVals" As #1 ' int after LF char received
  ' I use COM4 in MMB4W under Linux/wine. See Annex in manual
    null$ = Input$(128,#1)' get rid of any junk in com input buffer
  ' Enable timer tick interrupts to start data collection at 00 seconds
  SetTick 1000,RequestData  ' Call every second, update time display then,
  ' if even second, send "R" to front end that initiates a data dump into
  ' COM1 from the front end data collection processor.

  Pause 5000
  ActiveScreen = 2
  UpDateGUIScreen
  ' end of initialisation routines
  ''''''''''''''''''''''''''''''''
Main:
  Do    'Main program loop, for every 1 sec interrupt, update time display,
  ' test for even numbered second and if so, send data request, if data
  ' received, get the data, do 2 second, minute and end of day
  ' processing as appropriate
    Do While NewDataFlag = 0   ' wait until data request sent
      ' from back end to front end (by SetTick interrupt) which then
      ' generates fresh data and sends it to back end (this program)
      ' by a COM1 interrupt.
      If TouchDetected = 1 Then ' while waiting for data, check for touch
      ' yes, so update display
        UpdateGUIScreen ' vals updated on next 2 sec interrupt
        TouchDetected = 0 ' reset ready for next touch
      EndIf
    Loop
   ' should only get here if COM1 interrupt occured with
    ' good fresh data received so now process
'    if TwoSecEvenFlag = 0 then ' must be odd sec
      TwoSecProcess  ' real time data
'      TwoSecEvenFlag = 0
'    endif
    If MinFlag = 1 Then
      MinProcess   ' minute average data
      MinFlag = 0
    EndIf
    If EODFlag = 1 Then
      EODProcess   ' end of day tidy up
      EODFlag = 0
    EndIf
    If RTCFlag = 1 Then
      RTC GETTIME
      RTCFlag = 0
    EndIf
    UpdateGUIVals ' all processed so update values of screen display
  Loop
  ' end of main process loop

''''''''''''''''''''''''''''''''
' main data processing subroutines
Sub TwoSecProcess  'only on odd seconds - data request sent on even secs
  ' Entered with the following instantaneous values from front end
  ' battery volts, solar volts, solar amps, vehicle amps
  ' 240VAC charger amps, solar/car charger amps
  ' total input amps (solar/car + 240VAC), load amps
  ' battery amps (display as green charging, red discharging)
  Local Ax2  '2 second array pointer
  ' 2 second processing code - every interrupt sets 2 sec flag if even second
  ' Push most recent values into arrays for minute averaging
  ' then set up for minute averaging
  For Ax2 = 29 To 1 Step -1  ' shuffle everyone down one spot
    BVArray(Ax2) = BVArray(Ax2-1)
    SVArray(Ax2) = SVArray(Ax2-1)
    CVArray(Ax2) = CVArray(Ax2-1)
    SAArray(Ax2) = SAArray(Ax2-1)
    CAArray(Ax2) = CAArray(Ax2-1)
    DCAArray(Ax2) = DCAArray(Ax2-1)
    ACAArray(Ax2) = ACAArray(Ax2-1)
    LAArray(Ax2) = LAArray(Ax2-1)
  Next Ax2

    ' so now extract the front end data from DataVals$
    BVolts = Val(Field$(DataVals$,1,delim$))
    If BVolts < 12.2 Then
      Print BVolts ' debug
      BatWarn1
    EndIf
    SVolts = Val(Field$(DataVals$,2,delim$))
    CVolts = Val(Field$(DataVals$,3,delim$))
    If CVolts < 10 Then
      CVolts = 0
    EndIf
    SAmps  = Val(Field$(DataVals$,4,delim$))
    If SVolts = 0 Then
      SAmps = 0
    EndIf
    CAmps  = Val(Field$(DataVals$,5,delim$))
    If CVolts = 0 Then
       CAmps = 0
    EndIf
    DCAmps = Val(Field$(DataVals$,6,delim$))
    ACAmps = Val(Field$(DataVals$,7,delim$))
    LAmps  = Val(Field$(DataVals$,8,delim$))

  ' then plug in new values to first spot in averaging array
  BVArray(0) = BVolts
  SVArray(0) = SVolts
  CVArray(0) = CVolts
  SAArray(0) = SAmps
  CAArray(0) = CAmps
  DCAArray(0) = DCAmps
  ACAArray(0) = ACAmps
  LAArray(0) = LAmps
  ' Battery Current is IAmps-LAmps - if positive
  '  it is a charging current into battery, if negative, it is a
  '  discharging current out of battery
  IAmps = DCAmps + ACAmps  ' from both mains and solar/car chargers
  BAmps = IAmps-LAmps

  ' - now update terminal display with all values
  BatChargeUpdate  ' sub to calculate display battery charge
  NewDataFlag = 0  ' clear new data flag ready for next interrupt
  ' End of two second processing code
End Sub

  '''''''''''''''''''''''''''''''''''
Sub MinProcess
  Local Ax1m ' 1 minute array pointer
  ' minute flag set so calculate the average of the last
  ' 60 seconds of readings
  PreviousABV = AverageBV   ' save for discharge calculation purposes
  ' clear ready for averaging
  AverageBV = 0
  AverageSV = 0
  AverageCV = 0
  AverageSC = 0
  AverageCC = 0
  AverageDCC = 0
  AverageACC = 0
  AverageLC = 0
  AverageBC = 0
  For Ax1m = 0 To 29   'don't forget only sampled every 2 secs
    AverageBV = AverageBV + BVArray(Ax1m)
    AverageSV = AverageSV + SVArray(Ax1m)
    AverageCV = AverageCV + CVArray(Ax1m)
    AverageSC = AverageSC + SAArray(Ax1m)
    AverageCC = AverageCC + CAArray(Ax1m)
    AverageDCC = AverageDCC + DCAArray(Ax1m)
    AverageACC = AverageACC + ACAArray(Ax1m)
    AverageLC = AverageLC + LAArray(Ax1m)
  Next Ax1m
  ' now average
  AverageBV = AverageBV / 30
  AverageSV = AverageSV / 30
  AverageCV = AverageCV / 30
  AverageSC = AverageSC / 30
  AverageCC = AverageCC / 30
  AverageDCC = AverageDCC / 30
  AverageACC = AverageACC / 30
  AverageLC = AverageLC / 30
  AverageIC = AverageDCC + AverageACC  ' sum X250 car/solar
                                       ' plus X25000 AC chargers
  AverageBC = AverageIC - AverageLC ' sum of amps in(+) & out(-)
  ' Convert both input, output & battery minute average
  '  currents into Amps Hours and Watt Hours
  ' Note: For a 200AHr battery system, at a discharge rate of 1A,
  ' 50% discharge (allowing for losses) should be around 80Hrs.
  ' For a nominal 12V battery value, this equates to a battery
  ' capacity of roughly 960WHrs to 50% discharge.
  ' So, to de-rate again to be safe, say that a reduction
  ' of 500WHrs from the battery should be a safe maximum.
  ' For WHrs, convert minute average to 1hr average
  InputAH = InputAH + AverageIC/60 ' AHrs
  LoadAH = LoadAH + AverageLC/60
  BatAH = (InputAH-LoadAH)*0.8  ' assume 80% efficiency
  InputWH = InputWH + ((AverageIC*AverageBV)/60) ' WHrs
  LoadWH = LoadWH + ((AverageLC*AverageBV)/60)   ' WHrs
  BatWH = (InputWH-LoadWH)*0.8  ' assume 80% efficiency
  ' Note: All WHr accumulating totals are plugged into
  ' last day values then reset to zero at 6AM.

  ' 3 Hr graph updates - latest minute average bat volts
  BV3HrArray(BV3HrPointer) = AverageBV
  BV3HrPointer = BV3HrPointer + 1
  If BV3HrPointer = 180 Then
    BV3HrPointer = 0
  EndIf
  ' now update 3 hr array with latest minute average bat amps
  BA3HrArray(BA3HrPointer) = AverageBC
  BA3HrPointer = BA3HrPointer + 1
  If BA3HrPointer = 180 Then
    BA3HrPointer = 0
  EndIf
  ' now update 3 hr array with latest minute average bat WHrs
  BP3HrArray(BP3HrPointer) = BatWH
  BP3HrPointer = BP3HrPointer + 1
  If BP3HrPointer = 180 Then
    BP3HrPointer = 0
  EndIf
  ' now update 3 hr array with latest minute average bat charge
  BC3HrArray(BC3HrPointer) = BatCharge
  BC3HrPointer = BC3HrPointer + 1
  If BC3HrPointer = 180 Then
    BC3HrPointer = 0
  EndIf
  ' 3 Hr graph updates - latest minute average solar volts
  SV3HrArray(SV3HrPointer) = AverageSV
  SV3HrPointer = SV3HrPointer + 1
  If SV3HrPointer = 180 Then
    SV3HrPointer = 0
  EndIf
  MinFlag = 0   ' reset the min flag
End Sub  ' End of minute code

''''''''''''''''''''''''''''''''''''''
' End of Day process - save WHr values for historical records,
' tidy up and reset. Note EOD currently 0600 Hrs.
Sub EODProcess  ' Check for end of day
  ' - for new day starting at 6AM
  ' shuffle all previous WHr values back a day to allow
  ' new values to be added in then clean out old
  ' running values for new day
  Day3BWH = Day2BWH
  Day2BWH = PrevBWH
  PrevBWH = BatWH
  BatWH = 0
  Day3IWH = Day2IWH
  Day2IWH = PrevIWH
  PrevIWH = InputWH
  InputWH = 0
  Day3LWH = Day2LWH
  Day2LWH = PrevLWH
  PrevLWH = LoadWH
  LoadWH = 0
  TotalBWH = PrevBWH + Day2BWH + Day3BWH
  TotalIWH = PrevIWH + Day2IWH + Day3IWH
  TotalLWH = PrevLWH + Day2LWH + Day3LWH
  ' clean out running AHr values for new day
  InputAH = 0
  LoadAH = 0
  BatAH = 0
  EODFlag = 0    ' clear end of day flag until next 6AM
End Sub  ' End of EOD code
''''''''''''''''''''''''''

''''''''''''''''''''''''''
' Interrupt handling subroutines
' 1 Second SETTICK Interrupt Routine Handling to send data request to
' front end - also tests/sets the SimFlag, MinFlag,RTCFlag and EODFlag
Sub RequestData      ' SetTick Timer Interrupt handler - every second
  secs = Val(Right$(Time$,2)) ' grab time into component variables
  mins = Val(Mid$(Time$,4,2))
  hrs = Val(Left$(Time$,2))
  CtrlVal(Time_Box)  = Time$+"     "+Date$
  EndIf
  ' test for xx:xx:00 ie, minute rollover
  If secs = 0 Then
    MinFlag = 1
  Else
    MinFlag = 0
  EndIf
  ' test for 06:00, ie. nominal end of day
  ' - for testing only, if SimFlag = 1 then EOD will be assumed
  '   to be every 3 minutes
  If secs = 0 And mins = 0 And hrs = 6 Then
    EODFlag = 1
  Else
    EODFlag = 0
  EndIf
  ' reset clock at midnight so not to interfere with EOD process
  If secs = 0 And mins = 0 And hrs = 0 Then
      RTCFlag = 1
  EndIf
  ' test for even numbered seconds - thus only send front end
  ' request every 2 secs
  If secs Mod 2 = 0 Then ' even 2 sec boundary so send data request
    TwoSecEvenFlag = 1
    CtrlVal(Hint_1) = "Data request to fe sent at ",Time$
    Print #1,ReqChr$;  ' the data request character to front end
    If FEUpFlag = 0 Then
      LinkDown = LinkDown + 1
      If debug = 1 Then
        Print "Front End - no response to last data request",Time$
        Print "Link down count = ",LinkDown
      EndIf
      CtrlVal(Hint_1) = "Front end down count = "+Str$(LinkDown)
    EndIf
  EndIf
End Sub
'-----------------------------
' COM1 Interrupt to receive data string from front end with all data values
Sub GetVals  ' Receive character interrupt handler
  Local junk1$
  ' structure of DataVal$ -
  ' battery volts vv.vv, solar volts vv.v, car volts vv.vv,
  ' solar amps ii.i, car amps ii.i,
  ' dc charge amps ii.i, ac charge amps ii.i,
  ' load amps ii.i, data valid zz.z  in the form
  ' vv.vv,vv.v,vv.vv,ii.i,ii.i,ii.i,ii.i,ii.i,zz.zz
  ' total 46 bytes
  Pause 5
  DataVals$ =  Input$(64,#1) '
  ' first check if valid data from front end or sim data
  If debug = 1 Then
    Print "Data from FE rec'd at ",Time$  ' debug
    Print DataVals$   ' debug
  EndIf
  If Left$(DataVals$,1) = "R" Then  ' loopback set so plug in simulate data
    DataVals$ = "12.65,19.55,13.85,12.5,20.2,15.8,0,3.7,zz.zz"
    junk1$ = Input$(128,#1) ' clean out input buffer
  EndIf
  If Left$(FIELD$(DataVals$,9,delim$),5) = "zz.zz" Then
  ' very simple checksum test
  'flip twosecled green/black - just a 'me and front end alive' indicator
  ' all good - either front data or simulate data ready
    If twosecled = 1 Then
      CtrlVal(Led_1) = 1
      twosecled = 0
    Else
      CtrlVal(Led_1) = 0
      twosecled = 1
    EndIf
    NewDataFlag = 1 ' set to indicate good fresh data
    FEUpFlag = 1    ' all good link wise
  Else
    bdcount = bdcount +1
    If debug = 1 Then
      Print "Bad data",bdcount ' probably data corruption
      Print "Data - ",DataVals$
    EndIf
    CtrlVal(Hint_1) = "Corrupt data from Front End"
    DataVals$ = ""
    NewDataFlag = 0  ' invalid data from front end
    FEUpFlag = 1  ' front end down or corrupt data
  EndIf
End Sub ' Returns with the following values as string data in DataVals$:-
  ' BVolts = 0 to 15 equalling battery volts
  ' SVolts = 0 to 25 equalling solar Array volts
  ' CVolts = 0 to 25 equalling solar Array volts
  ' SAmps = 0 to 50 equalling 0 to 50Amps
  ' CAmps = 0 to 50 equalling 0 to 50Amps
  ' DCAmps=0 to 50 equalling 0 to 50Amps
  ' ACAmps=0 to 50 equalling 0 to 50Amps
  ' LAmps=0 to 50 equalling 0 to 50Amps
  ' plus pseudo checksum zz.zz
  ' plus NewData flag set/clear
  '   or DataVals$ empty if no response or bad data
'******************************************************
' screen touch routines for GUI interface
Sub TouchDown
  LastScreen = ActiveScreen ' where were we last
  Select Case Touch(REF)
    Case Menu_1_Touch ' Battery Volts
      ActiveScreen = 2
    Case Menu_2_Touch ' Power Details
      ActiveScreen = 3
    Case Menu_3_Touch ' Power History
      ActiveScreen = 4
    Case Menu_4_Touch ' Input Summary
      ActiveScreen = 5
    Case Help_Button_Touch ' ? Button
      ActiveScreen = 6
    Case Back_Arrow_Touch ' < Button
      ActiveScreen = 7
    Case Hint_Touch ' Hint Bar
      ActiveScreen = 8
    Case Frame_1_Touch ' Battery Volts/Amps frame Graph Button
      ActiveScreen = 9
    Case Graph_Touch    ' toggle graph displays
      GraphToggle = GraphToggle + 1
      If GraphToggle = 6 Then
        GraphToggle = 1
      EndIf
    CASE BigTime_Touch ' set to displat time big font
      ActiveScreen 10
      S_Bright = 5
      BACKLIGHT S_Bright
    Case Else ' anywhere else on screen to change brightness
      S_Bright = S_Bright + 15
      If S_Bright > 100 Then
        S_Bright = 0
      EndIf
'      BackLight S_Bright
    End Select
    TouchDetected = 1
  End Sub ' end of touch down subroutine
'End of Interrupt handling routines
'''''''''''''''''''''

' GUI Subroutines
'''''''''''''''''''''
  Sub UpdateGUIScreen ' update GUI display on startup and every time
                      ' a touch is detected - called in main loop
    Local tmp$= ""
    Select Case ActiveScreen
      Case 1      'startup page with help
        GUI Page 1,3
        GUI bcolour RGB(cyan),Menu_1,Menu_2,Menu_3,Menu_4

      Case 2  ' battery volts, current, time to 40%
        GUI Page 1,2,5,4  '
        tmp$ = "Instantaneous Battery Volts/Amps~"
        CtrlVal(Hint_1) = tmp$+"... touch label for history graphs"
        GUI bcolour RGB(0,120,120),Menu_1
        GUI bcolour RGB(cyan),Menu_2,Menu_3,Menu_4
        GUI BCOLOUR BLK,Whiteboard
        CtrlVal(Frame_1) = "Battery~Volts/Amps~~~~~~~~"
        CtrlVal(Frame_2) = "Solar~Volts/Amps~~~~~~~~"
        CtrlVal(Frame_3) = "Car~Volts/Amps~~~~~~~~"
        CtrlVal(Frame_4) = "Charge~Percent/Mins~~~~~~~~"

      Case 3  ' running input and load watt hours
        GUI Page 1,2,4 '
        CtrlVal(Hint_1) = "Average Power Details"
        GUI bcolour RGB(0,120,120),Menu_2
        GUI bcolour RGB(cyan),Menu_1,Menu_3,Menu_4
        GUI BCOLOUR BLK,Whiteboard
        CtrlVal(Frame_1) = "DC Amps~AC Amps~~~~~~~~~"
        CtrlVal(Frame_2) = "Input Amps~Load Amps~Batt Amps~~~~~~~"
        CtrlVal(Frame_3) = "Input AHrs~Load AHrs~Batt AHrs~~~~~~~"
        CtrlVal(Frame_4) = "Input WHrs~Batt WHrs~Batt WHrs~~~~~~~"

      Case 4  ' watt hours over previous 4 days
        CtrlVal(Hint_1) = "3 Day Power History"
        GUI bcolour RGB(0,120,120),Menu_3
        GUI bcolour RGB(cyan),Menu_1,Menu_2,Menu_4
        GUI BCOLOUR BLK,Whiteboard
        CtrlVal(Frame_1) = "IWHrs 1 day~LWHrs 1 day~BWHrs 1 day~~~~~~~"
        CtrlVal(Frame_2) = "IWHrs 2 days~LWHrs 2 days~BWHrs 2 days~~~~~~~"
        CtrlVal(Frame_3) = "IWHrs 3 days~LWHrs 3 days~BWHrs 3 days~~~~~~~"
        CtrlVal(Frame_4) = "IWHrs Total~LWHrs Total~BWHrs Total~~~~~~~"
        GUI Page 1,2,4  '

      Case 5  ' 1 minute average values summary
        GUI Page 1,2,4  '
        CtrlVal(Hint_1) = "One Minute Averages"
        GUI bcolour RGB(0,120,120),Menu_4
        GUI bcolour RGB(cyan),Menu_1,Menu_2,Menu_3
        GUI BCOLOUR BLK,Whiteboard
        CtrlVal(Frame_1) = "Battery~Volts/Amps~~~~~~~~"
        CtrlVal(Frame_2) = "Solar~Volts/Amps~~~~~~~~"
        CtrlVal(Frame_3) = "Car~Volts/Amps~~~~~~~~"
        CtrlVal(Frame_4) = "Input Amps~Load Amps~~~~~~~~"

      Case 6  ' help display
        GUI Page 1,3
        CtrlVal(Help_Box) = Help$
        GUI bcolour RGB(cyan),Menu_1,Menu_2,Menu_3,Menu_4

      Case 7  ' large display battery volts and charge
        GUI Page 1,7,4 '
        GUI BCOLOUR BLK,Whiteboard
        CtrlVal(Hint_1) = " 2 second values "
        CtrlVal(LDB_Hdr_1) = "Battery Volts~11.8  12  12.4        14"
        CtrlVal(LDB_Hdr_2) = "Battery Charge~0%      40% 55%     100%"
        CtrlVal(BVBarGraph) = 0
        CtrlVal(BCBarGraph) = 0
        GUI bcolour RGB(cyan),Menu_1,Menu_2,Menu_3,Menu_4

      Case 8  ' hint box
        GUI Page 1,3
        GUI bcolour RGB(cyan),Menu_1,Menu_2,Menu_3,Menu_4

      Case 9  ' history graphs
        GUI Page 1,8,6
        GUI bcolour RGB(cyan),Menu_1,Menu_2,Menu_3,Menu_4
        Select Case GraphToggle
          Case 1
            BVGraph
          Case 2
            BAGraph
          Case 3
            BPGraph
          Case 4
            BCGraph
          Case 5
            SVGraph
          Case Else
        End Select
      CASE 10 ' big time display
        GUI Page 1,9
        GUI bcolour RGB(cyan),Menu_1,Menu_2,Menu_3,Menu_4

    End Select
  End Sub

  '*******************
  Sub UpdateGUIVals ' update GUI display values every 2 seconds
                    ' from TwoSecProcess subroutine or whenever
                    ' a touch is detected
      Select Case ActiveScreen
        Case 1
          CtrlVal(Hint_1) = ".... press < for large display"
          CtrlVal(Help_Box) = Help$

        Case 2 ' instantaneous values
          GUI FColour BLK,DB_Val_5
          GUI BColour RGB(140,150,255),DB_Val_5
          CtrlVal(DB_Val_5) = Str$(BVolts,2,2)
          GUI FColour BLK,DB_Val_6
          GUI BColour RGB(cyan),DB_Val_6
          CtrlVal(DB_Val_6) = Str$(SVolts,2,2)
          GUI FColour BLK,DB_Val_7
          GUI BColour RGB(255,200,200),DB_Val_7
         CtrlVal(DB_Val_7) = Str$(CVolts,2,2)
          GUI FColour BLK,DB_Val_8
          GUI BColour RGB(255,255,100),DB_Val_8
          CtrlVal(DB_Val_8) = Str$(BatCharge,3,0)+"%"
          If BAmps < 0 Then
            GUI FColour BLK,DB_Val_9
            GUI BColour RED,DB_Val_9
            CtrlVal(DB_Val_9) = Str$(BAmps,2,2)
          Else
           GUI FColour BLK,DB_Val_9
            GUI BColour RGB(green),DB_Val_9
            CtrlVal(DB_Val_9) = Str$(BAmps,2,2)
          EndIf
          GUI FColour BLK,DB_Val_10
          GUI BColour RGB(green),DB_Val_10
          CtrlVal(DB_Val_10) = Str$(SAmps,2,2)
          GUI FColour BLK,DB_Val_11
          GUI BColour RGB(green),DB_Val_11
          CtrlVal(DB_Val_11) = Str$(CAmps,2,2)
          GUI FColour BLK,DB_Val_12
          GUI BColour RGB(255,255,100),DB_Val_12
          CtrlVal(DB_Val_12) = Str$(Time2Discharge,3,0)

        Case 3 ' power details
          GUI FColour BLK,DB_Val_5
          GUI BColour RGB(GREEN),DB_Val_5
          CtrlVal(DB_Val_5) = Str$(AverageDCC,2,2)
          GUI FColour BLK,DB_Val_6
          GUI BColour RGB(green),DB_Val_6
          CtrlVal(DB_Val_6) = Str$(AverageIC,2,2)
          GUI FColour BLK,DB_Val_7
          GUI BColour RGB(green),DB_Val_7
          CtrlVal(DB_Val_7) = Str$(InputAH,3,0)
          GUI FColour BLK,DB_Val_8
          GUI BColour RGB(green),DB_Val_8
          CtrlVal(DB_Val_8) = Str$(InputWH,3,0)
          GUI FColour BLK,DB_Val_9
          GUI BColour RGB(green),DB_Val_9
          CtrlVal(DB_Val_9) = Str$(AverageACC,2,2)
          GUI FColour BLK,DB_Val_10
          GUI BColour RED,DB_Val_10
          CtrlVal(DB_Val_10) = Str$(AverageLC,2,2)
          GUI FColour BLK,DB_Val_11
          GUI BColour RED,DB_Val_11
          CtrlVal(DB_Val_11) = Str$(LoadAH,3,0)
          GUI FColour BLK,DB_Val_12
          GUI BColour RED,DB_Val_12
          CtrlVal(DB_Val_12) = Str$(LoadWH,3,0)
          GUI FColour BLK,DB_Val_9
          GUI BColour RGB(green),DB_Val_9

          If AverageBC < 0 Then
            GUI FColour BLK,DB_Val_14
            GUI BColour RED,DB_Val_14
            CtrlVal(DB_Val_14) = Str$(AverageBC,2,2)
          Else
            GUI FColour BLK,DB_Val_14
            GUI BColour RGB(green),DB_Val_14
            CtrlVal(DB_Val_14) = Str$(AverageBC,2,2)
          EndIf
          If BatAH < 0 Then
            GUI FColour BLK,DB_Val_15
            GUI BColour RED,DB_Val_15
            CtrlVal(DB_Val_15) = Str$(BatAH,3,0)
          Else
            GUI FColour BLK,DB_Val_15
            GUI BColour RGB(green),DB_Val_15
            CtrlVal(DB_Val_15) = Str$(BatAH,3,0)
          EndIf
          If BatWH < 0 Then
            GUI FColour BLK,DB_Val_16
            GUI BColour RED,DB_Val_16
            CtrlVal(DB_Val_16) = Str$(BatWH,3,0)
          Else
            GUI FColour BLK,DB_Val_16
            GUI BColour RGB(green),DB_Val_16
            CtrlVal(DB_Val_16) = Str$(BatWH,3,0)
          EndIf

        Case 4  ' power history
          GUI FColour BLK,DB_Val_5
          GUI BColour RGB(GREEN),DB_Val_5
          CtrlVal(DB_Val_5) = Str$(PrevIWH,4,0)
          GUI FColour BLK,DB_Val_6
          GUI BColour RGB(GREEN),DB_Val_6
          CtrlVal(DB_Val_6) = Str$(Day2IWH,4,0)
          GUI FColour BLK,DB_Val_7
          GUI BColour RGB(GREEN),DB_Val_7
          CtrlVal(DB_Val_7) = Str$(Day3IWH,4,0)
          GUI FColour BLK,DB_Val_8
          GUI BColour RGB(GREEN),DB_Val_8
          CtrlVal(DB_Val_8) = Str$(TotalIWH,4,0)
          GUI FColour BLK,DB_Val_9
          GUI BColour RED,DB_Val_9
          CtrlVal(DB_Val_9) = Str$(PrevLWH,4,0)
          GUI FColour BLK,DB_Val_10
          GUI BColour RED,DB_Val_10
          CtrlVal(DB_Val_10) = Str$(Day2LWH,4,0)
          GUI FColour BLK,DB_Val_11
          GUI BColour RED,DB_Val_11
          CtrlVal(DB_Val_11) = Str$(Day3LWH,4,0)
          GUI FColour BLK,DB_Val_12
          GUI BColour RED,DB_Val_12
          CtrlVal(DB_Val_12) = Str$(TotalLWH,4,0)
          If PrevBWH < 0 Then
            GUI FColour BLK,DB_Val_13
            GUI BColour RED,DB_Val_13
            CtrlVal(DB_Val_13) = Str$(PrevBWH,4,0)
          Else
            GUI FColour BLK,DB_Val_13
            GUI BColour RGB(green),DB_Val_13
            CtrlVal(DB_Val_13) = Str$(PrevBWH,4,0)
          EndIf
          If Day2BWH < 0 Then
            GUI FColour BLK,DB_Val_14
            GUI BColour RED,DB_Val_14
            CtrlVal(DB_Val_14) = Str$(Day2BWH,4,0)
          Else
            GUI FColour BLK,DB_Val_14
            GUI BColour RGB(green),DB_Val_14
            CtrlVal(DB_Val_14) = Str$(Day2BWH,4,0)
          EndIf
          If Day3BWH < 0 Then
            GUI FColour BLK,DB_Val_15
            GUI BColour RED,DB_Val_15
            CtrlVal(DB_Val_15) = Str$(Day3BWH,4,0)
          Else
            GUI FColour BLK,DB_Val_15
            GUI BColour RGB(green),DB_Val_15
            CtrlVal(DB_Val_15) = Str$(Day3BWH,4,0)
          EndIf
          If TotalBWH < 0 Then
            GUI FColour BLK,DB_Val_16
            GUI BColour RED,DB_Val_16
            CtrlVal(DB_Val_16) = Str$(TotalBWH,4,0)
          Else
            GUI FColour BLK,DB_Val_16
            GUI BColour RGB(green),DB_Val_16
            CtrlVal(DB_Val_16) = Str$(TotalBWH,4,0)
          EndIf

        Case 5  'minute average values
          GUI FColour BLK,DB_Val_5
          GUI BColour RGB(90,90,255),DB_Val_5
          CtrlVal(DB_Val_5) = Str$(AverageBV,2,2)
          GUI FColour BLK,DB_Val_6
          GUI BColour RGB(cyan),DB_Val_6
          CtrlVal(DB_Val_6) = Str$(AverageSV,2,2)
          GUI FColour BLK,DB_Val_7
          GUI BColour RGB(magenta),DB_Val_7
          CtrlVal(DB_Val_7) = Str$(AverageCV,2,2)
          GUI FColour BLK,DB_Val_8
          GUI BColour RGB(green),DB_Val_8
          CtrlVal(DB_Val_8) = Str$(AverageIC,2,2)
          If AverageBC < 0 Then
            GUI FColour BLK,DB_Val_9
            GUI BColour RED,DB_Val_9
            CtrlVal(DB_Val_9) = Str$(AverageBC,2,2)
          Else
            GUI FColour BLK,DB_Val_9
            GUI BColour RGB(green),DB_Val_9
            CtrlVal(DB_Val_9) = Str$(AverageBC,2,2)
          EndIf
          GUI FColour BLK,DB_Val_10
          GUI BColour RGB(green),DB_Val_10
          CtrlVal(DB_Val_10) = Str$(AverageSC,2,2)
          GUI FColour BLK,DB_Val_11
          GUI BColour RGB(green),DB_Val_11
          CtrlVal(DB_Val_11) = Str$(AverageCC,2,2)
          GUI FColour BLK,DB_Val_12
          GUI BColour RED,DB_Val_12
          CtrlVal(DB_Val_12) = Str$(AverageLC,2,2)

        Case 6  ' ? help box
          CtrlVal(Hint_1) = ".... press < for large display"
          CtrlVal(Help_Box) = Help$

        Case 7  ' large character Battery Volts
          CtrlVal(Hint_1) = "Battery Voltage and State of Charge"
          CtrlVal(LDB_Val_1) = Str$(BVolts,2,2)'not available in font 6 +"V"
          CtrlVal(LDB_Val_2) = Str$(BatCharge) 'not available in font 6 +"%"
          CtrlVal(BCBarGraph) = BatCharge
          CtrlVal(BVBarGraph) = BVolts

        Case 8  ' hint box
          CtrlVal(Hint_1) = "Be kind to your mother!"
          LinkError$ = "~~Bad data counter = "+Str$(bdcount)
          LinkError$ = LinkError$ + "~Link error counter = "+Str$(LinkDown)
          CtrlVal(Help_Box) = Help$ + LinkError$
          GUI bcolour RGB(cyan),Menu_1,Menu_2,Menu_3,Menu_4

        Case 9  ' history graphs
          CtrlVal(Hint_1) = "Last 3 hours history graphs"
          
        CASE 10 ' big time display
          CTRLVAL(BigTimeBox) = Time$
          
      End Select
  End Sub ' end of UpdateGUIVals subroutine

  '---------------------------------------------------
'
Sub BatChargeUpdate
  ' Calculate battery change as percentage and fill charge state graph
  ' now calculate charge and paint capacity graph
  ' 12.1V equates to 30% charge - note this is a rough calculation
  '  - it should be done under no load.
  ' *** work needed here ***
  '  - this code attemps to simulate a discharge curve for the purposes
  '    of calculating state of charge under load
  Local RemainBC,AverageBDChg
  Local range = 29       ' shape of discharge curve
  Local offset = 10.9   ' offset to position range of values
  Local curve = 6       ' shape determined by discharge current
  ' ... some rule of thumb values to position values in the
  ' correct range and with a discharge curve approximating a discharge
  ' rate of between 4 and 10 amps
  '  curve = 10-(AverageLC/2.1)
  BatCharge = (((BVolts-offset)^2)*range)+ curve
  ' now set up to display charge graphically
  BatCharge = Int(BatCharge)
  If BatCharge < 1 Then BatCharge = 1
  If BatCharge > 100 Then BatCharge = 100
  ' Calculate time to 30% discharge of battery
  RemainBC = BatCharge - 30 ' rough convert from 30% - 100%
  ' to 0Ahrs to 70AHrs remaining charge
  If RemainBC < 1 Then
    RemainBC = 1
  EndIf
  AverageBDChg = 10 ' assume an average discharge rate of 10AH
  Time2Discharge = (RemainBC/AverageBDChg)*60  'if we have 70 AHrs and we
  ' discharging at 10AHrs
  ' then we have 7 hours of usage till 30%
  ' - again these are only rough approximations
End Sub

  '------------------------------------------------------
  ' GUI Setup routine - only called once on startup but safer in a subroutine
Sub InitGUI  ' initialise graphics
  GUI setup 1 ' same for all pages
    GUI led Led_1,"",400,90,10,RED ' flash every 2 secs
    Font 4,2
    GUI displaybox Head_Box,0,0,800,50,WHT,RGB(25,100,215)
    Font 6
    GUI Displaybox Time_Box,0,53,800,53,BLK,GRN
    Font 4,2
    GUI Displaybox Back_Arrow,0,428,70,50,BLK,RGB(220,220,255)
    GUI Displaybox Help_Button,728,428,70,50,BLK,RGB(220,220,255)
    Font 4
    GUI Displaybox Menu_1,0,376,195,50,BLK,RGB(cyan)
    GUI Displaybox Menu_2,202,376,195,50,BLK,RGB(cyan)
    GUI Displaybox Menu_3,403,376,195,50,BLK,RGB(cyan)
    GUI Displaybox Menu_4,605,376,195,50,BLK,RGB(cyan)
    GUI displaybox Hint_1,75,428,650,50,BLK,WHT
    CtrlVal(Head_Box) = "Caravan System Monitor"
    CtrlVal(Menu_1) = "Battery~Details"
    CtrlVal(Menu_2) = "Power~Details"
    CtrlVal(Menu_3) = "Power~History"
    CtrlVal(Menu_4) = "Input~Averages"
    CtrlVal(Hint_1) = " .. waiting for minute rollover"
    ' areas for boxes above
    GUI area Back_Arrow_Touch,0,430,70,50 '
    GUI area Help_Button_Touch,730,430,70,50 '
    GUI area Hint_Touch,75,420,650,50 '
    GUI area Menu_1_Touch,0,376,195,50
    GUI area Menu_2_Touch,202,376,195,50
    GUI area Menu_3_Touch,403,376,195,50
    GUI area Menu_4_Touch,605,376,195,50
    GUI area BigTime_Touch,0,53,800,53

  GUI setup 2 ' 8 header boxes and 8 small display boxes
    Font 4
    GUI displaybox Frame_1,0,110,195,262,BLK,RGB(220,220,240)
    GUI displaybox Frame_2,202,110,195,262,BLK,RGB(220,220,240)
    GUI displaybox Frame_3,403,110,195,262,BLK,RGB(220,220,240)
    GUI displaybox Frame_4,605,110,195,262,BLK,RGB(220,220,240)
    Font 6
    GUI Displaybox DB_Val_5,0,190,190,60,BLK,RGB(gray)
    GUI Displaybox DB_Val_6,202,190,190,60,BLK,RGB(gray)
    GUI Displaybox DB_Val_7,403,190,190,60,BLK,RGB(gray)
    GUI Displaybox DB_Val_8,605,190,190,60,BLK,RGB(gray)
    GUI Displaybox DB_Val_9,0,252,190,60,BLK,RGB(gray)
    GUI Displaybox DB_Val_10,202,252,190,60,BLK,RGB(gray)
    GUI Displaybox DB_Val_11,403,252,190,60,BLK,RGB(gray)
    GUI Displaybox DB_Val_12,605,252,190,60,BLK,RGB(gray)
    GUI Displaybox DB_Val_13,0,314,190,56,BLK,RGB(gray)
    GUI Displaybox DB_Val_14,202,314,190,56,BLK,RGB(gray)
    GUI Displaybox DB_Val_15,403,314,190,56,BLK,RGB(gray)
    GUI Displaybox DB_Val_16,605,314,190,56,BLK,RGB(gray)

  GUI setup 3 ' help screen
    Font 2
    GUI Displaybox Help_Box,0,110,800,264,BLK,RGB(210,230,210)

  GUI setup 4 ' display area for all info
    GUI displaybox Whiteboard,0,110,800,264,BLK,BLK

  GUI setup 5 ' touch frame heading to go to historical graphs
    GUI area Frame_1_Touch,0,110,195,130

  GUI setup 6 ' Graph area backgrpond
    GUI displaybox Graph_BG,0,110,796,264,WHT,WHT

  GUI setup 7 ' large battery volts and battery charge
    Font 4
    GUI Displaybox LDB_Hdr_1,0,112,392,60,BLK,RGB(220,220,220)
    GUI Displaybox LDB_Hdr_2,403,112,392,60,BLK,RGB(220,220,220)
    Font 6,2
    GUI Displaybox LDB_Val_1,0,186,392,186,BLK,RGB(190,190,255)
    GUI Displaybox LDB_Val_2,403,186,392,186,BLK,MBLU
    GUI BARGAUGE BCBarGraph,410,174,380,10,BLK,WHT,0,100,RED,40,YEL,55,GRN,55,GRN
    GUI BARGAUGE BVBarGraph,12,174,380,10,BLK,WHT,11.3,13.8,RED,12,YEL,12.4,GRN,55,GRN

  GUI setup 8 ' area box to toggle graphs
    Font 2
    GUI displaybox Graph_Text,460,140,320,190,BLK,RGB(200,200,200)
    GUI area Graph_Touch,460,140,320,190 ' touch area of graph toggle

  GUI setup 9 ' large time display
    Font 6,2
    GUI Displaybox BigTime_Box,0,112,800,246,BLK,RGB(220,220,220)
  
  GUI Setup 10 ' dummy 
  ' end of GUI setups
End Sub
'-------------------------------------------------------------
' graphs that overlay GUI areas
' - called from the UpdateGUIScreen sub after a touch is detected

Sub BVGraph ' graph of battery voltage over last 3 hrs
  ' draw out running graph frame, ticks etc.
  Local volts,yrg,xrg,rg_abv,BV3HrPtr
  GUI Bcolour WHT,Graph_BG
  Colour BLK,RGB(220,220,255)
  GText$ = "~Battery voltage for~the last 3 hours."
  GText$ = GText$ +"~~TOUCH HERE TO~TOGGLE GRAPHS~~"
  Gtext$ = GText$ + "Press < to return to~battery main page."
  CtrlVal(Graph_Text) = GText$
  volts=15
  For yrg =130 To 330 Step 50
    Text 40,yrg-6,Str$(volts,2,0)+"V",LT,1,1,RGB(blue)
    volts=volts-1
  Next yrg
  Line 90,130,90,330,2,RGB(blue)
  For yrg = 130 To 330 Step 10 'minor ticks
    Pixel 88,yrg,BLK
    Pixel 89,yrg,BLK
  Next yrg
  For yrg = 130 To 330 Step 50  'major ticks
    Pixel 86,yrg,BLK
    Pixel 87,yrg,BLK
    Pixel 88,yrg,BLK
    Pixel 89,yrg,BLK
  Next yrg
  Line 90,330,450,330,2,RGB(blue) 'now for the bottom
  For xrg = 90 To 450 Step 30 ' 15 min ticks
    Pixel xrg,332,BLK
    Pixel xrg,333,BLK
  Next xrg
  For xrg = 90 To 450 Step 60 ' half hour ticks
    Pixel xrg,332,BLK
    Pixel xrg,333,BLK
    Pixel xrg,334,BLK
  Next xrg
  For xrg = 90 To 450 Step 120 ' hour ticks
    Pixel xrg,332,BLK
    Pixel xrg,333,BLK
    Pixel xrg,334,BLK
    Pixel xrg,335,BLK
  Next xrg
  Text 90,345," Battery voltage for the last 3 hours ",LT,1,1,RGB(blue)
  ' now fill graph
  BV3HrPtr = BV3HrPointer ' local array counter
  If BV3HrPtr < 0 Then ' if end of array, loop around
    BV3HrPtr = 179
  EndIf
  For xrg = 91 To 451 Step 2
    rg_abv = 330 - Int((BV3HrArray(BV3HrPtr)-11) * 50)
    If rg_abv > 330 Then
      rg_abv = 330
    EndIf
    If rg_abv  < 260 Then ' must be greater than 12.4V
      Line xrg,rg_abv,xrg,329,2,GRN
    ElseIf rg_abv > 260 And rg_abv < 270 Then ' between 12.4V and 12.2V
      Line xrg,rg_abv,xrg,329,2,YEL
    ElseIf rg_abv > 270 Then ' below 12.2V - DANGER
          Line xrg,rg_abv,xrg,329,2,RED
      If rg_abv => 328 Then ' 11V or less so no red
              Line 91,328,451,328,2,WHT
        Line 90,330,451,330,2,RGB(blue)
      EndIf
    EndIf
    BV3HrPtr = BV3HrPtr + 1
    If BV3HrPtr =180 Then
      BV3HrPtr = 0
    EndIf
  Next xrg
End Sub

Sub BAGraph ' graph of battery current over last 3 hrs
  ' draw out running graph frame, ticks etc.
  Local amps,yrg,xrg,rg_aba,BA3HrPtr
  GUI Bcolour WHT,Graph_BG
  Colour BLK,RGB(220,220,255)
  GText$ = "~Battery current for~the last 3 hours."
  GText$ = GText$ +"~~TOUCH HERE TO~TOGGLE GRAPHS~~"
  Gtext$ = GText$ + "Press < to return to~battery main page."
  CtrlVal(Graph_Text) = GText$
  amps=20
  For yrg =130 To 330 Step 25
    Text 40,yrg-6,Str$(amps,2,0)+"A",LT,1,1,RGB(blue)
    amps=amps-5
  Next yrg
  Line 90,130,90,330,2,RGB(blue)
  For yrg = 130 To 330 Step 5 'minor ticks
    Pixel 88,yrg,BLK
    Pixel 89,yrg,BLK
  Next yrg
  For yrg = 130 To 330 Step 10  'major ticks
    Pixel 87,yrg,BLK
    Pixel 88,yrg,BLK
    Pixel 89,yrg,BLK
  Next yrg
  Line 90,330,450,330,2,RGB(blue) 'now for the bottom
  For xrg = 90 To 450 Step 30 ' 15 min ticks
    Pixel xrg,331,BLK
    Pixel xrg,332,BLK
  Next xrg
  For xrg = 90 To 450 Step 60 ' half hour ticks
    Pixel xrg,331,BLK
    Pixel xrg,332,BLK
    Pixel xrg,333,BLK
  Next xrg
  For xrg = 90 To 450 Step 120 ' hour ticks
    Pixel xrg,331,BLK
    Pixel xrg,332,BLK
    Pixel xrg,333,BLK
    Pixel xrg,334,BLK
  Next xrg
  Line 90,230,450,230,1,RGB(blue) 'now the middle
  Text 90,345," Battery current for the last 3 hours ",LT,1,1,RGB(blue)
  ' now fill graph
  BA3HrPtr = BA3HrPointer
  If BA3HrPtr < 0 Then ' if end of array, loop around
    BA3HrPtr = 179
  EndIf
    ' paint out pixels into graph for running values
  For xrg = 91 To 451 Step 2
    rg_aba = 230 - Int((BA3HrArray(BA3HrPtr)) * 5)
    If rg_aba > 330 Then
      rg_aba = 330
    ElseIf rg_aba < 130 Then
      rg_aba = 130
    EndIf
    If rg_aba  < 230 Then
      Line xrg,rg_aba,xrg,230,2,GRN
    ElseIf rg_aba > 230 Then
      Line xrg,230,xrg,rg_aba,2,RED
    EndIf
    BA3HrPtr = BA3HrPtr + 1
    If BA3HrPtr =180 Then
      BA3HrPtr = 0
    EndIf
  Next xrg
End Sub

Sub BPGraph ' graph of power usage over last 3 hrs
  ' draw out running graph frame, ticks etc.
  Local power,yrg,xrg,rg_abp,BP3HrPtr
  GUI Bcolour WHT,Graph_BG
  Colour BLK,RGB(220,220,255)
  GText$ = "~Battery WHrs for~the last 3 hours."
  GText$ = GText$ +"~~TOUCH HERE TO~TOGGLE GRAPHS~~"
  Gtext$ = GText$ + "Press < to return to~battery main page."
  CtrlVal(Graph_Text) = GText$
  power=200
  For yrg =130 To 330 Step 25
    Text 12,yrg-6,Str$(power,3,0)+"WHrs",LT,1,1,RGB(blue)
    power=power-50
  Next yrg
  Line 90,130,90,330,2,RGB(blue)
  For yrg = 130 To 330 Step 5 'minor ticks
    Pixel 88,yrg,BLK
    Pixel 89,yrg,BLK
  Next yrg
  For yrg = 130 To 330 Step 25  'major ticks
    Pixel 87,yrg,BLK
    Pixel 88,yrg,BLK
    Pixel 89,yrg,BLK
  Next yrg
  Line 90,330,450,330,2,RGB(blue) 'now for the bottom
  For xrg = 90 To 450 Step 30 ' 15 min ticks
    Pixel xrg,331,BLK
    Pixel xrg,332,BLK
  Next xrg
  For xrg = 90 To 450 Step 60 ' half hour ticks
    Pixel xrg,331,BLK
    Pixel xrg,332,BLK
    Pixel xrg,333,BLK
  Next xrg
  For xrg = 90 To 450 Step 120 ' hour ticks
    Pixel xrg,331,BLK
    Pixel xrg,332,BLK
    Pixel xrg,333,BLK
    Pixel xrg,334,BLK
  Next xrg
  Line 90,230,450,230,1,RGB(blue) 'now the middle
  Text 90,345," Battery WHrs for the last 3 hours ",LT,1,1,RGB(blue)
  ' now fill graph
  BP3HrPtr = BP3HrPointer
  If BP3HrPtr < 0 Then ' if end of array, loop around
    BP3HrPtr = 179
  EndIf
    ' paint out pixels into graph for running values
  For xrg = 91 To 451 Step 2 ' step across x axis
    rg_abp = 230 - Int((BP3HrArray(BP3HrPtr)))
    If rg_abp > 330 Then
      rg_abp = 330
    ElseIf rg_abp < 130 Then
      rg_abp = 130
    EndIf
    If rg_abp  < 230 Then
      Line xrg,rg_abp,xrg,230,2,GRN
    ElseIf rg_abp > 230 Then
      Line xrg,230,xrg,rg_abp,2,RED
    EndIf
    BP3HrPtr = BP3HrPtr + 1
    If BP3HrPtr =180 Then
      BP3HrPtr = 0
    EndIf
  Next xrg
End Sub

Sub BCGraph ' graph of battery charge left over last 3 hrs
  ' draw out running graph frame, ticks etc.
  Local charge,yrg,xrg,rg_abc,BC3HrPtr
  GUI Bcolour WHT,Graph_BG
  Colour BLK,RGB(220,220,255)
  GText$ = "~Battery charge for~the last 3 hours."
  GText$ = GText$ +"~~TOUCH HERE TO~TOGGLE GRAPHS~~"
  Gtext$ = GText$ + "Press < to return to~battery main page."
  CtrlVal(Graph_Text) = GText$
  charge=100
  For yrg =130 To 330 Step 20
    Text 40,yrg-6,Str$(charge,2,0)+"%",LT,1,1,RGB(blue)
    charge=charge-10
  Next yrg
  Line 90,130,90,330,2,RGB(blue)
  For yrg = 130 To 330 Step 10 'minor ticks
    Pixel 88,yrg,BLK
    Pixel 89,yrg,BLK
  Next yrg
  For yrg = 130 To 330 Step 20  'major ticks
    Pixel 87,yrg,BLK
    Pixel 88,yrg,BLK
    Pixel 89,yrg,BLK
  Next yrg
  Line 90,330,450,330,2,RGB(blue) 'now for the bottom
  For xrg = 90 To 450 Step 30 ' 15 min ticks
    Pixel xrg,331,BLK
    Pixel xrg,332,BLK
  Next xrg
  For xrg = 90 To 450 Step 60 ' half hour ticks
    Pixel xrg,331,BLK
    Pixel xrg,332,BLK
    Pixel xrg,333,BLK
  Next xrg
  For xrg = 90 To 450 Step 120 ' hour ticks
    Pixel xrg,331,BLK
    Pixel xrg,332,BLK
    Pixel xrg,333,BLK
    Pixel xrg,334,BLK
  Next xrg
  Text 90,345," Battery state of charge for the last 3 hours ",LT,1,1,RGB(blue)
  ' now fill graph
  BC3HrPtr = BC3HrPointer
  If BC3HrPtr < 0 Then ' loop pointer around to start
    BC3HrPtr = 179
  EndIf
  For xrg = 91 To 451 Step 2
    rg_abc = 330 - Int((BC3HrArray(BC3HrPtr)) * 2)
    If rg_abc > 330 Then
      rg_abc = 330
    EndIf
    If rg_abc  < 220 Then
      Line xrg,rg_abc,xrg,329,2,GRN
    ElseIf rg_abc > 220 And rg_abc < 250 Then
      Line xrg,rg_abc,xrg,329,2,YEL
    ElseIf rg_abc > 250 Then
      Line xrg,rg_abc,xrg,329,2,RED
      If rg_abc => 328 Then ' 11V or less so no red
        Line 91,328,451,328,2,WHT
        Line 90,330,451,330,2,RGB(blue)
      EndIf
    EndIf
    BC3HrPtr = BC3HrPtr + 1
    If BC3HrPtr =180 Then
      BC3HrPtr = 0
    EndIf
  Next xrg
End Sub

Sub SVGraph ' graph of solar voltage over last 3 hrs
  ' draw out running graph frame, ticks etc.
  Local volts,yrg,xrg,rg_asv,SV3HrPtr
  GUI Bcolour WHT,Graph_BG
  Colour BLK,RGB(220,220,255)
  GText$ = "~Solar volts for~the last 3 hours."
  GText$ = GText$ +"~~TOUCH HERE TO~TOGGLE GRAPHS~~"
  Gtext$ = GText$ + "Press < to return to~battery main page."
  CtrlVal(Graph_Text) = GText$
  volts=25
  For yrg =130 To 330 Step 50
    Text 40,yrg-6,Str$(volts,2,0)+"V",LT,1,1,RGB(blue)
    volts=volts-5
  Next yrg
  Line 90,130,90,330,2,RGB(blue)
  For yrg = 130 To 330 Step 10 'minor ticks
    Pixel 88,yrg,BLK
    Pixel 89,yrg,BLK
  Next yrg
    For yrg = 130 To 330 Step 50  'major ticks
    Pixel 86,yrg,BLK
    Pixel 87,yrg,BLK
    Pixel 88,yrg,BLK
    Pixel 89,yrg,BLK
  Next yrg
  Line 90,330,450,330,2,RGB(blue) 'now for the bottom
  For xrg = 90 To 450 Step 30 ' 15 min ticks
    Pixel xrg,332,BLK
    Pixel xrg,333,BLK
  Next xrg
  For xrg = 90 To 450 Step 60 ' half hour ticks
    Pixel xrg,332,BLK
    Pixel xrg,333,BLK
    Pixel xrg,334,BLK
  Next xrg
  For xrg = 90 To 450 Step 120 ' hour ticks
    Pixel xrg,332,BLK
    Pixel xrg,333,BLK
    Pixel xrg,334,BLK
    Pixel xrg,335,BLK
  Next xrg
  Text 90,345," Solar voltage for the last 3 hours ",LT,1,1,RGB(blue)
  ' now fill graph
  SV3HrPtr = SV3HrPointer ' local array counter
  If SV3HrPtr < 0 Then ' if end of array, loop around
    SV3HrPtr = 179
  EndIf
  For xrg = 91 To 451 Step 2
    ' y range is 200 pixels so scale 20V, less than 5V is no solar
    rg_asv = 330 - Int((SV3HrArray(SV3HrPtr)-5) * 10)
    If rg_asv > 330 Then
      rg_asv = 330
    EndIf
    Line xrg,rg_asv,xrg,329,2,RGB(magenta)
    If rg_asv => 328 Then ' 11V or less so no red
      Line 91,328,451,328,2,WHT
      Line 90,330,451,330,2,RGB(blue)
    EndIf
    SV3HrPtr = SV3HrPtr + 1
    If SV3HrPtr =180 Then
      SV3HrPtr = 0
    EndIf
  Next xrg
End Sub

'-----------------------------------------------------
' Geoff's Message Field$ Routine (with one amendment
'   by CaptainBoing)
'-----------------------------------------------------


Sub BatWarn1
  If S_Bright <> 0 Then
    Pin(Buzzer) = 1
    Pause 400
    Pin(Buzzer) = 0
  EndIf
End Sub

Sub Toggle Arg1
  If Arg1 = 0 Then
    Arg1 = 1
  Else
    Arg1 = 0
  EndIf
End Sub
' End program.
''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
> 
