' Transpiled on 31-05-2021 13:55:00

' Copyright (c) 2021 Thomas Hugo Williams
' License MIT <https://opensource.org/licenses/MIT>
' For Colour Maximite 2, MMBasic 5.07
'
' MD5 algorithm from https://en.wikipedia.org/wiki/MD5
' XXTEA algorithm from https://en.wikipedia.org/wiki/XXTEA

Option Explicit On
Option Default None
Option Base 0

' BEGIN:     #Include "../splib/system.inc" ------------------------------------
' Copyright (c) 2020 Thomas Hugo Williams
' For Colour Maximite 2, MMBasic 5.06

Const sys.VERSION$ = "r1b3"
Const sys.NO_DATA$ = Chr$(&h7F)
Const sys.MAX_INCLUDES% = 20
Const sys.CRLF$ = Chr$(13) + Chr$(10)

Dim sys.err$
Dim sys.includes$(sys.MAX_INCLUDES%) Length 20

Sub sys.provides(f$)
  Local f_$ = LCase$(f$)
  Local i% = 1
  Do
    Select Case sys.includes$(i%)
      Case f_$ : sys.err$ = "file already included: " + f_$ + ".inc" : Exit Sub
      Case ""  : sys.includes$(i%) = f_$ : Exit Sub
    End Select
    Inc i%
    If i% > sys.MAX_INCLUDES% Then sys.err$ = "too many includes" : Exit Sub
  Loop
End Sub

Sub sys.requires(f1$, f2$, f3$, f4$, f5$, f6$, f7$, f8$, f9$, f10$)
  Local f$(10) Length 20
  f$(1) = f1$ : f$(2) = f2$ : f$(3) = f3$ : f$(4) = f4$ : f$(5) = f5$
  f$(6) = f6$ : f$(7) = f7$ : f$(8) = f8$ : f$(9) = f9$ : f$(10) = f10$

  Local i%, j%, ok%, fail%
  For i% = 1 To 10 : f$(i%) = LCase$(f$(i%)) : Next
  For i% = 1 To 10
    If f$(i%) <> "" Then
      ok% = 0
      For j% = 1 To sys.MAX_INCLUDES%
        Select Case sys.includes$(j%)
          Case f$(i%) : ok% = 1 : Exit For
          Case ""     : Exit For
        End Select
      Next j%
      If Not ok% Then
        If Not fail% Then
          sys.err$ = "required file(s) not included: " + f$(i%) + ".inc"
          fail% = 1
        Else
          Cat sys.err$, ", " + f$(i%) + ".inc"
        EndIf
      EndIf
    EndIf
  Next i%
End Sub

' Formats a firmware version as a 5-digit number, e.g.
'   5.05.06 => 50506
'   5.04    => 50400
'
' @param version$  the firmware version to format.
'                  If empty then formats the current firmware version number.
Function sys.firmware_version%(version$)
  Local i%, s$, v$ = version$
  If v$ = "" Then v$ = Str$(Mm.Info$(Version))
  For i% = 1 To Len(v$)
    If InStr("0123456789", Mid$(v$, i%, 1)) > 0 Then s$ = s$ + Mid$(v$, i%, 1)
  Next
  Do While Len(s$) < 5 : s$ = s$ + "0" : Loop
  sys.firmware_version% = Val(s$)
End Function

' Generates a pseudo random integer between 1 and 'range%' inclusive.
'
' @param  range%  if > 0 then upper bound of generated number,
'                 if = 0 then reinitialises seed based on Timer value,
'                 if < 0 then sets seed to Abs(range%)
Function sys.pseudo%(range%)
  Static x% = Timer ' 7
  Static a% = 1103515245
  Static c% = 12345
  Static m% = 2^31

  If range% = 0 Then
    x% = Timer
  ElseIf range% < 0 Then
    x% = Abs(range%)
  Else
    x% = (a% * x% + c%) Mod m%
    sys.pseudo% = 1 + CInt((range% - 1) * (x% / m%))
  EndIf
End Function
' END:       #Include "../splib/system.inc" ------------------------------------
' BEGIN:     #Include "../splib/array.inc" -------------------------------------
' Copyright (c) 2020-2021 Thomas Hugo Williams
' For Colour Maximite 2, MMBasic 5.06

On Error Skip 1 : Dim sys.VERSION$ = ""
If sys.VERSION$ = "" Then Error "'system.inc' not included"
sys.provides("array")
If sys.err$ <> "" Then Error sys.err$

' Gets the upper-bound that should be used to dimension an array of the given
' capacity, irrespective of OPTION BASE.
'
' e.g. To create a string array that can hold 10 elements:
'        Dim my_array$(array.new%(10))
Function array.new%(capacity%)
  array.new% = capacity% + Mm.Info(Option Base) - 1
End Function

' Binary search for a value in a SORTED array.
'
' @param  a$()    the array.
' @param  s$      the value to search for.
' @param  flags$  "i" to search case-insensitively,
' @param  lb%     the lower bound to search from,
'                 if 0/unset then search from the first element.
' @param  num%    the number of elements to search,
'                 if 0/unset then search all the elements (from lb%).
' @return         the index of the element containing the value,
'                 or -1 if not present.
Function array.bsearch%(a$(), s$, flags$, lb_%, num_%)
  Local lb% = Choice(lb_% = 0, Bound(a$(), 0), lb_%)
  Local num% = Choice(num_% = 0, Bound(a$(), 1) - Bound(a$(), 0) + 1, num_%)
  Local ub% = lb% + num% - 1
  Local i%

  If InStr(UCase$(flags$), "I") Then
    Local us$ = UCase$(s$)
    Local ua$
    Do While ub% >= lb%
      i% = (lb% + ub%) \ 2
      ua$ = UCase$(a$(i%))
      If us$ > ua$ Then
        lb% = i% + 1
      ElseIf us$ < ua$ Then
        ub% = i% - 1
      Else
        Exit Do
      EndIf
    Loop
  Else
    Do While ub% >= lb%
      i% = (lb% + ub%) \ 2
      If s$ > a$(i%) Then
        lb% = i% + 1
      ElseIf s$ < a$(i%) Then
        ub% = i% - 1
      Else
        Exit Do
      EndIf
    Loop
  EndIf

  If ub% >= lb% Then array.bsearch% = i% Else array.bsearch% = -1
End Function

' Gets the capacity (number of elements) that string array a$() can hold.
Function array.capacity%(a$())
  array.capacity% = Bound(a$(), 1) - Bound(a$(), 0) + 1
End Function

' Copies a string array.
'
' @param  src$()    the source array.
' @param  dst$()    the destination array.
' @param  src_idx%  the start index in the source,
'                   if 0/unset then use the index of the first element.
' @param  dst_idx%  the start index in the destination,
'                   if 0/unset then use the index of the first element.
' @param  num%      the number of elements to copy,
'                   if 0/unset then copy all the elements (from idx%) from the source.
Sub array.copy(src$(), dst$(), src_idx%, dst_idx%, num%)
  Local base% = Mm.Info(Option Base 0)
  Local i%
  Local j% = Max(base%, dst_idx%)
  Local lb% = Max(base%, src_idx%)
  Local ub% = src_idx% + num% - 1
  If num% = 0 Then ub% = Bound(src$(), 1)

  ' TODO: Use a memory copy instead of a loop.
  For i% = lb% To ub% : dst$(j%) = src$(i%) : Inc j% : Next
End Sub

' Fills all the elements of string array a$() to x$.
Sub array.fill(a$(), x$)
  Local lb% = Bound(a$(), 0)
  Local ub% = Bound(a$(), 1)
  Local i%
  For i% = lb% To ub% : a$(i%) = x$ : Next
End Sub

' Returns a string consisting of the concatenated elements of a float array
' joined together with a delimiter.
'
' @param   a!()    the array.
' @param   delim$  delimiter to place between each element, if empty/unset then uses comma.
' @param   lb%     lower bound to start from, if 0/unset then the 1st element.
' @param   num%    number of elements to join, if 0/unset then all elements.
' @param   slen%   maximum length of string to return, if 0/unset then 255 chars.
' @return          a string composed of the array elements separated by the delimiter. If the
'                  string had to be truncated to slen% then it is terminated with an ellipsis "..."
Function array.join_floats$(a!(), delim$, lb%, num%, slen%)
  Local delim_$ = Choice(delim$ = "", ",", delim$)
  Local lb_% = Choice(lb% = 0, Mm.Info(Option Base), lb%)
  Local ub_% = Choice(num% = 0, Bound(a!(), 1), lb_% + num% - 1)
  Local slen_% = Choice(slen% = 0, 255, slen%)

  Local s$ = Str$(a!(lb_%))
  Inc lb_%

  Do While lb_% <= ub_%
    Cat s$, Left$(delim_$, 255 - Len(s$))
    Cat s$, Left$(Str$(a!(lb_%)), 255 - Len(s$))
    Inc lb_%
  Loop

  If Len(s$) <= slen_% Then
    array.join_floats$ = s$
  Else
    array.join_floats$ = Left$(s$, slen_% - 3) + "..."
  EndIf
End Function

' Returns a string consisting of the concatenated elements of an integer array
' joined together with a delimiter.
'
' @param   a%()    the array.
' @param   delim$  delimiter to place between each element, if empty/unset then uses comma.
' @param   lb%     lower bound to start from, if 0/unset then the 1st element.
' @param   num%    number of elements to join, if 0/unset then all elements.
' @param   slen%   maximum length of string to return, if 0/unset then 255 chars.
' @return          a string composed of the array elements separated by the delimiter. If the
'                  string had to be truncated to slen% then it is terminated with an ellipsis "..."
Function array.join_ints$(a%(), delim$, lb%, num%, slen%)
  Local delim_$ = Choice(delim$ = "", ",", delim$)
  Local lb_% = Choice(lb% = 0, Mm.Info(Option Base), lb%)
  Local ub_% = Choice(num% = 0, Bound(a%(), 1), lb_% + num% - 1)
  Local slen_% = Choice(slen% = 0, 255, slen%)

  Local s$ = Str$(a%(lb_%))
  Inc lb_%

  Do While lb_% <= ub_%
    Cat s$, Left$(delim_$, 255 - Len(s$))
    Cat s$, Left$(Str$(a%(lb_%)), 255 - Len(s$))
    Inc lb_%
  Loop

  If Len(s$) <= slen_% Then
    array.join_ints$ = s$
  Else
    array.join_ints$ = Left$(s$, slen_% - 3) + "..."
  EndIf
End Function

' Returns a string consisting of the concatenated elements of a string array
' joined together with a delimiter.
'
' @param   a$()    the array.
' @param   delim$  delimiter to place between each element, if empty/unset then uses comma.
' @param   lb%     lower bound to start from, if 0/unset then the 1st element.
' @param   num%    number of elements to join, if 0/unset then all elements.
' @param   slen%   maximum length of string to return, if 0/unset then 255 chars.
' @return          a string composed of the array elements separated by the delimiter. If the
'                  string had to be truncated to slen% then it is terminated with an ellipsis "..."
Function array.join_strings$(a$(), delim$, lb%, num%, slen%)
  Local delim_$ = Choice(delim$ = "", ",", delim$)
  Local lb_% = Choice(lb% = 0, Mm.Info(Option Base), lb%)
  Local ub_% = Choice(num% = 0, Bound(a$(), 1), lb_% + num% - 1)
  Local slen_% = Choice(slen% = 0, 255, slen%)

  Local s$ = a$(lb_%)
  Inc lb_%

  Do While lb_% <= ub_%
    Cat s$, Left$(delim_$, 255 - Len(s$))
    Cat s$, Left$(a$(lb_%), 255 - Len(s$))
    Inc lb_%
  Loop

  If Len(s$) <= slen_% Then
    array.join_strings$ = s$
  Else
    array.join_strings$ = Left$(s$, slen_% - 3) + "..."
  EndIf
End Function
' END:       #Include "../splib/array.inc" -------------------------------------
' BEGIN:     #Include "../splib/crypt.inc" -------------------------------------
' Copyright (c) 2021 Thomas Hugo Williams
' License MIT <https://opensource.org/licenses/MIT>
' For Colour Maximite 2, MMBasic 5.07

' MD5 algorithm from https://en.wikipedia.org/wiki/MD5
' XXTEA algorithm from https://en.wikipedia.org/wiki/XXTEA

On Error Skip 1 : Dim sys.VERSION$ = ""
If sys.VERSION$ = "" Then Error "'system.inc' not included"
sys.requires("array")
sys.provides("crypt")
If sys.err$ <> "" Then Error sys.err$

' Calculates the MD5 hash of bytes held in a buffer.
'
' To calculate the MD5 hash of a string s$:
'   ok% = crypt.md5%(Peek(VarAddr s$) + 1, Len(s$), out%())
'
' To calculate the MD5 hash of a long string ls%():
'   ok% = crypt.md5%(Peek(VarAddr ls%()) + 8, LLen(ls%()), out%())
'
' @param  ad%     address of the buffer.
' @param  size%   number of bytes to process.
' @param  out%()  on return the MD5 hash in little-endian format (2 elements = 16 bytes = 128-bits)
' @return         1 on success, 0 on failure, see sys.err$ for details of failure.
Function crypt.md5%(ad%, size%, out%())

  If Bound(out%(), 1) - Bound(out%(), 0) <> 1 Then Error "out%() should have 2 elements"

  Local block%(array.new%(64))
  Const b_addr% = Peek(VarAddr block%())

  ' Storage for hash, each element treated as 32-bit unsigned integer.
  Local h%(array.new%(4))

  ' Storage for constants, each element treated as 32-bit unsigned integer.
  Local r%(array.new%(64))
  Local k%(array.new%(64))

  Local cur% = 0, num%

  Do
    ' Process blocks of 64 bytes / 512 bits at a time.
    num% = Min(size% - cur%, 64)
    Memory Set b_addr%, 0, 64
    Memory Copy ad% + cur%, b_addr%, num%

    If num% < 64 Then
      ' Append '1' bit to block%, the remainder will already be padded with zeros.
      Poke Byte b_addr% + num%, &h80

      If num% > 55 Then
        crypt.md5_block(block%(), h%(), r%(), k%())
        Memory Set b_addr%, 0, 64
      EndIf

      ' Last 64-bits of block% should be data length in bits as a 64-bit little-endian integer.
      Poke Integer b_addr% + 56, size% * 8
    EndIf

    crypt.md5_block(block%(), h%(), r%(), k%())

    Inc cur%, 64
  Loop While num% = 64

  Const BASE% = Mm.Info(Option Base)
  out%(BASE%)     = h%(BASE%)     Or h%(BASE% + 1) << 32
  out%(BASE% + 1) = h%(BASE% + 2) Or h%(BASE% + 3) << 32
  crypt.md5% = 1

End Function

' Formats 128-bit MD5 hash as a string.
'
' @param  md5%()  the MD5 hash in little-endian format (2 elements = 16 bytes = 128-bits)
' @return         the formatted MD5 hash.
Function crypt.md5_fmt$(md5%())
  If Bound(md5%(), 1) - Bound(md5%(), 0) <> 1 Then Error "md5%() should have 2 elements"
  Local i%, s$
  Const md5_addr% = Peek(VarAddr md5%())
  For i% = 0 To 15 : Cat s$, Hex$(Peek(Byte md5_addr% + i%), 2) : Next
  crypt.md5_fmt$ = LCase$(s$)
End Function

' Calculates the MD5 hash of a file.
'
' @param  fd%     file-descriptor for a file Open For Input.
' @param  out%()  on return the MD5 hash in little-endian format (2 elements = 16 bytes = 128-bits)
' @return         1 on success, 0 on failure, see sys.err$ for details of failure.
Function crypt.md5_file%(fd%, out%())

  If Bound(out%(), 1) - Bound(out%(), 0) <> 1 Then Error "out%() should have 2 elements"

  Local block%(array.new%(64))
  Const b_addr% = Peek(VarAddr block%())

  ' Storage for hash, each element treated as 32-bit unsigned integer.
  Local h%(array.new%(4))

  ' Storage for constants, each element treated as 32-bit unsigned integer.
  Local r%(array.new%(64))
  Local k%(array.new%(64))

  Local s$, num%, size%

  Do
    s$ = Input$(64, #fd%)
    num% = Len(s$)
    Inc size%, num%
    Memory Set b_addr%, 0, 64
    Memory Copy Peek(VarAddr s$) + 1, b_addr%, num%

    If num% < 64 Then
      ' Append '1' bit to block%, the remainder will already be padded with zeros.
      Poke Byte b_addr% + num%, &h80

      If num% > 55 Then
        crypt.md5_block(block%(), h%(), r%(), k%())
        Memory Set b_addr%, 0, 64
      EndIf

      ' Last 64-bits of block% should be file length in bits as a 64-bit little-endian integer.
      Poke Integer b_addr% + 56, size% * 8
    EndIf

    crypt.md5_block(block%(), h%(), r%(), k%())
  Loop While num% = 64

  Const BASE% = Mm.Info(Option Base)
  out%(BASE%)     = h%(BASE%)     Or h%(BASE% + 1) << 32
  out%(BASE% + 1) = h%(BASE% + 2) Or h%(BASE% + 3) << 32
  crypt.md5_file% = 1

End Function

' Process 512-bit block in MD5 hash calculation.
'
' @param  block%()  512-bit block to process (64 elements).
' @param  h%()      hash-values (4 elements).
' @param  r%()      per-round shift amounts (64 elements).
' @param  k%()      binary integer part of the sines of integers (Radians) as constants (64 elements).
Sub crypt.md5_block(block%(), h%(), r%(), k%())
  Const BASE% = Mm.Info(Option Base)

  ' Note that all variables should be treated as unsigned 32-bit integers
  ' and wrap modulo 2^32 when calculating.

  ' Initialise hash and constants if this is the first call.
  If r%(BASE%) = 0 Then
    Local data_addr% = Peek(CFunAddr crypt.md5_data)
    Memory Copy data_addr%,       Peek(VarAddr h%()), 64
    Memory Copy data_addr% + 32,  Peek(VarAddr r%()), 512
    Memory Copy data_addr% + 544, Peek(VarAddr k%()), 512
  EndIf

  ' Split block into 16 x 32-bit words.
  Local b_addr% = Peek(VarAddr block%())
  Local i%, w%(array.new%(16))
  For i% = 0 To 15
     w%(i% + BASE%) = Peek(Word b_addr% + 4 * i%)
  Next

  ' Main loop.
  Local a% = h%(BASE%), b% = h%(BASE% + 1), c% = h%(BASE% + 2), d% = h%(BASE% + 3), f%, g%, tmp%
  For i% = 0 To 63
    Select Case i%
      Case 0 To 15
        f% = d% Xor (b% And (c% Xor d%))
        g% = i%
      Case 16 To 31
        f% = c% Xor (d% And (b% Xor c%))
        g% = (5 * i% + 1) Mod 16
      Case 32 To 47
        f% = b% Xor c% Xor d%
        g% = (3 * i% + 5) Mod 16
      Case Else:
        f% = c% Xor (b% Or (d% Xor &hFFFFFFFF))
        g% = (7 * i%) Mod 16
    End Select

    tmp% = d%
    d% = c%
    c% = b%
    b% = (a% + f%) And &hFFFFFFFF
    b% = (b% + k%(i% + BASE%)) And &hFFFFFFFF
    b% = (b% + w%(g% + BASE%)) And &hFFFFFFFF
    b% = (b% << r%(i% + BASE%)) Or (b% >> (32 - r%(i% + BASE%)))
    b% = (b% + c%) And &hFFFFFFFF
    a% = tmp%

  Next

  ' Add this block's hash to the result.
  h%(BASE%)     = (h%(BASE%)     + a%) And &hFFFFFFFF
  h%(BASE% + 1) = (h%(BASE% + 1) + b%) And &hFFFFFFFF
  h%(BASE% + 2) = (h%(BASE% + 2) + c%) And &hFFFFFFFF
  h%(BASE% + 3) = (h%(BASE% + 3) + d%) And &hFFFFFFFF

End Sub

' Not a real CSUB, this provides data for initialising MD5 constants.
CSub crypt.md5_data()
  00000000

  ' 4 x 8 byte values to initialise hash 'h'
  67452301 00000000 EFCDAB89 00000000 98BADCFE 00000000 10325476 00000000

  ' 64 x 8 byte values for 'r'
  00000007 00000000 0000000C 00000000 00000011 00000000 00000016 00000000
  00000007 00000000 0000000C 00000000 00000011 00000000 00000016 00000000
  00000007 00000000 0000000C 00000000 00000011 00000000 00000016 00000000
  00000007 00000000 0000000C 00000000 00000011 00000000 00000016 00000000
  00000005 00000000 00000009 00000000 0000000E 00000000 00000014 00000000
  00000005 00000000 00000009 00000000 0000000E 00000000 00000014 00000000
  00000005 00000000 00000009 00000000 0000000E 00000000 00000014 00000000
  00000005 00000000 00000009 00000000 0000000E 00000000 00000014 00000000
  00000004 00000000 0000000B 00000000 00000010 00000000 00000017 00000000
  00000004 00000000 0000000B 00000000 00000010 00000000 00000017 00000000
  00000004 00000000 0000000B 00000000 00000010 00000000 00000017 00000000
  00000004 00000000 0000000B 00000000 00000010 00000000 00000017 00000000
  00000006 00000000 0000000A 00000000 0000000F 00000000 00000015 00000000
  00000006 00000000 0000000A 00000000 0000000F 00000000 00000015 00000000
  00000006 00000000 0000000A 00000000 0000000F 00000000 00000015 00000000
  00000006 00000000 0000000A 00000000 0000000F 00000000 00000015 00000000

  ' 64 x 8 byte values for 'k'
  D76AA478 00000000 E8C7B756 00000000 242070DB 00000000 C1BDCEEE 00000000
  F57C0FAF 00000000 4787C62A 00000000 A8304613 00000000 FD469501 00000000
  698098D8 00000000 8B44F7AF 00000000 FFFF5BB1 00000000 895CD7BE 00000000
  6B901122 00000000 FD987193 00000000 A679438E 00000000 49B40821 00000000
  F61E2562 00000000 C040B340 00000000 265E5A51 00000000 E9B6C7AA 00000000
  D62F105D 00000000 02441453 00000000 D8A1E681 00000000 E7D3FBC8 00000000
  21E1CDE6 00000000 C33707D6 00000000 F4D50D87 00000000 455A14ED 00000000
  A9E3E905 00000000 FCEFA3F8 00000000 676F02D9 00000000 8D2A4C8A 00000000
  FFFA3942 00000000 8771F681 00000000 6D9D6122 00000000 FDE5380C 00000000
  A4BEEA44 00000000 4BDECFA9 00000000 F6BB4B60 00000000 BEBFBC70 00000000
  289B7EC6 00000000 EAA127FA 00000000 D4EF3085 00000000 04881D05 00000000
  D9D4D039 00000000 E6DB99E5 00000000 1FA27CF8 00000000 C4AC5665 00000000
  F4292244 00000000 432AFF97 00000000 AB9423A7 00000000 FC93A039 00000000
  655B59C3 00000000 8F0CCC92 00000000 FFEFF47D 00000000 85845DD1 00000000
  6FA87E4F 00000000 FE2CE6E0 00000000 A3014314 00000000 4E0811A1 00000000
  F7537E82 00000000 BD3AF235 00000000 2AD7D2BB 00000000 EB86D391 00000000
End CSub

' Encrypts/decrypts file using XXTEA/CBC/PKCS#7 algorithm.
'
' @param  cmd$       command to perform, either "encrypt" or "decrypt".
' @param  in_fd%     reads input from this file descriptor.
' @param  out_fd%    writes output to this file descriptor.
' @param  k%()       encryption key (2 elements = 16 bytes = 128-bits).
' @param  iv%()      initialisation vector (2 elements = 16 bytes = 128-bits),
'                    this is ignored when cmd$ = "decrypt".
' @return            1 on success, 0 on failure, see sys.err$ for details of failure.
Function crypt.xxtea_file%(cmd$, in_fd%, out_fd%, k%(), iv%())

  If Bound(k%(), 1) - Bound(k%(), 0) <> 1 Then Error "k%() should have 2 elements"
  If Bound(iv%(), 1) - Bound(iv%(), 0) <> 1 Then Error "iv%() should have 2 elements"

  Local s$, num%, ok%
  Local block%(array.new%(2)) ' 16-bytes = 128-bits
  Const s_addr% = Peek(VarAddr s$)
  Const b_addr% = Peek(VarAddr block%())
  Local iv2%(array.new%(2)) ' 16-bytes = 128-bits
  Const iv2_addr% = Peek(VarAddr iv2%())

  If cmd$ = "encrypt" Then

    ' Write 128-bit initialisation vector.
    Poke Byte s_addr%, 16
    Memory Copy Peek(VarAddr iv%()), s_addr% + 1, 16
    Print #out_fd%, s$;

    ' iv2%() = iv%()
    Memory Copy Peek(VarAddr iv%()), iv2_addr%, 16

    Do

      ' Read 128-bit block.
      s$ = Input$(16, #in_fd%)
      num% = Len(s$)
      If num% > 0 Then Memory Copy s_addr% + 1, b_addr%, num%

      ' If we read a block of N bytes where N < 16 bytes long then we append PKCS#7
      ' padding of 16 - N bytes each of value 16 - N. In the event the input file
      ' size is an exact multiple of 16 then we append 16 bytes of value 16.
      If num% < 16 Then Memory Set b_addr% + num%, 16 - num%, 16 - num%

      ' Implement CBC by XORing iv2%() with block%().
      Poke Integer b_addr%, Peek(Integer b_addr%) Xor Peek(Integer iv2_addr%)
      Poke Integer b_addr% + 8, Peek(Integer b_addr% + 8) Xor Peek(Integer iv2_addr% + 8)

      ' Encrypt block.
      If Not crypt.xxtea_block%(cmd$, block%(), k%()) Then Exit Do

      ' Write 128-bit block.
      Poke Byte s_addr%, 16
      Memory Copy b_addr%, s_addr% + 1, 16
      Print #out_fd%, s$;

      ' iv2%() = block%()
      Memory Copy b_addr%, iv2_addr%, 16

    Loop Until num% < 16

  ElseIf cmd$ = "decrypt" Then

    Local iv3%(array.new%(2))
    Local iv3_addr% = Peek(VarAddr iv3%())

    ' Read 128-bit initialisation vector.
    s$ = Input$(16, #in_fd%)
    If Len(s$) < 16 Then
      sys.err$ = "encrypted file too short: " + Str$(Len(s$)) + " bytes"
      Exit Function
    EndIf
    Memory Copy s_addr% + 1, iv2_addr%, 16

    Do

      ' Read 128-bit block.
      s$ = Input$(16, #in_fd%)
      If Len(s$) < 16 Then
        sys.err$ = "encrypted file length not multiple of 16 bytes"
        Exit Function
      EndIf
      Memory Copy s_addr% + 1, b_addr%, 16

      ' iv3%() = block%()
      Memory Copy b_addr%, iv3_addr%, 16

      ' Decrypt block.
      If Not crypt.xxtea_block%(cmd$, block%(), k%()) Then Exit Function

      ' Implement CBC by XORing iv2%() with block%().
      Poke Integer b_addr%, Peek(Integer b_addr%) Xor Peek(Integer iv2_addr%)
      Poke Integer b_addr% + 8, Peek(Integer b_addr% + 8) Xor Peek(Integer iv2_addr% + 8)

      ' iv3%() = i2%()
      Memory Copy iv3_addr%, iv2_addr%, 16

      ' If we've reached the EOF then we do not want to write the entire
      ' block, the last byte in the block will indicate the number of bytes
      ' we should ignore, which may be the entire block.
      num% = Choice(Eof(#in_fd%), 16 - Peek(Byte b_addr% + 15), 16)

      ' Write block.
      Poke Byte s_addr%, num%
      If num% > 0 Then Memory Copy b_addr%, s_addr% + 1, num%
      Print #out_fd%, s$;

    Loop Until num% < 16

  Else

    Error "Unknown encryption command: " + cmd$

  EndIf

  crypt.xxtea_file% = 1

End Function

' Encrypts/decrypts block using XXTEA algorithm.
'
' @param  cmd$      command to perform, either "encrypt" or "decrypt".
' @param  block%()  data to encrypted, interpreted as 32-bit words.
' @param  k%()      encryption key (2 elements = 16 bytes = 128-bits).
' @return           1 on success, 0 on failure, see sys.err$ for details of failure.
Function crypt.xxtea_block%(cmd$, block%(), k%())
  Const DELTA% = &h9e3779b9 ' 4-bytes
  Const b_addr% = Peek(VarAddr block%())
  Const k_addr% = Peek(VarAddr k%())
  Const N% = 2 * (Bound(block%(), 1) - Bound(block%(), 0) + 1)
  Local rounds% = 6 + 52 \ N%
  Local sum%, y%, z%, e%, p%

  If cmd$ = "encrypt" Then

    sum% = 0
    y% = 0
    z% = Peek(Word b_addr% + 4 * (N% - 1))

    Do
      sum% = (sum% + DELTA%) And &hFFFFFFFF
      e% = (sum% >> 2) And 3
      For p% = 0 To N% - 2
        y% = Peek(Word b_addr% + 4 * (p% + 1))
        z% = (Peek(Word b_addr% + 4 * p%) + crypt.mx%(y%, z%, sum%, k_addr%, p%, e%)) And &hFFFFFFFF
        Poke Word b_addr% + 4 * p%, z%
      Next
      y% = Peek(Word b_addr%)
      z% = (Peek(Word b_addr% + 4*(N%-1)) + crypt.mx%(y%, z%, sum%, k_addr%, p%, e%)) And &hFFFFFFFF
      Poke Word b_addr% + 4 * (N% - 1), z%
      Inc rounds%, -1
    Loop While rounds%

  ElseIf cmd$ = "decrypt" Then

    sum% = rounds% * DELTA%
    y% = Peek(Word b_addr%)
    z% = 0

    Do
      e% = (sum% >> 2) And 3
      For p% = N% - 1 To 1 Step -1
        z% = Peek(Word b_addr% + 4 * (p% - 1))
        y% = (Peek(Word b_addr% + 4 * p%) - crypt.mx%(y%, z%, sum%, k_addr%, p%, e%)) And &hFFFFFFFF
        Poke Word b_addr% + 4 * p%, y%
      Next
      z% = Peek(Word b_addr% + 4 * (N% - 1))
      y% = (Peek(Word b_addr%) - crypt.mx%(y%, z%, sum%, k_addr%, p%, e%)) And &hFFFFFFFF
      Poke Word b_addr%, y%
      sum% = (sum% - DELTA%) And &hFFFFFFFF
      Inc rounds%, -1
    Loop While rounds%

  Else

    Error "Unknown encryption command: " + cmd$

  EndIf

  crypt.xxtea_block% = 1
End Function

Function crypt.mx%(y%, z%, sum%, k_addr%, p%, e%)
  Local lhs% = (z% >> 5) Xor ((y% << 2) And &hFFFFFFFF)
  Inc lhs%, (y% >> 3) Xor ((z% << 4) And &hFFFFFFFF)
  Local rhs% = (sum% Xor y%) + (Peek(Word k_addr% + 4 * ((p% And 3) Xor e%)) Xor z%)
  crypt.mx% = (lhs% And &hFFFFFFFF) Xor (rhs% And &hFFFFFFFF)
End Function
' END:       #Include "../splib/crypt.inc" -------------------------------------
' BEGIN:     #Include "../splib/string.inc" ------------------------------------
' Copyright (c) 2019-2021 Thomas Hugo Williams
' For Colour Maximite 2, MMBasic 5.06

On Error Skip 1 : Dim sys.VERSION$ = ""
If sys.VERSION$ = "" Then Error "'system.inc' not included"
sys.provides("string")
If sys.err$ <> "" Then Error sys.err$

' Pads a string with spaces to the left and right so that it will be centred
' within a fixed length field. If the string is longer than the field then
' this function just returns the string. If an odd number of spaces are
' required then the extra space is added to the left hand side of the string.
'
' @param  s$  the string to centre.
' @param  x   the field length.
Function str.centre$(s$, x%)
  If Len(s$) < x% Then
    str.centre$ = s$ + Space$((x% - Len(s$)) \ 2)
    str.centre$ = Space$(x% - Len(str.centre$)) + str.centre$
  Else
    str.centre$ = s$
  EndIf
End Function

' Compares s1$ and s2$ ignoring case considerations.
Function str.equals_ignore_case%(s1$, s2$)
  str.equals_ignore_case% = LCase$(s1$) = LCase$(s2$)
End Function

Function str.lpad$(s$, x%)
  str.lpad$ = s$
  If Len(s$) < x% Then str.lpad$ = Space$(x% - Len(s$)) + s$
End Function

' Tokenises a string.
'
' @param   s$     string to tokenise.
' @param   sep$   one or more token separator characters.
'                 If empty then use space and skip empty tokens.
' @param   skip%  1 to skip empty tokens, 0 to return them.
' @return  the first token. To retrieve subsequent tokens call this function
'          with no parameters, i.e. tk$ = str.next_token$().
'          Returns sys.NO_DATA$ if there are no more tokens.
'
' WARNING! Take care when calling this function naively in cases where s$ might
' be the empty string as that will return data from a previously incomplete
' tokenisation. If necessary call str.next_token$(sys.NO_DATA$) to clear the
' internal state first.
Function str.next_token$(s$, sep$, skip%)
  Static s_$ = sys.NO_DATA$, sep_$, skip_%, st%

  If s$ <> "" Then
    s_$ = s$
    sep_$ = Choice(sep$ = "", " ", sep$)
    skip_% = Choice(sep$ = "", 1, skip%)
    st% = 1
  EndIf

  ' Have we already processed then entire string?
  If s_$ = sys.NO_DATA$ Then str.next_token$ = sys.NO_DATA$ : Exit Function

  Local i%
  For i% = st% To Len(s_$)
    If InStr(sep_$, Mid$(s_$, i%, 1)) > 0 Then
      str.next_token$ = Mid$(s_$, st%, i% - st%)
      st% = i% + 1
      If skip_% And str.next_token$ = "" Then Continue For
      Exit Function
    EndIf
  Next

  ' Return the last token.
  Local tmp$ = Mid$(s_$, st%, i% - st%)
  str.next_token$ = Choice(tmp$ = "" And skip_%, sys.NO_DATA$, tmp$)
  s_$ = sys.NO_DATA$
End Function

Function str.quote$(s$, begin$, end$)
  Local begin_$ = Choice(begin$ = "", Chr$(34), Left$(begin$, 1))
  Local end_$ = Choice(end$ = "", begin_$, Left$(end$, 1))
  str.quote$ = begin_$ + s$ + end_$
End Function

Function str.rpad$(s$, x%)
  str.rpad$ = s$
  If Len(s$) < x% Then str.rpad$ = s$ + Space$(x% - Len(s$))
End Function

' Returns a copy of s$ with leading and trailing spaces removed.
Function str.trim$(s$)
  Local st%, en%
  For st% = 1 To Len(s$)
    If Peek(Var s$, st%) <> 32 Then Exit For
  Next
  For en% = Len(s$) To 1 Step -1
    If Peek(Var s$, en%) <> 32 Then Exit For
  Next
  If en% >= st% Then str.trim$ = Mid$(s$, st%, en% - st% + 1)
End Function

' If s$ both begins and ends with " then returns a copy of s$ with those characters removed,
' otherwise returns an unmodified copy of s$.
Function str.unquote$(s$)
  If Peek(Var s$, 1) = 34 Then
    If Peek(var s$, Len(s$)) = 34 Then
      str.unquote$ = Mid$(s$, 2, Len(s$) - 2)
      Exit Function
    EndIf
  EndIf
  str.unquote$ = s$
End Function
' END:       #Include "../splib/string.inc" ------------------------------------
' BEGIN:     #Include "../splib/list.inc" --------------------------------------
' Copyright (c) 2020 Thomas Hugo Williams
' For Colour Maximite 2, MMBasic 5.06

On Error Skip 1 : Dim sys.VERSION$ = ""
If sys.VERSION$ = "" Then Error "'system.inc' not included"
sys.provides("list")
If sys.err$ <> "" Then Error sys.err$

' Gets the upper-bound that should be used to dimension an array to hold a list
' of the given capacity, irrespective of OPTION BASE.
'
' e.g. To create a string array that can be used as a list of 10 elements:
'        Dim my_list$(list.new%(10))
Function list.new%(capacity%)
  list.new% = capacity% + Mm.Info(Option Base)
End Function

' Gets the capacity of the list.
Function list.capacity%(list$())
  list.capacity% = Bound(list$(), 1) - Bound(list$(), 0)
End Function

' Initialises the list.
Sub list.init(lst$())
  Local i%
  For i% = Bound(lst$(), 0) To Bound(lst$(), 1)
    lst$(i%) = sys.NO_DATA$
  Next
  lst$(Bound(lst$(), 1)) = "0"
End Sub

' Appends an element to the end of the list.
Sub list.add(lst$(), s$)
  Local lb% = Bound(lst$(), 0)
  Local ub% = Bound(lst$(), 1)
  Local sz% = Val(lst$(ub%))
  ' TODO: report error if adding to a full list.
  lst$(lb% + sz%) = s$
  lst$(ub%) = Str$(sz% + 1)
End Sub

' Clears the list and resets its size to 0.
Sub list.clear(lst$())
  Local lb% = Bound(lst$(), 0)
  Local ub% = Bound(lst$(), 1)
  Local sz% = Val(lst$(ub%))
  Local i%
  For i% = lb% To lb% + sz% - 1
    lst$(i%) = sys.NO_DATA$
  Next
  lst$(ub%) = "0"
End Sub

' Prints the contents of the list.
Sub list.dump(lst$())
  Local lb% = Bound(lst$(), 0)
  Local ub% = Bound(lst$(), 1)
  Local sz% = Val(lst$(ub%))
  Local i%
  For i% = lb% To lb% + sz% - 1
    Print "[" Str$(i%) "] " lst$(i%)
  Next
  Print "END"
End Sub

' Gets a list element with bounds checking.
' To get a list element without bounds checking just do s$ = lst$(index%) directly.
Function list.get$(lst$(), index%)
  Local lb% = Bound(lst$(), 0)
  Local ub% = Bound(lst$(), 1)
  Local sz% = Val(lst$(ub%))
  If index% >= lb% + sz% Then Error "index out of bounds: " + Str$(index%) : Exit Function
  list.get$ = lst$(index%)
End Function

' Inserts an element into the list.
Sub list.insert(lst$(), index%, s$)
  Local lb% = Bound(lst$(), 0)
  Local ub% = Bound(lst$(), 1)
  Local sz% = Val(lst$(ub%))
  Local i%
  If index% >= lb% + sz% + 1 Then Error "index out of bounds: " + Str$(index%) : Exit Function
  For i% = lb% + sz% To lb% + index% + 1 Step -1
    lst$(i%) = lst$(i% - 1)
  Next
  lst$(i%) = s$
  lst$(ub%) = Str$(sz% + 1)
End Sub

Function list.is_full%(lst$())
  Local ub% = Bound(lst$(), 1)
  '             = (ub% - lb%) = sz%
  list.is_full% = (ub% - Bound(lst$(), 0)) = Val(lst$(ub%))
End Function

' Returns the element at the end of the list.
' If the list is empty then returns sys.NO_DATA$
Function list.peek$(lst$())
  Local lb% = Bound(lst$(), 0)
  Local ub% = Bound(lst$(), 1)
  Local sz% = Val(lst$(ub%))
  If sz% > 0 Then list.peek$ = lst$(lb% + sz% - 1) Else list.peek$ = sys.NO_DATA$
End Function

' Removes and returns the element at the end of the list.
' If the list is empty then returns sys.NO_DATA$
Function list.pop$(lst$())
  Local lb% = Bound(lst$(), 0)
  Local ub% = Bound(lst$(), 1)
  Local sz% = Val(lst$(ub%))
  If sz% > 0 Then
    list.pop$ = lst$(lb% + sz% - 1)
    lst$(ub%) = Str$(sz% - 1)
  Else
    list.pop$ = sys.NO_DATA$
  EndIf
End Function

' Synonym for add().
Sub list.push(lst$(), s$)
  Local lb% = Bound(lst$(), 0)
  Local ub% = Bound(lst$(), 1)
  Local sz% = Val(lst$(ub%))
  lst$(lb% + sz%) = s$
  lst$(ub%) = Str$(sz% + 1)
End Sub

' Removes an element from the list.
Sub list.remove(lst$(), index%)
  Local lb% = Bound(lst$(), 0)
  Local ub% = Bound(lst$(), 1)
  Local sz% = Val(lst$(ub%))
  Local i%
  If index% >= lb% + sz% Then Error "index out of bounds: " + Str$(index%) : Exit Sub
  For i% = index% To lb% + sz% - 2
    lst$(i%) = lst$(i% + 1)
  Next
  lst$(i%) = sys.NO_DATA$
  lst$(ub%) = Str$(sz% - 1)
End Sub

' Sets a list element with bounds checking.
' To set a list element without bounds checking just do lst$(index%) = s$ directly.
Sub list.set(lst$(), index%, s$)
  Local lb% = Bound(lst$(), 0)
  Local ub% = Bound(lst$(), 1)
  Local sz% = Val(lst$(ub%))
  If index% >= lb% + sz% Then Error "index out of bounds: " + Str$(index%) : Exit Sub
  lst$(index%) = s$
End Sub

' Sorts the list.
Sub list.sort(lst$())
  Local lb% = Bound(lst$(), 0)
  Local ub% = Bound(lst$(), 1)
  Local sz% = Val(lst$(ub%))
  Sort lst$(), , , lb%, sz%
End Sub

' Gets the size of the list.
Function list.size%(list$())
  list.size% = Val(list$(Bound(list$(), 1)))
End Function
' END:       #Include "../splib/list.inc" --------------------------------------
' BEGIN:     #Include "../splib/file.inc" --------------------------------------
' Copyright (c) 2020-2021 Thomas Hugo Williams
' For Colour Maximite 2, MMBasic 5.06

On Error Skip 1 : Dim sys.VERSION$ = ""
If sys.VERSION$ = "" Then Error "'system.inc' not included"
sys.requires("array", "list", "string")
sys.provides("file")
If sys.err$ <> "" Then Error sys.err$

Const fil.PROG_DIR$ = fil.get_parent$(Mm.Info$(Current))

' Gets the number of files in directory d$ whose names match pattern$.
'
' @param   d$        directory to process.
' @param   pattern$  file pattern to match on, e.g. "*.bas".
' @param   type$     type of files to return, "dir", "file" or "all".
' @return  the number of matching files.
Function fil.count_files%(d$, pattern$, type$)
  Local f$

  Select Case LCase$(type$)
    Case "all"      : f$ = Dir$(d$ + "/*", All)
    Case "dir"      : f$ = Dir$(d$ + "/*", Dir)
    Case "file", "" : f$ = Dir$(d$ + "/*", File)
    Case Else       : Error "unknown type: " + type$
  End Select

  Do While f$ <> ""
    If fil.fnmatch%(pattern$, f$) Then Inc fil.count_files%
    f$ = Dir$()
  Loop
End Function

' Does the file/directory 'f$' exist?
'
' @return 1 if the file exists, otherwise 0
Function fil.exists%(f$)
  Local f_$ = fil.get_canonical$(f$)
  If f_$ = "A:" Then
    fil.exists% = 1
  Else
    fil.exists% = Mm.Info(FileSize f_$) <> -1
  EndIf
End Function

' Find files whose names match pattern$
'
' @param   path$     root directory to start looking from.
' @param   pattern$  file pattern to match on, e.g. "*.bas".
' @param   type$     type of files to return, "dir", "file" or "all".
' @return  the absolute path to the first match found. To retrieve subsequent
'          matches call this function with no parameters, i.e. f$ = fil.find().
'          Returns the empty string if there are no more matches.
'
' TODO: In order to return the files in alphabetical order this uses a 128K workspace.
'       I think there is another potential implementation where it just records the last
'       file returned by the previous call and restarts the search from there, however
'       that would be much slower and probably better implemented as a CSUB.
Function fil.find$(path$, pattern$, type$)
  Static stack$(list.new%(1000)) Length 128 ' 128K workspace.
  Static pattern_$
  Static type_$
  Local base% = Mm.Info(Option Base)
  Local f$, is_dir%, lb%, name$, num%

  If path$ <> "" Then
    list.init(stack$())
    f$ = fil.get_canonical$(path$)
    list.push(stack$(), f$)
    pattern_$ = pattern$
    If pattern_$ = "" Then pattern_$ = "*"
    type_$ = UCase$(type$)
    If type_$ = "" Then type_$ = "FILE"
    If InStr("|ALL|DIR|FILE|", "|" + type_$ + "|") < 1 Then Error "unknown type: " + type_$
  EndIf

  Do
'    list.dump(stack$())
    Do
      fil.find$ = list.pop$(stack$())
      If fil.find$ = sys.NO_DATA$ Then fil.find$ = "" : Exit Function
      name$ = fil.get_name$(fil.find$)
    Loop Until name$ <> ".git" ' Doesn't recurse into .git directories.

    ' If it is a directory then expand it.
    is_dir% = fil.is_directory%(fil.find$)
    If is_dir% Then
      lb% = base% + list.size%(stack$())
      If type_$ = "DIR" Then f$ = Dir$(fil.find$ + "/*", Dir) Else f$ = Dir$(fil.find$ + "/*", All)
      Do While f$ <> ""
        list.push(stack$(), fil.find$ + "/" + f$)
        f$ = Dir$()
      Loop

      ' Sort the newly pushed dirs/files so that those beginning 'a|A'
      ' are at the top and those beginning 'z|Z' are near the bottom.
      num% = base% + list.size%(stack$()) - lb%
      If num% > 0 Then Sort stack$(), , &b11, lb%, num%)
    EndIf

    ' I've profiled it and its faster to do the name checks before the type checks.
    If fil.fnmatch%(pattern_$, name$) Then
      Select Case type_$
        Case "ALL"  : Exit Do
        Case "DIR"  : If is_dir% Then Exit Do
        Case "FILE" : If Not is_dir% Then Exit Do
      End Select
    EndIf

  Loop

End Function

' Does name$ match pattern$ ?
'
' @param   pattern$  *nix style 'shell wildcard' pattern.
' @param   name$     the name to test.
' @return  1 if the name matches the pattern otherwise 0.
'
' Derived from the work of Russ Cox: https://research.swtch.com/glob
Function fil.fnmatch%(pattern$, name$)
  Local p$ = UCase$(pattern$)
  Local n$ = UCase$(name$)
  Local c%, px% = 1, nx% = 1, nextPx% = 0, nextNx% = 0

  Do While px% <= Len(p$) Or nx% <= Len(n$)

    If px% <= Len(p$) Then

      c% = Peek(Var p$, px%)
      Select Case c%
        Case 42 ' *
          ' Zero-or-more-character wildcard
          ' Try to match at sx%,
          ' if that doesn't work out,
          ' restart at nx%+1 next.
          nextPx% = px%
          nextNx% = nx% + 1
          Inc px%
          Goto fil.fnmatch_cont

        Case 63 ' ?
          ' Single-character wildcard
          If nx% <= Len(n$) Then
            Inc px%
            Inc nx%
            Goto fil.fnmatch_cont
          EndIf

        Case Else
          ' Ordinary character
          If nx% <= Len(n$) Then
            If c% = Peek(Var n$, nx%) Then
              Inc px%
              Inc nx%
              Goto fil.fnmatch_cont
            EndIf
          EndIf
      End Select

    EndIf

    If nextNx% > 0 Then
      If nextNx% <= Len(n$) + 1 Then
        px% = nextPx%
        nx% = nextNx%
        Goto fil.fnmatch_cont
      EndIf
    EndIf

    Exit Function

    fil.fnmatch_cont:

  Loop

  fil.fnmatch% = 1

End Function

' Gets the canonical path for file/directory 'f$'.
Function fil.get_canonical$(f_$)
  Local f$ = f_$

  If fil.is_absolute%(f$) Then
    If Instr(UCase$(f$), "A:") <> 1 Then f$ = "A:" + f$
  Else
    f$ = Mm.Info$(Directory) + f$
  EndIf

  Local elements$(list.new%(20)) Length 128
  list.init(elements$())
  Local lb% = Bound(elements$(), 0)
  Local tk$ = str.next_token$(f$, "/\", 1)
  Do While tk$ <> sys.NO_DATA$
    list.add(elements$(), tk$)
    tk$ = str.next_token$()
  Loop
  elements$(lb%) = "A:"

  Local num_elements% = list.size%(elements$())
  Local i% = lb%
  Do While i% < lb% + num_elements%
    If elements$(i%) = "." Then
      list.remove(elements$(), i%)
    ElseIf elements$(i%) = ".." Then
      list.remove(elements$(), i%)
      list.remove(elements$(), i% - 1)
      Inc i%, -1
    Else
      Inc i%
    EndIf
  Loop

  fil.get_canonical$ = array.join_strings$(elements$(), "/", lb%, list.size%(elements$()))
End Function

' Gets the dot file-extension, from filename f$.
' e.g. fil.get_extension("foo.bas") => ".bas"
Function fil.get_extension$(f$)
  Local i%
  For i% = Len(f$) To 1 Step -1
    Select Case Peek(Var f$, i%)
      Case 46     ' .
        fil.get_extension$ = Mid$(f$, i%)
        Exit Function
      Case 47, 92 ' / or \
        Exit For
    End Select
  Next
End Function

' Gets the files in directory d$ whose names match pattern$.
'
' @param       d$        directory to process.
' @param       pattern$  file pattern to match on, e.g. "*.bas".
' @param       type$     type of files to return, "dir", "file" or "all".
' @param[out]  out$()    the names of matching files are copied into this array.
Sub fil.get_files(d$, pattern$, type$, out$())
  Local f$

  Select Case LCase$(type$)
    Case "all"      : f$ = Dir$(d$ + "/*", All)
    Case "dir"      : f$ = Dir$(d$ + "/*", Dir)
    Case "file", "" : f$ = Dir$(d$ + "/*", File)
    Case Else       : Error "unknown type: " + type$
  End Select

  Local i% = Mm.Info(Option Base)
  Do While f$ <> ""
    If fil.fnmatch%(pattern$, f$) Then
      out$(i%) = f$
      Inc i%
    EndIf
    f$ = Dir$()
  Loop

  If i% > Mm.Info(Option Base) Then
    Sort out$(), , &b10, Mm.Info(Option Base), i% - Mm.Info(Option Base)
  EndIf
End Sub

' Gets the name of file/directory 'f$' minus any path information.
Function fil.get_name$(f$)
  Local i%

  For i% = Len(f$) To 1 Step -1
    If InStr("/\", Mid$(f$, i%, 1)) > 0 Then Exit For
  Next i%

  fil.get_name$ = Mid$(f$, i% + 1)
End Function

' Gets the parent directory of 'f$', or the empty string if it does not have one.
Function fil.get_parent$(f$)
  Local i%

  For i% = Len(f$) To 1 Step -1
    If InStr("/\", Mid$(f$, i%, 1)) > 0 Then Exit For
  Next i%

  If i% > 0 Then fil.get_parent$ = Left$(f$, i% - 1)
End Function

Function fil.is_absolute%(f$)
  fil.is_absolute% = 1
  If InStr(f$, "/") = 1 Then Exit Function
  If InStr(f$, "\") = 1 Then Exit Function
  If InStr(UCase$(f$), "A:\") = 1 Then Exit Function
  If InStr(UCase$(f$), "A:/") = 1 Then Exit Function
  If UCase$(f$) = "A:" Then Exit Function
  fil.is_absolute% = 0
End Function

Function fil.is_directory%(f$)
  Local f_$ = fil.get_canonical$(f$)
  If f_$ = "A:" Then
    fil.is_directory% = 1
  Else
    fil.is_directory% = Mm.Info(FileSize f_$) = -2
  EndIf
End Function

' Makes directory 'f$' if it does not already exist.
Sub fil.mkdir(f$)
  If Not fil.exists%(f$) Then MkDir f$
End Sub

' Returns a copy of f$ with any dot file-extension removed.
' e.g. fil.trim_extension("foo.bas") => "foo"
Function fil.trim_extension$(f$)
  Local i%
  For i% = Len(f$) To 1 Step -1
    Select Case Peek(Var f$, i%)
      Case 46     ' .
        fil.trim_extension$ = Mid$(f$, 1, i% - 1)
        Exit Function
      Case 47, 92 ' / or \
        Exit For
    End Select
  Next
  fil.trim_extension$ = f$
End Function
' END:       #Include "../splib/file.inc" --------------------------------------
' BEGIN:     #Include "../common/sptools.inc" ----------------------------------
' Copyright (c) 2020-2021 Thomas Hugo Williams
' For Colour Maximite 2, MMBasic 5.06

On Error Skip 1 : Dim sys.VERSION$ = ""
If sys.VERSION$ = "" Then Error "'system.inc' not included"
sys.requires("file")
sys.provides("sptools")
If sys.err$ <> "" Then Error sys.err$

Const spt.INSTALL_DIR$ = spt.get_install_dir$()
Const spt.RESOURCES_DIR$ = spt.INSTALL_DIR$ + "/resources"

' Gets the 'sptools' installation directory.
Function spt.get_install_dir$()

  ' First try recursing up the directory structure from the running program
  ' until a file called 'sptools.root' is found.
  Local d$ = fil.PROG_DIR$
  Do While d$ <> ""
    If fil.exists%(d$ + "/sptools.root") Then Exit Do
    d$ = fil.get_parent$(d$)
  Loop

  ' Otherwise try the default installation location.
  If d$ = "" Then d$ = "A:/sptools"

  If Not fil.is_directory%(d$) Then Error "directory not found: " + d$

  spt.get_install_dir$ = d$
End Function

Sub spt.print_version(name$)
  Print name$ " (SP Tools) Release " + sys.VERSION$ + " for Colour Maximite 2, MMBasic 5.07"
  Print "Copyright (c) 2020-2021 Thomas Hugo Williams"
  Print "A Toy Plastic Trumpet Production for Sockpuppet Studios."
  Print "License MIT <https://opensource.org/licenses/MIT>"
  Print "This is free software: you are free to change and redistribute it."
  Print "There is NO WARRANTY, to the extent permitted by law."
End Sub
' END:       #Include "../common/sptools.inc" ----------------------------------

Const PROG_NAME$ = LCase$(fil.trim_extension$(fil.get_name$(Mm.Info(Current))))

Dim cmd$
Dim in_file$
Dim out_file$
Dim password$
Dim version%
Dim ok%

ok% = parse_cmdline%()
If Not ok% Then print_usage() : End
If version% Then spt.print_version(PROG_NAME$) : End

If Not fil.exists%(in_file$) Then
  ok% = 0
  sys.err$ = "input file '" + in_file$ + "' not found"
EndIf

If ok% Then
  Select Case cmd$
    Case "decrypt" : ok% = cmd_decrypt%()
    Case "encrypt" : ok% = cmd_encrypt%()
    Case "md5"     : ok% = cmd_md5%()
    Case Else      : Error "Unimplemented command."
  End Select
EndIf

If ok% Then
  Print "OK"
ElseIf sys.err$ <> "" Then
  Print PROG_NAME$ + ": " + sys.err$)
EndIf

End

Function parse_cmdline%()

  Local token$ = str.next_token$(Mm.CmdLine$)
  Local opt$, value$

  Do
    token$ = Choice(token$ = sys.NO_DATA$, "", token$)
    If token$ = "" Then Exit Do
    If parse_option%(token$) Then
      If version% Then parse_cmdline% = 1 : Exit Function
    ElseIf sys.err$ <> "" Then
      Exit Function
    ElseIf cmd$ = "" Then
      cmd$ = LCase$(token$)
    ElseIf in_file$ = "" Then
      in_file$ = token$
    ElseIf out_file$ = "" Then
      out_file$ = token$
    Else
      sys.err$ = "unexpected argument '" + token$ + "'"
      Exit Function
    EndIf
    token$ = str.next_token$()
  Loop

  ' Validate command.
  Select Case cmd$
    Case "" : sys.err$ = "no command specified"
    Case "decrypt", "encrypt", "md5"
    Case Else : sys.err$ = "unknown command '" + cmd$ + "' specified"
  End Select
  If cmd$ = "md5" And password$ <> "" Then
    sys.err$ = "MD5 command does not support '--password' option"
  EndIf
  If sys.err$ <> "" Then Exit Function

  ' Validate input file.
  in_file$ = str.trim$(str.unquote$(in_file$))
  If in_file$ = "" Then sys.err$ = "no input file specified" : Exit Function

  ' Validate output file.
  out_file$ = str.trim$(str.unquote$(out_file$))
  If cmd$ = "md5" And out_file$ <> "" Then
    sys.err$ = "MD5 command does not expect output file"
    Exit Function
  EndIf
  If out_file$ = "" Then out_file$ = in_file$ + "." + cmd$

  parse_cmdline% = 1

End Function

Function parse_option%(token$)
  If InStr(token$, "-") <> 1 Then Exit Function

  Local p% = InStr(token$, "=")
  Local opt$ = Choice(p% = 0, token$, Mid$(token$, 1, p% - 1))
  Local value$ = Choice(p% = 0, "", Mid$(token$, p% + 1))

  Select Case opt$
    Case "-p", "--password"
      If value$ = "" Then
        sys.err$ = "option '" + opt$ + "' expects argument"
      Else
        password$ = value$
      EndIf
    Case "--version"
      If value$ = "" Then
        version% = 1
      Else
        sys.err$ = "option '" + opt$ + "' does not expect an argument"
      EndIf
    Case Else
      sys.err$ = "unknown option '" + opt$ + "'"
  End Select

  parse_option% = (sys.err$ = "")
End Function

Sub print_usage()
  Local in$ = str.quote$("input file")
  Local out$ = str.quote$("output file")

  If sys.err$ <> "" Then Print PROG_NAME$ + ": " + sys.err$ : Print

  Print "Usage *" PROG_NAME$ " [OPTION]... <command> " in$ " [" out$ "]"
  Print
  Print "Options:"
  Print "  -p, --password=<password>  Use <password> for encryption/decryption."
  Print "                             If omitted then user will be prompted."
  Print "  --version                  Output version information and exit."
  Print
  Print "Commands:"
  Print "  decrypt  Decrypt " in$ " using XXTEA algorithm."
  Print "  encrypt  Encrypt " in$ " using XXTEA algorithm."
  Print "  md5      Calculate MD5 checksum for " in$ "."
End Sub

Function cmd_decrypt%()
  If Not prompt_for_overwrite%() Then Print "CANCELLED" : Exit Function
  If Not prompt_for_password%()  Then Print "CANCELLED" : Exit Function

  Print "Decrypting from '" in_file$ "' to '" out_file$ "' ..."
  Local md5%(array.new%(2))
  If Not crypt.md5%(Peek(VarAddr password$) + 1, Len(password$), md5%()) Then Exit Function
  Open in_file$ For Input As #1
  Open out_file$ For Output As #2
  Local iv%(array.new%(2))
  Local ok% = crypt.xxtea_file%("decrypt", 1, 2, md5%(), iv%())
  Close #2
  Close #1
  cmd_decrypt% = ok%
End Function

Function prompt_for_overwrite%()
  Local s$ = "y"
  If fil.exists%(out_file$) Then
    Line Input "Overwrite existing '" + out_file$ + "' [y|N] ? ", s$
    s$ = LCase$(str.trim$(s$))
  EndIf
  prompt_for_overwrite% = (s$ = "y")
Exit Function

Function prompt_for_password%()
  password$ = str.trim$(password$)
  If password$ = "" Then
    Input "Password? ", password$
    password$ = str.trim$(password$)
  EndIf
  prompt_for_password% = password$ <> ""
Exit Function

Function cmd_encrypt%()
  If Not prompt_for_overwrite%() Then Print "CANCELLED" : Exit Function
  If Not prompt_for_password%()  Then Print "CANCELLED" : Exit Function

  Print "Encrypting from '" in_file$ "' to '" out_file$ "' ..."
  Local md5%(array.new%(2))
  If Not crypt.md5%(Peek(VarAddr password$) + 1, Len(password$), md5%()) Then Exit Function
  Open in_file$ For Input As #1
  Open out_file$ For Output As #2
  Local iv%(array.new%(2))
  fill_iv(iv%())
  Local ok% = crypt.xxtea_file%("encrypt", 1, 2, md5%(), iv%())
  Close #2
  Close #1
  cmd_encrypt% = ok%
End Function

Sub fill_iv(iv%())
  Local i%, iv_addr% = Peek(VarAddr iv%()))
  For i% = 0 To 15 : Poke Byte iv_addr% + i%, Fix(256 * Rnd()) : Next
End Sub

Function cmd_md5%()
  Open in_file$ For Input As #1
  Local md5%(array.new%(2))
  cmd_md5% = crypt.md5_file%(1, md5%())
  Close #1
  If cmd_md5% Then Print crypt.md5_fmt$(md5%())
End Function
