' Transpiled on 23-11-2022 11:06:44

'----------------------
'Raycast.bas by Martin Herhaus
'Raycast Engine MMBasic translated from
'https://youtu.be/gYRrGTC7GtA by 3DSage
'----------------------

' BEGIN:     #include "src/ctrl.ipp" -------------------------------------------
' Copyright (c) 2022 Thomas Hugo Williams
' License MIT <https://opensource.org/licenses/MIT>
'
' MMBasic Controller Library
'
' Preprocessor flag PICOMITE defined
' Preprocessor flag CTRL_ONE_PLAYER defined
' Preprocessor flag CTRL_NO_SNES defined
' Preprocessor flag CTRL_USE_ON_PS2 defined

Const ctrl.VERSION = 905  ' 0.9.5

' Button values as returned by controller driver subroutines.
Const ctrl.R      = &h01
Const ctrl.START  = &h02
Const ctrl.HOME   = &h04
Const ctrl.SELECT = &h08
Const ctrl.L      = &h10
Const ctrl.DOWN   = &h20
Const ctrl.RIGHT  = &h40
Const ctrl.UP     = &h80
Const ctrl.LEFT   = &h100
Const ctrl.ZR     = &h200
Const ctrl.X      = &h400
Const ctrl.A      = &h800
Const ctrl.Y      = &h1000
Const ctrl.B      = &h2000
Const ctrl.ZL     = &h4000

Const ctrl.OPEN  = -1
Const ctrl.CLOSE = -2
Const ctrl.SOFT_CLOSE = -3

' The NES standard specifies a 12 micro-second pulse, but all the controllers
' I've tested work with 1 micro-second, and possibly less.
Const ctrl.PULSE = 0.001 ' 1 micro-second

' When a key is down the corresponding byte of this 256-byte map is set,
' when the key is up then it is unset.
'
' Note that when using INKEY$ (as opposed to the CMM2 'KEYDOWN' function or
' the PicoMiteVGA 'ON PS2' command) to read the keyboard we cannot detect
' keyup events and instead automatically clear a byte after it is read.
Dim ctrl.key_map%(31 + Mm.Info(Option Base))

' Map used to convert PS/2 set 2 scan codes to entries in ctrl.key_map%().
' The scan code first has to be converted into a single byte value,
' see ctrl.on_ps2().
Dim ctrl.scan_map%(31)

' Initialises keyboard reading.
'
' @param  period%  CMM2 only - interval to read KEYDOWN state, default 40 ms.
' @param  nbr%     CMM2 only - timer nbr to read KEYDOWN state, default 4.
Sub ctrl.init_keys(period%, nbr%)
  ctrl.term_keys()
  Read Save
  Restore ctrl.scan_map_data
  Local i%
  For i% = Bound(ctrl.scan_map%(), 0) To Bound(ctrl.scan_map%(), 1)
    Read ctrl.scan_map%(i%)
  Next
  Read Restore
  On Ps2 ctrl.on_ps2()
End Sub

Sub ctrl.on_ps2()
  Local ps2% = Mm.Info(PS2)
  Select Case ps2%
    Case Is < &hE000 : Poke Var ctrl.key_map%(), Peek(Var ctrl.scan_map%(), ps2% And &hFF), 1
    Case Is < &hF000 : Poke Var ctrl.key_map%(), Peek(Var ctrl.scan_map%(), (ps2% And &hFF) + &h80), 1
    Case Is < &hE0F000 : Poke Var ctrl.key_map%(), Peek(Var ctrl.scan_map%(), ps2% And &hFF), 0
    Case Else : Poke Var ctrl.key_map%(), Peek(Var ctrl.scan_map%(), (ps2% And &hFF) + &h80), 0
  End Select
End Sub

' Terminates keyboard reading.
Sub ctrl.term_keys()
  On Ps2 0
  Memory Set Peek(VarAddr ctrl.key_map%()), 0, 256
  Do While Inkey$ <> "" : Loop
End Sub

Function ctrl.keydown%(i%)
  ctrl.keydown% = Peek(Var ctrl.key_map%(), i%)
End Function

Function ctrl.poll_multiple$(ctrls$(), mask%, duration%)
  Local expires% = Choice(duration%, Timer + duration%, &h7FFFFFFFFFFFFFFF), i%
  Do
    For i% = Bound(ctrls$(), 0) To Bound(ctrls$(), 1)
      If ctrl.poll_single%(ctrls$(i%), mask%) Then
        ctrl.poll_multiple$ = ctrls$(i%)
        Exit Do
      EndIf
    Next
  Loop While Timer < expires%
End Function

' Opens, polls (for a maximum of 5ms) and closes a controller.
'
' @param  ctrl$  controller driver function.
' @param  mask%  bit mask to match against.
' @return        1 if any of the bits in the mask match what is read from the
'                controller, otherwise 0.
Function ctrl.poll_single%(ctrl$, mask%)
  On Error Ignore
  Call ctrl$, ctrl.OPEN
  If Mm.ErrNo = 0 Then
    Local key%, t% = Timer + 5
    Do
      Call ctrl$, key%
      If key% And mask% Then
        ctrl.poll_single% = 1
        ' Wait for user to release key.
        Do While key% : Pause 5 : Call ctrl$, key% : Loop
        Exit Do
      EndIf
    Loop While Timer < t%
    Call ctrl$, ctrl.SOFT_CLOSE
  EndIf
  On Error Abort
End Function

' Gets a string representation of bits read from a controller driver.
'
' @param  x%  bits returned by driver.
' @return     string representation.
Function ctrl.bits_to_string$(x%)
  Static BUTTONS$(14) = ("R","Start","Home","Select","L","Down","Right","Up","Left","ZR","X","A","Y","B","ZL")

  If x% = 0 Then
    ctrl.bits_to_string$ = "No buttons down"
    Exit Function
  EndIf

  ctrl.bits_to_string$ = Str$(x%) + " = "
  Local count%, i%, s$
  For i% = 0 To Bound(BUTTONS$(), 1)
    If x% And 2^i% Then
      s$ = BUTTONS$(i%)
      If count% > 0 Then Cat ctrl.bits_to_string$, ", "
      Cat ctrl.bits_to_string$, s$
      Inc count%
    EndIf
  Next
End Function

' Reads the keyboard as if it were a controller.
'
' Note that the PicoMite has no KEYDOWN function so we are limited to
' reading a single keypress from the input buffer and cannot handle multiple
' simultaneous keys or properly handle a key being pressed and not released.
Sub keys_cursor(x%)
  If x% < 0 Then Exit Sub
  x% =    ctrl.keydown%(32)  * ctrl.A
  Inc x%, ctrl.keydown%(128) * ctrl.UP
  Inc x%, ctrl.keydown%(129) * ctrl.DOWN
  Inc x%, ctrl.keydown%(130) * ctrl.LEFT
  Inc x%, ctrl.keydown%(131) * ctrl.RIGHT
End Sub

' Atari joystick on PicoGAME Port A.
Sub atari_a(x%)
  Select Case x%
    Case Is >= 0
      x% =    Not Pin(GP14) * ctrl.A
      Inc x%, Not Pin(GP0)  * ctrl.UP
      Inc x%, Not Pin(GP1)  * ctrl.DOWN
      Inc x%, Not Pin(GP2)  * ctrl.LEFT
      Inc x%, Not Pin(GP3)  * ctrl.RIGHT
      Exit Sub
    Case ctrl.OPEN
      SetPin GP0, DIn : SetPin GP1, DIn : SetPin GP2, DIn : SetPin GP3, DIn : SetPin GP14, DIn
    Case ctrl.CLOSE, ctrl.SOFT_CLOSE
      SetPin GP0, Off : SetPin GP1, Off : SetPin GP2, Off : SetPin GP3, Off : SetPin GP14, Off
  End Select
End Sub

' Reads port A connected to a NES gamepad.
'
' Note that the extra pulse after reading bit 7 (Right) should not be necessary,
' but in practice some NES clone controllers require it to behave correctly.
'
'   GP2: Latch, GP3: Clock, GP1: Data
Sub nes_a(x%)
  Select Case x%
    Case Is >= 0
      Pulse GP2, ctrl.PULSE
      x% =    Not Pin(GP1) * ctrl.A      : Pulse GP3, ctrl.PULSE
      Inc x%, Not Pin(GP1) * ctrl.B      : Pulse GP3, ctrl.PULSE
      Inc x%, Not Pin(GP1) * ctrl.SELECT : Pulse GP3, ctrl.PULSE
      Inc x%, Not Pin(GP1) * ctrl.START  : Pulse GP3, ctrl.PULSE
      Inc x%, Not Pin(GP1) * ctrl.UP     : Pulse GP3, ctrl.PULSE
      Inc x%, Not Pin(GP1) * ctrl.DOWN   : Pulse GP3, ctrl.PULSE
      Inc x%, Not Pin(GP1) * ctrl.LEFT   : Pulse GP3, ctrl.PULSE
      Inc x%, Not Pin(GP1) * ctrl.RIGHT  : Pulse GP3, ctrl.PULSE
      Exit Sub
    Case ctrl.OPEN
      SetPin GP1, Din : SetPin GP2, Dout : SetPin GP3, Dout
      Pin(GP2) = 0 : Pin(GP3) = 0
      nes_a(0) ' Discard the first reading.
    Case ctrl.CLOSE, ctrl.SOFT_CLOSE
      SetPin GP1, Off : SetPin GP2, Off : SetPin GP3, Off
  End Select
End Sub

ctrl.scan_map_data:

Data &h9C92919395009900, &h0060099496989A00, &h0031710000008B00, &h00327761737A0000
Data &h0033346564786300, &h0035727466762000, &h0036796768626E00, &h003837756A6D0000
Data &h0039306F696B2C00, &h002D703B6C2F2E00, &h00003D5B00270000, &h000023005D0A0000
Data &h0008000000005C00, &h0000003734003100, &h001B383635322E30, &h0000392A2D332B9B
Data &h0000000097000000, &h0000000000000000, &h0000000000008B00, &h0000000000000000
Data &h0000000000000000, &h0000000000000000, &h0000000000000000, &h0000000000000000
Data &h0000000000000000, &h0000000000000000, &h0000000000000000, &h0000000000000000
Data &h0000000000000000, &h0000008682008700, &h0000808300817F84, &h0000889D00890000
' END:       #Include "src/ctrl.ipp" -------------------------------------------

Dim float px,py,pdy,pdx,pa
Dim integer fps%,bg%,FC%,n%,mapX,mapY,mapS,HV,HH,vmul,hnul
mapS=64:mapy=24:mapx=24
Dim Map%(mapy*mapx)
P2 =Pi/2
P3=3*Pi/2
P2I=2*PI
DR=0.0174533 'one degree in radians

'Init
Select Case Mm.Device$
Case "PicoMiteVGA"
 Mode 2
 Font 1
 FrameBuffer Create
 FrameBuffer Write F
 Dim DRIVERS$(2) = ("atari_a", "nes_a", "keys_wasd")
 Const CONTROLLER_TEXT$ = "       Press START, FIRE or SPACE       "
Case "Colour Maximite 2", "Colour Maximite 2 G2", "MMBasic for Windows"
 Mode 1,8 '800x600
 Font 1
 Page Write 1
 If Mm.Device$ = "MMBasic for Windows" Then
   Dim DRIVERS$(1) = ("keys_wasd", "keys_wasd")
   Const CONTROLLER_TEXT$ = "          Press SPACE to start          "
 Else
   Dim DRIVERS$(3) = ("atari_dx", "nes_dx", "wii_any_3", "keys_wasd")
   Const CONTROLLER_TEXT$ = "       Press START, FIRE or SPACE       "
 EndIf
Case Else
 Error "Unsupported device: " + Mm.Device$
End Select
vmul=MM.VRes/512
hmul=MM.HRes/1024
hv=MM.VRes/2
hh=MM.HRes/2

Restore MapData:For n%=0 To mapx*mapy-1:Read Map%(n%):Next n%
bg%= RGB(0,64,0)
CLS bg%
px=300:py=300
pdx=Cos(pa)*5:pdy=Sin(pa)*5
tm%=Timer:fps%=0

drawRays3D

Text hh, hv + 20, CONTROLLER_TEXT$, "CM", 1, 1, Rgb(Black), Rgb(White)
If Mm.Device$ = "PicoMiteVGA" Then FrameBuffer Copy F, N, B Else Page Copy 1 To 0
ctrl.init_keys()
Dim driver$ = ctrl.poll_multiple$(DRIVERS$(), ctrl.A Or ctrl.B Or ctrl.START)
Call driver$, ctrl.OPEN
Dim key% = 1
Do While key% <> 0 : Call driver$, key% : Loop
Text hh, hv + 20, Space$(Len(CONTROLLER_TEXT$)), "CM", 1, 1, bg%, bg%

'main
Do
' display
drawRays3D
buttons
 If Mm.Device$ = "PicoMiteVGA" Then
   FrameBuffer Copy F, N
Else
   Page Copy 1 To 0
 EndIf
inc fps%:
if fps%=500 then
tm%=timer-tm%

  text hh+20,hv+20,str$(500/(tm%/1000),4,2)+"   ":fps%=0:tm%=Timer
end if
Loop

Function dist(ax,ay,bx,by,ang)
  dist=Sqr((bx-ax)*(bx-ax)+(by-ay)*(by-ay))
End Function

Sub drawRays3D
 Local integer r,mx,my,mp,dof,xstp,wcl
 Local floadt rx,ry,ra,xo,yo,dis1
 ra=pa-DR*30:Inc ra,P2i*(ra<0):Inc ra,-P2I*(ra>P2I)
 Box 0,0,hh<<1,hv,,RGB(0,255,255),RGB(0,255,255)
 Box 0,hv/2,hh<<1,hv/2,,RGB(255,64,0),RGB(255,64,0)
 'Cast 120 Rays
 For r=0 To 119
     'check Horizontal Lines
     dof=0
     distH=1000000:hx=px:hy=py
     atan=-1/Tan(ra)
     If ra>Pi Then
       ry=((Int(py>>6))<<6)-0.0001:rx=(py-ry)*atan+px ' Looking up
       yo=-64:xo=-yo*atan
     else
       ry=((Int(py>>6))<<6)+64:rx=(py-ry)*atan+px ' Looking down
       yo=64:xo=-yo*atan
     EndIf
     If not ra Or ra=Pi Then rx=px:ry=py:dof=mapx 'looking straight left or right
     Do While dof<mapx
       mx=Int(rx)>>6:my=Int(ry)>>6:mp=my*mapx+mx
         If mp>0 And mp<mapX*mapY Then If map%(mp)<>0 Then
           hx=rx:hy=ry:distH=dist(px,py,hx,hy,ra):dof=mapx
         Else
           rx=rx+Xo:ry=ry+Yo:Inc dof
        End If
    Loop
   'check vertical Lines
   dof=0
    distV=1000000:vx=px:vy=py
   ntan=-Tan(ra)
   If ra>P2 And ra< P3 Then
     rx=((Int(px>>6))<<6)-0.0001:ry=(px-rx)*ntan+py ' Looking left
     xo=-64:yo=-xo*ntan
   else
     rx=((Int(px>>6))<<6)+64:ry=(px-rx)*ntan+py ' Looking right
     xo=64:yo=-xo*ntan
   End If
    If not ra Or ra=Pi Then rx=px:ry=py:dof=mapy 'looking straight up or down
    Do While dof<mapy
      mx=Int(rx)>>6:my=Int(ry)>>6:mp=my*mapx+mx
      If mp<(mapX*mapY)And mp>-1 Then If map%(mp)<>0 Then
        vx=rx:vy=ry:distV=dist(px,py,vx,vy,ra):dof=mapy
      Else
        rx=rx+Xo:ry=ry+Yo:Inc dof
     End If
    Loop
    If distH<distV Then
      rx=hx:ry=hy:dis1=DistH:wcl=RGB(0,64,255)
    else
      rx=vx:ry=vy:dis1=DistV:wcl=RGB(0,128,255)
    end if
   '
   '--- Draw 3D Walls
   '
    ca=pa-ra:Inc ca,P2I*(ca<0):Inc ca,-P2I*(ca>P2I)
    dis1=dis1*Cos(ca)
    lineH=int((mapS*hh)/dis1):lineH=Min(lineH,hv)
    lineO=int(lineh>>1)
    Box xstp,(hv>>1)-LineO,1+hh/60,Lineh,,wcl,wcl
    Inc xstp,hh/60
    Inc ra,dr/2:Inc ra,P2I*(Ra<0):Inc ra,-P2I*(ra>P2I)
Next
End Sub

Sub display
drawmap2d
drawPlayer
End Sub

Sub drawPlayer
'unused
Local xo,yo
xo=px*hmul:yo=py*vmul
fc%=RGB(yellow)
Box xo-1,yo-1,3,3,,fc%
Line xo,yo,xo+pdx*5*hmul,yo+pdy*5*vmul,,fc%
End Sub


Sub buttons
  Local k%
  Call driver$, k%
  If Not k% Then Call "keys_cursor", k%
  Select Case k%
    Case ctrl.A, ctrl.B
      Save Image "cast.bmp"
    Case ctrl.UP
      Inc px,2*pdx
      Inc py,2*pdy
    Case ctrl.DOWN
      Inc px,2*-pdx
      Inc py,2*-pdy
    Case ctrl.LEFT
      Inc pa,-0.1
      If pa<0 Then Inc pa,P2I
      pdx=Cos(pa)*5:pdy=Sin(pa)*5
    Case ctrl.RIGHT
      Inc pa, 0.1
      If pa>P2I Then Inc pa,-P2I
      pdx=Cos(pa)*5:pdy=Sin(pa)*5
  End Select
End Sub

Sub drawMap2D
'unused
Local integer x,y,xo,yo
YO=2*HV/mapy:xo=HH/mapx
xo=xo*hmul:yo=yo*vmul
For y=0 To mapy-1:For x=0 To mapx-1
   Box x*xo,y*yo,xo,yo,,0,RGB(white)*(map%(y*mapX+x)<>0)
 Next : Next
End Sub

Sub keys_wasd(x%)
  If x% < 0 Then Exit Sub
  x% = ctrl.keydown%(32) * ctrl.A
  Inc x%, ctrl.keydown%(Asc("w")) * ctrl.UP
  Inc x%, ctrl.keydown%(Asc("s")) * ctrl.DOWN
  Inc x%, ctrl.keydown%(Asc("a")) * ctrl.LEFT
  Inc x%, ctrl.keydown%(Asc("d")) * ctrl.RIGHT
End Sub

MapData:
Data 1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1
Data 1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,0,0,0,1
Data 1,0,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1
Data 1,0,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1
Data 1,0,1,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,0,0,0,1
Data 1,0,1,0,0,0,0,1,1,1,1,1,1,1,1,1,1,1,0,1,1,1,1,1
Data 1,0,1,0,0,0,0,1,0,1,0,1,0,1,0,1,1,0,0,0,1,1,1,1
Data 1,0,1,0,0,0,0,1,0,0,0,0,0,0,0,1,1,0,0,0,0,0,0,8
Data 1,0,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,1,1,1
Data 1,0,1,0,0,0,0,1,0,0,0,0,0,0,0,1,1,0,0,0,0,0,0,1
Data 1,0,0,0,0,0,0,1,0,0,0,0,0,0,0,1,1,0,0,0,1,1,1,1
Data 1,0,0,0,0,0,0,1,1,1,1,0,1,1,1,1,1,1,1,1,1,1,1,1
Data 1,1,1,1,1,1,1,1,1,1,1,0,1,1,1,1,1,1,1,1,1,1,1,1
Data 1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1
Data 1,1,1,1,1,1,0,1,1,1,1,0,1,1,1,1,1,1,1,1,1,1,1,1
Data 1,1,1,1,1,1,0,1,1,1,1,0,1,1,1,1,1,1,1,1,1,1,1,1
Data 1,0,0,0,0,0,0,0,0,1,1,0,1,1,0,0,0,0,0,1,0,0,0,1
Data 1,0,0,0,0,0,0,0,0,0,0,0,1,1,0,0,1,0,0,1,0,0,0,1
Data 1,0,0,0,0,0,0,0,0,1,1,0,1,1,0,0,0,0,0,1,1,0,1,1
Data 1,0,1,0,1,0,0,0,0,1,1,0,0,0,0,0,1,0,0,0,0,0,0,1
Data 1,0,0,1,0,0,0,0,0,1,1,0,1,1,0,0,0,0,0,1,1,0,1,1
Data 1,0,1,0,1,0,0,0,0,1,1,0,1,1,0,0,1,0,0,1,0,0,0,1
Data 1,0,0,0,0,0,0,0,0,1,1,0,1,1,0,0,0,0,0,1,0,0,0,1
Data 1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1
