' Transpiled on 25-04-2023 09:11:17

' Copyright (c) 2020-2023 Thomas Hugo Williams
' License MIT <https://opensource.org/licenses/MIT>
' For MMBasic 5.07

Option Explicit On
Option Default Integer

Const MAX_NUM_FILES = 5
Dim in.num_open_files = 1

' BEGIN: #Include #Include "../../splib/system.inc" ----------------------------
' Copyright (c) 2020-2023 Thomas Hugo Williams
' License MIT <https://opensource.org/licenses/MIT>
' For MMBasic 5.07

Const sys.VERSION$ = "2.0.0-a1"
Const sys.NO_DATA$ = Chr$(&h7F)
Const sys.CRLF$ = Chr$(13) + Chr$(10)

Const sys.SUCCESS = 0
Const sys.FAILURE = -1

Dim sys.break_flag%
Dim sys.err$

Const sys.MAX_INCLUDES% = 20
Dim sys.includes$(sys.MAX_INCLUDES%) Length 20

Sub sys.provides(f$)
  Local f_$ = LCase$(f$)
  Local i% = 1
  Do
    Select Case sys.includes$(i%)
      Case f_$ : sys.err$ = "file already included: " + f_$ + ".inc" : Exit Sub
      Case ""  : sys.includes$(i%) = f_$ : Exit Sub
    End Select
    Inc i%
    If i% > sys.MAX_INCLUDES% Then sys.err$ = "too many includes" : Exit Sub
  Loop
End Sub

Sub sys.requires(f1$, f2$, f3$, f4$, f5$, f6$, f7$, f8$, f9$, f10$)

  ' Use indexes from 1..10 even when Option Base 0.
  Local f$(10) Length 20
  f$(1) = f1$ : f$(2) = f2$ : f$(3) = f3$ : f$(4) = f4$ : f$(5) = f5$
  f$(6) = f6$ : f$(7) = f7$ : f$(8) = f8$ : f$(9) = f9$ : f$(10) = f10$
  Local i%
  For i% = 1 To 10 : f$(i%) = LCase$(f$(i%)) : Next

  Local j%, ok%, fail%
  For i% = 1 To 10
    If f$(i%) <> "" Then
      ok% = 0
      For j% = 1 To sys.MAX_INCLUDES%
        Select Case sys.includes$(j%)
          Case f$(i%) : ok% = 1 : Exit For
          Case ""     : Exit For
        End Select
      Next
      If Not ok% Then
        If Not fail% Then
          sys.err$ = "required file(s) not included: " + f$(i%) + ".inc"
          fail% = 1
        Else
          Cat sys.err$, ", " + f$(i%) + ".inc"
        EndIf
      EndIf
    EndIf
  Next
End Sub

' Formats a firmware version as a 5-digit number, e.g.
'   5.05.06 => 50506
'   5.04    => 50400
'
' @param version$  the firmware version to format.
'                  If empty then formats the current firmware version number.
Function sys.firmware_version%(version$)
  Local i%, s$, v$ = version$
  If v$ = "" Then v$ = Str$(Mm.Info$(Version))
  For i% = 1 To Len(v$)
    If InStr("0123456789", Mid$(v$, i%, 1)) > 0 Then s$ = s$ + Mid$(v$, i%, 1)
  Next
  Do While Len(s$) < 5 : s$ = s$ + "0" : Loop
  sys.firmware_version% = Val(s$)
End Function

' Overrides Ctrl-C behaviour such that:
'   - Ctrl-C will call sys.break_handler()
'   - Ctrl-D will perform an actual MMBasic break
Sub sys.override_break()
  sys.break_flag% = 0
  Option Break 4
  On Key 3, sys.break_handler()
End Sub

' Called as an ON KEY interrupt when Ctrl-C is overridden by sys.override_break().
' Increments the sys.break_flag%, if the flag is then > 1 then END the program.
Sub sys.break_handler()
  Inc sys.break_flag%
  If sys.break_flag% > 1 Then
    sys.restore_break()
    End
  EndIf
End Sub

' Restores default Ctrl-C behaviour.
Sub sys.restore_break()
  sys.break_flag% = 0
  On Key 3, 0
  Option Break 3
End Sub

Function sys.string_prop$(key$)
  Select Case LCase$(key$)
    Case "home"
      Select Case Mm.Device$
        Case "MMB4L"
          sys.string_prop$ = Mm.Info$(EnvVar "HOME")
        Case "MMBasic for Windows"
          sys.string_prop$ = Mm.Info$(EnvVar "HOMEDIR") + Mm.Info$(EnvVar "HOMEPATH")
        Case Else
          sys.string_prop$ = "A:"
      End Select
    Case "separator"
      sys.string_prop$ = Choice(Mm.Device$ = "MMBasic for Windows", "\", "/")
    Case "tmpdir"
      Select Case Mm.Device$
        Case "MMB4L"
          sys.string_prop$ = Choice(Mm.Info$(EnvVar "TMPDIR") = "", "/tmp", Mm.Info$(EnvVar "TMPDIR"))
        Case "MMBasic for Windows"
          sys.string_prop$ = Mm.Info$(EnvVar "TMP")
        Case Else
          sys.string_prop$ = "A:/tmp"
      End Select
    Case Else
      Error "Unknown property: " + key$
  End Select
End Function

' Are we running on one of the given devices ?
'
' @param  d1$, d2$, ... d5$  one of:
'                mmb4l  - MMBasic for Linux
'                mmb4w  - MMBasic for Windows
'                cmm2   - Colour Maximite 2 (G1)
'                cmm2g2 - Colour Maximite 2 (G2)
'                cmm2*  - any Colour Maximite 2
'                pm     - PicoMite
'                pmvga  - PicoMiteVGA
'                pm*    - any PicoMite
Function sys.is_device%(d1$, d2$, d3$, d4$, d5$)
  Local devices$(5 + Mm.Info(Option Base)) Length 16 = (d1$, d2$, d3$, d4$, d5$, "")
  Local d$, ii% = Mm.Info(Option Base)
  sys.is_device% = 1
  Do While Len(devices$(ii%)) > 0
    d$ = LCase$(devices$(ii%))
    Select Case Mm.Device$
      Case "MMB4L"
        If d$ = "mmb4l" Then
          Exit Function
        ElseIf d$ = "mmb4l-armv6l" And Mm.Info$(Arch) = "Linux armv6l" Then
          Exit Function
        EndIf
      Case "MMBasic for Windows"
        If d$ = "mmb4w" Then Exit Function
      Case "Colour Maximite 2"
        If d$ = "cmm2" Or d$ = "cmm2*" Then Exit Function
      Case "Colour Maximite 2 G2"
        If d$ = "cmm2g2" Or d$ = "cmm2*" Then Exit Function
      Case "PicoMite"
        If d$ = "pm" Or d$ = "pm*" Then Exit Function
      Case "PicoMiteVGA"
        If d$ = "pmvga" Or d$ = "pm*" Then Exit Function
      Case Else
        Error "Unknown device: " + Mm.Device$
    End Select
    Inc ii%
  Loop
  sys.is_device% = 0
End Function

Function sys.error%(code%, msg$)
  If Not code% Then Error "Invalid error code"
  sys.error% = code%
  sys.err$ = msg$
End Function
' END:   #Include "/home/thwill/github/cmm2-sptools/src/sptrans/tests/../../splib/system.inc"
' BEGIN: #Include #Include "../../splib/array.inc" -----------------------------
' Copyright (c) 2020-2023 Thomas Hugo Williams
' License MIT <https://opensource.org/licenses/MIT>
' For MMBasic 5.07

On Error Skip 1 : Dim sys.VERSION$ = ""
If sys.VERSION$ = "" Then Error "'system.inc' not included"
sys.provides("array")
If sys.err$ <> "" Then Error sys.err$

' Gets the upper-bound that should be used to dimension an array of the given
' capacity, irrespective of OPTION BASE.
'
' e.g. To create a string array that can hold 10 elements:
'        Dim my_array$(array.new%(10))
Function array.new%(capacity%)
  array.new% = capacity% + Mm.Info(Option Base) - 1
End Function

' Binary search for a value in a SORTED array.
'
' @param  a$()    the array.
' @param  s$      the value to search for.
' @param  flags$  "i" to search case-insensitively,
' @param  lb%     the lower bound to search from,
'                 if 0/unset then search from the first element.
' @param  num%    the number of elements to search,
'                 if 0/unset then search all the elements (from lb%).
' @return         the index of the element containing the value,
'                 or -1 if not present.
Function array.bsearch%(a$(), s$, flags$, lb_%, num_%)
  Local lb% = Choice(lb_% = 0, Bound(a$(), 0), lb_%)
  Local num% = Choice(num_% = 0, Bound(a$(), 1) - Bound(a$(), 0) + 1, num_%)
  Local ub% = lb% + num% - 1
  Local i%

  If InStr(UCase$(flags$), "I") Then
    Local us$ = UCase$(s$)
    Local ua$
    Do While ub% >= lb%
      i% = (lb% + ub%) \ 2
      ua$ = UCase$(a$(i%))
      If us$ > ua$ Then
        lb% = i% + 1
      ElseIf us$ < ua$ Then
        ub% = i% - 1
      Else
        Exit Do
      EndIf
    Loop
  Else
    Do While ub% >= lb%
      i% = (lb% + ub%) \ 2
      If s$ > a$(i%) Then
        lb% = i% + 1
      ElseIf s$ < a$(i%) Then
        ub% = i% - 1
      Else
        Exit Do
      EndIf
    Loop
  EndIf

  If ub% >= lb% Then array.bsearch% = i% Else array.bsearch% = -1
End Function

' Gets the capacity (number of elements) that string array a$() can hold.
Function array.capacity%(a$())
  array.capacity% = Bound(a$(), 1) - Bound(a$(), 0) + 1
End Function

' Copies a string array.
'
' @param  src$()    the source array.
' @param  dst$()    the destination array.
' @param  src_idx%  the start index in the source,
'                   if 0/unset then use the index of the first element.
' @param  dst_idx%  the start index in the destination,
'                   if 0/unset then use the index of the first element.
' @param  num%      the number of elements to copy,
'                   if 0/unset then copy all the elements (from idx%) from the source.
Sub array.copy(src$(), dst$(), src_idx%, dst_idx%, num%)
  Local base% = Mm.Info(Option Base 0)
  Local i%
  Local j% = Max(base%, dst_idx%)
  Local lb% = Max(base%, src_idx%)
  Local ub% = src_idx% + num% - 1
  If num% = 0 Then ub% = Bound(src$(), 1)

  ' TODO: Use a memory copy instead of a loop.
  For i% = lb% To ub% : dst$(j%) = src$(i%) : Inc j% : Next
End Sub

' Fills all the elements of string array a$() to x$.
Sub array.fill(a$(), x$)
  Local lb% = Bound(a$(), 0)
  Local ub% = Bound(a$(), 1)
  Local i%
  For i% = lb% To ub% : a$(i%) = x$ : Next
End Sub

' Returns a string consisting of the concatenated elements of a float array
' joined together with a delimiter.
'
' @param   a!()    the array.
' @param   delim$  delimiter to place between each element, if empty/unset then uses comma.
' @param   lb%     lower bound to start from, if 0/unset then the 1st element.
' @param   num%    number of elements to join, if 0/unset then all elements.
' @param   slen%   maximum length of string to return, if 0/unset then 255 chars.
' @return          a string composed of the array elements separated by the delimiter. If the
'                  string had to be truncated to slen% then it is terminated with an ellipsis "..."
Function array.join_floats$(a!(), delim$, lb%, num%, slen%)
  Local delim_$ = Choice(delim$ = "", ",", delim$)
  Local lb_% = Choice(lb% = 0, Mm.Info(Option Base), lb%)
  Local ub_% = Choice(num% = 0, Bound(a!(), 1), lb_% + num% - 1)
  Local slen_% = Choice(slen% = 0, 255, slen%)

  Local s$ = Str$(a!(lb_%))
  Inc lb_%

  Do While lb_% <= ub_%
    Cat s$, Left$(delim_$, 255 - Len(s$))
    Cat s$, Left$(Str$(a!(lb_%)), 255 - Len(s$))
    Inc lb_%
  Loop

  If Len(s$) <= slen_% Then
    array.join_floats$ = s$
  Else
    array.join_floats$ = Left$(s$, slen_% - 3) + "..."
  EndIf
End Function

' Returns a string consisting of the concatenated elements of an integer array
' joined together with a delimiter.
'
' @param   a%()    the array.
' @param   delim$  delimiter to place between each element, if empty/unset then uses comma.
' @param   lb%     lower bound to start from, if 0/unset then the 1st element.
' @param   num%    number of elements to join, if 0/unset then all elements.
' @param   slen%   maximum length of string to return, if 0/unset then 255 chars.
' @return          a string composed of the array elements separated by the delimiter. If the
'                  string had to be truncated to slen% then it is terminated with an ellipsis "..."
Function array.join_ints$(a%(), delim$, lb%, num%, slen%)
  Local delim_$ = Choice(delim$ = "", ",", delim$)
  Local lb_% = Choice(lb% = 0, Mm.Info(Option Base), lb%)
  Local ub_% = Choice(num% = 0, Bound(a%(), 1), lb_% + num% - 1)
  Local slen_% = Choice(slen% = 0, 255, slen%)

  Local s$ = Str$(a%(lb_%))
  Inc lb_%

  Do While lb_% <= ub_%
    Cat s$, Left$(delim_$, 255 - Len(s$))
    Cat s$, Left$(Str$(a%(lb_%)), 255 - Len(s$))
    Inc lb_%
  Loop

  If Len(s$) <= slen_% Then
    array.join_ints$ = s$
  Else
    array.join_ints$ = Left$(s$, slen_% - 3) + "..."
  EndIf
End Function

' Returns a string consisting of the concatenated elements of a string array
' joined together with a delimiter.
'
' @param   a$()    the array.
' @param   delim$  delimiter to place between each element, if empty/unset then uses comma.
' @param   lb%     lower bound to start from, if 0/unset then the 1st element.
' @param   num%    number of elements to join, if 0/unset then all elements.
' @param   slen%   maximum length of string to return, if 0/unset then 255 chars.
' @return          a string composed of the array elements separated by the delimiter. If the
'                  string had to be truncated to slen% then it is terminated with an ellipsis "..."
Function array.join_strings$(a$(), delim$, lb%, num%, slen%)
  Local delim_$ = Choice(delim$ = "", ",", delim$)
  Local lb_% = Choice(lb% = 0, Mm.Info(Option Base), lb%)
  Local ub_% = Choice(num% = 0, Bound(a$(), 1), lb_% + num% - 1)
  Local slen_% = Choice(slen% = 0, 255, slen%)

  Local s$ = a$(lb_%)
  Inc lb_%

  Do While lb_% <= ub_%
    Cat s$, Left$(delim_$, 255 - Len(s$))
    Cat s$, Left$(a$(lb_%), 255 - Len(s$))
    Inc lb_%
  Loop

  If Len(s$) <= slen_% Then
    array.join_strings$ = s$
  Else
    array.join_strings$ = Left$(s$, slen_% - 3) + "..."
  EndIf
End Function
' END:   #Include "/home/thwill/github/cmm2-sptools/src/sptrans/tests/../../splib/array.inc"
' BEGIN: #Include #Include "../../splib/list.inc" ------------------------------
' Copyright (c) 2020-2023 Thomas Hugo Williams
' License MIT <https://opensource.org/licenses/MIT>
' For MMBasic 5.07

On Error Skip 1 : Dim sys.VERSION$ = ""
If sys.VERSION$ = "" Then Error "'system.inc' not included"
sys.provides("list")
If sys.err$ <> "" Then Error sys.err$

' Gets the upper-bound that should be used to dimension an array to hold a list
' of the given capacity, irrespective of OPTION BASE.
'
' e.g. To create a string array that can be used as a list of 10 elements:
'        Dim my_list$(list.new%(10))
Function list.new%(capacity%)
  list.new% = capacity% + Mm.Info(Option Base)
End Function

' Gets the capacity of the list.
Function list.capacity%(list$())
  list.capacity% = Bound(list$(), 1) - Bound(list$(), 0)
End Function

' Initialises the list.
Sub list.init(lst$())
  Local i%
  For i% = Bound(lst$(), 0) To Bound(lst$(), 1)
    lst$(i%) = sys.NO_DATA$
  Next
  lst$(Bound(lst$(), 1)) = "0"
End Sub

' Appends an element to the end of the list.
Sub list.add(lst$(), s$)
  Local lb% = Bound(lst$(), 0)
  Local ub% = Bound(lst$(), 1)
  Local sz% = Val(lst$(ub%))
  ' TODO: report error if adding to a full list.
  lst$(lb% + sz%) = s$
  lst$(ub%) = Str$(sz% + 1)
End Sub

' Clears the list and resets its size to 0.
Sub list.clear(lst$())
  Local lb% = Bound(lst$(), 0)
  Local ub% = Bound(lst$(), 1)
  Local sz% = Val(lst$(ub%))
  Local i%
  For i% = lb% To lb% + sz% - 1
    lst$(i%) = sys.NO_DATA$
  Next
  lst$(ub%) = "0"
End Sub

' Prints the contents of the list.
Sub list.dump(lst$())
  Local lb% = Bound(lst$(), 0)
  Local ub% = Bound(lst$(), 1)
  Local sz% = Val(lst$(ub%))
  Local i%
  For i% = lb% To lb% + sz% - 1
    Print "[" Str$(i%) "] " lst$(i%)
  Next
  Print "END"
End Sub

' Gets a list element with bounds checking.
' To get a list element without bounds checking just do s$ = lst$(index%) directly.
Function list.get$(lst$(), index%)
  Local lb% = Bound(lst$(), 0)
  Local ub% = Bound(lst$(), 1)
  Local sz% = Val(lst$(ub%))
  If index% >= lb% + sz% Then Error "index out of bounds: " + Str$(index%) : Exit Function
  list.get$ = lst$(index%)
End Function

' Inserts an element into the list.
Sub list.insert(lst$(), index%, s$)
  Local lb% = Bound(lst$(), 0)
  Local ub% = Bound(lst$(), 1)
  Local sz% = Val(lst$(ub%))
  Local i%
  If index% >= lb% + sz% + 1 Then Error "index out of bounds: " + Str$(index%) : Exit Function
  For i% = lb% + sz% To lb% + index% + 1 Step -1
    lst$(i%) = lst$(i% - 1)
  Next
  lst$(i%) = s$
  lst$(ub%) = Str$(sz% + 1)
End Sub

Function list.is_full%(lst$())
  Local ub% = Bound(lst$(), 1)
  '             = (ub% - lb%) = sz%
  list.is_full% = (ub% - Bound(lst$(), 0)) = Val(lst$(ub%))
End Function

' Returns the element at the end of the list.
' If the list is empty then returns sys.NO_DATA$
Function list.peek$(lst$())
  Local lb% = Bound(lst$(), 0)
  Local ub% = Bound(lst$(), 1)
  Local sz% = Val(lst$(ub%))
  If sz% > 0 Then list.peek$ = lst$(lb% + sz% - 1) Else list.peek$ = sys.NO_DATA$
End Function

' Removes and returns the element at the end of the list.
' If the list is empty then returns sys.NO_DATA$
Function list.pop$(lst$())
  Local lb% = Bound(lst$(), 0)
  Local ub% = Bound(lst$(), 1)
  Local sz% = Val(lst$(ub%))
  If sz% > 0 Then
    list.pop$ = lst$(lb% + sz% - 1)
    lst$(ub%) = Str$(sz% - 1)
  Else
    list.pop$ = sys.NO_DATA$
  EndIf
End Function

' Synonym for add().
Sub list.push(lst$(), s$)
  Local lb% = Bound(lst$(), 0)
  Local ub% = Bound(lst$(), 1)
  Local sz% = Val(lst$(ub%))
  lst$(lb% + sz%) = s$
  lst$(ub%) = Str$(sz% + 1)
End Sub

' Removes an element from the list.
Sub list.remove(lst$(), index%)
  Local lb% = Bound(lst$(), 0)
  Local ub% = Bound(lst$(), 1)
  Local sz% = Val(lst$(ub%))
  Local i%
  If index% >= lb% + sz% Then Error "index out of bounds: " + Str$(index%) : Exit Sub
  For i% = index% To lb% + sz% - 2
    lst$(i%) = lst$(i% + 1)
  Next
  lst$(i%) = sys.NO_DATA$
  lst$(ub%) = Str$(sz% - 1)
End Sub

' Sets a list element with bounds checking.
' To set a list element without bounds checking just do lst$(index%) = s$ directly.
Sub list.set(lst$(), index%, s$)
  Local lb% = Bound(lst$(), 0)
  Local ub% = Bound(lst$(), 1)
  Local sz% = Val(lst$(ub%))
  If index% >= lb% + sz% Then Error "index out of bounds: " + Str$(index%) : Exit Sub
  lst$(index%) = s$
End Sub

' Sorts the list.
Sub list.sort(lst$())
  Local lb% = Bound(lst$(), 0)
  Local ub% = Bound(lst$(), 1)
  Local sz% = Val(lst$(ub%))
  Sort lst$(), , , lb%, sz%
End Sub

' Gets the size of the list.
Function list.size%(list$())
  list.size% = Val(list$(Bound(list$(), 1)))
End Function
' END:   #Include "/home/thwill/github/cmm2-sptools/src/sptrans/tests/../../splib/list.inc"
' BEGIN: #Include #Include "../../splib/string.inc" ----------------------------
' Copyright (c) 2020-2023 Thomas Hugo Williams
' License MIT <https://opensource.org/licenses/MIT>
' For MMBasic 5.07

On Error Skip 1 : Dim sys.VERSION$ = ""
If sys.VERSION$ = "" Then Error "'system.inc' not included"
sys.provides("string")
If sys.err$ <> "" Then Error sys.err$

' Pads a string with spaces to the left and right so that it will be centred
' within a fixed length field. If the string is longer than the field then
' this function just returns the string. If an odd number of spaces are
' required then the extra space is added to the left hand side of the string.
'
' @param  s$  the string to centre.
' @param  x   the field length.
Function str.centre$(s$, x%)
  If Len(s$) < x% Then
    str.centre$ = s$ + Space$((x% - Len(s$)) \ 2)
    str.centre$ = Space$(x% - Len(str.centre$)) + str.centre$
  Else
    str.centre$ = s$
  EndIf
End Function

' Compares s1$ and s2$ ignoring case considerations.
Function str.equals_ignore_case%(s1$, s2$)
  str.equals_ignore_case% = LCase$(s1$) = LCase$(s2$)
End Function

' Does a string contain only printable and/or whitespace ASCII ?
'
' @param  s$  the string.
' @return     1 if the string only contains printable and/or whitespace ASCII, otherwise 0.
Function str.is_plain_ascii%(s$)
  Local i%
  For i% = 1 To Len(s$)
    Select Case Peek(Var s$, i%)
      Case 9, 10, 13   : ' Tab, line-feed and carriage-return are acceptable.
      Case < 32, > 126 : Exit Function
      Case Else        : ' ASCII printable characters.
    End Select
  Next
  str.is_plain_ascii% = 1
End Function

Function str.lpad$(s$, x%)
  str.lpad$ = s$
  If Len(s$) < x% Then str.lpad$ = Space$(x% - Len(s$)) + s$
End Function

' Tokenises a string separated by delimiters.
'
'  - delimiters are ignored between pairs of double-quotes.
'  - a double-quote preceeded by a backslash does not end a pair of
'    double-quotes.
'  - double-quotes and backslashes are included verbatim in the returned tokens.
'
' @param   s$     string to tokenise.
' @param   dlm$   one or more token delimiter characters.
'                 If empty then use space and skip empty tokens.
' @param   skip%  1 to skip empty tokens, 0 to return them.
' @return  the first token. To retrieve subsequent tokens call this function
'          with no parameters, i.e. tk$ = str.next_token$().
'          Returns sys.NO_DATA$ if there are no more tokens.
'
' WARNING! Take care when calling this function naively in cases where s$ might
' be the empty string as that will return data from a previously incomplete
' tokenisation. If necessary call str.next_token$(sys.NO_DATA$) to clear the
' internal state first.
Function str.next_token$(s$, dlm$, skip%)
  Static s_$, dlm_$, skip_%, p%

  If s$ <> "" Then
    s_$ = s$
    dlm_$ = Choice(dlm$ = "", " ", dlm$)
    skip_% = Choice(dlm$ = "", 1, skip%)
    p% = 1 ' Index to process from.
  EndIf

  ' If we've reached the end of the string then return either NO_DATA or an
  ' empty token depending on the value of skip_% and then force skip_% =
  If p% > Len(s_$) Then
    str.next_token$ = Choice(skip_%, sys.NO_DATA$, "")
    skip_% = 1
    Exit Function
  EndIf

  Local ch%, state% = 0
  For p% = p% To Len(s_$)
    ch% = Peek(Var s_$, p%)

    Select Case state%
      Case 0  ' Base state
        If ch% = &h22 Then
          state% = 1
        ElseIf InStr(dlm_$, Chr$(ch%)) Then
          If skip_% Then
            If Len(str.next_token$) = 0 Then Continue For
          EndIf
          Inc p%
          Exit For
        EndIf

      Case 1  ' Inside double-quote
        Select Case ch%
          Case &h22 : state% = 0  ' double-quote
          Case &h5C : state% = 2  ' backslash
        End Select

      Case 2  ' Inside double-quote and following backslash
        state% = 1

    End Select

    Cat str.next_token$, Chr$(ch%)
  Next

  ' If we reach the end of the string and the last character is not a delimiter
  ' then force skip_% = 1 so we do not return an empty token on the next call.
  If p% > Len(s_$) Then
    If Not InStr(dlm_$, Chr$(ch%)) Then skip_% = 1
  EndIf
End Function

' Gets a string "quoted" with given characters.
'
' @param  s$      the string.
' @param  begin$  the character to put at the start, defaults to double-quote.
' @param  end$    the character to put at the end, defaults to double-quote.
' @return         the "quoted" string.
Function str.quote$(s$, begin$, end$)
  Local begin_$ = Choice(begin$ = "", Chr$(34), Left$(begin$, 1))
  Local end_$ = Choice(end$ = "", begin_$, Left$(end$, 1))
  str.quote$ = begin_$ + s$ + end_$
End Function

' Gets copy of 'haystack$' with occurrences of 'needle$' replaced by 'rep$'.
Function str.replace$(haystack$, needle$, rep$)
  Local p%, st%, s$
  Do
    Inc st%
    p% = InStr(st%, haystack$, needle$)
    If p% < 1 Then Exit Do
    Cat s$, Mid$(haystack$, st%, p% - st%) + rep$
    st% = p% + Len(needle$) - 1
  Loop
  Cat s$, Mid$(haystack$, st%)
  str.replace$ = s$
End Function

' Gets a string padded to a given width with spaces to the right.
'
' @param  s$  the string.
' @param  w%  the width.
' @return     the padded string.
'             If Len(s$) > w% then returns the unpadded string.
Function str.rpad$(s$, x%)
  str.rpad$ = s$
  If Len(s$) < x% Then str.rpad$ = s$ + Space$(x% - Len(s$))
End Function

' Returns a copy of s$ with leading and trailing spaces removed.
Function str.trim$(s$)
  Local st%, en%
  For st% = 1 To Len(s$)
    If Peek(Var s$, st%) <> 32 Then Exit For
  Next
  For en% = Len(s$) To 1 Step -1
    If Peek(Var s$, en%) <> 32 Then Exit For
  Next
  If en% >= st% Then str.trim$ = Mid$(s$, st%, en% - st% + 1)
End Function

' If s$ both begins and ends with " then returns a copy of s$ with those characters removed,
' otherwise returns an unmodified copy of s$.
Function str.unquote$(s$)
  str.unquote$ = s$
  If Len(s$) < 2 Then Exit Function
  If Peek(Var s$, 1) <> 34 Then Exit Function
  If Peek(var s$, Len(s$)) = 34 Then
    str.unquote$ = Mid$(s$, 2, Len(s$) - 2)
  EndIf
End Function
' END:   #Include "/home/thwill/github/cmm2-sptools/src/sptrans/tests/../../splib/string.inc"
' BEGIN: #Include #Include "../../splib/file.inc" ------------------------------
' Copyright (c) 2020-2023 Thomas Hugo Williams
' License MIT <https://opensource.org/licenses/MIT>
' For MMBasic 5.07

On Error Skip 1 : Dim sys.VERSION$ = ""
If sys.VERSION$ = "" Then Error "'system.inc' not included"
sys.provides("file")
If sys.err$ <> "" Then Error sys.err$

If InStr(Mm.Device$, "PicoMite") > 0 Then
  ' TODO: this is just a placeholder.
  Const file.PROG_DIR$ = "A:/"
Else
  Const file.PROG_DIR$ = file.get_parent$(Mm.Info$(Current))
EndIf

' Gets the number of files in directory d$ whose names match pattern$.
'
' @param   d$        directory to process.
' @param   pattern$  file pattern to match on, e.g. "*.bas".
' @param   type$     type of files to return, "dir", "file" or "all".
' @return  the number of matching files.
Function file.count_files%(d$, pattern$, type$)
  If Not file.is_directory%(d$) Then
    file.count_files% = sys.error%(sys.FAILURE, "Not a directory '" + d$ + "'")
    Exit Function
  EndIf

  Local f$
  Select Case LCase$(type$)
    Case "all"      : f$ = Dir$(d$ + "/*", All)
    Case "dir"      : f$ = Dir$(d$ + "/*", Dir)
    Case "file", "" : f$ = Dir$(d$ + "/*", File)
    Case Else
      file.count_files% = sys.error%(sys.FAILURE, "Invalid file type '" + type$ + "'")
      Exit Function
    EndIf
  End Select

  Do While f$ <> ""
    If file.fnmatch%(pattern$, f$) Then Inc file.count_files%
    f$ = Dir$()
  Loop
End Function

' Does the file/directory 'f$' exist?
'
' @return 1 if the file exists, otherwise 0
Function file.exists%(f$)
  Local f_$ = f$, i%
  For i% = 1 To Len(f_$)
    If Peek(Var f_$, i%) = 92 Then Poke Var f_$, i%, 47
  Next
  Select Case f_$
    Case ""
      ' file.exists% = 0
    Case "."
      file.exists% = 1
    Case "A:", "B:" ' TODO
      file.exists% = Mm.Info(Exists Dir f$ + "/")
    Case Else
      If Mm.Device$ = "MMB4L" Or Mm.Device$ = "MMBasic for Windows" Then
        file.exists% = Mm.Info(Exists f_$)
      Else
        file.exists% = Mm.Info(Exists File f_$) Or Mm.Info(Exists Dir f_$)
      EndIf
  End Select
End Function

' Find files whose names match pattern$
'
' @param   path$     root directory to start looking from.
' @param   pattern$  file pattern to match on, e.g. "*.bas".
' @param   type$     type of files to return, "dir", "file" or "all".
' @return  the absolute path to the first match found. To retrieve subsequent
'          matches call this function with no parameters, i.e. f$ = file.find().
'          Returns the empty string if there are no more matches.
'
' TODO: In order to return the files in alphabetical order this uses a 64-256K
'       workspace. I think there is an alternative implementation where it
'       just records the last file returned by the previous call and restarts
'       the search from there, however that would be much slower and probably
'       better implemented as a CSUB.
'       Alternatively need to use a more efficient way to store strings
'       (a pool?), or take advantage of the fact they have common prefixes.
Function file.find$(path$, pattern$, type$)
  ' ~256K workspace for MMB4L and MMB4W, ~64K workspace for other platforms.
  If sys.is_device%("mmb4l", "mmb4w") Then
    Static stack$(1000) Length 255
  Else
    Static stack$(500) Length 128
  EndIf
  Static stack_ptr%, pattern_$, type_$
  Local f$, is_dir%, lb%, name$, num%

  If path$ <> "" Then
    stack_ptr% = Bound(stack$(), 0)
    pattern_$ = Choice(pattern$ = "", "*", pattern$)
    type_$ = Choice(type$ = "", "FILE", LCase$(type$))
    f$ = file.get_canonical$(path$)
    stack$(stack_ptr%) = f$ : Inc stack_ptr%
    If Not InStr("|all|dir|file|", "|" + type_$ + "|") Then
      Error "Invalid file type '" + type_$ + "'"
      Exit Function ' So we stop even if errors are being ignored.
    EndIf
  EndIf

  Do
    Do
      If stack_ptr% = Mm.Info(Option Base) Then file.find$ = "" : Exit Function
      Inc stack_ptr%, -1
      file.find$ = stack$(stack_ptr%)
      name$ = file.get_name$(file.find$)
    Loop Until name$ <> ".git" ' Doesn't recurse into .git directories.

    ' If it is a directory and not a symbolic-link then expand it.
    is_dir% = 0
    If file.is_directory%(file.find$) Then
      If Not file.is_symlink%(file.find$) Then
        is_dir% = 1
        lb% = stack_ptr%

        ' Add all sub-directories.
        f$ = Dir$(file.find$ + "/*", Dir)
        Do While f$ <> ""
          If f$ = ".." Then
            Error("DIR$() returned '..'")
            Exit Function
          ElseIf stack_ptr% > Bound(stack$(), 1) Then
            Error "Too many files"
            Exit Function ' So we stop even if errors are being ignored.
          EndIf
          stack$(stack_ptr%) = file.find$  + "/" + f$
          Inc stack_ptr%
          f$ = Dir$()
        Loop

        ' Add all files matching pattern.
        If type_$ <> "dir" Then
          ' f$ = Dir$(file.find$ + "/*", File)
          f$ = Dir$(file.find$ + "/" + pattern_$, File)
          Do While f$ <> ""
            If f$ = ".." Then
              Error("DIR$() returned '..'")
              Exit Function
            ElseIf stack_ptr% > Bound(stack$(), 1) Then
              Error "Too many files"
              Exit Function ' So we stop even if errors are being ignored.
            EndIf
            ' If file.fnmatch%(pattern_$, f$) Then
            stack$(stack_ptr%) = file.find$  + "/" + f$
            Inc stack_ptr%
            ' EndIf
            f$ = Dir$()
          Loop
        EndIf

        ' Sort the newly pushed dirs/files so that those beginning 'a|A'
        ' are at the top and those beginning 'z|Z' are near the bottom.
        num% = stack_ptr% - lb%
        If num% > 0 Then Sort stack$(), , &b11, lb%, num%
      EndIf
    EndIf

    If is_dir% Then
      If InStr("all|dir", type_$) Then
        If file.fnmatch%(pattern_$, name$) Then Exit Do
      EndIf
    Else
      If InStr("all|file", type_$) Then Exit Do
    EndIf

  Loop

End Function

' Does name$ match pattern$ ?
'
' @param   pattern$  *nix style 'shell wildcard' pattern.
' @param   name$     the name to test.
' @return  1 if the name matches the pattern otherwise 0.
'
' Derived from the work of Russ Cox: https://research.swtch.com/glob
Function file.fnmatch%(pattern$, name$)
  Local p$ = UCase$(pattern$)
  Local n$ = UCase$(name$)
  Local c%, px% = 1, nx% = 1, nextPx% = 0, nextNx% = 0

  Do While px% <= Len(p$) Or nx% <= Len(n$)

    If px% <= Len(p$) Then

      c% = Peek(Var p$, px%)
      Select Case c%
        Case 42 ' *
          ' Zero-or-more-character wildcard
          ' Try to match at sx%,
          ' if that doesn't work out,
          ' restart at nx%+1 next.
          nextPx% = px%
          nextNx% = nx% + 1
          Inc px%
          Goto file.fnmatch_cont

        Case 63 ' ?
          ' Single-character wildcard
          If nx% <= Len(n$) Then
            Inc px%
            Inc nx%
            Goto file.fnmatch_cont
          EndIf

        Case Else
          ' Ordinary character
          If nx% <= Len(n$) Then
            If c% = Peek(Var n$, nx%) Then
              Inc px%
              Inc nx%
              Goto file.fnmatch_cont
            EndIf
          EndIf
      End Select

    EndIf

    If nextNx% > 0 Then
      If nextNx% <= Len(n$) + 1 Then
        px% = nextPx%
        nx% = nextNx%
        Goto file.fnmatch_cont
      EndIf
    EndIf

    Exit Function

    file.fnmatch_cont:

  Loop

  file.fnmatch% = 1

End Function

' Gets the canonical path for file/directory 'f$'.
Function file.get_canonical$(f$)
  Local tmp$ = f$
  Local current$ = Cwd$
  If InStr("/\", Right$(current$, 1)) Then current$ = Left$(current$, Len(current$) - 1)

  Select Case Left$(tmp$, 1)
    Case "/", "\"
      tmp$ = "A:" + tmp$
    Case "~"
      If tmp$ = "~" Then
        tmp$ = sys.string_prop$("home")
      ElseIf Left$(tmp$, 2) = "~/" Or Left$(tmp$, 2) = "~\" Then
        tmp$ = sys.string_prop$("home") + Mid$(tmp$, 2)
      Else
        tmp$ = Cwd$ + "/" + tmp$
      EndIf
    Case "a" To "z", "A" To "Z"
      If Mid$(tmp$, 2, 1) = ":" Then
        ' Do nothing
      Else
        tmp$ = Cwd$ + "/" + tmp$
      EndIf
    Case Else
      tmp$ = Cwd$ + "/" + tmp$
  End Select

  ' Convert backslash to slash to simplify algorithm.
  ' On Windows we convert back to backslash before returning the result.
  Local i%
  For i% = 1 To Len(tmp$)
    If Peek(Var tmp$, i%) = 92 Then Poke Var tmp$, i%, 47
  Next

  ' Capitalise first letter of a drive specification.
  If Mid$(tmp$, 2, 1) = ":" Then Poke Var tmp$, 1, Asc(UCase$(Chr$(Peek(Var tmp$, 1))))

  ' Ensure there is a trailing slash,
  ' this helps with the next step and is ultimately stripped off.
  If Right$(tmp$, 1) <> "/" Then Cat tmp$, "/"

  ' Process dot, dot-dot and consecutive slash characters.
  file.get_canonical$ = tmp$
  tmp$ = ""
  Local ch%, j%, s$ = ""
  For i% = 1 To Len(file.get_canonical$) + 1
    ch% = Peek(Var file.get_canonical$, i%)
    Select Case ch%
      Case 47 ' /
        Select Case s$
          Case "/", "/."
            ' Do nothing
          Case "/.."
            ' Back track
            For j% = Len(tmp$) To 1 Step -1
              If Peek(Var tmp$, j%) = 47 Then
                Poke Var tmp$, 0, j% - 1 : Exit For
              EndIf
            Next
          Case Else
            Cat tmp$, s$
        End Select
        s$ = "/"
      Case Else
        Cat s$, Chr$(ch%)
    End Select
  Next

  ' On Windows convert slash to backslash.
  If Mm.Device$ = "MMBasic for Windows" Then
    For i% = 1 To Len(tmp$)
      If Peek(Var tmp$, i%) = 47 Then Poke Var tmp$, i%, 92
    Next
  EndIf

  file.get_canonical$ = tmp$
End Function

' Gets the dot file-extension, from filename f$.
' e.g. file.get_extension("foo.bas") => ".bas"
Function file.get_extension$(f$)
  Local i%
  For i% = Len(f$) To 1 Step -1
    Select Case Peek(Var f$, i%)
      Case 46     ' .
        file.get_extension$ = Mid$(f$, i%)
        Exit Function
      Case 47, 92 ' / or \
        Exit For
    End Select
  Next
End Function

' Gets the files in directory d$ whose names match pattern$.
'
' @param       d$        directory to process.
' @param       pattern$  file pattern to match on, e.g. "*.bas".
' @param       type$     type of files to return, "dir", "file" or "all".
' @param[out]  out$()    the names of matching files are copied into this array.
' @return                the number of files matching the pattern.
Function file.get_files%(d$, pattern$, type$, out$())
  If Not file.is_directory%(d$) Then
    file.get_files% = sys.error%(sys.FAILURE, "Not a directory '" + d$ + "'")
    Exit Function
  EndIf

  Local f$
  Select Case LCase$(type$)
    Case "all"      : f$ = Dir$(d$ + "/*", All)
    Case "dir"      : f$ = Dir$(d$ + "/*", Dir)
    Case "file", "" : f$ = Dir$(d$ + "/*", File)
    Case Else
      file.get_files% = sys.error%(sys.FAILURE, "Invalid file type '" + type$ + "'")
      Exit Function
  End Select

  Do While f$ <> ""
    If file.fnmatch%(pattern$, f$) Then
      out$(file.get_files% + Mm.Info(Option Base)) = f$
      Inc file.get_files%
    EndIf
    f$ = Dir$()
  Loop

  If file.get_files% > 0 Then
    Sort out$(), , &b10, Mm.Info(Option Base), file.get_files%
  EndIf
End Function

' Gets the name of file/directory 'f$' minus any path information.
Function file.get_name$(f$)
  Local i%

  For i% = Len(f$) To 1 Step -1
    If InStr("/\", Mid$(f$, i%, 1)) > 0 Then Exit For
  Next i%

  file.get_name$ = Mid$(f$, i% + 1)
End Function

' Gets the parent directory of 'f$', or the empty string if it does not have one.
Function file.get_parent$(f$)
  Local i%

  For i% = Len(f$) To 1 Step -1
    If InStr("/\", Mid$(f$, i%, 1)) > 0 Then Exit For
  Next i%

  If i% > 0 Then file.get_parent$ = Left$(f$, i% - 1)
End Function

Function file.is_absolute%(f$)
  Select Case Left$(f$, 1)
    Case "a", "A"
      Select Case Mid$(f$, 2, 2)
        Case ":", ":\", ":/" : file.is_absolute% = 1
      End Select
    Case "/", "\"
      file.is_absolute% = 1
  End Select
End Function

Function file.is_directory%(f$)
  ' TODO: handle ~ prefix
  Local f_$ = f$, i%
  For i% = 1 To Len(f_$)
    If Peek(Var f_$, i%) = 92 Then Poke Var f_$, i%, 47
  Next
  Select Case f_$
    Case ""
      ' file.is_directory% = 0
    Case ".", "/"
      file.is_directory% = 1
    Case "A:", "B:" ' TODO
      file.is_directory% = Mm.Info(Exists Dir f_$ + "/")
    Case Else
      file.is_directory% = Mm.Info(Exists Dir f_$)
  End Select
End Function

Function file.is_symlink%(f$)
  If Mm.Device$ = "MMB4L" Then
    file.is_symlink% = Mm.Info(Exists SymLink f$)
  EndIf
End Function

' Makes directory 'f$' and its parents if they do not already exist.
Function file.mkdir%(f$)
  Local ad%, faddr% = Peek(VarAddr f$), parent$
  sys.err$ = ""
  For ad% = faddr% To faddr% + Len(f$)
    Select Case Peek(Byte ad%)
      Case 47, 92 ' / and \
        parent$ = Left$(f$, ad% - faddr% - 1)
        If file.exists%(parent$) Then
          If Not file.is_directory%(parent$) Then
            file.mkdir% = sys.error%(sys.FAILURE, "File exists")
            Exit Function
          EndIf
        Else
          If parent$ <> "" Then MkDir parent$
        EndIf
    End Select
  Next
  If file.exists%(f$) Then
    If Not file.is_directory%(f$) Then file.mkdir% = sys.error%(sys.FAILURE, "File exists")
  Else
    MkDir f$
  EndIf
End Function

' Performs depth-first recursion of the filesystem from a given root$
' calling the supplied callback$ function for each file encountered.
'
' @param  root$      Root of the traversal.
' @param  callback$  Callback function to call, first parameter should be a string (the absolute
'                    filename of the file node being processed) and return 0 on success.
' @param  xtra%      Value to be passed as second argument to callback.
'                    This could be used to pass the address of a value other than an integer.
' @return            0 on success, or if a callback function returns a non-zero value then the
'                    traversal stops and that value is returned.
Function file.depth_first%(root$, callback$, xtra%)
  Local root_$ = file.get_canonical$(root$)
  If Not file.exists%(root_$) Then
    file.depth_first% = sys.error%(sys.FAILURE, "No such file or directory '" + root_$ + "'")
    Exit Function
  EndIf

  Local depth% = 0, current$ = root_$, name$, parent$
  Local f$ = Choice(file.is_directory%(root_$) And Not file.is_symlink%(root_$), Dir$(root_$ + "/*", All), "")
  Do While file.depth_first% = sys.SUCCESS
    If f$ = ".." Then
      Error("DIR$() returned '..'")
      Exit Function
    ElseIf f$ = "" Then ' No more entries.
      If current$ <> root_$ Then
        ' Protection from accidentally navigating above root$ which would
        ' have disasterous consequences if performing a recursive delete.
        Inc depth%, -1
        If depth% < 0 Then
          Error "Safety check failed!"
          Exit Function ' So we stop even if errors are being ignored.
        EndIf

        ' Update f$ to point to the entry after current$ in its parent;
        ' traversal will continue from there.
        name$ = file.get_name$(current$)
        parent$ = file.get_parent$(current$)
        f$ = Dir$(parent$ + "/*", All)
        Do While f$ <> name$ : f$ = Dir$() : Loop
        f$ = Dir$()
      EndIf

      ' Note that unless we are acting on the root$ we will have navigated to
      ' the parent before calling the callback$, this is essential if the
      ' callback deletes or renames the directory.
      file.depth_first% = Call(callback$, current$, xtra%)
      If current$ = root_$ Then Exit Do
      current$ = parent$
    Else
      f$ = current$ + "/" + f$
      If file.is_symlink%(f$) Then
        file.depth_first% = Call(callback$, f$, xtra%)
        f$ = Dir$()
      ElseIf file.is_directory%(f$) Then
        current$ = f$
        f$ = Dir$(current$ + "/*", All)
        Inc depth%
      Else
        file.depth_first% = Call(callback$, f$, xtra%)
        f$ = Dir$()
      EndIf
    EndIf
  Loop
End Function

Function file.delete%(f$, yes_to_all%)
  Local f_$ = file.get_canonical$(f$), is_root%
  Select Case Len(f_$)
    Case 1 : is_root% = InStr("/\", f_$)
    Case 2 : is_root% = Mid$(f_$, 2, 1) = ":"
    Case 3 : is_root% = Mid$(f_$, 2, 1) = ":" And InStr("/\", Mid$(f_$, 3, 1))
  End Select
  If is_root% Then
    file.delete% = sys.error%(sys.FAILURE, "Cannot delete '" + f_$ + "'")
  Else
    file.delete% = file.depth_first%(f_$, "file.delete_callback%", yes_to_all%)
  EndIf
End Function

Function file.delete_callback%(f$, yes_to_all%)
  Local s$ = Choice(yes_to_all%, "y", "n")
  If s$ = "n" Then Line Input "Delete '" + f$ + "' [y|N] ? ", s$
  If LCase$(s$) = "y" Then
    If file.is_directory%(f$) And Not file.is_symlink%(f$) Then RmDir f$ Else Kill f$
  EndIf
End Function

' Returns a copy of f$ with any dot file-extension removed.
' e.g. file.trim_extension("foo.bas") => "foo"
Function file.trim_extension$(f$)
  Local i%
  For i% = Len(f$) To 1 Step -1
    Select Case Peek(Var f$, i%)
      Case 46     ' .
        file.trim_extension$ = Mid$(f$, 1, i% - 1)
        Exit Function
      Case 47, 92 ' / or \
        Exit For
    End Select
  Next
  file.trim_extension$ = f$
End Function
' END:   #Include "/home/thwill/github/cmm2-sptools/src/sptrans/tests/../../splib/file.inc"
' BEGIN: #Include #Include "../../splib/map.inc" -------------------------------
' Copyright (c) 2020-2023 Thomas Hugo Williams
' License MIT <https://opensource.org/licenses/MIT>
' For MMBasic 5.07

On Error Skip 1 : Dim sys.VERSION$ = ""
If sys.VERSION$ = "" Then Error "'system.inc' not included"
sys.requires("array")
sys.provides("map")
If sys.err$ <> "" Then Error sys.err$

' Gets the upper-bound that should be used to dimension an array to hold a map
' of the given capacity, irrespective of OPTION BASE.
'
' e.g. To create a string array that can be used as a map of 10 elements:
'        Dim my_map$(map.new%(10))
Function map.new%(capacity%)
  map.new% = 2 * capacity% + Mm.Info(Option Base)
End Function

' Initialises the map.
Sub map.init(mp$())
  Local lb% = Bound(mp$(), 0)
  Local ub% = Bound(mp$(), 1)
  Local i%
  For i% = lb% To ub% : mp$(i%) = sys.NO_DATA$ : Next
  mp$(ub%) = "0"
End Sub

' Gets the capacity of the map.
Function map.capacity%(mp$())
  map.capacity% = (Bound(mp$(), 1) - Bound(mp$(), 0)) \ 2
End Function

' Clears the keys and values and resets the size to 0.
Sub map.clear(mp$())
  Local lb% = Bound(mp$(), 0)
  Local ub% = Bound(mp$(), 1)
  Local off% = (ub% - lb%) \ 2
  Local sz% = Val(mp$(ub%))
  Local i%
  For i% = lb% To lb% + sz% - 1
    mp$(i%) = sys.NO_DATA$
    mp$(i% + off%) = sys.NO_DATA$
  Next
  mp$(ub%) = "0"
End Sub

' Prints the contents of the map.
Sub map.dump(mp$())
  Local lb% = Bound(mp$(), 0)
  Local ub% = Bound(mp$(), 1)
  Local off% = (ub% - lb%) \ 2
  Local sz% = Val(mp$(ub%))
  Local i%, length%
  For i% = lb% To lb% + sz% - 1 : length% = Max(length%, Len(mp$(i%))) : Next
  For i% = lb% To lb% + sz% - 1
    Print "[" Str$(i%) "] ";
    Print mp$(i%);
    Print Space$(length% - Len(mp$(i%)));
    Print " => ";
    Print mp$(i% + off%)
  Next
  Print "END"
End Sub

' Gets the value corresponding to a key, or sys.NO_DATA$ if the key is not present.
Function map.get$(mp$(), k$)
  Local lb% = Bound(mp$(), 0)
  Local ub% = Bound(mp$(), 1)
  Local off% = (ub% - lb%) \ 2
  Local sz% = Val(mp$(ub%))

  Local i% = array.bsearch%(mp$(), k$, "", lb%, sz%)
  If i% > -1 Then
    map.get$ = mp$(i% + off%)
  Else
    map.get$ = sys.NO_DATA$
  EndIf
End Function

Function map.get_key_index%(mp$(), k$)
  map.get_key_index% = array.bsearch%(mp$(), k$, "", Bound(mp$(), 0), Val(mp$(Bound(mp$(), 1))))
End Function

Function map.is_full%(map$())
  Local ub% = Bound(map$(), 1)
  '             = (ub% - lb%) \ 2 = sz%
  map.is_full% = (ub% - Bound(map$(), 0)) \ 2 = Val(map$(ub%))
End Function


' Adds a key/value pair.
Sub map.put(mp$(), k$, v$)
  Local lb% = Bound(mp$(), 0)
  Local ub% = Bound(mp$(), 1)
  Local off% = (ub% - lb%) \ 2
  Local sz% = Val(mp$(ub%))

  Local i% = array.bsearch%(mp$(), k$, "", lb%, sz%)
  If i% <> -1 Then mp$(i% + off%) = v$ : Exit Sub

  If sz% = off% Then Error "map full" : Exit Sub

  sz% = sz% + 1
  mp$(lb% + sz% - 1) = k$
  mp$(lb% + sz% + off% - 1) = v$
  mp$(ub%) = Str$(sz%)
  If sz% > 1 Then
    If k$ < mp$(lb% + sz% - 2) Then map.sort(mp$())
  EndIf
End Sub

' Removes a key/value pair.
Sub map.remove(mp$(), k$)
  Local lb% = Bound(mp$(), 0)
  Local ub% = Bound(mp$(), 1)
  Local off% = (ub% - lb%) \ 2
  Local sz% = Val(mp$(ub%))

  Local i% = array.bsearch%(mp$(), k$, "", lb%, sz%)
  If i% = -1 Then Exit Sub

  mp$(i%) = sys.NO_DATA$
  mp$(i% + off%) = sys.NO_DATA$
  If sz% > 1 Then map.sort(mp$())
  mp$(ub%) = Str$(sz% - 1)
End Sub

' Gets the size / number of entries in the map.
Function map.size%(mp$())
  map.size% = Val(mp$(Bound(mp$(), 1)))
End Function

' Sorts the map to place the keys in ascending order.
Sub map.sort(mp$())
  Local lb% = Bound(mp$(), 0)
  Local ub% = Bound(mp$(), 1)
  Local off% = (ub% - lb%) \ 2
  Local sz% = Val(mp$(ub%))
  Local idx%(map.new%(off%))

  ' Sort keys.
  Sort mp$(), idx%(), , lb%, sz%

  ' Copy values to tmp$().
  Local i%
  Select Case Mm.Device$
    Case "PicoMite", "PicoMiteVGA"
      Local tmp$(map.new%(sz%)) Length Peek(Byte Peek(VarHeader mp$()) + 34)
    Case Else
      Local tmp$(map.new%(sz%))
  End Select
  ub% = lb% + sz% - 1
  For i% = lb% To ub% : tmp$(i%) = mp$(i% + off%) : Next

  ' Copy re-ordered values back into mp$().
  For i% = lb% To ub% : mp$(i% + off%) = tmp$(idx%(i%)) : Next
End Sub
' END:   #Include "/home/thwill/github/cmm2-sptools/src/sptrans/tests/../../splib/map.inc"
' BEGIN: #Include #Include "../../splib/set.inc" -------------------------------
' Copyright (c) 2020-2023 Thomas Hugo Williams
' License MIT <https://opensource.org/licenses/MIT>
' For MMBasic 5.07

On Error Skip 1 : Dim sys.VERSION$ = ""
If sys.VERSION$ = "" Then Error "'system.inc' not included"
sys.requires("array")
sys.provides("set")
If sys.err$ <> "" Then Error sys.err$

' Gets the upper-bound that should be used to dimension an array to hold a set
' of the given capacity, irrespective of OPTION BASE.
'
' e.g. To create a string array that can be used as a set of 10 elements:
'        Dim my_set$(set.new%(10))
Function set.new%(capacity%)
  set.new% = capacity% + Mm.Info(Option Base)
End Function

' Gets the capacity of the set.
Function set.capacity%(set$())
  set.capacity% = Bound(set$(), 1) - Bound(set$(), 0)
End Function

' Initialises the set.
Sub set.init(set$())
  Local lb% = Bound(set$(), 0)
  Local ub% = Bound(set$(), 1)
  Local i%
  For i% = lb% To ub% : set$(i%) = sys.NO_DATA$ : Next
  set$(ub%) = "0"
End Sub

' Clears the set and resets its size to 0.
Sub set.clear(set$())
  Local lb% = Bound(set$(), 0)
  Local ub% = Bound(set$(), 1)
  Local sz% = Val(set$(ub%))
  Local i%
  For i% = lb% To lb% + sz% - 1 : set$(i%) = sys.NO_DATA$ : Next
  set$(ub%) = "0"
End Sub

' Prints the contents of the set.
Sub set.dump(set$())
  Local lb% = Bound(set$(), 0)
  Local ub% = Bound(set$(), 1)
  Local sz% = Val(set$(ub%))
  Local i%
  For i% = lb% To lb% + sz% - 1 : Print "[" Str$(i%) "] " set$(i%) : Next
  Print "END"
End Sub

' Gets the index of a value in the set, or -1 if not present.
Function set.get%(set$(), s$)
  set.get% = array.bsearch%(set$(), s$, "", Bound(set$(), 0), Val(set$(Bound(set$(), 1))))
End Function

Function set.is_full%(set$())
  Local ub% = Bound(set$(), 1)
  '             = (ub% - lb%) = sz%
  set.is_full% = (ub% - Bound(set$(), 0)) = Val(set$(ub%))
End Function

' Adds a value to the set.
Sub set.put(set$(), s$)
  Local lb% = Bound(set$(), 0)
  Local ub% = Bound(set$(), 1)
  Local sz% = Val(set$(ub%))

  If array.bsearch%(set$(), s$, "", lb%, sz%) <> -1 Then Exit Sub

  If sz% = ub% - lb% Then Error "set full"

  set$(lb% + sz%) = s$
  sz% = sz% + 1
  set$(ub%) = Str$(sz%)

  If sz% <= 1 Then Exit Sub
  If set$(lb% + sz% - 1) >= set$(lb% + sz% - 2) Then Exit Sub

  Sort set$(), , , lb%, sz%
End Sub

' Removes a value from the set if present.
Sub set.remove(set$(), s$)
  Local lb% = Bound(set$(), 0)
  Local ub% = Bound(set$(), 1)
  Local sz% = Val(set$(ub%))

  Local i% = array.bsearch%(set$(), s$, "", lb%, sz%)
  If i% = -1 Then Exit Sub

  set$(i%) = sys.NO_DATA$
  set$(ub%) = Str$(sz% - 1)

  Sort set$(), , , lb%, sz%
End Sub

' Gets the size of the set.
Function set.size%(set$())
  set.size% = Val(set$(Bound(set$(), 1)))
End Function
' END:   #Include "/home/thwill/github/cmm2-sptools/src/sptrans/tests/../../splib/set.inc"
' BEGIN: #Include #Include "../../splib/vt100.inc" -----------------------------
' Copyright (c) 2020-2023 Thomas Hugo Williams
' License MIT <https://opensource.org/licenses/MIT>
' For MMBasic 5.07

On Error Skip 1 : Dim sys.VERSION$ = ""
If sys.VERSION$ = "" Then Error "'system.inc' not included"
sys.provides("vt100")
If sys.err$ <> "" Then Error sys.err$

Function vt100$(s$)
  vt100$ = Chr$(27) + "[" + s$
End Function

Function vt100.colour$(c$)
  Select Case LCase$(c$)
    Case "black"   : vt100.colour$ = vt100$("30m")
    Case "red"     : vt100.colour$ = vt100$("31m")
    Case "green"   : vt100.colour$ = vt100$("32m")
    Case "yellow"  : vt100.colour$ = vt100$("33m")
    Case "blue"    : vt100.colour$ = vt100$("34m")
    Case "magenta", "purple" : vt100.colour$ = vt100$("35m")
    Case "cyan"    : vt100.colour$ = vt100$("36m")
    Case "white"   : vt100.colour$ = vt100$("37m")
    Case "reset"   : vt100.colour$ = vt100$("0m")
    Case Else      : Error "Unknown VT100 colour: " + c$
  End Select
End Function

' END:   #Include "/home/thwill/github/cmm2-sptools/src/sptrans/tests/../../splib/vt100.inc"
' BEGIN: #Include #Include "../../sptest/unittest.inc" -------------------------
' Copyright (c) 2020-2023 Thomas Hugo Williams
' License MIT <https://opensource.org/licenses/MIT>
' For MMBasic 5.07

On Error Skip 1 : Dim sys.VERSION$ = ""
If sys.VERSION$ = "" Then Error "'system.inc' not included"
sys.requires("array", "file", "list", "string", "vt100")
sys.provides("unittest")
If sys.err$ <> "" Then Error sys.err$

If Mm.Device$ = "MMBasic for Windows" Then Option Console Both

Const ut.MAX_FAILURES% = 20 ' Maximum number of failures reported per test method.
Const ut.MAX_TESTS% = 100
Const ut.MAX_ARRAY_DIFFS% = 5

Dim ut.asserts_count%
Dim ut.failure_count%
Dim ut.failure_len%(array.new%(ut.MAX_FAILURES%))
Dim ut.failure_start%(array.new%(ut.MAX_FAILURES%))
Dim ut.failure_txt%(1000)
Dim ut.test_names$(list.new%(ut.MAX_TESTS%)) Length 128
Dim ut.test_subs$(list.new%(ut.MAX_TESTS%)) Length 40

list.init(ut.test_names$())
list.init(ut.test_subs$())

Sub add_test(name$, sub$)
  list.add(ut.test_names$(), name$)
  list.add(ut.test_subs$(), Choice(sub$ = "", name$, sub$))
End Sub

Sub ut.add_failure(msg$)
  Inc ut.failure_count%
  If ut.failure_count% <= ut.MAX_FAILURES% Then
    ut.failure_start%(ut.failure_count% + Mm.Info(Option Base) - 1) = LLen(ut.failure_txt%()) + 1
    LongString Append ut.failure_txt%(), Str$(ut.asserts_count%) + ": " + msg$
    ut.update_failure()
  EndIf
End Sub

Sub ut.update_failure()
  If ut.failure_count% <= ut.MAX_FAILURES% Then
    Local i% = ut.failure_count% + Mm.Info(Option Base) - 1
    ut.failure_len%(i%) = LLen(ut.failure_txt%()) - ut.failure_start%(i%) + 1
  EndIf
End Sub

Sub assert_fail(msg$)
  Inc ut.asserts_count%
  ut.add_failure(msg$)
End Sub

Sub assert_false(z%, msg$)
  Inc ut.asserts_count%
  If z% Then ut.add_failure(Choice(msg$ = "", "assert_false() failed", msg$))
End Sub

Sub assert_true(z%, msg$)
  Inc ut.asserts_count%
  If Not z% Then ut.add_failure(Choice(msg$ = "", "assert_true() failed", msg$))
End Sub

Sub assert_hex_equals(expected%, actual%, chars%)
  Inc ut.asserts_count%
  If expected% <> actual% Then
    Local s$
    Cat s$, "Assert equals failed, expected &h"
    Cat s$, Choice(chars%, Hex$(expected%, chars%), Hex%(expected%))
    Cat s$, " but actually &h"
    Cat s$, Choice(chars%, Hex$(actual%, chars%), Hex%(actual%))
    ut.add_failure(s$)
  EndIf
End Sub

Sub assert_int_equals(expected%, actual%)
  Inc ut.asserts_count%
  If expected% <> actual% Then
    Local s$ = "Assert equals failed, expected " + Str$(expected%)
    s$ = s$ + " but actually " + Str$(actual%)
    ut.add_failure(s$)
  EndIf
End Sub

Sub assert_int_neq(unexpected%, actual%)
  Inc ut.asserts_count%
  If unexpected% = actual% Then
    Local s$ = "Assert not equals failed, did not expect " + Str$(unexpected%)
    ut.add_failure(s$)
  EndIf
End Sub

Sub assert_float_equals(expected!, actual!, delta!)
  Inc ut.asserts_count%
  If Not equals_float%(expected!, actual!, delta!) Then
    Local s$ = "Assert equals failed, expected " + Str$(expected!)
    s$ = s$ + " but actually " + Str$(actual!)
    ut.add_failure(s$)
  EndIf
End Sub

Function equals_float%(a!, b!, delta!)
  equals_float% = (a! >= b! - delta!) And (a! <= b! + delta!)
End Function

Sub assert_string_equals(expected_$, actual_$)
  Inc ut.asserts_count%
  If expected_$ <> actual_$ Then
    Local expected$ = str.quote$(expected_$)
    Local actual$ = str.quote$(actual_$)
    If Len(expected_$) = 1 Then expected$ = "Chr$(" + Str$(Asc(expected_$)) + ")"
    If Len(actual_$) = 1 Then actual$ = "Chr$(" + Str$(Asc(actual_$)) + ")"

    ut.add_failure("Assert equals failed, expected ")
    LongString Append ut.failure_txt%(), ut.sanitise_string$(expected$)
    LongString Append ut.failure_txt%(), " but actually "
    LongString Append ut.failure_txt%(), ut.sanitise_string$(actual$)
    ut.update_failure()
  EndIf
End Sub

Function ut.sanitise_string$(s$)
  Local c%, i%, s2$
  For i% = 1 To Len(s$)
    c% = Peek(Var s$, i%)
    Cat s2$, Choice(c% < 32 Or c% > 126, "<" + Str$(c%) + ">", Chr$(c%))
  Next
  ut.sanitise_string$ = s2$
End Function

Sub assert_int_array_equals(expected%(), actual%())
  Inc ut.asserts_count%

  If Bound(expected%(), 1) <> Bound(actual%(), 1) Then
    ut.add_array_size_failure(Bound(expected%(), 1), Bound(actual%(), 1))
    Exit Sub
  EndIf

  Local count%, i%, j% = Mm.Info(Option Base)
  Local indexes%(array.new%(ut.MAX_ARRAY_DIFFS%))
  Local e$(array.new%(ut.MAX_ARRAY_DIFFS%))
  Local a$(array.new%(ut.MAX_ARRAY_DIFFS%))

  For i% = Bound(expected%(), 0) To Bound(expected%(), 1)
    If expected%(i%) = actual%(i%) Then Continue For
    Inc count%
    If count% > ut.MAX_ARRAY_DIFFS% Then Continue For
    indexes%(j%) = i%
    e$(j%) = Str$(expected%(i%))
    a$(j%) = Str$(actual%(i%))
    Inc j%
  Next

  If count% > 0 Then ut.add_array_content_failure(count%, indexes%(), e$(), a$())
End Sub

Sub ut.add_array_size_failure(expected_size%, actual_size%)
  Local msg$ = "Assert array equals failed, expected size "
  Cat msg$, Str$(expected_size%)
  Cat msg$, " but was "
  Cat msg$, Str$(actual_size%)
  ut.add_failure(msg$)
End Sub

Sub ut.add_array_content_failure(count%, indexes%(), e$(), a$())
  Local i%, width%
  Local lb% = Bound(indexes%(), 0)
  Local ub% = Min(count% + lb% - 1, Bound(indexes%(), 1))
  For i% = lb% To ub%
    width% = Max(width%, Len(Str$(indexes%(i%))))
  Next

  ut.add_failure("Assert array equals failed, expected:")
  For i% = lb% To ub%
    ut.array_diff(width%, indexes%(i%), e$(i%))
  Next
  ut.and_more(count%)

  LongString Append ut.failure_txt%(), sys.CRLF$ + "    but actually:")
  For i% = lb% To ub%
    ut.array_diff(width%, indexes%(i%), a$(i%))
  Next
  ut.and_more(count%)

  ut.update_failure()
End Sub

Sub ut.array_diff(width%, index%, value$)
  LongString Append ut.failure_txt%(), sys.CRLF$
  LongString Append ut.failure_txt%(), Format$(index%, "      [%" + Str$(width%) + ".0f] ")
  LongString Append ut.failure_txt%(), value$
End Sub

Sub ut.and_more(count%)
  If count% <= ut.MAX_ARRAY_DIFFS% Then Exit Sub
  LongString Append ut.failure_txt%(), sys.CRLF$
  LongString Append ut.failure_txt%(), "      ... and "
  LongString Append ut.failure_txt%(), Str$(count% - ut.MAX_ARRAY_DIFFS%)
  LongString Append ut.failure_txt%(), " more"
End Sub

Sub assert_float_array_equals(expected!(), actual!(), delta!())
  Inc ut.asserts_count%

  If Bound(expected!(), 1) <> Bound(actual!(), 1) Then
    ut.add_array_size_failure(Bound(expected!(), 1), Bound(actual!(), 1))
    Exit Sub
  EndIf

  Local count%, i%, j% = Mm.Info(Option Base)
  Local indexes%(array.new%(ut.MAX_ARRAY_DIFFS%))
  Local e$(array.new%(ut.MAX_ARRAY_DIFFS%))
  Local a$(array.new%(ut.MAX_ARRAY_DIFFS%))

  For i% = Bound(expected!(), 0) To Bound(expected!(), 1)
    If equals_float%(expected!(i%), actual!(i%), delta!(i%)) Then Continue For
    Inc count%
    If count% > ut.MAX_ARRAY_DIFFS% Then Continue For
    indexes%(j%) = i%
    e$(j%) = Str$(expected!(i%))
    a$(j%) = Str$(actual!(i%))
    Inc j%
  Next

  If count% > 0 Then ut.add_array_content_failure(count%, indexes%(), e$(), a$())
End Sub

Sub assert_string_array_equals(expected$(), actual$())
  Inc ut.asserts_count%

  If Bound(expected$(), 1) <> Bound(actual$(), 1) Then
    ut.add_array_size_failure(Bound(expected$(), 1), Bound(actual$(), 1))
    Exit Sub
  EndIf

  Local count%, i%, j% = Mm.Info(Option Base)
  Local indexes%(array.new%(ut.MAX_ARRAY_DIFFS%))
  Local e$(array.new%(ut.MAX_ARRAY_DIFFS%))
  Local a$(array.new%(ut.MAX_ARRAY_DIFFS%))

  For i% = Bound(expected$(), 0) To Bound(expected$(), 1)
    If expected$(i%) = actual$(i%) Then Continue For
    Inc count%
    If count% > ut.MAX_ARRAY_DIFFS% Then Continue For
    indexes%(j%) = i%
    e$(j%) = str.quote$(ut.sanitise_string$(expected$(i%)))
    a$(j%) = str.quote$(ut.sanitise_string$(actual$(i%)))
    Inc j%
  Next

  If count% > 0 Then ut.add_array_content_failure(count%, indexes%(), e$(), a$())
End Sub

Sub assert_no_error()
  Inc ut.asserts_count%
  If sys.err$ <> "" Then
    ut.add_failure("Expected no error, but actually sys.err$ = " + str.quote$(sys.err$))
  EndIf

  ' CMM2 file operations set Mm.ErrMsg$ = "Success " including a space when they succeed.
  If Mm.ErrNo <> 0 Then
    ut.add_failure("Expected no error, but actually Mm.ErrMsg$ = " + str.quote$(Mm.ErrMsg$))
  EndIf
End Sub

Sub assert_error(expected$)
  Inc ut.asserts_count%
  If sys.err$ <> expected$ Then
    Local s$ = "Expected Error " + str.quote$(expected$) + ", but actually " + str.quote$(sys.err$)
    ut.add_failure(s$)
  EndIf
End Sub

Sub assert_raw_error(expected$)
  Inc ut.asserts_count%
  Local s$
  If Mm.ErrMsg$ = expected$ Then
    If Mm.ErrNo = 0 And expected$ <> "" Then
      Cat s$, "Expected Error " + str.quote$(expected$)
      Cat s$, ", but Mm.ErrNo = 0"
    EndIf
  Else If Not InStr(Mm.ErrMsg$, expected$) Then
    Cat s$, "Expected Error " + str.quote$(expected$)
    Cat s$, ", but actually " + str.quote$(Mm.ErrMsg$)
  EndIf
  If s$ <> "" Then ut.add_failure(s$)
End Sub

' @param  repeat$  if this is not empty then after running the test suite the program
'                  should run itself again appending 'repeat$' to its command line.
Sub run_tests(repeat$)
  Local num_failed_tests% = 0
  Local num_tests% = list.size%(ut.test_names$())
  Local test$ = Mm.Info$(Current) + " " + Mm.CmdLine$;

  Print test$;
  If ut.is_verbose%() Then Print

  Local t% = Timer

  Local i%
  Local lb% = Mm.Info(Option Base)
  Local ub% = lb% + num_tests% - 1
  For i% = lb% To ub%
    If Not ut.run_single_test%(i%) Then Inc num_failed_tests%
  Next

  If ut.is_verbose%() Then
    Print "  Execution time: " Str$((Timer - t%) / 1000) " s"
    Print "  ";
  Else
    Print Space$(Max(1, 85 - Len(test$)));
  EndIf

  If num_tests% = 0 Then
    ut.print_colour("magenta")
    Print "NO TESTS";
  Else If num_failed_tests% = 0 Then
    ut.print_colour("green")
    Print "PASS (" Str$(num_tests%) "/" Str$(num_tests%) ")";
  Else
    ut.print_colour("red")
    Print "FAIL (" Str$(num_failed_tests%) "/" Str$(num_tests%) ")";
  EndIf
  ut.print_colour("reset")
  Print

  If repeat$ <> "" Then
    ut.run_same(repeat$)
  ElseIf InStr(Mm.CmdLine$, "--all") Then
    ut.run_next()
  EndIf
End Sub

' @return 1 if the test passes, 0 if it fails.
Function ut.run_single_test%(idx%)
  Local catch_errors% = ut.is_catch_errors%()

  If ut.is_verbose%() Then
    Print "  " + ut.test_names$(idx%); ":"; Space$(Max(1, 50 - Len(ut.test_names$(idx%))));
  EndIf

  ut.asserts_count% = 0
  ut.failure_count% = 0
  LongString Clear ut.failure_txt%()

  On Error Skip 1 ' Skip error caused by missing optional setup_test() function.
  setup_test()

  On Error Clear
  If catch_errors% Then On Error Ignore
  sys.err$ = ""
  Call ut.test_subs$(idx%)
  If catch_errors% And Mm.ErrNo <> 0 Then add_mmbasic_error()
  On Error Abort

  On Error Skip 1 ' Skip error caused by missing optional teardown_test() function.
  teardown_test()

  If ut.is_verbose%() Then
    If ut.asserts_count% = 0 Then
      ut.print_colour("magenta")
      Print "NO ASSERTIONS";
    Else If ut.failure_count% = 0 Then
      ut.print_colour("green")
      Print "PASS ("; Str$(ut.asserts_count%); "/"; Str$(ut.asserts_count%); ")";
    Else If ut.failure_count% > 0 Then
      ut.print_colour("red")
      Print "FAIL ("; Str$(ut.failure_count%); "/"; Str$(ut.asserts_count%); ")";
      Local i%, j%
      Local lb% = Mm.Info(Option Base)
      Local ub% = lb% + Min(ut.failure_count%, ut.MAX_FAILURES%) - 1
      For i% = lb% To ub%
        Print
        Print "    ";
        j% = ut.failure_start%(i%)
        Do
          Print LGetStr$(ut.failure_txt%(), j%, Min(255, ut.failure_len%(i%)));
          Inc j%, Min(255, ut.failure_len%(i%))
        Loop While j% < ut.failure_len%(i%)
      Next
      If ut.failure_count% > ut.MAX_FAILURES% Then
        Print
        Print "    +" Str$(ut.failure_count% - ut.MAX_FAILURES%) + " more";
      EndIf
    EndIf
    ut.print_colour("reset")
    Print
  EndIf

  ut.run_single_test% = ut.failure_count% = 0
End Function

' Should verbose output be generated?
Function ut.is_verbose%()
  ut.is_verbose% = InStr(" " + Mm.CmdLine$ + " ", " --all ") < 1
  If Not ut.is_verbose% Then ut.is_verbose% = InStr(" " + Mm.CmdLine$ + " ", " --verbose ") > 0
End Function

' Should MMBasic errors be caught and reported as assertion failures?
Function ut.is_catch_errors%()
  ut.is_catch_errors% = InStr(" " + Mm.CmdLine$ + " ", " --catch-errors ")
End Function

Sub add_mmbasic_error()
  Inc ut.asserts_count%
  ut.add_failure("MMBasic " + Mid$(Mm.ErrMsg$, InStr(Mm.ErrMsg$, "Error")))
End Sub

' First:  creates the file "/tmp/sptest.list" containing the full paths (one per line)
'         of each sptest suite/file found by walking the file-system from the current
'         working directory.
' Second: executes the first sptest suite/file found.

Sub ut.run_first()
  Print "Building list of tests ..."
  Open sys.string_prop$("tmpdir") + "/sptest.lst" For Output As #1
  Print #1, Time$
  Local f$ = file.find$(Cwd$, "*st*.bas", "file")
  Local first$
  Do While f$ <> ""
    If ut.is_test_file%(f$) Then
      Print #1, file.get_canonical$(f$)
      If first$ = "" Then first$ = f$
    EndIf
    f$ = file.find$()
  Loop
  Close #1

  If first$ = "" Then
    Print "No tests found."
    End
  Else
    Print "Executing tests ..."
    Local cmd$ = "Run " + str.quote$(first$) + ", --all"
    If InStr(" " + Mm.CmdLine$ + " ", " --catch-errors") Then Cat cmd$, " --catch-errors"
    If InStr(" " + Mm.CmdLine$ + " ", " --verbose ")     Then Cat cmd$, " --verbose"
    Execute cmd$
  EndIf
End Sub

Function ut.is_test_file%(f$)
  Local name$ = file.get_name$(f$)

  ' Check name matches pattern for sptest files.
  If file.fnmatch%("test_*.bas", name$) Then ut.is_test_file% = 1
  If file.fnmatch%("tst_*.bas", name$) Then ut.is_test_file% = 1
  If file.fnmatch%("*_test.bas", name$) Then ut.is_test_file% = 1
  If file.fnmatch%("*_tst*.bas", name$) Then ut.is_test_file% = 1

  If Not ut.is_test_file% Then Exit Function

  ' Scan first 50 lines of file for #Include of "unittest.inc".
  Local i%, s$
  ut.is_test_file% = 0
  Open f$ For Input As #2
  For i% = 1 To 50
    If Eof(#2) Then Exit For
    Line Input #2, s$
    If file.fnmatch%("*#include*unittest.inc*", s$) Then ut.is_test_file% = 1 : Exit For
  Next
  Close #2

End Function

Function ut.parse_time%(t$)
  Inc ut.parse_time%, Val(Mid$(t$, 1, 2)) * 60 * 60
  Inc ut.parse_time%, Val(Mid$(t$, 4, 2)) * 60
  Inc ut.parse_time%, Val(Mid$(t$, 7, 2))
End Function

Sub ut.run_next()
  Local f$, start_time$
  Open sys.string_prop$("tmpdir") + "/sptest.lst" For Input As #1
  Line Input #1, start_time$

  Do
    Line Input #1, f$
    If f$ = file.get_canonical$(Mm.Info$(Current)) Then Exit Do
  Loop While Not Eof(#1)

  If Eof(#1) Then
    Close #1
    Local start% = ut.parse_time%(start_time$)
    Local end% = ut.parse_time%(Time$)
    Print "Total execution time: " + Str$(end% - start%) + " s"
  Else
    Line Input #1, f$
    Close #1
    Local cmd$ = "Run " + str.quote$(f$) + ", --all"
    If ut.is_catch_errors%() Then Cat cmd$, " --catch-errors"
    If ut.is_verbose%()      Then Cat cmd$, " --verbose"
    Execute cmd$
  EndIf
End Sub

' Runs the current test file again appending 'cmd$' to its command line.
Sub ut.run_same(cmd$)
  If sys.is_device%("cmm2") Then
    Execute "Run " + str.quote$(Mm.Info$(Current)) + ", " + Mm.CmdLine$ + " " + cmd$
  Else
    Run Mm.Info$(Current), Mm.CmdLine$ + " " + cmd$
  EndIf
End Sub

Sub ut.print_colour(c$)
  If Mm.Device$ = "PicoMite" Then
    Print vt100.colour$(c$);
    Exit Sub
  EndIf

  ' Serial / ANSI colour.
  Local old_console$ = Mm.Info(Option Console)
  If LCase$(old_console$) <> "serial" Then Option Console Serial
  Print vt100.colour$(c$);
  Select Case LCase$(old_console$)
    Case "both"   : Option Console Both
    Case "screen" : Option Console Screen
  End Select

  ' Screen / VGA colour.
  Select Case c$
    Case "green"   : Colour(RGB(Green))
    Case "magenta" : Colour(RGB(Magenta))
    Case "red"     : Colour(RGB(Red))
    Case "reset"   : Colour(RGB(White))
    Case Else      : Error "unsupported colour: " + c$
  End Select
End Sub

' Creates an empty file.
Sub ut.create_file(f$, fnbr%)
  Local fnbr_% = Choice(fnbr%, fnbr%, 1)
  Open f$ For Output As fnbr_%
  Close fnbr_%
End Sub

' Writes the contents of DATA to a file.
'
' @param  filename$    The file to write.
' @param  data_label$  Label from which to start reading DATA.
Sub ut.write_data_file(filename$, data_label$)
  Local s$, type$
  Read Save
  Restore data_label$
  Read type$
  Open filename$ For Output As #1
  Select Case type$
    Case "text/crlf", "text/lf"
      Do
        Read s$
        If Right$(s$, 5) = "<EOF>" Then
          Print #1, Left$(s$, Len(s$) - 5);
          Exit Do
        ElseIf type$ = "text/crlf" Then
          Print #1, s$ + Chr$(13) + Chr$(10);
        ElseIf type$ = "text/lf" Then
          Print #1, s$ + Chr$(10);
        Else
          Error "Unknown text data type '" + type$ + "'"
        EndIf
      Loop
    Case Else
      Error "Unknown data type '" + type$ + "'"
  End Select
  Close #1
  Read Restore
End Sub
' END:   #Include "/home/thwill/github/cmm2-sptools/src/sptrans/tests/../../sptest/unittest.inc"
' BEGIN: #Include #Include "../../common/sptools.inc" --------------------------
' Copyright (c) 2020-2023 Thomas Hugo Williams
' License MIT <https://opensource.org/licenses/MIT>
' For MMBasic 5.07

On Error Skip 1 : Dim sys.VERSION$ = ""
If sys.VERSION$ = "" Then Error "'system.inc' not included"
sys.requires("file")
sys.provides("sptools")
If sys.err$ <> "" Then Error sys.err$

' Gets the 'sptools' installation directory.
Function spt.get_install_dir$()

  ' First try recursing up the directory structure from the running program
  ' until a file called 'sptools.root' is found.
  Local d$ = file.PROG_DIR$
  Do While d$ <> ""
    If file.exists%(d$ + "/sptools.root") Then Exit Do
    d$ = file.get_parent$(d$)
  Loop

  ' Otherwise try the default installation location.
  If d$ = "" Then d$ = "A:/sptools"

  If Not file.is_directory%(d$) Then Error "directory not found: " + d$

  spt.get_install_dir$ = d$
End Function

Sub spt.print_version(name$)
  Print name$ " (SP Tools) Version " + sys.VERSION$ + " for MMBasic 5.07"
  Print "Copyright (c) 2020-2023 Thomas Hugo Williams"
  Print "A Toy Plastic Trumpet Production for Sockpuppet Studios."
  Print "License MIT <https://opensource.org/licenses/MIT>"
  Print "This is free software: you are free to change and redistribute it."
  Print "There is NO WARRANTY, to the extent permitted by law."
End Sub
' END:   #Include "/home/thwill/github/cmm2-sptools/src/sptrans/tests/../../common/sptools.inc"
' BEGIN: #Include #Include "../keywords.inc" -----------------------------------
' Copyright (c) 2020-2023 Thomas Hugo Williams
' License MIT <https://opensource.org/licenses/MIT>
' For MMBasic 5.07.07

On Error Skip 1 : Dim sys.VERSION$ = ""
If sys.VERSION$ = "" Then Error "'system.inc' not included"
sys.requires("map", "sptools")
sys.provides("keywords")
If sys.err$ <> "" Then Error sys.err$

Sub keywords.init()
  Local num_keywords%, max_keyword_len%, i%, s$
  Read Save
  Restore keywords.data
  Read num_keywords%, max_keyword_len%
  Dim keywords$(map.new%(num_keywords%)) Length max_keyword_len%
  map.init(keywords$())
  For i% = 1 To num_keywords%
    Read s$
    map.put(keywords$(), LCase$(s$), s$)
  Next
  Read Restore
End Sub

Function keywords.contains%(s$)
  keywords.contains% = map.get$(keywords$(), LCase$(s$)) <> sys.NO_DATA$
End Function

Function keywords.get$(s$)
  keywords.get$ = map.get$(keywords$(), LCase$(s$))
End Function

keywords.data:
Data 443, 14
Data "#Gps","#Include","@","Abort","Abs","Acos","Adc","Ain","All","Altitude"
Data "And","Append","Arc","Area","As","Asc","Asin","Atan2","Atn","Auto"
Data "AutoRefresh","Autorun","Autosave","Backlight","Bargauge","Base","Baudrate"
Data "Bcolour","Be","Beep","Bezier","Bin$","Bin2Str$","Bitmap","Black","Blit"
Data "Blue","Bmp","Box","Break","Brown","Button","Byte","C16","C20","C40","C8"
Data "Calibrate","Camera","Cancel","Caption","Capture","Case","Cat","CFunAddr"
Data "CFunction","ChDir","Checkbox","Chr$","Cin","CInt","Circle","Clear","Close"
Data "Cls","Color","ColorCode","Colour","ColourCode","Concat","Console","Const"
Data "Continue","Controller","Controls","Copy","Cos","Cpu","Create","Csub"
Data "Ctrlval","Current","Cursor","Cwd$","Cyan","Dac","Data","Date$","DateTime$"
Data "Day$","Default","DefineFont","Deg","Delete","Dht22","Dim","Din","Dir"
Data "Dir$","Disable","Display","DisplayBox","Distance","Do","Dop","Dout","Down"
Data "Draw3d","Ds18b20","Edit","Else","ElseIf","Enable","End","EndIf","Eof"
Data "Epoch","Erase","Error","Es","Eval","Exit","Exp","Explicit","Fcolour","Fft"
Data "Field$","File","Files","Fin","Fix","Flac","FlashPages","Float","Font"
Data "For","Format$","Fr","Frame","Frequency","Function","Fauge","Geoid"
Data "GetReg","GetScanLine","GetTime","Gosub","Goto","Gps","Gr","Gray","Green"
Data "Gui","Hex$","Hide","Humid","I2C","I2C2","I2C3","If","Ignore","Ili9163"
Data "Ili9341","Ii9341_16","Ii9841","Image","Inc","Init","Inkey$","Input"
Data "Input$","InStr","Int","Intb","Integer","Interrupt","Inth","Intl","Inverse"
Data "Invert","IR","IReturn","Is","It","Jpg","Json$","Key","Keyboard","Keydown"
Data "Keypad","Kill","Landscape","LastRef","LastX","LastY","Latitude","LCase$"
Data "Lcd","LcdPanel","LCompare","Led","Left$","Len","Length","Let","LGetStr$"
Data "Library","Line","LInStr","List","LLen","Load","Loc","Local","Lof","Log"
Data "Longitude","LongString","Loop","Lower","Magenta","Magnitude","Map","Math"
Data "Max","Memory","Mid$","Min","MkDir","Mm.Backup","Mm.CmdLine$","Mm.Device$"
Data "Mm.ErrMsg$","Mm.ErrNo","Mm.FontHeight","Mm.FontWidth","Mm.HPos","Mm.HRes"
Data "Mm.I2C","Mm.Info","Mm.Info$","Mm.OneWire","Mm.Persist","Mm.Ver","Mm.VPos"
Data "Mm.VRes","Mm.Watchdog","Mod","Mode","ModFile","Mouse","Move","Movement"
Data "Mp3","MsgBox","Name","New","NewEdit","Next","NoConsole","NoEcho"
Data "NoInterrupt","NoInvert","None","Not","NumberBox","Nunchuk","Oc","Oct$"
Data "Off","On","OneWire","Oout","Open","Option","Or","Output","OwSearch","Page"
Data "Pause","Peek","Phase","Pi","Pin","Pixel","Play","Png","Poke","Polygon"
Data "Port","Portrait","Pos","Print","ProgMem","Pu","PullDown","PullUp","Pulse"
Data "Pulsin","Pwm","Q_Create","Q_Euler","Q_Invert","Q_Mult","Q_Rotate"
Data "Q_Vector","Rad","Radio","Random","Randomize","RBox","Read","Red","Redraw"
Data "Ref","Refresh","Register","Rem","Rename","Replace","Reset","Restart"
Data "Restore","Resume","Return","Rgb","Right$","Rlandscape","RmDir","Rnd"
Data "Rotate","RPortrait","Rtc","Run","Satellites","Save","Scale","ScrollH"
Data "ScrollR","ScrollV","SdCard","Search","Seek","Select","Send","SensorFusion"
Data "Serial","Servo","SetPin","SetReg","SetTick","SetTime","Setup","Sgn","Show"
Data "Sin","Skip","Slave","Sleep","Sort","Space$","Spc","Speed","Spi","Spi2"
Data "Spi3","SpinBox","Sprite","Sqr","Ssd1963_4","Ssd1963_5","Ssd1963_5_16"
Data "Ssd1963_5_640","Ssd1963_5_buff","Ssd1963_5a","Ssd1963_7","Ssd1963_7_16"
Data "Ssd1963_7_640","Ssd1963_7_buff","Ssd1963_7a","Ssd1963_8","Ssd1963_8_16"
Data "Ssd1963_8_640","Ssd1963_8_buff","St7735","Start","StartLine","Static"
Data "Step","Stop","Str$","Str2Bin","String","String$","Sub","Switch","Tab"
Data "Tan","Tempr","Test","Text","TextBox","Then","Time$","Timer","Title","To"
Data "Tone","Touch","Trace","Track","Triangle","Trigger","Trim","Troff","Tron"
Data "Tts","Turtle","UCase$","UK","Until","Up","Upper","US","UsbKeyboard","Val"
Data "Valid","Var","VarAddr","VarTbl","Vcc","Volume","Watchdog","Wav","WEnd"
Data "While","White","Word","Write","Ws2812","XModem","Xor","Yellow"
' END:   #Include "/home/thwill/github/cmm2-sptools/src/sptrans/tests/../keywords.inc"
' BEGIN: #Include #Include "../lexer.inc" --------------------------------------
' Copyright (c) 2020-2023 Thomas Hugo Williams
' License MIT <https://opensource.org/licenses/MIT>
' For MMBasic 5.07

On Error Skip 1 : Dim sys.VERSION$ = ""
If sys.VERSION$ = "" Then Error "'system.inc' not included"
sys.requires("set", "keywords")
sys.provides("lexer")
If sys.err$ <> "" Then Error sys.err$

Const TK_IDENTIFIER = 1
Const TK_NUMBER = 2
Const TK_COMMENT = 3
Const TK_STRING = 4
Const TK_KEYWORD = 5
Const TK_SYMBOL = 6
Const TK_DIRECTIVE = 7
Const TK_LABEL = 8
Const TK_OPTION = 9

Const LX_MAX_TOKENS = 255 ' In theory every character may be a separate token.
Dim lx.type(LX_MAX_TOKENS - 1)
Dim lx.start(LX_MAX_TOKENS - 1)
Dim lx.len(LX_MAX_TOKENS - 1)

Dim lx.char$
Dim lx.line$
Dim lx.next_char$
Dim lx.num         ' number of tokens.
Dim lx.pos
Dim lx.csub        ' 1 if we are tokenising within a CSUB definition, otherwise 0.
Dim lx.font        ' 1 if we are tokenising within a DEFINEFONT definition, otherwise 0.

' Parses a line of BASIC code.
'
' @param   the line to parse.
' @return  sys.SUCCESS on success, otherwise sys.FAILURE and sets 'sys.err$'.
Function lx.parse_basic%(line$)
  lx.reset(line$)
  lx.advance()

  Do While lx.char$ <> Chr$(10)
    If lx.char$ = " " Then
      lx.advance()
    ElseIf InStr("&.0123456789", lx.char$) Then
      If lx.csub Or lx.font Then
        lx.parse_identifier()
      Else
        lx.parse_number()
      EndIf
    ElseIf lx.char$ = "'" Then
      lx.parse_comment_or_directive()
    ElseIf lx.char$ = Chr$(34) Then
      lx.parse_string()
    ElseIf InStr("@#_abcdefghijklmnopqrstuvwxyz", lx.char$) Then
      lx.parse_identifier()
    Else
      lx.parse_symbol()
    EndIf

    If sys.err$ <> "" Then
      lx.parse_basic% = sys.FAILURE
      Exit Do
    EndIf
  Loop
End Function

Sub lx.reset(line$)
  ' Clear old token data
  Do While lx.num > 0
    Inc lx.num, -1
    lx.type(lx.num) = 0
    lx.start(lx.num) = 0
    lx.len(lx.num) = 0
  Loop

  sys.err$ = ""
  lx.line$ = line$
  lx.next_char$ = ""
  lx.pos = 0
End Sub

Sub lx.advance()
  Inc lx.pos
  lx.char$ = lx.next_char$
  If lx.char$ = "" Then
    lx.char$ = Choice(lx.pos <= Len(lx.line$), LCase$(Chr$(Peek(Var lx.line$, lx.pos))), Chr$(10))
  EndIf

  If lx.pos + 1 <= Len(lx.line$) Then
    lx.next_char$ = LCase$(Chr$(Peek(Var lx.line$, lx.pos + 1)))
  Else
    lx.next_char$ = Chr$(10)
  EndIf

End Sub

Sub lx.parse_number()
  If InStr(".0123456789", lx.char$) Then
    lx.parse_decimal()
  ElseIf lx.char$ = "&" Then
    If InStr("abcdefh0123456789", lx.next_char$) Then
      lx.parse_hex_or_binary()
    ElseIf lx.next_char$ = "o" Then
      lx.parse_octal()
    ElseIf lx.next_char$ = "&" Then
      lx.parse_symbol()
    Else
      sys.err$ = "Unknown literal type &" + lx.next_char$
    EndIf
  EndIf
End Sub

Sub lx.parse_decimal()
  Local start = lx.pos

  lx.advance_while("0123456789")

  If lx.char$ = "." Then
    lx.advance()
    lx.advance_while("0123456789")
  EndIf

  If lx.char$ = "e" Then
    ' Handle E numbers:
    '  - if there is just a trailing E, e.g. 1234E then the E is part of the
    '    number literal.
    '  - however if followed by anything other than +, -, or <digit>,
    '    e.g. 1234ENDPROC then the E is part of a following identifier.
    If InStr(" -+0123456789" + Chr$(10), lx.next_char$) Then
      lx.advance()
      If lx.char$ = "-" Or lx.char$ = "+" Then lx.advance()
      lx.advance_while("0123456789")
    EndIf
  ElseIf lx.char$ = ":" Then
    ' Handle numeric labels.
    If lx.num = 0 Then lx.advance()
  EndIf

  lx.store(TK_NUMBER, start, lx.pos - start)
  If Right$(lx.token$(lx.num - 1), 1) = ":" Then lx.type(lx.num - 1) = TK_LABEL
End Sub

Sub lx.store(type, start, length)
  If length = 0 Then Error "Empty token"
  lx.type(lx.num) = type
  lx.start(lx.num) = start
  lx.len(lx.num) = length
  Inc lx.num
End Sub

Sub lx.advance_while(allowed$)
  Do While InStr(allowed$, lx.char$) : lx.advance() : Loop
End Sub

Sub lx.parse_hex_or_binary()
  Local start = lx.pos

  ' To facilitate transpiling BBC Basic source code the lexer accepts
  ' hex numbers which begin with just an & instead of &h.
  '
  ' This does introduce an unresolved ambiguity, e.g. is "&b0" the binary
  ' number 0 or the hex number B0 ?
  '
  ' TODO: provide an option to switch between BBC Basic and MMBasic
  '       "&-prefixed" literals.

  lx.advance()
  lx.advance()
  lx.advance_while("0123456789abcdefABCDEF")
  lx.store(TK_NUMBER, start, lx.pos - start)
End Sub

Sub lx.parse_octal()
  Local start = lx.pos

  lx.advance()
  lx.advance()
  lx.advance_while("01234567")
  lx.store(TK_NUMBER, start, lx.pos - start)
End Sub

Sub lx.parse_comment_or_directive()
  If lx.num = 0 Then
    ' Only the first token on a line will be recognised as a directive.
    If lx.next_char$ = "!" Then
      lx.parse_directive()
      Exit Sub
    EndIf
  EndIf
  lx.parse_comment()
End Sub

Sub lx.parse_directive()
  Local start = lx.pos

  lx.advance()
  lx.advance()
  lx.advance_while("-_abcdefghijklmnopqrstuvwxyz0123456789")
  lx.store(TK_DIRECTIVE, start, lx.pos - start)

  If lx.token_lc$(lx.num - 1) = "'!replace" Then lx.parse_replace_directive()
End Sub

Sub lx.parse_replace_directive()
  Local start

  Do While lx.char$ <> Chr$(10)
    If lx.char$ = " " Then
      lx.advance()
    ElseIf InStr("{}", lx.char$) Then
      lx.parse_symbol()
    Else
      start = lx.pos
      lx.advance()
      lx.advance_until(" {}" + Chr$(10))
      ' TODO: not really a KEYWORD, but a REPLACEMENT token.
      lx.store(TK_KEYWORD, start, lx.pos - start)
    EndIf

    If sys.err$ <> "" Then Exit Do
  Loop
End Sub

Sub lx.parse_comment()
  lx.store(TK_COMMENT, lx.pos, Len(lx.line$) - lx.pos + 1)
  lx.char$ = Chr$(10)
End Sub

Sub lx.parse_string()
  Local start = lx.pos

  lx.advance()
  lx.advance_until(Chr$(10) + Chr$(34))
  If lx.char$ = Chr$(10) Then sys.err$ = "No closing quote" : Exit Sub
  lx.store(TK_STRING, start, lx.pos - start + 1)
  lx.advance()
End Sub

Sub lx.advance_until(disallowed$)
  Do While Not InStr(disallowed$, lx.char$) : lx.advance() : Loop
End Sub

Sub lx.parse_identifier()
  Local start = lx.pos

  lx.advance()
  lx.advance_while("._abcdefghijklmnopqrstuvwxyz0123456789")

  If lx.char$ = " " Then
    ' Handle old-school REM statements.
    If LCase$(Mid$(lx.line$, start, lx.pos - start)) = "rem") Then
      lx.store(TK_COMMENT, start, Len(lx.line$) - start + 1)
      lx.char$ = Chr$(10)
      Exit Sub
    EndIf
  ElseIf lx.char$ = ":" Then
    ' Handle labels.
    If lx.num = 0 Then lx.advance()
  ElseIf InStr("$!%", lx.char$) Then
    ' Handle trailing type symbol.
    lx.advance()
  EndIf

  Local ln% = lx.pos - start
  If keywords.contains%(Mid$(lx.line$, start, ln%)) Then
    lx.store(TK_KEYWORD, start, ln%)
    Select Case lx.token_lc$(lx.num - 1)
      Case "csub" :       lx.csub = Not lx.csub
      Case "definefont" : lx.font = Not lx.font
    End Select
  Else
    lx.store(TK_IDENTIFIER, start, ln%)
    If Right$(lx.token$(lx.num - 1), 1) = ":" Then lx.type(lx.num - 1) = TK_LABEL
  EndIf
End Sub

Sub lx.parse_symbol()
  Local start = lx.pos

  Select Case lx.char$
    Case "<", ">", "="
      lx.advance()
      Select Case lx.char$
        Case "<", ">", "="
          lx.store(TK_SYMBOL, start, 2)
          lx.advance()
        Case Else
          lx.store(TK_SYMBOL, start, 1)
      End Select
    Case "&"
      lx.advance()
      Select Case lx.char$
        Case "&"
          lx.store(TK_SYMBOL, start, 2)
          lx.advance()
        Case Else
          lx.store(TK_SYMBOL, start, 1)
      End Select
    Case "|"
      lx.advance()
      Select Case lx.char$
        Case "|"
          lx.store(TK_SYMBOL, start, 2)
          lx.advance()
        Case Else
          lx.store(TK_SYMBOL, start, 1)
      End Select
    Case Else
      lx.store(TK_SYMBOL, start, 1)
      lx.advance()
  End Select
End Sub

' Gets the text of token 'i'.
'
' If i > the number of tokens then returns the empty string.
Function lx.token$(i)
  lx.token$ = Choice(i < lx.num, Mid$(lx.line$, lx.start(i), lx.len(i)), "")
End Function

' Gets the lower-case text of token 'i'.
'
' If i > the number of tokens then returns the empty string.
Function lx.token_lc$(i)
  lx.token_lc$ = Choice(i < lx.num, LCase$(Mid$(lx.line$, lx.start(i), lx.len(i))), "")
End Function

' Gets the directive corresponding to token 'i' without the leading single quote.
'
' Throws an Error if token 'i' is not a directive.
Function lx.directive$(i)
  If lx.type(i) <> TK_DIRECTIVE Then Error "{" + lx.token$(i) + "} is not a directive"
  lx.directive$ = Mid$(lx.line$, lx.start(i) + 1, lx.len(i) - 1)
End Function

' Gets the string corresponding to token 'i' without the surrounding quotes.
'
' Throws an Error if token 'i' is not a string literal.
Function lx.string$(i)
  If lx.type(i) <> TK_STRING Then Error "{" + lx.token$(i) + "} is not a string literal"
  lx.string$ = Mid$(lx.line$, lx.start(i) + 1, lx.len(i) - 2)
End Function

' Gets the number corresponding to token 'i'.
'
' Throws an Error if token 'i' is not a number literal.
Function lx.number!(i)
  If lx.type(i) <> TK_NUMBER Then Error "{" + lx.token$(i) + "} is not a number literal"
  lx.number! = Val(lx.token$(i))
End Function

' @return  0 on success, otherwise -1 and sets 'sys.err$'.
Function lx.parse_command_line%(line$)
  lx.reset(line$)
  lx.advance()

  Do While lx.char$ <> Chr$(10)
    If lx.char$ = " " Then
      lx.advance()
    ' ElseIf InStr("&.0123456789", lx.char$) Then
    '   lx.parse_number()
    ' ElseIf lx.char$ = "'" Then
    '   lx.parse_comment_or_directive()
    ElseIf lx.char$ = Chr$(34) Then
      lx.parse_string()
    ElseIf InStr("./@#_abcdefghijklmnopqrstuvwxyz", lx.char$) Then
      lx.parse_argument()
    ElseIf lx.char$ = "-" Then
      lx.parse_option()
    Else
      lx.parse_symbol()
    EndIf

    If sys.err$ <> "" Then
      lx.parse_command_line% = -1
      Exit Do
    EndIf
  Loop
End Function

Sub lx.parse_option()

  Const start = lx.pos

  Select Case lx.char$
    Case "-"
      lx.advance()
      If lx.char$ = "-" Then lx.advance()
    ' Case "/"
    '   lx.advance()
    Case Else
      Error "Unexpected badly formed option" ' Should never happen.
  End Select

  Const legal$ = "-_abcdefghijklmnopqrstuvwxyz0123456789"
  Local ok% = InStr(legal$, lx.char$) > 0
  If ok% Then lx.advance_while(legal$)
  ok% = ok% And (InStr("= " + Chr$(10), lx.char$) > 0)

  If ok% Then
    lx.store(TK_OPTION, start, lx.pos - start)
  Else
    If Not InStr("= " + Chr$(10), lx.char$) Then lx.advance()
    sys.err$ = "Illegal command-line option format: " + Mid$(lx.line$, start, lx.pos - start)
  EndIf

End Sub

Sub lx.parse_argument()
  Local start% = lx.pos
  lx.advance()
  lx.advance_while(".-_/abcdefghijklmnopqrstuvwxyz0123456789")
  lx.store(TK_IDENTIFIER, start, lx.pos - start)
End Sub

' Gets the command-line option corresponding to token 'i'.
'
' Throws an Error if token 'i' is not a command-line option.
Function lx.option$(i)
  If lx.type(i) <> TK_OPTION Then Error "{" + lx.token$(i) + "} is not a command-line option"
  Local sz% = Choice(Mid$(lx.line$, lx.start(i), 2) = "--", 2, 1)
  lx.option$ = Mid$(lx.line$, lx.start(i) + sz%, lx.len(i) - sz%)
End Function

Sub lx.dump()
  Local i%, type$
  Print "[[[" lx.line$ "]]]"
  For i% = 0 To lx.num - 1
    Select Case lx.type%(i%)
      Case TK_IDENTIFIER : type$ = "IDENTIFIER"
      Case TK_NUMBER     : type$ = "NUMBER    "
      Case TK_COMMENT    : type$ = "COMMENT   "
      Case TK_STRING     : type$ = "STRING    "
      Case TK_KEYWORD    : type$ = "KEYWORD   "
      Case TK_SYMBOL     : type$ = "SYMBOL    "
      Case TK_DIRECTIVE  : type$ = "DIRECTIVE "
      Case TK_LABEL      : type$ = "LABEL     "
      Case TK_OPTION     : type$ = "OPTION    "
      Case Else          : type$ = "<UNKNOWN> "
    End Select
    Print Str$(i%) ": " type$ ", " Str$(lx.start%(i%)) ", " Str$(lx.len%(i%)) ", [[[" lx.token$(i%) "]]]"
  Next
End Sub

' Inserts token at given index moving the current token at that position
' and all the tokens following it one place to the right.
'
' 1. Whitespace preceeding a token is considered part of that token,
'    unless it is the first token.
' 2. Whitespace following the last token is considered part of that
'    token.
Sub lx.insert_token(idx%, token$, type%)
  If idx% > lx.num Then Error "Invalid token index: " + Str$(idx%)
  Local length% = Len(lx.line$), prefix$, suffix$

  ' Separator to insert before the token we are inserting.
  Local sep$ = Choice(idx% = 0, "", Choice(token$ = "(", "", " "))

  Select Case idx%
    Case lx.num
      prefix$ = lx.line$
    Case 0
      prefix$ = Left$(lx.line$, lx.start(idx%) - 1)
      suffix$ = " " + Mid$(lx.line$, lx.start(idx%))
    Case Else
      Local prefix_len% = lx.start(idx% - 1) + lx.len(idx% - 1) - 1
      prefix$ = Left$(lx.line$, prefix_len%)
      suffix$ = Mid$(lx.line$, prefix_len% + 1)
  End Select
  lx.line$ = prefix$ + sep$ + token$ + suffix$

  ' Shift following tokens up one index.
  Local shift% = Len(token$) + Len(sep$) + (idx% = 0)
  Local j%
  For j% = lx.num - 1 To idx% Step -1
    lx.type(j% + 1) = lx.type(j%)
    lx.start(j% + 1) = lx.start(j%) + shift%
    lx.len(j% + 1) = lx.len(j%)
  Next

  ' Insert new token.
  lx.type(idx%) = type%
  lx.start(idx%) = Len(prefix$ + sep$) + 1
  lx.len(idx%) = Len(token$)

  Inc lx.num, 1
End Sub

' Removes token at given index.
'
' 1. Whitespace preceeding a token is considered part of that token,
'    unless it is the first token.
' 2. Whitespace following the last token is considered part of that
'    token.
Sub lx.remove_token(idx%)
  If idx% >= lx.num Then Error "Invalid token index: " + Str$(idx%)
  Local length% = Len(lx.line$), prefix$, suffix$

  Select Case idx%
    Case 0
      prefix$ = Left$(lx.line$, lx.start(0) - 1)
      suffix$ = Choice(lx.num = 1, "", Mid$(lx.line$, lx.start(1)))
    Case lx.num - 1
      prefix$ = Left$(lx.line$, lx.start(idx% - 1) + lx.len(idx% - 1) - 1)
    Case Else
      prefix$ = Left$(lx.line$, lx.start(idx% - 1) + lx.len(idx% - 1) - 1)
      suffix$ = Mid$(lx.line$, lx.start(idx%) + lx.len(idx%))
  End Select
  lx.line$ = prefix$ + suffix$

  Local shift% = length% - Len(lx.line$)
  Local j%
  For j% = idx% To lx.num - 2
    lx.type(j%) = lx.type(j% + 1)
    lx.start(j%) = lx.start(j% + 1) - shift%
    lx.len(j%) = lx.len(j% + 1)
  Next

  Inc lx.num, -1
  lx.type(lx.num) = 0
  lx.start(lx.num) = 0
  lx.len(lx.num) = 0
End Sub

' Replaces token at given index.
Sub lx.replace_token(idx%, replacement$, type%)
  If idx% >= lx.num Then Error "Invalid token index: " + Str$(idx%)
  lx.line$ = Left$(lx.line$, lx.start(idx%) - 1) + replacement$ + Mid$(lx.line$, lx.start(idx%) + lx.len(idx%))
  lx.type(idx%) = type%
  Local shift% = Len(replacement$) - lx.len(idx%)
  lx.len(idx%) = Len(replacement$)
  Local j%
  For j% = idx% + 1 To lx.num - 1
    Inc lx.start(j%), shift%
  Next
End Sub
' END:   #Include "/home/thwill/github/cmm2-sptools/src/sptrans/tests/../lexer.inc"
' BEGIN: #Include #Include "../options.inc" ------------------------------------
' Copyright (c) 2020-2023 Thomas Hugo Williams
' License MIT <https://opensource.org/licenses/MIT>
' For MMBasic 5.07

On Error Skip 1 : Dim sys.VERSION$ = ""
If sys.VERSION$ = "" Then Error "'system.inc' not included"
sys.requires("set")
sys.provides("options")
If sys.err$ <> "" Then Error sys.err$

Dim opt.colour       '  0    : no syntax colouring of console output
                     '  1    : VT100 syntax colouring of console output
Dim opt.comments     ' -1    : preserve comments
                     '  0    : omit all comments
Dim opt.empty_lines  ' -1    : preserve empty lines
                     '  0    : omit all empty lines
                     '  1    : include empty line between each Function/Sub
Dim opt.format_only  '  0    : transpile
                     '  1    : just format / pretty-print
Dim opt.include_only '  0    : transpile
                     '  1    : just inline #INCLUDEd files
Dim opt.indent_sz    ' -1    : preserve indenting
                     '  0..N : automatically indent by N spaces per level
Dim opt.keywords     ' -1    : preserve keyword capitalisation
                     '  0    : lowercase keywords
                     '  1    : PascalCase keywords
                     '  2    : UPPERCASE keywords
Dim opt.quiet        '  0    : run with normal console output
                     '  1    : run with minimal console output
Dim opt.spacing      ' -1    : preserve spacing
                     '  0    : omit all unnecessary (non-indent) spaces
                     '  1    : space compactly
                     '  2    : space generously
Dim opt.infile$      ' input file/path
Dim opt.outfile$     ' output file/path, empty for output to console

Sub opt.init()
  opt.colour = 0
  opt.comments = -1
  opt.empty_lines = -1
  opt.format_only = 0
  opt.include_only = 0
  opt.indent_sz = -1
  opt.keywords = -1
  opt.quiet = 0
  opt.spacing = -1
  opt.infile$ = ""
  opt.outfile$ = ""
End Sub

' Do the options require pretty-printing ?
Function opt.pretty_print%()
  Local z% = opt.colour Or opt.comments <> -1 Or opt.empty_lines <> -1
  z% = z% Or opt.indent_sz <> -1 Or opt.keywords <> -1 Or opt.spacing <> -1
  opt.pretty_print% = z%
End Function

' Do the options require !directives to be processed ?
Function opt.process_directives%()
  opt.process_directives% = Not (opt.include_only Or opt.format_only)
End Function

' Sets the value for an option.
'
' If name$ or value$ are invalid then sets sys.err$.
Sub opt.set(name$, value$)
  Select Case LCase$(name$)
    Case "colour"       : opt.set_colour(value$)
    Case "comments"     : opt.set_comments(value$)
    Case "crunch"       : opt.set_crunch(value$)
    Case "empty-lines"  : opt.set_empty_lines(value$)
    Case "format-only"  : opt.set_format_only(value$)
    Case "include-only" : opt.set_include_only(value$)
    Case "indent"       : opt.set_indent_sz(value$)
    Case "keywords"     : opt.set_keywords(value$)
    Case "no-comments"  : opt.set_no_comments(value$)
    Case "quiet"        : opt.set_quiet(value$)
    Case "spacing"      : opt.set_spacing(value$)
    Case "infile"       : opt.set_infile(value$)
    Case "outfile"      : opt.set_outfile(value$)
    Case Else
      sys.err$ = "unknown option: " + name$
  End Select
End Sub

Sub opt.set_colour(value$)
  Select Case LCase$(value$)
    Case "default", "off", "0", "" : opt.colour = 0
    Case "on", "1"                 : opt.colour = 1
    Case Else
      sys.err$ = "expects 'on|off' argument"
  End Select
End Sub

Sub opt.set_comments(value$)
  Select Case LCase$(value$)
    Case "preserve", "default", "on", "-1", "" : opt.comments = -1
    Case "none", "omit", "off", "0"            : opt.comments = 0
    Case Else
      sys.err$ = "expects 'on|off' argument"
  End Select
End Sub

Sub opt.set_crunch(value$)
  Select Case LCase$(value$)
    Case "default", "off", "0", "" : ' do nothing
    Case "on", "1"
      opt.comments = 0
      opt.empty_lines = 0
      opt.indent_sz = 0
      opt.spacing = 0
    Case Else
      sys.err$ = "expects 'on|off' argument"
  End Select
End Sub

Sub opt.set_empty_lines(value$)
  Select Case LCase$(value$)
    Case "preserve", "default", "on", "-1", "" : opt.empty_lines = -1
    Case "none", "omit", "off", "0"             : opt.empty_lines = 0
    Case "single", "1"                         : opt.empty_lines = 1
    Case Else
      sys.err$ = "expects 'on|off|single' argument"
  End Select
End Sub

Sub opt.set_format_only(value$)
  Select Case LCase$(value$)
    Case "default", "off", "0", "" : opt.format_only = 0
    Case "on", "1"                : opt.format_only = 1
    Case Else                     : sys.err$ = "expects 'on|off' argument"
  End Select
End Sub

Sub opt.set_include_only(value$)
  Select Case LCase$(value$)
    Case "default", "off", "0", "" : opt.include_only = 0
    Case "on", "1"                : opt.include_only = 1
    Case Else                     : sys.err$ = "expects 'on|off' argument"
  End Select
End Sub

Sub opt.set_indent_sz(value$)
  Select Case LCase$(value$)
    Case "preserve", "default", "on", "-1", "" : opt.indent_sz = -1
    Case Else
      If Str$(Val(value$)) = value$ And Val(value$) >= 0 Then
        opt.indent_sz = Val(value$)
      Else
        sys.err$= "expects 'on|<number>' argument"
      EndIf
    End Select
  End Select
End Sub

Sub opt.set_keywords(value$)
  Select Case LCase$(value$)
    Case "preserve", "default", "on", "-1", "" : opt.keywords = -1
    Case "lower", "l", "0"                     : opt.keywords = 0
    Case "pascal", "mixed", "m", "p", "1"      : opt.keywords = 1
    Case "upper", "u", "2"                     : opt.keywords = 2
    Case Else
      sys.err$ = "expects 'preserve|lower|pascal|upper' argument"
  End Select
End Sub

Sub opt.set_no_comments(value$)
  Select Case LCase$(value$)
    Case "default", "off", "0", "" : opt.comments = -1
    Case "on", "1"                : opt.comments = 0
    Case Else                     : sys.err$ = "expects 'on|off' argument"
  End Select
End Sub

Sub opt.set_quiet(value$)
  Select Case LCase$(value$)
    Case "default", "off", "0", "" : opt.quiet = 0
    Case "on", "1"                 : opt.quiet = 1
    Case Else
      sys.err$ = "expects 'on|off' argument"
  End Select
End Sub

Sub opt.set_spacing(value$)
  Select Case LCase$(value$)
    Case "preserve", "default", "on", "-1", "" : opt.spacing = -1
    Case "minimal", "off", "omit", "0"         : opt.spacing = 0
    Case "compact", "1"                        : opt.spacing = 1
    Case "generous", "2"                       : opt.spacing = 2
    Case Else
      sys.err$ = "expects 'on|minimal|compact|generous' argument"
  End Select
End Sub

Sub opt.set_infile(value$)
  opt.infile$ = value$
End Sub

Sub opt.set_outfile(value$)
  opt.outfile$ = value$
End Sub
' END:   #Include "/home/thwill/github/cmm2-sptools/src/sptrans/tests/../options.inc"
' BEGIN: #Include #Include "../defines.inc" ------------------------------------
' Copyright (c) 2020-2023 Thomas Hugo Williams
' License MIT <https://opensource.org/licenses/MIT>
' For MMBasic 5.07

On Error Skip 1 : Dim sys.VERSION$ = ""
If sys.VERSION$ = "" Then Error "'system.inc' not included"
sys.requires("set")
sys.provides("defines")
If sys.err$ <> "" Then Error sys.err$

' Set of active defines
'   - set at command line using -D<flag>,
'   - or in program using !DEFINE directive.
Const def.ID_MAX_LENGTH% = 64
Dim def.defines$(set.new%(10)) Length def.ID_MAX_LENGTH%

Sub def.init()
  set.init(def.defines$())
End Sub

Sub def.undefine(id$)
  Local s$ = str.trim$(id$), lower$ = LCase$(s$)

  If s$ = "" Then
    sys.err$ = "invalid identifier"
  Else If Len(s$) > def.ID_MAX_LENGTH% Then
    sys.err$ = "identifier too long, max 64 chars"
  Else If InStr("|1|true|on|0|false|off|", "|" + lower$ + "|") Then
    sys.err$ = "'" + s$ + "' cannot be undefined"
  Else If def.is_defined%(lower$) Then
    set.remove(def.defines$(), lower$)
  Else
    sys.err$ = "'" + s$ + "' is not defined"
  EndIf
End Sub

' @return  1 if the id$ is defined, 0 if it is not.
Function def.is_defined%(id$)
  Local s$ = str.trim$(id$)
  If Len(s$) > def.ID_MAX_LENGTH% Then
    sys.err$ = "identifier too long, max 64 chars"
  Else If InStr("|1|true|on|", "|" + LCase$(s$) + "|") Then
    def.is_defined% = 1
  Else If InStr("|0|false|off|", "|" + LCase$(s$) + "|") Then
    def.is_defined% = 0
  Else
    def.is_defined% = set.get%(def.defines$(), LCase$(s$)) <> -1
  Endif
End Function

Sub def.define(id$)
  Local i%, s$ = str.trim$(id$), lower$ = LCase$(s$)
  sys.err$ = "invalid identifier"

  If s$ = "" Then Exit Sub

  If Len(s$) > def.ID_MAX_LENGTH% Then
    sys.err$ = "identifier too long, max 64 chars"
    Exit Sub
  EndIf

  If InStr("|1|true|on|0|false|off|", "|" + lower$ + "|") Then
    sys.err$ = "'" + s$ + "' cannot be defined"
    Exit Sub
  EndIf

  If Not InStr("_abcdefghijklmnopqrstuvwxyz", Mid$(lower$, 1, 1)) Then Exit Sub
  For i% = 2 To Len(lower$)
    If Not InStr("_abcdefghijklmnopqrstuvwxyz0123456789", Mid$(lower$, i%, 1)) Then Exit Sub
  Next

  If set.get%(def.defines$(), lower$) <> -1 Then
    sys.err$ = "'" + s$ + "' is already defined"
    Exit Sub
  EndIf

  If set.is_full%(def.defines$()) Then
     sys.err$ = "too many defines"
     Exit Sub
  EndIf

  sys.err$ = ""
  set.put(def.defines$(), lower$)
End Sub
' END:   #Include "/home/thwill/github/cmm2-sptools/src/sptrans/tests/../defines.inc"
' BEGIN: #Include #Include "../expression.inc" ---------------------------------
' Copyright (c) 2023 Thomas Hugo Williams
' License MIT <https://opensource.org/licenses/MIT>
' For MMBasic 5.07

On Error Skip 1 : Dim sys.VERSION$ = ""
If sys.VERSION$ = "" Then Error "'system.inc' not included"
sys.requires("defines")
sys.provides("expression")
If sys.err$ <> "" Then Error sys.err$

' Evaluates an !IF preprocessor expression on the current line.
'
' @param[in]  idx      token index to start evaluating the expression from.
' @param[out] result%  1 for true, 0 for false.
' @return              sys.SUCCESS on success, any other value is an error,
'                      see 'sys.err$' for details.
Function ex.eval%(idx%, result%)
  Local i%, expr$, x%, t$

  For i% = idx% To lx.num - 1
    t$ = lx.token_lc$(i%)
    Select Case t$
      Case "defined"
        ' Ignore for now
      Case "!", "not"
        Cat expr$, " Not "
      Case "&&", "and"
        Cat expr$, " And "
      Case "||", "or"
        Cat expr$, " Or "
      Case "xor"
        Cat expr$, " Xor "
      Case "(", ")"
        Cat expr$, t$
      Case Else
        Select Case lx.type(i%)
          Case TK_IDENTIFIER, TK_NUMBER, TK_KEYWORD
            Cat expr$, Str$(def.is_defined%(t$))
          Case Else
            expr$ = ""
            Exit For
        End Select
    End Select
  Next

  ' Print "Expression: " expr$
  If expr$ = "" Then
    ex.eval% = sys.FAILURE
  Else
    ex.eval% = sys.SUCCESS
    On Error Ignore
    result% = Eval(expr$)
    If Mm.ErrNo <> 0 Then ex.eval% = sys.FAILURE
    On Error Abort
  EndIf

  If ex.eval% <> sys.SUCCESS Then sys.err$ = "Invalid expression syntax"
End Function
' END:   #Include "/home/thwill/github/cmm2-sptools/src/sptrans/tests/../expression.inc"
' BEGIN: #Include #Include "../trans.inc" --------------------------------------
' Copyright (c) 2020-2023 Thomas Hugo Williams
' License MIT <https://opensource.org/licenses/MIT>
' For MMBasic 5.07

On Error Skip 1 : Dim sys.VERSION$ = ""
If sys.VERSION$ = "" Then Error "'system.inc' not included"
sys.requires("options", "set", "defines", "expression")
sys.provides("trans")
If sys.err$ <> "" Then Error sys.err$

Dim tr.num_comments(MAX_NUM_FILES - 1)

' For each source file we maintain a stack of currently open !if directives.
' If 'i' is the base-0 index of the source file then
'   if_stack(i, if_stack_sz - 1)
' is a boolean combination of constants describing the latest !if directive.
Const tr.IF_ACTIVE     = &h00001 ' !if clause is active.
Const tr.ELSE_ACTIVE   = &h00010 ' !else clause is active.
Const tr.ELSE_INACTIVE = &h00100 ' !else clause is inactive.
Const tr.IF_COMMENT    = &h01000 ' Inside an active/inactive !if_comment.
Const tr.IF_UNCOMMENT  = &h10000 ' Inside an active/inactive !if_uncomment.
Const tr.MAX_NUM_IFS = 10
Dim tr.if_stack(MAX_NUM_FILES - 1, tr.MAX_NUM_IFS - 1)
Dim tr.if_stack_sz(MAX_NUM_FILES - 1)

' The list of replacements (from -> to).
Const tr.MAX_REPLACEMENTS% = 200
Dim tr.num_replacements
Dim tr.replacements$(tr.MAX_REPLACEMENTS% - 1, 1) Length 80

' Set to name of file after processing #INCLUDE.
Dim tr.include$

' If set we ignore/omit lines until we process an !ENDIF directive
' corresponding to the directive that set this flag.
Dim tr.omit_flag%

' Was the last non-omitted line empty after transpilation ?
Dim tr.empty_line_flag%

' Was the last line omitted ?
Dim tr.omitted_line_flag%

Const tr.INCLUDE_FILE = 1
Const tr.OMIT_LINE = 2

' Just transpile/inline #INCLUDE statements.
'
' @return  sys.FAILURE      on error, see 'sys.err$' for details.
'          sys.SUCCESS      if the current line should be included in the
'                           output.
'          tr.INCLUDE_FILE  if current line started with #INCLUDE statement, the
'                           filename/path is returned in 'tr.include$'.
'          tr.OMIT_LINE     if the current line should be omitted from the
'                           output.
Function tr.transpile_includes%()
  tr.include$ = ""
  If lx.token_lc$(0) = "#include" Then
    tr.process_include()
    tr.transpile_includes% = Choice(sys.err$ = "", tr.INCLUDE_FILE, sys.FAILURE)
  EndIf
End Function

' Full transpilation.
'
' @return  0  on error, see 'sys.err$' for details.
'          1  on general success.
'          2  if current line started with #INCLUDE statement,
'             the filename/path is returned in 'tr.include$'.
'          3  if line was deleted/omitted by the transpiler.
Function tr.transpile%()
  tr.include$ = ""

  ' Remove any trailing comment from directive.
  If lx.type(0) = TK_DIRECTIVE Then
    If lx.type(lx.num - 1) = TK_COMMENT Then lx.remove_token(lx.num - 1)
  EndIf

  Select Case lx.token_lc$(0)
    Case "'!endif", "'!else", "'!elif"
      ' TODO: Requiring the trailing brackets in the CALL may be an MMBasic bug.
      Call "tr.process_" + Mid$(lx.token_lc$(0), 3) + "()"
      If sys.err$ = "" Then
        lx.reset("") ' Is this really necessary ?
        tr.omitted_line_flag% = 1
        tr.transpile% = tr.OMIT_LINE
      Else
        tr.transpile% = sys.FAILURE
      EndIf
      Exit Function
  End Select

  If tr.omit_flag% Then
    If lx.type(0) = TK_DIRECTIVE Then
      Select Case lx.directive$(0)
        Case "!comment_if", "!if", "!ifdef", "!ifndef", "!uncomment_if"
          tr.push_if(0) ' Still need to record omitted !if's.
      End Select
    EndIf
    lx.reset("")
    tr.omitted_line_flag% = 1
    tr.transpile% = tr.OMIT_LINE
    Exit Function
  EndIf

  tr.add_comments()
  tr.apply_replacements()
  If sys.err$ <> "" Then
    tr.transpile% = sys.FAILURE
    Exit Function
  EndIf

  If lx.num = 0 Then                  ' IF this line is empty
    If tr.omitted_line_flag% Then     ' AND the last line was omitted
      If tr.empty_line_flag% Then     ' AND the last non-omitted line was empty
        tr.transpile% = tr.OMIT_LINE  ' THEN omit this line.
        Exit Function
      EndIf
    EndIf
  EndIF

  If lx.token_lc$(0) = "#include" Then
    tr.process_include()
    tr.transpile% = Choice(sys.err$ = "", tr.INCLUDE_FILE, sys.FAILURE)
    tr.omitted_line_flag% = 0
    Exit Function
  EndIf

  If lx.type(0) <> TK_DIRECTIVE Then
    tr.empty_line_flag% = (lx.num = 0)
    tr.transpile% = Choice(opt.comments = 0, tr.remove_comments%(), sys.SUCCESS)
    tr.omitted_line_flag% = tr.transpile% = tr.OMIT_LINE
    Exit Function
  EndIf

  tr.transpile% = tr.OMIT_LINE

  Select Case lx.directive$(0)
    Case "!comments"     : tr.process_comments()
    Case "!comment_if"   : tr.process_if()
    Case "!define"       : tr.process_define()
    Case "!elif"         : tr.process_elif()
    Case "!else"         : tr.process_else()
    Case "!empty-lines"  : tr.process_empty_lines()
    Case "!error"        : tr.process_error()
    Case "!if"           : tr.process_if()
    Case "!ifdef"        : tr.process_if()
    Case "!ifndef"       : tr.process_if()
    Case "!info"         : tr.transpile% = tr.process_info%()
    Case "!indent"       : tr.process_indent()
    Case "!replace"      : tr.process_replace()
    Case "!spacing"      : tr.process_spacing()
    Case "!uncomment_if" : tr.process_if()
    Case "!undef"        : tr.process_undef()
    Case "!unreplace"    : tr.process_replace()  ' Same SUB for !replace & !unreplace.
    Case Else            : sys.err$ = "Unknown " + Mid$(lx.directive$(0), 1) + " directive"
  End Select

  If sys.err$ <> "" Then
    tr.transpile% = sys.FAILURE
  ElseIf tr.transpile% = tr.OMIT_LINE Then
    ' lx.parse_basic("' PROCESSED: " + Mid$(lx.line$, lx.start(0) + 1))
    lx.reset("")
    tr.omitted_line_flag% = 1
  EndIf
End Function

Sub tr.process_elif()
  Select Case tr.pop_if()
    Case 0
      tr.process_if()
    Case tr.IF_ACTIVE
      tr.omit_flag% = 1
      tr.push_if(tr.IF_ACTIVE)
    Case Else
      sys.err$ = "!elif directive without !if"
  End Select
End Sub

Sub tr.process_else()
  If lx.num <> 1 Then
    sys.err$ = "!else directive has too many arguments"
    Exit Sub
  EndIf

  Local x% = tr.pop_if()
  Select Case x%
    Case -1
      If sys.err$ = "" Then Error "Internal error"
    Case 0
      tr.omit_flag% = 0
      tr.push_if(tr.ELSE_ACTIVE)
    Case Else
      If x% And (tr.ELSE_ACTIVE Or tr.ELSE_INACTIVE) Then
        sys.err$ = "Too many !else directives"
      Else
        tr.omit_flag% = 1
        tr.push_if(tr.ELSE_INACTIVE)
      EndIf
  End Select
End Sub

Sub tr.process_endif()
  If lx.num <> 1 Then
    sys.err$ = "!endif directive has too many arguments"
    Exit Sub
  EndIf

  Local x% = tr.pop_if()
  If x% And (tr.IF_ACTIVE Or tr.ELSE_ACTIVE) Then
    If x% And tr.IF_COMMENT Then
      tr.update_num_comments(-1)
    ElseIf x% And tr.IF_UNCOMMENT Then
      tr.update_num_comments(+1)
    EndIf
  Else
    If Not (x% And (tr.IF_COMMENT Or tr.IF_UNCOMMENT)) Then
      tr.omit_flag% = 0
    EndIf
  EndIf
End Sub

Sub tr.update_num_comments(x)
  Local i = in.num_open_files% - 1
  tr.num_comments(i) = tr.num_comments(i) + x
End Sub

Function tr.pop_if(directive$)
  Local i = in.num_open_files% - 1
  If tr.if_stack_sz(i) > 0 Then
    tr.if_stack_sz(i) = tr.if_stack_sz(i) - 1
    tr.pop_if = tr.if_stack(i, tr.if_stack_sz(i))
  Else
    sys.err$ = lx.directive$(0) + " directive without !if"
    tr.pop_if = -1
  EndIf
End Function

Sub tr.add_comments()
  Local nc = tr.num_comments(in.num_open_files% - 1)
  Local result%
  If nc > 0 Then
    result% = lx.parse_basic%(String$(nc, "'") + " " + lx.line$)
  ElseIf nc < 0 Then
    Do While nc < 0 And lx.num > 0 And lx.type(0) = TK_COMMENT
      If Mid$(lx.line$, lx.start(0), 1) = "'" Then
        result% = lx.parse_basic%(Left$(lx.line$, lx.start(0) - 1) + Mid$(lx.line$, lx.start(0) + 1))
      ElseIf LCase$(Mid$(lx.line$, lx.start(0), 3)) = "rem" Then
        result% = lx.parse_basic%(Left$(lx.line$, lx.start(0) - 1) + Mid$(lx.line$, lx.start(0) + 3))
      EndIf
      Inc nc, 1
    Loop
  EndIf
End Sub

' Applies replacements to the currently parsed line, lx.line$.
Sub tr.apply_replacements()
  If lx.num = 0 Then Exit Sub
  If tr.num_replacements% = 0 Then Exit Sub
  If lx.type(0) = TK_DIRECTIVE Then Exit Sub ' Don't replace within directives.

  Local capture$, i%, j%, k%, old_num%, s$, to_$
  For i% = 0 To tr.num_replacements% - 1

    ' Skip 'dead' entries.
    If tr.replacements$(i%, 0) = Chr$(0) Then Continue For

    j% = 0
    ' Need to use a DO rather than a FOR because the latter does not re-evaluate its end-point
    ' at the beginning of each iteration.
    Do While j% < lx.num
      k% = tr.match%(j%, i%, capture$)
      If k% > -1 Then
        s$ = Left$(lx.line$, lx.start(j%) - 1)
        to_$ = str.replace$(tr.replacements$(i%, 1), "|", " ")
        to_$ = str.replace$(to_$, "%1", Field$(capture$, 1, "|"))
        If Len(s$) + Len(to_$) + Len(lx.line$) - lx.start(k%) - lx.len(k%) > 255 Then
          sys.err$ = "applying replacement makes line > 255 characters"
          Exit Sub
        EndIf
        ' WARNING! In firmware 5.07 CAT does not error when it overflows 255 characters.
        Cat s$, to_$
        Cat s$, Mid$(lx.line$, lx.start(k%) + lx.len(k%))
        old_num% = lx.num%
        If lx.parse_basic%(s$) <> sys.SUCCESS Then Exit Sub

        ' Adjust j% so we don't resume searching for matches until immediately
        ' after the current replacement.
        Inc j%, Max(0, 1 + lx.num% - old_num% + k% - j%)
      Else
        Inc j%
      EndIf
    Loop
  Next

End Sub

' Attempts to match current token array against a 'from' specification.
'
' @param  ti  index into the token array from which to attempt the match.
' @param  ri  index into the replacements array for the 'from' we are trying to match.
' @return     the token index where the match ends, or -1 if there is no match.
Function tr.match%(ti%, ri%, capture$)
  Local done%, from$, i%, t$

  tr.match% = -1
  capture$ = ""

  Do
    from$ = Field$(tr.replacements$(ri, 0), i% + 1, "|")
    t$ = lx.token$(ti% + i%)

    If from$ = "" Then
      tr.match% = ti% + i% - 1
      done% = 1
    ElseIf InStr("%% %d %h", Right$(from$, 2)) Mod 3 = 1 Then
      done% = Not tr.capture%(from$, t$, capture$)
    Else
      done% = from$ <> LCase$(t$)
    EndIf

    Inc i%
  Loop Until done%
End Function

Function tr.capture%(pattern$, token$, capture$)
  Local allowed$

  Select Case Right$(pattern$, 2)
    Case "%%" : allowed$ = "*"                      ' match any character.
    Case "%d" : allowed$ = "0123456789"             ' match decimal digits.
    Case "%h" : allowed$ = "abcdefABCDEF0123456789" ' match hexadecimal digits.
    Case Else : Error "Unknown pattern."
  End Select

  If Len(pattern$) = 2 Then
    ' pattern$ is just a pattern, e.g. %%, %d, %h
    ' - we try to match and capture the entire token$.
    If tr.contains_only%(allowed$, token$) Then
      Cat capture$, token$ + "|"
      tr.capture% = 1
    EndIf

  ElseIf InStr(LCase$(token$), Left$(pattern$, Len(pattern$) - 2)) = 1 Then
    ' pattern$ is a prefix followed by a pattern, e.g. foo%%
    ' - we try to match and capture everything in token$ after the prefix.
    Local t$ = Mid$(token$, Len(pattern$) - 1)
    If tr.contains_only%(allowed$, t$) Then
      Cat capture$, t$ + "|"
      tr.capture% = 1
    EndIf
  EndIf
End Function

' Are all the characters in 's$' present in 'allowed$' ?
' If 'allowed$' is "*" then always returns 1 (true).
Function tr.contains_only%(allowed$, s$)
  If allowed$ <> "*" Then
    Local i%
    For i% = 1 To Len(s$)
      If Not InStr(allowed$, Mid$(s$, 1, 1)) Then Exit Function
    Next
  EndIf
  tr.contains_only% = 1
End Function

Sub tr.process_undef()
  If lx.num < 2 Then sys.err$ = "expects a <flag> argument"
  If lx.num > 2 Then sys.err$ = "has too many arguments"
  If sys.err$ = "" Then def.undefine(lx.token$(1))
  If sys.err$ <> "" Then sys.err$ = "!undef directive " + sys.err$
End Sub

Sub tr.process_comments()
  If lx.num > 2 Then sys.err$ = "has too many arguments"
  If sys.err$ = "" Then opt.set_comments(lx.token_lc$(1))
  If sys.err$ <> "" Then sys.err$ = "!comments directive " + sys.err$
End Sub

Sub tr.process_if()

  ' Replace !IFDEF and !IFNDEF directives with equivalent !IF DEFINED.
  Select Case lx.directive$(0)
    Case "!ifdef"
      If lx.num <> 2 Then
        sys.err$ = "expects 1 argument"
      Else
        Local dummy% = lx.parse_basic%("'!if defined (" + lx.token$(1) + ")")
      EndIf
    Case "!ifndef"
      If lx.num <> 2 Then
        sys.err$ = "expects 1 argument"
      Else
        Local dummy% = lx.parse_basic%("'!if not defined (" + lx.token$(1) + ")")
      EndIf
    Case Else
      If lx.num < 2 Then sys.err$ = "expects at least 1 argument"
  End Select

  If sys.err$ <> "" Then
    sys.err$ = lx.directive$(0) + " directive " + sys.err$
    Exit Sub
  EndIf

  Local x%
  If ex.eval%(1, x%) <> 0 Then Exit Sub
  If x% < 0 Or x% > 1 Then Error "Value is not true (1) or false (0): " + x%

  Select Case lx.directive$(0)
    Case "!comment_if"
      tr.push_if(x% Or tr.IF_COMMENT)
      If x% Then tr.update_num_comments(+1)

    Case "!if", "!elif"
      tr.push_if(Choice(x%, tr.IF_ACTIVE, 0))
      tr.omit_flag% = Not x%

    Case "!uncomment_if"
      tr.push_if(x% Or tr.IF_UNCOMMENT)
      If x% Then tr.update_num_comments(-1)

    Case Else
      Error "Unknown !if directive: " + lx.directive$(0)
  End Select
End Sub

Sub tr.push_if(x)
  Local i = in.num_open_files% - 1
  If tr.if_stack_sz(i) = tr.MAX_NUM_IFS Then Error "Too many !if directives"
  tr.if_stack(i, tr.if_stack_sz(i)) = x
  Inc tr.if_stack_sz(i)
End Sub

Sub tr.process_empty_lines()
  If lx.num > 2 Then sys.err$ = "has too many arguments"
  If sys.err$ = "" Then opt.set_empty_lines(lx.token_lc$(1))
  If sys.err$ <> "" Then sys.err$ = "!empty-lines directive " + sys.err$
End Sub

Sub tr.process_error()
  If lx.num <> 2 Or lx.type(1) <> TK_STRING Then
    sys.err$ = "!error directive has missing " + str.quote$("message") + " argument"
  Else
    sys.err$ = lx.string$(1)
  EndIf
End Sub

Sub tr.process_include()
  If lx.num <> 2 Or lx.type(1) <> TK_STRING Then
    sys.err$ = "#INCLUDE expects a <file> argument"
  Else
    tr.include$ = lx.string$(1)
  EndIf
End Sub

Function tr.process_info%()
  If lx.num <> 3 Then
    sys.err$ = "expects two arguments"
  ElseIf lx.token_lc$(1) <> "defined" Then
    sys.err$ = "has invalid first argument: " + lx.token_lc$(1)
  Else
    Local defined% = def.is_defined%(lx.token$(2))
    If defined% Then
      lx.replace_token(0, "' Preprocessor value " + UCase$(lx.token$(2)) + " defined", TK_COMMENT)
      lx.remove_token(2)
      lx.remove_token(1)
      tr.process_info% = sys.SUCCESS
    Else
      tr.process_info% = tr.OMIT_LINE
    EndIf
  EndIf
  If sys.err$ <> "" Then
    sys.err$ = "!info directive " + sys.err$
    tr.process_info% = sys.FAILURE
  EndIf
End Function

Sub tr.process_indent()
  If lx.num > 2 Then sys.err$ = "has too many arguments"
  If sys.err$ = "" Then opt.set_indent_sz(lx.token_lc$(1))
  If sys.err$ <> "" Then sys.err$ = "!indent directive " + sys.err$
End Sub

Sub tr.process_replace()
  Local in_group%
  Local i%
  Local gidx% = 0
  Local unreplace% = lx.directive$(0) = "!unreplace"
  Local groups$(1) = ( "", Choice(unreplace%, Chr$(0), "") )

  For i% = 1 To lx.num - 1
    Select Case lx.token_lc$(i%)
      Case "{"
        If in_group% Then
          sys.err$ = "has unexpected '{'"
        Else
          in_group% = 1
        EndIf
      Case "}"
        If in_group% Then
          in_group% = 0
          If gidx% = 0 Then
            If groups$(gidx%) = "" Then sys.err$ = "has empty <from> group"
          EndIf
          Inc gidx%
        Else
          sys.err$ = "has unexpected '}'"
        EndIf
      Case Else
        If gidx% > 1 Or (unreplace% And gidx% > 0) Then
          sys.err$ = "has too many arguments"
        ElseIf in_group% Then
          If groups$(gidx%) <> "" Then Cat groups$(gidx%), "|"
          Cat groups$(gidx%), lx.token$(i%)
        Else
          groups$(gidx%) = lx.token$(i%)
          Inc gidx%
        EndIf
    End Select

    If sys.err$ <> "" Then Exit For
  Next

  If sys.err$ = "" Then
    If in_group% Then
      sys.err$ = "has missing '}'"
    ElseIf gidx% < 1 Then
      sys.err$ = "expects <from> argument"
    Else
      tr.add_replacement(groups$(0), groups$(1))
    EndIf
  EndIf

  If sys.err$ <> "" Then sys.err$ = lx.directive$(0) + " directive " + sys.err$
End Sub

Sub tr.process_define()
  If lx.num < 2 Then sys.err$ = "expects <id> argument"
  If lx.num > 2 Then sys.err$ = "has too many arguments"
  If sys.err$ = "" Then def.define(lx.token$(1))
  If sys.err$ <> "" Then sys.err$ = "!define directive " + sys.err$
End Sub

Sub tr.process_spacing()
  If lx.num > 2 Then sys.err$ = " has too many arguments")
  If sys.err$ = "" Then opt.set_spacing(lx.token_lc$(1))
  If sys.err$ <> "" Then sys.err$ = "!spacing directive " + sys.err$
End Sub

Sub tr.clear_replacements()
  Local i%
  For i% = 0 To tr.num_replacements% - 1
    tr.replacements$(i%, 0) = ""
    tr.replacements$(i%, 1) = ""
  Next
  tr.num_replacements% = 0
End Sub

' Adds a new entry to the replacement list.
'
'   - if the entry exists then flags it as "dead" and adds a new entry to the
'     end of the list.
'   - if to_$ = Chr$(0) then expects entry to exist and flags it as "dead".
Sub tr.add_replacement(from$, to_$)
  Local f_lower$ = LCase$(from$)
  Local existing% = -1, i%

  ' Find existing replacement (if any) and flag entry dead.
  For i% = 0 To tr.num_replacements% - 1
    If f_lower$ = tr.replacements$(i%, 0) Then
      existing% = i%
      Exit For
    EndIf
  Next

  If existing% > -1 Then
    ' Flag existing entry as dead.
    tr.replacements$(existing%, 0) = Chr$(0)
    tr.replacements$(existing%, 1) = Chr$(0)
    If to_$ = Chr$(0) Then Exit Sub
  ElseIf to_$ = Chr$(0) Then
    sys.err$ = "could not find '" + from$ + "'"
    Exit Sub
  EndIf

  ' Handle too many entries.
  If tr.num_replacements% >= tr.MAX_REPLACEMENTS% Then
    sys.err$ = "too many replacements (max " + Str$(tr.MAX_REPLACEMENTS%) + ")"
    Exit Sub
  EndIf

  ' Actually add the new entry.
  tr.replacements$(tr.num_replacements%, 0) = f_lower$
  tr.replacements$(tr.num_replacements%, 1) = to_$
  Inc tr.num_replacements%
End Sub

Sub tr.dump_replacements()
  Local i%, s$
  For i% = 0 To tr.num_replacements% - 1
    Print Str$(i%) ": " ;
    Print Choice(tr.replacements$(i%, 0) = Chr$(0), "<null>", tr.replacements$(i%, 0));
    Print " => ";
    Print Choice(tr.replacements$(i%, 1) = Chr$(0), "<null>", tr.replacements$(i%, 1))
  Next
End Sub

' Removes all comment tokens from the current line.
'
' @return  tr.OMIT_LINE if as a result of calling this function the line
'          is now empty, otherwise 1.
Function tr.remove_comments%()
  If lx.num = 0 Then Exit Function

  Local i%
  Do While i% < lx.num
    If lx.type(i%) = TK_COMMENT Then lx.remove_token(i%)
    Inc i%
  Loop
  tr.remove_comments% = Choice(lx.num = 0, tr.OMIT_LINE, sys.SUCCESS)
End Function
' END:   #Include "/home/thwill/github/cmm2-sptools/src/sptrans/tests/../trans.inc"

keywords.init()

add_test("test_transpile_includes")
add_test("test_parse_replace")
add_test("test_parse_replace_given_errors")
add_test("test_parse_unreplace")
add_test("test_parse_unreplace_given_errs")
add_test("test_parse_given_too_many_rpl")
add_test("test_apply_replace")
add_test("test_apply_replace_groups")
add_test("test_apply_replace_patterns")
add_test("test_replace_fails_if_too_long")
add_test("test_replace_with_fewer_tokens")
add_test("test_replace_with_more_tokens")
add_test("test_replace_given_new_rpl")
add_test("test_apply_unreplace")
add_test("test_undef_given_defined")
add_test("test_undef_given_undefined")
add_test("test_undef_given_id_too_long")
add_test("test_undef_is_case_insensitive")
add_test("test_comment_if")
add_test("test_comment_if_not")
add_test("test_uncomment_if")
add_test("test_uncomment_if_not")
add_test("test_unknown_directive")
add_test("test_ifdef_given_set")
add_test("test_ifdef_given_unset")
add_test("test_ifdef_given_0_args")
add_test("test_ifdef_given_2_args")
add_test("test_ifdef_is_case_insensitive")
add_test("test_ifdef_nested_1")
add_test("test_ifdef_nested_2")
add_test("test_ifdef_nested_3")
add_test("test_ifdef_nested_4")
add_test("test_ifndef_given_set")
add_test("test_ifndef_given_unset")
add_test("test_ifndef_given_0_args")
add_test("test_ifndef_given_2_args")
add_test("test_ifndef_is_case_insensitive")
add_test("test_ifndef_nested_1")
add_test("test_ifndef_nested_2")
add_test("test_ifndef_nested_3")
add_test("test_ifndef_nested_4")
add_test("test_define_given_defined")
add_test("test_define_given_undefined")
add_test("test_define_given_id_too_long")
add_test("test_define_is_case_insensitive")
add_test("test_omit_directives_from_output")
add_test("test_endif_given_no_if")
add_test("test_endif_given_args")
add_test("test_endif_given_trail_comment")
add_test("test_error_directive")
add_test("test_omit_and_line_spacing")
add_test("test_comments_directive")
add_test("test_always_defined_values")
add_test("test_always_undefined_values")
add_test("test_if_given_true")
add_test("test_if_given_false")
add_test("test_if_given_nested")
add_test("test_else_given_if_active")
add_test("test_else_given_else_active")
add_test("test_else_given_no_if")
add_test("test_too_many_elses")
add_test("test_elif_given_if_active")
add_test("test_elif_given_elif_1_active")
add_test("test_elif_given_elif_2_active")
add_test("test_elif_given_else_active")
add_test("test_elif_given_no_expression")
add_test("test_elif_given_invalid_expr")
add_test("test_elif_given_no_if")
add_test("test_elif_given_comment_if")
add_test("test_elif_given_uncomment_if")
add_test("test_elif_given_ifdef")
add_test("test_elif_given_shortcut_expr")
add_test("test_info_defined")

run_tests()

End

Sub setup_test()
  opt.init()
  def.init()

  ' TODO: extract into trans.init() or trans.reset().
  tr.clear_replacements()
  tr.include$ = ""
  tr.omit_flag% = 0
  tr.empty_line_flag% = 0
  tr.omitted_line_flag% = 0

  Local i%, j%
  For i% = Bound(tr.num_comments(), 0) To Bound(tr.num_comments(), 1)
    tr.num_comments(i%) = 0
  Next
  For i% = Bound(tr.if_stack(), 0) To Bound(tr.if_stack(), 1)
    For j% = Bound(tr.if_stack(), 0) To Bound(tr.if_stack(), 2)
      tr.if_stack(i%, j%) = 0
    Next
    tr.if_stack_sz(i%) = 0
  Next

  sys.err$ = ""
End Sub

Sub test_transpile_includes()
  ' Given #INCLUDE statement.
  setup_test()
  assert_int_equals(sys.SUCCESS, lx.parse_basic%("#include " + str.quote$("foo/bar.inc")))
  assert_int_equals(tr.INCLUDE_FILE, tr.transpile_includes%())
  assert_no_error()
  assert_string_equals("foo/bar.inc", tr.include$)

  ' Given #INCLUDE statement preceded by whitespace.
  setup_test()
  assert_int_equals(sys.SUCCESS, lx.parse_basic%("  #include " + str.quote$("foo/bar.inc")))
  assert_int_equals(tr.INCLUDE_FILE, tr.transpile_includes%())
  assert_no_error()
  assert_string_equals("foo/bar.inc", tr.include$)

  ' Given #INCLUDE statement followed by whitespace.
  setup_test()
  assert_int_equals(sys.SUCCESS, lx.parse_basic%("#include " + str.quote$("foo/bar.inc") + "  "))
  assert_int_equals(tr.INCLUDE_FILE, tr.transpile_includes%())
  assert_no_error()
  assert_string_equals("foo/bar.inc", tr.include$)

  ' Given other statement.
  setup_test()
  assert_int_equals(sys.SUCCESS, lx.parse_basic%("Print " + str.quote$("Hello World")))
  assert_int_equals(sys.SUCCESS, tr.transpile_includes%())
  assert_no_error()
  assert_string_equals("", tr.include$)

  ' Given missing argument.
  setup_test()
  assert_int_equals(sys.SUCCESS, lx.parse_basic%("#include"))
  assert_int_equals(sys.FAILURE, tr.transpile_includes%())
  assert_error("#INCLUDE expects a <file> argument")
  assert_string_equals("", tr.include$)

  ' Given non-string argument.
  setup_test()
  assert_int_equals(sys.SUCCESS, lx.parse_basic%("#include foo"))
  assert_int_equals(sys.FAILURE, tr.transpile_includes%())
  assert_error("#INCLUDE expects a <file> argument")
  assert_string_equals("", tr.include$)

  ' Given too many arguments.
  setup_test()
  assert_int_equals(sys.SUCCESS, lx.parse_basic%("#include " + str.quote$("foo/bar.inc") + " " + str.quote$("wombat.inc")))
  assert_int_equals(sys.FAILURE, tr.transpile_includes%())
  assert_error("#INCLUDE expects a <file> argument")
  assert_string_equals("", tr.include$)

  ' Given #INCLUDE is not the first token on the line.
  setup_test()
  assert_int_equals(sys.SUCCESS, lx.parse_basic%("Dim i% : #include " + str.quote$("foo/bar.inc")))
  assert_int_equals(sys.SUCCESS, tr.transpile_includes%())
  assert_no_error()
  assert_string_equals("", tr.include$)
End Sub

Sub test_parse_replace()
  expect_transpile_omits("'!replace DEF Sub")
  expect_replacement(0, "def", "Sub")

  expect_transpile_omits("'!replace ENDPROC { End Sub }")
  expect_replacement(1, "endproc", "End|Sub")

  expect_transpile_omits("'!replace { THEN ENDPROC } { Then Exit Sub }")
  expect_replacement(2, "then|endproc", "Then|Exit|Sub")

  expect_transpile_omits("'!replace GOTO%% { Goto %1 }")
  expect_replacement(3, "goto%%", "Goto|%1")

  expect_transpile_omits("'!replace { THEN %% } { Then Goto %1 }")
  expect_replacement(4, "then|%%", "Then|Goto|%1")

  expect_transpile_omits("'!replace '%% { CRLF$ %1 }")
  expect_replacement(5, "'%%", "CRLF$|%1")

  expect_transpile_omits("'!replace &%h &h%1")
  expect_replacement(6, "&%h", "&h%1")

  expect_transpile_omits("'!replace foo")
  expect_replacement(7, "foo", "")

  expect_transpile_omits("'!replace {foo}")
  expect_replacement(7, Chr$(0), Chr$(0))
  expect_replacement(8, "foo", "")

  expect_transpile_omits("'!replace foo {}")
  expect_replacement(8, Chr$(0), Chr$(0))
  expect_replacement(9, "foo", "")
End Sub

Sub test_parse_replace_given_errors()
  expect_transpile_error("'!replace", "!replace directive expects <from> argument")
  assert_int_equals(0, tr.num_replacements%)

  expect_transpile_error("'!replace {}", "!replace directive has empty <from> group")
  assert_int_equals(0, tr.num_replacements%)

  expect_transpile_error("'!replace {} y", "!replace directive has empty <from> group")
  assert_int_equals(0, tr.num_replacements%)

  expect_transpile_error("'!replace { x y", "!replace directive has missing '}'")
  assert_int_equals(0, tr.num_replacements%)

  expect_transpile_error("'!replace { x } { y z", "!replace directive has missing '}'")
  assert_int_equals(0, tr.num_replacements%)

  expect_transpile_error("'!replace { x } { y } z", "!replace directive has too many arguments")
  assert_int_equals(0, tr.num_replacements%)

  expect_transpile_error("'!replace { x } { y } { z }", "!replace directive has too many arguments")
  assert_int_equals(0, tr.num_replacements%)

  expect_transpile_error("'!replace { {", "!replace directive has unexpected '{'")
  assert_int_equals(0, tr.num_replacements%)

  expect_transpile_error("'!replace foo }", "!replace directive has unexpected '}'")
  assert_int_equals(0, tr.num_replacements%)
End Sub

Sub test_parse_given_too_many_rpl()
  Local i%

  For i% = 0 To tr.MAX_REPLACEMENTS% - 1
    expect_transpile_omits("'!replace a" + Str$(i%) + " b")
  Next

  expect_transpile_error("'!replace foo bar", "!replace directive too many replacements (max 200)")
End Sub

Sub test_parse_unreplace()
  expect_transpile_omits("'!replace foo bar")
  expect_transpile_omits("'!replace wom bat")
  expect_transpile_omits("'!unreplace foo")

  assert_int_equals(2, tr.num_replacements%)
  expect_replacement(0, Chr$(0), Chr$(0))
  expect_replacement(1, "wom", "bat")
End Sub

Sub test_parse_unreplace_given_errs()
  ' Test given missing argument to directive.
  expect_transpile_error("'!unreplace", "!unreplace directive expects <from> argument")

  ' Test given directive has too many arguments.
  expect_transpile_error("'!unreplace { a b } c", "!unreplace directive has too many arguments")

  ' Test given replacement not present.
  expect_transpile_omits("'!replace wom bat")
  expect_transpile_error("'!unreplace foo", "!unreplace directive could not find 'foo'")
  assert_int_equals(1, tr.num_replacements%)
  expect_replacement(0, "wom", "bat")
End Sub

Sub test_apply_replace()
  expect_transpile_omits("'!replace x      y")
  expect_transpile_omits("'!replace &hFFFF z")
  expect_transpile_succeeds("Dim x = &hFFFF ' comment")

  expect_token_count(5)
  expect_token(0, TK_KEYWORD, "Dim")
  expect_token(1, TK_IDENTIFIER, "y")
  expect_token(2, TK_SYMBOL, "=")
  expect_token(3, TK_IDENTIFIER, "z")
  expect_token(4, TK_COMMENT, "' comment")
  assert_string_equals("Dim y = z ' comment", lx.line$)
End Sub

Sub test_apply_replace_groups()
  expect_transpile_omits("'!replace ab { cd ef }")
  expect_transpile_succeeds("ab gh ij")
  expect_token_count(4)
  expect_token(0, TK_IDENTIFIER, "cd")
  expect_token(1, TK_IDENTIFIER, "ef")
  expect_token(2, TK_IDENTIFIER, "gh")
  expect_token(3, TK_IDENTIFIER, "ij")

  setup_test()
  expect_transpile_omits("'!replace {ab cd} ef")
  expect_transpile_succeeds("ab cd gh ij")
  expect_token_count(3)
  expect_token(0, TK_IDENTIFIER, "ef")
  expect_token(1, TK_IDENTIFIER, "gh")
  expect_token(2, TK_IDENTIFIER, "ij")
End Sub

Sub test_apply_replace_patterns()
  setup_test()
  expect_transpile_omits("'!replace { DEF PROC%% } { SUB proc%1 }")
  expect_transpile_succeeds("foo DEF PROCWOMBAT bar")
  expect_token_count(4)
  expect_token(0, TK_IDENTIFIER, "foo")
  expect_token(1, TK_KEYWORD,    "SUB")
  expect_token(2, TK_IDENTIFIER, "procWOMBAT") ' Note don't want to change case of WOMBAT.
  expect_token(3, TK_IDENTIFIER, "bar")

  setup_test()
  expect_transpile_omits("'!replace GOTO%d { Goto %1 }")
  expect_transpile_succeeds("foo GOTO1234 bar")
  expect_token_count(4)
  expect_token(0, TK_IDENTIFIER, "foo")
  expect_token(1, TK_KEYWORD   , "Goto")
  expect_token(2, TK_NUMBER,     "1234")
  expect_token(3, TK_IDENTIFIER, "bar")

  setup_test()
  expect_transpile_omits("'!replace { THEN %d } { Then Goto %1 }")

  ' Test %d pattern matches decimal digits ...
  expect_transpile_succeeds("foo THEN 1234 bar")
  expect_token_count(5)
  expect_token(0, TK_IDENTIFIER, "foo")
  expect_token(1, TK_KEYWORD   , "Then")
  expect_token(2, TK_KEYWORD   , "Goto")
  expect_token(3, TK_NUMBER,     "1234")
  expect_token(4, TK_IDENTIFIER, "bar")

  ' ... but it should not match other characters.
  expect_transpile_succeeds("foo THEN wombat bar")
  expect_token_count(4)
  expect_token(0, TK_IDENTIFIER, "foo")
  expect_token(1, TK_KEYWORD   , "THEN")
  expect_token(2, TK_IDENTIFIER, "wombat")
  expect_token(3, TK_IDENTIFIER, "bar")

  setup_test()
  expect_transpile_omits("'!replace { PRINT '%% } { ? : ? %1 }")
  expect_transpile_succeeds("foo PRINT '" + str.quote$("wombat") + " bar")
  expect_token_count(6)
  expect_token(0, TK_IDENTIFIER, "foo")
  expect_token(1, TK_SYMBOL,     "?")
  expect_token(2, TK_SYMBOL,     ":")
  expect_token(3, TK_SYMBOL,     "?")
  expect_token(4, TK_STRING,     str.quote$("wombat"))
  expect_token(5, TK_IDENTIFIER, "bar")

  setup_test()
  expect_transpile_omits("'!replace '%% { : ? %1 }")
  expect_transpile_succeeds("foo PRINT '" + str.quote$("wombat") + " bar")
  expect_token_count(6)
  expect_token(0, TK_IDENTIFIER, "foo")
  expect_token(1, TK_KEYWORD,    "PRINT")
  expect_token(2, TK_SYMBOL,     ":")
  expect_token(3, TK_SYMBOL,     "?")
  expect_token(4, TK_STRING,     str.quote$("wombat"))
  expect_token(5, TK_IDENTIFIER, "bar")

  setup_test()
  expect_transpile_omits("'!replace REM%% { ' %1 }")
  expect_transpile_succeeds("foo REM This is a comment")
  expect_token_count(2)
  expect_token(0, TK_IDENTIFIER, "foo")
  expect_token(1, TK_COMMENT, "' This is a comment")

  setup_test()
  expect_transpile_omits("'!replace { Spc ( } { Space$ ( }")
  expect_transpile_succeeds("foo Spc(5) bar")
  expect_token_count(6)
  expect_token(0, TK_IDENTIFIER, "foo")
  expect_token(1, TK_KEYWORD,    "Space$")
  expect_token(2, TK_SYMBOL,     "(")
  expect_token(3, TK_NUMBER,     "5")
  expect_token(4, TK_SYMBOL,     ")")
  expect_token(5, TK_IDENTIFIER, "bar")

  ' Test %h pattern matches hex digits ...
  setup_test()
  expect_transpile_omits("'!replace GOTO%h { Goto %1 }")
  expect_transpile_succeeds("foo GOTOabcdef0123456789 bar")
  expect_token_count(4)
  expect_token(0, TK_IDENTIFIER, "foo")
  expect_token(1, TK_KEYWORD,    "Goto")
  expect_token(2, TK_IDENTIFIER, "abcdef0123456789")
  expect_token(3, TK_IDENTIFIER, "bar")

  ' ... but it should not match other characters.
  expect_transpile_succeeds("foo GOTOxyz bar")
  expect_token_count(3)
  expect_token(0, TK_IDENTIFIER, "foo")
  expect_token(1, TK_IDENTIFIER, "GOTOxyz")
  expect_token(2, TK_IDENTIFIER, "bar")
End Sub

Sub test_replace_fails_if_too_long()
  expect_transpile_omits("'!replace foo foobar")

  ' Test where replaced string should be 255 characters.
  Local s$ = String$(248, "a")
  Cat s$, " foo"
  assert_int_equals(252, Len(s$))
  expect_transpile_succeeds(s$)
  expect_token_count(2)
  expect_token(0, TK_IDENTIFIER, String$(248, "a"))
  expect_token(1, TK_IDENTIFIER, "foobar")


  ' Test where replaced string should be 256 characters.
  s$ = String$(251, "a")
  Cat s$, " foo"
  assert_int_equals(255, Len(s$))
  expect_transpile_error(s$, "applying replacement makes line > 255 characters")
End Sub

Sub test_replace_with_fewer_tokens()
  ' Replace 1 token with 0.
  expect_transpile_omits("'!replace bar")
  expect_transpile_succeeds("foo bar wom")
  expect_token_count(2)
  expect_token(0, TK_IDENTIFIER, "foo")
  expect_token(1, TK_IDENTIFIER, "wom")

  ' Removal of all tokens.
  setup_test()
  expect_transpile_omits("'!replace bar")
  expect_transpile_succeeds("bar bar bar", 1)
  expect_token_count(0)

  ' Replace 2 tokens with 1.
  setup_test()
  expect_transpile_omits("'!replace { foo bar } wom")
  expect_transpile_succeeds("foo bar foo bar snafu")
  expect_token_count(3)
  expect_token(0, TK_IDENTIFIER, "wom")
  expect_token(1, TK_IDENTIFIER, "wom")
  expect_token(2, TK_IDENTIFIER, "snafu")

  ' Note that we don't end up with the single token "foo" because once we have
  ' applied a replacement we do not recursively apply that replacement to the
  ' already replaced text.
  setup_test()
  expect_transpile_omits("'!replace { foo bar } foo")
  expect_transpile_succeeds("foo bar bar")
  expect_token_count(2)
  expect_token(0, TK_IDENTIFIER, "foo")
  expect_token(1, TK_IDENTIFIER, "bar")

  ' Replace 3 tokens with 1 - again note we don't just end up with "foo".
  setup_test()
  expect_transpile_omits("'!replace { foo bar wom } foo")
  expect_transpile_succeeds("foo bar wom bar wom")
  expect_token_count(3)
  expect_token(0, TK_IDENTIFIER, "foo")
  expect_token(1, TK_IDENTIFIER, "bar")
  expect_token(2, TK_IDENTIFIER, "wom")

  ' Replace 3 tokens with 2 - and again we don't just end up with "foo bar".
  setup_test()
  expect_transpile_omits("'!replace { foo bar wom } { foo bar }")
  expect_transpile_succeeds("foo bar wom wom")
  expect_token_count(3)
  expect_token(0, TK_IDENTIFIER, "foo")
  expect_token(1, TK_IDENTIFIER, "bar")
  expect_token(2, TK_IDENTIFIER, "wom")
End Sub

Sub test_replace_with_more_tokens()
  ' Replace 1 token with 2 - note that we don't get infinite recursion because
  ' once we have applied the replacement text we not not recusively apply the
  ' replacement to the already replaced text.
  expect_transpile_omits("'!replace foo { foo bar }")
  expect_transpile_succeeds("foo wom")
  expect_token_count(3)
  expect_token(0, TK_IDENTIFIER, "foo")
  expect_token(1, TK_IDENTIFIER, "bar")
  expect_token(2, TK_IDENTIFIER, "wom")

  setup_test()
  expect_transpile_omits("'!replace foo { bar foo }")
  expect_transpile_succeeds("foo wom foo")
  expect_token_count(5)
  expect_token(0, TK_IDENTIFIER, "bar")
  expect_token(1, TK_IDENTIFIER, "foo")
  expect_token(2, TK_IDENTIFIER, "wom")
  expect_token(3, TK_IDENTIFIER, "bar")
  expect_token(4, TK_IDENTIFIER, "foo")

  ' Ensure replacement applied for multiple matches.
  setup_test()
  expect_transpile_omits("'!replace foo { bar foo }")
  expect_transpile_succeeds("foo foo")
  expect_token_count(4)
  expect_token(0, TK_IDENTIFIER, "bar")
  expect_token(1, TK_IDENTIFIER, "foo")
  expect_token(2, TK_IDENTIFIER, "bar")
  expect_token(3, TK_IDENTIFIER, "foo")

  ' Replace 3 tokens with 4.
  setup_test()
  expect_transpile_omits("'!replace { foo bar wom } { foo bar wom foo }")
  expect_transpile_succeeds("foo bar wom bar wom")
  expect_token_count(6)
  expect_token(0, TK_IDENTIFIER, "foo")
  expect_token(1, TK_IDENTIFIER, "bar")
  expect_token(2, TK_IDENTIFIER, "wom")
  expect_token(3, TK_IDENTIFIER, "foo")
  expect_token(4, TK_IDENTIFIER, "bar")
  expect_token(5, TK_IDENTIFIER, "wom")
End Sub

Sub test_replace_given_new_rpl()
  expect_transpile_omits("'!replace foo bar")
  expect_transpile_succeeds("foo wom bill")
  expect_token_count(3)
  expect_token(0, TK_IDENTIFIER, "bar")
  expect_token(1, TK_IDENTIFIER, "wom")
  expect_token(2, TK_IDENTIFIER, "bill")

  expect_transpile_omits("'!replace foo snafu")
  expect_transpile_succeeds("foo wom bill")
  expect_token_count(3)
  expect_token(0, TK_IDENTIFIER, "snafu")
  expect_token(1, TK_IDENTIFIER, "wom")
  expect_token(2, TK_IDENTIFIER, "bill")
End Sub

Sub test_apply_unreplace()
  expect_transpile_omits("'!replace foo bar")
  expect_transpile_omits("'!replace wom bat")
  expect_transpile_omits("'!replace bill ben")
  expect_replacement(0, "foo", "bar")
  expect_replacement(1, "wom", "bat")
  expect_replacement(2, "bill", "ben")
  expect_replacement(3, "", "")

  expect_transpile_succeeds("foo wom bill")
  expect_token_count(3)
  expect_token(0, TK_IDENTIFIER, "bar")
  expect_token(1, TK_IDENTIFIER, "bat")
  expect_token(2, TK_IDENTIFIER, "ben")

  expect_transpile_omits("'!unreplace wom")
  expect_replacement(0, "foo", "bar")
  expect_replacement(1, Chr$(0), Chr$(0))
  expect_replacement(2, "bill", "ben")
  expect_replacement(3, "", "")

  expect_transpile_succeeds("foo wom bill")
  expect_token_count(3)
  expect_token(0, TK_IDENTIFIER, "bar")
  expect_token(1, TK_IDENTIFIER, "wom")
  expect_token(2, TK_IDENTIFIER, "ben")
End Sub

Sub test_undef_given_defined()
  def.define("foo")
  def.define("bar")

  expect_transpile_omits("'!undef foo")
  assert_int_equals(0, def.is_defined%("foo"))
  assert_int_equals(1, def.is_defined%("bar"))

  expect_transpile_omits("'!undef bar")
  assert_int_equals(0, def.is_defined%("foo"))
  assert_int_equals(0, def.is_defined%("bar"))
End Sub

Sub test_undef_given_undefined()
  expect_transpile_error("'!undef foo", "!undef directive 'foo' is not defined")
  expect_transpile_error("'!undef BAR", "!undef directive 'BAR' is not defined")
End Sub

Sub test_undef_given_id_too_long()
  Local line$ = "'!undef flag5678901234567890123456789012345678901234567890123456789012345"
  Local emsg$ = "!undef directive identifier too long, max 64 chars"
  expect_transpile_error(line$, emsg$)
End Sub

Sub test_undef_is_case_insensitive()
  def.define("foo")
  def.define("BAR")

  expect_transpile_omits("'!undef FOO")
  assert_int_equals(0, def.is_defined%("foo"))
  assert_int_equals(1, def.is_defined%("BAR"))

  expect_transpile_omits("'!undef bar")
  assert_int_equals(0, def.is_defined%("foo"))
  assert_int_equals(0, def.is_defined%("BAR"))
End Sub

Sub test_comment_if()
  ' 'foo' is set, code inside !comment_if block should be commented.
  expect_transpile_omits("'!define foo")
  expect_transpile_omits("'!comment_if foo")
  expect_transpile_succeeds("one")
  expect_token_count(1)
  expect_token(0, TK_COMMENT, "' one")
  expect_transpile_succeeds("two")
  expect_token_count(1)
  expect_token(0, TK_COMMENT, "' two")
  expect_transpile_omits("'!endif")

  ' Code outside the block should not be commented.
  expect_transpile_succeeds("three")
  expect_token_count(1)
  expect_token(0, TK_IDENTIFIER, "three")
End Sub

Sub test_comment_if_not()
  ' 'foo' is NOT set, code inside !comment_if NOT block should be commented.
  expect_transpile_omits("'!comment_if not foo")
  expect_transpile_succeeds("one")
  expect_token_count(1)
  expect_token(0, TK_COMMENT, "' one")
  expect_transpile_succeeds("two")
  expect_token_count(1)
  expect_token(0, TK_COMMENT, "' two")
  expect_transpile_omits("'!endif")

  ' 'foo' is set, code inside !comment_if NOT block should NOT be commented.
  expect_transpile_omits("'!define foo")
  expect_transpile_omits("'!comment_if not foo")
  expect_transpile_succeeds("three")
  expect_token_count(1)
  expect_token(0, TK_IDENTIFIER, "three")
  expect_transpile_omits("'!endif")
End Sub

Sub test_uncomment_if()
  ' 'foo' is set, code inside !uncomment_if block should be uncommented.
  expect_transpile_omits("'!define foo")
  expect_transpile_omits("'!uncomment_if foo")

  expect_transpile_succeeds("' one")
  assert_string_equals(" one", lx.line$)
  expect_token_count(1)
  expect_token(0, TK_IDENTIFIER, "one")

  expect_transpile_succeeds("REM two")
  assert_string_equals(" two", lx.line$)
  expect_token_count(1)
  expect_token(0, TK_IDENTIFIER, "two")

  expect_transpile_succeeds("'' three")
  assert_string_equals("' three", lx.line$)
  expect_token_count(1)
  expect_token(0, TK_COMMENT, "' three")

  expect_transpile_omits("'!endif")

  ' Code outside the block should not be uncommented.
  expect_transpile_succeeds("' four")
  assert_string_equals("' four", lx.line$)
  expect_token_count(1)
  expect_token(0, TK_COMMENT, "' four")
End Sub

Sub test_uncomment_if_not()
  ' 'foo' is NOT set, code inside !uncomment_if NOT block should be uncommented.
  expect_transpile_omits("'!uncomment_if not foo")

  expect_transpile_succeeds("' one")
  assert_string_equals(" one", lx.line$)
  expect_token_count(1)
  expect_token(0, TK_IDENTIFIER, "one")

  expect_transpile_succeeds("REM two")
  assert_string_equals(" two", lx.line$)
  expect_token_count(1)
  expect_token(0, TK_IDENTIFIER, "two")

  expect_transpile_succeeds("'' three")
  assert_string_equals("' three", lx.line$)
  expect_token_count(1)
  expect_token(0, TK_COMMENT, "' three")
  expect_transpile_omits("'!endif")

  ' 'foo' is set, code inside !uncomment_if NOT block should NOT be uncommented.
  expect_transpile_omits("'!define foo")
  expect_transpile_omits("'!uncomment_if not foo")
  expect_transpile_succeeds("' four")
  assert_string_equals("' four", lx.line$)
  expect_token_count(1)
  expect_token(0, TK_COMMENT, "' four")
  expect_transpile_omits("'!endif")
End Sub

Sub test_ifdef_given_set()
  ' FOO is set so all code within !ifdef FOO is included.
  expect_transpile_omits("'!define FOO")
  expect_transpile_omits("'!ifdef FOO")
  expect_transpile_succeeds("one")
  expect_token_count(1)
  expect_token(0, TK_IDENTIFIER, "one")
  expect_transpile_succeeds("two")
  expect_token_count(1)
  expect_token(0, TK_IDENTIFIER, "two")
  expect_transpile_omits("'!endif")

  ' Code outside the block is included.
  expect_transpile_succeeds("three")
  expect_token_count(1)
  expect_token(0, TK_IDENTIFIER, "three")
End Sub

Sub test_ifdef_given_unset()
  ' FOO is unset so all code within !ifdef FOO is excluded.
  expect_transpile_omits("'!ifdef FOO")
  expect_transpile_omits("one")
  expect_transpile_omits("two")
  expect_transpile_omits("'!endif")

  ' Code outside the block is included.
  expect_transpile_succeeds("three")
  expect_token_count(1)
  expect_token(0, TK_IDENTIFIER, "three")
End Sub

Sub test_ifdef_given_0_args()
  expect_transpile_error("'!ifdef", "!ifdef directive expects 1 argument")
End Sub

Sub test_ifdef_given_2_args()
  expect_transpile_error("'!ifdef not bar", "!ifdef directive expects 1 argument")
End Sub

Sub test_ifdef_is_case_insensitive()
  def.define("foo")
  expect_transpile_omits("'!ifdef FOO")
  expect_transpile_succeeds("one")
  expect_token_count(1)
  expect_token(0, TK_IDENTIFIER, "one")
  expect_transpile_omits("'!endif")
End Sub

Sub test_ifdef_nested_1()
  ' FOO and BAR are both unset so all code within !ifdef FOO is excluded.
  expect_transpile_omits("'!ifdef FOO")
  expect_transpile_omits("one")
  expect_transpile_omits("'!ifdef BAR")
  expect_transpile_omits("two")
  expect_transpile_omits("'!endif")
  expect_transpile_omits("'!endif")

  ' Code outside the block is included.
  expect_transpile_succeeds("three")
  expect_token_count(1)
  expect_token(0, TK_IDENTIFIER, "three")
End Sub

Sub test_ifdef_nested_2()
  ' FOO is set and BAR is unset so code within !ifdef BAR is excluded.
  expect_transpile_omits("'!define FOO")
  expect_transpile_omits("'!ifdef FOO")
  expect_transpile_succeeds("one")
  expect_token_count(1)
  expect_token(0, TK_IDENTIFIER, "one")
  expect_transpile_omits("'!ifdef BAR")
  expect_transpile_omits("two")
  expect_transpile_omits("'!endif")
  expect_transpile_omits("'!endif")

  ' Code outside the block is included.
  expect_transpile_succeeds("three")
  expect_token_count(1)
  expect_token(0, TK_IDENTIFIER, "three")
End Sub

Sub test_ifdef_nested_3()
  ' BAR is set and FOO is unset so all code within !ifdef FOO is excluded.
  expect_transpile_omits("'!define BAR")
  expect_transpile_omits("'!ifdef FOO")
  expect_transpile_omits("one")
  expect_transpile_omits("'!ifdef BAR")
  expect_transpile_omits("two")
  expect_transpile_omits("'!endif")
  expect_transpile_omits("'!endif")

  ' Code outside the block is included.
  expect_transpile_succeeds("three")
  expect_token_count(1)
  expect_token(0, TK_IDENTIFIER, "three")
End Sub

Sub test_ifdef_nested_4()
  ' FOO and BAR are both set so all code within !ifdef FOO and !ifdef BAR is included.
  expect_transpile_omits("'!define FOO")
  expect_transpile_omits("'!define BAR")
  expect_transpile_omits("'!ifdef FOO")
  expect_transpile_succeeds("one")
  expect_token_count(1)
  expect_token(0, TK_IDENTIFIER, "one")
  expect_transpile_omits("'!ifdef BAR")
  expect_transpile_succeeds("two")
  expect_token_count(1)
  expect_token(0, TK_IDENTIFIER, "two")
  expect_transpile_omits("'!endif")
  expect_transpile_omits("'!endif")

  ' Code outside the block is included.
  expect_transpile_succeeds("three")
  expect_token_count(1)
  expect_token(0, TK_IDENTIFIER, "three")
End Sub

Sub test_ifndef_given_set()
  ' FOO is set so all code within !ifndef FOO is excluded.
  expect_transpile_omits("'!define FOO")
  expect_transpile_omits("'!ifndef FOO")
  expect_transpile_omits("one")
  expect_transpile_omits("two")
  expect_transpile_omits("'!endif")

  ' Code outside the block is included.
  expect_transpile_succeeds("three")
  expect_token_count(1)
  expect_token(0, TK_IDENTIFIER, "three")
End Sub

Sub test_ifndef_given_unset()
  ' FOO is unset so all code within !ifndef FOO is included.
  expect_transpile_omits("'!ifndef FOO")
  expect_transpile_succeeds("one")
  expect_token_count(1)
  expect_token(0, TK_IDENTIFIER, "one")
  expect_transpile_succeeds("two")
  expect_token_count(1)
  expect_token(0, TK_IDENTIFIER, "two")
  expect_transpile_omits("'!endif")

  ' Code outside the block is included.
  expect_transpile_succeeds("three")
  expect_token_count(1)
  expect_token(0, TK_IDENTIFIER, "three")
End Sub

Sub test_ifndef_given_0_args()
  expect_transpile_error("'!ifndef", "!ifndef directive expects 1 argument")
End Sub

Sub test_ifndef_given_2_args()
  expect_transpile_error("'!ifndef not bar", "!ifndef directive expects 1 argument")
End Sub

Sub test_ifndef_is_case_insensitive()
  def.define("foo")
  expect_transpile_omits("'!ifndef FOO")
  expect_transpile_omits("one")
  expect_token_count(0)
  expect_transpile_omits("'!endif")
End Sub

Sub test_ifndef_nested_1()
  ' FOO and BAR are both unset so all code within !ifndef FOO is included.
  expect_transpile_omits("'!ifndef FOO")
  expect_transpile_succeeds("one")
  expect_token_count(1)
  expect_token(0, TK_IDENTIFIER, "one")
  expect_transpile_omits("'!ifndef BAR")
  expect_transpile_succeeds("two")
  expect_token_count(1)
  expect_token(0, TK_IDENTIFIER, "two")
  expect_transpile_omits("'!endif")
  expect_transpile_omits("'!endif")

  ' Code outside the block is included.
  expect_transpile_succeeds("three")
  expect_token_count(1)
  expect_token(0, TK_IDENTIFIER, "three")
End Sub

Sub test_ifndef_nested_2()
  ' FOO is set and BAR is unset so all code within !ifndef FOO is excluded.
  expect_transpile_omits("'!define FOO")
  expect_transpile_omits("'!ifndef FOO")
  expect_transpile_omits("one")
  expect_transpile_omits("'!ifndef BAR")
  expect_transpile_omits("two")
  expect_transpile_omits("'!endif")
  expect_transpile_omits("'!endif")

  ' Code outside the block is included.
  expect_transpile_succeeds("three")
  expect_token_count(1)
  expect_token(0, TK_IDENTIFIER, "three")
End Sub

Sub test_ifndef_nested_3()
  ' BAR is set and FOO is unset so all code within !ifndef BAR is excluded.
  expect_transpile_omits("'!define BAR")
  expect_transpile_omits("'!ifndef FOO")
  expect_transpile_succeeds("one")
  expect_token_count(1)
  expect_token(0, TK_IDENTIFIER, "one")
  expect_transpile_omits("'!ifndef BAR")
  expect_transpile_omits("two")
  expect_transpile_omits("'!endif")
  expect_transpile_omits("'!endif")

  ' Code outside the block is included.
  expect_transpile_succeeds("three")
  expect_token_count(1)
  expect_token(0, TK_IDENTIFIER, "three")
End Sub

Sub test_ifndef_nested_4()
  ' FOO and BAR are both set so all code within !ifndef FOO and !ifndef BAR is excluded.
  expect_transpile_omits("'!define FOO")
  expect_transpile_omits("'!define BAR")
  expect_transpile_omits("'!ifndef FOO")
  expect_transpile_omits("one")
  expect_transpile_omits("'!ifndef BAR")
  expect_transpile_omits("two")
  expect_transpile_omits("'!endif")
  expect_transpile_omits("'!endif")

  ' Code outside the block is included.
  expect_transpile_succeeds("three")
  expect_token_count(1)
  expect_token(0, TK_IDENTIFIER, "three")
End Sub

Sub test_define_given_defined()
  expect_transpile_omits("'!define foo")
  assert_int_equals(1, def.is_defined%("foo"))

  expect_transpile_error("'!define foo", "!define directive 'foo' is already defined")
End Sub

Sub test_define_given_undefined()
  expect_transpile_omits("'!define foo")
  assert_int_equals(1, def.is_defined%("foo"))

  expect_transpile_omits("'!define BAR")
  assert_int_equals(1, def.is_defined%("BAR"))
End Sub

Sub test_define_given_id_too_long()
  Local id$ = "flag567890123456789012345678901234567890123456789012345678901234"

  expect_transpile_omits("'!define " + id$)
  assert_int_equals(1, def.is_defined%(id$))

  expect_transpile_error("'!define " + id$ + "5", "!define directive identifier too long, max 64 chars")
End Sub

Sub test_define_is_case_insensitive()
  expect_transpile_omits("'!define foo")
  assert_int_equals(1, def.is_defined%("FOO"))

  expect_transpile_error("'!define FOO", "!define directive 'FOO' is already defined")
End Sub

Sub test_omit_directives_from_output()
  setup_test()
  expect_transpile_omits("'!define FOO")
  expect_transpile_omits("'!undef FOO")

  setup_test()
  expect_transpile_omits("'!comments on")

  setup_test()
  expect_transpile_omits("'!comment_if FOO")
  expect_transpile_omits("'!define FOO")
  expect_transpile_omits("'!comment_if FOO")

  setup_test()
  expect_transpile_omits("'!empty-lines 1")

  setup_test()
  expect_transpile_omits("'!ifdef FOO")
  expect_transpile_omits("'!define FOO")
  expect_transpile_omits("'!ifdef FOO")

  setup_test()
  expect_transpile_omits("'!ifndef FOO")
  expect_transpile_omits("'!define FOO")
  expect_transpile_omits("'!ifndef FOO")

  setup_test()
  expect_transpile_omits("'!indent 1")

  setup_test()
  expect_transpile_omits("'!replace FOO BAR")

  setup_test()
  expect_transpile_omits("'!define FOO")

  setup_test()
  expect_transpile_omits("'!spacing 1")

  setup_test()
  expect_transpile_omits("'!uncomment_if FOO")
  expect_transpile_omits("'!define FOO")
  expect_transpile_omits("'!uncomment_if FOO")

  setup_test()
  expect_transpile_omits("'!replace FOO BAR")
  expect_transpile_omits("'!unreplace FOO")

  setup_test()
  expect_transpile_omits("'!ifdef FOO")
  expect_transpile_omits("'!endif")
  expect_transpile_omits("'!define FOO")
  expect_transpile_omits("'!ifdef FOO")
  expect_transpile_omits("'!endif")
End Sub

Sub test_unknown_directive()
  expect_transpile_error("'!wombat foo", "Unknown !wombat directive")
End Sub

Sub test_endif_given_no_if()
  expect_transpile_omits("'!ifndef FOO")
  expect_transpile_omits("'!endif")
  expect_transpile_error("'!endif", "!endif directive without !if")
End Sub

Sub test_endif_given_args()
  expect_transpile_omits("'!ifndef FOO")
  expect_transpile_error("'!endif wombat", "!endif directive has too many arguments")
End Sub

Sub test_endif_given_trail_comment()
  expect_transpile_omits("'!if defined(FOO)")
  expect_transpile_omits("'!endif ' my comment")

  ' Note that only the first token on a line will be recognised as a directive.
  expect_transpile_omits("'!if defined(FOO)")
  expect_transpile_omits("'!endif '!if defined(FOO)")
End Sub

Sub test_error_directive()
  expect_transpile_error("'!error " + str.quote$("This is an error"), "This is an error")
  expect_transpile_error("'!error", "!error directive has missing " + str.quote$("message") + " argument")
  expect_transpile_error("'!error 42", "!error directive has missing " + str.quote$("message") + " argument")
End Sub

' If the result of transpiling a line is such that the line is omitted
' and that omission then causes two empty lines to run sequentially then
' we automatically omit the second empty line.
Sub test_omit_and_line_spacing()
  expect_transpile_succeeds("", 1)
  expect_transpile_succeeds("", 1)

  expect_transpile_omits("'!define foo")
  assert_int_equals(0, tr.omit_flag%)

  ' Should be omitted, because the last line was omitted AND
  ' the last non-omitted line was empty.
  expect_transpile_omits("")
  assert_int_equals(0, tr.omit_flag%)

  expect_transpile_omits("'!ifndef foo")
  assert_int_equals(1, tr.omit_flag%)

  expect_transpile_omits("bar")
  assert_int_equals(1, tr.omit_flag%)

  expect_transpile_omits("'!endif")
  assert_int_equals(0, tr.omit_flag%)

  ' Should be omitted, because the last line was omitted AND
  ' the last non-omitted line was empty.
  expect_transpile_omits("")
  assert_int_equals(0, tr.omit_flag%)
End Sub

Sub test_comments_directive()
  expect_transpile_omits("'!comments off")
  assert_int_equals(0, opt.comments)

  expect_transpile_omits("' This is a comment")
  assert_string_equals("", lx.line$)

  expect_transpile_succeeds("Dim a = 1 ' This is also a comment")
  expect_token_count(4)
  expect_token(0, TK_KEYWORD, "Dim")
  expect_token(1, TK_IDENTIFIER, "a")
  expect_token(2, TK_SYMBOL, "=")
  expect_token(3, TK_NUMBER, "1")
  assert_string_equals("Dim a = 1", lx.line$)

  expect_transpile_omits("'!comments on")
  assert_int_equals(-1, opt.comments)

  expect_transpile_succeeds("' This is a third comment")
  expect_token_count(1)
  expect_token(0, TK_COMMENT, "' This is a third comment")
End Sub

Sub test_always_defined_values()
  Local values$(4) = ("1", "true", "TRUE", "on", "ON")
  Local i%

  For i% = Bound(values$(), 0) To Bound(values$(), 1)
    expect_transpile_omits("'!ifdef " + values$(i%))
    expect_transpile_succeeds("should_not_be_omitted")
    expect_token_count(1)
    expect_token(0, TK_IDENTIFIER, "should_not_be_omitted")
  Next

  For i% = Bound(values$(), 0) To Bound(values$(), 1)
    expect_transpile_omits("'!ifndef " + values$(i%))
    expect_transpile_omits("should_be_omitted")
    expect_transpile_omits("'!endif")
  Next
End Sub

Sub test_always_undefined_values()
  Local values$(4) = ("0", "false", "FALSE", "off", "OFF")
  Local i%

  For i% = Bound(values$(), 0) To Bound(values$(), 1)
    expect_transpile_omits("'!ifndef " + values$(i%))
    expect_transpile_succeeds("should_not_be_omitted")
    expect_token_count(1)
    expect_token(0, TK_IDENTIFIER, "should_not_be_omitted")
    expect_transpile_omits("'!endif")
  Next

  For i% = Bound(values$(), 0) To Bound(values$(), 1)
    expect_transpile_omits("'!ifdef " + values$(i%))
    expect_transpile_omits("should_be_omitted")
    expect_transpile_omits("'!endif")
  Next
End Sub

Sub test_if_given_true()
  expect_transpile_omits("'!if defined(true)")
  expect_transpile_succeeds(" if_clause")
  expect_token_count(1)
  expect_token(0, TK_IDENTIFIER, "if_clause")
  expect_transpile_omits("'!endif")
  assert_true(tr.if_stack_sz(0) = 0, "IF stack is not empty")
End Sub

Sub test_if_given_false()
  expect_transpile_omits("'!if defined(false)")
  expect_transpile_omits(" if_clause")
  expect_transpile_omits("'!endif")
  assert_true(tr.if_stack_sz(0) = 0, "IF stack is not empty")
End Sub

Sub test_if_given_nested()
  expect_transpile_omits("'!if false")
  expect_transpile_omits("  '!if true")
  expect_transpile_omits("  '!endif")
  expect_transpile_omits("'!endif")
End Sub

Sub test_else_given_if_active()
  expect_transpile_omits("'!if defined(true)")
  expect_transpile_succeeds(" if_clause")
  expect_token_count(1)
  expect_token(0, TK_IDENTIFIER, "if_clause")
  expect_transpile_omits("'!else")
  expect_transpile_omits("    else_clause")
  expect_transpile_omits("'!endif")
  assert_true(tr.if_stack_sz(0) = 0, "IF stack is not empty")
End Sub

Sub test_else_given_else_active()
  expect_transpile_omits("'!if defined(false)")
  expect_transpile_omits("    if_clause")
  expect_transpile_omits("'!else")
  expect_transpile_succeeds(" else_clause")
  expect_token_count(1)
  expect_token(0, TK_IDENTIFIER, "else_clause")
  expect_transpile_omits("'!endif")
  assert_true(tr.if_stack_sz(0) = 0, "IF stack is not empty")
End Sub

Sub test_else_given_no_if()
  expect_transpile_error("'!else", "!else directive without !if")
End Sub

Sub test_too_many_elses()
  expect_transpile_omits("'!if defined(true)")
  expect_transpile_omits("'!else")
  expect_transpile_error("'!else", "Too many !else directives")

  setup_test()
  expect_transpile_omits("'!if defined(false)")
  expect_transpile_omits("'!else")
  expect_transpile_error("'!else", "Too many !else directives")
End Sub

Sub test_elif_given_if_active()
  expect_transpile_omits("'!if defined(true)")
  expect_transpile_succeeds(" if_clause")
  expect_token_count(1)
  expect_token(0, TK_IDENTIFIER, "if_clause")
  expect_transpile_omits("'!elif defined(true)")
  expect_transpile_omits("    elif_clause_1")
  expect_transpile_omits("'!elif defined(true)")
  expect_transpile_omits("    elif_clause_2")
  expect_transpile_omits("'!else")
  expect_transpile_omits("    else_clause")
  expect_transpile_omits("'!endif")
  assert_true(tr.if_stack_sz(0) = 0, "IF stack is not empty")
End Sub

Sub test_elif_given_elif_1_active()
  expect_transpile_omits("'!if defined(false)")
  expect_transpile_omits("    if_clause")
  expect_transpile_omits("'!elif defined(true)")
  expect_transpile_succeeds(" elif_clause_1")
  expect_token_count(1)
  expect_token(0, TK_IDENTIFIER, "elif_clause_1")
  expect_transpile_omits("'!elif defined(true)")
  expect_transpile_omits("    elif_clause_2")
  expect_transpile_omits("'!else")
  expect_transpile_omits("    else_clause")
  expect_transpile_omits("'!endif")
  assert_true(tr.if_stack_sz(0) = 0, "IF stack is not empty")
End Sub

Sub test_elif_given_elif_2_active()
  expect_transpile_omits("'!if defined(false)")
  expect_transpile_omits("    if_clause")
  expect_transpile_omits("'!elif defined(false)")
  expect_transpile_omits("    elif_clause_1")
  expect_transpile_omits("'!elif defined(true)")
  expect_transpile_succeeds(" elif_clause_2")
  expect_token_count(1)
  expect_token(0, TK_IDENTIFIER, "elif_clause_2")
  expect_transpile_omits("'!else")
  expect_transpile_omits("    else_clause")
  expect_transpile_omits("'!endif")
  assert_true(tr.if_stack_sz(0) = 0, "IF stack is not empty")
End Sub

Sub test_elif_given_else_active()
  expect_transpile_omits("'!if defined(false)")
  expect_transpile_omits("    if_clause")
  expect_transpile_omits("'!elif defined(false)")
  expect_transpile_omits("    elif_clause_1")
  expect_transpile_omits("'!elif defined(false)")
  expect_transpile_omits("    elif_clause_2")
  expect_transpile_omits("'!else")
  expect_transpile_succeeds(" else_clause")
  expect_token_count(1)
  expect_token(0, TK_IDENTIFIER, "else_clause")
  expect_transpile_omits("'!endif")
  assert_true(tr.if_stack_sz(0) = 0, "IF stack is not empty")
End Sub

Sub test_elif_given_no_expression()
  expect_transpile_omits("'!if defined(false)")
  expect_transpile_omits("    if_clause")
  expect_transpile_error("'!elif", "!elif directive expects at least 1 argument")
End Sub

Sub test_elif_given_invalid_expr()
  expect_transpile_omits("'!if defined(false)")
  expect_transpile_omits("    if_clause")
  expect_transpile_error("'!elif a +", "Invalid expression syntax")
End Sub

Sub test_elif_given_no_if()
  expect_transpile_error("'!elif defined(true)", "!elif directive without !if")
End Sub

Sub test_elif_given_comment_if()
  expect_transpile_omits("'!comment_if true")
  expect_transpile_error("'!elif defined(true)", "!elif directive without !if")

  setup_test()
  expect_transpile_omits("'!comment_if false")
  expect_transpile_error("'!elif defined(true)", "!elif directive without !if")
End Sub

Sub test_elif_given_uncomment_if()
  expect_transpile_omits("'!uncomment_if true")
  expect_transpile_error("'!elif defined(true)", "!elif directive without !if")

  setup_test()
  expect_transpile_omits("'!uncomment_if false")
  expect_transpile_error("'!elif defined(true)", "!elif directive without !if")
End Sub

Sub test_elif_given_ifdef()
  expect_transpile_omits("'!ifdef true")
  expect_transpile_succeeds(" if_clause")
  expect_token_count(1)
  expect_token(0, TK_IDENTIFIER, "if_clause")
  expect_transpile_omits("'!elif defined(true)")
  expect_transpile_omits("    elif_clause_1")
  expect_transpile_omits("'!endif")
  assert_true(tr.if_stack_sz(0) = 0, "IF stack is not empty")

  setup_test()
  expect_transpile_omits("'!ifdef false")
  expect_transpile_omits("    if_clause")
  expect_transpile_omits("'!elif defined(true)")
  expect_transpile_succeeds(" elif_clause_1")
  expect_token_count(1)
  expect_token(0, TK_IDENTIFIER, "elif_clause_1")
  expect_transpile_omits("'!endif")
  assert_true(tr.if_stack_sz(0) = 0, "IF stack is not empty")
End Sub

' Test given shortcut expression:
'   !ELIF foo
' rather than full:4
'   !ELIF DEFINED(foo)
Sub test_elif_given_shortcut_expr()
  expect_transpile_omits("'!if false")
  expect_transpile_omits(" if_clause")
  expect_transpile_omits("'!elif true")
  expect_transpile_succeeds(" elif_clause")
  expect_token_count(1)
  expect_token(0, TK_IDENTIFIER, "elif_clause")
  expect_transpile_omits("'!endif")
  assert_true(tr.if_stack_sz(0) = 0, "IF stack is not empty")
End Sub

Sub test_info_defined()
  expect_transpile_omits("'!info defined foo")
  expect_transpile_omits("'!define foo")
  expect_transpile_succeeds("'!info defined foo")
  expect_token_count(1)
  expect_token(0, TK_COMMENT, "' Preprocessor value FOO defined")
  expect_transpile_omits("'!undef foo")
  expect_transpile_omits("'!info defined foo")
  expect_transpile_error("'!info", "!info directive expects two arguments")
  expect_transpile_error("'!info defined", "!info directive expects two arguments")
  expect_transpile_error("'!info foo bar", "!info directive has invalid first argument: foo")
End Sub

Sub expect_replacement(i%, from$, to_$)
  assert_true(from$ = tr.replacements$(i%, 0), "Assert failed, expected from$ = '" + from$ + "', but was '" + tr.replacements$(i%, 0) + "'")
  assert_true(to_$  = tr.replacements$(i%, 1), "Assert failed, expected to_$ = '"   + to_$  + "', but was '" + tr.replacements$(i%, 1) + "'")
End Sub

Sub expect_token_count(num)
  assert_no_error()
  assert_true(lx.num = num, "expected " + Str$(num) + " tokens, found " + Str$(lx.num))
End Sub

Sub expect_token(i, type, s$)
  assert_true(lx.type(i) = type, "expected type " + Str$(type) + ", found " + Str$(lx.type(i)))
  assert_string_equals(s$, lx.token$(i))
End Sub

Sub expect_transpile_omits(line$)
  If lx.parse_basic%(line$) <> sys.SUCCESS Then
    assert_fail("Parse failed: " + line$)
  Else
    Local result% = tr.transpile%()
    If result% = tr.OMIT_LINE Then
      If lx.num <> 0 Then
        assert_fail("Omitted line contains " + Str$(lx.num) + " tokens: " + line$)
      EndIf
    Else
      assert_fail("Transpiler failed to omit line, result = " + Str$(result%) + " : " + line$)
    EndIf
  EndIf
  assert_no_error()
End Sub

Sub expect_transpile_succeeds(line$, allow_zero_tokens%)
  If lx.parse_basic%(line$) <> sys.SUCCESS Then
    assert_fail("Parse failed: " + line$)
  Else
    Local result% = tr.transpile%()
    If result% = sys.SUCCESS Then
      If Not allow_zero_tokens% And lx.num < 1 Then
        assert_fail("Transpiled line contains zero tokens: " + line$)
      EndIf
    Else
      assert_fail("Transpiler did not return SUCCESS, result = " + Str$(result%) + " : " + line$)
    EndIf
  EndIf
  assert_no_error()
End Sub

Sub expect_transpile_error(line$, msg$)
  If lx.parse_basic%(line$) <> sys.SUCCESS Then
    assert_fail("Parse failed: " + line$)
  Else
    Local result% = tr.transpile%()
    If result% = sys.FAILURE Then
      assert_error(msg$)
    Else
      assert_fail("Transpiler did not return ERROR, result = " + Str$(result%) + " : " + line$)
    EndIf
  EndIf
End Sub
