  '******************************************************
  ' MM DISCHARGER
  ' Universal discharger for rechargeable batteries v046
  ' (1 or 7 cells NiCd, NiMH or 1 cell Li-Ion 18650)
	'
	' Easily to convert into a 4 day data logger
	' (because of its self-scaling capability 1 min - 4 days)
  '
  ' System: MMBasic 4.7 (b32)
  ' Micromite 2 (PIC32MX170/28, Micks BP170 + 2.4" TFT)
  ' by twofingers 11-2015 at TBS
  '
  ' -----------------------------------------------------
  ' Uses a ILI9341 TFT and/or serial monitor (vt100).
  ' Automatic discharger resistor selection, self-scaling.
  '------------------------------------------------------
  ' Purpose: 
  ' The measurement of the capacity of rechargeable batteries.
  ' This program discharges one rechargeable battery 
  ' (NiMH, LiIon) to measure its capacitance. Results (mAh)
  ' and the discharge curve are shown on the TFT.
  '
  ' You can also use a log file with a serial monitor (eg TT)
  ' and import it in Excel.
  ' 
  ' !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
  ' Note/Warning:
  ' Frequent discharge can/will
  ' shorten rechargeable batteries life!
  ' Any cell has a limited lifecycle.
  ' !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!! 
  ' For comparison: 
  ' A noname alkaline battery has 505mAh.
  '
  '******************************************************
  '
  ' Used PINs (MX170/28):
  ' 23 as Input
  ' 16 as Output
  ' 17 as Output
  ' 18 as Output
  '
  '*******************************************************
  ' This code may be freely distributed and changed.
  ' Provided AS IS without any warranty.
  ' Use it at your own risk. All information is provided for
  ' educational purposes only!
  ' ------------------------------------------------------
  '  ++++++ Credit to Geoff for his great MMBasic ++++++
  '*******************************************************
  '
  ' This is the schematic example for a simple 9V discharger:
  '
  '    A IN
  '   o-----------+         o---------------o----------+
  '               |         |               |          |
  '               |        .-. Disch.R     .-.         |
  '               |        | | 470R     R3 | |         |
  '  <- MM2       |        | |  R2     820k| |         |
  '               |        '-'             '-'         |
  '               |         |               |          |
  '               |         |               |          | Battery
  '               +---------))--------------+----+     |
  '   D OUT                 |               |    |    ---
  '   o--------o------+    D|               |    |     -
  '            |      |  ||-| N-MOSFET      |    |     |
  '            |      |  ||<) IRLML2502     |   ---    |
  '           .-.     +--||-| T1           .-.  ---    |
  '       R1  | |     G    S|          R4  | |   | C1  |
  '      100k | |           |         100k | |   | 100n|
  '           '-'           |              '-'   |     |
  '            |            |               |    |     |
  '   o--------o------------o---------------o----o-----+
  '   GnD
  '
  '
  ' Recomended values (R2):
  ' 1.2V NiMH  1500-2500mA: C/10 = 200mA = 4,7R/Threshold: 1V
  ' 3.6V LiIon 2000-3000mA: C/10 = 200mA = 18R /Threshold: 3V
  ' 9V   NiMH    150-250mA: C/10 =  20mA = 470R/Threshold: 7V
  '--------------------------------------------
  '
  ' (3x) R1=100K, 1/8W
  ' (3x) R2a=470R, 1/4W (b=18R, 1W,  c=4.7R, 1/2W)
  ' R3=820k, 1/8W
  ' R4=100k, 1/8W
  '
  ' C1=~100n
  ' (3x) T1=N-MosFet, logic level (IRL540, IRLML2502, ...)
  '      voltage drop on T1 = 1.3mV
	'
  ' see ->>MMDIS.JPG for the real thing
  '------------------------------------------
  ' Command line settings for TFT Display
  ' OPTION LCDPANEL ILI9341, L, 4, 5, 6
  ' OPTION TOUCH 7, 2
  ' GUI TEST TOUCH
  ' GUI CALIBRATE
  '-----------------------------------------
  
  Do ' let the show begin
    
    Option AUTORUN ON ' since we use a watchdog
    Option explicit
    
    Const PRG_VERS=.46
    
    ' user configurable const ---------------------
    ' PIC32MX170 28 Pin wiring eg
    Const Input_PIN =23   ' measures the batt. voltage on voltage divider
    ' for OUTPUT_PINs see setOutPin(3)
    
    Const maxHOURS=72
    Const maxDischargeTime=maxHOURS*3600 ' in sec
    
    Const Voltage_Divider=0.1077 ' 820k/100k eg 100k/(820k+100k), theor. = 0,1087
    ' please check this value with a DMM
    ' ie low drain on battery when Not charging
    
    ' pxPerHour should be a power of 2, range 4096(~1min) - 4(~72h)
    Dim integer pxPerHour=4096*4'<- modify this for your battery capacity
    ' starting parameter for voltage curve on TFT
    
    Dim float setDResistor(3)=(0,4.7,18 , 470)'R discharger, check your stuff plz
    Dim float setThresh(3)   =(0,1.0,3.0, 7.0)'V cutoff voltage, threshold
    Dim float setBHighLim(3) =(0,1.7,4.5,11.0)'V upper limits for batt type
    Dim integer setOutPin(3) =(0, 16, 17,  18)'pin numbers, user configurable
    Dim float setVscMin(3)   =(0,0.9,2  ,   6)'lower scale limit
    Dim float setscVolt(3)   =(0,20 ,4  ,   2)'2V/x scale divisions
    Dim string setBatName$(3)=("","1,2V NiMH","3.6V Li-Ion","9V NiMH")
    
    ' program const ---------------------
    Const HIGH =1
    Const LOW  =0
    
    Const TRUE =1
    Const FALSE=0
    
    Const ENTER=13 ' enter key
    
    Const RBORDER=25 ' space for voltage scale
    Const BOTTOM=MM.VRes-25 '215, space for time scale
    Const USCREEN=MM.HRes-RBORDER '295, hor. usable diagram screen space
    Const TOP=65  ' vert. usable diagram screen space
    Const VOFFSET=30 ' voltage scale multiplier/steps
    
    
    ' variable initialisation
    Dim integer UserStop=FALSE ' keyboard control
    Dim integer plot_Pos=0   ' graphic pixel output
    Dim integer ticker=0,ex_ticker=0
    Dim integer i            ' general counter
    Dim integer min_counter=0,min_counter_last=0
    Dim integer B_lim, vSampl=30 ' number of samples
    Dim integer BattType=0 '0, 1.2, 3.6, 9V
    Dim integer refresh=FALSE
    Dim integer touchDetect=FALSE
    
    Dim float ti        ' timer in sec
    Dim float v=0,vx,V_sec ' voltage, voltage to sum
    Dim float mAh=0     ' capacity
    Dim float C_log(maxDischargeTime/60) ' one buffer for the min
    Dim float C_Seclog(4000) ' one buffer (1h+) for the sec
    Dim float mp_ratio
    
    Dim string ttime$, v_old$, key$
    Dim string Vstr$, Astr$
    
    '--------------------------------------------
    
    SetPin Input_PIN ,AIN ' The transducer will seduce you
    
    If MM.Watchdog Then do_WD_Handler ' oops, we had a restart !
    
    SetupDisplay
    
    Do   ' look out for batteries
      v=0: For i= 1 To vSampl:v=v+Pin(Input_PIN):Pause 15:Next i
      v=v/vSampl/Voltage_Divider
      
      text_center "Volt:"+Str$(v,2,2),80,2
     
      If v <0.1 Then
        If BattType<>0 Then Line 0, 105, MM.HRes, 105,100,RGB(white)
        BattType=0
        text_center "  insert a Battery  ",200,2
        Print "       please       ";Chr$(13);
        Pause 333
        Print "  insert a Battery  ";Chr$(13);
        Pause 333
      ElseIf v <setBHighLim(1) And v>setThresh(1) Then ' 1.2V NiMH
        If BattType<>1 Then Line 0, 105, MM.HRes, 105,100,RGB(white)
        BattType=1
        BDisplay(BattType) 'gets the touch
      ElseIf v <setBHighLim(2) And v>setThresh(2) Then ' 3.6V LiIon
        If BattType<>2 Then Line 0, 105, MM.HRes, 105,100,RGB(white) 'cls
        BattType=2
        BDisplay(BattType) 'gets the touch
      ElseIf v >setThresh(3) Then '9V NiMH
        If BattType<>3 Then Line 0, 105, MM.HRes, 105,100,RGB(white)
        BattType=3
        BDisplay(BattType) 'gets the touch
      Else ' Gaps: 0-1V, 1.7V-3.0V, 4.5V-7.0V
        BattType=0
        If BattType<>0 Then Line 0, 105, MM.HRes, 105,100,RGB(white)
        Line 0, 105, MM.HRes, 105,100,RGB(white)
        text_center "   Battery empty    ",200,2
        Print "   Battery empty    ";Chr$(13);
        Pause 333
        Print "   Volt:"+Str$(v,2,2)+"   ";Chr$(13);
        Pause 333
      EndIf
      key$=Inkey$
    Loop While (key$<>Chr$(ENTER) Or Not BattType) And Not touchDetect
    
		' Const are actually vars
    Const THRESHOLD=setThresh(BattType)
    Const VscMin=setVscMin(BattType)
    Const scVolt=setscVolt(BattType)
    Const scTH=(Threshold-VscMin)*scVolt
    Const Output_PIN=setOutPin(BattType)
    Const DISCHARGE_R=setDResistor(BattType)
    
    Print:Print
    Print "Selected: ";setBatName$(BattType)",  Threshold:";Str$(THRESHOLD,2,2);
    Print "V, Discharge resistor:";DISCHARGE_R;"R"
    Print
    Print "Press <SPACE> to abort"
    
    MDisplay
    
    On KEY New_KB_Handler
    Option Break 0
    
    SetPin Output_PIN, DOUT
    Pin(Output_PIN)=HIGH 'open the gate
    
    C_log(0)=v: vx=v
    v_old$="  V0: "+Str$(v,2,2)
    
    SetTick 1000,Tick_handler
    Timer=0
    
    Do '*********** discharging *******************************
      WatchDog 6666 ' since we have to turn off the MOSFET in any event (>1001ms)
      
      IDisplay 'InfoText
      
      'sec synchonizing,  wait until 1 sec has passed ##
      Do:Loop While ticker<=ex_ticker And Touch(X)=-1
      ex_ticker=ex_ticker+1 'And that's all the time that it takes
      
      V=0: For i= 1 To vSampl:v=v+Pin(Input_PIN):Pause 15:Next i
      V=V/vSampl/Voltage_Divider
      
      mAh=mAh+v/Discharge_R/3.600 'mAh = V/R/3600sec*1000
      
      ti=Int(Timer/1000) 'timer sec
      plot_pos=ti*pxPerHour/3600
      V_sec=V_sec+v
      min_counter=Int(ti/60)
      
      B_lim=USCREEN-(USCREEN Mod 2)
      If pxPerHour>B_lim Then C_Seclog(ti)=V
      If min_counter>min_counter_last Then
        min_counter_last=min_counter
        C_log(min_counter)=V_sec/60 ' calculate the average voltage per min
        V_sec=0
      EndIf
      
      Plot(plot_pos,v)
      
      Print ti,">";Str$(v,1,4),Str$(1000*(v-vx),2,2),Str$(v/Discharge_R/3.600)
      vx=v
      
      '-----------------------------
      If plot_pos>=B_lim Then ' shrink the curve horizontally
        pxPerHour=pxPerHour/2
        
        MDisplay
        
        If pxPerHour>B_lim Then
          mp_ratio=ti/(B_lim/2)
          For i=0 To B_lim/2
            Plot(i,C_Seclog(i*mp_ratio))
          Next i
        Else
          mp_ratio=min_counter/(B_lim/2)
          For i=0 To B_lim/2
            Plot(i,C_log(i*mp_ratio))
          Next i
        EndIf
      EndIf
      '-----------------------------
      If Touch(X)<>-1 Then userStop=TRUE
      
    Loop While (v>THRESHOLD) And (Timer/1000<maxDischargeTime) And (Not userStop)
    ' end discharging *****************************************************
    
    Pin(Output_PIN)=LOW 'Close the gate
    On KEY 0
    Option Break 3
    WatchDog OFF
    SetTick 0,0
    Pause 500
        
    Print
    If userStop Then
      Print "BREAK! " Str$(v,2,2)+"V"+ v_old$+"V ";
    Else
      Print "Ready! " Str$(v,2,2)+"V"+ v_old$+"V ";
    EndIf
    Print Str$(mAh,2,2)+"mAh  "+Sec2Hms$(Ti)
    
    IDisplay 'InfoText
    
    If userStop Then
      Text 0,30, "User break! "+Sec2Hms$(Ti), , 2,2,RGB(YELLOW)
    Else
      Text 0,30, "Ready!      "+Sec2Hms$(Ti), , 2,2,RGB(GREEN)
    EndIf
    
    Print:Print "Press <ENTER> to restart"
    Do:Loop While Inkey$<>Chr$(ENTER) And Touch(X)=-1
    Clear ' forget anything you have learned
  Loop ' let's do the time warp again
End    ' madness takes it's toll ... 
'****************************************************************
  
  
Sub Plot(x,voltage) ' x=plot position
  Pixel x,(BOTTOM-((voltage-VscMin)*scVolt)*15),RGB(Yellow)
End Sub
  
  
Sub Tick_handler ' inc ticker once a second
  ticker=ticker+1
End Sub
  
  
Sub SetupDisplay ' Titel
  Colour RGB(BLACK), RGB(WHITE)
  CLS
  
  text_center "MM Discharger "+Str$(PRG_VERS,1,2),10
  Line 0,40,MM.HRes,40,5,RGB(red)
End Sub
  
  
Sub BDisplay(Btype)' Infos on titel
  text_center setBatName$(BattType),120,3
  text_center "found",160,2
  text_center "Press screen",200,2
  Print "       "+setBatName$(BType)+"    ";
  Print " open your log file and ";
  Print "press ENTER to start";Chr$(13);
  T_Pause(300)
  Print "   Volt:"+Str$(v,2,2)+"   ";Chr$(13);
  text_center "to start",200,2
  T_Pause(300)
End Sub
  
  
Sub IDisplay 'InfoText on MDisplay
  Vstr$="V:"+Str$(v,2,2)
  Astr$=Str$(mAh,2,2)+"mAh"
  Text 0,0, Vstr$+Space$(20-Len(Vstr$+Astr$))+Astr$, , 2,2,RGB(WHITE)
  ttime$="Time: "+Sec2Hms$(Timer/1000)
  text_center(v_old$+"  R:"+Str$(Discharge_R,2)+"  "+ttime$,30,1)
End Sub
  
  
Sub MDisplay '###
  Local float vLinesPerScreen  'vLinesPerScreen= lines per screen
  Local float sv
  Local integer i
  Local integer pxPerVolt=30
  Local integer tLine=BOTTOM-(scTH)*15
  
  Local integer scale=(32/pxPerHour)+(pxPerHour>64),pxPerMin
  Local integer mm(4)=(0,60,12,6,2),c=1
  
  
  'Const TOP=65
  'Const VOFFSET=30
  'Const BOTTOM=215
  'Const USCREEN=MM.HRes-25
  
  Colour RGB(YELLOW), RGB(BLACK)
  If refresh Then
    Box 0, 50, MM.HRes-25, MM.VRes-50,0,,RGB(BLACK)
  Else
    CLS
    refresh=TRUE
  EndIf
  
  For i = BOTTOM To TOP Step -pxPerVolt ' hor lines, Volt
    Line 0,i,USCREEN,i,,RGB(RED)
    sv=((i-TOP)/(pxPerVolt/2))/scVolt+VscMin
    Text USCREEN+0,MM.VRes-i+VOFFSET, Str$(sv), ,2,1,RGB(YELLOW)
  Next i
  
  Line 0,BOTTOM,USCREEN,BOTTOM,,RGB(WHITE)
  
  For i = 1 To USCREEN Step 10 ' Threshold
    Line i,tLine,i+3,tLine,,RGB(GREEN)
  Next i
  
  Do While c<=4 And pxPerHour>100' minute scale 1,5,10,30
    pxPerMin=pxPerHour/mm(c)
    vLinesPerScreen=USCREEN/pxPerMin
    
    If (vLinesPerScreen>1 And vLinesPerScreen<10) Then
      For i = pxPerMin To USCREEN Step pxPerMin ' vert lines, time
        Line i,MM.VRes-20+5,i,100-45,,RGB(cyan)
        Text i-8,220, Str$(i/pxPerHour*60,2,0)+"m", ,2,1,RGB(WHITE) 'h scale
      Next i
    EndIf
    c=c+1
  Loop
  
  For i = 0 To USCREEN Step pxPerHour*scale ' vert lines, time
    Line i,MM.VRes-20+5,i,100-45,,RGB(BLUE)
    Text i-8,220, Str$(i/pxPerHour,2,0)+"h ", ,2,1,RGB(WHITE) 'h scale
  Next i
  
End Sub
  
  
Sub text_center tstring$,h_pos,size
  Local tsize=size
  Local dspace, nfw=8 ' normal font width
  Local t$=tstring$
  
  If tsize=0 Then tsize = 2
  dspace=Int((MM.HRes/nfw/tsize-Len(t$))/2)
  t$=Space$(dspace)+t$+Space$(dspace-dspace/2) ' delete left & right
  Text 0, h_pos, t$, , 2, tsize
End Sub
  
  
Sub New_KB_Handler ' my CTRL-C handler
  Local key_value
  key_value=Asc(Inkey$)'
  
  If key_value=32 Or key_value=3 Then ' space bar or CTRL-C  means stop
    UserStop= TRUE
    Pause 500
  EndIf
  
  If key_value=Asc("#") Then 'debug
    pxPerHour=pxPerHour/2
    mDISPLAY
  EndIf
  
  If key_value=Asc("d") Then ' debug_list
    debug_list
  EndIf
End Sub
  
  
Sub debug_list
  For i=0 To min_counter:Print i,C_log(i):Next i
  Print "ti ";ti;"  ti/60 ";ti/60
End Sub
  
  
Function Sec2Hms$(sec) ' convert sec to "00:00:00" string
  ' limit = 4,17 days (359999 sec="99:99:99")
  Local integer Hour,Mins,Secs
	local integer hms.days ' make this global if you want it
  
  hms.days= Int(sec/86400)  ' just if you need to know
  Hour= Int(sec/3600)
  Mins= Int((sec Mod 3600)/60)
  Secs= Int((sec Mod 3600) Mod 60)
  
  Sec2Hms$=Str$(Hour,2,0,"0")+":"+Str$(Mins,2,0,"0")+":"+Str$(Secs,2,0,"0")
End Function
  
  
Sub do_WD_Handler 'We return to Transylvania
  WatchDog OFF
  For i = 1 To 3
    Pin(setOutPin(i))=LOW 'close the gates
  Next i
  On KEY 0
  Option Break 3
  
  text_center("Watchdog restart",80,2)
  text_center(Time$+" "+Date$,120,2)
  text_center "Program terminated",200,2
  Print "Program terminated"    ' please examinate your code
End 'So baby don't cry like there's no tomorrow ...
End Sub
  
  
Sub T_Pause(ms)'Touch-a touch-a touch me
  i=0
  Do
    Pause ms/30
    i=i+1
    touchDetect=Touch(X)<>-1 ' touch me, creature of the night!
  Loop While i<ms/10 And Not touchDetect
End Sub