'   ___  ___              _  _            _        _     _____ 
'   |  \/  |             (_)| |          (_)      | |   / __  \
'   | .  . |  __ _ __  __ _ | |__   _ __  _   ___ | | __`' / /'
'   | |\/| | / _` |\ \/ /| || '_ \ | '__|| | / __|| |/ /  / /  
'   | |  | || (_| | >  < | || |_) || |   | || (__ |   < ./ /___
'   \_|  |_/ \__,_|/_/\_\|_||_.__/ |_|   |_| \___||_|\_\\_____/
'
' A Breakout clone for the Colour Maximite 2 computer
'
' Written by vegipete, September 2020
'
' Playable with keyboard, paddle or Nunchuk
'
' version history:
'   v .71   original release for the Colour Maximite
'   v .82   update for the Colour Maximite 2
'             - language changes to suit new/changed MMBasic
'             - layout ajusted to suit higher resolution
'             - paddle connections changed
'             - Nunchuk support added
'             - increased speed at higher levels
'             - volume control added
'             - all support files moved into main program
'
' Notes: 
'   Beyond level 160ish, play can get stuck as the ball passes through targets.
'   Since I doubt any human can play that fast, this doesn't really matter.
'
'   Code is only marginally commented - sorry.
'
'   If the volume is set to minimum, finishing a level appears to hang as the
'   tune plays silently. Have patience. Or modify the PlayTune subroutine to
'   return faster when the volume is lowest. ( volevel = 1)

Cls

' table of note frequencies for 8th octave
' 0    = rest
' 1-7  = ABCDEFG
' 8-12 = ACDFG#
dim notes(12) = (0,3520,3951,4186,4699,5274,5588,6272,3729,4435,4978,5920,6644)

' brick colours     blue      green     cyan       red     purple     yellow    white
dim CLR(7) = (0,&h0000FF,&h00FF00,&h00FFFF,&hFF0000,&hFF00FF,&hFFFF00,&hFFFFFF)

const brickH = 56
const brickV = 18
const blsty = 33  ' ball starting position above bottom of screen

' create ball sprites
page write 1
circle 8,8,8,1,1,rgb( blue),rgb( blue)
circle 7,7,5,1,1,rgb( cyan),rgb( cyan)
circle 6,6,2,1,1,rgb(white),rgb(white)
sprite read 1,0,0,17,17,1
sprite read 2,0,0,17,17,1
sprite read 3,0,0,17,17,1
sprite read 4,0,0,17,17,1
sprite read 5,0,0,17,17,1
sprite read 6,0,0,17,17,1
sprite read 7,0,0,17,17,1
page write 0

'set up hardware
pinpot = 26 'pin 26 is analog paddle input
pinbut = 28 'pin 28 is digital input, pin(pinbut) = 0 when pressed, = 1 when released
setpin pinpot,AIN,8   ' analog y2
setpin pinbut,DIN     ' button D, pulled high externally

' test for nunchuk
gotnun = 1
on error skip
wii nunchuk open 3
if MM.ERRNO = 16 then gotnun = 0

' set volume at startup
volevel = 5
SetVolume(volevel)

'initialize variables
level = 1
score = 0
brickcount = 0
balleft = 0
Dim bricks(14,7)
delta = 0
kboard = 0
demomode = 1
circdiag = 0    ' don't show circuit diagram

'draw screen
box 0,28,MM.HRES,MM.VRES,1,rgb(magenta),rgb(magenta)
box 5,58,MM.HRES-10,MM.VRES,1,0,0

text  10,15,"MaxiBrick2","LM",5,1,rgb(red),-1
text  11,15,"MaxiBrick2","LM",5,1,rgb(red),-1
text 300,14,"v0.82","LM",4,1,rgb(Green)
text 395,14,"by Vegipete","LM",4,1,rgb(Yellow)
text 550,14,"picprog.strongedge.net","LM",4,1,rgb(Cyan)

text  10,43,"Bricks Left:","LM",4,1,0,-1
text 275,43,"Balls:","LM",4,1,0,-1
text 440,43,"Level:","LM",4,1,0,-1
text 640,43,"Score:","LM",4,1,0,-1

StartDemo

UpdateScoreBar

' get starting position for bat and ball
xposn = ReadPaddle()
if xposn <  37 then xposn =  37
if xposn > 763 then xposn = 763
yposn = MM.VRes - 12
text xposn,yposn,"1","CM",10,1 'draw the bat at the bottom
xball = xposn
xinc = .3
yball = MM.VRes - blsty
yinc = -1
Sprite show 1,xball,yball,1
DrawBricks

usenun = 0
usepad = 0
usekey = 0

' main loop, for both playing and demo modes
Do
  Timer = 0   ' not tested if needed for CMM2 version

  'Update ball position
  If pausmode = 0 Then
    if graball = 0 then
      xball = xball + xinc * (level + 9) / 10.1
      if xball > MM.HRES-1 then xball = MM.HRES - 1
      if xball < 0 then xball = 0
      yball = yball + yinc * (level + 7) /  8.3
      if yball > MM.VRES-10 then yball = MM.VRES - 10
      if yball < 40 then yball = 40
    else
      xball = xposn - 8    ' lock ball to paddle
      yball = MM.VRes - blsty
      if usepad then
        if pin(pinbut) = 0 then graball = 0 : xinc = 0
      endif
      if usenun then
        if nunchuk(Z, 3) = 1 then graball = 0 : xinc = 0
      endif
    endif
    sprite show 1,xball,yball,1
  EndIf

  'Update bat position
  If demomode Then
    xposnew = xball + delta + 8 ' lock bat to ball x-coord, with slight random offset
  Else
    If usekey Then
      ky = KeyDown(1)
      If ky = 104 Then xposnew = xposnew - 4
      If ky = 106 Then xposnew = xposnew - 2
      If ky = 107 Then xposnew = xposnew - 0
      If ky = 108 Then xposnew = xposnew + 2
      If ky =  59 Then xposnew = xposnew + 4
      if ky =  32 and graball then graball = 0
    Elseif usepad then
      xposnew = ReadPaddle()
    elseif usenun then
      ' oOher nunchuks may need other calibration values.
      ' Reading calibration from the nunchuck might make sense.
      xposnew = (nunchuk(ax,3) - 300) * 1.9
    endif
  EndIf
  If xposnew <  37 Then xposnew =  37
  If xposnew > 763 Then xposnew = 763

  If Abs(xposnew - xposn) > 2 Then
    text xposn,yposn,"0","CM",10,1 'erase the old bat
    text xposnew,yposn,"1","CM",10,1 'draw new bat
    xposn = xposnew
  EndIf

  'Test ball at edges of playing area
  If (xball < 5) Then xinc = Abs(xinc) : play Tone 800,1,40 : 'Left?
  If (xball > MM.HRes - 20) Then xinc = - Abs(xinc) : play Tone 1,800,40
  If (yball < 58) Then yinc = Abs(yinc) : play Tone 900,900,40 : 'Top?

  'Test ball hitting (or missing) bat
  If (yball > MM.VRes - blsty) Then        'Ball at bottom of screen?
    If ((xball > xposn - 45) And (xball < xposn + 29)) or demomode Then  'Ball hitting bat?
      if graball = 0 then play Tone 700,700,40
      yinc = -Abs(yinc)
      yball = yball - rnd * level / 50  ' perturb the ball position slightly
      spin = xball - xposn + 8
      xinc = xinc + spin / 20
      If xinc > 1.6 Then xinc = 1.6
      If xinc < -1.6 Then xinc = -1.6
      delta = 8 - Rnd * 16
    Else    'missed the ball
      balleft = balleft - 1
      If balleft = 0 Then
        Sprite hide balleft + 2
        EndGame   'game over >:-(
        Do While Inkey$ <> ""
        Loop  'clear input buffer
      Else
        play Tone 440,440,300 : Pause 350
        play Tone 392,392,300 : Pause 350
        Sprite hide balleft + 2
        play Tone 330,330,1000 : Pause 1050
        yinc = -1
        xinc = 0
        xball = xposn - 8
        yball = MM.VRes - blsty
        graball = 1
      EndIf
    EndIf
  EndIf

  'Test for ball hitting bricks
  i = Int((xball - 4) / brickH) + 1
  if i <  1 then i =  1
  if i > 14 then i = 14
  k = Int((yball - 105) / brickV)
  If k < 8 And k > 0 Then
    If bricks(i, k) Then
      If demomode = 0 Then score = score + 1
      bn = 2000 - 100 * k : play Tone bn,bn,30
      brickcount = brickcount - 1
      UpdateScoreBar
      yinc = -yinc
        
      Sprite hide 1
      DrawOneBrick(i, k, 0) 'Erase brick
      Sprite show 1, xball, yball, 1
      bricks(i,k) = 0
      
      If brickcount = 0 Then
        level = level + 1

        text 400,200,"ALL BRICKS SMASHED!","CM",5,1,rgb(yellow),-1
        
        PlayTune(400,"F52C52F52A62G52@02F52D52F52C62B62@02",50)
        PlayTune(400,"A62G54A64B64G52B64A64F54A62",50)
        PlayTune(400,"G52C64B64C64@04G54E54C52",50)

        text 400,200,"                   ","CM",5,1,0
        
        Sprite hide 1
        DrawBricks
        Sprite show 1, xball, yball, 1
      EndIf
    EndIf
  EndIf

  'Test for keypress and handle it
  kp = Asc(Inkey$)
  If kp <> 0 Then
    select case kp
      case 252      'CTRL-SHIFT-F12'
        SaveBMP "MB"+Left$(Right$(Time$,5),2)+"-"+Right$(Time$,2)
      case 101      ' e = toggle easy mode
        easymode = 1 - easymode
      case 27       ' esc - toggle pause
        pausmode = 1 - pausmode
      case 80, 112  ' P, p - show paddle wiring
        circdiag = circdiag + demomode
        if circdiag > 3 then circdiag = 3   ' 1 = rising edge, 3 = falling edge
      case 81, 113  ' Q, q - quit
        end
      case 44       ' , - volume down
        volevel = volevel - (volevel > 1)
        SetVolume(volevel)
      case 46       ' . - volume up
        volevel = volevel + (volevel < 20)
        SetVolume(volevel)
      case 32       ' [space] - start/restart demo mode
        if demomode then
          sprite hide 1
          DrawBricks
          yball = 500
          Sprite show 1,xball,yball,1
        else
          Do While balleft > 0
            Sprite hide balleft + 1
            balleft = balleft - 1
          Loop
          StartDemo
          Sprite show 1, xball, yball, 1
        endif
      case 107      ' k - use keyboard
        if demomode then usekey = 1 : StartNew
    end select
  EndIf

  ' test for paddle button press to start game
  if (demomode = 1) and (pin(pinbut) = 0) then
    do : loop until pin(pinbut) ' wait for button release
    usepad = 1
    StartNew
  endif

  ' test for Nunchuk Z button press to start game
  if (demomode = 1) and (gotnun = 1) then
    if nunchuk(Z, 3) then
      do : loop until nunchuk(Z, 3) = 0 ' wait for Z button release
      usenun = 1
      StartNew
    endif
  endif

  ' test for request to show/hide wiring diagram
  if circdiag = 1 then
    sprite hide 1
    text 650,290,"0","LT",11
    Sprite show 1, xball, yball, 1
    circdiag = 2
  elseif circdiag = 3 then
    sprite hide 1
    box 650,290,128,248,1,0,0
    Sprite show 1, xball, yball, 1
    circdiag = 0
  endif
  
  'Wait for fixed time so animation is smooth
  Do
  Loop Until Timer > 4
  If easymode Then
    Do
    Loop Until Timer > 8
  EndIf

Loop
Cls
End

'******************************
function ReadPaddle()
  ReadPaddle = int(pin(pinpot) * 750 / 3.3) + 32
end function

'******************************
'Set everything up to start a new game
Sub StartNew
  Local i

  sprite hide 1

  'erase demo mode message
  box 5,58,MM.HRES-10,MM.VRES,1,0,0

  'start paddle in the middle
  text xposn,yposn,"0","CM",10,1 'erase the old bat
  xposn = MM.HRES/2
  text xposn,yposn,"1","CM",10,1 'draw new bat

  'set up starting ball position and speed
  graball = 1   ' hold the ball on the paddle to start
  xball = xposn - 8
  If xball < 5 Then xball = 5
  If xball > MM.HRes - 20 Then xball = MM.HRes - 20
  xinc = 0    'straight up
  
  yball = MM.VRES - blsty
  yinc = -1
  
  Sprite show 1,xball,yball,1

  score = 0
  level = 1
  brickcount = 0
  balleft = 3
  demomode = 0
  DrawBricks
  UpdateScoreBar

  'draw the players balls
  For i = 0 To balleft-1 : Sprite show 2+i,350+20*i,35,1 : Next

End Sub

'******************************
'Game over - so sad
Sub EndGame
  text 400,260,"GAME OVER","CT",5,1,rgb(magenta)
  Sprite hide 1

  'Play some Chopin
  PlayTune(700,"A51A51A54A51C51B54B51A54A51A54",50)
  play Tone 440,440,1000

  'Now scroll "GAME OVER" off bottom of screen
  For i = 261 To 650
    text 400,i,"GAME OVER","CT",5,1,rgb(magenta)
    line 290,i-1,350,i-1,1,0
    pause 10
  Next

  usenun = 0
  usepad = 0
  usekey = 0

  StartDemo
  Sprite show 1, xball, yball, 1
End Sub

'******************************
'Play a tune contained in a string
'
'spd is the duration of one whole note
'tune$ is a string containing the tune
'gap is the duration between notes
'
'each note consists of 3 (or 4 for sharps) characters
'Note(@-A-G, optional # for sharp)/Octave(1-8)/Duration(1=whole,2=half,etc)
'For a rest, use @0 as the note and octave
'
'Example: 'Mary Had a Little Lamb'
'PlayTune(800,"E52D52C52D52E52E52E51@08D52D52D51@08E52G52G51@08",50)
'PlayTune(800,"E52D52C52D52E52E52E52E52D52D52E52D52C51",50)
'
Sub PlayTune(spd, tune$, gap)
  Local i,fr,oc,du
  i = 1
  tune$ = UCase$(tune$)
  Do
    'read note letter and convert to 1 to 8
    fr = Asc(Mid$(tune$,i,1)) - 63 : i = i + 1
    If fr < 1 Or fr > 8 Then fr = 1
    'if sharp then shift up 7
    If Mid$(tune$,i,1) = "#" Then
      fr = fr + 8
      i = i + 1
    EndIf
    ' 2-8 are base notes, 10-16 are sharps (B# and E# don't actually exist)
    ' 1 & 9 are rests
    
    if fr < 0 or fr > 12 then fr = 0
    fr = notes(fr)

    'read requested octave
    oc = 8 - Val(Mid$(tune$,i,1)) : i = i + 1
    Do While oc > 0
      fr = fr / 2
      oc = oc - 1
    Loop
    'read note fraction, eg: 1 = full, 4 = quarter
    du = Val(Mid$(tune$,i,1)) : i = i + 1
    If fr < 1 Then fr = 1
    play Tone fr,fr,spd/du
    Pause gap + spd/du
  Loop Until i > Len(tune$)
End Sub

sub SetVolume(sv)
  vv = exp(sv / 4.34) - 1
  play volume vv, vv
end sub

'******************************
'Start demo mode
Sub StartDemo
  demomode = 1    'demo mode will continue from where the game ended
  easymode = 0
  graball = 0
  text 400,300,"Demo Mode","CM",5,1,rgb(green)
  text 401,300,"Demo Mode","CM",5,1,rgb(green),-1
  text 240,330,"Press [k] to play using keyboard,","LM",4,1,rgb(green)
  text 250,350,"[h-j-k-l-;] to move, [space] to serve","LM",4,1,rgb(green)
  text 240,370,"Press paddle button to use paddle","LM",4,1,rgb(green)
  text 240,390,"Press Nunchuk Z to use Nunchuk","LM",4,1,rgb(green)
  text 240,420,"Other keys:","LM",4,1,rgb(green)
  text 250,440,"[Esc] to pause","LM",4,1,rgb(green)
  text 250,460,"[e] to toggle Easy mode during play","LM",4,1,rgb(green)
  text 250,480,"[SPACE BAR] to return to Demo mode","LM",4,1,rgb(green)
  text 250,500,"[,] [.] to adjust volume","LM",4,1,rgb(green)
  text 250,520,"[p] to view paddle wiring diagram","LM",4,1,rgb(green)
  text 250,540,"[q] to quit","LM",4,1,rgb(green)
End Sub

'******************************
'Update all onscreen values
Sub UpdateScoreBar

  ' erase old values
  box 150,28, 75,30,1,rgb(magenta),rgb(magenta)
  box 500,28, 75,30,1,rgb(magenta),rgb(magenta)
  box 700,28,100,30,1,rgb(magenta),rgb(magenta)

  ' draw new values
  text 150,46,Format$(brickcount, "%03G"),"LM",5,1,0,-1 'rgb(magenta)
  text 500,46,Format$(level, "%03G"),"LM",5,1,0,-1 'rgb(magenta)
  text 700,46,Format$(score, "%04G"),"LM",5,1,0,-1 'rgb(magenta)
End Sub

'******************************
sub DrawOneBrick(bx, by, c)
  text bx*brickH - 20, by*brickV + 120,"1","CM",9,1,c
end sub

'******************************
'Fill the array with bricks and draw them on the screen
Sub DrawBricks
  brickcount = 0
  For i = 1 To 14
    For k = 1 To 7
      bricks(i,k) = k
      DrawOneBrick(i, k, CLR(bricks(i,k)))
      brickcount = brickcount + 1
    Next k
  Next i
End Sub

'******************************
' Various game bitmaps, stored as fonts just because
'
' Brick font
' "0" is blank
' "1" is brick
DefineFont #9
  03301036
  00000000 00000000 00000000 00000000 00000000 00000000 00000000 00000000
  00000000 00000000 00000000 00000000 00000000 00000000 00000000 00000000
  00000000 00000000 00000000 00000000 00000000 00000000 00000000 00000000
  00000000 00000000 00000000 FFFFFF7F FFFBFFFF FFFFFFFF FFFFFFFF FFFFFFFF
  FFFFFFFF FFFFFFFF FFFFFFFF FFFFFFFF FFFFFFFF FFFFFFFF FFFFFFFF FFFFFFFF
  FFFFFFFF FFFFFFFF FFFFFFFF FFFFFFFF FFFFFFFF FFFFFFFF FFFFFFFF FFFFFFFF
  FFFFFFFF FFFFFFFF FFFFFFFF FFFFFFFF FFFFDFFF FEFFFFFF 00000000 04000000
  E9C46752 4A110030 20252444 10E94500 0180F410 4240BC54 9202005E 48E90091
  00000000 00000000 00000000 18040000 730651C9 A9941000 00922464 50C55242
  09011092 C8C3944A C9980700 00210951 00000000 00000000 00000000 00000000
  00000000
End DefineFont

' Bat font
' "0" is blank
' "1" is bat
DefineFont #10
  02300840
  00000000 00000000 00000000 00000000 00000000 00000000 00000000 00000000
  00000000 00000000 00000000 00000000 00000000 00000000 00000000 00000000
  FFFFFF3F FCFFFFFF FFFFFF7F FEFFFFFF FFFFFFFF FFFFFFFF FFFFFFFF FFFFFFFF
  FFFFFFFF FFFFFFFF FFFFFFFF FFFFFFFF FFFFFF7F FEFFFFFF FFFFFF3F FCFFFFFF
End DefineFont

' circuit diagram font 128x248 "0"
DefineFont #11
  0130F880
  FFFFFFFF FFFFFFFF 00FCFFFF 00000000 00000080 00000000 00040000 00000000
  00000080 00000000 00040000 00000000 00000080 00000000 00040000 00000000
  00000080 00000000 00040000 00000000 00000080 00000000 00040000 00000000
  00000080 00000000 00040000 00000000 FFFFFF83 0080FFFF 00040000 00000000
  10000082 00880000 00040000 00000000 10000082 00880000 00040000 00000000
  10800082 E08F0007 00040000 00000000 10800182 00888008 00040000 00000000
  10800082 00888000 00040000 00000000 108000FE 00800003 00040000 00000000
  10800002 00800004 00040000 00000000 10800002 20860008 00040000 00000000
  10C00102 4089800F 00040000 00000000 10000002 80890000 00040000 00000000
  10000002 00890000 00040000 00000000 FFFFFF03 E08FFFFF 00040000 00000000
  10000002 00800000 00040000 00000000 10000002 00800000 00040000 00000000
  10C00102 C0870001 00040000 00000000 10200202 20880003 00040000 00000000
  10200002 20880005 00040000 00000000 10C00002 20880009 00040000 00000000
  10200002 C087800F 00040000 00000000 10200202 00800001 00040000 00000000
  10C00102 00800001 00040000 00000000 10000002 00860000 00040000 00000000
  10000002 00890000 00040000 00000000 FFFFFF03 0089FFFF 00040000 00000000
  10000002 00890000 00040000 00000000 10000002 E08F0000 00040000 00000000
  10E00302 00800007 00040000 00000000 10000202 00800008 00040000 00000000
  10C00302 00800008 00040000 00000000 10200002 0080000F 00040000 00000000
  10200002 00808008 00040000 00000000 10200202 00808008 00040000 00000000
  10C00102 C0870007 00040000 00000000 10000002 20880000 00040000 00000000
  10000002 20880000 00040000 00000000 FFFFFF03 2088FFFF 00040000 00000000
  10000002 C0870000 00040000 00000000 10000002 00800000 00040000 00000000
  10E00302 00800007 00040000 00000000 10200002 00988008 00040000 00000000
  10200002 00868008 00040000 00000000 10400002 00810007 00040000 00000000
  10800002 C0808008 00040000 00000000 10800002 30808008 00040000 00000000
  10800002 00800007 00040000 00000000 10000002 00800000 00040000 00000000
  10000002 20880000 00040000 00000000 FFFFFF03 E08FFFFF 00040000 00000000
  10000002 20880000 00040000 00000000 10000002 00800000 00040000 00000000
  10C00102 0080E020 00040000 00000000 10200202 00801061 00040000 00000000
  10200202 00801021 00040000 00000000 10E00102 00801021 00040000 00000000
  10200002 00801021 00040000 00000000 10200002 00801021 00040000 00000000
  10C00102 0080E070 00040000 00000000 10000002 00800000 00040000 00000000
  10000002 00800000 00040000 00000000 FFFFFF03 0080FFFF 00040000 00000000
  10000002 00800000 00040000 00000000 10000002 00800000 00040000 00000000
  10080402 0080E020 00040000 00000000 10180C02 00801061 00040000 00000000
  10080402 00801020 00040000 00000000 10080402 00806020 00040000 00000000
  10080402 00808020 00040000 00000000 10080402 00800021 00040000 00000000
  101C0E02 0080F071 00040000 00000000 10000002 00800000 00040000 00000000
  10000002 00800000 00040000 00000000 FFFFFF03 0080FFFF 00040000 00000000
  10000002 00800000 00040000 00000000 10000002 00800000 00040000 00000000
  101C0402 00802020 00040000 00000000 10220C02 00806060 00040000 00000000
  10020402 0080A020 00040000 00000000 100C0402 00802021 00040000 00000000
  10020402 0080F021 00040000 00000000 10220402 00802020 00040000 00000000
  101C0E02 00802070 00040000 00000000 10000002 00800000 00040000 00000000
  10000002 00800000 00040000 00000000 FFFFFF03 0080FFFF 00040000 00000000
  10000002 00800000 000E0000 00000000 10000002 00800000 00FEFF00 00000000
  103E0402 0080E020 000E8000 00000000 10200C02 00800061 00048000 00000000
  103C0402 00800021 00048000 00000000 10020402 0080E021 00048000 00000000
  10020402 00801021 00048000 00000000 10220402 00801021 00048000 00000000
  101C0E02 0080E070 00048000 00000000 10000002 00800000 00048000 00000000
  10000002 00800000 00048000 00000000 FFFFFF03 0080FFFF 00048000 00000000
  10000002 00800000 00048001 00000000 10000002 00800000 00040006 00000000
  103E0402 0080E020 00040018 00000000 10020C02 00801061 00040006 00000000
  10020402 00801021 00048001 00000000 10040402 0880E020 00046080 00000000
  10080402 05801021 00041800 00000000 10080402 02801021 00046000 00000000
  10080E02 3F80E070 00048081 00000000 10000002 00800000 00040006 00000000
  10000002 00800000 00040018 00000000 FFFFFF03 1F80FFFF 00040006 00000000
  10000002 20800000 00048081 00000000 10000002 20800000 00046080 00000000
  101C0402 2080E070 00041880 00000000 10220C02 1F801089 00046000 00000000
  10220402 00801009 00048001 00000000 101E0402 00801031 00040006 00000000
  10020402 00801041 00040098 00000000 10020402 3F801081 00040086 0000900F
  101C0E02 1080E0F8 00048081 00001008 10000002 00800000 00046000 0000120F
  10000002 00800000 00041800 00009400 FFFFFF03 0080FFFF 00046000 00009800
  10000002 00800000 00048001 00009408 10000002 00800000 00040006 00001207
  10080E02 0080E070 00040018 00000000 10181102 00801089 00040006 00000000
  10080102 00801008 00048001 00202020 10080602 00806030 00048000 00202020
  10080802 00808040 00048000 00505050 10081002 00800081 00048000 00505050
  101C1F02 0080F0F9 F8078000 FE888888 10000002 00800000 08008000 82888888
  10000002 00800000 05008000 02050505 FFFFFF03 0080FFFF 05008000 02050505
  10000002 00800000 02008000 02020202 10000002 00800000 121D8000 02020202
  101C0E02 00802070 10218000 02002000 10221102 00806088 50218000 02002000
  10020102 0080A008 50218000 02007000 100C0602 00802031 A01C8000 02007000
  10020802 0080F041 00008000 0200F800 10221002 00802080 00008000 02002000
  101C1F02 008020F8 00008000 02002000 10000002 00800000 00008000 02002000
  10000002 00800000 00008000 02002000 FFFFFF03 0080FFFF 00008000 02002000
  10000002 00800000 00008000 02002000 10000002 00800000 00008000 02002000
  103E0E02 0080E070 00008000 02002000 10201102 00800089 00000000 02002000
  103C0102 00800009 00000000 02002000 10020602 FFFFE031 FFFFFFFF 0200E0FF
  10020802 00801041 00000000 02000000 10221002 00801081 00000000 02000000
  101C1F02 0080E0F8 00008000 02000000 10000002 00800000 00008000 02000000
  10000002 00800000 00008000 02000000 FFFFFF03 0080FFFF 00008000 02000000
  10000002 00800000 FF1F8000 020000FF 10000002 00800000 00008000 02000000
  103E0E02 0080E070 00008000 02000000 10021102 00801089 000E8000 0200000E
  10020102 00801009 001FC001 0700001F 10040602 FFFFE030 00FFFFFF FFFFFF1F
  10080802 00801041 001FC001 0700001F 10081002 00801081 000E0000 0200000E
  10081F02 0080E0F8 00000000 02000000 10000002 00800000 00000000 02000000
  10000002 00800000 00000000 02000000 FFFFFF03 0080FFFF 00000000 02000000
  10000002 00800000 00000000 02000000 10000002 00800000 00000000 02000000
  101C0E02 0080E070 00000000 02000000 10221102 00801089 00000000 02000000
  10220102 00801009 00000000 02000000 101E0602 FFFF1031 FFFFFFFF FEFFFFFF
  10020802 00801009 00000000 00000000 10021002 00801089 00000000 00000000
  101C1F02 0080E070 00000000 00000000 10000002 00800000 00000000 00000000
  10000002 00800000 00000000 00000000 FFFFFF03 0080FFFF 00000000 00000000
  10000002 00800000 00000000 00000000 10000002 00800000 00000000 00000000
  10080E02 0080E070 00000000 00000000 10181102 00801089 00000000 00000000
  10080102 78801008 F0419E47 4E74DE45 10080602 44806030 004151A4 51269144
  10080102 45808008 00415114 50259144 10081102 79800089 C0415114 D0249E44
  101C0E02 4180F071 004151F4 57249454 10000002 41800000 00415114 51249254
  10000002 41800000 F07D9E17 4F74D129 FFFFFF03 0080FFFF 00000000 00000000
  10000002 00800000 00000000 00000000 10000002 7F800000 FFFFFFFF FFFFFFFF
  101C0E02 00802070 00000000 00000000 10221102 00806088 00000000 00000000
  10020102 0080A008 00000000 00000000 100C0602 00802031 C0E7E388 00000000
  10020102 0080F009 008410C9 00000000 10221102 00802088 088410A9 00000000
  101C0E02 00802070 00871099 00000000 10000002 00800000 08841089 00000000
  10000002 00800000 00841089 00000000 FFFFFF03 0080FFFF C087E088 00000000
  10000002 00800000 00000000 00000000 10000002 00800000 00000000 00000000
  103E0E02 0080E070 1C918803 00448E07 10201102 00800089 229B4D04 00644404
  103C0102 00800009 02950A04 00544404 10020602 0080E031 0C910804 004C8407
  10020102 00801009 10910804 00440404 10221102 00801089 20914804 00440404
  101C0E02 0080E070 3E918803 00440E04 10000002 00800000 00000000 00000000
  10000002 00800000 00000000 00000000 FFFFFF03 0080FFFF 9E482402 0070783E
  10000002 00800000 914D2403 00884420 10000002 00800000 914AA402 00804420
  103E0E02 0080E070 9E486402 00707838 10021102 00801089 91482402 00085020
  10020102 00801009 91482402 00884820 10040602 0080E030 9E882302 0070443E
  10080102 00801009 00000000 00000000 10081102 00801089 00000000 00000000
  10080E02 0080E070 0F010000 0000001F 10000002 00800000 88020000 00000090
  10000002 00800000 48040000 00000090 FFFFFF03 0080FFFF 4F040000 0000001C
  10000002 00800000 CA070000 00000010 10000002 00800000 49040000 00000010
  101C0E02 0080E010 48040000 0000009F 10221102 00801031 00000000 00000000
  10220102 00801051 00000000 00000000 101E0602 00801091 F9443E1E 00CFC7E1
  10020102 008010F9 81442011 80082412 10021102 00801011 81442011 80080412
  101C0E02 0080E010 E128381E 8008C7E1 10000002 00800000 81282014 80082440
  10000002 00800000 81102012 80082422 FFFFFF03 0080FFFF F9103E11 00CFC711
End DefineFont
