/*******************************************************************************
*
* Driver for SSD1289 Display written as CFunctions
*
* (c) Peter Mather 2015 with acknowledgements to Peter Carnegie & Geoff Graham
*
*
* This CFunction MUST be compiled with Optimization Level 1, -O1
* -O2,-O3,-Os will compile successfully, but generate Exceptions at runtime.
*
* When Generating the CFunction, use MERGE CFunction mode, and name the CFunction
* Driver_SSD1289
*
* Entry point is function long long main(long long *MyAddress,long long *func,
* long long *RST,
* long long *orientation)
*
* V1.0 2015-07-21 Peter Mather
* V1.01 Change orientation codes to 1:landscape, 2:portrait, 3:rlandscape, 4:rportrait
* V1.1 2016-01-28 Selectable 44 or 64-pin
* V1.21 2016-06-24 Improved bitmap error handling
*
******************************************************************************/
#include <stdarg.h>
#include <xc.h> // Required for SFR defs
#include <sys/attribs.h> // Required to use __longramfunc__

#define Version 120 //Version 1.20

#include "../cfunctions.h"
#define MX470
#define bclrport *(volatile unsigned int *)(0xbf886134) //latch registers
#define bsetport *(volatile unsigned int *)(0xbf886138) //latch registers
#define cport *(volatile unsigned int *)(0xbf886230) //latch registers
#define cread *(volatile unsigned int *)(0xbf886220) //latch registers
#define ctrisinv *(volatile unsigned int *)(0xbf88621C) //latch registers
#define cclrport *(volatile unsigned int *)(0xbf886234) //latch registers
#define csetport *(volatile unsigned int *)(0xbf886238) //latch registers
#define eport *(volatile unsigned int *)(0xbf886430) //latch registers
#define eread *(volatile unsigned int *)(0xbf886420) //latch registers
#define etrisinv *(volatile unsigned int *)(0xbf88641C) //latch registers
#define eclrport *(volatile unsigned int *)(0xbf886434) //latch registers
#define esetport *(volatile unsigned int *)(0xbf886438) //latch registers
#define DEVID (*(volatile unsigned int *)0xBF80F220)

//Offsets into the persistent RAM of variables
#ifdef MX470
#define RS_Pin_No 27
#define WR_Pin_No 24
#define RS_Pin 0x1000
#define WR_Pin 0x0800
#define clrport bclrport
#define setport bsetport
#define port eport
#define read eread
#define trisinv etrisinv
#define RSLo {clrport=RS_Pin;wt=DEVID;}
#define RSHi {setport=RS_Pin;wt=DEVID;}
#define WRLo {clrport=WR_Pin;wt=DEVID;}
#define WRHi {setport=WR_Pin;wt=DEVID;}
#else
#define RS_Pin_No 4
#define WR_Pin_No 5
#define RS_Pin 0x100
#define WR_Pin 0x200
#define clrport cclrport
#define setport csetport
#define port cport
#define read cread
#define trisinv ctrisinv
#define RSLo clrport=RS_Pin
#define RSHi setport=RS_Pin
#define WRLo clrport=WR_Pin
#define WRHi setport=WR_Pin
#endif
#define Both WR_Pin | RS_Pin
#define RDLo (*(volatile unsigned int *)RDclrport)=RDpin
#define RDHi (*(volatile unsigned int *)RDsetport)=RDpin
#define LANDSCAPE 1
#define PORTRAIT 2
#define RLANDSCAPE 3
#define RPORTRAIT 4


/*******************************************************************************
*
* Write Data to a register on the Chip
*
******************************************************************************/

void writeRegister(unsigned int command, unsigned int data)
{
  #ifdef MX470
  int wt;
  RSLo;
  #endif
  port=(((command>>8) & 0xFF) | WR_Pin);
  WRLo;
  // int i=DEVID;
  WRHi;
  port=(((command) & 0xFF) | WR_Pin);
  WRLo;
  // i=DEVID;
  #ifdef MX470
  RSHi;
  #endif
  WRHi;
  port=(((data>>8) & 0xFF) | Both);
  WRLo;
  // i=DEVID;
  WRHi;
  port=(((data) & 0xFF) | Both);
  WRLo;
  // i=DEVID;
  WRHi;
  return;
}
/*******************************************************************************
*
* defines start/end coordinates for memory access from host to SSD1289
* also maps the start and end points to suit the orientation
*
* This function is a modified version of the function inside the MMBasic Interpreter
* for MM+ on 'MX470 chips
*
*******************************************************************************/
void defineregion(long x, long y, long width,long height)
{ //SSD1289
  long x1=x,x2=x+width-1,y1=y,y2=y+height-1;
  unsigned long xstart,xend,ystart,yend,xp,yp,Vertical,Horizontal;
  
  if(HRes<VRes){
    Vertical=VRes;
    Horizontal=HRes;
  }
  else 
  {
    Vertical=HRes;
    Horizontal=VRes;
  }
  
  switch(Option->DISPLAY_ORIENTATION)
  {
    case PORTRAIT:
      xstart = x1;
      xend = x2;
      ystart = y1;
      yend = y2;
      xp=xstart; yp=ystart;
      break;
    case RPORTRAIT:
      xstart = (Horizontal - 1) - x2;
      xend = (Horizontal - 1) - x1;
      ystart = (Vertical - 1) - y2;
      yend = (Vertical - 1) - y1;
      xp=xend; yp=yend;
      break;
    case LANDSCAPE:
      xstart = (Horizontal - 1) - y2;
      xend = (Horizontal - 1) - y1;
      ystart = x1;
      yend = x2;
      xp=xend; yp=ystart;
      break;
    default:
      xstart = y1;
      xend = y2;
      ystart = (Vertical - 1) - x2;
      yend = (Vertical - 1) - x1;
      xp=xstart; yp=yend;
      break;
  }
  writeRegister(0x44,xstart | (xend<<8));
  writeRegister(0x45,ystart);
  writeRegister(0x46,yend);
  writeRegister(0x4E,xp);
  writeRegister(0x4F,yp);
  
  #ifdef MX470
  int wt;
  RSLo;
  #endif
  
  port= WR_Pin;
  WRLo;
  WRHi;
  port=(0x22 | WR_Pin);
  WRLo;
  WRHi;
}



/***********************************************************************************************
* Display the bitmap of a char on the TFT panel
*
* This function is NEVER called by MMBasic programs
*
* This is optimised for the
* x, y - the top left of the char
* width, height - size of the char's bitmap
* scale - how much to scale the bitmap
* fg, bg - foreground and background colour
* bitmap - pointer to the butmap
*
* This function is a modified version of the function inside the MMBasic Interpreter
* for MM+ on 'MX470 chips
***********************************************************************************************/
void DrawBitmap_SSD1289(int x1, int y1, int width, int height, int scale, int fc, int bc, unsigned char *bitmap )
{
  long i, j, k, m;
  unsigned int fch,fcm,fcl,bch,bcm,bcl;
  int vertCoord, horizCoord, XStart, XEnd, YEnd;
  
  // adjust when part of the bitmap is outside the displayable coordinates
  vertCoord = y1; if(y1 < 0) y1 = 0; // the y coord is above the top of the screen
  XStart = x1; if(XStart < 0) XStart = 0; // the x coord is to the left of the left marginn
  XEnd = x1 + (width * scale) - 1; 
  if(XEnd >= HRes) XEnd = HRes - 1; // the width of the bitmap will extend beyond the right margin
  YEnd = y1 + (height * scale) - 1; 
  if(YEnd >= VRes) YEnd = VRes - 1;// the height of the bitmap will extend beyond the bottom margin

  fch=(fc>>16) & 0x3FC | Both;
  fcm=(fc>>8) & 0x3FC | Both;
  fcl=fc & 0x3FC | Both;
  bch=(bc>>16) & 0x3FC | Both;
  bcm=(bc>>8) & 0x3FC | Both;
  bcl=bc & 0x3FC | Both;


  defineregion(XStart,y1,XEnd-XStart+1,YEnd-y1+1);
  #ifdef MX470
  int wt;
  RSHi;
  #endif
  for(i = 0; i < height; i++) 
  { // step thru the font scan line by line
    for(j = 0; j < scale; j++) 
    { // repeat lines to scale the font
      if(vertCoord++ < 0) continue; // we are above the top of the screen
      if(vertCoord > VRes) return; // we have extended beyond the bottom of the screen
      horizCoord = x1;
      for(k = 0; k < width; k++) 
      { // step through each bit in a scan line
        for(m = 0; m < scale; m++) 
        { // repeat pixels to scale in the x axis
          if(horizCoord++ < 0) continue; // we have not reached the left margin
          if(horizCoord > HRes) continue; // we are beyond the right margin

          if((bitmap[((i * width) + k)/8] >> (((height * width) - ((i * width) + k) - 1) %8)) & 1) 
          {
            port=fch;WRLo;WRHi;
            port=fcm;WRLo;WRHi;
            port=fcl;WRLo;WRHi;
          } 
          else 
          {
            port=bch;WRLo;WRHi;
            port=bcm;WRLo;WRHi;
            port=bcl;WRLo;WRHi;
          }
        }
      }
    }
  }
}

/*******************************************************************************
*
* Called by MMBasic Interpreter to draw a rectangle
* on the SSD1289 panel
*
* This function is NEVER called by MMBasic programs
*
*
******************************************************************************/

void DrawRectangle_SSD1289(int x1, int y1, int x2, int y2, int fc)
{
  int t;
  long width, height;
  unsigned long i,ch,cm,cl;
  
  // make sure the coordinates are kept within the display area
  if(x2 <= x1) { t = x1; x1 = x2; x2 = t; }
  if(y2 <= y1) { t = y1; y1 = y2; y2 = t; }
  if(x1 < 0) x1 = 0; if(x1 >= HRes) x1 = HRes - 1;
  if(x2 < 0) x2 = 0; if(x2 >= HRes) x2 = HRes - 1;
  if(y1 < 0) y1 = 0; if(y1 >= VRes) y1 = VRes - 1;
  if(y2 < 0) y2 = 0; if(y2 >= VRes) y2 = VRes - 1;
  width=x2-x1+1;height=y2-y1+1;
  i=width*height;
  defineregion(x1,y1,width,height);
  
  #ifdef MX470
  int wt;
  RSHi;
  #endif
  
  // convert the colours to 565 format
  ch=(fc>>16) & 0x3FC | Both;
  cm=(fc>>8) & 0x3FC | Both;
  cl=fc & 0x3FC | Both;

  if((ch==cl) && (ch==cm))
  {
  port=ch; WRLo; WRHi;
  WRLo; WRHi;
  WRLo; WRHi;
  i--;
    while (i--)
    {
      WRLo;
      WRHi;
      WRLo;
      WRHi;
      WRLo;
      WRHi;
    }
  }
  else 
  {
    while (i--) 
    {
      port=ch; WRLo; WRHi;
      port=cm; WRLo; WRHi;
      port=cl; WRLo; WRHi;
    }
  }
}

__attribute__((noinline)) void getFPC(void *a, void *b, volatile unsigned int *c)

{
  *c = (unsigned int) (__builtin_return_address (0) - (b -a)) ;
}

/*******************************************************************************
*
* Driver_SSD1289 : Initialise the CFunction Driver Sub-System
*
* Function called to initialise the driver SubSystem
*
* OpenDriver_SSD1289 is ALWAYS called from an MMBasic program
* On exit, vectors DrawRectangleVector, and DrawBitmapVector will
* be set to point to the CFunctions DrawRectangle_SSD1289 and
* DrawBitmap_SSD1289 respectively
*
******************************************************************************/
//CFunction Driver_SSD1289
#ifdef MX470
void main(long long *orientation)
{
#else
void main(long long *RST,long long *orientation)
{
  #endif //Close Driver Request ?
  int t,HorizontalRes,VerticalRes;


  int j;
  unsigned int p[8];
  volatile unsigned int libAddr ;
  getFPC(NULL,&&getFPCLab,&libAddr) ; // warning can be ignored, stupid editor
  getFPCLab: { }

  static const unsigned short _initData[] = {
  0x00,0x0001,0x03,0xA8A4,0x0C,0x0000,0x0D,0x080C,0x0E,0x2B00,0x1E,0x00B7,
  0x01,0x2B3F,0x02,0x0600,0x10,0x0000,0x11,0x6070,0x05,0x0000,0x06,0x0000,
  0x16,0xEF1,0x17,0x0003,0x07,0x0233,0x0B,0x0000,0x0F,0x0000,0x41,0x0000,
  0x42,0x0000,0x48,0x0000,0x49,0x013F,0x4A,0x0000,0x4B,0x0000,0x30,0x0707,
  0x31,0x0204,0x32,0x0204,0x33,0x0502,0x34,0x0507,0x35,0x0204,0x36,0x0204,
  0x37,0x0502,0x3A,0x0302,0x3B,0x0302,0x23,0x0000,0x24,0x0000,0x25,0x8000};
  unsigned short * initData = (unsigned short *)((void *)_initData + libAddr );
  static const char _testData[]="Loading SSD1289 driver\r\n";
  unsigned char * testData = (unsigned char *)((void *)_testData + libAddr );

  MMPrintString(testData);
  Option->DISPLAY_ORIENTATION=*orientation;
  #ifdef MX470
  int wt;
  Option->LCD_Reset=28;
  p[0]=60;
  p[1]=61;
  p[2]=62;
  p[3]=63;
  p[4]=64;
  p[5]=1;
  p[6]=2;
  p[7]=3;

  #else
  Option->LCD_Reset=*RST;
  p[0]=25;
  p[1]=26;
  p[2]=27;
  p[3]=36;
  p[4]=37;
  p[5]=38;
  p[6]=2;
  p[7]=3;
  #endif
  for(j=0;j<8;j++)
  {
    ExtCfg(p[j],EXT_DIG_OUT,0);
    ExtCfg(p[j],EXT_BOOT_RESERVED,0);
    PinSetBit(p[j],LATCLR);
  }

  //Control Signals
  ExtCfg(RS_Pin_No,EXT_DIG_OUT,0);ExtCfg(RS_Pin_No,EXT_BOOT_RESERVED,0);
  RSHi;

  ExtCfg(WR_Pin_No,EXT_DIG_OUT,0);ExtCfg(WR_Pin_No,EXT_BOOT_RESERVED,0);
  WRHi;

  ExtCfg(Option->LCD_Reset,EXT_DIG_OUT,0);ExtCfg(Option->LCD_Reset,EXT_BOOT_RESERVED,0);
  PinSetBit(Option->LCD_Reset, LATSET);


  HorizontalRes=240;
  VerticalRes=320;

  /************************************************
  *
  * Initialise the SSD1289
  *
  ***********************************************/

  //Reset the SSD1289
  PinSetBit(Option->LCD_Reset,LATSET);
  uSec(10000);
  PinSetBit(Option->LCD_Reset,LATCLR);
  uSec(10000);
  PinSetBit(Option->LCD_Reset,LATSET);
  uSec(10000);
  for(j=0;j<72;j+=2)writeRegister(initData[j],initData[j+1]);

  //Set Hres and VRes
  switch(Option->DISPLAY_ORIENTATION)
  {
    case PORTRAIT:
    t=0x4070;
    break;

    case RPORTRAIT:
    t=0x4040;
    break;

    case LANDSCAPE:
    t=0x4068;
    break;

    default:
    t=0x4058;
  }
  writeRegister(0x11,t);

  if(Option->DISPLAY_ORIENTATION&1)
  { //landscape is not default
    VRes=HorizontalRes;
    HRes=VerticalRes;
  } 
  else 
  {
    HRes=HorizontalRes;
    VRes=VerticalRes;
  }

  //Save the current values of the Vectors

  //Set the DrawRectangle vector to point to our function
  DrawRectangleVector= (unsigned int)&DrawRectangle_SSD1289 + libAddr;

  //Set the DrawBitmap vector to point to our function
  DrawBitmapVector=(unsigned int)&DrawBitmap_SSD1289 + libAddr;

  //Draw a black box = CLS
  DrawRectangle(0,0,HRes-1,VRes-1,0x000000);

} 