OPTION EXPLICIT  ' all variables must be defined before use

' GetFile Demonstration Program
' Shows usage for the GetFile dialog function.
'
' by vegipete, October 2020
'   version 1.0   Original release
'   version 1.1   all navigation by arrow keys only, can return directory names too
'   version 1.2   remembers selection when moving back up directories, can specify
'                 file criteria, forcing selection type
'   version 2.0   rewrite with mouse in mind
'
'===================
' required setup for function
DIM MousePort = 0     ' mouse port, set to -1 if no mouse
DIM mouse_d(5) = (0,0,0,0,0,0) ' persistent mouse data
CONST DOUBLETIME = 333  ' max time (ms) to register double click
CONST MOUSE_L_DOWN = 1
CONST MOUSE_L_UP = 2
CONST MOUSE_L_DOUBLE = 6

DIM UI_elements(15,7)
'              (  ,0)   type
'              (  ,1)   x location
'              (  ,2)   y location
'              (  ,3)   width
'              (  ,4)   height
'              (  ,5)
'              (  ,6)
'              (  ,7)
DIM UI_elemtxt$(15) length 20 ' caption string

CONST DIRCOUNT = 75   ' max number of sub-directories
CONST FILCOUNT = 255  ' max number of files
CONST NAMELENGTH = 64 ' The maximum file/path length is 127 characters.
DIM UB_dirs$(DIRCOUNT) length NAMELENGTH  ' store list of directories
DIM UB_fils$(FILCOUNT) length NAMELENGTH  ' store list of files
DIM UB_hist$(DIRCOUNT) length 8 ' store directory number visited along path
DIM UB_msel$(1)                 ' flag of items multi-selected 0=DIR, 1=FILES
DIM UB_vals(10) as integer      ' misc internal values
'          ( 0)  dialog box top left corner X
'          ( 1)  dialog box top left corner Y
'          ( 2)  height in characters
'          ( 3)  width in characters
'          ( 4)  height in pixels
'          ( 5)  width in pixels
'          ( 6)  text region height in pixels
'          ( 7)  select file if 0, select directory if 1
'          ( 8)  single select if 0, otherwise max number of selections
'          ( 9)  multi-select directories if 1
'          (10)
'DIM UB_dpcl(9) as integer       ' standard colours
DIM UB_cols(3) = (&hA0A040,&h101010,&hFFFFFF,&h303030) 'array of 4 colour values
'    UB_cols(0)  frame               (&hA0A040 sort-of gold     )
'    UB_cols(1)  body                (&h101010 really dark grey )
'    UB_cols(2)  text                (&hFFFFFF white            )
'    UB_cols(3)  shadow              (&h303030 dark sort-of gold)

' end of setup for function
'===================

'===================
' Start of Demonstration Code
'===================
DIM result$
DIM msg$, a$

mode 1,8
cls

FillBackground
MouseStart
MouseShow

'--------------------
font 4
msg$ = "Improve the user experience."
UB_Message(msg$,2000,,50,,len(msg$))

font 2
msg$ = "Make file and directory selection a breeze with..."
UB_Message(msg$,2000,,100,,35)

font 3
UB_cols(2) = &hFFA040   ' orange colour text
msg$ = "GetFileNameDialog!"
UB_Message(msg$,2500,,200,,len(msg$))

'--------------------
font 1
UB_cols(2) = &hFFFFFF   ' white text
msg$ = "Get ready to try it out! When the dialog box is open, use the arrow keys to navigate"
msg$ = msg$ + " and choose a file. Press [Enter] to select the file. Press [Esc] to cancel."
UB_Message(msg$,0,,250,,65)

UB_GetFile(result$)

msg$ = "Awesome! You chose:" + chr$(13)
if result$ = "" then
  msg$ = msg$ + "Nothing. That's OK. It was your choice."
else
  msg$ = msg$ + result$
endif
UB_Message(msg$,2000,,200,,44)

'--------------------
msg$ = "You can also specify the type of file to display. For example,"
msg$ = msg$ + chr$(13) + "'UB_GetFile(result$," + chr$(34) + "*.bas" + chr$(34) + ")'"
msg$ = msg$ + chr$(13) + "will show only BASIC files."
UB_Message(msg$,0,,250,,50)

UB_GetFile(result$,"*.bas")

msg$ = "Marvelous! You chose:" + chr$(13)
if result$ = "" then
  msg$ = msg$ + "Nothing. That's OK. It was your choice."
else
  msg$ = msg$ + result$
endif
UB_Message(msg$,2000,,300,,54)

'--------------------
msg$ = "It is also possible to select a directory!" + chr$(13)
msg$ = msg$ + "The command to do this is:"  + chr$(13)
msg$ = msg$ + "'UB_GetFile(result$," + chr$(34) + "<DIR>" + chr$(34) + ")'"
UB_Message(msg$,0,,250,,50)

UB_GetFile(result$,"<DIR>")

msg$ = "Fantastic! You chose:" + chr$(13)
if result$ = "" then
  msg$ = msg$ + "Nothing. That's OK. It was your choice."
else
  msg$ = msg$ + result$
endif
UB_Message(msg$,2000,,300,,54)

'--------------------
msg$ = "As you can see, GetFileNameDialog is great."
msg$ = msg$ + " You will notice too that the screen is restored when the dialog"
msg$ = msg$ + " closes. This makes GetFileNameDialog simple to add to your own"
msg$ = msg$ + " programs. So what are you waiting for? Get programming today!"
UB_Message(msg$,0,,350,,40)

font 2
UB_cols(2) = &h20FF20   ' green colour text
msg$ = "The End."
UB_Message(msg$,1500,,200,,len(msg$))
MouseStop
print @(0,500)
end

'*******************************
' Fill the screen with something - random coloured hexagons
sub FillBackground
  dim integer xorg(6) = (-10,-5,5,10,5,-5,-10)
  dim integer xgon(6)
  dim integer ygon(6) = (0,9,9,0,-9,-9,0)
  dim i,j,v

  for j = 1 to 28
    for v = 0 to 6
      xgon(v) = xorg(v)
      ygon(v) = ygon(v) + 9
    next v

    for i = 1 to 26
      for v = 0 to 6
        xgon(v) = xgon(v) + 30
      next v
      polygon 6, xgon(), ygon(),rgb(white),rgb(rnd*255,rnd*255,rnd*255)
    next i

    for v = 0 to 6
      xgon(v) = xorg(v) + 15
      ygon(v) = ygon(v) + 9
    next v

    for i = 1 to 25
      for v = 0 to 6
        xgon(v) = xgon(v) + 30
      next v
      polygon 6, xgon(), ygon(),rgb(white),rgb(rnd*255,rnd*255,rnd*255)
    next i

  next j
end sub

'===================
' End of Demonstration Code
'===================

'*****************************************************************
' Sub MessageBox(message$ [,type][,x_position][,y_position][,height][,width])
'
' This routines displays a message box on the screen.
' The box can either be timed or require a keypress/mouse click to close.
'
' Input:
'   message$    The message to display
'                 some attempt is made to split longer lines on space characters
'               special characters:
'                 CHR$(13)      - line break
'                 CHR$(26)+"n"  - change to colour #n, where n = (0,9)
'                     (black,red,green,blue,yellow,cyan,magenta,brown,gray,white)
'                     Note: it is up to the programmer so select sensible colours.
'                 CHR$(24)      - cancel change
'   type        type of message box
'                 0: (default) "Press any key to continue..."
'                 value:  time in milliseconds before close and exit (keypress for early exit)
'
'   x_position  top left corner of message box on screen
'   y_position    default is centered on screen
'
'   height      size of message box in characters - default height is enough to fit message
'   width         default 40 characters wide

sub UB_Message(msg$ , type, xpos, ypos, high, wide)
  local integer y,UB_cwidth,UB_x,UB_y,UB_lines,UB_width,UB_height
  local float temptime
  local UB_1$, UB_2$
  local UB_c$ = "Press a key to continue..."  ' string for localization

  'SetupColours  ' fill the colour table (again)

  ' message box dimensions
  ' test parameters, fill in defaults if needed
  UB_cwidth = wide : if UB_cwidth = 0 then UB_cwidth = 40       ' width in characters
  if UB_cwidth < 16 then UB_cwidth = 16                         ' minimum width
  UB_width = 30 + UB_cwidth * MM.INFO(FONTWIDTH)                ' width in pixels

  '==================
  ' calculate vertical size based on source text
  ' code mostly repeated below but there are too many paramaters to subroutine-ize
  UB_1$ = msg$
  for y = len(msg$) to 1 step -1
    if (mid$(UB_1$,y,1) < chr$(32)) and (mid$(UB_1$,y,1) <> chr$(13)) then
      UB_1$ = left$(UB_1$,y-1) + mid$(UB_1$,y+1)  ' chop out CTRL character
    endif
  next y
  y = 0
  do
    do while left$(UB_1$,1) = " "
      UB_1$ = mid$(UB_1$,2)   ' hack off leading spaces
    loop
    if UB_1$ = "" then exit do
    SplitString(UB_1$, UB_2$, UB_cwidth)
    y = y + 1
    UB_1$ = UB_2$
  loop
  y = y + 2
  '==================

  UB_lines = high ': if UB_lines = 0 then UB_lines = 8     ' height in characters
  if UB_lines < y then UB_lines = y
  if type = 0 then UB_lines = UB_lines + 1          ' another line for "Press a key..."
  UB_height = 15 + UB_lines * MM.INFO(FONTHEIGHT)   ' height in pixels

  UB_x=xpos : if UB_x=0 then UB_x=(MM.HRES -  UB_width)/2  ' location of top left corner
  UB_y=ypos : if UB_y=0 then UB_y=(MM.VRES - UB_height)/2 '    of message box

  MouseHide
  ' save underlying screen image in buffer #63
  blit read 63, UB_x, UB_y, UB_width, UB_height
  ' draw dialog box
  rbox UB_x+7, UB_y+7, UB_width- 8, UB_height- 8, 10, UB_cols(3), UB_cols(3)
  rbox UB_x  , UB_y  , UB_width- 8, UB_height- 8, 10, UB_cols(0), UB_cols(0)
  rbox UB_x+5, UB_y+5, UB_width-18, UB_height-18,  5, UB_cols(1), UB_cols(1)  ' text area

  '==================
  ' this time, actually print the text
  UB_1$ = msg$
  y = 0
  do
    do while left$(UB_1$,1) = " "
      UB_1$ = mid$(UB_1$,2)   ' hack off leading spaces
    loop
    if UB_1$ = "" then exit do
    SplitString(UB_1$, UB_2$, UB_cwidth)
    text UB_x+8,UB_y+y*MM.INFO(FONTHEIGHT)+10,UB_1$,"LT",,,UB_cols(2),UB_cols(1)
    y = y + 1
    UB_1$ = UB_2$
    if y > UB_lines-1 then exit do    ' don't overflow message box - shouldn't happen
  loop
  MouseShow
  '==================

  ' if type is non-zero, display message for type millisecs, allow keypress to exit early
  if type then
    temptime = timer + abs(type)
  else
    MouseHide
    text UB_x+UB_width-15,UB_y+UB_height-15,UB_c$,"RB",7,1,UB_cols(2),UB_cols(1)
    MouseShow
    temptime = 1E499    ' far in the future - essentially never
  endif
  do : loop until (inkey$ <> "") or (timer > temptime) or (mouse_d(3) = MOUSE_L_UP)
  mouse_d(3) = 0

  ' restore original screen image  (box not needed with  f/w v5.05.06+)
  MouseHide
  box UB_x, UB_y, UB_width, UB_height, 1, 0, 0 ' must clear to black first
  blit write 63, UB_x, UB_y ', 0   ' now restore all non-black pixels
  blit close 63
  MouseShow
  do : loop until inkey$ = ""   ' clear the keyboard buffer

end sub

'==================
' SplitString : used for word wrapping
' Split 'in$' at a deliminator in 'd$' such that 'in$' is a long as possible but not
' longer than 'size'. Return the left over string, without the deliminator, in 'tail$'
' CHR$(13) allows for line breaks
sub SplitString(in$, tail$, size, d$)
  local integer sp
  local string head$,delim$

  tail$ = ""
  sp = instr(in$, chr$(13))
  if sp > 0 and sp < size then ' is there a line break in the first size characters?
    tail$ = mid$(in$,sp+1)
    in$ = left$(in$,sp-1)
    exit sub
  endif
  if size >= len(in$) then exit sub
  delim$ = d$
  if delim$ = "" then delim$ = " "
  sp = size + 1
  do
    'if instr(delim$, mid$(in$,sp,1)) then
    if delim$ = mid$(in$,sp,1) then
      tail$ = mid$(in$,sp+1)  'right$(in$,len(in$) - sp)
      head$ = left$(in$,sp)
      exit do
    endif
    sp = sp - 1
  loop until sp = 0
  if sp then
    in$ = head$
  else    ' no deliminator found
    tail$ = mid$(in$,size+1)    ' break at message box width
    in$ = left$(in$,size)
  endif
  do while instr(delim$, right$(in$,1)) ' hack off trailing spaces
    in$ = mid$(in$, 1, len(in$) - 1)
  loop
end sub

'sub SetupColours
'  UB_dpcl(0) = &h000000 ' black
'  UB_dpcl(1) = &hFF0000 ' red
'  UB_dpcl(2) = &h00FF00 ' green
'  UB_dpcl(3) = &h0000FF ' blue
'  UB_dpcl(4) = &hFFFF00 ' yellow
'  UB_dpcl(5) = &h00FFFF ' cyan
'  UB_dpcl(6) = &hFF00FF ' magenta
'  UB_dpcl(7) = &hFF8000 ' brown
'  UB_dpcl(8) = &h808080 ' gray
'  UB_dpcl(9) = &hFFFFFF ' white
'end sub

'*****************************************************************
' Sub FileDialog(result$() [,spec$][,capt$][,x_position][,y_position][,height][,width])
'
' This routine displays a centered dialog box on the screen, allows
' the user to choose a file and returns the full path of the chosen
' file. The underlying screen is restored when the dialog closes.
' UP and DOWN arrows to select, ENTER to choose selection
' ESC to cancel, LEFT arrow to go up directory
'
'   version 1.0   Original release vegipete, Oct 2020
'   version 1.1   all navigation by arrow keys only, can return directory names too
'   version 1.2   remembers selection when moving back up directories, can specify
'                 file criteria, forcing selection type
'   version 1.3   multi-file select, base code for save file
'
' Input:
'   result$(): array to hold results, dim(1) for single result, more for multi
'   spec$: optional - wildcard matching, plus following
'     standard functions:
'       "*"        show all files, allow any file (no DIR) to be selected - default
'       "*.BAS"    show only request file types, standard MMBasic wildcard matching
'       "<DIR>"    show all files, allow only selection of directories
'     multi-select functions (only if result$ is an array, otherwise defaults to standard):
'       "<M>*"     show and multi-select all file types
'       "<M>*.BAS" show and multi-select specified file types
'       "<MDIR>"   show all, multi-select only directories
'       "<MALL>"   show all, multi-select files and directories
'     save file function:
'       "<SAVE>["starting value"]" show all, returns path of displayed directory + string
'                              from text box, which was initialized with the optional string.
'   p_cap$:      optional title for the dialog box
'   x_position:  optional, default = centered dialog
'   y_position:  optional, default = centered dialog
'   height:      optional, height in characters of directory/file list, default 15
'   width:       optional, width in characters of directory/file list, default 32
'
' Output:
'   result$(0) contains full path of [first] item chosen, or "" if nothing
'   result$(1-n) contain full path of remaining items chosen, or "" if nothing
'   Note: the directory part of the path will be capitalized. This is just
'   how the CWD$ function works. Fortunately, MMBasic is case insensitive.
'
' The following global variables should be declared before use:
' CONST DIRCOUNT = 50   ' max number of sub-directories
' CONST FILCOUNT = 255  ' max number of files
' CONST NAMELENGTH = 64
' DIM UB_dirs$(DIRCOUNT) length NAMELENGTH  ' store list of directories
' DIM UB_fils$(FILCOUNT) length NAMELENGTH  ' store list of files
' DIM UB_hist$(DIRCOUNT) length 8 ' store directory number visited along path
' DIM UB_vals(10) as integer      ' misc internal values
' DIM UB_cols(3) = (&hA0A040,&h101010,&hFFFFFF,&h303030) 'array of 4 colour values
'    'UB_cols(0)  frame               (&hA0A040 sort-of gold     )
'    'UB_cols(1)  body                (&h101010 really dark grey )
'    'UB_cols(2)  text                (&hFFFFFF white            )
'    'UB_cols(3)  shadow              (&h303030 dark sort-of gold)
'
' Routines Used:  (included below)
'   sub UB_ReadDir   ' reads current directory into the above arrays
'   sub UB_ListDir(first, nlines, hilite)  ' shows a portion of the current directory
'
' Entry for get single file
Sub UB_GetFile(result$, p_spec$, p_cap$, p_x, p_y, p_height, p_width)
  local oneresult$(2)
  local GF_spec$ = ucase$(p_spec$)

  if left$(GF_spec$,3) = "<M>" or left$(GF_spec$,4) = "<M>*" then
    GF_spec$ = "*"
  elseif left$(GF_spec$,2) = "<M" then
    GF_spec$ = "<" + mid$(GF_spec$,3)
  endif
  UB_GetMultiFile(oneresult$(), GF_spec$, p_cap$, p_x, p_y, p_height, p_width)
  result$ = oneresult$(0)
end sub

' Entry all GetFile functions
Sub UB_GetMultiFile(result$(), p_spec$, p_cap$, p_x, p_y, p_height, p_width)
  local UB_spec$, UB_startdir$, UB_mode
  local UB_k, UB_changed, UB_tmp
  local UB_top_item, UB_sel_item, UB_top_last, UB_chosen
  local UB_capt$
  local MIFH = MM.INFO(FONTHEIGHT)

  ' test parameters, fill in defaults if needed
  UB_spec$ = ucase$(p_spec$) : if UB_spec$ = "" then UB_spec$ = "*"
  UB_vals(7) = 0
  if left$(UB_spec$,5) = "<DIR>" then UB_vals(7) = 1  ' select 1 directory
  UB_vals(8) = 0
  if left$(UB_spec$,2) = "<M" then
    UB_vals(8) = bound(result$(),1)   ' multi-select
    UB_msel$(0) = string$(255,0)      ' set all directory select flags to 0
    UB_msel$(1) = string$(255,0)      ' set all file select flags to 0
  endif
  UB_vals(9) = 0
  if left$(UB_spec$,5) = "<MDIR" then UB_vals(9) = 1
  if left$(UB_spec$,5) = "<MALL" then UB_vals(9) = 1
  ' figure out dialog box title
  if p_cap$ = "" then
    if UB_vals(8) then    ' multiple select?
      if left$(UB_spec$,6) = "<MDIR>" then
        UB_capt$ = "Select Directories..."  ' strings for localization
      elseif left$(UB_spec$,6) = "<MALL>" then
        UB_capt$ = "Select Items..."        ' strings for localization
      else
        UB_capt$ = "Select Files..."        ' strings for localization
      endif
    elseif UB_vals(7) then
      UB_capt$ = "Select Directory..."      ' strings for localization
    else
      UB_capt$ = "Select File..."           ' strings for localization
    endif
  else
    UB_capt$ = p_cap$   ' set caption to passed parameter
  endif

  ' dialog box dimensions
  UB_vals(2) = p_height : if UB_vals(2) = 0 then UB_vals(2) = 15 ' height in characters
  UB_vals(4) = 50 + (UB_vals(2) - 1) * MIFH       ' height in pixels
  UB_vals(6) = UB_vals(4) - MIFH*3 + 2    ' text region height in pixels
  UB_mode = 0      ' getfile mode
  if left$(UB_spec$,6) = "<SAVE>" then
    UB_vals(4) = UB_vals(4) + MIFH + 3 ' make room for filename text line
    UB_mode = 1    ' savefile mode
  endif
  if MousePort >= 0 then    ' make box bigger for Cancel/Accept buttons for mouse
    UB_vals(4) = UB_vals(4) + MIFH + 5 ' make room for buttons
  endif

  UB_vals(3) = p_width : if UB_vals(3) = 0 then UB_vals(3) = 32   ' width in characters
  UB_vals(5) = 46 + UB_vals(3) * MM.INFO(FONTWIDTH)              ' width in pixels
  ' location of top left corner of dialog box
  UB_vals(0) = p_x : if UB_vals(0) = 0 then UB_vals(0) = (MM.HRES - UB_vals(5))/2
  UB_vals(1) = p_y : if UB_vals(1) = 0 then UB_vals(1) = (MM.VRES - UB_vals(4))/2

  ' determine starting point in directory structure, set up history
  UB_startdir$ = cwd$      ' save starting directory
  for UB_k = 1 to DIRCOUNT ' set all elements to 1 - 1st item selected
    UB_hist$(UB_k) = "1,1"
  next UB_k
  UB_hist$(0) = "1"      ' initially at top directory level
  if UB_startdir$ <> "A:/" then  ' determine starting directory depth
    UB_startdir$ = UB_startdir$ + "/"
    for UB_k = 1 to len(UB_startdir$)
      if mid$(UB_startdir$,UB_k,1) = "/" then
        UB_hist$(0) = str$(val(UB_hist$(0)) + 1) ' another level deeper
      endif
    next UB_k
  endif

  MouseHide
  ' save underlying screen image in buffer #64
  blit read 64, UB_vals(0), UB_vals(1), UB_vals(5), UB_vals(4)
  ' draw dialog box
  rbox UB_vals(0)+7,UB_vals(1)+ 7,UB_vals(5)- 8,UB_vals(4)-8,10,UB_cols(3),UB_cols(3) ' drop shadow
  rbox UB_vals(0)  ,UB_vals(1)   ,UB_vals(5)- 8,UB_vals(4)-8,10,UB_cols(0),UB_cols(0) ' frame
  rbox UB_vals(0)+5,UB_vals(1)+22,UB_vals(5)-18,UB_vals(6),5,UB_cols(1), UB_cols(1) ' text area

  text UB_vals(0)+10,UB_vals(1)+6,UB_capt$, "LT", 1, 1, 0, -1   ' d-box title caption

  if UB_vals(8) then    ' multiple select?
    text UB_vals(0)+UB_vals(5)-12,UB_vals(1)+1,"2", "RT", 11, 1, 0, -1  ' Arrow/Ent/Esc/space
  else
    text UB_vals(0)+UB_vals(5)-12,UB_vals(1)+1,"1", "RT", 11, 1, 0, -1  ' Arrow/Ent/Esc
  endif

  if MousePort >= 0 then    ' draw Cancel/Accept buttons for mouse
    UB_tmp = UB_vals(1) + UB_vals(4) - MIFH*2 - 2  ' y location
    UI_ClearAllElements     ' start fresh
    UB_k = UI_CreateElement(1,UB_vals(0)+150,UB_tmp,"Cancel")
    UI_DrawElement UB_k
    UB_k = UI_CreateElement(2,UB_vals(0)+ 50,UB_tmp,"Select")
    UI_DrawElement UB_k
  endif

  '--------------------
  UB_ReadDir(UB_spec$,UB_top_item,UB_sel_item,UB_top_last)
  UI_DrawScrollBar UB_vals(0)+6,UB_vals(1)+23,UB_vals(2),1,val(UB_dirs$(0))+val(UB_fils$(0))
  UB_ListDir(UB_top_item, UB_vals(2), UB_sel_item)  ' populate the dialog box
  MouseShow

  do  ' user interation loop
    UB_k = asc(inkey$)
    UB_changed = 0

    ' synthesize up/down arrow key strokes if mouse wheel turned
    if mouse_d(2) < 0 then UB_k = 128 : mouse_d(2) = mouse_d(2) + 1
    if mouse_d(2) > 0 then UB_k = 129 : mouse_d(2) = mouse_d(2) - 1

    ' handle left-mouse-down event - move hilite if in list region
    if mouse_d(3) = MOUSE_L_DOWN then
      UB_tmp = UB_InDirListArea()
      if UB_tmp then
        if UB_tmp <= min(UB_vals(2),val(UB_dirs$(0)) + val(UB_fils$(0))) then ' not a blank line
          UB_sel_item = UB_tmp  ' move selection
          UB_changed = 1
        endif
      endif
      mouse_d(3) = 0  ' clear mouse-down event
      endif
    endif
    
    ' evaluate left-mouse-button-up on button
    if mouse_d(3) = MOUSE_L_UP then
      if UI_MouseOverElement() = 1 then UB_k = 27   ' [Cancel] button same as [ESC] key
      if UI_MouseOverElement() = 2 then   ' [Select] button
        if (UB_sel_item = 1) then ' can't select 1st item
          UB_k = 130  ' same as [LEFT] key  - dir up was selected
        else
          UB_k = 13   ' same as [ENTER] key
        endif
      endif
      mouse_d(3) = 0  ' clear the up event
    endif

    ' evaluate left-mouse-button double-click
    ' maybe enter directory or select file
    if mouse_d(3) = MOUSE_L_DOUBLE then
      UB_tmp = UB_InDirListArea()
      if UB_tmp then     ' OOOOO! Double-click on something in list - maybe.
        if UB_tmp <= min(UB_vals(2),val(UB_dirs$(0)) + val(UB_fils$(0))) then ' not a blank line
          ' clicked on something - dir or file?
          if (UB_tmp = 1) then  'top item
            UB_k = 130  ' same as [LEFT] key  - dir up was selected
          else
            UB_chosen = UB_top_item + UB_tmp - 1
            if UB_chosen <= val(UB_dirs$(0)) then ' item number in directory range?
              UB_k = 131  ' equivalent to [RIGHT] Arrow
            else
              UB_sel_item = UB_tmp
              UB_k = 13
            endif
          endif
        endif
      endif
      mouse_d(3) = 0  ' clear double-click event
    endif

    select case UB_k
      case  27  ' ESC
        result$(0) = ""  ' Cancel so return blank
        exit do
      case 128  ' UP arrow
        if UB_sel_item = 1 then  ' is the top item selected?
          if UB_top_item > 1 then  ' at top of list?
            UB_top_item = UB_top_item - 1  ' no so shift list up one
            UB_changed = 1
          endif
        else
          UB_sel_item = UB_sel_item - 1  ' shift selection up one
          UB_changed = 1
        endif
      case 129  ' DOWN arrow
        if UB_sel_item = UB_vals(2) then  ' is the bottom item selected?
          if UB_top_item < UB_top_last then  ' at bottom of list?
            UB_top_item = UB_top_item + 1  ' no so shift list down one
            UB_changed = 1
          endif
        else if UB_sel_item < val(UB_dirs$(0)) + val(UB_fils$(0)) then
          ' don't shift down past last item
          UB_sel_item = UB_sel_item + 1  ' shift selection down one
          UB_changed = 1
        endif
      case 130  ' LEFT Arrow - directory up if not root
        if cwd$ <> "A:/" then ' in a sub-directory?
          chdir ".."     'directory up chosen
          UB_ReadDir(UB_spec$,UB_top_item,UB_sel_item,UB_top_last)
          UB_hist$(0) = str$(val(UB_hist$(0)) - 1)
          UB_top_item = val(field$(UB_hist$(val(UB_hist$(0))),1,","))
          UB_sel_item = val(field$(UB_hist$(val(UB_hist$(0))),2,","))
          UB_changed = 1
        endif
      case 131  ' RIGHT Arrow - directory down if directory selected
        UB_chosen = UB_top_item + UB_sel_item - 1
        if UB_chosen <= val(UB_dirs$(0)) then ' item number in directory range?

          UB_hist$(val(UB_hist$(0))) = str$(UB_top_item) + "," + str$(UB_sel_item)
          'UB_hist$(UB_hist$(0)) = UB_chosen    ' save selection number if we come back up
          UB_hist$(0) = str$(val(UB_hist$(0)) + 1)

          if right$(cwd$,1) = "/" then   ' root level
            if UB_chosen > 1 then
              chdir cwd$ + UB_dirs$(UB_chosen)  ' tunnel down a directory from root
            endif
          else  ' not root level
            if UB_chosen > 1 then
              chdir cwd$ + "/" + UB_dirs$(UB_chosen)  ' tunnel down a directory
            else  ' first item is directory up
              chdir ".."     'directory up chosen
              UB_ReadDir(UB_spec$,UB_top_item,UB_sel_item,UB_top_last)
              UB_hist$(0) = str$(val(UB_hist$(0)) - 1)
              UB_top_item = val(field$(UB_hist$(val(UB_hist$(0))),1,","))
              UB_sel_item = val(field$(UB_hist$(val(UB_hist$(0))),2,","))
              UB_changed = 1
            endif
          endif
          UB_ReadDir(UB_spec$,UB_top_item,UB_sel_item,UB_top_last)
          UB_changed = 1
        endif
      case  32  ' [SPACE BAR] - multi-select / deselect
        if UB_vals(8) then  ' is multi-select mode on?
          UB_chosen = UB_top_item + UB_sel_item - 1
          if UB_chosen <= val(UB_dirs$(0)) then ' item number in directory range?
            if UB_vals(9) then  ' is multi-select directory mode on?
              ' toggle UB_msel$(UB_chosen)
            endif
          else    ' item is in file range

          endif
        ' toggle items on and off
        '    UB_msel$(0) = string$(255,0)      ' set all directory select flags to 0
        '    UB_msel$(1) = string$(255,0)      ' set all file select flags to 0

        endif
      case  13  ' ENTER - something has been selected
        UB_chosen = UB_top_item + UB_sel_item - 1
        if UB_chosen <= val(UB_dirs$(0)) then ' item number in directory range?
          if UB_vals(7) then  ' can a directory be selected?
            if right$(cwd$,1) = "/" then
              result$(0) = cwd$ + UB_dirs$(UB_chosen) + "/"  ' directory at root level
            else
              result$(0) = cwd$ + "/" + UB_dirs$(UB_chosen) + "/"   ' directory deeper
            endif     ' Note: cwd$ returns all uppercase
            exit do
          endif
        else    ' Yahoo! A filename has been chosen
          if not UB_vals(7) then   ' was other than directory selection chosen?
            UB_chosen = UB_chosen - val(UB_dirs$(0))
            if UB_fils$(UB_chosen) <> "" then  ' in case directory has no (specified) file
              if right$(cwd$,1) = "/" then
                result$(0) = cwd$ + UB_fils$(UB_chosen)  ' filename at root level
              else
                result$(0) = cwd$ + "/" + UB_fils$(UB_chosen)  ' filename deeper
              endif     ' Note: cwd$ returns all uppercase
              exit do
            endif
          endif
        endif
    end select
    if UB_changed then   ' something changed so redisplay directory list
      MouseHide
      UI_DrawScrollBar UB_vals(0)+6,UB_vals(1)+23,UB_vals(2),UB_top_item,val(UB_dirs$(0))+val(UB_fils$(0))
      UB_ListDir(UB_top_item, UB_vals(2), UB_sel_item)
      MouseShow
    endif
  loop
  '--------------------

  ' restore original screen image  (box not needed with  f/w v5.05.06+)
  MouseHide
  box UB_vals(0), UB_vals(1), UB_vals(5), UB_vals(4), 1, 0, 0 ' must clear to black first
  blit write 64, UB_vals(0), UB_vals(1) ', 0   ' now restore all non-black pixels
  blit close 64
  MouseShow

  ' restore starting directory
  chdir UB_startdir$

  do : loop until inkey$ = ""   ' clear the keyboard buffer
end sub

'*****************************************************************
' Read directories and specified files in the current directory
' spec$        identifies what type of file to read
' d_top_item,d_sel_item,d_top_last       returned values
sub UB_ReadDir(spec$,d_top_item,d_sel_item,d_top_last)
  local item_cnt, i

  for i = 1 to DIRCOUNT
    UB_dirs$(i) = ""   ' clear the array
  next i
  for i = 1 to FILCOUNT
    UB_fils$(i) = ""   ' clear the array
  next i

  ' read directories first
  UB_dirs$(0) = ""  ' 0 items to begin
  item_cnt = 2
  UB_dirs$(1) = "[..]"
  if (cwd$ = "A:/") then UB_dirs$(1) = "[ROOT]"    ' at root level?

  UB_dirs$(item_cnt) = left$(Dir$("*", DIR),NAMELENGTH) ' WARNING - possible truncation
  Do While UB_dirs$(item_cnt) <> "" and item_cnt < DIRCOUNT - 1
    If UB_dirs$(item_cnt) <> "." Then item_cnt = item_cnt + 1 ' ignore "."
    UB_dirs$(item_cnt) = Dir$()
  Loop
  if UB_dirs$(item_cnt) = "" then item_cnt = item_cnt - 1

  ' Sort only found directories, case insensitive, skip 1st item
  if item_cnt > 1 then Sort UB_dirs$(),,2,2,item_cnt-1
  UB_dirs$(0) = str$(item_cnt)   ' store number of items

  ' now read files
  UB_fils$(0) = ""  ' 0 items to begin
  item_cnt = 1
  if UB_vals(7) then  ' read all files if selecting only a dir
    UB_fils$(item_cnt) = left$(Dir$("*", FILE),NAMELENGTH) ' WARNING - possible truncation
  else
    UB_fils$(item_cnt) = left$(Dir$(spec$, FILE),NAMELENGTH) ' WARNING - possible truncation
  endif
  Do While UB_fils$(item_cnt) <> "" and item_cnt < FILCOUNT - 1
    If UB_fils$(item_cnt) <> "." Then item_cnt = item_cnt + 1 ' ignore "."
    UB_fils$(item_cnt) = Dir$()
  Loop
  if UB_fils$(item_cnt) = "" then item_cnt = item_cnt - 1

  ' Sort files and shift non-blank entries to front of array
  if item_cnt then Sort UB_fils$(),,2,1,item_cnt    ' sort only found items, case insensitive
  UB_fils$(0) = str$(item_cnt)   ' store number of items

  d_top_item = 1
  d_sel_item = 1
  d_top_last = val(UB_dirs$(0)) + val(UB_fils$(0)) - UB_vals(2) + 1

end sub

'*****************************************************************
' Display (part of) directory
' Show 'nlines' number of items, starting with item 'first',
' hilite given item
sub UB_ListDir(first, nlines, hilite)
  local integer i, item, x, y
  local d_txt$, pre$
  local MIFH = MM.INFO(FONTHEIGHT)

  if UB_vals(8) then
    pre$ = chr$(128) + chr$(129) ' empty box, checked box
  else
    pre$ = "  "
  endif
  x = UB_vals(0)+22
  y = UB_vals(1)+24

  for i = 0 to nlines - 1
    item = first + i
    if item > val(UB_dirs$(0)) then   ' is it a filename?
      d_txt$ = UB_fils$(item - val(UB_dirs$(0)))
    else    ' no, it's a directory
      if UB_vals(9) then  ' multi-select directory mode?
        if (item = 1) then ' special case for 1st item of root dir
          if UB_dirs$(1) = "[ROOT]" then d_txt$ = UB_dirs$(1)
        else
          d_txt$ = "<DIR> " + UB_dirs$(item)
        endif
      else
        d_txt$ = "<DIR> " + UB_dirs$(item)
      endif
    endif
    if len(d_txt$) > UB_vals(3) then d_txt$ = left$(d_txt$,UB_vals(3)-1) + chr$(148)
    d_txt$ = left$(d_txt$ + space$(UB_vals(3)),UB_vals(3))

    if d_txt$ = space$(UB_vals(3)),UB_vals(3)) then ' blank line
      text x,y+i*MIFH," ","LT",1,1,UB_cols(2),UB_cols(1)
      text x+10,y+i*MIFH,d_txt$,"LT",1,1,UB_cols(2),UB_cols(1)
    else
      if i = hilite - 1 then
        if UB_vals(9) then text x,y+i*MIFH,mid$(pre$,2,1),"LT",1,1,UB_cols(1),UB_cols(0)
        text x+10,y+i*MIFH,d_txt$,"LT",1,1,UB_cols(1),UB_cols(0)
      else
        text x,y+i*MIFH,mid$(pre$,1,1),"LT",1,1,UB_cols(2),UB_cols(1)
        text x+10,y+i*MIFH,d_txt$,"LT",1,1,UB_cols(2),UB_cols(1)
      endif
    endif
  next i
end sub

' test if mouse is in the directory listing region
' return text y position if inside, 0 otherwise
function UB_InDirListArea()
  local tmp
  UB_InDirListArea = 0
  if (mouse_d(0) > UB_vals(0)+31) and (mouse_d(0) < UB_vals(0)+UB_vals(5)-15) then
    tmp = mouse_d(1) - UB_vals(1) - MM.INFO(FONTHEIGHT)
    tmp = tmp \ MM.INFO(FONTHEIGHT)
    UB_InDirListArea = tmp
  endif
end function

'*****************************************************************
' Draw a scoll bar with the thumb in the appropriate place
' x,y = top left corner
' h = number of text lines high
' p = thumb position 1 = top, s-h = bottom
' s = size, total number of items in list
'   if s <= h then all are showing, no scroll needed.
'   ratio of s to h indicates vertical size of thumb
sub UI_DrawScrollBar(x,y,h,p,s)
  local ii,th
  local FH = MM.INFO(FONTHEIGHT)
  do : loop until getscanline > MM.VRES
  rbox x,y,15,h*FH+1,3,,0
  if s <= h then  ' entire list fits so no scroll bar needed - draw as grey
    for ii = 2 to 12 step 2
      line x+ii,y+ 1,x+ii,y+(h-0)*FH-1
    next ii
  else
    p = min(p,s-h+1)
    text x+3,y+1,chr$(146)
    text x+3,y+1+(h-1)*FH,chr$(147),,,,,-1
    for ii = 2 to 12 step 2
      line x+ii,y+FH,x+ii,y+(h-1)*FH
    next ii
    if h > 3 then ' skip scroll bar for really short dialog box
      th = max(1,(h-2)*h/s)
      rbox x+1,y+FH+(h-th-2)*(p-1)*FH/(s-h),13,th*FH,3,UB_cols(0),UB_cols(0)
    endif
  endif
end sub

' create a thing, return ID number, 0 if failed
' t  = type
' x  = x posn
' y  = y posn
' c$ = cation text
function UI_CreateElement(t,x,y,c$)
  local integer n
  if UI_elements(0,0) > 15 then ' no room for more
    UI_CreateElement = 0
    exit function
  else
    inc UI_elements(0,0)
    n = UI_elements(0,0)
    UI_elemtxt$(n) = c$     ' store caption string
    UI_elements(n,0) = t    ' type
    UI_elements(n,1) = x    ' location x
    UI_elements(n,2) = y    ' location y
    UI_elements(n,3) = MM.INFO(FONTWIDTH)*len(c$)+2    ' width
    UI_elements(n,4) = MM.INFO(FONTHEIGHT)+3    ' height
  endif
  UI_CreateElement = UI_elements(0,0)
end function

' draw specified element
sub UI_DrawElement n
  local tx,ty

  tx = UI_elements(n,1)
  ty = UI_elements(n,2)
  select case UI_elements(n,0)
    case 1    ' simple button - single outline
      rbox tx,ty,UI_elements(n,3),UI_elements(n,4),3,UB_cols(1)
      tx = tx + UI_elements(n,3)/2
      ty = ty + UI_elements(n,4)/2
      text tx,ty,UI_elemtxt$(n),"CM",,,UB_cols(1),-1
    case 2    ' default button - double outline
      rbox tx,ty,UI_elements(n,3),UI_elements(n,4),3,UB_cols(1)
      rbox tx-2,ty-2,UI_elements(n,3)+4,UI_elements(n,4)+4,4,UB_cols(1)
      tx = tx + UI_elements(n,3)/2
      ty = ty + UI_elements(n,4)/2
      text tx,ty,UI_elemtxt$(n),"CM",,,UB_cols(1),-1
    case 3
  end select
end sub

' return element number (or 0) under mouse
' mouse position is in mouse_d(0),mouse_d(1)
function UI_MouseOverElement()
  local n
  UI_MouseOverElement = 0
  for n = 1 to UI_elements(0,0)
    if mouse_d(0) < UI_elements(n,1) + UI_elements(n,3) then    ' x right
      if mouse_d(1) < UI_elements(n,2) + UI_elements(n,4) then  ' y bottom
        if mouse_d(0) > UI_elements(n,1) then     ' x left
          if mouse_d(1) > UI_elements(n,2) then   ' y top
            UI_MouseOverElement = n
            exit function
          endif
        endif
      endif
    endif
  next n
end function

' clear all elements
sub UI_ClearAllElements
  UI_elements(0,0) = 0
end sub

'*****************************************************************
' Mouse routines
' If MousePort >= 0 then use mouse
'
' MouseStart    - sets up and turns on the mouse, does not show it though
' MouseStop     - turns it all off again
' MouseHide     - moves mouse out of sight, use when changing the screen
' MouseShow     - make it visible again
'
sub MouseStart
  local i, F = RGB(NOTBLACK), B = UB_cols(0) 'RGB(YELLOW)
  local mousepointerdata(12*19)
  if MousePort >= 0 then
    restore MousePointerBitMap
    for i = 0 to 12*19 - 1 : read mousepointerdata(i) : next  ' read custom mouse pointer
    sprite loadarray 1,12,19,mousepointerdata()   ' sprite-ify pointer data
    sprite show 1,-1,MM.VRES-1,1    ' position mouse where it can't be seen
    controller mouse open MousePort
  endif
end sub

'*****************************************************************
sub MouseStop
  if MousePort >= 0 then
    settick 0, MouseUpdate  ' turn off interrupt
    controller mouse close
    sprite close 1
  endif
end sub

'*****************************************************************
' ISR to handle mouse activity
sub MouseUpdate
  local tx, ty
  if MousePort >= 0 then

    ' check for and handle mouse moved
    tx = mouse(x,MousePort)
    ty = mouse(y,MousePort)
    if mouse_d(0) <> tx or mouse_d(1) <> ty then
      mouse_d(0) = min(MM.HRES-2,tx)
      mouse_d(1) = min(MM.VRES-3,ty)
      sprite show 1,mouse_d(0),mouse_d(1),1
    endif

    ' record wheel movement
    mouse_d(2) = mouse(z,MousePort)   ' read the wheel

    ' handle left button down, up and double-click events
    tx = mouse(L,MousePort)     ' read left button
    ty = (tx xor mouse_d(5))    ' set if changed
    if ty then
      if tx then   ' new press
        mouse_d(3) = MOUSE_L_DOWN
      elseif mouse_d(5) then  ' new release
        mouse_d(3) = MOUSE_L_UP
        if timer - mouse_d(4) < DOUBLETIME then
          mouse_d(3) = MOUSE_L_DOUBLE
        endif
        mouse_d(4) = timer  ' save time of latest release
      endif
      mouse_d(5) = tx   ' save new mouse button state
    endif

  endif
end sub

'*****************************************************************
sub MouseHide
  if MousePort >= 0 then
    settick 0, MouseUpdate  ' stop the interrupt
    sprite show 1,-1,MM.VRES-1,1    ' position mouse where it can't be seen
  endif
end sub

'*****************************************************************
sub MouseShow
  if MousePort >= 0 then
    sprite show 1,mouse_d(0),mouse_d(1),1
    settick 25, MouseUpdate   ' (re)start interrupt
  endif
end sub

'*****************************************************************
' mouse pointer - because firmware version is hollow
MousePointerBitMap:
data F,0,0,0,0,0,0,0,0,0,0,0
data F,F,0,0,0,0,0,0,0,0,0,0
data F,B,F,0,0,0,0,0,0,0,0,0
data F,B,B,F,0,0,0,0,0,0,0,0
data F,B,B,B,F,0,0,0,0,0,0,0
data F,B,B,B,B,F,0,0,0,0,0,0
data F,B,B,B,B,B,F,0,0,0,0,0
data F,B,B,B,B,B,B,F,0,0,0,0
data F,B,B,B,B,B,B,B,F,0,0,0
data F,B,B,B,B,B,B,B,B,F,0,0
data F,B,B,B,B,B,B,B,B,B,F,0
data F,B,B,B,B,B,B,B,B,B,B,F
data F,B,B,B,B,B,B,F,F,F,F,F
data F,B,B,B,F,B,B,F,0,0,0,0
data F,B,B,F,0,F,B,B,F,0,0,0
data F,B,F,0,0,F,B,B,F,0,0,0
data F,F,0,0,0,0,F,B,B,F,0,0
data 0,0,0,0,0,0,F,B,B,F,0,0
data 0,0,0,0,0,0,0,F,F,0,0,0

'*****************************************************************
' two small gliphs showing arrow keys, tab/ent/space
DefineFont #11
  02311538
  00000000 3F000000 C1FF1FFF FF43F0FF FBE7FFB0 B7FF5FFC FCF1E7FE ACB7185F
  46FEE02F EF96B1F7 375FFEFB FBEFB6B7 B7D75FFE FEFBEFB6 B7B03842 3FEEFB6E
  CCFF1FFF 0000E6FF 0E080000 00000002 E6FF0C00 00000000 00EEFB0E 0F000000
  0000FEFB FB0F0000 000000FE FEFB0F00 00000000 00FEE00F 07000000 0000FCF1
  FB070000 000000FC F0FF0100 00000000 00000000 00000000 FF3F0000 FFC1FF1F
  B0FF43F0 FCFBE7FF FEB7FF5F 5FFCF1E7 2FACB718 F746FEE0 FBEF96B1 B7375FFE
  FEFBEFB6 B6B7D75F 42FEFBEF 6EB7B038 FF3FEEFB FFCCFF1F 000000E6 020E0800
  FFFFFF3F 7FE6FFCC EEFFFFFF 237EEEFB FBEF678C 6BED7DFE FEFBEFDB C36B6D7E
  7FFEFBEF EFDF4BAD 637CFEE0 F1E763AC FFEF7FFC FCFBE7FF FFFFEF3F 00F0FFC1
  00000000 00000000
End DefineFont
'*****************************************************************
