mode 1
autosave
'=============================================================================|
' PicoRocks in Space - Teaser
'
' written by Vegipete, Jan 2024
' Ported from CMM2
'
' needs:
' OPTION CPUSPEED 252000 (KHz)
'

Option DEFAULT INTEGER
Option LCDPANEL NOCONSOLE
option angle degrees
MODE 2
cls 0

const MAXROCKS = 50
dim float rock(MAXROCKS,5)  ' 50 rocks: scale,angle,x,y,vx,vy
dim trock (MAXROCKS)  ' type of rock
math set 0, rock()  ' erase them all to start
math set 0,trock()

' rock vector shapes
' Note: any resemblance to a country is purely coincidental
dim r1x(14)=( 0, 3, 4, 3, 5, 4, 1,-2,-4,-5,-3,-5,-3,-2, 0)
dim r1y(14)=( 5, 4, 3, 1, 0,-4,-5,-4,-4,-1,-1, 2, 3, 5, 5)
dim r2x(14)=( 0,-2,-3,-4,-5,-4,-4,-3,-1, 4, 4, 5, 4, 2, 0)
dim r2y(14)=(-6,-4,-3,-4,-1, 2, 3, 3, 5, 3, 1, 0,-3,-4,-6)
dim r3x(14)=( 1,-2,-2,-4,-4,-5,-5,-3,-2, 0, 4, 3, 5, 3, 1)
dim r3y(14)=(-4,-5,-3,-4,-2, 0, 3, 4, 3, 5, 3, 2, 1,-5,-4)
dim r4x(14)=( 0,-2,-4,-4,-5,-3,-1, 2, 1, 3, 5, 3, 4, 3, 0)
dim r4y(14)=(-5,-4,-4,-1, 1, 4, 5, 5, 3, 4, 1,-2,-3,-4,-5)

FRAMEBUFFER Create   ' hidden layer to draw scene
FRAMEBUFFER Write F : CLS 0

Const SPRADDR = Peek(cfunaddr Ship)
dim float pv,px,py,pvx,pvy

ph = 0 : pv = 0
px = MM.HRES/2-6 : pvx = 0
py = MM.VRES/2-6 : pvy = 0

' create some random rocks
k = 1
for i = 1 to 10
  trock(i)  = 1 + (k and 3) ' rock type
  rock(i,0) = 1+int(rnd*3)  ' rock size
  rock(i,2) = rnd*MM.HRES   ' x location
  rock(i,3) = rnd*MM.VRES   ' y location
  rock(i,4) = 1-rnd*2   ' x velocity
  rock(i,5) = 1-rnd*2   ' y velocity
  inc k
next

mot = timer

do
  k = asc(inkey$)
  if k then
    burn = 0
    select case k
      Case 128  ' up
        inc pvx, sin(ph*5)/2
        inc pvy,-cos(ph*5)/2
        mot = timer + 100   ' keep flame visible for a while
      Case 130  ' left
        inc ph,-1
        if ph < 0 then ph = 71
      Case 131  ' right
        inc ph
        if ph > 71 then ph = 0
    end select
  endif
  
  inc px,pvx
  if px > MM.HRES then
    inc px, -(MM.HRES+6)
  elseif px < 5 then
    inc px, MM.HRES+6
  endif
  
  inc py,pvy
  if py > MM.VRES then
    inc py, -(MM.VRES+6)
  elseif py < 0 then
    inc py, MM.VRES+6
  endif
  
  ' velocity decays away slowly
  pvx = pvx * .995
  pvy = pvy * .995

  ' move rocks
  for i = 0 to MAXROCKS
    if trock(i) then   ' rock exists
      inc rock(i,2),rock(i,4)
      if rock(i,2) > MM.HRES then
        inc rock(i,2), -MM.HRES
      elseif rock(i,2) < 0 then
        inc rock(i,2), MM.HRES
      endif
      
      inc rock(i,3),rock(i,5)
      if rock(i,3) > MM.VRES then
        inc rock(i,3), -MM.VRES
      elseif rock(i,3) < 0 then
        inc rock(i,3), MM.VRES
      endif
    endif
  next
  
  off = (mot > timer) * 6624
  cls
  Sprite memory SPRADDR+off+ph*92,px,py
  DrawRocks
  FRAMEBUFFER copy F,N,B
loop

' Go through the list of rocks and draw any that exist
sub DrawRocks
  local i
  local xb(14),yb(14)
  
  for i = 0 to MAXROCKS
    if trock(i) then   ' rock exists
      select case trock(i) ' what shape of rock is it?
        case 1
          math scale r1x(),rock(i,0),xb()
          math scale r1y(),rock(i,0),yb()
        case 2
          math scale r2x(),rock(i,0),xb()
          math scale r2y(),rock(i,0),yb()
        case 3
          math scale r3x(),rock(i,0),xb()
          math scale r3y(),rock(i,0),yb()
        case 4
          math scale r4x(),rock(i,0),xb()
          math scale r4y(),rock(i,0),yb()
      end select
    endif
    math add xb(),rock(i,2),xb()  ' position rock
    math add yb(),rock(i,3),yb()
    polygon 15,xb(),yb()
  next i
end sub

' 72 ships per rotation, 92 bytes each
'  0- 71 no thrust
' 72-143 engine on
CSub Ship
00000000
000D000D 06000000 00000000 00006000 60000000 00000060 06060000 00000000
00006060 00600000 00000006 06000600 60000000 00006000 06060000 00000006 
06660666 00600000 00006000 00000000 00000000 00000000 00000000 000D000D 
60000000 00000000 00060000 00000000 00000066 60600000 00000000 00060060 
06000000 00000060 06006000 60000000 00006000 06060000 00000060 06660666 
00600000 00006000 00000000 00000000 00000000 00000000 000D000D 60000000 
00000000 00066000 00000000 00000066 06060000 00000000 00006060 00600000 
00000006 06000600 06000000 00006000 06666000 60000006 00660006 00000000 
00000600 00000000 00000000 00000000 00000000 000D000D 00000000 00000006 
00660000 00000000 00000660 60600000 00000000 00060060 06000000 00000060 
06000600 06000000 00006000 00666000 60000006 00606006 00000000 00000660 
60000000 00000000 00000000 00000000 000D000D 00000000 00000006 00660000 
00000000 00000660 60600000 00000000 00060060 06000000 00000060 00600600 
06000000 00000600 60666000 60000000 00066006 00000000 00000066 06000000 
00000000 00000000 00000000 000D000D 00000000 00000000 06000000 00000000 
00006600 06000000 00000006 00060600 06000000 00000060 06000600 06000000 
00006000 00666600 00000006 00066000 00000000 00000066 06000000 00000000 
00000000 00000000 000D000D 00000000 00000000 06000000 00000000 00006600 
06000000 00000006 00060600 06600000 00000060 06000060 00600000 00006000 
00666660 00000006 00066000 00000000 00000066 06000000 00000000 00000000 
00000000 000D000D 00000000 00000000 06000000 00000000 00006600 66000000 
00000000 00060600 06600000 00000060 06000060 00600000 00000600 60666660 
00000000 00060600 00000000 00000006 00600000 00000000 00000000 00000000 
000D000D 00000000 00000000 60000000 00000000 00066600 06000000 00000006 
00600660 00600000 00000060 06000066 00660000 00006000 60666000 00000000 
00060600 00000000 00000006 00600000 00000000 00000000 00000000 000D000D 
00000000 00000000 00000000 00000000 00060000 60000000 00000006 00606600 
06600000 00000060 06000066 66660000 00000606 60600000 00000000 00006600 
60000000 00000006 00060000 00000000 00000000 00000000 000D000D 00000000 
00000000 00000000 00000000 00660000 60000000 00000066 06006660 00660000 
60000600 06000006 66000000 00006000 60060000 00000000 00006600 60000000 
00000006 00060000 00000000 00000000 00000000 000D000D 00000000 00000000 
00000000 00000000 00000000 00000000 00000660 06666000 66660000 60000600 
06000006 66000000 00006000 60060000 00000000 00006060 66000000 00000000 
00006000 00000000 00000000 00000000 000D000D 00000000 00000000 00000000 
00000000 00000000 00000000 00000666 06066666 00066600 00000600 06000066 
60000000 00006000 60060000 00000000 00006060 66000000 00000000 00006000 
00000000 00000000 00000000 000D000D 00000000 00000000 00000000 00000000 
00000000 00000000 00000666 06066666 00066600 00000600 06000066 60000000 
00000600 06060000 00000000 00000660 06000000 00000000 00000000 00000000 
00000000 00000000 000D000D 00000000 00000000 00000000 00000000 00000000 
00000000 00000000 66666000 66666606 00066000 60000066 60000000 00006600 
06060000 00000000 00000666 00600000 00000000 00000000 00000000 00000000 
00000000 000D000D 00000000 00000000 00000000 00000000 00000000 00000000 
66600000 66666666 00006006 00066000 60000060 60000000 00006600 06060000 
00000000 00000666 00600000 00000000 00000000 00000000 00000000 00000000 
000D000D 00000000 00000000 00000000 00000000 00000000 00000000 66000000 
00006666 00060000 00666660 00000600 60000066 00066000 66006000 00000000 
00000666 00600000 00000000 00000000 00000000 00000000 00000000 000D000D 
00000000 00000000 00000000 00000000 00000000 00006660 60000000 00666660 
00060000 00666000 60000600 06000006 00006660 00666000 60000000 00000006 
00000000 00000000 00000000 00000000 00000000 00000000 000D000D 00000000 
00000000 00000000 00000000 00000000 00000660 60000000 00006666 00060000 
00006660 00000600 06000660 00666600 06666000 60000000 00000006 00000000 
00000000 00000000 00000000 00000000 00000000 000D000D 00000000 00000000 
00000000 00000000 00000000 00000660 60000000 00000066 60060000 00000066 
60000600 06000006 66600000 66606000 60000066 00000066 00000000 00000000 
00000000 00000000 00000000 00000000 000D000D 00000000 00000000 00000000 
00600000 00000000 00066600 00000000 00006606 00600000 00000660 00000600 
06000066 66666600 06666600 00000000 00000000 00000000 00000000 00000000 
00000000 00000000 00000000 000D000D 00000000 00000000 00000000 00600000 
00000000 00066600 00000000 00000606 00600000 00000066 60000060 00600000 
06600000 66666660 00066666 00000000 00000000 00000000 00000000 00000000 
00000000 00000000 000D000D 00000000 00000000 00000000 00600000 00000000 
00066600 00000000 00000606 00600000 00000066 60000066 66660000 06600066 
60000000 00066666 00000000 00000000 00000000 00000000 00000000 00000000 
00000000 000D000D 00000000 00000000 00000000 06000000 00000000 00066000 
00000000 00000606 00600000 00000006 06000066 06660000 00060000 66660000 
00000606 66000000 00000006 00000000 00000000 00000000 00000000 00000000 
000D000D 00000000 00000000 00000060 66000000 00000000 00606000 00000000 
00006006 00600000 00000006 06000066 06660000 00060000 66660000 00000606 
66000000 00000006 00000000 00000000 00000000 00000000 00000000 000D000D 
00000000 00000000 00000060 66000000 00000000 00606000 00000000 00006006 
00660000 60000006 06000006 66000000 00060006 66000000 00000666 60000000 
00000006 00000000 00000000 00000000 00000000 00000000 000D000D 00000000 
00000000 00000600 60000000 00000006 00606000 00000000 00006006 00660000 
60000060 06000006 66000000 00060000 66600000 00000600 66600000 00000000 
00660000 00000000 00000000 00000000 00000000 000D000D 00000000 00000000 
00000600 60000000 00000006 00660000 00000000 00006060 06666600 00000006 
06000066 60000000 00006006 66000000 00000060 06600000 00000000 00060000 
00000000 00000000 00000000 00000000 000D000D 00000000 00000000 00006000 
00000000 00000006 06060000 00000000 00006066 00066600 00000060 06000066 
60000000 00006000 06600000 00000060 06060000 00000000 00066600 00000000 
00000060 00000000 00000000 000D000D 00000000 00000000 00006000 00000000 
00000006 06060000 66600000 00006066 00006000 00000006 00600060 60000000 
00006000 06600000 00000006 00660000 00000000 00006600 00000000 00000006 
00000000 00000000 000D000D 00000000 00000000 00060000 00000000 00000066 
06600000 66600000 00060066 00006000 00000060 06000060 60000000 00006000 
06600000 00000006 06060000 00000000 00006600 00000000 00000006 00000000 
00000000 000D000D 00000000 00000000 00060000 00000000 00000066 06600000 
66000000 00060066 00060000 00000060 06000600 00000000 00006006 06000000 
00000006 06060000 00000000 00006600 00000000 00000006 00000000 00000000 
000D000D 00000000 00000000 00060000 00000000 00000066 06600660 60000000 
00006066 00060000 00000006 00600600 00000000 00006006 00600000 00000006 
00606000 00000000 00000660 66000000 00000000 00060000 00000000 000D000D 
00000000 00000000 00600000 00000000 00000660 60600660 60000000 00060066 
00060000 00000060 06000600 00000000 00006006 00600000 00000006 00606000 
00000000 00000660 66000000 00000000 00060000 00000000 000D000D 00000000 
00000000 00000000 00000000 00000600 66006660 60000000 00060660 00060000 
00000060 06000600 60000000 00006000 60600000 00000000 00060600 00000000 
00000066 06600000 00000000 00006000 00000000 000D000D 00000000 00000000 
00000000 00600000 00006000 66066600 00000006 00600606 00600000 00000060 
06006000 00000000 00006006 00600000 00000006 00606000 00000000 00000066 
06000000 00000000 00006000 00000000 000D000D 00000000 00000000 00000000 
00600000 00006000 66066600 00000006 00060606 00600000 00000060 06000600 
00000000 00006006 60600000 00000000 00060600 60000000 00000060 00600000 
00000000 00000600 00000000 000D000D 00000000 00000000 00000000 00600000 
00006000 66066600 60000006 00060600 00600000 00000060 00600600 60000000 
00000600 60060000 00000000 00006060 60000000 00000006 00060000 00000000 
00000060 00000000 000D000D 00000000 00000000 00000000 06000000 00000000 
00066000 00000066 00666606 00600000 00000600 06000600 00000000 00006006 
60600000 00000000 00060600 60000000 00000006 00660000 00000000 00000060 
00000000 000D000D 00000000 00000000 00000060 66000000 00000000 00606000 
00000066 00666006 00600000 00000600 06000600 60000000 00000600 60060000 
00000000 00006060 66000000 00000000 00066000 00000000 00000006 00000000 
000D000D 00000000 00000000 00000600 60000000 00000006 00660000 00000066 
00666060 06000000 00000600 06006000 60000000 00000600 60060000 00000000 
00006060 66000000 00000000 00066000 00000000 00000006 00000000 000D000D 
00000000 00000000 00000600 60000000 00000006 00660000 00000000 06666006 
00600000 00000600 06000600 60000000 00000600 06060000 00000000 00000606 
06600000 00000000 00000600 00000000 00000000 00000000 000D000D 00000000 
00000000 00000600 60000000 00000006 00660000 00000000 66666006 00600000 
00006000 60000600 60000000 00006600 06060000 00000000 00000606 06600000 
00000000 00000600 00000000 00000000 00000000 000D000D 00000000 00000000 
00006000 00000000 00000006 06060000 00000000 66666060 06000000 00006000 
60000600 60000000 00006600 06060000 00000000 00000660 06600000 00000000 
00000600 00000000 00000000 00000000 000D000D 00000000 00000000 00006000 
00000000 00000006 06060000 00000000 00666060 00600000 00066000 60000600 
60000006 00006000 66006000 00000000 00000606 06660000 00000000 00000060 
00000000 00000000 00000000 000D000D 00000000 00000000 00060000 00000000 
00000066 06600000 00000000 00006060 06000000 00066666 60000600 60000006 
00006600 06606000 00000000 00000066 00060000 00000000 00000000 00000000 
00000000 00000000 000D000D 00000000 00000000 00060000 00000000 00000066 
06600000 00000000 00060060 00600000 00000660 00000600 06000066 00066000 
66600600 60000000 00000066 00066000 00000000 00000000 00000000 00000000 
00000000 000D000D 00000000 00000000 00600000 00000000 00000660 60600000 
00000000 00060060 00600000 00000660 00000600 06000066 00066660 00666600 
66000000 00000000 00000000 00000000 00000000 00000000 00000000 00000000 
000D000D 00000000 00000000 00600000 00000000 00000660 60600000 00000000 
00060060 00600000 00000060 60000600 06000006 06660000 66660600 66000006 
00000006 00000000 00000000 00000000 00000000 00000000 00000000 000D000D 
00000000 00000000 00000000 00000000 00000600 66000000 00000000 00060600 
06000000 00000060 60000600 06000006 06660000 66660600 66000006 00000006 
00000000 00000000 00000000 00000000 00000000 00000000 000D000D 00000000 
00000000 00000000 00000000 00006000 66000000 00000006 00060600 06600000 
00000060 60000060 00660006 06666660 00666666 00000000 00000000 00000000 
00000000 00000000 00000000 00000000 00000000 000D000D 00000000 00000000 
00000000 00000000 00006000 66000000 00000006 00060600 06600000 00000060 
60000060 00660000 00600000 66666666 00006666 00000000 00000000 00000000 
00000000 00000000 00000000 00000000 000D000D 00000000 00000000 00000000 
00000000 00006000 66000000 00000006 00600660 00660000 60000060 06000006 
66666000 00060000 66600000 00000666 00000000 00000000 00000000 00000000 
00000000 00000000 00000000 000D000D 00000000 00000000 00000000 00000000 
00000000 00000000 00000066 00666000 66600000 00000600 06000066 00666000 
00060000 66666000 00000060 66600000 00000000 00000000 00000000 00000000 
00000000 00000000 000D000D 00000000 00000000 00000000 00000000 00000000 
00000000 00000066 00666600 06666000 66000600 06000000 66600000 00060000 
66600000 00000066 66000000 00000000 00000000 00000000 00000000 00000000 
00000000 000D000D 00000000 00000000 00000000 00000000 00000000 60000000 
60000066 00606666 00006660 00000600 06000066 60000000 00060066 60000000 
00000066 66000000 00000000 00000000 00000000 00000000 00000000 00000000 
000D000D 00000000 00000000 00000000 00000000 00000000 00000000 00000000 
06666600 06666660 60000600 06000006 66000000 00006000 06600000 00000006 
06660000 00000000 00006000 00000000 00000000 00000000 00000000 000D000D 
00000000 00000000 00000000 00000000 00000000 00000000 66660000 66666666 
00006600 00006000 60000060 60000000 00006006 06000000 00000006 06660000 
00000000 00006000 00000000 00000000 00000000 00000000 000D000D 00000000 
00000000 00000000 00000000 00000000 00000000 66660000 00000066 60006600 
00066666 60000060 60000006 00006006 06000000 00000006 06660000 00000000 
00006000 00000000 00000000 00000000 00000000 000D000D 00000000 00000000 
00000000 00000000 00000000 00000666 06000000 00066666 00060000 00066600 
60000600 00000006 00006006 06000000 00000006 00660000 00000000 00000600 
00000000 00000000 00000000 00000000 000D000D 00000000 00000000 00000000 
00000000 00000000 00000666 06000000 00066666 00060000 00066600 60000600 
00000006 00006006 00600000 00000006 00606000 00000000 00000660 60000000 
00000000 00000000 00000000 000D000D 00000000 00000000 00000000 00000000 
00000000 00000066 66000000 00000666 00060000 00000666 00000600 00000066 
00066006 00600000 00000006 00606000 00000000 00000660 60000000 00000000 
00000000 00000000 000D000D 00000000 00000000 00000000 00066000 00000000 
00006660 06000000 00006660 00060000 00000660 00000600 60000066 00066000 
00600000 00000006 00606000 00000000 00000066 06000000 00000000 00000000 
00000000 000D000D 00000000 00000000 00000000 00060000 00000000 00006600 
60000000 00000660 00600000 00000066 60000600 00000006 06666606 60600000 
00000000 00066000 00000000 00000066 06000000 00000000 00000000 00000000 
000D000D 00000000 60000000 00000000 06660000 00000000 00060600 60000000 
00006600 00600000 00000060 60000600 60000006 06660000 60600000 00000006 
00060600 00000000 00000006 00600000 00000000 00000000 00000000 000D000D 
00000000 00000000 00000006 06600000 00000000 00066000 00000000 00006606 
00600000 00000060 60006000 00000000 00600006 60600000 00006666 00060600 
00000000 00000006 00600000 00000000 00000000 00000000 000D000D 00000000 
00000000 00000006 06600000 00000000 00060600 00000000 00006606 00600000 
00000060 60000600 60000000 00600000 60060000 00006666 00006600 60000000 
00000006 00060000 00000000 00000000 00000000 000D000D 00000000 00000000 
00000006 06600000 00000000 00060600 00000000 00000606 00600000 00000006 
06000600 60000000 00060000 60060000 00000666 00006600 60000000 00000006 
00060000 00000000 00000000 00000000 000D000D 00060000 00000000 00000660 
66000000 00000000 00606000 00000000 00006006 00600000 00000006 06006000 
00000000 00060006 60600000 00000066 66006600 60000000 00000006 00060000 
00000000 00000000 00000000 000D000D 00060000 00000000 00000660 66000000 
00000000 00606000 00000000 00006006 00600000 00000006 06000600 60000000 
00060000 60060000 00000066 66006060 66000000 00000000 00006000 00000000 
00000000 00000000 000D000D 00600000 00000000 00006600 60000000 00000006 
06060000 00000000 00006060 00600000 00000060 06000600 60000000 00060000 
66060000 00000060 66600660 06000000 00000000 00000000 00000000 00000000 
00000000 000D000D 00600000 00000000 00000600 60000000 00000006 00606000 
00000000 00006006 00600000 00000006 00600600 60000000 00006000 06006000 
00000006 06660666 00600000 00006000 00000000 00000000 00000000 00000000 
000D000D 06000000 00000000 00006000 60000000 00000060 06060000 00000000 
00006060 00600000 00000006 06000600 60000000 00006000 06060000 00000006 
06660666 00600000 0000600E 0ACA0000 00000000 00000A00 00000000 000D000D 
60000000 00000000 00060000 00000000 00000066 60600000 00000000 00060060 
06000000 00000060 06006000 60000000 00006000 06060000 00000060 06660666 
00600000 0000600E 0ACA0000 00000000 000000A0 00000000 000D000D 60000000 
00000000 00066000 00000000 00000066 06060000 00000000 00006060 00600000 
00000006 06000600 06000000 00006000 06666000 60000006 00660006 E0000000 
00000600 00ACA000 00000000 000000A0 00000000 000D000D 00000000 00000006 
00660000 00000000 00000660 60600000 00000000 00060060 06000000 00000060 
06000600 06000000 00006000 00666000 60000006 00606006 E0000000 00000660 
60ACA000 00000000 0000000A 00000000 000D000D 00000000 00000006 00660000 
00000000 00000660 60600000 00000000 00060060 06000000 00000060 00600600 
06000000 00000600 60666000 60000000 00066006 E0A00000 00000066 060AC000 
00000000 0000000A 00000000 000D000D 00000000 00000000 06000000 00000000 
00006600 06000000 00000006 00060600 06000000 00000060 06000600 06000000 
00006000 00666600 00000006 00066000 0EA00000 00000066 060ACA00 00000000 
00000000 00000000 000D000D 00000000 00000000 06000000 00000000 00006600 
06000000 00000006 00060600 06600000 00000060 06000060 00600000 00006000 
00666660 00000006 000660E0 0CA00000 00000066 0600AA00 00000000 00000000 
00000000 000D000D 00000000 00000000 06000000 00000000 00006600 66000000 
00000000 00060600 06600000 00000060 06000060 00600000 00000600 60666660 
00000000 000606E0 00CA0000 00000006 0060AA00 00000000 00000000 00000000 
000D000D 00000000 00000000 60000000 00000000 00066600 06000000 00000006 
00600660 00600000 00000060 06000066 00660000 00006000 60666000 A0000000 
0006060E 0AC00000 00000006 006000A0 00000000 00000000 00000000 000D000D 
00000000 00000000 00000000 00000000 00060000 60000000 00000006 00606600 
06600000 00000060 06000066 66660000 00000606 60600000 A0000000 0000660E 
60AA0000 00000006 00060000 00000000 00000000 00000000 000D000D 00000000 
00000000 00000000 00000000 00660000 60000000 00000066 06006660 00660000 
60000600 06000006 66000000 00006000 60060A00 C0000000 0000660E 60A0A000 
00000006 00060000 00000000 00000000 00000000 000D000D 00000000 00000000 
00000000 00000000 00000000 00000000 00000660 06666000 66660000 60000600 
06000006 66000000 00006000 6006E0A0 CA000000 00006060 660A0000 00000000 
00006000 00000000 00000000 00000000 000D000D 00000000 00000000 00000000 
00000000 00000000 00000000 00000666 06066666 00066600 00000600 06000066 
60000000 00006000 6006E0A0 AA000000 00006060 66000000 00000000 00006000 
00000000 00000000 00000000 000D000D 00000000 00000000 00000000 00000000 
00000000 00000000 00000666 06066666 00066600 00000600 06000066 600A0000 
00000600 06060EC0 AA000000 00000660 06000000 00000000 00000000 00000000 
00000000 00000000 000D000D 00000000 00000000 00000000 00000000 00000000 
00000000 00000000 66666000 66666606 00066000 60000066 60EA0000 00006600 
060600CA A0000000 00000666 00600000 00000000 00000000 00000000 00000000 
00000000 000D000D 00000000 00000000 00000000 00000000 00000000 00000000 
66600000 66666666 00006006 A0066000 60000060 60EC0000 00006600 060600AA 
00000000 00000666 00600000 00000000 00000000 00000000 00000000 00000000 
000D000D 00000000 00000000 00000000 00000000 00000000 00000000 66000000 
00006666 00060000 A0666660 00000600 60ECA066 00066000 660060A0 00000000 
00000666 00600000 00000000 00000000 00000000 00000000 00000000 000D000D 
00000000 00000000 00000000 00000000 00000000 00006660 60000000 00666660 
00060A00 C0666000 6000060E 060AA006 00006660 00666000 60000000 00000006 
00000000 00000000 00000000 00000000 00000000 00000000 000D000D 00000000 
00000000 00000000 00000000 00000000 00000660 60000000 00006666 00060A00 
CA006660 0000060E 060A0660 00666600 06666000 60000000 00000006 00000000 
00000000 00000000 00000000 00000000 00000000 000D000D 00000000 00000000 
00000000 00000000 00000000 00000660 60000000 00000066 60060AA0 C0000066 
6000060E 060A0006 66600000 66606000 60000066 00000066 00000000 00000000 
00000000 00000000 00000000 00000000 000D000D 00000000 00000000 00000000 
00600000 00000000 00066600 00A00000 00006606 0060ECA0 A0000660 00000600 
06000066 66666600 06666600 00000000 00000000 00000000 00000000 00000000 
00000000 00000000 00000000 000D000D 00000000 00000000 00000000 00600000 
00000000 00066600 00AA0000 00000606 0060EC00 A0000066 60000060 00600000 
06600000 66666660 00066666 00000000 00000000 00000000 00000000 00000000 
00000000 00000000 000D000D 00000000 00000000 00000000 00600000 00000000 
000666A0 00CA0000 00000606 0060EA00 00000066 60000066 66660000 06600066 
60000000 00066666 00000000 00000000 00000000 00000000 00000000 00000000 
00000000 000D000D 00000000 00000000 00000000 06000000 00000000 000660AA 
0EC00000 00000606 00600A00 00000006 06000066 06660000 00060000 66660000 
00000606 66000000 00000006 00000000 00000000 00000000 00000000 00000000 
000D000D 00000000 00000000 00000060 66000000 00000000 006060AA ECA00000 
00006006 00600000 00000006 06000066 06660000 00060000 66660000 00000606 
66000000 00000006 00000000 00000000 00000000 00000000 00000000 000D000D 
00000000 00000000 00000060 660A0000 00000000 006060CA E0A00000 00006006 
00660000 60000006 06000006 66000000 00060006 66000000 00000666 60000000 
00000006 00000000 00000000 00000000 00000000 00000000 000D000D 00000000 
00000000 00000600 60A0A000 00000006 00606EC0 0A000000 00006006 00660000 
60000060 06000006 66000000 00060000 66600000 00000600 66600000 00000000 
00660000 00000000 00000000 00000000 00000000 000D000D 00000000 00000000 
00000600 60AA0000 00000006 00660EA0 00000000 00006060 06666600 00000006 
06000066 60000000 00006006 66000000 00000060 06600000 00000000 00060000 
00000000 00000000 00000000 00000000 000D000D 00000000 A0000000 00006000 
0AC00000 00000006 06060EA0 00000000 00006066 00066600 00000060 06000066 
60000000 00006000 06600000 00000060 06060000 00000000 00066600 00000000 
00000060 00000000 00000000 000D000D 00000000 00000000 000060AA 00CA0000 
00000006 0606E000 66600000 00006066 00006000 00000006 00600060 60000000 
00006000 06600000 00000006 00660000 00000000 00006600 00000000 00000006 
00000000 00000000 000D000D 00000000 00000000 000600AA 0CA00000 00000066 
0660E000 66600000 00060066 00006000 00000060 06000060 60000000 00006000 
06600000 00000006 06060000 00000000 00006600 00000000 00000006 00000000 
00000000 000D000D 00000000 00000000 00060ACA 0EA00000 00000066 06600000 
66000000 00060066 00060000 00000060 06000600 00000000 00006006 06000000 
00000006 06060000 00000000 00006600 00000000 00000006 00000000 00000000 
000D000D 000A0000 00000000 00060AC0 E0A00000 00000066 06600660 60000000 
00006066 00060000 00000006 00600600 00000000 00006006 00600000 00000006 
00606000 00000000 00000660 66000000 00000000 00060000 00000000 000D000D 
000A0000 00000000 0060ACA0 E0000000 00000660 60600660 60000000 00060066 
00060000 00000060 06000600 00000000 00006006 00600000 00000006 00606000 
00000000 00000660 66000000 00000000 00060000 00000000 000D000D 00A00000 
00000000 0000ACA0 E0000000 00000600 66006660 60000000 00060660 00060000 
00000060 06000600 60000000 00006000 60600000 00000000 00060600 00000000 
00000066 06600000 00000000 00006000 00000000 000D000D 00A00000 00000000 
000ACA00 00600000 0000600E 66066600 00000006 00600606 00600000 00000060 
06006000 00000000 00006006 00600000 00000006 00606000 00000000 00000066 
06000000 00000000 00006000 00000000 000D000D 0A000000 00000000 000ACA00 
00600000 0000600E 66066600 00000006 00060606 00600000 00000060 06000600 
00000000 00006006 60600000 00000000 00060600 60000000 00000060 00600000 
00000000 00000600 00000000 000D000D A0000000 00000000 000ACA00 00600000 
0000600E 66066600 60000006 00060600 00600000 00000060 00600600 60000000 
00000600 60060000 00000000 00006060 60000000 00000006 00060000 00000000 
00000060 00000000 000D000D A0000000 00000000 00ACA000 06000000 000000E0 
00066000 00000066 00666606 00600000 00000600 06000600 00000000 00006006 
60600000 00000000 00060600 60000000 00000006 00660000 00000000 00000060 
00000000 000D000D 00000000 0000000A 00ACA060 66000000 000000E0 00606000 
00000066 00666006 00600000 00000600 06000600 60000000 00000600 60060000 
00000000 00006060 66000000 00000000 00066000 00000000 00000006 00000000 
000D000D 00000000 0000000A 00CA0600 60000000 0000A0E6 00660000 00000066 
00666060 06000000 00000600 06006000 60000000 00000600 60060000 00000000 
00006060 66000000 00000000 00066000 00000000 00000006 00000000 000D000D 
00000000 00000000 0ACA0600 60000000 0000AE06 00660000 00000000 06666006 
00600000 00000600 06000600 60000000 00000600 06060000 00000000 00000606 
06600000 00000000 00000600 00000000 00000000 00000000 000D000D 00000000 
00000000 0AA00600 60000000 0000A006 E0660000 00000000 66666006 00600000 
00006000 60000600 60000000 00006600 06060000 00000000 00000606 06600000 
00000000 00000600 00000000 00000000 00000000 000D000D 00000000 00000000 
0AA06000 00000000 000AC006 E6060000 00000000 66666060 06000000 00006000 
60000600 60000000 00006600 06060000 00000000 00000660 06600000 00000000 
00000600 00000000 00000000 00000000 000D000D 00000000 00000000 A0006000 
00000000 0000CA06 06060000 000000AE 00666060 00600000 00066000 60000600 
60000006 00006000 66006000 00000000 00000606 06660000 00000000 00000060 
00000000 00000000 00000000 000D000D 00000000 00000000 00060000 00000000 
000AA066 06600000 000000AE 00006060 06000000 00066666 60000600 60000006 
00006600 06606000 00000000 00000066 00060000 00000000 00000000 00000000 
00000000 00000000 000D000D 00000000 00000000 00060000 00000000 00A0A066 
06600000 000000CE 0A060060 00600000 00000660 00000600 06000066 00066000 
66600600 60000000 00000066 00066000 00000000 00000000 00000000 00000000 
00000000 000D000D 00000000 00000000 00600000 00000000 000A0660 60600000 
00000AC0 A0E60060 00600000 00000660 00000600 06000066 00066660 00666600 
66000000 00000000 00000000 00000000 00000000 00000000 00000000 00000000 
000D000D 00000000 00000000 00600000 00000000 00000660 60600000 00000AA0 
ACE60060 00600000 00000060 60000600 06000006 06660000 66660600 66000006 
00000006 00000000 00000000 00000000 00000000 00000000 00000000 000D000D 
00000000 00000000 00000000 00000000 00000600 66000000 00000AA0 CE060600 
06000000 000A0060 60000600 06000006 06660000 66660600 66000006 00000006 
00000000 00000000 00000000 00000000 00000000 00000000 000D000D 00000000 
00000000 00000000 00000000 00006000 66000000 000000A6 C0060600 0660000A 
000AE060 60000060 00660006 06666660 00666666 00000000 00000000 00000000 
00000000 00000000 00000000 00000000 00000000 000D000D 00000000 00000000 
00000000 00000000 00006000 66000000 00000006 A0060600 0660000A 000CE060 
60000060 006600A0 00600000 66666666 00006666 00000000 00000000 00000000 
00000000 00000000 00000000 00000000 000D000D 00000000 00000000 00000000 
00000000 00006000 66000000 00000006 A0600660 00660000 60ACE060 06000006 
666660A0 00060000 66600000 00000666 00000000 00000000 00000000 00000000 
00000000 00000000 00000000 000D000D 00000000 00000000 00000000 00000000 
00000000 00000000 00000066 00666000 66600000 00AA0600 06000066 006660CE 
0A060000 66666000 00000060 66600000 00000000 00000000 00000000 00000000 
00000000 00000000 000D000D 00000000 00000000 00000000 00000000 00000000 
00000000 00000066 00666600 06666000 660A0600 06000000 66600ACE 0A060000 
66600000 00000066 66000000 00000000 00000000 00000000 00000000 00000000 
00000000 000D000D 00000000 00000000 00000000 00000000 00000000 60000000 
60000066 00606666 00006660 000A0600 06000066 600000CE AA060066 60000000 
00000066 66000000 00000000 00000000 00000000 00000000 00000000 00000000 
000D000D 00000000 00000000 00000000 00000000 00000000 00000000 00000000 
06666600 06666660 60000600 06000006 660000A0 ACE06000 06600000 0000A006 
06660000 00000000 00006000 00000000 00000000 00000000 00000000 000D000D 
00000000 00000000 00000000 00000000 00000000 00000000 66660000 66666666 
00006600 00006000 60000060 600000A0 0CE06006 06000000 000AA006 06660000 
00000000 00006000 00000000 00000000 00000000 00000000 000D000D 00000000 
00000000 00000000 00000000 00000000 00000000 66660000 00000066 60006600 
00066666 60000060 60000006 0AE06006 06000000 000AC006 A6660000 00000000 
00006000 00000000 00000000 00000000 00000000 000D000D 00000000 00000000 
00000000 00000000 00000000 00000666 06000000 00066666 00060000 00066600 
60000600 00000006 0A006006 06000000 0000CE06 A0660000 0000000A 00000600 
00000000 00000000 00000000 00000000 000D000D 00000000 00000000 00000000 
00000000 00000000 00000666 06000000 00066666 00060000 00066600 60000600 
00000006 00006006 00600000 0000A0E6 A0606000 0000000A 00000660 60000000 
00000000 00000000 00000000 000D000D 00000000 00000000 00000000 00000000 
00000000 00000066 66000000 00000666 00060000 00000666 00000600 00000066 
00066006 00600000 0000A0E6 C0606000 0000000A 000A0660 60000000 00000000 
00000000 00000000 000D000D 00000000 00000000 00000000 00066000 00000000 
00006660 06000000 00006660 00060000 00000660 00000600 60000066 00066000 
00600000 00000A06 CE606000 00000000 00A0A066 06000000 00000000 00000000 
00000000 000D000D 00000000 00000000 00000000 00060000 00000000 00006600 
60000000 00000660 00600000 00000066 60000600 00000006 06666606 60600000 
00000000 AE066000 00000000 000AA066 06000000 00000000 00000000 00000000 
000D000D 00000000 60000000 00000000 06660000 00000000 00060600 60000000 
00006600 00600000 00000060 60000600 60000006 06660000 60600000 00000006 
AE060600 00000000 0000CA06 00600000 000000A0 00000000 00000000 000D000D 
00000000 00000000 00000006 06600000 00000000 00066000 00000000 00006606 
00600000 00000060 60006000 00000000 00600006 60600000 00006666 00E60600 
00000000 000AC006 A0600000 0000000A 00000000 00000000 000D000D 00000000 
00000000 00000006 06600000 00000000 00060600 00000000 00006606 00600000 
00000060 60000600 60000000 00600000 60060000 00006666 00E06600 60000000 
0000A006 A0060000 0000000A 00000000 00000000 000D000D 00000000 00000000 
00000006 06600000 00000000 00060600 00000000 00000606 00600000 00000006 
06000600 60000000 00060000 60060000 00000666 00006600 60000000 0000AE06 
CA060000 0000000A 00000000 00000000 000D000D 00060000 00000000 00000660 
66000000 00000000 00606000 00000000 00006006 00600000 00000006 06006000 
00000000 00060006 60600000 00000066 66006600 60000000 0000A0E6 CA060000 
00000000 000A0000 00000000 000D000D 00060000 00000000 00000660 66000000 
00000000 00606000 00000000 00006006 00600000 00000006 06000600 60000000 
00060000 60060000 00000066 66006060 66000000 000000E0 ACA06000 00000000 
000A0000 00000000 000D000D 00600000 00000000 00006600 60000000 00000006 
06060000 00000000 00006060 00600000 00000060 06000600 60000000 00060000 
66060000 00000060 66600660 06000000 000000E0 ACA00000 00000000 0000A000 
00000000 000D000D 00600000 00000000 00000600 60000000 00000006 00606000 
00000000 00006006 00600000 00000006 00600600 60000000 00006000 06006000 
00000006 06660666 00600000 0000600E 0ACA0000 00000000 0000A000 00000000

end csub