  
  '++++++++++++ Begin VL53L0X functions and subs ++++++++++++++++
function setSignalRateLimit (sens_idx as integer, sens_rate_lim as float) as integer
  if (sens_rate_lim < 0) or (sens_rate_lim > 511.99) then
    setSignalRateLimit = 0      'false
    exit function
  else
    setSignalRateLimit = 1      'true
    writeReg(sens_idx, &h44, (sens_rate_lim * (1 << 7)), 2)
  end if
end function
  
  '++++++++++++++++++++++++++++++++++++++
  'set new I2C address to sensor
sub setI2Caddr (old_addr as integer, new_addr as integer)
  I2C write old_addr, 0, 2, &H8A, new_addr
end sub
  
  
  '++++++++++++++++++++++++++++++++++++++
  ' Returns a range reading in millimeters when continuous mode is active
  ' (readRangeSingleMillimeters() also calls this function after starting a
  ' single-shot range measureme
function readRangeContMM(sens_idx as integer) as integer
  local integer range
  
  start_Timeout(sens_idx)
  do while ((readReg(sens_idx, &H13) and &H07) = 0)     'read RESULT_INTERRUPT_STATUS
    '// "Wait until start bit has been cleared"
    if check_Timeout(sens_idx, VL53L0X_IO_Timeout) = 1 then
      readRangeContMM = 65535       'mark timeout
      exit function
    end if
  loop
  
  ' assumptions: Linearity Corrective Gain is 1000 (default);
  ' fractional ranging is not enabled
  range = readReg(sens_idx, &H14 + 10,2)   'Reg: RESULT_RANGE_STATUS, 16 bit
  
  writeReg(sens_idx, &H0B, &H01)        'Reg: SYSTEM_INTERRUPT_CLEAR
  
  readRangeContMM = range
end function
  '---------------------------------------
  
  '++++++++++++++++++++++++++++++++++++++
  ' Starts a single-shot range measurement without waiting
  ' based on VL53L0X_PerformSingleRangingMeasurement()
sub startRangeSingleMM_BG(sens_idx as integer) as integer
  writeReg(sens_idx, &h80, &h01)    'reg: POWER_MANAGEMENT_GO1_POWER_FORCE
  writeReg(sens_idx, &hFF, &h01)    'reg:
  writeReg(sens_idx, &h00, &h00)    'reg: SYSRANGE_START
  writeReg(sens_idx, &h91, VL53L0X_stop_variable(sens_idx))    'reg:
  writeReg(sens_idx, &h00, &h01)    'reg: SYSRANGE_START
  writeReg(sens_idx, &hFF, &h00)    'reg:
  writeReg(sens_idx, &h80, &h00)    'reg: POWER_MANAGEMENT_GO1_POWER_FORCE
  
  writeReg(sens_idx, &h00, &h01)
  start_Timeout(sens_idx)
end sub
  '---------------------------------------
  
  '++++++++++++++++++++++++++++++++++++++
  'perform a sensor distance read, no wait for data
  'returns the range reading in millimeters or
  '65634 if measurement is not completed yet
  '65535 if a timeout occured
function readRangeSingleMM_BG(sens_idx as integer) as integer
  if (readReg(sens_idx, &h00) and &h01) = 1 then      'check if start bit is still set
    readRangeSingleMM = 65534                         'mark no data available yet
    if check_Timeout(sens_idx, VL53L0X_IO_Timeout) = 1 then 'check for timout
      readRangeSingleMM = 65535                       'mark timeout
    end if
  else
    readRangeSingleMM_BG = readRangeContMM(sens_idx)
  end if
end function
  '---------------------------------------
  
  '++++++++++++++++++++++++++++++++++++++
  'perform a sensor distance read and wait until data is valid or timeout occured
  'return distance or 65535 if a timeout occured
function readRangeSingleMM_WT(sens_idx as integer) as integer
  '// "Wait until start bit has been cleared"
  start_Timeout(sens_idx)
  do while (readReg(sens_idx, &h00) and &h01) = 1
    if check_Timeout(sens_idx, VL53L0X_IO_Timeout) = 1 then
      readRangeSingleMM = 65535
      exit function
    end if
  loop
  readRangeSingleMM_WT = readRangeContMM(sens_idx)
end function  
  '++++++++++++++++++++++++++++++++++++++
  ' Performs a single-shot range measurement and returns the reading in
  ' millimeters
  ' based on VL53L0X_PerformSingleRangingMeasurement()
function readRangeSingleMM(sens_idx as integer) as integer
  
  writeReg(sens_idx, &h80, &h01)    'reg: POWER_MANAGEMENT_GO1_POWER_FORCE
  writeReg(sens_idx, &hFF, &h01)    'reg:
  writeReg(sens_idx, &h00, &h00)    'reg: SYSRANGE_START
  writeReg(sens_idx, &h91, VL53L0X_stop_variable(sens_idx))    'reg:
  writeReg(sens_idx, &h00, &h01)    'reg: SYSRANGE_START
  writeReg(sens_idx, &hFF, &h00)    'reg:
  writeReg(sens_idx, &h80, &h00)    'reg: POWER_MANAGEMENT_GO1_POWER_FORCE
  
  writeReg(sens_idx, &h00, &h01)
  
  '// "Wait until start bit has been cleared"
  start_Timeout(sens_idx)
  do while (readReg(sens_idx, &h00) and &h01) = 1
    if check_Timeout(sens_idx, VL53L0X_IO_Timeout) = 1 then
      readRangeSingleMM = 65535
      exit function
    end if
  loop
  readRangeSingleMM = readRangeContMM(sens_idx)
end function
  '---------------------------------------
  
  '++++++++++++++++++++++++++++++++++++++
sub start_timeout(sens_idx as integer)
  VL53L0X_timer(sens_idx) = timer
end sub
  '---------------------------------------
  
  '++++++++++++++++++++++++++++++++++++++
function check_timeout(sens_idx as integer, timeout as integer) as integer
  if timer - VL53L0X_timer(sens_idx) >= timeout then
    check_timeout = 1
  else
    check_timeout = 0
  end if
end function
  '---------------------------------------
  
  '++++++++++++++++++++++++++++++++++++++
  ' Start continuous ranging measurements. If period_ms (optional) is 0 or not
  ' given, continuous back-to-back mode is used (the sensor takes measurements as
  ' often as possible); otherwise, continuous timed mode is used, with the given
  ' inter-measurement period in milliseconds determining how often the sensor
  ' takes a measurement.
  ' based on VL53L0X_StartMeasurement()
sub startContinuous(sens_idx as integer, period_ms as integer)
  local integer osc_calibrate_val
  
  writeReg(sens_idx, &H80, &H01)
  writeReg(sens_idx, &HFF, &H01)
  writeReg(sens_idx, &H00, &H00)
  writeReg(sens_idx, &H91, VL53L0X_stop_variable(sens_idx))
  writeReg(sens_idx, &H00, &H01)
  writeReg(sens_idx, &HFF, &H00)
  writeReg(sens_idx, &H80, &H00)
  
  if period_ms <> 0 then
    
    ' continuous timed mode
    osc_calibrate_val = readReg(sens_idx, &HF8,2)  'Reg: OSC_CALIBRATE_VAL
    if osc_calibrate_val <> 0 then
      period_ms = period_ms * osc_calibrate_val
    end if
    writeReg(sens_idx, &H04, period_ms, 4)    'Reg: SYSTEM_INTERMEASUREMENT_PERIOD
    writeReg(sens_idx, &H00, &H04) ' VL53L0X_REG_SYSRANGE_MODE_TIMED
  else
    ' continuous back-to-back mode
    writeReg(sens_idx, &H00, &H02) ' VL53L0X_REG_SYSRANGE_MODE_BACKTOBACK
  end if
  
end sub
  '---------------------------------------
  
  '++++++++++++++++++++++++++++++++++++++
  'Stop continuous measurements
  'based on VL53L0X_StopMeasurement()
sub stopContinuous(sens_idx as integer)
  writeReg(sens_idx, &h00, &h01)    'Reg: SYSRANGE_START, VL53L0X_REG_SYSRANGE_MODE_SINGLESHOT
  writeReg(sens_idx, &hFF, &h01)
  writeReg(sens_idx, &h00, &h00)
  writeReg(sens_idx, &h91, &h00)
  writeReg(sens_idx, &h00, &h01)
  writeReg(sens_idx, &hFF, &h00)
end sub
  '---------------------------------------
  
  '++++++++++++++++++++++++++++++++++++++
  'read 1 to 8 bytes from registers
  'reg ... register address of first register to read
  'num_bt ... number of bytes to read
  'i2c_addr ... I2C address of device
function readreg (VL_idx as integer, reg as integer, num_bt as integer) as integer
  local integer reg_cont(8), r_idx, e_idx
  if num_bt = 0 then num_bt=1                 'default to 1 byte transfer
  
  I2C write VL53L0X_i2c(VL_idx),1,1,reg
  I2C read VL53L0X_i2c(VL_idx),0,num_bt,reg_cont()
  
  readreg = 0
  for r_idx = 0 to num_bt-1
    e_idx = num_bt - r_idx -1
    readreg = readreg + (reg_cont(r_idx)<<(8*e_idx))
  next r_idx
  
end function
  '---------------------------------------
  
  '++++++++++++++++++++++++++++++++++++++
  'write 1 to 8 bytes to registers starting with reg
  'reg ... register address of first register to write
  'content ... integer containing data to be written
  'num_bt ... number of bytes to write
  'i2c_addr ... I2C address of device
sub writereg (VL_idx as integer, reg as integer, content as integer, num_bt as integer)
  local integer reg_cont(9), r_idx, e_idx
  
  if num_bt = 0 then num_bt=1                 'default to 1 byte transfer
  
  reg_cont(0) = reg           'register to write is first data
  for r_idx = 0 to num_bt-1
    e_idx = num_bt - r_idx -1
    reg_cont(r_idx+1) = content>>(8*e_idx)
  next r_idx
  
  I2C write VL53L0X_i2c(VL_idx),0,num_bt+1, reg_cont()
  
end sub
  '---------------------------------------
  
  
  '++++++++++++++++++++++++++++++++++++++
function init_sensor (sens_idx as integer, sens_i2c as integer, sens_io_timeout as integer) as integer
  local integer sens_stat
  local integer spad_count, ref_spad_map
  local integer spad_type_is_aperture, spads_enabled, first_spad_to_enable, idx, stat
  'local integer measurement_timing_budget_us
  
  on error skip
  dim integer VL53L0X_stop_variable(10), VL53L0X_i2c(10), VL53L0X_timer(10), VL53L0X_Meas_Tim_Budget_us(10), VL53L0X_io_timeout
  
  if sens_io_timeout <> 0 then            'if timout value given
    VL53L0X_io_timeout = sens_io_timeout  'replace the old value
  end if
  
  VL53L0X_i2c(sens_idx) = sens_i2c    'save sensor i2c address
  
  writeReg(sens_idx, &H89,(readReg(sens_idx,&H89) or &H01))    ' set bit 0 ... switch to 2V8 mode
  
  '"Set I2C standard mode"
  writeReg(sens_idx, &H88, &H00)
  writeReg(sens_idx, &H80, &H01)
  writeReg(sens_idx, &HFF, &H01)
  writeReg(sens_idx, &H00, &H00)
  VL53L0X_stop_variable(sens_idx) = readReg(sens_idx, &H91) 'read by init and used when starting measurement;
  writeReg(sens_idx, &H00, &H01)
  writeReg(sens_idx, &HFF, &H00)
  writeReg(sens_idx, &H80, &H00)
  
  'disable SIGNAL_RATE_MSRC (bit 1) and SIGNAL_RATE_PRE_RANGE (bit 4) limit checks
  writeReg(sens_idx, &h60, (readReg(sens_idx, &h60)) or &H12)   'reg: MSRC_CONFIG_CONTROL
  
  'set final range signal rate limit to 0.25 MCPS (million counts per second)
  sens_stat = setSignalRateLimit (sens_idx,0.25)
  
  writeReg(sens_idx, &h01, &hFF)    'reg: SYSTEM_SEQUENCE_CONFIG
  
  'VL53L0X_DataInit() end
  
  'VL53L0X_StaticInit() begin
  
  if getSpadInfo(sens_idx, spad_count, spad_type_is_aperture) = 1 then      'error at get SPAD info
    init_sensor = 1           'mark error
    exit function
  end if
  
  'The SPAD map (RefGoodSpadMap) is read by VL53L0X_get_info_from_device() in
  'the API, but the same data seems to be more easily readable from
  'GLOBAL_CONFIG_SPAD_ENABLES_REF_0 through _6, so read it from there
  
  ref_spad_map = readReg(sens_idx, &HB0, 6)
  
  ' -- VL53L0X_set_reference_spads() begin (assume NVM values are valid)
  
  writeReg(sens_idx, &hFF, &h01)
  writeReg(sens_idx, &h4F, &h00)    'reg: DYNAMIC_SPAD_REF_EN_START_OFFSET
  writeReg(sens_idx, &h4E, &h2C)    'reg: DYNAMIC_SPAD_NUM_REQUESTED_REF_SPAD
  writeReg(sens_idx, &HFF, &H00)
  writeReg(sens_idx, &HB6, &hB4)    'reg: GLOBAL_CONFIG_REF_EN_START_SELECT
  
  if spad_type_is_aperture = 1 then
    first_spad_to_enable = 12
  else
    first_spad_to_enable = 0
  end if
  spads_enabled = 0
  
  for idx = 0 to 47
    if (idx < first_spad_to_enable) or (spads_enabled = spad_count) then
      ref_spad_map = ref_spad_map and inv(1<<idx)
    else if (ref_spad_map>>((int(idx/8))*8)) and &h01 = 1 then
      inc spads_enabled
    end if
  next idx
  
  writeReg(sens_idx, &hB0, ref_spad_map, 6)     'Reg: GLOBAL_CONFIG_SPAD_ENABLES_REF_0
  
  '-- VL53L0X_set_reference_spads() begin (assume NVM values are valid)
  
  
  ' -- VL53L&H_load_tuning_settings() begin
  ' DefaultTuningSettings from vl53l&H_tuning.h
  
  writeReg(sens_idx, &HFF, &H01)
  writeReg(sens_idx, &H00, &H00)
  
  writeReg(sens_idx, &HFF, &H00)
  writeReg(sens_idx, &H09, &H00)
  writeReg(sens_idx, &H10, &H00)
  writeReg(sens_idx, &H11, &H00)
  
  writeReg(sens_idx, &H24, &H01)
  writeReg(sens_idx, &H25, &HFF)
  writeReg(sens_idx, &H75, &H00)
  
  writeReg(sens_idx, &HFF, &H01)
  writeReg(sens_idx, &H4E, &H2C)
  writeReg(sens_idx, &H48, &H00)
  writeReg(sens_idx, &H30, &H20)
  
  writeReg(sens_idx, &HFF, &H00)
  writeReg(sens_idx, &H30, &H09)
  writeReg(sens_idx, &H54, &H00)
  writeReg(sens_idx, &H31, &H04)
  writeReg(sens_idx, &H32, &H03)
  writeReg(sens_idx, &H40, &H83)
  writeReg(sens_idx, &H46, &H25)
  'writeReg(sens_idx, &H60, &H00)
  writeReg(sens_idx, &H27, &H00)
  writeReg(sens_idx, &H50, &H06)
  writeReg(sens_idx, &H51, &H00)
  writeReg(sens_idx, &H52, &H96)
  writeReg(sens_idx, &H56, &H08)
  writeReg(sens_idx, &H57, &H30)
  writeReg(sens_idx, &H61, &H00)
  writeReg(sens_idx, &H62, &H00)
  writeReg(sens_idx, &H64, &H00)
  writeReg(sens_idx, &H65, &H00)
  writeReg(sens_idx, &H66, &HA0)
  
  writeReg(sens_idx, &HFF, &H01)
  writeReg(sens_idx, &H22, &H32)
  writeReg(sens_idx, &H47, &H14)
  writeReg(sens_idx, &H49, &HFF)
  writeReg(sens_idx, &H4A, &H00)
  
  writeReg(sens_idx, &HFF, &H00)
  writeReg(sens_idx, &H7A, &H0A)
  writeReg(sens_idx, &H7B, &H00)
  writeReg(sens_idx, &H78, &H21)
  
  writeReg(sens_idx, &HFF, &H01)
  writeReg(sens_idx, &H23, &H34)
  writeReg(sens_idx, &H42, &H00)
  writeReg(sens_idx, &H44, &HFF)
  writeReg(sens_idx, &H45, &H26)
  writeReg(sens_idx, &H46, &H05)
  writeReg(sens_idx, &H40, &H40)
  writeReg(sens_idx, &H0E, &H06)
  writeReg(sens_idx, &H20, &H1A)
  writeReg(sens_idx, &H43, &H40)
  
  writeReg(sens_idx, &HFF, &H00)
  writeReg(sens_idx, &H34, &H03)
  writeReg(sens_idx, &H35, &H44)
  
  writeReg(sens_idx, &HFF, &H01)
  writeReg(sens_idx, &H31, &H04)
  writeReg(sens_idx, &H4B, &H09)
  writeReg(sens_idx, &H4C, &H05)
  writeReg(sens_idx, &H4D, &H04)
  
  writeReg(sens_idx, &HFF, &H00)
  writeReg(sens_idx, &H44, &H00)
  writeReg(sens_idx, &H45, &H20)
  writeReg(sens_idx, &H47, &H08)
  writeReg(sens_idx, &H48, &H28)
  writeReg(sens_idx, &H67, &H00)
  writeReg(sens_idx, &H70, &H04)
  writeReg(sens_idx, &H71, &H01)
  writeReg(sens_idx, &H72, &HFE)
  writeReg(sens_idx, &H76, &H00)
  writeReg(sens_idx, &H77, &H00)
  
  writeReg(sens_idx, &HFF, &H01)
  writeReg(sens_idx, &H0D, &H01)
  
  writeReg(sens_idx, &HFF, &H00)
  writeReg(sens_idx, &H80, &H01)
  writeReg(sens_idx, &H01, &HF8)
  
  writeReg(sens_idx, &HFF, &H01)
  writeReg(sens_idx, &H8E, &H01)
  writeReg(sens_idx, &H00, &H01)
  writeReg(sens_idx, &HFF, &H00)
  writeReg(sens_idx, &H80, &H00)
  
  ' -- VL53L0X_load_tuning_settings() end
  
  ' "Set interrupt config to new sample ready"
  ' -- VL53L0X_SetGpioConfig() begin
  
  writeReg(sens_idx, &H0A, &H04)        'Reg: SYSTEM_INTERRUPT_CONFIG_GPIO
  writeReg(sens_idx, &H84, (readReg(sens_idx, &H84) and inv(&H10)))   ' Reg: GPIO_HV_MUX_ACTIVE_HIGH  active low
  writeReg(sens_idx, &h0B, &H01)        'Reg: SYSTEM_INTERRUPT_CLEAR
  
  ' -- VL53L0X_SetGpioConfig() end
  
  VL53L0X_Meas_Tim_Budget_us(sens_idx) = getMeasurementTimingBudget(sens_idx)
  
  ' "Disable MSRC and TCC by default"
  ' MSRC = Minimum Signal Rate Check
  ' TCC = Target CentreCheck
  ' -- VL53L0X_SetSequenceStepEnable() begin
  
  writeReg(sens_idx, &H01, &HE8)          'Reg: SYSTEM_SEQUENCE_CONFIG
  
  ' -- VL53L0X_SetSequenceStepEnable() end
  
  ' "Recalculate timing budget"
  stat = setMeasurementTimingBudget(sens_idx, VL53L0X_Meas_Tim_Budget_us(sens_idx))
  
  ' VL53L0X_StaticInit() end
  
  ' VL53L0X_PerformRefCalibration() begin (VL53L0X_perform_ref_calibration())
  
  ' -- VL53L0X_perform_vhv_calibration() begin
  
  writeReg(sens_idx, &H01, &H01)          'Reg: SYSTEM_SEQUENCE_CONFIG
  if performSingleRefCalibration(sens_idx, &h40) = 1 then
    init_sensor = 1
    exit function
  end if
  
  ' -- VL53L0X_perform_vhv_calibration() end
  
  ' -- VL53L0X_perform_phase_calibration() begin
  
  writeReg(sens_idx, &H01, &H02)          'Reg: SYSTEM_SEQUENCE_CONFIG
  if performSingleRefCalibration(sens_idx, &h00) = 1 then
    init_sensor = 1
    exit function
  end if
  
  ' -- VL53L0X_perform_phase_calibration() end
  
  ' "restore the previous Sequence Config"
  writeReg(sens_idx, &H01, &HE8)          'Reg: SYSTEM_SEQUENCE_CONFIG
  
  ' VL53L0X_PerformRefCalibration() end
  
  init_sensor = 0                         'success
end function
  '---------------------------------------
  
  '++++++++++++++++++++++++++++++++++++++
  'Get reference SPAD (single photon avalanche diode) count and type
  'based on VL53L0X_get_info_from_device(),
  'but only gets reference SPAD count and type
function getSpadInfo(sens_idx as integer, spad_count as integer, type_is_aperture as integer) as integer
  
  local integer tmp, stat
  
  writeReg(sens_idx, &h80, &h01)
  writeReg(sens_idx, &hFF, &h01)
  writeReg(sens_idx, &h00, &h00)
  
  writeReg(sens_idx, &hFF, &h06)
  writeReg(sens_idx, &h83, (readReg(sens_idx, &h83) or &h04))
  writeReg(sens_idx, &hFF, &h07)
  writeReg(sens_idx, &h81, &h01)
  
  writeReg(sens_idx, &h80, &h01)
  
  writeReg(sens_idx, &h94, &h6b)
  
  writeReg(sens_idx, &h83, &h00)
  start_Timeout(sens_idx)
  do while readReg(sens_idx, &h83) = &h00
    if check_timeout(sens_idx, 100) = 1 then
      getSpadInfo = 1                          'mark timeout
      exit function
    end if
  loop
  writeReg(sens_idx, &h83, &h01)
  
  tmp = readReg(sens_idx, &h92)
  spad_count = tmp and &h7f
  type_is_aperture = (tmp >> 7) and &h01
  
  writeReg(sens_idx, &h81, &h00)
  writeReg(sens_idx, &hFF, &h06)
  writeReg(sens_idx, &h83, (readReg(sens_idx, &h83)  and inv(&h04)))
  writeReg(sens_idx, &hFF, &h01)
  writeReg(sens_idx, &h00, &h01)
  
  writeReg(sens_idx, &hFF, &h00)
  writeReg(sens_idx, &h80, &h00)
  
  getSpadInfo = 0
end function
  '---------------------------------------
  
  '++++++++++++++++++++++++++++++++++++++
  'Convert sequence step timeout from microseconds to MCLKs with given VCSEL period in PCLKs
  'based on VL53L0X_calc_timeout_mclks()
function timeoutMicrosecondsToMclks(timeout_period_us as integer, vcsel_period_pclks as integer) as integer
  local integer macro_period_ns
  
  macro_period_ns = calcMacroPeriod(vcsel_period_pclks)
  
  timeoutMicrosecondsToMclks = (((timeout_period_us * 1000) + (macro_period_ns / 2)) / macro_period_ns)
end function
  '---------------------------------------
  
  
  '++++++++++++++++++++++++++++++++++++++
  ' Get sequence step enables
  ' based on VL53L0X_GetSequenceStepEnables()
sub getSequenceStepEnables(sens_idx as integer, tcc as integer, dss as integer, msrc as integer, pre_range as integer, final_range as integer)
  local integer sequence_config
  
  sequence_config = readReg(sens_idx, &h01)     'Reg: SYSTEM_SEQUENCE_CONFIG
  
  tcc = (sequence_config >> 4) and &H01
  dss = (sequence_config >> 3) and &H01
  msrc = (sequence_config >> 2) and &H01
  pre_range = (sequence_config >> 6) and &H01
  final_range = (sequence_config >> 7) and &H01
end sub
  '---------------------------------------
  
  '++++++++++++++++++++++++++++++++++++++
  ' Get sequence step timeouts
  ' based on get_sequence_step_timeout(),
  ' but gets all timeouts instead of just the requested one, and also stores
  ' intermediate values
sub getSequenceStepTimeouts(sens_idx%, pre_range%, pre_range_vcsel_period_pclks%, msrc_dss_tcc_mclks%, msrc_dss_tcc_us%, pre_range_mclks%, pre_range_us%, final_range_vcsel_period_pclks%, final_range_mclks%, final_range_us%)
  
  const VcselPeriodPreRange=0, VcselPeriodFinalRange=1
  
  pre_range_vcsel_period_pclks% = getVcselPulsePeriod(VcselPeriodPreRange)
  
  msrc_dss_tcc_mclks% = readReg(sens_idx%, &h46) + 1    'Reg: MSRC_CONFIG_TIMEOUT_MACROP
  msrc_dss_tcc_us% = timeoutMclksToMicroseconds(msrc_dss_tcc_mclks%, pre_range_vcsel_period_pclks%)
  
  pre_range_mclks% = decodeTimeout(readReg(sens_idx%, &h51, 2))  'Reg: PRE_RANGE_CONFIG_TIMEOUT_MACROP_HI
  pre_range_us% = timeoutMclksToMicroseconds(pre_range_mclks%, pre_range_vcsel_period_pclks%)
  
  final_range_vcsel_period_pclks% = getVcselPulsePeriod(sens_idx%, VcselPeriodFinalRange)
  
  final_range_mclks% = decodeTimeout(readReg(sens_idx%, &h71, 2))   'Reg: FINAL_RANGE_CONFIG_TIMEOUT_MACROP_HI
  
  if pre_range% = 1 then
    final_range_mclks% = final_range_mclks% - pre_range_mclks%
  end if
  
  final_range_us% = timeoutMclksToMicroseconds(final_range_mclks%, final_range_vcsel_period_pclks%)
end sub
  '---------------------------------------
  
  '++++++++++++++++++++++++++++++++++++++
  ' Decode sequence step timeout in MCLKs from register value
  ' based on VL53L0X_decode_timeout()
  ' Note: the original function returned a uint32_t, but the return value is
  ' always stored in a uint16_t.
function decodeTimeout(reg_val as integer) as integer
  
  ' format: "(LSByte * 2^MSByte) + 1"
  decodeTimeout = ((reg_val and &h00FF) << ((reg_val and &hFF00) >> 8)) + 1
end function
  '---------------------------------------
  
  '++++++++++++++++++++++++++++++++++++++
  ' Encode sequence step timeout register value from timeout in MCLKs
  ' based on VL53L0X_encode_timeout()
function encodeTimeout(timeout_mclks as integer) as integer
  local integer ls_byte, ms_byte
  ' format: "(LSByte * 2^MSByte) + 1"
  
  if (timeout_mclks > 0) then
    ls_byte = timeout_mclks - 1
    do while ((ls_byte and &hFFFFFF00) > 0)
      ls_byte = ls_byte >> 1
      inc ms_byte
    loop
    encodeTimeout =  (ms_byte << 8) or (ls_byte and &hFF)
  else
    encodeTimeout = 0
  end if
end function
  '---------------------------------------
  
  '++++++++++++++++++++++++++++++++++++++
  ' Convert sequence step timeout from MCLKs to microseconds with given VCSEL period in PCLKs
  ' based on VL53L0X_calc_timeout_us()
function timeoutMclksToMicroseconds(timeout_period_mclks as integer, vcsel_period_pclks as integer) as integer
  local integer macro_period_ns
  macro_period_ns = calcMacroPeriod(vcsel_period_pclks)
  timeoutMclksToMicroseconds = ((timeout_period_mclks * macro_period_ns) + 500) / 1000
end function
  '---------------------------------------
  
  '++++++++++++++++++++++++++++++++++++++
  ' Calculate macro period in *nanoseconds* from VCSEL period in PCLKs
  ' based on VL53L0X_calc_macro_period_ps()
  ' PLL_period_ps = 1655; macro_period_vclks = 2304
function calcMacroPeriod(vcsel_period_pclks as integer) as integer
  calcMacroPeriod = (((2304 * (vcsel_period_pclks) * 1655) + 500) / 1000)
end function
  '---------------------------------------
  
  '++++++++++++++++++++++++++++++++++++++
  ' Get the VCSEL pulse period in PCLKs for the given period type.
  ' based on VL53L0X_get_vcsel_pulse_period()
function getVcselPulsePeriod(sens_idx as integer, type as integer) as integer
  const VcselPeriodPreRange=0, VcselPeriodFinalRange=1
  
  if type = VcselPeriodPreRange then
    getVcselPulsePeriod = decodeVcselPeriod(readReg(sens_idx, &h50)) 'Reg: PRE_RANGE_CONFIG_VCSEL_PERIOD
  else if type = VcselPeriodFinalRange then
    getVcselPulsePeriod = decodeVcselPeriod(readReg(sens_idx, &h70)) 'Reg: INAL_RANGE_CONFIG_VCSEL_PERIOD
  else
    getVcselPulsePeriod = 255
  end if
end function
  '---------------------------------------
  
  '++++++++++++++++++++++++++++++++++++++
  ' Decode VCSEL (vertical cavity surface emitting laser) pulse period in PCLKs
  ' from register value
  ' based on VL53L0X_decode_vcsel_period()
function decodeVcselPeriod(reg_val as integer) as integer
  decodeVcselPeriod = (((reg_val) + 1) << 1)
end function
  '---------------------------------------
  
  '++++++++++++++++++++++++++++++++++++++
  ' Set the measurement timing budget in microseconds, which is the time allowed
  ' for one measurement; the ST API and this library take care of splitting the
  ' timing budget among the sub-steps in the ranging sequence. A longer timing
  ' budget allows for more accurate measurements. Increasing the budget by a
  ' factor of N decreases the range measurement standard deviation by a factor of
  ' sqrt(N). Defaults to about 33 milliseconds; the minimum is 20 ms.
  ' based on VL53L0X_set_measurement_timing_budget_micro_seconds()
function setMeasurementTimingBudget(sens_idx as integer, budget_us as integer) as integer
  local integer used_budget_us, tcc , dss, msrc, pre_range, final_range
  local integer pre_range_vcsel_period_pclks, msrc_dss_tcc_mclks, msrc_dss_tcc_us, pre_range_mclks, pre_range_us, final_range_vcsel_period_pclks, final_range_mclks, final_range_us, final_range_timeout_us
  local integer final_range_timeout_mclks
  
  const StartOverhead     = 1910
  const EndOverhead        = 960
  const MsrcOverhead       = 660
  const TccOverhead        = 590
  const DssOverhead        = 690
  const PreRangeOverhead   = 660
  const FinalRangeOverhead = 550
  
  used_budget_us = StartOverhead + EndOverhead
  
  getSequenceStepEnables(sens_idx, tcc , dss, msrc, pre_range, final_range)
  'getSequenceStepTimeouts(&enables, &timeouts)
  getSequenceStepTimeouts(sens_idx, pre_range, pre_range_vcsel_period_pclks, msrc_dss_tcc_mclks, msrc_dss_tcc_us, pre_range_mclks, pre_range_us, final_range_vcsel_period_pclks, final_range_mclks, final_range_us)
  
  if tcc = 1 then
    used_budget_us = used_budget_us + (timeouts.msrc_dss_tcc_us + TccOverhead)
  end if
  
  if dss = 1 then
    used_budget_us = used_budget_us + 2 * (msrc_dss_tcc_us + DssOverhead)
  else if msrc = 1 then
    used_budget_us = used_budget_us + (msrc_dss_tcc_us + MsrcOverhead)
  end if
  
  if pre_range = 1 then
    used_budget_us = used_budget_us + (pre_range_us + PreRangeOverhead)
  end if
  
  if final_range = 1 then
    used_budget_us = used_budget_us + FinalRangeOverhead
    
    ' "Note that the final range timeout is determined by the timing
    ' budget and the sum of all other timeouts within the sequence.
    ' If there is no room for the final range timeout, then an error
    ' will be set. Otherwise the remaining time will be applied to
    ' the final range."
    
    if used_budget_us > budget_us then
      ' "Requested timeout too big."
      setMeasurementTimingBudget = 1
      EXIT FUNCTION
    END IF
    
    final_range_timeout_us = budget_us - used_budget_us
    
    ' set_sequence_step_timeout() begin
    ' (SequenceStepId == VL53L0X_SEQUENCESTEP_FINAL_RANGE)
    
    ' "For the final range timeout, the pre-range timeout
    '  must be added. To do this both final and pre-range
    '  timeouts must be expressed in macro periods MClks
    '  because they have different vcsel periods."
    
    final_range_timeout_mclks = timeoutMicrosecondsToMclks(final_range_timeout_us, final_range_vcsel_period_pclks)
    
    if pre_range = 1 then
      final_range_timeout_mclks = final_range_timeout_mclks + pre_range_mclks
    end if
    
    writeReg(sens_idx, &h71, encodeTimeout(final_range_timeout_mclks),2)  'Reg: FINAL_RANGE_CONFIG_TIMEOUT_MACROP_HI
    
    ' set_sequence_step_timeout() end
    
    VL53L0X_Meas_Tim_Budget_us(sens_idx) = budget_us ' store for internal reuse
    setMeasurementTimingBudget = 0
end function
    '---------------------------------------
    
    '++++++++++++++++++++++++++++++++++++++
    ' Get the measurement timing budget in microseconds
    ' based on VL53L0X_get_measurement_timing_budget_micro_seconds()
    ' in us
function getMeasurementTimingBudget(sens_idx as integer) as integer
    
    local integer budget_us, tcc , dss, msrc, pre_range, final_range
    local integer pre_range_vcsel_period_pclks, msrc_dss_tcc_mclks, msrc_dss_tcc_us, pre_range_mclks, pre_range_us, final_range_vcsel_period_pclks, final_range_mclks, final_range_us
    
    const StartOverhead     = 1910
    const EndOverhead        = 960
    const MsrcOverhead       = 660
    const TccOverhead        = 590
    const DssOverhead        = 690
    const PreRangeOverhead   = 660
    const FinalRangeOverhead = 550
    
    ' "Start and end overhead times always present"
    budget_us = StartOverhead + EndOverhead
    
    getSequenceStepEnables(sens_idx, tcc , dss, msrc, pre_range, final_range)
    'getSequenceStepTimeouts(&enables, &timeouts)
    getSequenceStepTimeouts(sens_idx, pre_range, pre_range_vcsel_period_pclks, msrc_dss_tcc_mclks, msrc_dss_tcc_us, pre_range_mclks, pre_range_us, final_range_vcsel_period_pclks, final_range_mclks, final_range_us)
    
    if tcc = 1 then
      budget_us = budget_us + (msrc_dss_tcc_us + TccOverhead)
    end if
    
    if dss = 1 then
      budget_us = budget_us + 2 * (msrc_dss_tcc_us + DssOverhead)
    else if msrc = 1 then
      budget_us = budget_us + (msrc_dss_tcc_us + MsrcOverhead)
    end if
    
    if pre_range = 1 then
      budget_us = budget_us + (pre_range_us + PreRangeOverhead)
    end if
    
    if final_range = 1 then
      budget_us = budget_us + (final_range_us + FinalRangeOverhead)
    end if
    
    VL53L0X_Meas_Tim_Budget_us(sens_idx) = budget_us ' store for internal reuse
    getMeasurementTimingBudget = budget_us
    
end function
    '---------------------------------------
    
    '++++++++++++++++++++++++++++++++++++++
    ' based on VL53L0X_perform_single_ref_calibration()
function performSingleRefCalibration(sens_idx as integer, vhv_init_byte as integer) as integer
    
    writeReg(sens_idx, &h00, (&h01 or vhv_init_byte))  'Reg: SYSRANGE_START, VL53L0X_REG_SYSRANGE_MODE_START_STOP
    
    start_Timeout(sens_idx)
    do while ((readReg(sens_idx, &h13) and &h07) = 0) 'reg: RESULT_INTERRUPT_STATUS
      if (check_Timeout(sens_idx, 100)) = 1 then
        performSingleRefCalibration = 1
        exit function
      end if
    loop
    
    writeReg(sens_idx, &h0b, &h01)  'reg: SYSTEM_INTERRUPT_CLEAR
    writeReg(sens_idx, &h00, &h00)  'reg: SYSRANGE_START
    
    performSingleRefCalibration = 0
end function
    '---------------------------------------
    
    '++++++++++++++++++++++++++++++++++++++
    ' Set the VCSEL (vertical cavity surface emitting laser) pulse period for the
    ' given period type (pre-range or final range) to the given value in PCLKs.
    ' Longer periods seem to increase the potential range of the sensor.
    ' Valid values are (even numbers only):
    '  pre:  12 to 18 (initialized default: 14)
    '  final: 8 to 14 (initialized default: 10)
    ' based on VL53L0X_set_vcsel_pulse_period()
function setVcselPulsePeriod(sens_idx as integer, type as integer, period_pclks as integer) as integer
    
    local integer vcsel_period_reg = encodeVcselPeriod(period_pclks)
    local integer tcc , dss, msrc, pre_range, final_range
    local integer pre_range_vcsel_period_pclks, msrc_dss_tcc_mclks, msrc_dss_tcc_us, pre_range_mclks, pre_range_us, final_range_vcsel_period_pclks, final_range_mclks, final_range_us, final_range_timeout_us
    local integer new_pre_range_timeout_mclks, new_msrc_timeout_mclks, new_final_range_timeout_mclks
    local integer stat, sequence_config
    
    getSequenceStepEnables(sens_idx, tcc , dss, msrc, pre_range, final_range)
    'getSequenceStepTimeouts(&enables, &timeouts)
    getSequenceStepTimeouts(sens_idx, pre_range, pre_range_vcsel_period_pclks, msrc_dss_tcc_mclks, msrc_dss_tcc_us, pre_range_mclks, pre_range_us, final_range_vcsel_period_pclks, final_range_mclks, final_range_us)
    
    ' "Apply specific settings for the requested clock period"
    ' "Re-calculate and apply timeouts, in macro periods"
    
    ' "When the VCSEL period for the pre or final range is changed,
    ' the corresponding timeout must be read from the device using
    ' the current VCSEL period, then the new VCSEL period can be
    ' applied. The timeout then must be written back to the device
    ' using the new VCSEL period.
    '
    ' For the MSRC timeout, the same applies - this timeout being
    ' dependant on the pre-range vcsel period."
    
    const VcselPeriodPreRange=0, VcselPeriodFinalRange=1
    
    if type = VcselPeriodPreRange then
      ' "Set phase check limits"
      select case period_pclks
        case 12
          writeReg(sens_idx, &h57, &h18)
          
        case 14
          writeReg(sens_idx, &h57, &h30)
          
        case 16
          writeReg(sens_idx, &h57, &h40)
          
        case 18
          writeReg(sens_idx, &h57, &h50)
          
        case else
          ' invalid period
          setVcselPulsePeriod = 1
          exit function
      end select
      writeReg(sens_idx, &h56, &h08)
      
      ' apply new VCSEL period
      writeReg(sens_idx, &h50, vcsel_period_reg)
      
      ' update timeouts
      
      ' set_sequence_step_timeout() begin
      ' (SequenceStepId == VL53L0X_SEQUENCESTEP_PRE_RANGE)
      
      new_pre_range_timeout_mclks = timeoutMicrosecondsToMclks(pre_range_us, period_pclks)
      
      writeReg(sens_idx, &h51, encodeTimeout(new_pre_range_timeout_mclks), 2)
      
      ' set_sequence_step_timeout() end
      
      ' set_sequence_step_timeout() begin
      ' (SequenceStepId == VL53L0X_SEQUENCESTEP_MSRC)
      
      new_msrc_timeout_mclks = timeoutMicrosecondsToMclks(msrc_dss_tcc_us, period_pclks)
      
      if new_msrc_timeout_mclks > 256 then
        writeReg(sens_idx, &h46, 255)
      else
        writeReg(sens_idx, &h46, new_msrc_timeout_mclks - 1)
      end if
      
      ' set_sequence_step_timeout() end
    else if type = VcselPeriodFinalRange then
      select case period_pclks
        case 8
          writeReg(sens_idx, &h48, &h10)
          writeReg(sens_idx, &h47, &h08)
          writeReg(sens_idx, &h32, &h02)
          writeReg(sens_idx, &h30, &h0C)
          writeReg(sens_idx, &hFF, &h01)
          writeReg(sens_idx, &h30, &h30)
          writeReg(sens_idx, &hFF, &h00)
          
        case 10
          writeReg(sens_idx, &h48, &h28)
          writeReg(sens_idx, &h47, &h08)
          writeReg(sens_idx, &h32, &h03)
          writeReg(sens_idx, &h30, &h09)
          writeReg(sens_idx, &hFF, &h01)
          writeReg(sens_idx, &h30, &h20)
          writeReg(sens_idx, &hFF, &h00)
          
        case 12
          writeReg(sens_idx, &h48, &h38)
          writeReg(sens_idx, &h47, &h08)
          writeReg(sens_idx, &h32, &h03)
          writeReg(sens_idx, &h30, &h08)
          writeReg(sens_idx, &hFF, &h01)
          writeReg(sens_idx, &h30, &h20)
          writeReg(sens_idx, &hFF, &h00)
          
        case 14
          writeReg(sens_idx, &h48, &h48)
          writeReg(sens_idx, &h47, &h08)
          writeReg(sens_idx, &h32, &h03)
          writeReg(sens_idx, &h30, &h07)
          writeReg(sens_idx, &hFF, &h01)
          writeReg(sens_idx, &h30, &h20)
          writeReg(sens_idx, &hFF, &h00)
          
        case else
          ' invalid period
          setVcselPulsePeriod = 1
          exit function
      end select
      
      ' apply new VCSEL period
      writeReg(sens_idx, &h70, vcsel_period_reg)    'reg: FINAL_RANGE_CONFIG_VCSEL_PERIOD
      
      ' update timeouts
      
      ' set_sequence_step_timeout() begin
      ' (SequenceStepId == VL53L0X_SEQUENCESTEP_FINAL_RANGE)
      
      ' "For the final range timeout, the pre-range timeout
      '  must be added. To do this both final and pre-range
      '  timeouts must be expressed in macro periods MClks
      '  because they have different vcsel periods."
      
      new_final_range_timeout_mclks = timeoutMicrosecondsToMclks(final_range_us, period_pclks)
      
      if pre_range = 1 then
        new_final_range_timeout_mclks = new_final_range_timeout_mclks + pre_range_mclks
      end if
      
      writeReg(sens_idx, &h71, encodeTimeout(new_final_range_timeout_mclks),2) 'reg: FINAL_RANGE_CONFIG_TIMEOUT_MACROP_HI
      
      ' set_sequence_step_timeout end
    else
      ' invalid type
      setVcselPulsePeriod = 1
      exit function
    end if
    
    ' "Finally, the timing budget must be re-applied"
    
    stat = setMeasurementTimingBudget(sens_idx, VL53L0X_Meas_Tim_Budget_us(sens_idx))
    
    ' "Perform the phase calibration. This is needed after changing on vcsel period."
    ' VL53L0X_perform_phase_calibration() begin
    
    sequence_config = readReg(sens_idx, &h01)    'reg: SYSTEM_SEQUENCE_CONFIG
    writeReg(sens_idx, &h01, &h02)    'reg: SYSTEM_SEQUENCE_CONFIG
    stat = performSingleRefCalibration(&h00)
    writeReg(sens_idx, &h01, sequence_config)    'reg: SYSTEM_SEQUENCE_CONFIG
    
    ' VL53L0X_perform_phase_calibration() end
    setVcselPulsePeriod = 0
end function
    
    
    ' Encode VCSEL pulse period register value from period in PCLKs
    ' based on VL53L0X_encode_vcsel_period()
function encodeVcselPeriod(period_pclks as integer) as integer
    encodeVcselPeriod = (((period_pclks) >> 1) - 1)
end function
    '---------------------------------------
    
    '++++++++++++++++++++++++++++++++++++++
    
sub select_tof_mode (sens_idx as integer, tof_mode as integer)
    const VcselPeriodPreRange=0, VcselPeriodFinalRange=1
    local integer stat
    
    select case tof_mode
      case tof_long_range
        ' lower the return signal rate limit (default is 0.25 MCPS)
        stat = setSignalRateLimit(sens_idx, 0.1)
        ' increase laser pulse periods (defaults are 14 and 10 PCLKs)
        stat = setVcselPulsePeriod(sens_idx, VcselPeriodPreRange, 18)
        stat = setVcselPulsePeriod(sens_idx, VcselPeriodFinalRange, 14)
        
      case tof_high_speed
        ' reduce timing budget to 20 ms (default is about 33 ms)
        stat = setMeasurementTimingBudget(sens_idx, 20000)
        
      case tof_high_accuracy
        ' increase timing budget to 200 ms
        stat = setMeasurementTimingBudget(sens_idx, 200000)
        
      case tof_range_default
        stat = setSignalRateLimit(sens_idx, 0.25)
        
    end select
end sub
    '---------------------------------------