  '
  OPTION EXPLICIT
  OPTION DEFAULT NONE
  '
  'Program to display the radar data in list form or on graphical display
  'Radar sensor used is LD2450LD
  'Note: zone filtering is currently not implemented
  '
  '
  '************ HW dependencies, change as required ****************
  '
  '++++++ definition of serial line: RX/TX pins, com-number ++++++
  '       baudrate, buffersize, ISR name, buffersize for interrupt
  
  'config 1
  'setpin gp0,gp1,com1
  'const s_comspec = "com1:256000, 256, ser_int, 30"
  
  'config 2
  setpin gp20,gp21,com2
  const s_comspec = "com2:256000, 256, ser_int, 30"
  '****************** end dependencies *****************************
  
  '**********+ LD2450LD parameter definitions ***************
  '
  'definition of some length parameters for the sensor data exchange
  '+++++++++ record specs +++++++++++++++++++++++++++++++++++
  const data_rec_len = 24      'length of data record from radar
  const data_hdr_len = 4       'offset for 1st data byte in record
  const data_trail_len = 2      'Length of data record trailer
  
  const cmd_hdr_len = 4       'command header length
  const cmd_trail_len = 4     'command trailer length
  const cmd_rec_len = 0       'command length is variable
  
  const ack_hdr_len = 4       'ack header length
  const ack_trail_len = 4     'ack trailer length
  
  dim integer rec_len     'size of record to process
  dim integer hdr_len     'size of actual header to process
  dim integer trail_len   'size of actual trailer to ptocess
  '--------- end record specs
  
  'defintiton of command and reply strings for the sensor
  'the definition conforms to the operations manual of the LD2450LD sensor
  'in the actual sent/received packets data value low/high bytes are corrected to higb byte first
  'this is done in the send/receive routines
  '+++++++ definition of synch/ACK characters +++++++++++++++
  dim string syn_data_s = "AAFF0300", syn_data_e = "55CC"       'data header/trailer
  dim string syn_ack_s = "FDFCFBFA", syn_ack_e = "04030201"     'ACK header/trailer
  '-------- end synch chars def -----------------------------
  
  '+++++++ definition of command characters +++++++++++++++++
  dim string cmd_head = "FDFCFBFA", cmd_trail = "04030201"      'cmd header/trailer
  dim string cmd_m_track = "0090", cmd_s_track = "0080"         'cmd code to track multi/single object
  dim string cmd_beg_conf = "00FF0001", cmd_end_conf = "00FE"   'cmd begin/end configuration
  '-------- end command chars def ---------------------------
  '**********+ end LD2450LD parameter definitions ************
  
  '+++++++++ variables section ++++++++++++++++++++++++++++++
  dim string inchar, cons_char
  dim integer idx, stat, synch, targ
  dim float t_tmr, alpha, dist
  dim integer param_arry(3,4)                 'contains the radar measurements for 3 targets
  dim integer int_dis                         'if 1 serial line interrupts are ignored
  dim integer trk_mode                        '0 ... single target, 1 ... multi target
  dim integer dsp_enab, prt_enab, stat_enab   'if 1 display, print and statistics are enabled
  '--------- end variables ----------------------------------
  
  '++++++++++ eventflag definitions +++++++++++++++++++++++++
  dim integer  ef_clu                             '64 bit eventflag cluster
  const ser_ef = 0, radar_ef = 1, ser_lock = 2, cons_ef = 3    'eventflag numbers
  const keep_ef = 0, clear_ef = 1                 'used in ef sub/functions
  '---------- end eventflags --------------------------------
  
  '+++++++++ graphics definitions +++++++++++++++++++++++++++
  const v_fact = mm.vres/12000
  const h_fact = mm.hres/6000
  const x_0 = mm.hres/2, y_0 = mm.vres
  dim integer x_pos, y_pos, x_draw, y_draw
  '--------- end graphics -----------------------------------
  
  '+++++++++++ serial channel definitions ++++++++++++++++++++
  const s_chan = 1                        'serial channel number
  '------------ end serial channel ---------------------------
  
  '--------- end variables/constants ------------------------
  
  '++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  '+++++++++++ Begin of executable code +++++++++++++++++++++
  '++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  
  '++++++++ init section ++++++++++++++++++++++++++++++++++++
  int_dis = 1       'serial data are handled synchronous not by interrupt for setting up the radar
  
  open s_comspec as s_chan
  'inchar = input$(255,  s_chan)     'purge input buffer
  
  'setup and initialize sensor
  stat = init_sensor (s_chan, cmd_m_track)
  trk_mode = 1        'tracking mode is multi
  if stat <> 0 then
    print "%Radar-F-Init, init failed with error ";stat, bin_str_to_hex(inchar)
end           'end program if sensor init fails
  end if
  
  'now we enable interrupt processing on serial line to drive the processing
  int_dis = 0
  
  on key char_isr             'enable character interrupts from console
  
  'begin neasurement
  start_measure()
  
  'say hello to the outside world
  print
  print "%Radar-I-Start, measurement starts"
  print
  
  'simulate console input to display help text
  cons_char = "H"
  cons_menue
  
  '----------------------------------------------------------
  '--------------- end init section -------------------------
  '----------------------------------------------------------
  
  'processing of sensor data is as follows:
  ' serial interrupr sets ser_ef
  ' if ser_ef is set, the data are parsed/checked and computed into param_arry()
  ' if parsing is complete, ser_ef is reset and radar_ef is set to signal new data is available
  'if radar_ef is set, the data are just printed out
  
  '++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  '+++++++++++++++ main code ++++++++++++++++++++++++++++++++
  '++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  
  do                            'main loop
    '++++++++++++++++ begin idle loop +++++++++++++++++++++++
    do while not test_ef(ef_clu, ser_ef)         'idle loop, wait for serial eventflag to be set
      'idle time statstics
      set_exec_timer(0)       'start timer for idle loop
      
      'time statistics of radar data processing
      t_tmr = get_exec_timer(1)
      if t_tmr <> 0 and stat_enab = 1 then print "radar tmr "; t_tmr      'print timer for radar processing
      
      '++++++ process data from radar measurement +++++++++++++
      'currently we just calculate distance and angle for taget 0 and print them out
      
      do_exec_ef(ef_clu, radar_ef, "proc_data", clear_ef)       'execute UAR if ef is set
      do_exec_ef(ef_clu, cons_ef, "cons_menue", clear_ef)       'execute console menue if ef is set
      
      '------ end radar data processing -----------------------
      
      '+++++
      '----> execute here your other code but do not exceed about 86 mSec runtime
      '----> otherwise you may lose some measurements
      '-----
      
    loop
    '---------------- end idle loop -------------------------
    
    'arrive here if sensor data received and thus ser_ef was set
    
    'statistics for idle loop
    if stat_enab then print "idle tmr "; get_exec_timer(0)      'print timer for idle loop
    
    clr_ef(ef_clu, ser_ef)                  'reset serial eventflag
    set_exec_timer(1)          'start timer for radar processing
    
    'get in synch with the record header if not already established
    stat = check_synch(inchar, syn_data_s)       'check if in synch
    if stat then do_synch s_chan, inchar, rec_len, hdr_len, syn_data_s   're-synch if not
    
    'calculate radar data
    for targ = 0 to 2
      calc_val(param_arry(), targ, inchar, hdr_len)      'compute values
    next targ
    
    set_ef (ef_clu, radar_ef)        'signal new radar data available
  loop
  '----------------------------------------------
  '-------------- end main code -----------------
  '----------------------------------------------
  
  '+++++++++++ DEV AREA +++++++++++++++++++
  
  '+++ ISR for console character input ++++
sub char_isr
  do
    cons_char = ucase$(inkey$)
  loop until cons_char <> ""
  set_ef (ef_clu, cons_ef)        'signal console input available
end sub

sub cons_menue
  select case cons_char
    case "D"
      dsp_enab = not dsp_enab       'toggle flag
      cls
    case "P"
      prt_enab = not prt_enab       'toggle flag
    case "S"
      stat_enab = not stat_enab     'toggle flag
    case "X"
end
    case "1"
      stat = init_sensor (s_chan, cmd_s_track)
      trk_mode = 0        'tracking mode is single
      print "%Radar-I-mode, mode set to single"
    case "M"
      stat = init_sensor (s_chan, cmd_m_track)
      print "%Radar-I-mode, mode set to multi"
      trk_mode = 1        'tracking mode is multi
    case "H"
      print "+++++++++++"
      print "D ... enable display"
      print "P ... print data on console"
      print "S ... print timer statistics on console"
      print "1 ... single target tracking"
      print "M ... multi-target tracking"
      print "X ... Exit"
      print "H ... this help text"
      print "-----------"
    case else
      print cons_char;" is not a valid command ...."
  end select
end sub
  '----------------------------------------------------
  
  '---------- END DEV AREA ----------------
  
  '++++++++++ SUBs and FUNCTIONs ++++++++++++++++
sub proc_data
  local integer idx
  'processing of the radar data handed over in param_arry()
  
  for idx = 0 to 2
    if param_arry(idx,0) <> 0 then                      'check if meaningful data are available
      alpha = 180 + atn(param_arry(idx,1)/param_arry(idx,0))*360/pi   'calc distance vector angle
      dist = sqr(param_arry(idx,0)^2 + param_arry(idx,1)^2)           'calc vector distance
    end if
  next idx
  
  if prt_enab then print_data    'print data on console
  if dsp_enab then draw_data     'draw position on display
end sub
  '----------------------------------------------------
  
sub draw_data
  circle x_draw, y_draw, 5, 1, 1, rgb(black), rgb(black)      'delete old position
  if param_arry(0,0) <> 0 then
    x_pos = param_arry(0,0)*h_fact      'scale to diaplay
    y_pos = param_arry(0,1)*v_fact
    x_draw = x_0 + x_pos                'get display position
    y_draw = y_0 - y_pos
    circle x_draw, y_draw, 5, 1, 1, rgb(white), rgb(white)    'draw new position
  end if
end sub
  '----------------------------------------------------
  
sub print_data
  local integer idx
  for idx = 0 to 2
    if param_arry(idx,0) <> 0 then                      'check if printout possible
      print "TG: ";idx, param_arry(idx,0), param_arry(idx,1), param_arry(idx,2), param_arry(idx,3),    'print results
      print format$(alpha,"% 3.0f"), format$(dist,"% 4.0f")
    end if
  next idx
end sub
  '----------------------------------------------------
  
sub do_exec_ef(ef_clstr as integer, ef_no as integer, uar as string, bit_reset as integer)
  'ef_clstr ... eventflag cluster as integer 64 bit
  'ef_no ... eventflag number 0 ... 63
  'bit_reset: 0 ... flag left unchanged, 1 ... reset flag after test
  if bit(ef_clstr, ef_no) = 1 then execute uar     'execute user-action-routine if ef is set
  if bit_reset = 1 then bit(ef_clstr, ef_no) = 0      'reset ef if requried
end sub
  '----------------------------------------------------
  
sub start_measure()
  'setup params for standard measurement records
  rec_len = data_hdr_len + data_rec_len + data_trail_len
  hdr_len = data_hdr_len
  trail_len = data_trail_len
  
  'synch to data header
  do_synch s_chan, inchar, rec_len, hdr_len, syn_data_s
end sub
  '----------------------------------------------------
  
function init_sensor(s_chan as integer, init_mode as string) as integer
  'ini sensor by sending begin-config, mode, end-config
  'function return codes are:
  ' 0 ... success
  ' 1 ... begin config failed
  ' 2 ... mode command failed
  ' 3 ... end config failed
  
  rec_len = data_hdr_len + data_rec_len + data_trail_len    'use as warning limit in ISR
  init_sensor = 0       'mark success
  
  'send begin configuration
  inc init_sensor       'mark first failure value
  stat = send_cmd(s_chan, cmd_head, cmd_beg_conf, cmd_trail)
  stat = get_ack(s_chan, inchar)
  if stat then
    exit function
  end if
  
  'send mode command
  inc init_sensor       'mark next failure value
  stat = send_cmd(s_chan, cmd_head, init_mode, cmd_trail)
  stat = get_ack(s_chan, inchar)
  if stat then
    exit function
  end if
  
  inc init_sensor       'mark next failure value
  stat = send_cmd(s_chan, cmd_head, cmd_end_conf, cmd_trail)
  stat = get_ack(s_chan, inchar)
  if stat then
    exit function
  end if
  init_sensor = 0       'mark success
end function
  '----------------------------------------------------
  
Function get_ack(s_chan as integer, ack_p as string) as integer
  local integer ack_size
  local string ack_s1, ack_s2
  
  do_synch s_chan, inchar, 0, ack_hdr_len, syn_ack_s        'wait for synch chars
  
  do while loc(s_chan) < 2        'make sure the ack length chars are already here
  loop
  
  'assemble ack-length
  ack_s1 = input$(1, s_chan)        'get low byte
  ack_s2 = input$(1, s_chan)        'get high byte
  ack_size = (asc(ack_s2)<<8) + asc(ack_s1) 'convert to integer
  
  do while loc(s_chan) < ack_size         'make sure the whole ack-message is ready
  loop
  ack_p = input$(ack_size, s_chan)       'get ack message part between length bytes and trailer
  
  get_ack = 0         'set funtion body to 0
  
  'get ack return value 0 ... success, 1 ... failure into the function body
  'assemble high ack-byte
  ack_s1 = mid$(ack_p,11,1)              'get ack value low byte 2
  ack_s2 = mid$(ack_p,12,1)              'get ack value high byte 2
  get_ack = (asc(ack_s1) + (asc(ack_s2)<<4))<<8     'set high byte
  
  'assemble low ack-byte
  ack_s1 = mid$(ack_p,9,1)              'get ack value low byte 1
  ack_s2 = mid$(ack_p,10,1)              'get ack value high byte 1
  get_ack = get_ack + asc(ack_s1) + (asc(ack_s2)<<4)    'assign full ack-value
  
  'make sure the trailer is here but do we not check it
  do while loc(s_chan) < ack_trail_len
  loop
  ack_s1 = input$(ack_trail_len, s_chan)   'get ack trailer
  
end function
  '----------------------------------------------------
  
function send_cmd(s_chan as integer, cmd_head as string, cmd_str as string, cmd_trail as string) as integer
  'send command to sensor
  'command is as string of hex bytes as defined in the manual, commands are reformatted to low byte first
  'header and trailer are strings and left unchanged to comply with the manual
  local string outstr, in_str
  outstr = hex_to_bin_str(cmd_head)                                     'add header
  outstr = outstr + hex_to_bin_str(sw_byte(hex$(len(cmd_str)/2,4)))     'add command/data length
  outstr = outstr + hex_to_bin_str(sw_byte(cmd_str))                    'add command/data
  outstr = outstr + hex_to_bin_str(cmd_trail)                           'add trailer
  print #s_chan, outstr;                                                'send data to sensor
end function
  '----------------------------------------------------
  
function sw_byte(in_str as string) as string
  'swap bytes in hex format "0102" to "0201"
  local integer idx
  sw_byte = ""
  for idx = 1 to len(in_str) step 4
    sw_byte = sw_byte + mid$(in_str,idx+2,2) + mid$(in_str,idx,2)
  next idx
end function
  '----------------------------------------------------
  
function bin_str_to_hex (bin_str as string) as string
  local integer idx
  bin_str_to_hex = ""
  for idx = 1 to len(bin_str)
    bin_str_to_hex = bin_str_to_hex + hex$(asc(mid$(bin_str,idx,1)),2)
  next idx
end function
  '----------------------------------------------------
  
function hex_to_bin_str(in_str as string) as string
  'convert 2 chars of in_str in hex into single char in out_str
  local integer idx, t_int1, t_int2
  local string t_char1, t_char2
  
  hex_to_bin_str = ""
  for idx = 1 to len(in_str) step 2
    t_int1 = val("&h"+mid$(in_str,idx,1))
    t_int2 = val("&h"+mid$(in_str,idx+1,1))
    t_int1 = (t_int1<<4) + t_int2
    hex_to_bin_str = hex_to_bin_str + chr$(t_int1)
  next idx
end function
  '----------------------------------------------------
  
sub set_ef (ef_clstr as integer, ef_no as integer)
  bit(ef_clstr, ef_no) = 1
end sub
  '----------------------------------------------------
  
sub clr_ef (ef_clstr as integer, ef_no as integer)
  bit(ef_clstr, ef_no) = 0
end sub
  '----------------------------------------------------
  
function test_ef (ef_clstr as integer, ef_no as integer, bit_reset as integer) as integer
  'ef_clstr ... eventflag cluster as integer 64 bit
  'ef_no ... eventflag number 0 ... 63
  'bit_reset: 0 ... flag left unchanged, 1 ... reset flag after test
  
  test_ef = bit(ef_clstr, ef_no)      'get ef value
  if bit_reset = 1 then bit(ef_clstr, ef_no) = 0      'reset ef if requried
end function
  '----------------------------------------------------
  
sub waitfr_ef(byref ef_clstr as integer, ef_no as integer)
  'test if any ef specified in mask is set, otherwise do spinwait
  do while bit(ef_clstr, ef_no) = 0
  loop
end sub
  '----------------------------------------------------
  
sub set_exec_timer (t_nbr as integer)
  'start timer 0 to 9 if read out before (timer-value = 0)
  on error skip 1
  dim float __exec_tmr(10)
  if __exec_tmr(t_nbr) = 0 then __exec_tmr(t_nbr) = timer
end sub
  '----------------------------------------------------
  
function get_exec_timer (t_nbr as integer) as float
  'get timer-value if non-zero and reset timer-value to 0
  if __exec_tmr(t_nbr) <> 0 then
    get_exec_timer  = timer  - __exec_tmr(t_nbr)
    __exec_tmr(t_nbr) = 0
  end if
end function
  '----------------------------------------------------
  
sub calc_val (param_arry() as integer, tg as integer, in_str as string, hdr_len as integer)
  local integer ofs
  'calculate values for target
  'param_arry:
  'first index is target number (0 ... 2) passed in tg in the call
  'second index is target value:
  '  0 ... x-distance
  '  1 ... y-distance
  '  2 ... velocity
  '  3 ... resolution
  set_ef(ef_clu, ser_lock)          'lock datastructure
  
  ofs = tg*8            'offset within record to each target data block
  
  '++++++++++++ get X-value
  param_arry(tg,0) = asc(mid$(in_str, hdr_len+ofs+1, 1)) + (asc(mid$(in_str, hdr_len+ofs+2, 1))<<8)
  'correction of x-value to pos/neg values
  if (asc(mid$(in_str, hdr_len+ofs+2, 1)) and &h80) then
    param_arry(tg,0) = &h8000 - param_arry(tg,0)
  end if
  
  'compute y, velocity and resolution only if x-value is valid
  if param_arry(tg,0) <> 0 then       'if non-zero x-value, compute rest
    '++++++++++++ get Y-value
    param_arry(tg,1) = asc(mid$(in_str, hdr_len+ofs+3, 1)) + (asc(mid$(in_str, hdr_len+ofs+4, 1))<<8)
    'correction of y-value
    param_arry(tg,1) = param_arry(tg,1) - &h8000
    
    '++++++++++++ get velocity
    param_arry(tg,2) = asc(mid$(in_str, hdr_len+ofs+5, 1)) + (asc(mid$(in_str, hdr_len+ofs+6, 1))<<8)
    'correction of velocity value
    if (asc(mid$(in_str, hdr_len+ofs+6, 1)) and &h80) then
      param_arry(tg,2) = &h8000 - param_arry(tg,2)
    end if
    
    '++++++++++++ get resolution
    param_arry(tg,3) = asc(mid$(in_str, hdr_len+ofs+7, 1)) + (asc(mid$(in_str, hdr_len+ofs+8, 1))<<8)
  end if
  
  clr_ef(ef_clu, ser_lock)          'release datastructure
end sub
  '----------------------------------------------------
  
sub do_synch (in_chan as integer, in_str as string, rec_len as integer, hdr_len as integer, syn_data_s as string)
  'synch on the defined header bytes
  local integer t_synch = 0, idx
  
  do while t_synch = 0                              'loop while out-of-sync
    do while hex$(asc(input$(1,in_chan)),2) <> mid$(syn_data_s, 1, 2)      'synch to first char
    loop
    
    for idx = 3 to hdr_len*2-1 step 2                   'for all defined synch char tuples
      if hex$(asc(input$(1,in_chan)),2) <> mid$(syn_data_s, idx, 2) then     'synch to next chars
        t_synch = 0                                   'mark not-in-sync
        exit for                                      'if not in sequence, exit
      else
        t_synch = 1                                   'mark in-sync
      end if
    next idx                                          'try next char
  loop
  
  if rec_len > 0 then
    do while loc(in_chan) < rec_len-hdr_len        'wait for remaining chars after header
    loop
    in_str = input$(rec_len-hdr_len,in_chan)       'and throw away the rest of this message
  end if
  print "%Radar-I-Sync, Sync established"
end sub
  '----------------------------------------------------
  
function check_synch (in_str as string, syn_data as string) as integer
  local integer idx
  
  for idx = 1 to hdr_len
    if hex$(Asc(mid$(inchar, idx, 1)),2) = mid$(syn_data, idx*2-1 ,2) then     'synch to next chars
      check_synch = 0
    else
      check_synch = 1
      exit for
    end if
  next idx
  
  if check_synch = 1 then print "%Radar-W-Sync, Sync lost"
  
end function
  '----------------------------------------------------
  
  'conveniant sub to print out binary coded strings in hex format
  'used for debugging
sub prt_str_hex(in_str as string)
  local integer idx
  for idx = 1 to len(in_str)
    print hex$(asc(mid$(in_str, idx, 1)),2);
  next idx
  print
end sub
  '----------------------------------------------------
  
  '++++++++++ Interrupt service routines (ISR) +++++++++
  
  '++++++++ serial line interrupt ++++++++++++++++++++++
sub ser_int
  'if recordbuffer exceeds 1 record print warning. This is not necessarily a problem
  if loc(s_chan) > rec_len then print "%Radar-W-Buffsz, input buffer size ";loc(s_chan)
  inchar = input$(rec_len,s_chan)         'get record
  ' it is possible to lock the datastructure with this semaphore
  if not int_dis then                       'if interrupts on serial are enabled
    if not test_ef(ef_clu, ser_lock) then               'if lock-flag set, datastructures are locked
      set_ef(ef_clu, ser_ef)                            'set eventflag only if lock-flag = 0
    else
      print "%Radar-W-Serial_ISR, Structure lock, data lost"
    end if
  end if
end sub
  '----------------------------------------------------
