  '***********************************
  '*          radar target visualisation for LD2450 / RD-03D sensors
  '*
  '* partially from gerald at thebackshed-forum
  '***********************************
  
  'PicoMite MMBasic RP2040 Edition V6.00.03
  'OPTION SYSTEM SPI GP6,GP3,GP4
  'OPTION COLOURCODE ON
  'OPTION CPUSPEED (KHz) 200000
  'OPTION LCDPANEL ILI9341, LANDSCAPE,GP7,GP2,GP5
  '
  
  OPTION EXPLICIT
  OPTION DEFAULT NONE
  option escape
  
  cls
  dim string vers
  vers = "Radar-V2.0.3"
  
  dim string uars, uars1
  '
  'Program to display the radar data in list form or on graphical display
  'Radar sensor used is LD2450
  'Note: zone filtering is currently not implemented
  '
  '
  '************ HW dependencies, change as required ****************
  '
  '++++++ definition of serial line: RX/TX pins, com-number ++++++
  '       baudrate, buffersize, ISR name, buffersize for interrupt
  
  'config 1
  'setpin gp0,gp1,com1
  'const s_comspec = "com1:256000, 256, ser_int, 30"
  
  'config 2
  setpin gp20,gp21,com2
  const s_comspec = "com2:256000, 256, ser_int, 30"
  
  '+++++++++++++ Touch interrupt pin +++++++++++++++++++++++++++++
  const touch_pin = "GP9"    'define the Touch Interrupt Pin
  
  '****************** end dependencies *****************************
  
  
  '**********+ LD2450LD parameter definitions ***************
  '
  'definition of some length parameters for the sensor data exchange
  '+++++++++ record specs +++++++++++++++++++++++++++++++++++
  const data_rec_len = 24      'length of data record from radar
  const data_hdr_len = 4       'offset for 1st data byte in record
  const data_trail_len = 2      'Length of data record trailer
  
  const cmd_hdr_len = 4       'command header length
  const cmd_trail_len = 4     'command trailer length
  const cmd_rec_len = 0       'command length is variable
  
  const ack_hdr_len = 4       'ack header length
  const ack_trail_len = 4     'ack trailer length
  
  dim integer rec_len     'size of record to process
  dim integer hdr_len     'size of actual header to process
  dim integer trail_len   'size of actual trailer to ptocess
  '--------- end record specs
  
  'defintiton of command and reply strings for the sensor
  'the definition conforms to the operations manual of the LD2450LD sensor
  'in the actual sent/received packets data value low/high bytes are corrected to higb byte first
  'this is done in the send/receive routines
  '+++++++ definition of synch/ACK characters +++++++++++++++
  dim string syn_data_s = "AAFF0300", syn_data_e = "55CC"       'data header/trailer
  dim string syn_ack_s = "FDFCFBFA", syn_ack_e = "04030201"     'ACK header/trailer
  '-------- end synch chars def -----------------------------
  
  '+++++++ definition of command characters +++++++++++++++++
  dim string cmd_head = "FDFCFBFA", cmd_trail = "04030201"      'cmd header/trailer
  dim string cmd_m_track = "0090", cmd_s_track = "0080"         'cmd code to track multi/single object
  dim string cmd_beg_conf = "00FF0001", cmd_end_conf = "00FE"   'cmd begin/end configuration
  '-------- end command chars def ---------------------------
  '**********+ end LD2450LD parameter definitions ************
  
  '+++++++++ variables section ++++++++++++++++++++++++++++++
  dim string inchar, cons_char
  dim integer idx, stat, synch, targ
  dim float t_tmr, alpha, dist
  dim integer param_arry(3,4)                 'contains the radar measurements for 3 targets
  'dim integer trk_mode                        '0 ... single target, 1 ... multi target
  'dim integer dsp_enab, prt_enab, stat_enab   'if 1 display, print and statistics are enabled
  '--------- end variables ----------------------------------
  
  '++++++++++ eventflag definitions +++++++++++++++++++++++++
  dim integer  ef_clu                             '64 bit eventflag cluster
  const ser_ef = 0, radar_ef = 1, ser_lock = 2, cons_ef = 3    'eventflag numbers
  const keep_ef = 0, clear_ef = 1                 'used in ef sub/functions
  '---------- end eventflags --------------------------------
  
  '+++++++++ flags definitions ++++++++++++++++++++++++++++++
  dim integer fl_clu
  const int_dis = 1                                 'if set serial line interrupts are ignored
  const dsp_enab = 2, prt_enab = 3, stat_enab = 4   'if set display, print and statistics are enabled
  const trk_mode = 5                                'clear ... single target, set ... multi target
  const z_mode = 6                                  '0 ... suppress / 1 ... eanble zero x-axis tuples
  
  '+++++++++ graphics definitions +++++++++++++++++++++++++++
  const h_mm = 8000               'define horizontal max range in mm
  const v_mm = h_mm               'define vertical max range in mm
  const grid_col = rgb(green)     'define grid color (arcs and lines)
  const grid_txt = rgb(yellow)    'define grid text color
  const grid_unit = " mm"         'define grid text unit
  const v_fact = mm.vres/v_mm     'hor scaling factor in pixel/mm
  const h_fact = mm.hres/h_mm     'vert scaling factor in pixel/mm
  const x_0 = mm.hres/2, y_0 = mm.vres    'origin for the radar grid relativ to display coordinate system
  dim integer x_pos, y_pos, x_draw(3), y_draw(3)
  
  dim integer x_res, y_res
  dim float lin(5), bordr(4), f_idx
  'define border lines via x or y offset only
  bordr(0) = -mm.hres/2   'left border
  bordr(1) = mm.hres/2    'right border
  bordr(2) = mm.vres      'top border
  bordr(3) = 0            'bottom border
  '--------- end graphics -----------------------------------
  
  '+++++++++ PMG stuff ++++++++++++++++++++++++++++++++++++++
  'define number of GUI objects
  const max_pmg_obj = 10
  
  dim integer obj_stat(max_pmg_obj)   'status array for objects
  dim integer n_lin, n_obj_lin, x_sz, y_sz, x_spc, y_spc
  dim integer wid, hei, y_offs
  '------------ end PMG stuff --------------------------------
  
  '+++++++++++ serial channel definitions ++++++++++++++++++++
  const s_chan = 1                        'serial channel number
  '------------ end serial channel ---------------------------
  
  pmg_init(max_pmg_obj, touch_pin)    'init internal PMG data structures and the Touch Interrupt Pin
  pmg_get_size(6, 8, n_lin, n_obj_lin, x_sz, y_sz, x_spc, y_spc)
  y_offs = mm.vres/2*0.8          'general y offset
  
  cls
  pmg_rbox(0, 0, 0, 60, 20, -5, rgb(white), rgb(myrtle),"Menue")   'BOX with rounded corners
  draw_grid
  
  font 7:text 0,mm.vres-mm.fontheight,vers  'shows version in left bottom corner
  
  '--------- end variables/constants ------------------------
  
  '++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  '+++++++++++ Begin of executable code +++++++++++++++++++++
  '++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  
  '++++++++ init section ++++++++++++++++++++++++++++++++++++
  set_ef(fl_clu,int_dis)       'serial data are handled synchronous not by interrupt for setting up the radar
  set_ef(fl_clu,dsp_enab)
  
  
  
  open s_comspec as s_chan
  'inchar = input$(255,  s_chan)     'purge input buffer
  
  'stat = init_sensor (s_chan, cmd_s_track)
  'clr_ef(fl_clu,trk_mode)        'tracking mode is single
  
  'setup and initialize sensor
  stat = init_sensor (s_chan, cmd_s_track)
  set_ef(fl_clu,trk_mode)        'tracking mode is single
  if stat <> 0 then
    print "%Radar-F-Init, init failed with error ";stat, bin_str_to_hex(inchar)
end           'end program if sensor init fails
  end if
  
  'now we enable interrupt processing on serial line to drive the processing
  clr_ef(fl_clu,int_dis)
  
  on key char_isr             'enable character interrupts from console
  
  'begin neasurement
  start_measure()
  
  'say hello to the outside world
  print
  print "%Radar-I-Start, measurement starts"
  print
  
  'simulate console input to display help text
  cons_char = "H"
  cons_menue
  
  '----------------------------------------------------------
  '--------------- end init section -------------------------
  '----------------------------------------------------------
  
  'processing of sensor data is as follows:
  ' serial interrupr sets ser_ef
  ' if ser_ef is set, the data are parsed/checked and computed into param_arry()
  ' if parsing is complete, ser_ef is reset and radar_ef is set to signal new data is available
  'if radar_ef is set, the data are just printed out
  
  '++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  '+++++++++++++++ main code ++++++++++++++++++++++++++++++++
  '++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  
  do                            'main loop
    '++++++++++++++++ begin idle loop +++++++++++++++++++++++
    do while not test_ef(ef_clu, ser_ef)         'idle loop, wait for serial eventflag to be set
      'idle time statstics
      set_exec_timer(0)       'start timer for idle loop
      
      'time statistics of radar data processing
      t_tmr = get_exec_timer(1)
      if t_tmr <> 0 and test_ef(fl_clu, stat_enab) = 1 then print "radar tmr "; t_tmr      'print timer for radar processing
      
      '++++++ process data from radar measurement +++++++++++++
      'currently we just calculate distance and angle for taget 0 and print them out
      
      do_exec_ef(ef_clu, radar_ef, "proc_data", 1, clear_ef)       'execute UAR if ef is set
      do_exec_ef(ef_clu, cons_ef, "cons_menue", 1, clear_ef)       'execute console menue if ef is set
      
      '------ end radar data processing -----------------------
      
      '++++++++ menue/submenue handler via PMG routines +++++++
      if __pmg_obj_t <> -1 then
        select case __pmg_obj_t
          case 0        'menue object, draw setup submenue
            clr_ef(fl_clu,dsp_enab)       'disable ball display
            cls                           'clear screen
            
            'draw menue boxes
            x_pos = 0
            y_pos = 0
            wid = x_sz      'width
            hei = y_sz      'heigth
            pmg_next_pos(x_pos, y_pos, x_sz, y_sz, x_spc, y_spc, n_obj_lin, n_lin)  'skip first box position
            pmg_rbox(1, x_pos, y_offs+y_pos, wid, hei, -5, rgb(white), rgb(myrtle),"Single")   'BOX with rounded corners
            pmg_next_pos(x_pos, y_pos, x_sz, y_sz, x_spc, y_spc, n_obj_lin, n_lin)    'calc next position
            pmg_rbox(2, x_pos, y_offs+y_pos, wid, hei, -5, rgb(white), rgb(myrtle),"Multi")   'BOX with rounded corners
            pmg_next_pos(x_pos, y_pos, x_sz, y_sz, x_spc, y_spc, n_obj_lin, n_lin)    'calc next position
            pmg_rbox(3, x_pos, y_offs+y_pos, wid, hei, -5, rgb(white), rgb(myrtle),"Print")   'BOX with rounded corners
            pmg_next_pos(x_pos, y_pos, x_sz, y_sz, x_spc, y_spc, n_obj_lin, n_lin)    'calc next position
            pmg_next_pos(x_pos, y_pos, x_sz, y_sz, x_spc, y_spc, n_obj_lin, n_lin)    'skip next position
            pmg_rbox(9, x_pos, y_offs+y_pos, wid, hei, -5, rgb(white), rgb(myrtle),"Exit")   'BOX with rounded corners
            
          case 1          'single target
            cons_char = "1"
            cons_menue
          case 2          'multi target
            cons_char = "M"
            cons_menue
          case 3          'print
            cons_char = "P"
            cons_menue
            if test_ef(fl_clu,prt_enab) then
              pmg_rbox(3,,,,,,,,"Pr on ")     'redraw "print" menue with changed annotation
            else
              pmg_rbox(3,,,,,,,,"Pr off")     'redraw "print" menue with changed annotation
            end if
          case 4
          case 5
          case 6
          case 7
          case 8
            
          case 9          'exit the settings submenue
            cls
            pmg_rbox(0,,,,,,,,"Menue")   'redraw menue button
            draw_grid
            font 7:text 0,mm.vres-mm.fontheight,vers  'shows version in left bottom corner
            set_ef(fl_clu,dsp_enab)   'enable ball display
        end select
        
        'reset touch object id
        __pmg_obj_t = -1
      end if
      '------------ end menues handler -----------------------------------------
      
      '+++++
      '----> execute here your other code but do not exceed about 86 mSec runtime
      '----> otherwise you may lose some measurements
      '-----
      
    loop
    '---------------- end idle loop -------------------------
    
    'arrive here if sensor data received and thus ser_ef was set
    
    'statistics for idle loop
    if test_ef(fl_clu, stat_enab) then print "idle tmr "; get_exec_timer(0)      'print timer for idle loop
    
    clr_ef(ef_clu, ser_ef)                  'reset serial eventflag
    set_exec_timer(1)          'start timer for radar processing
    
    'get in synch with the record header if not already established
    stat = check_synch(inchar, syn_data_s)       'check if in synch
    if stat then do_synch s_chan, inchar, rec_len, hdr_len, syn_data_s   're-synch if not
    
    'calculate radar data
    for targ = 0 to 2
      calc_val(param_arry(), targ, inchar, hdr_len)      'compute values
    next targ
    
    set_ef (ef_clu, radar_ef)        'signal new radar data available
  loop
  '----------------------------------------------
  '-------------- end main code -----------------
  '----------------------------------------------
  
  '+++++++++++ DEV AREA +++++++++++++++++++
  
sub draw_grid
  local float txt_mark
  local string circ_text
  
  'draw lines
  for f_idx = -157.5 to -22.5 step 22.5   'select line angles
    lin(0) = 0                            'line origin is x/y 0,0
    lin(1) = 0
    lin(2) = mm.hres/2*cos(pi/180*f_idx)  'line end x, max x is mm.hre/2
    lin(3) = mm.vres*sin(pi/180*f_idx)    'line end y, may y is mm.vres
    
    'get intercept point of line with border of display
    get_intercept(lin(), bordr(), x_res, y_res)
    
    'draw line from origin to border
    'x origin is mm.hres/2
    'y origin is mm.vres
    'this gives an origin of bottom-middle
    line mm.hres/2,mm.vres, x_res+(mm.hres/2), mm.vres-y_res,,Rgb(green)
  next f_idx
  
  'draw arcs
  for f_idx = 1 to 4
    txt_mark = v_mm/4*(f_idx)
    circle mm.hres/2,mm.vres,mm.vres*f_idx/4,,,grid_col
    text mm.hres/2,mm.vres*(4-f_idx)/4+(mm.fontheight*0.7),format$(txt_mark,"%g")+grid_unit,"CB",7,1,grid_txt
  next f_idx
  
end sub
  
sub get_intercept(lin() as float, bordr()as float, x_pos as integer, y_pos as integer)
  local float x1, y1, k1, dx1, dy1
  
  'common calculations
  dx1 = lin(2)-lin(0)   'delta x
  dy1 = lin(3)-lin(1)   'delta y
  if dx1 = 0 then dx1 = 1e-10   'prevent divide by zero error
  k1 = dy1/dx1    'slope
  if k1 = 0 then k1=1e-10       'prevent divide by zero error
  
  'specific calculations
  
  'x/y of intercept point with top border
  y1 = bordr(2)        'upper border y value
  x1 = (y1-lin(4))/k1 'intercept point x-coordinate
  y1 = k1*x1+lin(4)   'y-coordinate, check only
  x_pos = x1
  y_pos = y1
  if x_pos >= bordr(0) and x_pos <= bordr(1) then exit sub
  
  'x/y of intercept point with right border
  x1 = bordr(1)        'right border x value
  y1 = k1*x1+lin(4)   'intercept point, y-coordinate
  x1 = (y1-lin(4))/k1 'x-coordinate, check only
  x_pos = x1
  y_pos = y1
  if y_pos >= bordr(3) and y_pos <= bordr(2) then exit sub
  
  'x/y of intercept point with left border
  x1 = bordr(0)        'left border x value
  y1 = k1*x1+lin(4)   'intercept point, y-coordinate
  x1 = (y1-lin(4))/k1 'x-coordinate, check only
  x_pos = x1
  y_pos = y1
  if y_pos >= bordr(3) and y_pos <= bordr(2) then exit sub
  
  'x/y of intercept point with bottom border
  y1 = bordr(3)        'lower border y value
  x1 = (y1-lin(4))/k1 'intercept point x-coordinate
  y1 = k1*x1+lin(4)   'y-coordinate, check only
  x_pos = x1
  y_pos = y1
  
end sub
  
  '+++ ISR for console character input ++++
sub char_isr
  do
    cons_char = ucase$(inkey$)
  loop until cons_char <> ""
  set_ef (ef_clu, cons_ef)        'signal console input available
end sub
  
sub cons_menue
  if asc(cons_char) < 32 then exit sub    'discard control chars
  select case cons_char
    case "D"
      toggle_ef(fl_clu,dsp_enab)       'toggle flag
      do_exec_ef(fl_clu, dsp_enab, "print \qD on\q", 1, keep_ef)
      do_exec_ef(fl_clu, dsp_enab, "print \qD off\q", 0, keep_ef)
      'cls
    case "P"
      toggle_ef(fl_clu,prt_enab)       'toggle flag
      do_exec_ef(fl_clu, prt_enab, "print \qP on\q", 1, keep_ef)
      do_exec_ef(fl_clu, prt_enab, "print \qP off\q", 0, keep_ef)
    case "S"
      toggle_ef(fl_clu,stat_enab)       'toggle flag
      do_exec_ef(fl_clu, stat_enab, "print \qS on\q", 1, keep_ef)
      do_exec_ef(fl_clu, stat_enab, "print \qS off\q", 0, keep_ef)
    case "X"
end
    case "1"
      print "%Radar-I-mode, mode set to single"
      stat = init_sensor (s_chan, cmd_s_track)
      clr_ef(fl_clu,trk_mode)        'tracking mode is single
    case "M"
      stat = init_sensor (s_chan, cmd_m_track)
      print "%Radar-I-mode, mode set to multi"
      set_ef(ef_clu,trk_mode)        'tracking mode is multi
    case "Z"
      toggle_ef(fl_clu,z_mode)       'toggle flag
      do_exec_ef(fl_clu, z_mode, "print \qZ on\q", 1, keep_ef)
      do_exec_ef(fl_clu, z_mode, "print \qZ off\q", 0, keep_ef)
    case "R"                        'redraw grid
      cls
      draw_grid
      font 7:text 0,mm.vres-mm.fontheight,vers  'shows version in left bottom corner
    case "H"
      print "+++++++++++"
      print "  Version is ";vers
      print "D ... enable display"
      print "P ... print data on console"
      print "S ... print timer statistics on console"
      print "1 ... single target tracking"
      print "M ... multi-target tracking"
      print "Z ... enable/suppress zero x-axis values"
      print "R ... redraw grid-lines"
      print "X ... Exit"
      print "H ... this help text"
      print "-----------"
    case else
      print cons_char;" is not a valid command ...."
  end select
end sub
  '----------------------------------------------------
  
sub do_exec_ef(ef_clstr as integer, ef_no as integer, uar as string, tst_val as integer, bit_reset as integer)
  'ef_clstr ... eventflag cluster as integer 64 bit
  'ef_no ... eventflag number 0 ... 63
  'bit_reset: 0 ... flag left unchanged, 1 ... reset flag after test
  'tst_val ... value to test ef for (0 or 1)
  
  if bit(ef_clstr, ef_no) = tst_val then                'if ef has the right value
    if(uar <> "") then                                  'and if there is something to execute
      execute uar                                       'execute user-action-routine if ef is set
      if bit_reset = 1 then bit(ef_clstr, ef_no) = 0    'reset ef if requried
    end if
  end if
end sub
  '----------------------------------------------------
  
  '---------- END DEV AREA ----------------
  
  '++++++++++ SUBs and FUNCTIONs ++++++++++++++++
sub proc_data
  local integer idx
  'processing of the radar data handed over in param_arry()
  
  for idx = 0 to 2
    if param_arry(idx,0)<> 0 then                                     'check if meaningful data are available
      alpha = 180 + atn(param_arry(idx,1)/param_arry(idx,0))*360/pi   'calc distance vector angle
      dist = sqr(param_arry(idx,0)^2 + param_arry(idx,1)^2)           'calc vector distance
    end if
  next idx
  
  if test_ef(fl_clu, prt_enab) then print_data    'print data on console
  if test_ef(fl_clu, dsp_enab) then draw_data     'draw position on display
end sub
  '----------------------------------------------------
  
sub draw_data
  for idx = 0 to 2        'restore screen areas
    on error skip 2
    blit write idx+1, x_draw(idx)-5, y_draw(idx)-5
    blit close idx+1
  next idx
  
  for idx = 0 to 2      'calc new ball positions to save screen areas
    x_pos = param_arry(idx,0)*h_fact      'scale to diaplay
    y_pos = param_arry(idx,1)*v_fact
    x_draw(idx) = x_0 + x_pos                'get display position
    y_draw(idx) = y_0 - y_pos
    blit read idx+1, x_draw(idx)-5, y_draw(idx)-5, 12, 12   'save area to be overwritten
  next idx
  
  for idx = 0 to 2      'draw new ball positions
    if param_arry(idx,0) <> 0 then
      circle x_draw(idx), y_draw(idx), 5, 1, 1, rgb(white), rgb(white)    'draw new position
    end if
  next idx
  
end sub
  '----------------------------------------------------
  
sub print_data
  local integer idx
  for idx = 0 to 2
    if (test_ef(fl_clu, z_mode) = 1) or (param_arry(idx,0)<> 0) then    'check if printout possible
      print "TG: ";idx, param_arry(idx,0), param_arry(idx,1), param_arry(idx,2), param_arry(idx,3),    'print results
      print format$(alpha,"% 3.0f"), format$(dist,"% 4.0f")
    end if
  next idx
end sub
  '----------------------------------------------------
  
sub start_measure()
  'setup params for standard measurement records
  rec_len = data_hdr_len + data_rec_len + data_trail_len
  hdr_len = data_hdr_len
  trail_len = data_trail_len
  
  'synch to data header
  do_synch s_chan, inchar, rec_len, hdr_len, syn_data_s
end sub
  '----------------------------------------------------
  
function init_sensor(s_chan as integer, init_mode as string) as integer
  'init sensor by sending begin-config, mode, end-config
  'function return codes are:
  ' 0 ... success
  ' 1 ... begin config failed
  ' 2 ... mode command failed
  ' 3 ... end config failed
  
  rec_len = data_hdr_len + data_rec_len + data_trail_len    'use as warning limit in ISR
  init_sensor = 0       'mark success
  
  'send begin configuration
  inc init_sensor       'mark first failure value
  stat = send_cmd(s_chan, cmd_head, cmd_beg_conf, cmd_trail)
  stat = get_ack(s_chan, inchar)
  if stat then
    exit function
  end if
  
  'send mode command
  inc init_sensor       'mark next failure value
  stat = send_cmd(s_chan, cmd_head, init_mode, cmd_trail)
  stat = get_ack(s_chan, inchar)
  if stat then
    exit function
  end if
  
  inc init_sensor       'mark next failure value
  stat = send_cmd(s_chan, cmd_head, cmd_end_conf, cmd_trail)
  stat = get_ack(s_chan, inchar)
  if stat then
    exit function
  end if
  init_sensor = 0       'mark success
end function
  '----------------------------------------------------
  
Function get_ack(s_chan as integer, ack_p as string) as integer
  local integer ack_size
  local string ack_s1, ack_s2
  
  do_synch s_chan, inchar, 0, ack_hdr_len, syn_ack_s        'wait for synch chars
  
  do while loc(s_chan) < 2        'make sure the ack length chars are already here
  loop
  
  'assemble ack-length
  ack_s1 = input$(1, s_chan)        'get low byte
  ack_s2 = input$(1, s_chan)        'get high byte
  ack_size = (asc(ack_s2)<<8) + asc(ack_s1) 'convert to integer
  
  do while loc(s_chan) < ack_size         'make sure the whole ack-message is ready
  loop
  ack_p = input$(ack_size, s_chan)       'get ack message part between length bytes and trailer
  
  get_ack = 0         'set funtion body to 0
  
  'get ack return value 0 ... success, 1 ... failure into the function body
  'assemble high ack-byte
  ack_s1 = mid$(ack_p,11,1)              'get ack value low byte 2
  ack_s2 = mid$(ack_p,12,1)              'get ack value high byte 2
  get_ack = (asc(ack_s1) + (asc(ack_s2)<<4))<<8     'set high byte
  
  'assemble low ack-byte
  ack_s1 = mid$(ack_p,9,1)              'get ack value low byte 1
  ack_s2 = mid$(ack_p,10,1)              'get ack value high byte 1
  get_ack = get_ack + asc(ack_s1) + (asc(ack_s2)<<4)    'assign full ack-value
  
  'make sure the trailer is here but do we not check it
  do while loc(s_chan) < ack_trail_len
  loop
  ack_s1 = input$(ack_trail_len, s_chan)   'get ack trailer
  
end function
  '----------------------------------------------------
  
function send_cmd(s_chan as integer, cmd_head as string, cmd_str as string, cmd_trail as string) as integer
  'send command to sensor
  'command is as string of hex bytes as defined in the manual, commands are reformatted to low byte first
  'header and trailer are strings and left unchanged to comply with the manual
  local string outstr, in_str
  outstr = hex_to_bin_str(cmd_head)                                     'add header
  outstr = outstr + hex_to_bin_str(sw_byte(hex$(len(cmd_str)/2,4)))     'add command/data length
  outstr = outstr + hex_to_bin_str(sw_byte(cmd_str))                    'add command/data
  outstr = outstr + hex_to_bin_str(cmd_trail)                           'add trailer
  print #s_chan, outstr;                                                'send data to sensor
end function
  '----------------------------------------------------
  
function sw_byte(in_str as string) as string
  'swap bytes in hex format "0102" to "0201"
  local integer idx
  sw_byte = ""
  for idx = 1 to len(in_str) step 4
    sw_byte = sw_byte + mid$(in_str,idx+2,2) + mid$(in_str,idx,2)
  next idx
end function
  '----------------------------------------------------
  
function bin_str_to_hex (bin_str as string) as string
  local integer idx
  bin_str_to_hex = ""
  for idx = 1 to len(bin_str)
    bin_str_to_hex = bin_str_to_hex + hex$(asc(mid$(bin_str,idx,1)),2)
  next idx
end function
  '----------------------------------------------------
  
function hex_to_bin_str(in_str as string) as string
  'convert 2 chars of in_str in hex into single char in out_str
  local integer idx, t_int1, t_int2
  local string t_char1, t_char2
  
  hex_to_bin_str = ""
  for idx = 1 to len(in_str) step 2
    t_int1 = val("&h"+mid$(in_str,idx,1))
    t_int2 = val("&h"+mid$(in_str,idx+1,1))
    t_int1 = (t_int1<<4) + t_int2
    hex_to_bin_str = hex_to_bin_str + chr$(t_int1)
  next idx
end function
  '----------------------------------------------------
  
sub set_ef (ef_clstr as integer, ef_no as integer)
  bit(ef_clstr, ef_no) = 1
end sub
  '----------------------------------------------------
  
sub clr_ef (ef_clstr as integer, ef_no as integer)
  bit(ef_clstr, ef_no) = 0
end sub
  '----------------------------------------------------
  
sub toggle_ef(ef_clstr as integer, ef_no as integer)
  bit(ef_clstr, ef_no) = not bit(ef_clstr, ef_no)
end sub
  '----------------------------------------------------
  
function test_ef (ef_clstr as integer, ef_no as integer, bit_reset as integer) as integer
  'ef_clstr ... eventflag cluster as integer 64 bit
  'ef_no ... eventflag number 0 ... 63
  'bit_reset: 0 ... flag left unchanged, 1 ... reset flag after test
  
  test_ef = bit(ef_clstr, ef_no)      'get ef value
  if bit_reset = 1 then bit(ef_clstr, ef_no) = 0      'reset ef if requried
end function
  '----------------------------------------------------
  
sub waitfr_ef(byref ef_clstr as integer, ef_no as integer)
  'test if any ef specified in mask is set, otherwise do spinwait
  do while bit(ef_clstr, ef_no) = 0
  loop
end sub
  '----------------------------------------------------
  
sub set_exec_timer (t_nbr as integer)
  'start timer 0 to 9 if read out before (timer-value = 0)
  on error skip 1
  dim float __exec_tmr(10)
  if __exec_tmr(t_nbr) = 0 then __exec_tmr(t_nbr) = timer
end sub
  '----------------------------------------------------
  
function get_exec_timer (t_nbr as integer) as float
  'get timer-value if non-zero and reset timer-value to 0
  if __exec_tmr(t_nbr) <> 0 then
    get_exec_timer  = timer  - __exec_tmr(t_nbr)
    __exec_tmr(t_nbr) = 0
  end if
end function
  '----------------------------------------------------
  
sub calc_val (param_arry() as integer, tg as integer, in_str as string, hdr_len as integer)
  
  local integer ofs
  'calculate values for target
  'param_arry:
  'first index is target number (0 ... 2) passed in tg in the call
  'second index is target value:
  '  0 ... x-distance
  '  1 ... y-distance
  '  2 ... velocity
  '  3 ... resolution
  'set_ef(ef_clu, )          'lock datastructure
  
  ofs = tg*8            'offset within record to each target data block
  
  '++++++++++++ get X-value
  param_arry(tg,0) = asc(mid$(in_str, hdr_len+ofs+1, 1)) + (asc(mid$(in_str, hdr_len+ofs+2, 1))<<8)
  'correction of x-value to pos/neg values
  if (asc(mid$(in_str, hdr_len+ofs+2, 1)) and &h80) then
    param_arry(tg,0) = &h8000 - param_arry(tg,0)
  end if
  
  'compute y, velocity and resolution only if x-value is valid
  if param_arry(tg,0) <> 0 then       'if non-zero x-value, compute rest
    '++++++++++++ get Y-value
    param_arry(tg,1) = asc(mid$(in_str, hdr_len+ofs+3, 1)) + (asc(mid$(in_str, hdr_len+ofs+4, 1))<<8)
    'correction of y-value
    param_arry(tg,1) = param_arry(tg,1) - &h8000
    
    '++++++++++++ get velocity
    param_arry(tg,2) = asc(mid$(in_str, hdr_len+ofs+5, 1)) + (asc(mid$(in_str, hdr_len+ofs+6, 1))<<8)
    'correction of velocity value
    if (asc(mid$(in_str, hdr_len+ofs+6, 1)) and &h80) then
      param_arry(tg,2) = &h8000 - param_arry(tg,2)
    end if
    
    '++++++++++++ get resolution
    param_arry(tg,3) = asc(mid$(in_str, hdr_len+ofs+7, 1)) + (asc(mid$(in_str, hdr_len+ofs+8, 1))<<8)
  end if
  
  'clr_ef(ef_clu, ser_lock)          'release datastructure
end sub
  '----------------------------------------------------
  
sub do_synch (in_chan as integer, in_str as string, rec_len as integer, hdr_len as integer, syn_data_s as string)
  'synch on the defined header bytes
  local integer t_synch = 0, idx
  
  do while t_synch = 0                              'loop while out-of-sync
    do while hex$(asc(input$(1,in_chan)),2) <> mid$(syn_data_s, 1, 2)      'synch to first char
    loop
    
    for idx = 3 to hdr_len*2-1 step 2                   'for all defined synch char tuples
      if hex$(asc(input$(1,in_chan)),2) <> mid$(syn_data_s, idx, 2) then     'synch to next chars
        t_synch = 0                                   'mark not-in-sync
        exit for                                      'if not in sequence, exit
      else
        t_synch = 1                                   'mark in-sync
      end if
    next idx                                          'try next char
  loop
  
  if rec_len > 0 then
    do while loc(in_chan) < rec_len-hdr_len        'wait for remaining chars after header
    loop
    in_str = input$(rec_len-hdr_len,in_chan)       'and throw away the rest of this message
  end if
  print "%Radar-I-Sync, Sync established"
end sub
  '----------------------------------------------------
  
function check_synch (in_str as string, syn_data as string) as integer
  local integer idx
  
  for idx = 1 to hdr_len
    if hex$(Asc(mid$(inchar, idx, 1)),2) = mid$(syn_data, idx*2-1 ,2) then     'synch to next chars
      check_synch = 0
    else
      check_synch = 1
      exit for
    end if
  next idx
  
  if check_synch = 1 then print "%Radar-W-Sync, Sync lost"
  
end function
  '----------------------------------------------------
  
  'conveniant sub to print out binary coded strings in hex format
  'used for debugging
sub prt_str_hex(in_str as string)
  local integer idx
  for idx = 1 to len(in_str)
    print hex$(asc(mid$(in_str, idx, 1)),2);
  next idx
  print
end sub
  '----------------------------------------------------
  
  '++++++++++ Interrupt service routines (ISR) +++++++++
  
  '++++++++ serial line interrupt ++++++++++++++++++++++
sub ser_int
  'if recordbuffer exceeds 1 record print warning. This is not necessarily a problem
  if loc(s_chan) > rec_len then print "%Radar-W-Buffsz, input buffer size reached ";loc(s_chan)
  inchar = input$(rec_len,s_chan)         'get record
  ' it is possible to lock the datastructure with this semaphore
  if not test_ef(fl_clu,int_dis) then                       'if interrupts on serial are enabled
    if not test_ef(ef_clu, ser_lock) then               'if lock-flag set, datastructures are locked
      set_ef(ef_clu, ser_ef)                            'set eventflag only if lock-flag = 0
    else
      print "%Radar-W-Serial_ISR, Structure lock, data lost"
    end if
  end if
end sub
  '----------------------------------------------------
  
  '+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  '     PMG subs
  '+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  
sub pmg_next_pos(x_pos as integer, y_pos as integer, x_sz as integer, y_sz as integer, x_spc as integer, y_spc as integer, n_obj_lin as integer, n_lin as integer)
  
  ' compute next position of BOX
  x_pos = x_pos+x_spc+x_sz        'next x-position
  if x_pos + x_spc + x_sz > mm.hres then  'if overflow
    x_pos = 0                     'new line, x-pos = 0
    y_pos = y_pos + y_spc + y_sz  'new line y-pos
  end if
end sub
  '-----------------------------------------------
  
sub pmg_get_size (nbr_obj as integer, nbr_chars as integer, nbr_lines as integer, nbr_obj_line as integer, x_size as integer, y_size as integer, x_space as integer, y_space as integer)
  
  'input:
  ' nbr_obj ... number of objects to be placed
  ' nbr_chars ... max number of chars in annotation
  '
  'output:
  ' nbr_lines ... number of lines required
  ' nbr_obj_line ... number of objects per line
  ' x_size ... x size of an object
  ' y_size ... y size of an object
  ' x_space ... x spacing of objects
  ' y_space ... y spacing of lines
  
  local float x_txt, y_txt, t_x_nbr, t_y_nbr, t_x_spc, t_y_spc, t_y_lin
  
  x_txt = cint((mm.fontwidth*nbr_chars)*1.1)    'get needed x size + 10%
  y_txt = cint(mm.fontheight*2)               'get needed y size + 10%
  t_x_nbr = int(mm.hres/(x_txt+1))            'numer of objects per line, 1 pixel min spaceing
  t_y_nbr = int(mm.vres/(y_txt+1))            'numnber of available lines, 1 pixel min spaceing
  t_x_spc = cint((mm.hres-(t_x_nbr*x_txt))/t_x_nbr) 'x spaceing
  t_y_spc = cint((mm.vres-(t_y_nbr*y_txt))/t_y_nbr) 'y spaceing
  t_y_lin = int((nbr_obj/t_x_nbr)+1)
  
  nbr_lines = t_y_lin
  nbr_obj_line = t_x_nbr
  x_size = x_txt
  y_size = y_txt
  x_space = t_x_spc
  y_space = t_y_spc
  
end sub
  '-----------------------------------------------
  
sub pmg_init(pmg_obj_max as integer, t_pin as string)
  dim integer __pmg_obj_max = pmg_obj_max   'setup maximum objects
  dim integer __pmg_obj_t = -1    'object last touched
  
  'setup object coordinate arrays
  dim integer __pmg_tch_x0(pmg_obj_max)
  dim integer __pmg_tch_y0(pmg_obj_max)
  dim integer __pmg_tch_x1(pmg_obj_max)
  dim integer __pmg_tch_y1(pmg_obj_max)
  dim integer __pmg_obj_par(pmg_obj_max,8)
  
  'setup interrupt pin
  setpin mm.info(pinno t_pin), intl, __t_isr
end sub
  '-----------------------------------------------
  
sub pmg_circle(pmg_obj_id as integer,obj_x0 as integer, obj_y0 as integer, obj_r as integer, obj_lw as integer, obj_ar as float, obj_col as integer, obj_fill as integer, obj_ann as string)
  
  '!!! the touch sensitive area around the circle is a square !!!
  'store touch area coordinates
  __pmg_tch_x0(pmg_obj_id) = obj_x0 - (obj_r*obj_ar)
  __pmg_tch_y0(pmg_obj_id) = obj_y0 - (obj_r)
  __pmg_tch_x1(pmg_obj_id) = obj_x0 + (obj_r*obj_ar)
  __pmg_tch_y1(pmg_obj_id) = obj_y0 + (obj_r)
  
  circle obj_x0, obj_y0, obj_r, obj_lw, obj_ar, obj_col, obj_fill
  
  text obj_x0, obj_y0,obj_ann,"CM",,,,obj_fill    'add annotation
end sub
  '-----------------------------------------------
  
sub pmg_rbox(pmg_obj_id as integer, obj_x0 as integer, obj_y0 as integer, obj_w as integer, obj_h as integer, obj_crad as integer, obj_col as integer, obj_fill as integer, obj_ann as string)
  
  ' obj_crad controls if BOX or RBOX is used
  ' >= 0 is BOX and used as box line-width
  ' < 0 is RBOX and used as box corner radius
  
  if obj_w <> 0 and obj_h <> 0 then   'data provided
    'store touch area coordinates
    __pmg_tch_x0(pmg_obj_id) = obj_x0
    __pmg_tch_y0(pmg_obj_id) = obj_y0
    __pmg_tch_x1(pmg_obj_id) = obj_x0 + obj_w
    __pmg_tch_y1(pmg_obj_id) = obj_y0 + obj_h
    
    'store box attributes
    __pmg_obj_par(pmg_obj_id,0) = obj_x0
    __pmg_obj_par(pmg_obj_id,1) = obj_y0
    __pmg_obj_par(pmg_obj_id,2) = obj_w
    __pmg_obj_par(pmg_obj_id,3) = obj_h
    __pmg_obj_par(pmg_obj_id,4) = obj_crad
    __pmg_obj_par(pmg_obj_id,5) = obj_col
    __pmg_obj_par(pmg_obj_id,6) = obj_fill
  end if
  
  'draw box
  if __pmg_obj_par(pmg_obj_id,4) < 0 then    'draw RBOX
    rbox __pmg_obj_par(pmg_obj_id,0), __pmg_obj_par(pmg_obj_id,1), __pmg_obj_par(pmg_obj_id,2), __pmg_obj_par(pmg_obj_id,3), __pmg_obj_par(pmg_obj_id,4)*-1, __pmg_obj_par(pmg_obj_id,5), __pmg_obj_par(pmg_obj_id,6)
    __pmg_obj_par(pmg_obj_id,7) = 1     'object type RBOX
  else                    'draw BOX
    'box obj_x0, obj_y0, obj_w, obj_h, obj_crad, obj_col, obj_fill
    box __pmg_obj_par(pmg_obj_id,0), __pmg_obj_par(pmg_obj_id,1), __pmg_obj_par(pmg_obj_id,2), __pmg_obj_par(pmg_obj_id,3), __pmg_obj_par(pmg_obj_id,4), __pmg_obj_par(pmg_obj_id,5), __pmg_obj_par(pmg_obj_id,6)
    __pmg_obj_par(pmg_obj_id,7) = 2     'object type BOX
  end if
  text __pmg_obj_par(pmg_obj_id,0)+(__pmg_obj_par(pmg_obj_id,2)/2), __pmg_obj_par(pmg_obj_id,1)+(__pmg_obj_par(pmg_obj_id,3)/2),obj_ann,"CM",,,,__pmg_obj_par(pmg_obj_id,6)    'add annotation
  
end sub
  '-----------------------------------------------
  
sub __t_isr
  'touch interrupt service routine
  local integer idx
  __pmg_obj_t = -1      'mark no touch
  Do While Touch(x)+1 'continue reading touch until released (then Touch(x)=-1)
    for idx = 0 to __pmg_obj_max
      If (Touch(x) > __pmg_tch_x0(idx)) and (Touch(x) < __pmg_tch_x1(idx)) and (Touch(y) > __pmg_tch_y0(idx)) and (Touch(y) < __pmg_tch_y1(idx)) then
        __pmg_obj_t = idx
        exit sub
      EndIf
    next idx
  Loop
end sub
  '-----------------------------------------------
