/*
 * CAN Clock 
 * 
 * Plus:
 * DHT22 temperature and humidity sensor
 * BMP085 barometric pressure sensor
 * Piezo buzzer
 * Pushbutton switch
 * Light sensor
 * Sound pressure level sensor
 *
 */

#include <TimeLib.h>
#include <EEPROM.h>
#include "Adafruit_LEDBackpack.h"

Adafruit_7segment matrix = Adafruit_7segment();

boolean drawDots = false;

#include "DHT.h"

#define DHTPIN 3     // what pin we're connected to

// Uncomment whatever type you're using!
//#define DHTTYPE DHT11   // DHT 11 
#define DHTTYPE DHT22   // DHT 22  (AM2302)
//#define DHTTYPE DHT21   // DHT 21 (AM2301)

DHT dht(DHTPIN, DHTTYPE);

#include "Wire.h"

#include <Adafruit_BMP085.h>

Adafruit_BMP085 bmp;

#define piezo 8

#define button1 7
#define buttonrepeat 500
#define displaystates 6
unsigned int displaystate=0;
unsigned long lastbutton=0;

unsigned int lastAlarmDay=0;

unsigned int CAN_base_address; // pulled from FLASH 0x00 0x01

unsigned long piezo_off_time;
bool piezo_on = 0;

boolean debug=false;

#include <mcp_can.h>
#include <SPI.h>

unsigned long cycles=0;
unsigned int lastcyclecount;

unsigned char lastmin=0xff;

long unsigned int rxId;
unsigned char len = 0;
unsigned char rxBuf[8];

unsigned long lasttsec=0;
unsigned long lasthsec=0;
unsigned long last1sec=0;
unsigned long last10sec=0;
unsigned long last60sec=0;
unsigned long last5min=0;

float h=0;
float t=0;

char buffer[150];
char pointer=0;

unsigned char hb[8] = {0xff, 0, 0, 0, 0, 0, 0, 0};

unsigned char stmp[8] = {0, 1, 2, 3, 4, 5, 6, 7};

MCP_CAN CAN0(10);                               // Set CS to pin 10

unsigned int txBuf[9];
unsigned char txPtr = 0;

void setup() {
//  Serial.begin(115200);
  
  CAN0.begin(CAN_500KBPS);
  pinMode(2, INPUT);                            // Setting pin 2 for CAN /INT input
  
  pinMode(piezo, OUTPUT); // Piezo output

  pinMode(A0, INPUT); // light sensor
  
  load_config();

  matrix.begin(0x70);
  matrix.setBrightness(0); // 0-15

  dht.begin();
  
//  Wire.begin(); // Already started by library?
  bmp.begin();
  
  piezo_beep();

  pinMode(button1, INPUT_PULLUP);

  //--------------------------------------------------------------------------------------------
  // Optional - set CAN address:
//  EEPROM.update(0, 0x00);
//  EEPROM.update(1, 0x40);
  // Comment out for normal operation
  //--------------------------------------------------------------------------------------------
}

// Should return 0-f; returns 255 if invalid character
unsigned char hextoval(char hexchar) {
  if (hexchar > 47 && hexchar < 58) return hexchar-48;
  if (hexchar > 64 && hexchar < 71) return hexchar-55;
  if (hexchar > 96 && hexchar < 103) return hexchar-87;
  return 255;
}

void send_heartbeat() {
//  hb[0] = 0xff; // Is set in variable definition to 0xff (heartbeat code)
  unsigned long lastnow = now();
  hb[2] = lastnow >> 24;
  hb[3] = lastnow >> 16;
  hb[4] = lastnow >> 8;
  hb[5] = lastnow;
  hb[6] = lastcyclecount >> 8;
  hb[7] = lastcyclecount;
  CAN0.sendMsgBuf(CAN_base_address, 0, 8, hb);
  hb[1]++;
  return;
}

void send_error_can(uint8_t errorcode) {
  stmp[0]=0;
  stmp[1]=errorcode;
  CAN0.sendMsgBuf(CAN_base_address, 0, 2, stmp);
  return;
}

void send_ok_can(uint8_t okcode) {
  stmp[0]=1;
  stmp[1]=okcode;
  CAN0.sendMsgBuf(CAN_base_address, 0, 2, stmp);
  return;
}

void load_config() {
  CAN_base_address = EEPROM.read(0) << 8 | EEPROM.read(1);
  if (CAN_base_address>0x7ff) CAN_base_address=0x7ff;
  send_heartbeat();
}

void can_scan() {
  if(!digitalRead(2)) {                        // If pin 2 is low, read receive buffer
    CAN0.readMsgBuf(&len, rxBuf);              // Read data: len = data length, buf = data byte(s)
    rxId = CAN0.getCanId();                    // Get message ID
    unsigned int masked = rxId & 0xfff0;
    if (masked == 0 && len>0) {
      uint8_t command=rxBuf[0];
      switch (command) { // Process broadcast messages
        case 0xf0:
          if (len==5) {
            unsigned long synctime = 0;
            synctime += (unsigned long) rxBuf[1] << 24;
            synctime += (unsigned long) rxBuf[2] << 16;
            synctime += (unsigned long) rxBuf[3] << 8;
            synctime += rxBuf[4];
            setTime(synctime);
            display_update();
//            send_ok_can(0xf0);
          } else {
            send_error_can(0xf0);
          }
          break;
        default:
          if (len>=3 && (rxBuf[1]<<8 | rxBuf[2] == CAN_base_address)) {
            switch (command) { // Process messages for me
              case 0xaa: // set display brightness
                if (len==4) {
                  matrix.setBrightness(rxBuf[3]); // 0-15
                  piezo_beep();
                  send_ok_can(0xaa);
                } else {
                  send_error_can(0xaa);
                  //Serial.println("Invalid length");
                }
                break;
              case 0xbb: // Beep piezo
                if (len==3) {
                  piezo_beep();
                  send_ok_can(0xbb);
                } else {
                  send_error_can(0xbb);
                  //Serial.println("Invalid length");
                }
                break;
              case 0xcc: // Write display
                if (len==5) {
                  unsigned int val = rxBuf[3]<<8 | rxBuf[4];
                  matrix.print(val, HEX);
                  matrix.writeDisplay();
                  delay(1000);
                  send_ok_can(0xcc);
                } else {
                  send_error_can(0xcc);
                  //Serial.println("Invalid length");
                }
                break;
              case 0x80: // EEPROM get
                if (len==5) {
                  stmp[0]=0x80;
                  stmp[1]=rxBuf[3];
                  stmp[2]=rxBuf[4];
                  stmp[3]=EEPROM.read(rxBuf[3]<<8 | rxBuf[4]);
                  CAN0.sendMsgBuf(CAN_base_address, 0, 4, stmp);
                } else {
                  send_error_can(0x80);
                }
                break;
              case 0x90: // EEPROM set
                if (len==6) {
                  EEPROM.update(rxBuf[3]<<8 | rxBuf[4], rxBuf[5]);
                  stmp[0]=0x90;
                  stmp[1]=rxBuf[3];
                  stmp[2]=rxBuf[4];
                  stmp[3]=EEPROM.read(rxBuf[3]<<8 | rxBuf[4]);
                  load_config();
                  CAN0.sendMsgBuf(CAN_base_address, 0, 4, stmp);
                } else {
                  send_error_can(0x80);
                }
                break;
            }
          }
      }
    }
  }
  return;
}  

void display_update() {
  signed int tt;
  int32_t pp;
  unsigned int ambientLight = analogRead(A0);

  if (ambientLight>40) matrix.setBrightness(15); // 0-15
  else matrix.setBrightness(0); // 0-15
  
  int wkday = weekday();
  unsigned int minutes = minute();
  unsigned int hours = hour();

  if (wkday != lastAlarmDay) { // We haven't sounded alarm yet today
    if (wkday > 1 && wkday < 7) { // Is weekday (not Sun / Sat)
      if (hours >= 6 && minutes >= 0) { // Is later than 6am
        // sound alarm
        piezo_beep();
        lastAlarmDay = wkday;
      }
    }
  }

  switch (displaystate) {
    case 0:
      //Serial.println("Clock");
       minutes = minute();
      if (lastmin != minutes) {
        hours = hour();
    //    if (hours==0) hours=12;
        if (hours>9) {
          matrix.writeDigitNum(0, (hours / 10) % 10);
        } else {
          matrix.println(0); // clears display (apart from digit 4 '0' which will get overwritten by minutes)
          //matrix.writeDigitNum(0, 255); // 255 clears digit (but also clears colon :(
        }
        matrix.writeDigitNum(1, hours % 10);
        matrix.drawColon(true);
        matrix.writeDigitNum(3, (minutes / 10) % 10);
        matrix.writeDigitNum(4, minutes % 10);
        matrix.writeDisplay();
      }
      lastmin=minutes;
      break;
    case 1:
      lastmin=99;
      matrix.print(ambientLight, DEC);
      matrix.drawColon(false);
      matrix.writeDisplay();
      break;
    case 2:
      lastmin=99;
      tt = bmp.readTemperature()*100;
      matrix.print(tt, DEC);
      matrix.drawColon(true);
      matrix.writeDigitRaw(4, 0);
      matrix.writeDisplay();
      break;
    case 3:
      lastmin=99;
      matrix.print((unsigned int) t*100, DEC);
      matrix.drawColon(true);
      matrix.writeDigitRaw(2, 0x12);
      matrix.writeDigitRaw(4, 0);
      matrix.writeDisplay();
      break;
    case 4:
      lastmin=99;
      pp = bmp.readPressure();
      matrix.print(pp/100, DEC);
      matrix.writeDigitRaw(2, 0x04);
      matrix.writeDisplay();
      break;
    case 5:
      lastmin=99;
      matrix.print((unsigned int) h, DEC);
      matrix.writeDigitRaw(2, 0x08);
      matrix.writeDisplay();
      break;
  }
}

void send_dht_can() {
  h = dht.readHumidity();
  t = dht.readTemperature();
  if (isnan(t) || isnan(h)) {
    stmp[0] = 0;
    CAN0.sendMsgBuf(CAN_base_address+1, 0, 1, stmp);

    CAN0.sendMsgBuf(CAN_base_address+2, 0, 1, stmp);
  } else {
    unsigned long lastnow = now();
    stmp[0] = lastnow >> 24;
    stmp[1] = lastnow >> 16;
    stmp[2] = lastnow >> 8;
    stmp[3] = lastnow;
    signed int tt = t*100;
    stmp[4] = tt >> 8;
    stmp[5] = tt;
    CAN0.sendMsgBuf(CAN_base_address+2, 0, 6, stmp);

    unsigned int hh = h*100;
    stmp[4] = hh >> 8;
    stmp[5] = hh;
    CAN0.sendMsgBuf(CAN_base_address+3, 0, 6, stmp);
  }
}

void send_bmp_can() {
  unsigned long lastnow = now();
  stmp[0] = lastnow >> 24;
  stmp[1] = lastnow >> 16;
  stmp[2] = lastnow >> 8;
  stmp[3] = lastnow;
  signed int tt = bmp.readTemperature()*100;
  int32_t pp = bmp.readPressure();
  stmp[4] = tt >> 8;
  stmp[5] = tt;
  CAN0.sendMsgBuf(CAN_base_address+4, 0, 6, stmp);

  stmp[4] = pp >> 24;
  stmp[5] = pp >> 16;
  stmp[6] = pp >> 8;
  stmp[7] = pp;
  CAN0.sendMsgBuf(CAN_base_address+5, 0, 8, stmp);
}

void send_luminance_can() {
  unsigned long lastnow = now();
  stmp[0] = lastnow >> 24;
  stmp[1] = lastnow >> 16;
  stmp[2] = lastnow >> 8;
  stmp[3] = lastnow;
  unsigned int aa = analogRead(A0);
  stmp[4] = aa >> 8;
  stmp[5] = aa;
  CAN0.sendMsgBuf(CAN_base_address+1, 0, 6, stmp);
}

void ticks() {
  // after a tenth of a second
  if (millis()-lasttsec>100) {
    lasttsec = millis();
  }
  // after 1/2 second
  if (millis()-lasthsec>500) {

    display_update();
/*
    // Flashing colon - timing doesn't work very well with other stuff happening
    matrix.drawColon(drawDots);
    matrix.writeDisplay();
    if (drawDots) drawDots=false;
    else drawDots=true;
*/
    lasthsec = millis();
  }
  // after 1 second
  if (millis()-last1sec>1000) {

//    Serial.println(analogRead(A0)); luminance
    
    last1sec = millis();
  }
  // after 10 seconds
  if (millis()-last10sec>10000) {

    last10sec = millis();
  }
  // after 60 seconds
  if (millis()-last60sec>60000) {

    send_dht_can();
    
    send_bmp_can();

    send_luminance_can();

    last60sec = millis();
  }
  // after 300 seconds (5 mins)
  if (millis()-last5min>300000) {

    lastcyclecount = cycles/300;
    cycles=0;
    send_heartbeat();

    last5min = millis();
  }
}

void piezo_beep() {
  digitalWrite(piezo, HIGH);
  piezo_on=true;
  piezo_off_time=millis()+250;
}

void scan_piezo() {
  if (piezo_on) {
    if (millis()>piezo_off_time) {
      digitalWrite(piezo, 0);
      piezo_on=false;
    }
  }
}

void process_button() {
  if (millis()-lastbutton > buttonrepeat) {
    if (!digitalRead(button1)) {
      displaystate++;
      if (displaystate>=displaystates) displaystate=0;
      //Serial.println(displaystate);
      lastbutton=millis();
    }
  }
}

void loop() {
  can_scan(); // Get and process CAN packet if there is one
  ticks();
  scan_piezo();
  cycles++;
  process_button();
}

